// Copyright 1994 Brad Pitzel
//
// Feel free to use/distribute/modify as long as credit/copyrights for myself 
// are included.

// Based very heavily on the Object structure from xasteroids. :)

#ifndef __BPObj__
#define __BPObj__

#include <math.h>
#include <vgagl.h>

// Note: bitmaps should be square: width=height
// Objects drawn in bitmap should be approx. round
// since the collision detection method used in this prog is
// pretty simple/dumb.  

class ObjType {
    public:
    
    	ObjType( int sh, double m, const Bitmap& b) : 
    		shape(sh), mass(m),
    		Vtime(0), Vsize(0) 
    		{
    		setBitmap(b);
    		};

	// yuck! virtual functions, which result in nicely designed
	// objects/code in sasteroids, also result in SLOW code. I would
	// have used them more, except... slow slow slow.    	
    	inline virtual void draw() const 
    		{
		Vbitmap->put( (int)x-Vsize, (int)y-Vsize );
    		}
	
	// do the stuff that should happen in one 'frame' of time, i.e.
	// move the obj, etc..
    	inline virtual void tick()		
    		{	
    		if (Vtime>0)
    			{
    			Vtime--;
    			if (!Vtime) die();
    			}
    			
    		this->x += this->xvel*speedscale;
		while (this->x < 0) this->x += (double) WIDTH;
		while (this->x > WIDTH) this->x -= (double) WIDTH;
		
		this->y += this->yvel*speedscale;
		while (this->y < 0) this->y += (double)HEIGHT;
		while (this->y > HEIGHT) this->y -= (double)HEIGHT;
		}

	void setBitmap( const Bitmap &b ) 
		{ 
		Vbitmap=&b; 
		Vsize = b.width() >> 1;
		}

	//Bitmap& bitmap() { return b; }

	// give object a fixed amount of time to live..
	inline void	setTime( int time ) { Vtime = time; }
	
	inline void	die() { alive=0; Vtime=0; }
	
	inline int	size() const { return Vsize; }
	
    	int shape, alive;
	double mass, x, y, xvel, yvel, rot;
	
    protected:
	const Bitmap *Vbitmap;
	
    private:
	int	Vsize, Vtime;
     };



// Player's ship.  Only create one instance of it in the game.
//
class Ship : public ObjType {
    public:

    	Ship( int sh, double m, const Bitmap& b) : 
    		ObjType(sh,m,b), pos(0), shieldOn(0)
    		{
    		}

	
	// if shield on, returns the 'energy level' (approx to timing
	// remaining) for the shield, between 0 and 1.0
	inline double shieldPercent() const
			{ 
			if (shieldOn)
				return ((double)energy)/((double)shieldMax());
			else	return 0.0;
			}				

    	inline void draw() const
    		{
    		ObjType::draw();
    		if (shieldOn)
    			gl_circle( (int)this->x, (int)this->y, 10, 254 );
		}
		    	
	void tick() 
		{
		ObjType::tick();
		if (this->shieldOn)
			{
			this->energy--;
			if (this->energy <=0)
				{
				this->energy=0;
				this->shieldOn=0;
				}
			}
		if (thrustCnt) 
			{
			thrustCnt--;
			if (thrustCnt<=0) 
				{
				thrustCnt=0;
				Ship::setBitmap();
				}
			}
		}
		    	
    	void setBitmap()
    		{
    		if (thrustCnt) 	Vbitmap = &(shipThr[pos]);
    		else		Vbitmap = &(ship[pos]);
    		}

    	inline void rotLeft()
    		{
		this->rot -= M_PI/8;
		this->pos--;
		this->pos %= SHIPTURNS;
		this->Ship::setBitmap();
		}

    	inline void rotRight()
    		{
		this->rot += M_PI/8;
		this->pos++;
		this->pos %= SHIPTURNS;
		this->Ship::setBitmap();
		}

	inline void thrust()
		{
		this->xvel += cos(this->rot);
		this->yvel += sin(this->rot);
				    
		thrustCnt = 5;
		Ship::setBitmap();
		}

	inline void  shield()
		{
		if (this->energy)
			this->shieldOn = 1;
		}

	void newShip(int newX, int newY)
		{
	    	this->x = newX;
		this->y = newY;
		this->xvel = 0;
		this->yvel = 0;
		this->rot = 3*M_PI_2;
		this->pos    = 0;
		this->energy = shieldMax();
		this->shieldOn = 0;
		this->thrustCnt = 0;
		this->Ship::setBitmap();
		this->alive=1;
		};

	inline int shielded() const { return shieldOn; }

    private:

	inline int shieldMax() const { return 200; }  // max shield value
    
	unsigned char pos;
    	int	 energy,shieldOn;
    	int	 thrustCnt;
    	
    };
    
#endif
