/* --------------------------------- grfast.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Vga graphics driver, uses low level graphics primitives, 256 colors only.
*/

#undef USECLEAR

#include "fly.h"
#include "gr.h"

#include <conio.h>


#ifdef DEBUG_GR
extern void	FAR GrfMoveS3 (int x1, int y1);
extern void	FAR GrfDrawS3 (int x2, int y2, Uint c);
#endif

extern void far _fastcall _GrSetColor (int color, int r, int g, int b);

extern void	FAR LogStats (void);
extern Ulong	FAR _BankSwitches;

#define DOSYNC		0x0001
#define BIGPAGE		0x0006
#define POSTSYNC	0x0008
#define INITED		0x1000

#define GRF_STATS	(GRT_MASK+1)

#define MAX_SYNC_WAIT	1000L	/* 1 second is long enough */

static int	FAR width = 0, FAR height = 0;
static long	FAR pagesize = 0L;
static long	FAR psize[] = {0L, 512L*1024L, 1024L*1024L, 2048L*1024L};

LOCAL_FUNC int FAR
GrfSetActive (int page)
{
	GrSetActive (page * pagesize);
	return (0);
}

LOCAL_FUNC int FAR
GrfSetVisual (int page)		/* done */
{
	Ulong	lasttime;
	int	ret;
	int	port;

	if (2 == CS->device->colors)
		port = 0x3ba;		/* monochrome */
	else
		port = 0x3da;		/* colour */
	lasttime = st.lasttime;

	if (CS->device->flags & DOSYNC) {
		while (inp (port) & 0x01) {	/* wait for Display Enabled */
			sys_poll (31);
			if (st.lasttime - lasttime > MAX_SYNC_WAIT) {
				LogPrintf ("%s: sync timed out\n", Gr->name);
				die ();
			}
		}
	}

	ret = GrSetVisual (page * pagesize);

	if (CS->device->flags & (DOSYNC|POSTSYNC)) {
		while (inp (port) & 0x08) {	/* wait for Vert Sync*/
			sys_poll (32);
			if (st.lasttime - lasttime > MAX_SYNC_WAIT) {
				LogPrintf ("%s: sync timed out\n", Gr->name);
				die ();
			}
		}
		while (!(inp (port) & 0x08)) {	/* wait for Vert Sync end */
			sys_poll (33);
			if (st.lasttime - lasttime > MAX_SYNC_WAIT) {
				LogPrintf ("%s: sync timed out\n", Gr->name);
				die ();
			}
		}
	}
	return (ret);
}

#undef MAX_SYNC_WAIT

LOCAL_FUNC int FAR
GrfSetWriteMode (int mode)
{
	switch (mode) {
	default:
	case T_MSET:
		mode = 0;
		break;
	case T_MOR:
		mode = GrOR;
		break;
	case T_MXOR:
		mode = GrXOR;
		break;
	}
	GrSetWriteMode (mode);
	return (0);
}

LOCAL_FUNC int FAR
GrfSetPalette (int index, long c)
{
	_GrSetColor (index, C_RGB_R (c), C_RGB_G (c), C_RGB_B (c));
	return (0);
}

LOCAL_FUNC int FAR
GrfOptions (char *options)
{
	char		*p;
	struct chip	*c;
	long		temp;

	if (F(p = get_piarg (options, 1)))
		return (1);
	for (c = chips;; ++c) {
		if (!c->name) {
			LogPrintf ("Bad video type %s\n", p);
			STRfree (p);
			return (1);
		}
		if (!stricmp (p, c->name)) {
			Gr->flags |= c->type;
			break;
		}
	}
	STRfree (p);
	if (get_arg (options, "stats"))
		Gr->flags |= GRF_STATS;

#ifdef DEBUG_GR
	if (get_arg (options, "accel")) {
		Gr->DrawTo = GrfDrawS3;
		Gr->MoveTo = GrfMoveS3;
	} else if (get_arg (options, "usec")) {
		Gr->DrawTo = GrDrawToC;
		Gr->MoveTo = GrMoveToC;
	} else if (get_arg (options, "usebasic")) {
		Gr->DrawTo = GrDrawTo;
		Gr->MoveTo = GrMoveTo;
	} else {
		Gr->DrawTo = GrDrawToA;
		Gr->MoveTo = GrMoveToA;
	}
#endif

	if (!get_narg (options, "shutters=", &temp))
		st.misc[7] = (int)temp;
	else
		st.misc[7] = 0;

	return (0);
}

LOCAL_FUNC int FAR
GrfInit (DEVICE *dev, char *options)
{
	int	i;
	int	row;

	if (GrfOptions (options))
		return (1);

	if (dev->sizex == 0 || dev->sizey == 0) {
		LogPrintf ("Bad WxH in .vmd file\n");
		return (1);
	}
	width = dev->sizex;
	height = dev->sizey;
	pagesize = T(i = dev->flags & BIGPAGE)
			? psize[i/2]
			: (long)width * (long)height;
	if (!dev->mode) {
		LogPrintf ("Must have video mode in .vmd file\n");
		return (1);
	}
	GrSetXY (dev->sizex, dev->sizey);
	GrSetBiosMode (dev->mode);
	GrSetType (Gr->flags & GRT_MASK, (int)(dev->npages*pagesize/1024L),
		dev->sizex);

#ifdef DEBUG_GR
	while (GrAllocCell () >= 0)		/* get 2-255 */
		;
#endif

	if (GrfSetVisual (0))
		dev->npages = 1;

	GrfSetWriteMode (T_MSET);
	GrfSetPalette (CC_BLACK, C_BLACK);

	for (i = dev->npages; --i >= 0;) {
		GrfSetActive (i);
#if 0
		GrClear (0, 0, dev->sizex, dev->sizey, st.colors[CC_BLACK]);
#else
		for (row = 0; row < dev->sizey; ++row) {
			Gr->MoveTo (0, row);
			Gr->DrawTo (dev->sizex-1, row, st.colors[CC_BLACK]);
		}
#endif
	}

	Gr->flags |= INITED;

	return (0);
}

LOCAL_FUNC void FAR
GrfTerm (DEVICE *dev)
{
	if (!(Gr->flags & INITED))
		return;
	Gr->flags &= ~INITED;

	LogPrintf ("BankSwitches %s\n", show_ul (_BankSwitches));
	if (Gr->flags & GRF_STATS)
		LogStats ();

	GrSetType (GRT_NONE, 256, 200);
	GrSetBiosMode (0x03);		/* text 80x25 */
}

#ifdef USECLEAR
LOCAL_FUNC void FAR
GrfClear (SCREEN *scr)
{
	GrClear (scr->minx, scr->miny, scr->sizex, scr->sizey, scr->BgColor);
}
#endif

LOCAL_FUNC int FAR
GrfShutters (int eye)
{
	if (st.misc[7]) {
		if (eye >= 0)
			outp (st.misc[7]+4, 1+2*eye);
		else if (-1 == eye)
			outp (st.misc[7]+4, 1);		/* on */
		else if (-2 == eye)
			outp (st.misc[7]+4, 0);		/* off */
		return (0);				/* have shutters */
	} else
		return (1);				/* no shutters */
}


struct GrDriver NEAR GrFast = {
	"GrFast",
	0,
	NULL,	/* extra */
	0,
	GrfInit,
	GrfTerm,
	GrMoveToA,
	GrDrawToA,
	GrfSetVisual,
	GrfSetActive,
#ifdef USECLEAR
	GrfClear,
#else
	0,	/* GrfClear() too slow */
#endif
	GrfSetWriteMode,
	GrfSetPalette,
	GrEllipse,
	0,	/* Flush */
	GrfShutters
};
#undef DOSYNC
#undef BIGPAGE
#undef POSTSYNC
#undef GRF_STATS
#undef INITED
#undef MAX_SYNC_WAIT
