/*
 *  chapter.cpp
 *
 *  Description:
 *
 *      This program reads one of the files "lib\help\*.txt" of Angband 2.7.9v6 
 *      or later, and outputs a file in another format. The latter file can be used
 *      for online documentation in a graphical Angband implementation, or
 *      to create another version of the text files with better chapter numbering
 *      and layout or to provide a HTML version of the documentation.
 *
 *      Currently, only OS/2 .IPF format and text output is implemented. Feel free to         
 *      add other emitter classes; the program has been tuned to simplify this process.
 *      Look at the IPFEmitter class (search for "IPFEmitter::" in your editor) for a 
 *      working example. The base class for deriving emitter classes is GeneralEmitter.
 *      Refer to its documentation for more help.
 *
 *      You may use GCC 2.7.0 or any other version not too old. Other compilers have
 *      not been tested; the program only uses parts of the C++ draft.
 *
 *      The width of this file doesn't exceed 94 characters.
 *
 *  Copyright (C) 1996 Ekkehard Kraemer (ekraemer@pluto.camelot.de)
 *
 *  This software is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but without any warranty; without even the implied warranty of
 *  merchantability or fitness for a particular purpose.  See the
 *  GNU General Public License for more details.
 *
 */

#include <io.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <fstream.h>
#include <assert.h>
#include <stdio.h>

// Global constants
#define MAXLINELEN 1024                                     // Maximum input line 
                                                            // *after* convChars

// TextEmitter constants
#define LINELEN 66                                          // Len of TextEmitter line
                                                            // (including indention)
#define FIRSTINDENT 6                                       // Indent of first line of par.
#define NORMALINDENT 6   // or smaller than FIRSTINDENT     // Indent of every line of par.
                                                            // and of "item" of a list
#define LISTINDENT 10                                       // Indent of every line of list
#define MAXSPACES 15                                        // Maximum amount of spaces
                                                            // inserted to expand line

/*
 * General emitter class.
 *
 *   This class is used to "emit" the output file. To implement a new emitter,
 *   create a new class that inherits from GeneralEmitter. Implement all abstract
 *   functions and those of the others that you need. To test it,
 *   find the place in main() where the emitters are constructed and insert yours 
 *   there.
 *
 *   Send your finished class to ekraemer@pluto.camelot.de and tell me which
 *   language/OS this class is for. I'll add your class to the current code
 *   and make it available on my home page and on r.g.r.a.
 *
 *   You should *not* have to change anything at all in the rest of the code.
 *   If you have to, please report it, because it would most probably indicate a 
 *   design flaw.
 *
 *  Public Functions
 *
 *     virtual void beginChapter(ostream &o,const char *chapter,
 *                               unsigned level,const char *id);
 *
 *         This function must write the header of the following chapter
 *         to stream 'o'. The chapter title is 'chapter'. If you need to create
 *         an ID for each chapter, use 'id', which was constructed from the
 *         "name-prefix" command line parameter as prefix and continuously
 *         appended outputs from getAdditionalID.
 *
 *         'level' denotes the hierarchical level of the chapter. Values start 
 *         at 0 and are not limited (in fact, they are limited to 2 because
 *         of the input file structure).
 *
 *     virtual void endChapter(ostream &o);
 *
 *         This function may be implemented to finish the current (sub-)chapter.
 *         E.g., some markup language may need a "end of chapter" tag.
 *         It is called *before* the list of subchapters is generated
 *         (I.e., startSubChapters, putSubChapterLink, endSubChapters) and
 *         *before* the following subchapters.
 *
 *     virtual void endDoc(ostream &o);
 *
 *         This function may be implemented to write something to the end of 
 *         the document. E.g., some markup language may need a "end of document"
 *         tag.
 *
 *     virtual void putLine(ostream &o,const char *line);
 *
 *         This function may be overwritten to write one line of text to the
 *         output file. The 'line' has already been character-converted
 *         (see convChars). It should remove leading and ending white spaces
 *         (use stripSpaces to do this).
 *
 *     virtual void putParagraph(ostream &o);
 *
 *         This function should be implemented to write the "new paragraph"
 *         token of hypertext markup languages.
 *
 *     virtual void beginFixedFont(ofstream &o);
 *     virtual void endFixedFont(ofstream &o);
 *
 *         This functions may be implemented to set/unset fixed fonts in languages
 *         where proportional fonts are used by default. If fixed font is active,
 *         putLine should *not* remove leading white spaces (this font is used for 
 *         tables).
 *
 *     virtual void beginList(ostream &fo);
 *     virtual void beginListItem(ostream &fo,const char *item);
 *     virtual void endListItem(ostream &fo);
 *     virtual void endList(ostream &fo);
 *
 *         This functions may be implemented to properly display item lists in
 *         markup languages. beginList starts a list, beginListItem introduces
 *         the item 'item', endListItem ends it and and endList ends the whole list. 
 *         There will be no putParagraph after endList and endListItem.
 *         Note: if the language has something like an index, it would be nice
 *         to store 'item' there during beginListItem.
 *
 *     virtual void convChars(char *line);
 *
 *         This function may be implemented to remove "deadly" things from each
 *         line. E.g., normally each markup language has one or more 
 *         "introduce tag"-characters, and these characters can't occur in
 *         the middle of the text, but are normally replaced by a tag.
 *         E.g., instead of '&', the string "&colon." may occur in a text.
 *
 *         To simplify conversion, you just need to call convChars(line,tab),
 *         passing it the line and an array of ConvTab entries (which is 
 *         delimited by a {0,0} pair), because this function handles the most
 *         usual case. See IPFEmitter::convChars.
 *
 *         The function is called for each text line and each header title.
 *
 *     virtual void startSubChapters(ofstream &fo,const char *title);
 *
 *         This function may be implemented to introduce a series of 
 *         putSubChapterLink calls and should, e.g., introduce a unnumbered list.
 *
 *     virtual void putSubChapterLink(ofstream &fo,const char *chapter,
 *                                    const char *id);
 *
 *         This function may be implemented to write links to subchapters
 *         of the current chapter. They should appear on their own line.
 *         'chapter' is the target chapter title, 'id' the hypertext ID.
 *
 *     virtual void endSubChapters(ofstream &fo);
 *
 *         This function may be implemented to end the list of subchapters.
 *
 *     virtual const char *getAdditionalID(unsigned id,int first);
 *
 *         This function may be overwritten to return a part of a chapter's ID.
 *         For example, for a 'id' of 4 it could return ".4" or "/4" or the like.
 *         If 'first' isn't 0, then the delimiter should be ommited (i.e., "4"
 *         instead of ".4").
 *         This is used to consistently identify all chapters for hyperlinks.
 *         If your emitter doesn't need (or use) chapter IDs, you may return "".
 *
 *         'id' starts at 0, so for human-readable results you should add 1.
 *
 *         The return value should be a pointer to a static buffer.
 *
 *
 * Protected Functions
 *
 *   virtual void convChars(char *line,const ConvTab *tab);
 *
 *         See convChars(line). This function takes an array of type ConvTab,
 *         tests each character of 'line' against each entry of 'tab'
 *         and, if apropriate, replaces tab[].chr with tab[].str.
 *         'tab' *must* end with a {0,0} pair.
 *
 *         Overwrite convChars(line) instead, if you have to.
 *
 */

class GeneralEmitter
{
protected:
    struct ConvTab
    {
        char chr;
        char *str;
    };

    virtual void convChars(char *line,const ConvTab *tab);

public:
    virtual void beginChapter(ostream &o,const char *chapter,
                              unsigned level,const char *id)=0;
    virtual void endChapter(ostream &) {}
    virtual void endDoc(ostream &) {}
    virtual void convChars(char *) {}
    virtual const char *getAdditionalID(unsigned int,int first);

    virtual void putLine(ostream &o,const char *line);
    virtual void putParagraph(ostream &o) { putLine(o,""); }
    virtual void beginList(ostream &) { }
    virtual void endList(ostream &o) { putParagraph(o); }
    virtual void beginListItem(ostream &o,const char *item) { putLine(o,item); }
    virtual void endListItem(ostream &o) { putParagraph(o); }
    virtual void beginFixedFont(ofstream &) {}
    virtual void endFixedFont(ofstream &) {}

    virtual void startSubChapters(ofstream &,const char *) {}
    virtual void putSubChapterLink(ofstream &,const char *,const char *) {}
    virtual void endSubChapters(ofstream &) {}
};

/*
 * Emitter class for the OS/2 .IPF online help and documentation language.
 *
 * You may take this class as an example of how to implement other emitters.
 */

class IPFEmitter : public GeneralEmitter
{
    int inFixed;
public:
    virtual void beginChapter(ostream &o,const char *chapter,
                              unsigned level,const char *id);
    virtual void convChars(char *line);
    virtual const char *getAdditionalID(unsigned int,int first);

    virtual void startSubChapters(ofstream &,const char *);
    virtual void putParagraph(ostream &o);
    virtual void putSubChapterLink(ofstream &fo,const char *chapter,const char *id);
    virtual void endSubChapters(ofstream &); 

    virtual void putLine(ostream &o,const char *line);
    virtual void beginList(ostream &fo);
    virtual void endList(ostream &fo);
    virtual void beginListItem(ostream &o,const char *item);
    virtual void endListItem(ostream &) { }
    virtual void beginFixedFont(ofstream &o);
    virtual void endFixedFont(ofstream &o);

    IPFEmitter() { inFixed=0; }
};

/*
 * Emitter class for normal, untagged, human-readable text. Left as reference
 * (the absolute minimum) - not used.
 */

class SimpleTextEmitter : public GeneralEmitter
{
public:
    virtual void beginChapter(ostream &o,const char *chapter,
                              unsigned level,const char *id);
};

/*
 * Emitter class for formatted text output.
 *
 */

class TextEmitter : public GeneralEmitter
{
    int inFixed,inFirstLine;
    int firstIndent,normalIndent;
    char curLine[LINELEN+1];

    void putWord(ostream &fo,char *word);
    void flush(ostream &fo,int expand=0);
    int lineLen();

public:
    virtual void beginChapter(ostream &o,const char *chapter,
                              unsigned level,const char *id);
    virtual void putParagraph(ostream &o);
    virtual void putLine(ostream &o,const char *line);
    virtual void beginList(ostream &fo);
    virtual void endList(ostream &fo);
    virtual void beginListItem(ostream &o,const char *item);
    virtual void endListItem(ostream &);
    virtual void beginFixedFont(ofstream &o);
    virtual void endFixedFont(ofstream &o);

    TextEmitter() 
    { 
        firstIndent=FIRSTINDENT; normalIndent=NORMALINDENT; 
        inFixed=0; inFirstLine=1; 
    }
};

/*
 *  Main work class. No need to fiddle with it to implement other emitters.
 */

class Converter
{
    class Page
    {
        unsigned id;
        char *title;
        char *idString;
        char **content;
        int size,maxSize;
        Page **descendants;
        unsigned descs;
        GeneralEmitter *emit;

        void writeChapter(ofstream &fo,int level);
        void writeOtherChapters(ofstream &fo,unsigned level,Page *next,Page *parent);
        int isTable(int t,int &firstCol);
        int isListItem(int t);
        int writeList(ofstream &fo,int &t);
        int writeTable(ofstream &fo,int &t);

    public:

        void append(const char *line);
        Page *newChapter(const char *tit);
        Page &lastChapter();
        const char *getId() { return idString; }
        void write(ofstream &fo,unsigned level,Page *next,Page *parent);

        Page(GeneralEmitter *emit,const char *idPrefix,const char *t,unsigned i);
        Page(GeneralEmitter *emit,const char *idPrefix);
        ~Page() { delete idString; }
    };

    const char *idName;
    GeneralEmitter *emit;
    Page mainPage;

    int getChapter(char *line);
    void read(ifstream &fi);

public:

    void convert(ifstream &fi,ofstream &fo);

    Converter(GeneralEmitter *emit,const char *idName);
};

/*
 * Main program. 
 */

int main(int argc,char **argv)
{
    if (argc!=5)                                            // Output usage text
    {
        cerr << 
            "Usage: " << argv[0] << " type name-prefix input-file output-file\n\n"
            "'type' may be any of:\n"
            "           1  --  Text\n"
            "           2  --  OS/2 IPF\n\n"
            "'name-prefix' will identify all chapters of this particular file.\n"
            "           If you don't need this, use \"\" as prefix.\n\n"
            "'input-file' is one of the files out of lib\\help\\*.txt of Angband 2.7.9v6\n"
            "           or later.\n\n"
            "'output-file' is the file which is newly written.\n";
            
        exit(1);
    }

    int type=atoi(argv[1]);
    GeneralEmitter *emit;

    switch (type)
    {
        case 1:         emit=new TextEmitter;   break;     
        case 2:         emit=new IPFEmitter;    break;      

        default:                                            // Unknown emitter
            cerr << "Unknown emitter " << type << ".\n";    
            exit(1);
    }

    ifstream fi(argv[3]);                                   // Open input file
    if (!fi.good())
    {
        cerr << "Can't open input file '" << argv[3] << "'.\n";
        exit(1);
    }

    ofstream fo(argv[4]);                                   // Open output file
    if (!fo.good())
    {
        cerr << "Can't open output file '" << argv[4] << "'.\n";
        exit(1);
    }

    Converter converter(emit,argv[2]);
    converter.convert(fi,fo);                               // Convert files
    delete emit;

    if (!fo.good())
    {
        cerr << "Error writing to '" << argv[4] << "'.\n";
        exit(1);
    }

    return 0;                                               // Exit
}

void stripSpaces(char *line)
{
    while (isspace(line[0]))                                // Remove white spaces
        memmove(line,line+1,strlen(line)+1);
    while (isspace(line[strlen(line)-1])) 
        line[strlen(line)-1]=0;
}

// --- Converter ------------------------------------------------------------------

Converter::Converter(GeneralEmitter *e,const char *i) 
    : idName(i), emit(e), mainPage(emit,idName)
{
}

void Converter::convert(ifstream &fi,ofstream &fo)
{
    read(fi);
    mainPage.write(fo,0,NULL,NULL);
}

void Converter::read(ifstream &fi)
{
    Page *current=&mainPage;
    int skipEmptyLines=1;

    do
    {
        char line[MAXLINELEN];
        fi.getline(line,MAXLINELEN);                        // Read next line of input
        if (fi.eof()) break;
        if (!fi.good())                                     // Abort if error
        {
            cerr << "Error reading from input file.\n";
            exit(1);
        }

        int type=getChapter(line);                          // Check if it's a chapter

        switch (type)
        {
            case 0:                                         // No, it's a line of text
            {
                while (isspace(line[strlen(line)-1]))
                       line[strlen(line)-1]=0;
                int empty=!strlen(line);                    // Add it to the chapter
                if (!skipEmptyLines || !empty)              // and manage multiple
                {                                           // empty lines.
                    current->append(line);
                    if (empty) skipEmptyLines=1;
                    else skipEmptyLines=0;
                }                
                break;
            }

            case 1:                                         // Chapter
                current=mainPage.newChapter(line);
                break;

            case 2:
                current=mainPage.lastChapter().newChapter(line);
                break;
        }
    } while (fi.good());
}


int Converter::getChapter(char *line)
{
    int rc=0;

    if (!strncmp(line,"===",3) &&                           // Chapter is something like:
        !strncmp(line+strlen(line)-3,"===",3)) rc=1;        // === chapter ===

    if (!strncmp(line,"---",3) &&                           // Chapter is something like:
        !strncmp(line+strlen(line)-3,"---",3)) rc=2;        // --- subchapter ---

    if (!rc) return 0;                                      // Normal line

    memcpy(line,line+3,strlen(line+2));                     // Remove '===' or '---'
    line[strlen(line)-3]=0;

    stripSpaces(line);

    return rc;
}

// --- Converter::Page ------------------------------------------------------------

Converter::Page::Page(GeneralEmitter *e,const char *preId,const char *t,unsigned i) 
    : id(i), emit(e)
{ 
    size=maxSize=descs=0; 
    content=0; 
    descendants=0; 
    title=strdup(t); 
    assert(title); 

    const char *additional=emit->getAdditionalID(id,0);
    idString=new char[strlen(preId)+strlen(additional)+1];
    assert(idString);
    strcpy(idString,preId);
    strcat(idString,additional);
}

Converter::Page::Page(GeneralEmitter *e,const char *preId) 
    : emit(e)
{ 
    id=size=maxSize=descs=0; 
    content=0; 
    descendants=0; 
    title=0; 

    const char *additional=emit->getAdditionalID(id,1);
    idString=new char[strlen(preId)+strlen(additional)+1];
    assert(idString);
    strcpy(idString,preId);
    strcat(idString,additional);
}

void Converter::Page::append(const char *line)
{
    if (size>=maxSize)
    {
        maxSize+=100;
        content=(char**)realloc(content,sizeof(char*)*maxSize);
        assert(content);
    }
    content[size++]=strdup(line);
    assert(content[size-1]);
}

Converter::Page *Converter::Page::newChapter(const char *tit)
{
    if (!descs && !title)
    {
        title=strdup(tit);
        assert(title);
    }
    descs++;
    descendants=(Page**)realloc(descendants,sizeof(Page*)*descs);
    assert(descendants);
    descendants[descs-1]=new Page(emit,idString,tit,descs-1);
    assert(descendants[descs-1]);
    return descendants[descs-1];
}

Converter::Page &Converter::Page::lastChapter()
{
    if (!descs) return *this;
    return *(descendants[descs-1]);
}

int Converter::Page::isListItem(int t)
{
    if (strlen(content[t])  &&   (t<1 || !strlen(content[t-1])) &&   t<(int)size-1)
    {
        int spaces1=0;
        while (isspace(content[t][spaces1])) spaces1++;
                
        int spaces2=0;
        while (isspace(content[t+1][spaces2])) spaces2++;
                
        if (spaces1<spaces2) return 1;
    }

    return 0;
}

int Converter::Page::writeList(ofstream &fo,int &t)
{
    if (!isListItem(t)) return 0;

    emit->beginList(fo);
    
    do
    {
        char buf[MAXLINELEN];
        strcpy(buf,content[t]);
        stripSpaces(buf);
        emit->convChars(buf);
        
        emit->beginListItem(fo,buf);

        t++;

        while (t<(int)size && strlen(content[t]))
        {
            strcpy(buf,content[t++]);
            emit->convChars(buf);
            emit->putLine(fo,buf);
        }

        emit->endListItem(fo);

        if (t>=(int)size) break;
        t++;
    } while (isListItem(t));

    emit->endList(fo);

    return 1;
}

int Converter::Page::isTable(int t,int &firstCol)
{
    /*
     * Tables begin with a line of text, preceded by no line of text.
     */
    if ((t>0 && content[t-1][0]) || !content[t][0]) return 0; 
    
    int hot=0;
    int cols=0;

    for (int pos=0; pos<MAXLINELEN; pos++)
    {
        int len=0;
        int haveSome=0;
        int line=0;
        int hadEmpty=0;

        for (line=t; line<=size; line++)
        {
            if (line<size)
                len=strlen(content[line]);
            else 
                len=0;
            if (!len)
            {
                if (line<t+3) 
                {
                    if (isspace(content[line-1][0]) &&      // Headers start with spaces
                        !hadEmpty)                          // Admit one empty
                    {                                       // line per table (for
                        hadEmpty=1;                         // headers)
                        continue;
                    }                    
                    return 0;                               // Headers are at most 2 lines
                }                                           
                break;                                      
            }
            if (len<=pos) continue;                         // Skip short lines
            haveSome=1;
            if (!isspace(content[line][pos])) 
            {
                if (!hot) firstCol=pos;
                hot=1;                                      // Only count columns of spaces
                break;                                      // after the first non-space.
            }
        }

        if (!haveSome) return 0;

        if ((!len || line==size) && hot && line>t+3)        // Must be a table if at 
        {                                                   // least 4 lines and at
            cols++;                                         // least 3 colns of spaces
            if (cols>3) return hadEmpty?2:1;                // return number of empty lines+1
        }
    }
        
    return 0;
}

int Converter::Page::writeTable(ofstream &fo,int &t)
{
    int firstCol=0;
    int emptyLines=isTable(t,firstCol);

    if (!emptyLines) return 0;

    emit->beginFixedFont(fo);

    while (t<size)
    {
        if (!strlen(content[t]))
        {
            emptyLines--;
            if (emptyLines<1) break;
        }
        char buf[MAXLINELEN];
        strcpy(buf,content[t]);
        char *p=buf;
        if ((int)strlen(buf)>firstCol) p+=firstCol;
        emit->convChars(p);
        emit->putLine(fo,p);
        t++;
    }
    
    emit->endFixedFont(fo);

    return 1;
}

void Converter::Page::writeChapter(ofstream &fo,int level)
{
    emit->beginChapter(fo,title,level,idString);

    int t=0;

    while (t<size)
    {
        if (!writeList(fo,t) &&
            !writeTable(fo,t))
        {
            if (strlen(content[t]))
            {
                char buf[MAXLINELEN];
                strcpy(buf,content[t]);
                emit->convChars(buf);
                emit->putLine(fo,buf);
            }
            else 
                emit->putParagraph(fo);

            t++;
        }
    }
        
    emit->endChapter(fo);
}

void Converter::Page::writeOtherChapters(ofstream &fo,unsigned level,Page *next,Page *parent)
{
    if (descs) 
    {
        char buf[100];
        sprintf(buf,size?"Topics:":"More Topics:");
        emit->convChars(buf);
        emit->startSubChapters(fo,buf);
        for (unsigned t=0; t<descs; t++)
            emit->putSubChapterLink(fo,descendants[t]->title,descendants[t]->getId());
        emit->endSubChapters(fo);
    }

    if (next)
    {
        char buf[100];
        sprintf(buf,"Next chapter:");
        emit->convChars(buf);
        emit->startSubChapters(fo,buf);
        emit->putSubChapterLink(fo,next->title,next->getId());
        emit->endSubChapters(fo);
    }
        
    if (parent)
    {
        char buf[100];
        sprintf(buf,"Parent chapter:");
        emit->convChars(buf);
        emit->startSubChapters(fo,buf);
        emit->putSubChapterLink(fo,parent->title,parent->getId());
        emit->endSubChapters(fo);
    }
        
    for (unsigned t=0; t<descs; t++)
    {
        Page *next=NULL;
        if (t<descs-1) 
            next=descendants[t+1];
        descendants[t]->write(fo,level+1,next,this);
    }
}

void Converter::Page::write(ofstream &fo,unsigned level,Page *next,Page *parent)
{
    if (!title || !strlen(title)) title=strdup("Content");
    assert(title);
    
    while (size && !strlen(content[size-1])) size--;
    
    writeChapter(fo,level);
    writeOtherChapters(fo,level,next,parent);
}

// --- GeneralEmitter -------------------------------------------------------------

void GeneralEmitter::convChars(char *line,const ConvTab *convTab)
{
    for (char *p=line; *p; p++)                             // Look at all characters
    {
        for (int t=0; convTab[t].chr || convTab[t].str; t++) // Compare to all entries
        {
            if (*p==convTab[t].chr)                         // Hit
            {
                memmove(p+strlen(convTab[t].str)-1,p,strlen(p)+1); // Replace character
                memcpy(p,convTab[t].str,strlen(convTab[t].str));   //    with string
                p+=strlen(convTab[t].str)-1;
                break;
            }
        }
    }
}

void GeneralEmitter::putLine(ostream &fo,const char *line)
{
    fo << line << '\n';
}

const char *GeneralEmitter::getAdditionalID(unsigned id,int first)
{
    static char buf[10];
    sprintf(buf,"%s%u",first?"":".",id+1);
    return buf;
}

// --- IPFEmitter -----------------------------------------------------------------

void IPFEmitter::beginChapter(ostream &fo,const char *chapter,unsigned level,const char *id)
{
    fo << ":h" << (level+2) << " name=" << id << '.' << chapter << '\n';
    fo << ":hp2." << chapter << ":ehp2.:p.\n";
}

void IPFEmitter::putParagraph(ostream &fo)
{
    fo << ":p.\n";
}

void IPFEmitter::convChars(char *line)
{
    /*
     * This table (and only this table) was taken from 'txttoipf' by
     * Dennis Frost (dfrost@wvuvphs1.hsc.wvu.edu)
     *
     * Thanks and appologies for stealing your work.
     */
    static ConvTab convTab[]=
    {
        {'&', "&amp."},         {':', "&colon."},
        {'^', "&caret."},       {',', "&comma."},
        {'\'', "&apos."},       {'*', "&asterisk."},
        {'\\',"&bslash."},      {'"', "&cdq."},
        {'-', "&dash."},        {'$', "&dollar."},
        {'.', "&per."},         {'=', "&eq."},
        {'!', "&xclm."},        {'>', "&gtsym."},
        {'<', "&ltsym."},       {'{', "&lbrace."},
        {'[', "&lbracket."},    {'(', "&lpar."},
        {'#', "&numsign."},     {'`', "&osq."},
        {'%', "&percent."},     {'+', "&plus."},
        {'}', "&rbrace."},      {']', "&rbracket."},
        {')', "&rpar."},        {';', "&semi."},
        {'/', "&slash."},       {'@', "&atsign."},
        {'|', "&splitvbar."},   {'~', "&tilde."},
        {'_', "&us."},          {0,0}
    };
    
    GeneralEmitter::convChars(line,convTab);
}

void IPFEmitter::putSubChapterLink(ofstream &fo,const char *chapter,const char *id) 
{
    fo << ":li.:link reftype=hd refid=" << id << '.' << chapter << ":elink.\n";
}

const char *IPFEmitter::getAdditionalID(unsigned id,int first)
{
    static char buf[10];
    sprintf(buf,"%s%u",first?"":"/",id+1);
    return buf;
}

void IPFEmitter::startSubChapters(ofstream &fo,const char *title)
{
    fo << ":p.:hp7." << title << ":ehp7.:ul compact.\n";
}

void IPFEmitter::endSubChapters(ofstream &fo)
{
    fo << ":eul.\n";
}

void IPFEmitter::beginFixedFont(ofstream &fo)
{
    fo << ":xmp.\n";
    inFixed=1;
}

void IPFEmitter::endFixedFont(ofstream &fo)
{
    fo << ":exmp.\n";
    inFixed=0;
}

void IPFEmitter::beginList(ostream &fo)
{
    fo << ":dl break=all.\n";
}

void IPFEmitter::endList(ostream &fo)
{
    fo << ":edl.:p.\n";
}

void IPFEmitter::beginListItem(ostream &fo,const char *item)
{
    fo << ":dt.:hp2." << item << ":ehp2.:dd.\n";
    fo << ":i2 refid=gameidx." << item << '\n';
}

void IPFEmitter::putLine(ostream &fo,const char *line)
{
    char *tmp=strdup(line);
    assert(tmp);
    if (!inFixed) stripSpaces(tmp);
    fo << tmp << '\n';
    free(tmp);
}

// --- SimpleTextEmitter ------------------------------------------------------------

void SimpleTextEmitter::beginChapter(ostream &fo,const char *chapter,unsigned,const char *id)
{
    unsigned pos=fo.tellp();
    fo << '\n' << id << ' ' << chapter << "\n";
    pos=fo.tellp()-pos-2;
    for (; pos; pos--) fo << '=';
    fo << "\n\n";
}

// --- TextEmitter ------------------------------------------------------------

void TextEmitter::beginChapter(ostream &fo,const char *chapter,unsigned,const char *id)
{
    flush(fo);

    unsigned pos=fo.tellp();
    fo << '\n' << id << ' ' << chapter << "\n";
    pos=fo.tellp()-pos-2;
    for (; pos; pos--) fo << '=';
    fo << "\n\n";
}

void TextEmitter::putParagraph(ostream &o)
{
    flush(o);
}

void TextEmitter::beginList(ostream &fo)
{
    flush(fo);

    firstIndent=NORMALINDENT;
    normalIndent=LISTINDENT;
}

void TextEmitter::endList(ostream &)
{
    firstIndent=FIRSTINDENT;
    normalIndent=NORMALINDENT;
}

void TextEmitter::beginListItem(ostream &o,const char *item)
{
    putLine(o,item);
    flush(o,-1);
}

void TextEmitter::endListItem(ostream &fo) 
{
    flush(fo);
}

void TextEmitter::beginFixedFont(ofstream &fo)
{
    flush(fo);
    inFixed=1;
}

void TextEmitter::endFixedFont(ofstream &fo)
{
    inFixed=0;
    fo << '\n';
}

void TextEmitter::putLine(ostream &fo,const char *line)
{
    if (inFixed)                                            // Fixed "font":
    {                                                       // Flush all lines 
        for (int t=0; t<normalIndent; t++)                  // Indent line
            fo << ' ';
        fo << line << '\n';                                 
        return;
    }

    const char *front=line;

    do
    {
        while (isspace(*front)) front++;                    // Skip spaces
        
        if (!*front) break;                                 // Check for end of line

        const char *back=front;                             // Find end of next word
        while (!isspace(*back) && *back) back++;            // back points to ' ' or 0.

        assert(back-front<lineLen());                       // Hack, but what the heck

        char buf[LINELEN+1];                                // Copy word to temporary
        memcpy(buf,front,back-front);                       // buffer
        buf[back-front]=0;

        putWord(fo,buf);                                    // Output word

        front=back;                                         // Set front after word
    } while (1);                                            // Loop until infty
}

void TextEmitter::putWord(ostream &fo,char *word)
{
    if (strlen(curLine)+strlen(word)>=(unsigned)lineLen()-1) // Flush line if word 
        flush(fo,1);                                        // doesn't fit

    if (strlen(curLine)) strcat(curLine," ");               // Add space if necessary

    strcat(curLine,word);                                   // Add word
}

void TextEmitter::flush(ostream &fo,int expand)
{
    if (!curLine[0]) return;

    int indent=inFirstLine?firstIndent:normalIndent;

    if (expand==1 && lineLen()-strlen(curLine)<MAXSPACES)   // Expand line to LINELEN
    {                                                       // characters
        char *pos=curLine;

        int expanded=0;

        while ((int)strlen(curLine)<lineLen())
        {
            if (/* *pos=='.' || *pos==',' || *pos==':' ||   // Add space after
                *pos==';' || *pos=='!' || *pos=='?' ||      // punctuation or space
                *pos==')' || */ *pos==' ')
            {
                memmove(pos+1,pos,strlen(pos)+1);
                pos++;
                *pos=' ';
                while (*pos && isspace(*pos)) pos++;        // Skip other spaces
                expanded=1;
            }

            if (*pos) pos++;
            if (!*pos) 
            {
                pos=curLine;                                // Begin once again
                if (!expanded) break;                       // But only if useful
            }
        }
    }
    
    assert(strlen(curLine)<=LINELEN);                       // Paranoia

    for (int t=0; t<indent; t++)                            // Indent line
        fo << ' ';
    
    fo << curLine << '\n';                                  // Output line
    if (expand==0) fo << '\n';                              // Last line of paragraph
    curLine[0]=0;                                           // Delete line
    inFirstLine=!expand;
}

int TextEmitter::lineLen()
{
    return LINELEN-(inFirstLine?firstIndent:normalIndent);
}
