/* File: borg1.h */

/* Purpose: Header file for "borg1.c" -BEN- */

#ifndef INCLUDED_BORG1_H
#define INCLUDED_BORG1_H

#include "angband.h"



#ifdef ALLOW_BORG


/*
 * This file provides support for "borg1.c".
 */



/*** Some constants ***/


/*
 * Maximum possible dungeon size
 */
#define AUTO_MAX_X  MAX_WID
#define AUTO_MAX_Y  MAX_HGT



/*
 * Flags for the "info" field of grids
 *
 * Note that some of the flags below are not "perfect", in particular,
 * several of the flags should be treated as "best guesses", see below.
 *
 * The "BORG_MARK" flag means that the grid has been "observed", though
 * the terrain feature may or may not be memorized.  Note the use of the
 * "FEAT_NONE", "FEAT_FLOOR", and "FEAT_INVIS" feature codes below.
 *
 * The "BORG_GLOW" flag means that a grid is probably "perma-lit", but
 * sometimes it is really only "recently" perma-lit, but was then made
 * dark with a darkness attack, and it is now torch-lit or off-screen.
 *
 * The "BORG_DARK" flag means that a grid is probably not "perma-lit",
 * but sometimes it is really only "recently" dark, but was then made
 * "lite" with a "call lite" spell, and it is now torch-lit or off-screen.
 *
 * The "BORG_LITE" flag means that a grid is probably lit by the player
 * torch, but this may not be true if the nearby "BORG_VIEW" flags are
 * not correct, or if the "lite radius" has changed recently.
 *
 * The "BORG_VIEW" flag means that a grid is probably in line of sight
 * of the player, but this may not be true if some of the grids between
 * the player and the grid contain previously unseen walls/doors/etc.
 *
 * The "BORG_TEMP" flag means that a grid has been added to the array
 * "auto_temp_x"/"auto_temp_y", though normally we ignore this flag.
 *
 * The "BORG_XTRA" flag is used for various "extra" purposes, primarily
 * to assist with the "update_view()" code.
 */
#define BORG_MARK   0x01    /* observed grid */
#define BORG_GLOW   0x02    /* probably perma-lit */
#define BORG_DARK   0x04    /* probably not perma-lit */
#define BORG_OKAY   0x08    /* on the current panel */
#define BORG_LITE   0x10    /* lit by the torch */
#define BORG_VIEW   0x20    /* in line of sight */
#define BORG_TEMP   0x40    /* temporary flag */
#define BORG_XTRA   0x80    /* extra flag */


/*
 * Maximum size of the "lite" array
 */
#define AUTO_LITE_MAX 1536


/*
 * Maximum size of the "view" array
 */
#define AUTO_VIEW_MAX 1536


/*
 * Number of grids in the "temp" array
 */
#define AUTO_TEMP_MAX 1536


/*
 * Number of grids in the "flow" array
 */
#define AUTO_FLOW_MAX 1536



/*
 * Enable the "borg_note()" usage of the Recall Window
 * Also specify the number of "rolling rows" to use
 */
#define BORG_NOTE_ROWS      12



/*
 * Size of Keypress buffer
 */
#define KEY_SIZE 8192




/*
 * Object information
 */

typedef struct auto_take auto_take;

struct auto_take
{
    s16b    k_idx;      /* Kind index */

    bool    known;      /* Verified kind */

    bool    seen;       /* Assigned motion */

    bool    extra;      /* Unused */

    byte    x, y;       /* Location */

    s16b    when;       /* When last seen */
};


/*
 * Monster information
 */

typedef struct auto_kill auto_kill;

struct auto_kill
{
    s16b    r_idx;      /* Race index */

    bool    known;      /* Verified race */
    bool    awake;      /* Probably awake */

    bool    seen;       /* Assigned motion */
    bool    used;       /* Assigned message */

    byte    x, y;       /* Location */

    byte    ox, oy;     /* Old location */

    byte    speed;      /* Estimated speed */
    byte    moves;      /* Estimates moves */

    s16b    power;      /* Estimated hit-points */
    s16b    other;      /* Estimated something */

    s16b    when;       /* When last seen */
};



/*
 * OPTION: Use the "room" support
 *
 * Note -- this option causes slight bugs in "borg-ext.c" that
 * I have not yet tracked down.  You have been warned...
 */
/* #define BORG_ROOMS */


#ifdef BORG_ROOMS


/*
 * OPTION: Allow rooms to intersect.  This is "pretty" but inefficient.
 */
/* #define CROSS_ROOMS */


/*
 * Maximum number of rooms.  This may be too small.
 * But if AUTO_ROOMS * sizeof(auto_room) > 64K then some
 * machines may not be able to allocate the room array.
 */
#define AUTO_ROOMS  (AUTO_MAX_X * AUTO_MAX_Y / 8)



/*
 * Forward declare
 */
typedef struct auto_room auto_room;


/*
 * A room in the dungeon.  20 bytes.
 *
 * The "rooms" are designed to (1) optimize the storage of information
 * about the map and (2) simplify navigation.  Thus, we create rooms by
 * grouping relevant rectangles of grids together.  This includes all
 * "normal" rooms in the dungeon, most "corridors", occasional "thick"
 * corridors, all left-over floor grids, and many regions in the town.
 *
 * A room index of "zero" means that the grid is not in any room yet
 * A room index N such that 0<N<MAX_ROOMS refers to a single room
 * A room index MAX_ROOMS+N means the grid is inside "N" different rooms
 *
 * A few options in the code itself determine if two rooms can overlap.
 * Likewise, the code handles issues like one room absorbing another,
 * and destroying any rooms that are suddenly found to include a grid
 * which is actually a wall.  This allows the Borg to recover from
 * earthquakes, and also lets it assume that all monsters and objects
 * are actually on floors, since if they move, revealing a wall, the
 * Borg will simply take back all previous assumptions.
 *
 * The rooms are maintained in a "room array" of a size AUTO_ROOMS.
 * The first and last rooms in this array are used to maintain two
 * in-place lists, a stack of "free" rooms, and a priority queue
 * of rooms used to propagate the current "flow".
 *
 * The "self" field gives the index of a room.
 *
 * The "free" field is either zero, meaning that the room is being used,
 * or it holds the index of the next free room.  Note that the first
 * room always points to the next free room, and the last room always
 * points to itself (indicating we have run out of rooms).
 *
 * The "when" field allows us to make decisions based on the actual
 * "time" at which a room was visited.  Currently this field is used
 * only by the "revisit" function to choose "interesting" rooms.  We
 * can probably delete this field and still function.
 *
 * The "location" of the corners of the room are encoded in x1/y1/x2/y2.
 */
struct auto_room
{
    s16b self;      /* Self index */

    s16b free;      /* Index of next room in free room list */

    s16b when;      /* When last visited */

    byte x1, y1;    /* Upper left corner */
    byte x2, y2;    /* Bottom right corner */
};


#endif



/*
 * Forward declare
 */
typedef struct auto_grid auto_grid;

/*
 * A grid in the dungeon.  Several bytes.
 *
 * There is a set of eight bit flags (see below) called "info".
 *
 * There is a terrain feature type, which may be incorrect.  It is
 * more or less based on the standard "feature" values, but some of
 * the legal values are never used, such as "secret door", and others
 * are used in bizarre ways, such as "invisible trap".
 *
 * There is an object index into the "object tracking" array.
 *
 * There is a monster index into the "monster tracking" array.
 *
 * There is a byte "hmmm" which is currently unused.
 *
 * There is a byte "xtra" which tracks how much "searching" has been done
 * in the grid or in any grid next to the grid.
 *
 * To perform "navigation" from one place to another, the "flow" routines
 * are used, which place "cost" information into the "cost" fields.  Then,
 * if the path is clear, the "cost" information is copied into the "flow"
 * fields, which are used for the actual navigation.  This allows multiple
 * routines to check for "possible" flowing, without hurting the current
 * flow, which may have taken a long time to construct.  We also assume
 * that the Borg never needs to follow a path longer than 250 grids long.
 * Note that the "cost" fields have been moved into external arrays.
 *
 * Hack -- note that the "char" zero will often crash the system!
 */
struct auto_grid
{
    byte feat;      /* Grid type */
    byte info;      /* Grid flags */

    byte take;      /* Object index */
    byte kill;      /* Monster index */

#ifdef BORG_ROOMS
    s16b room;      /* Room index */
#endif

    byte hmmm;      /* Extra field (unused) */

    byte xtra;      /* Extra field (search count) */
};


/*
 * Forward declare
 */
typedef struct auto_data auto_data;

/*
 * Hack -- one byte of info per grid
 *
 * We use a structure to encapsulate the data into a "typed" form.
 */
struct auto_data
{
    byte data[AUTO_MAX_Y][AUTO_MAX_X];
};




/*** Some macros ***/


/*
 * Determine "twice" the distance between two points
 * This results in "diagonals" being "correctly" ranged,
 * that is, a diagonal appears "furthur" than an adjacent.
 */
#define double_distance(Y1,X1,Y2,X2) \
    (distance(((int)(Y1))<<1,((int)(X1))<<1,((int)(Y2))<<1,((int)(X2))<<1))



/*** Some variables ***/


/*
 * Some variables
 */

extern bool auto_active;        /* Actually active */

extern bool auto_cancel;        /* Being cancelled */



/* 
 * Hack -- optional cheating flags
 */

extern bool auto_cheat_equip;       /* Cheat for "equip mode" */

extern bool auto_cheat_inven;       /* Cheat for "inven mode" */

extern bool auto_cheat_spell;       /* Cheat for "browse mode" */

extern bool auto_cheat_panel;       /* Cheat for "panel mode" */

extern bool auto_do_star_id;


/*
 * Various silly flags
 */

extern bool auto_flag_save;     /* Save savefile at each level */

extern bool auto_flag_dump;     /* Save savefile at each death */

extern bool borg_save; /* do a save next time we get to press a key! */


/*
 * Use a simple internal random number generator
 */

extern bool auto_rand_quick;        /* Save system setting */

extern u32b auto_rand_value;        /* Save system setting */

extern u32b auto_rand_local;        /* Save personal setting */


/*
 * Hack -- time variables
 */

extern s16b c_t;        /* Current "time" */


/*
 * Hack -- Other time variables
 */

extern s16b when_call_lite; /* When we last did call light */
extern s16b when_wizard_lite;   /* When we last did wizard light */

extern s16b when_detect_traps;  /* When we last detected traps */
extern s16b when_detect_doors;  /* When we last detected doors */
extern s16b when_detect_walls;  /* When we last detected walls */


/*
 * Some information
 */

extern s16b goal;       /* Flowing (goal type) */

extern bool goal_rising;    /* Currently returning to town */

extern bool goal_leaving;   /* Currently leaving the level */

extern bool goal_fleeing;   /* Currently fleeing the level */

extern bool goal_ignoring;  /* Currently ignoring monsters */

extern bool goal_recalling; /* Currently waiting for recall */

extern s16b borg_times_twitch; /* how often twitchy on this level */

extern bool stair_less;     /* Use the next "up" staircase */
extern bool stair_more;     /* Use the next "down" staircase */

extern s32b auto_began;     /* When this level began */
extern s32b auto_time_town; /* how long it has been since I was in town */

extern s16b avoidance;      /* Current danger thresh-hold */

extern bool auto_failure;   /* Notice failure */

extern bool auto_simulate;  /* Simulation flag */
extern bool borg_attacking; /* Are we attacking a monster? */

extern bool auto_completed; /* Completed the level */

/* defence flags */
extern bool borg_prot_from_evil;
extern bool borg_speed;
extern bool borg_bless;
extern bool borg_temp_fire;
extern bool borg_temp_cold;
extern bool borg_temp_acid;
extern bool borg_temp_poison;
extern bool borg_temp_elec;
extern s16b borg_goi;
extern bool borg_shield;

/*
 * Shop goals
 */

extern s16b goal_shop;      /* Next shop to visit */
extern s16b goal_ware;      /* Next item to buy there */
extern s16b goal_item;      /* Next item to sell there */


/*
 * Other variables
 */

extern int w_x;         /* Current panel offset (X) */
extern int w_y;         /* Current panel offset (Y) */

extern int c_x;         /* Current location (X) */
extern int c_y;         /* Current location (Y) */

extern int g_x;         /* Goal location (X) */
extern int g_y;         /* Goal location (Y) */

extern int bad_obj_x[10];   /* Dropped cursed artifact at location (X) */
extern int bad_obj_y[10];   /* Dropped cursed artifact at location (Y) */


/*
 * State variables extracted from the screen
 */

extern bool do_weak;        /* Currently Weak */
extern bool do_hungry;      /* Currently Hungry */

extern bool do_full;        /* Currently Full */
extern bool do_gorged;      /* Currently Gorged */

extern bool do_blind;       /* Currently Blind */
extern bool do_afraid;      /* Currently Afraid */
extern bool do_confused;    /* Currently Confused */
extern bool do_poisoned;    /* Currently Poisoned */

extern bool do_cut;         /* Currently bleeding */
extern bool do_stun;        /* Currently stunned */
extern bool do_heavy_stun;      /* Currently stunned */

extern bool do_image;       /* May be hallucinating */
extern bool do_study;       /* May learn spells */

extern bool do_fix_lev;     /* Drained LEV */
extern bool do_fix_exp;     /* Drained EXP */

extern bool do_fix_stat[6]; /* Drained Stats */



/*
 * Some estimated state variables
 */

extern s16b my_stat_max[6]; /* Current "maximal" stat values    */
extern s16b my_stat_cur[6]; /* Current "natural" stat values    */
extern s16b my_stat_use[6]; /* Current "resulting" stat values  */
extern s16b my_stat_ind[6]; /* Current "additions" to stat values   */
extern bool my_need_stat_check[6];  /* do I need to check my stats */

extern s16b my_stat_add[6];  /* aditions to stats */

extern s16b home_stat_add[6];

extern s16b my_ac;      /* Base armor       */
extern s16b my_to_ac;       /* Plusses to ac    */
extern s16b my_to_hit;      /* Plusses to hit   */
extern s16b my_to_dam;      /* Plusses to dam   */

extern byte my_immune_acid; /* Immunity to acid     */
extern byte my_immune_elec; /* Immunity to lightning    */
extern byte my_immune_fire; /* Immunity to fire     */
extern byte my_immune_cold; /* Immunity to cold     */

extern byte my_resist_acid; /* Resist acid      */
extern byte my_resist_elec; /* Resist lightning */
extern byte my_resist_fire; /* Resist fire      */
extern byte my_resist_cold; /* Resist cold      */
extern byte my_resist_pois; /* Resist poison    */

extern byte my_resist_conf; /* Resist confusion */
extern byte my_resist_sound;    /* Resist sound     */
extern byte my_resist_lite; /* Resist light     */
extern byte my_resist_dark; /* Resist darkness  */
extern byte my_resist_chaos;    /* Resist chaos     */
extern byte my_resist_disen;    /* Resist disenchant    */
extern byte my_resist_shard;    /* Resist shards    */
extern byte my_resist_nexus;    /* Resist nexus     */
extern byte my_resist_blind;    /* Resist blindness */
extern byte my_resist_neth; /* Resist nether    */
extern byte my_resist_fear; /* Resist fear      */

extern byte my_sustain_str; /* Keep strength    */
extern byte my_sustain_int; /* Keep intelligence    */
extern byte my_sustain_wis; /* Keep wisdom      */
extern byte my_sustain_dex; /* Keep dexterity   */
extern byte my_sustain_con; /* Keep constitution    */
extern byte my_sustain_chr; /* Keep charisma    */

extern byte my_aggravate;   /* Aggravate monsters   */
extern byte my_teleport;    /* Random teleporting   */

extern byte my_ffall;       /* No damage falling    */
extern byte my_lite;        /* Permanent light  */
extern byte my_free_act;    /* Never paralyzed  */
extern byte my_see_inv;     /* Can see invisible    */
extern byte my_regenerate;  /* Regenerate hit pts   */
extern byte my_hold_life;   /* Resist life draining */
extern byte my_telepathy;   /* Telepathy        */
extern byte my_slow_digest; /* Slower digestion */

/* various slays */
extern byte my_slay_animal;
extern byte my_slay_evil;
extern byte my_slay_undead;
extern byte my_slay_demon;
extern byte my_slay_orc;
extern byte my_slay_troll;
extern byte my_slay_giant;
extern byte my_slay_dragon;
extern byte my_kill_dragon;
extern byte my_impact;
extern byte my_brand_acid;
extern byte my_brand_elec;
extern byte my_brand_fire;
extern byte my_brand_cold;

extern s16b my_see_infra;   /* Infravision range    */

extern s16b my_skill_dis;   /* Skill: Disarming     */
extern s16b my_skill_dev;   /* Skill: Magic Devices     */
extern s16b my_skill_sav;   /* Skill: Saving throw      */
extern s16b my_skill_stl;   /* Skill: Stealth factor    */
extern s16b my_skill_srh;   /* Skill: Searching ability */
extern s16b my_skill_fos;   /* Skill: Searching frequency   */
extern s16b my_skill_thn;   /* Skill: Combat (normal)   */
extern s16b my_skill_thb;   /* Skill: Combat (shooting) */
extern s16b my_skill_tht;   /* Skill: Combat (throwing) */
extern s16b my_skill_dig;   /* Skill: Digging       */

extern s16b my_num_blow;    /* Number of blows  */
extern s16b my_num_fire;    /* Number of shots  */

extern s16b my_speed;       /* Approximate speed    */
extern s16b my_other;       /* Approximate other    */

extern byte my_ammo_tval;   /* Ammo -- "tval"   */
extern byte my_ammo_sides;  /* Ammo -- "sides"  */
extern s16b my_ammo_power;  /* Shooting multipler   */
extern s16b my_ammo_range;  /* Shooting range   */

extern s16b my_cur_lite;    /* Radius of lite */

extern s16b my_need_enchant_to_a;   /* Need some enchantment */
extern s16b my_need_enchant_to_h;   /* Need some enchantment */
extern s16b my_need_enchant_to_d;   /* Need some enchantment */



/*
 * Hack -- basic "power"
 */

extern s32b my_power;


/*
 * Various "amounts" (for the player)
 */

extern s16b amt_fuel;
extern s16b amt_food;
extern s16b amt_ident;
extern s16b amt_star_ident;
extern s16b amt_recall;
extern s16b amt_phase;
extern s16b amt_escape;
extern s16b amt_teleport;
extern s16b amt_teleport_staff;

extern s16b amt_heal;
extern s16b amt_ez_heal;
extern s16b amt_mana;
extern s16b amt_ez_mana;
extern s16b amt_cure_critical;
extern s16b amt_cure_serious;

extern s16b amt_detect_trap;
extern s16b amt_detect_door;

extern s16b amt_missile;

extern s16b amt_book[9];

extern s16b amt_add_stat[6];
extern s16b amt_fix_stat[6];

extern s16b amt_fix_exp;

extern s16b amt_speed;

extern s16b amt_enchant_to_a;
extern s16b amt_enchant_to_d;
extern s16b amt_enchant_to_h;


/*
 * Various "amounts" (for the home)
 */

extern s16b num_food;
extern s16b num_ident;
extern s16b num_star_ident;
extern s16b num_recall;
extern s16b num_phase;
extern s16b num_escape;
extern s16b num_teleport;

extern s16b num_cure_critical;
extern s16b num_cure_serious;

extern s16b num_missile;

extern s16b num_book[9];

extern s16b num_fix_stat[6];

extern s16b num_fix_exp;
extern s16b num_mana;
extern s16b num_heal;

extern s16b num_enchant_to_a;
extern s16b num_enchant_to_d;
extern s16b num_enchant_to_h;

extern s16b num_artifact;

extern s16b home_slot_free;
extern s16b home_damage;
extern s16b num_duplicate_items;
extern s16b num_slow_digest;
extern s16b num_regenerate;
extern s16b num_telepathy;
extern s16b num_lite;
extern s16b num_see_inv;
extern s16b num_ffall;
extern s16b num_free_act;
extern s16b num_hold_life;
extern s16b num_immune_acid;
extern s16b num_immune_elec;
extern s16b num_immune_fire;
extern s16b num_immune_cold;
extern s16b num_resist_acid;
extern s16b num_resist_elec;
extern s16b num_resist_fire;
extern s16b num_resist_cold;
extern s16b num_resist_pois;
extern s16b num_resist_conf;
extern s16b num_resist_sound;
extern s16b num_resist_lite;
extern s16b num_resist_dark;
extern s16b num_resist_chaos;
extern s16b num_resist_disen;
extern s16b num_resist_shard;
extern s16b num_resist_nexus;
extern s16b num_resist_blind;
extern s16b num_resist_neth;

extern s16b num_speed;
extern s16b num_edged_weapon;
extern s16b num_bad_gloves;
extern s16b num_weapons;
extern s16b num_bow;
extern s16b num_rings;
extern s16b num_neck;
extern s16b num_armor;
extern s16b num_cloaks;
extern s16b num_shields;
extern s16b num_hats;
extern s16b num_gloves;
extern s16b num_boots;


/*
 * Hack -- extra state variables
 */

extern int auto_feeling;    /* Current level "feeling" */

extern int auto_max_level;  /* Maximum player level */
extern int auto_max_depth;  /* Maximum dungeon depth */
extern int fear_depth;  /* fear below this dungeon depth */



/*
 * Hack -- current shop index
 */

extern s16b shop_num;       /* Current shop index */



/*
 * State variables extracted from the screen
 */

extern int auto_depth;      /* Current dungeon "level" */

extern int auto_level;      /* Current level */

extern bool borg_king;      /* We won! */

extern s32b auto_exp;       /* Current experience */

extern s32b auto_gold;      /* Current gold */

extern int auto_speed;      /* Current speed */

extern int auto_ac;     /* Current armor */

extern int auto_mhp;        /* Maximum hitpoints */
extern int auto_chp;        /* Current hitpoints */

extern int auto_msp;        /* Maximum spell points */
extern int auto_csp;        /* Current spell points */

extern int auto_stat[6];    /* Current stats */

extern int auto_book[9];    /* Current book slots */


/*
 * State variables extracted from the inventory/equipment
 */

extern int auto_cur_wgt;    /* Current weight */


/*
 * Constant state variables
 */

extern int auto_race;       /* Current race */
extern int auto_class;      /* Current class */



/*
 * Constant state structures
 */

extern player_race *rb_ptr; /* Player race info */
extern player_class *cb_ptr;    /* Player class info */
extern player_magic *mb_ptr;    /* Player magic info */



/*
 * Number of turns to step for (zero means forever)
 */
extern u16b auto_step;      /* Step count (if any) */


/*
 * Status message search string
 */
extern char auto_match[128];    /* Search string */


/*
 * Log file
 */
extern FILE *auto_fff;      /* Log file */


/*
 * Hack -- single character constants
 */

extern const char p1, p2, c1, c2, b1, b2;


/*
 * Hack -- the detection arrays
 */

extern bool auto_detect_wall[6][6];

extern bool auto_detect_trap[6][6];

extern bool auto_detect_door[6][6];


/*
 * Locate the store doors
 */

extern byte *track_shop_x;
extern byte *track_shop_y;


/*
 * Track "stairs up"
 */

extern s16b track_less_num;
extern s16b track_less_size;
extern byte *track_less_x;
extern byte *track_less_y;


/*
 * Track "stairs down"
 */

extern s16b track_more_num;
extern s16b track_more_size;
extern byte *track_more_x;
extern byte *track_more_y;


/*
 * The object list.  This list is used to "track" objects.
 */

extern s16b auto_takes_cnt;

extern s16b auto_takes_nxt;

extern auto_take *auto_takes;


/*
 * The monster list.  This list is used to "track" monsters.
 */

extern s16b auto_kills_cnt;

extern s16b auto_kills_nxt;

extern auto_kill *auto_kills;


/*
 * Hack -- depth readiness
 */
extern int auto_fear_depth;
extern bool finished_level;
extern bool borg_collect_potions_morgoth;
extern int borg_ready_morgoth;

/*
 * Hack -- extra fear per "region"
 */

extern u16b auto_fear_region[6][18];


/*
 * Hack -- count racial appearances per level
 */

extern s16b *auto_race_count;


/*
 * Hack -- count racial kills (for uniques)
 */

extern s16b *auto_race_death;


/*
 * Classification of map symbols
 */

extern bool auto_is_take[256];      /* Symbol may be an object */

extern bool auto_is_kill[256];      /* Symbol may be a monster */


/*
 * Current "grid" list
 */

extern auto_grid *auto_grids[AUTO_MAX_Y];   /* Current "grid list" */


#ifdef BORG_ROOMS

/*
 * Some "room" variables
 */

extern int auto_room_max;       /* First totally free room */

extern auto_room *auto_rooms;       /* Current "room list" */

extern auto_room *auto_room_head;   /* &auto_rooms[0] */

extern auto_room *auto_room_tail;   /* &auto_rooms[AUTO_MAX_ROOMS-1] */


#endif


/*
 * Maintain a set of grids (liteable grids)
 */

extern s16b auto_lite_n;
extern byte auto_lite_y[AUTO_LITE_MAX];
extern byte auto_lite_x[AUTO_LITE_MAX];


/*
 * Maintain a set of grids (viewable grids)
 */

extern s16b auto_view_n;
extern byte auto_view_y[AUTO_VIEW_MAX];
extern byte auto_view_x[AUTO_VIEW_MAX];


/*
 * Maintain a set of grids (scanning arrays)
 */

extern s16b auto_temp_n;
extern byte auto_temp_y[AUTO_TEMP_MAX];
extern byte auto_temp_x[AUTO_TEMP_MAX];


/*
 * Maintain a set of grids (flow calculations)
 */

extern s16b auto_flow_n;
extern byte auto_flow_y[AUTO_FLOW_MAX];
extern byte auto_flow_x[AUTO_FLOW_MAX];


/*
 * Hack -- use "flow" array as a queue
 */

extern int flow_head;
extern int flow_tail;


/*
 * Some variables
 */

extern auto_data *auto_data_flow;   /* Current "flow" data */

extern auto_data *auto_data_cost;   /* Current "cost" data */

extern auto_data *auto_data_hard;   /* Constant "hard" data */

extern auto_data *auto_data_know;   /* Current "know" flags */

extern auto_data *auto_data_icky;   /* Current "icky" flags */


/*
 * Strategy flags -- recalculate things
 */

extern bool auto_danger_wipe;       /* Recalculate danger */

extern bool auto_update_view;       /* Recalculate view */

extern bool auto_update_lite;       /* Recalculate lite */


/*
 * Strategy flags -- examine the world
 */

extern bool auto_do_inven;      /* Acquire "inven" info */

extern bool auto_do_equip;      /* Acquire "equip" info */

extern bool auto_do_panel;      /* Acquire "panel" info */

extern bool auto_do_frame;      /* Acquire "frame" info */

extern bool auto_do_spell;      /* Acquire "spell" info */

extern byte auto_do_spell_aux;      /* Hack -- book for "auto_do_spell" */

extern bool auto_do_browse;     /* Acquire "store" info */

extern byte auto_do_browse_what;    /* Hack -- store for "auto_do_browse" */

extern byte auto_do_browse_more;    /* Hack -- pages for "auto_do_browse" */


/*
 * Strategy flags -- run certain functions
 */

extern bool auto_do_crush_junk;

extern bool auto_do_crush_hole;

extern bool auto_do_crush_slow;

/* am I fighting a unique */
extern bool borg_fighting_unique;



/*** Some functions ***/

/*
 * Queue a keypress
 */
extern errr borg_keypress(char k);

/*
 * Queue several keypresses
 */
extern errr borg_keypresses(cptr str);

/*
 * Dequeue a keypress
 */
extern char borg_inkey(bool take);

/*
 * Flush the keypresses
 */
extern void borg_flush(void);


/*
 * Obtain some text from the screen (single character)
 */
extern errr borg_what_char(int x, int y, byte *a, char *c);

/*
 * Obtain some text from the screen (multiple characters)
 */
extern errr borg_what_text(int x, int y, int n, byte *a, char *s);


/*
 * Log a message to a file
 */
extern void borg_info(cptr what);

/*
 * Log a message, Search it, and Show/Memorize it in pieces
 */
extern void borg_note(cptr what);


/*
 * Abort the Borg, noting the reason
 */
extern void borg_oops(cptr what);


/*
 * Take a "memory note"
 */
extern bool borg_tell(cptr what);

/*
 * Change the player name
 */
extern bool borg_change_name(cptr str);

/*
 * Dump a character description
 */
extern bool borg_dump_character(cptr str);

/*
 * Save the game (but do not quit)
 */
extern bool borg_save_game(void);


/*
 * Update the "frame" info from the screen
 */
extern void borg_update_frame(void);


/*
 * Initialize this file
 */
extern void borg_init_1(void);


#endif

#endif

