/*
**                      Softwarepraktikum iMaze
**                              1993/94
**                Joerg Czeranski    Hans-Ulrich Kiel
**
** Datei: server.c
**
** Kommentar:
**  Das Hauptprogramm des Servers
**  inklusive des Startens der anderen Prozesse
*/


#include <stdio.h>
#include <string.h>
#include <signal.h>

#include "global.h"
#include "fehler.h"
#include "speicher.h"
#include "labyrinth.h"
#include "server_netz.h"
#include "server.h"
#include "init_spieler.h"

static char sccsid[] = "@(#)server.c	1.60 24 Oct 1994";


/* Identifikationstext, mit dem jede Labyrinthdatei beginnt: */
static char MAGIC[] = "iMazeLab1\n";

u_int feldlaenge, feldbreite; /* Feldgroesse */
block **spielfeld;            /* Daten des Spielfeldes */

static void *session_deskriptor; /* Deskriptor zur Kommunikation mit der
                                    einzigen Session, die gestartet wird */


/*
** ende_handler
**  wird aufgerufen, wenn der Server vom User beendet wird;
**  schickt dem Session-Prozess die Nachricht; der Server-Prozess wird
**  erst beendet, wenn der Session-Prozess geantwortet hat
**
** Parameter:
**  signum: Dummy-Parameter
*/
static void ende_handler(signum)
int signum;
{
	char daten; /* Puffer fuer die Nachricht */

	/* Nachricht NT_SPIEL_BEENDEN an Session-Prozess senden */
	daten = NT_SPIEL_BEENDEN;
	nachricht_senden(session_deskriptor, NULL, sizeof daten, &daten);
}


/*
** server
**  loest die Initialisierung von neuen Clients aus und kommuniziert
**  mit dem Session-Prozess
**
** Seiteneffekte:
**  diese Prozedur ist eine Endlosschleife und wird nur durch Fehler
**  oder Signale per exit verlassen
*/
static void server()
{
	/* Endlosschleife, s.o. */
	for (;;)
	{
		/* auf Nachrichten von den anderen Prozessen und neue
		   Verbindungen von Clients warten */
		if (nachricht_oder_verbindung_erwarten(1,
			&session_deskriptor) == NULL)
		/* eine Verbindung steht evtl. bereit */
		{
			void *verbindung; /* Deskriptor fuer die Verbindung zum
			                     neuen Client */
			char *spruch;     /* Spruch des neuen Spielers */
			char *daten;      /* Puffer zum Uebertragen des Spruchs
			                     an den Sessions-Prozess */

			/* versuchen, die Verbindung entgegenzunehmen */
			if ((verbindung = verbindung_annehmen()) == NULL)
				/* keine Verbindung bekommen oder ein
				   Initialisierungsprozess wurde abgespalten, um die
				   Verbindung anzunehmen */
				continue;

			/* die Verbindung wurde angenommen; evtl. ist dies ein
			   abgespaltener Initialisierungsprozess */

			/* dem Client die Initialisierungsdaten senden
			   und die Verbindung vom Prolog auf das Spiel schalten */
			if (init_spieler(verbindung, feldbreite, feldlaenge,
				spielfeld, &spruch) ||
				verbindung_auf_spiel(session_deskriptor, verbindung))
			{
				/* Speicher fuer Spruch bei Fehler wieder freigeben */
				if (spruch != NULL)
					speicher_freigeben((void **)&spruch);

				/* bei einem Fehler die Verbindung abbrechen und weiter
				   warten; einen evtl. gestarteten Initialisierungsprozess
				   implizit beenden */
				verbindung_abbrechen(verbindung);
				continue;
			}

			/* Speicher fuer NT_NEUER_SPIELER und den Spruch belegen */
			speicher_belegen((void **)&daten,
				spruch == NULL ? 2 : strlen(spruch) + 2);

			/* die Nachricht beginnt mit NT_NEUER_SPIELER */
			daten[0] = NT_NEUER_SPIELER;

			/* falls Spruch bekannt, an die Nachricht anhaengen */
			if (spruch == NULL)
				daten[1] = 0;
			else
			{
				strcpy(daten + 1, spruch);

				/* Speicher fuer Spruch wieder freigeben */
				speicher_freigeben((void **)&spruch);
			}

			/* dem Session-Prozess den Verbindungsdeskriptor per Nachricht
			   uebergeben */
			nachricht_senden(session_deskriptor, verbindung,
				strlen(daten + 1) + 1, daten);

			/* Speicher fuer die Nachricht wieder freigeben */
			speicher_freigeben((void **)&daten);

			/* der Verbindungsdeskriptor wird vom Server nicht mehr
			   benoetigt, aufraeumen; einen evtl. gestarteten
			   Initialisierungsprozess implizit beenden */
			verbindung_freigeben(verbindung);
		}
		else
		/* eine Nachricht von einer Session liegt evtl. an */
		{
			void *daten;      /* Datenteil der Nachricht */
			int laenge;       /* Laenge des Datenteils */
			void *verbindung; /* Verbindungsdeskriptor, auf den die
			                     Nachricht sich bezieht */
			char typ;         /* Typ der Nachricht; siehe server.h */

			/* Nachricht (Typ und Deskriptor) entgegennehmen */
			if (nachricht_empfangen(session_deskriptor, &verbindung,
				&laenge, &daten))
				/* keine Nachricht bekommen; weiter warten */
				continue;

			/* Datenteil besteht nicht genau aus dem Typ? */
			if (laenge != 1)
			{
				/* dann Speicher wieder freigeben */
				if (laenge)
					speicher_freigeben(&daten);

				/* und Nachricht ignorieren */
				continue;
			}

			typ = *(char *)daten; /* Typ der Nachricht */

			/* Speicher wieder freigeben */
			speicher_freigeben(&daten);

			/* wenn der Session-Prozess sich beendet hat, den
			   Server-Prozess auch beenden */
			if (typ == NT_SPIEL_ENDE)
				exit(0);

			/* alle weiteren Nachrichten ignorieren und einen
			   evtl. mitgesandten Deskriptor wieder freigeben */
			if (verbindung != NULL)
				verbindung_freigeben(verbindung);
		}
	}
}


/*
** aussenwand_korrigieren
**  korrigiert eine Aussenwand des Spielfeldes, falls diese unsichtbar,
**  begehbar oder nicht schusssicher ist
**
** Parameter:
**  wand: Zeiger auf eine Wand im Spielfeld
**
** Rueckgabewert:
**  0, falls nichts veraendert wurde, 1 sonst
**
** Seiteneffekte:
**  *wand wird noetigenfalls korrigiert
*/
static int aussenwand_korrigieren(wand)
struct wand *wand;
{
	int veraendert; /* Flag, ob etwas veraendert wurde */

	veraendert = 0;

	/* die Wand muss schusssicher sein */
	if (!wand->schusssicher)
		wand->schusssicher = 1,
		veraendert = 1;

	/* die Wand muss unbegehbar sein */
	if (!wand->unbegehbar)
		wand->unbegehbar = 1,
		veraendert = 1;

	/* die Wand muss die Farbe einer Wand haben */
	if (wand->farbe < 1 || wand->farbe > 7)
		wand->farbe = 1,
		veraendert = 1;

	/* zurueckmelden, ob etwas veraendert wurde */
	return veraendert;
}


/*
** ladelab
**  laedt das Labyrinth aus einer Datei
**
** Parameter:
**  dateiname: der Dateiname
**
** Seiteneffekte:
**  spielfeld, feldbreite und feldlaenge werden initialisiert
*/
static void ladelab(dateiname)
char *dateiname;
{
	FILE *datei;                  /* Deskriptor fuer die Labyrinthdatei */
	char magic[sizeof MAGIC - 1]; /* Identifikationstext in der Datei */
	u_char daten[4];              /* Puffer zum Lesen aus der Datei */
	int x, y, i;                  /* Indizes fuer das Spielfeld */
	int warnung;                  /* Labyrinth enthaelt
	                                 unzulaessige Daten */

	/* falls der Dateiname "-" ist, von stdin lesen, sonst Datei oeffnen */
	if (strcmp(dateiname, "-"))
	{
		/* Datei zum Lesen oeffnen */
		datei = fopen(dateiname, "rb");

		/* bei Fehler, diesen anzeigen und Programm abbrechen */
		if (datei == NULL)
		{
			static char *meldung[] = { "iMaze - Initialization Error", "",
				"Can't read labyrinth file", NULL, NULL };

		lese_fehler:
			/* Fehlermeldung zusammenstellen und ausgeben */
			meldung[3] = dateiname;
			uebler_fehler(meldung, NULL);
		}
	}
	else
		/* stdin benutzen */
		datei = stdin;

	/* Identifikationstext aus der Datei lesen */
	if (fread(magic, 1, sizeof magic, datei) != sizeof magic)
		/* bei Fehler "Initialization Error" anzeigen */
		goto lese_fehler;

	/* Falls der Identifikationstext nicht der erwartete ist,
	   Fehler anzeigen und Programm abbrechen */
	if (memcmp(MAGIC, magic, sizeof magic))
	{
		static char *meldung[] = { "iMaze - Initialization Error", "",
			"Invalid labyrinth file format", NULL, NULL };

	format_fehler:
		/* Fehlermeldung zusammenstellen und ausgeben */
		meldung[3] = dateiname;
		uebler_fehler(meldung, NULL);
	}

	/* Feldgroesse aus der Datei lesen */
	if (fread(daten, 1, 2, datei) != 2)
		/* bei Fehler "Initialization Error" anzeigen */
		goto lese_fehler;
	feldbreite = daten[0];
	feldlaenge = daten[1];

	/* Feldgroesse ueberpruefen */
	if (feldbreite < 1 || feldbreite > MAXFELDBREITE ||
		feldlaenge < 1 || feldlaenge > MAXFELDBREITE)
		/* bei ungueltier Groesse "Initialization Error" anzeigen */
		goto format_fehler;

	/* Speicher fuer das Spielfeld belegen */
	speicher_belegen((void **)&spielfeld, feldbreite * sizeof(block *));

	for (x = 0; x < feldbreite; x++)
		speicher_belegen((void **)&spielfeld[x],
			feldlaenge * sizeof(block));

	/* bei unzulaessigem Spielfeldinhalt Warnung ausgeben,
	   aber weitermachen */
	warnung = 0;

	/* Spielfeld einlesen */
	for (y = 0; y < feldlaenge; y++)
		for (x = 0; x < feldbreite; x++)
		{
			/* einen Block (4 Waende) aus der Datei lesen */
			if (fread(daten, 1, 4, datei) != 4)
				/* bei Fehler "Initialization Error" anzeigen */
				goto lese_fehler;

			/* alle 4 Waende in Spielfeld kopieren */
			for (i = 0; i < 4; i++)
			{
				struct wand *wand; /* Zeiger in das Spielfeld */

				wand = &spielfeld[x][y][i];

				/* Farbe und Flags "schusssicher" und "unbegehbar" aus der
				   Datei uebernehmen */
				wand->farbe = daten[i] & 15;
				wand->schusssicher = daten[i] >> 6 & 1;
				wand->unbegehbar = daten[i] >> 7 & 1;

				/* Daten unzulaessig? */
				warnung |=
					wand->unbegehbar && wand->farbe >= 8 ||
					!wand->unbegehbar && wand->farbe >= 1 &&
					wand->farbe <= 7 ||
					wand->schusssicher ^ (wand->farbe != 0) ||
					(daten[i] >> 4 & 3) != 0;
			}
		}

	/* falls am Rand keine Waende sind, korrigieren */
	for (x = 0; x < feldbreite; x++)
	{
		warnung |= aussenwand_korrigieren(&spielfeld[x][0][NORD]);
		warnung |= aussenwand_korrigieren(&spielfeld[x][feldlaenge - 1][SUED]);
	}
	for (y = 0; y < feldlaenge; y++)
	{
		warnung |= aussenwand_korrigieren(&spielfeld[0][y][WEST]);
		warnung |= aussenwand_korrigieren(&spielfeld[feldbreite - 1][y][OST]);
	}

	/* Datei schliessen, wenn es nicht stdin war */
	if (datei != stdin)
		fclose(datei);

	/* evtl. Warnung ausgeben */
	if (warnung)
	{
		fprintf(stderr, "iMazesrv: Invalid labyrinth\n");
		fprintf(stderr, "iMazesrv: %s\n", dateiname);
		fprintf(stderr, "iMazesrv:\n");
		fprintf(stderr, "iMazesrv: Who cares?\n");
	}
}


/* bis hier lokaler Teil                       */
/***********************************************/
/* ab hier globaler Teil                       */


/*
** uebler_fehler
**  zeigt eine Fehlermeldung an und beendet das Programm
**
** Parameter:
**  meldung: Feld von Strings, die die Zeilen des auszugebenden Textes
**           beinhalten
**  knopf: Text der auf einem Bestaetigungs-Knopf stehen sollte;
**         wird ignoriert
**
** Seiteneffekte:
**  Ende des Programms
*/
void uebler_fehler(meldung, knopf)
char **meldung;
char *knopf;
{
	int i; /* Zaehlvariable fuer Ausgeben der Meldung */

	/* Ausgabe der Meldung auf standard-error */
	for (i = 0; meldung[i] != NULL; i++)
		fprintf(stderr, "iMazesrv: %s\n", meldung[i]);

	/* Programm abbrechen */
	exit(1);
}


/*
** main
**  die Hauptroutine, der Server und eine Session werden gestartet
**
** Parameter:
**  argc: Anzahl der Argumente inklusive Programmname
**  argv: Argumentliste
*/
void main(argc, argv)
int argc;
char **argv;
{
	/* diese Signale immer ignorieren */
	signal(SIGHUP, SIG_IGN);
	signal(SIGALRM, SIG_IGN);

	/* fuer diese Signale gibt es fuer einen Prozess einen Handler,
	   erstmal ignorieren */
	signal(SIGINT, SIG_IGN);
	signal(SIGTERM, SIG_IGN);

	/* Netzwerkroutinen initialisieren */
	netzwerk_init(&argc, argv);

	/* falls keine Labyrinthdatei angegeben wurde, Usage ausgeben */
	if (argc != 2)
	{
		fprintf(stderr, "usage: %s [network-options] lab-file\n", argv[0]);
		exit(1);
	}

	/* Labyrinth laden */
	ladelab(argv[1]);

	/* eine Session starten */
	if ((session_deskriptor = session_starten()) != NULL)
	{
		/* dies ist der Server-Prozess */

		/* fuer diese Signale einen Handler installieren, der ein
		   sauberes Programmende erlaubt */
		signal(SIGINT, ende_handler);
		signal(SIGTERM, ende_handler);

		server();
	}
	else
		/* dies ist der Session-Prozess */
		session();
}
