/* wiz.c */

#include "copyright.h"

/* Wizard-only commands */

#include <string.h>
#include <math.h>
#include <sys/time.h>
#include "config.h"
#include "db.h"
#include "interface.h"
#include "match.h"
#include "externs.h"

#ifdef MEM_CHECK
#include "mem_check.h"
#endif

#define  ANY_OWNER	-2


extern dbref find_player_by_desc(); /* from bsd.c, used in do_boot */
extern void boot_desc();

#ifdef USE_MAILER
extern void clear_mail();
#endif

#ifdef WCREAT
void do_pcreate(creator, player_name, player_password)
     dbref creator;
     const char *player_name;
     const char *player_password;
{
    dbref player;

    if (!Create_Player(creator)) {
	notify(creator, "You do not have the power over body and mind!");
	return;
    }

    player = create_player(player_name, player_password, "None");
    if (player == NOTHING) {
	notify(creator, tprintf("failure creating '%s'", player_name));
	return;
    }
    notify(creator, tprintf("New player '%s' created with password '%s'",
			    player_name, player_password));
    do_log(LT_WIZ, creator, player, "Player creation");
}
#endif

#ifdef QUOTA
void do_quota(player, arg1, arg2, set_q)
     dbref player;
     char *arg1, *arg2;
     int set_q;			/* 0 = @quota, 1 = @squota */
{
    
    dbref who, thing;
    int owned, limit;
    
    /* determine the victim */
    if (!arg1 || !*arg1 || !strcmp(arg1, "me"))
	who = player;
    else {
	who = lookup_player(arg1);
	if (who == NOTHING) {
	    notify(player, "No such player.");
	    return;
	}
    }
    
    /* check permissions */
    if (!Wizard(player) && (*arg2 != '\0')) {
	notify(player, "Only wizards may change a quota.");
	return;
    }
    if (!Do_Quotas(player) && (player != who)) {
	notify(player, "You can't look at someone else's quota.");
	return;
    }
    
    /* count up all owned objects */
    owned = -1;			/* a player is never included in his own
				 * quota */
    for (thing = 0; thing < db_top; thing++) {
	if (db[thing].owner == who)
#ifdef DESTROY
	    if ((Flags(thing) & (TYPE_THING | GOING)) != (TYPE_THING | GOING))
#endif
		++owned;
    }
    
    /* the quotas of priv'ed players are unlimited and cannot be set. */
    if (NoPay(who)) {
	notify(player, tprintf("Objects: %d   Limit: UNLIMITED", owned));
	return;
    }
    
    /* if we're not doing a change, determine the mortal's quota limit. 
     * RQUOTA is the objects _left_, not the quota itself.
     */
    
    if (!set_q) {
	limit = get_current_quota(who);
	notify(player, tprintf("Objects: %d   Limit: %d",
			       owned, owned + limit));
	return;
    } 
    
    /* set a new quota */
    if (!arg2 || !*arg2) {
	notify(player, "What do you want to set the quota to?");
	return;
    }
    
    limit = atoi(arg2);
    if (limit < owned)		/* always have enough quota for your objects */
	limit = owned;
    
    atr_add(Owner(who), "RQUOTA", tprintf("%d", limit - owned), GOD, NOTHING);
    
    notify(player, tprintf("Objects: %d   Limit: %d", owned, limit));
}


void do_allquota(player, arg1)
     dbref player;
     char *arg1;
{
    int limit, owned;
    dbref who, thing;
    
    if (!God(player)) {
	notify(player, "Who do you think you are, GOD?");
	return;
    }
    
    if (!arg1 || !*arg1) {
	notify(player, "What do you want to set all quotas to?");
	return;
    }
    
    limit = atoi(arg1);
    
    for (who = 0; who < db_top; who++) {
	if (Typeof(who) != TYPE_PLAYER)
	    continue;
	
	/* count up all owned objects */
	owned = -1;			/* a player is never included in his own
					 * quota */
	for (thing = 0; thing < db_top; thing++) {
	    if (db[thing].owner == who)
#ifdef DESTROY
		if ((Flags(thing) & (TYPE_THING | GOING)) != (TYPE_THING | GOING))
#endif
		    ++owned;
	}
	
	if (limit <= owned)
	    atr_add(who, "RQUOTA", "0", GOD, NOTHING);
	else
	    atr_add(who, "RQUOTA", tprintf("%d", limit - owned), GOD, NOTHING);
    }
}
#endif				/* QUOTA */

int tport_dest_ok(player, victim, dest)
     dbref player;
     dbref victim;
     dbref dest;
{
    /* can player legitimately send something to dest */
    
    if (Tel_Anywhere(player))
	return 1;
    
    if (controls(player, dest))
	return 1;
    
    /* beyond this point, if you don't control it and it's not a room, no hope */
    if (Typeof(dest) != TYPE_ROOM)
	return 0;
    
    /* Check for a teleport lock. It fails if the player is not wiz or
     * royalty, and the room is tport-locked against the victim, and the
     * victim does not control the room.
     */
    if (!eval_boolexp(victim, db[dest].enterkey, dest, 0, ENTERLOCK))
	return 0;
    
    if (Toggles(dest) & ROOM_JUMP_OK)
	return 1;
    
    return 0;
}

int tport_control_ok(player, victim, loc)
     dbref player;
     dbref victim;
     dbref loc;
{
    /* can player legitimately move victim from loc */
    
    if (God(victim) && !God(player))
	return 0;
    
    if (Tel_Anything(player))
	return 1;
    
    if (controls(player, victim))
	return 1;
    
    /* mortals can't @tel priv'ed players just on basis of location ownership */
    
    if (controls(player, loc) && (!Hasprivs(victim) || Owns(player, victim)))
	return 1;
    
    return 0;
}

void do_teleport(player, arg1, arg2)
     dbref player;
     const char *arg1;
     const char *arg2;
{
    dbref victim;
    dbref destination;
    dbref loc;
    const char *to;
    dbref absroom;       /* "absolute room", for NO_TEL check */
    int rec = 0;         /* recursion counter */
    
    /* get victim, destination */
    if (*arg2 == '\0') {
	victim = player;
	to = arg1;
    } else {
	init_match(player, arg1, NOTYPE);
	match_neighbor();
	match_possession();
	match_me();
	match_absolute();
	match_player();
	
	if ((victim = noisy_match_result()) == NOTHING) {
	    return;
	}
	to = arg2;
    }
    
    /* get destination */
    
    if (!strcasecmp(to, "home")) {
	destination = db[victim].exits;
    } else {
	init_match(player, to, TYPE_PLAYER);
	match_everything();
	destination = match_result();
    }
    
    switch (destination) {
      case NOTHING:
	notify(player, "No match.");
	break;
      case AMBIGUOUS:
	notify(player, "I don't know which destination you mean!");
	break;
      default:
	/* check victim, destination types, teleport if ok */
	if(recursive_member(destination, victim, 0) || (victim == destination)) {
	    notify(player, "Bad destination.");
	    return;
	}
	
	if (!Tel_Anywhere(player) &&
	    (Typeof(victim) == TYPE_EXIT ||
	     Typeof(victim) == TYPE_ROOM ||
	     ((Typeof(victim) == TYPE_PLAYER) &&
	      (Typeof(destination) == TYPE_PLAYER)))) {
	    notify(player, "Bad destination.");
	    return;
	}
	
	loc = Location(victim);
	
	/* if royal or wiz and destination is player, tel to location */
	if ((Typeof(destination) == TYPE_PLAYER) &&
	    (Tel_Anywhere(player)) && (Typeof(victim) == TYPE_PLAYER) &&
	    (destination != HOME)) {
	    if (loc != Location(destination))
		did_it(victim, victim, NULL, NULL, "OXTPORT", NULL, NULL, loc);
	    safe_tel(victim, Location(destination));
	    if (loc != Location(destination))
		did_it(victim, victim, "TPORT", NULL, "OTPORT", NULL, "ATPORT",
		       Location(destination));
	    return;
	}
	
	/* check needed for NOTHING. Especially important for unlinked exits */
	if ((absroom = db[victim].location) == NOTHING) {
	    if (Typeof(victim) != TYPE_EXIT) {
		notify(victim, "You're in the Void. This is not a good thing.");
		do_move(victim, "home", 0);
		return;
	    }
	    /* if the thing is an unlinked exit, no need for NO_TEL check */
	}
	else {
	    /* valid location, perform other checks */
	    
	    /* if player is inside himself, send him home */
	    if (absroom == victim) {
		notify(player, "What are you doing inside of yourself?");
		do_move(victim, "home", 0);
		return;
	    }
	    
	    /* find the "absolute" room */
	    while (GoodObject(absroom) && (Typeof(absroom) != TYPE_ROOM)
		   && (rec < 15)) {
		absroom = db[absroom].location;
		rec++;
	    }
	    
	    /* if there are a lot of containers, send him home */
	    if (rec > 15) {
		notify(victim, "You're in too many containers.");
		do_move(victim, "home", 0);
		return;
	    }
	    
	    /* note that we check the NO_TEL status of the victim rather
	     * than the player that issued the command. This prevents someone
	     * in a NO_TEL room from having one of his objects @tel him out.
	     * The control check, however, is detemined by command-giving
	     * player. */
	    
	    /* now check to see if the absolute room is set NO_TEL */
	    if (IS(absroom, TYPE_ROOM, ROOM_NO_TEL)
		&& !controls(player, absroom)
		&& !Tel_Anywhere(player)) {
		notify(player, "Teleports are not allowed in this room.");
		return;
	    }
	}
	
	if (Typeof(destination) != TYPE_EXIT) {
	    if (tport_control_ok(player, victim, getloc(victim)) &&
		tport_dest_ok(player, victim, destination)) {
		if (loc != destination)
		    did_it(victim, victim, NULL, NULL, "OXTPORT", NULL, NULL, loc);
		safe_tel(victim, destination);
		if (loc != destination)
		    did_it(victim, victim, "TPORT", NULL, "OTPORT", NULL, "ATPORT",
			   destination);
		if ((victim != player) &&
		    !((Typeof(victim) == TYPE_THING) &&
		      (db[victim].flags & THING_PUPPET) &&
		      (db[victim].owner == db[player].owner)))
		    notify(player, "Teleported.");
		return;
	    }
	    /* we can't do it */
	    did_it(player, destination, "EFAIL", "Permission denied.", 
		   "OEFAIL", NULL, "AEFAIL", getloc(player));
	    return;
	} else {
	    /* attempted teleport to an exit */
	    if (controls(player, victim) ||
		controls(player, db[victim].location)) {
		do_move(victim, to, 0);
	    } else {
		notify(victim,
		       tprintf("%s tries to impose his will on you and fails.",
			       db[player].name));
	    }
	}
    }
}

void do_force(player, what, command)
     dbref player;
     const char *what;
     char *command;
{
    dbref victim;
    
    if ((victim = match_controlled(player, what)) == NOTHING) {
	notify(player, "Sorry.");
	return;
    }
    if ((Typeof(victim) == TYPE_ROOM) || (Typeof(victim) == TYPE_EXIT)) {
	notify(player, "You can only force players and things.");
	return;
    }
    if (Wizard(player) && options.log_forces) {
	if (db[victim].owner != db[player].owner)
	    do_log(LT_WIZ, player, victim, "** FORCE: %s", command);
	else
	    do_log(LT_WIZ, player, victim, "FORCE: %s", command);
    }
    if (God(victim) && !God(player)) {
	notify(player, "You can't force God!");
	return;
    }
#ifdef INHERIT_FLAG
    if (!Inherit(player) && Inherit(victim)) {
	notify(player, "Authorization failed.");
	return;
    }
#endif
    /* force victim to do command */
    parse_que(victim, command, player);
}

int force_by_number(player, command)
     dbref player;
     char *command;
{
    /* the command handler has given us something of the format
     * <#dbref> <command>. Split it up and pass it to @force handler.
     * We can hack it up in-place, since we won't need it the input again.
     */
    
    char *s;
    
    for (s = command; *s && !isspace(*s); s++)
	;
    if (!*s)
	return (0);
    *s++ = '\0';
    
    do_force(player, command, s);
    return (1);
}

void do_stats(player, name)
     dbref player;
     const char *name;
{
    dbref garbage;
    dbref rooms;
    dbref exits;
    dbref things;
    dbref players;
    dbref total;
    dbref i;
    dbref owner;
    
    if (*name == '\0')
	owner = ANY_OWNER;
    else if (*name == '#') {
	owner = atoi(&name[1]);
	if (owner < 0 || db_top <= owner)
	    owner = NOTHING;
	else if (Typeof(owner) != TYPE_PLAYER)
	    owner = NOTHING;
    } else if (strcmp(name, "me") == 0)
	owner = player;
    else
	owner = lookup_player(name);
    if (owner == NOTHING) {
	notify(player, tprintf("%s: No such player.", name));
	return;
    }
    if (!Search_All(player))
	if (owner != ANY_OWNER && owner != player) {
	    notify(player, "You need a search warrant to do that!");
	    return;
	}
    total = rooms = exits = things = players = garbage = 0;
    for (i = 0; i < db_top; i++) {
	if (owner == ANY_OWNER || owner == db[i].owner) {
	    if (db[i].flags & GOING) {
		garbage++;
	    } else {
		total++;
		switch (Typeof(i)) {
		  case TYPE_ROOM:
		    rooms++;
		    break;
		  case TYPE_EXIT:
		    exits++;
		    break;
		  case TYPE_THING:
		    things++;
		    break;
		  case TYPE_PLAYER:
		    players++;
		    break;
		  default:
		    break;
		}
	    }
	}
    }
    if (owner == ANY_OWNER) {
	total = total + garbage;
	notify(player, tprintf("%d objects = %d rooms, %d exits, %d things, %d players, %d garbage.",
			       total, rooms, exits, things, players, garbage));
    } else {
	notify(player, tprintf("%d objects = %d rooms, %d exits, %d things, %d players.",
			       total, rooms, exits, things, players));
    }
#ifdef MEM_CHECK
    if(God(player))
	list_mem_check(player);
#endif
#ifdef SLOW_STATISTICS
    if(God(player))
	dump_malloc_data(player);
#endif
#if defined(DEBUG) && defined(USE_SMALLOC)
    if (Hasprivs(player))
	notify(player, tprintf("%d bytes memory used.", memused()));
    if (God(player))
	verify_sfltable(player);
#endif
}

XFUNCTION(fun_lstats)
{
    int total, rooms, exits, things, players, garbage;
    int i;
    dbref owner;
    
    if (!strcasecmp(args[0], "all"))
	owner = ANY_OWNER;
    else if (!strcasecmp(args[0], "me"))
	owner = privs;
    else
	owner = lookup_player(args[0]);
    
    /* no name matched, try to match by number */
    if (owner == NOTHING) {
	init_match(privs, args[0], TYPE_PLAYER);
	match_absolute();
	owner = match_result();
    }
    
    if (owner == NOTHING) {
	strcpy(buff, "#-1 NO SUCH PLAYER");
	return;
    }
    
    if (!Search_All(privs))
	if (owner != ANY_OWNER && owner != privs) {
	    strcpy(buff, "#-1 PERMISSION DENIED");
	    return;
	}
    
    total = rooms = exits = things = players = garbage = 0;
    for (i = 0; i < db_top; i++) {
	if (owner == ANY_OWNER || owner == db[i].owner) {
	    if (db[i].flags & GOING) {
		garbage++;
	    } else {
		total++;
		switch (Typeof(i)) {
		  case TYPE_ROOM:
		    rooms++;
		    break;
		  case TYPE_EXIT:
		    exits++;
		    break;
		  case TYPE_THING:
		    things++;
		    break;
		  case TYPE_PLAYER:
		    players++;
		    break;
		  default:
		    break;
		}
	    }				
	}				
    }
    
    if (owner == ANY_OWNER)
	total = total + garbage;
    /* note: God owns garbage */
    strcpy(buff, tprintf("%d %d %d %d %d %d", 
			 total, rooms, exits, things, players, garbage));
}

void do_toad(player, name)
     dbref player;
     const char *name;
{
    dbref victim;
    char tbuf1[BUFFER_LEN];
    ATTR *atemp;
    char alias[PLAYER_NAME_LIMIT];
    
    if (!Wizard(player)) {
	notify(player, "Only a Wizard can toad a person.");
	return;
    }
    init_match(player, name, TYPE_PLAYER);
    match_neighbor();
    match_absolute();
    match_player();
    if ((victim = noisy_match_result()) == NOTHING)
	return;
    
    if (Typeof(victim) != TYPE_PLAYER) {
	notify(player, "Try @destroy instead!");
    } else if (Wizard(victim)) {
	notify(player, "You can't toad a Wizard.");
    } else if (db[victim].contents != NOTHING) {
	notify(player,"You need to destroy their contents first.");
    } else {
	/* we're ok */
	/* do it */
	s_Pass(victim, NULL);
	db[victim].flags = TYPE_THING;
	db[victim].owner = player;	/* you get it */
	s_Pennies(victim, 1);
	
#ifdef USE_MAILER
	clear_mail(victim, 0);	/* wipe out the mail */
#endif
	
	/* notify people */
	notify(victim,
	       "You have been turned into a slimy toad.");
	notify(player, tprintf("You toaded %s!", db[victim].name));
	do_log(LT_WIZ, player, victim, "*** TOAD ***");
	
	/* reset name */
	delete_player(victim, NULL);
	if ((atemp = atr_get_noparent(victim, "ALIAS")) != NULL) {
	    strcpy(alias, uncompress(atemp->value));
	    delete_player(victim, alias);
	}
	boot_off(victim);
	sprintf(tbuf1,
		"A wart toad named %s", db[victim].name);
	SET(db[victim].name, tbuf1);
	do_chownall(player, name, "");
    }
}

void do_newpassword(player, name, password)
     dbref player;
     const char *name;
     const char *password;
{
    dbref victim;
    if (!Wizard(player)) {
	notify(player, "Your delusions of grandeur have been duly noted.");
	return;
    } else if ((victim = lookup_player(name)) == NOTHING) {
	notify(player, "No such player.");
    } else if (*password != '\0' && !ok_password(password)) {
	/* Wiz can set null passwords, but not bad passwords */
	notify(player, "Bad password.");
    } else if (God(victim) && !God(player)) {
	notify(player, "You cannot change that player's password.");
    } else {
	/* it's ok, do it */
	s_Pass(victim, crypt(password,"XX"));
	notify(player, "Password changed.");
	notify(victim, tprintf("Your password has been changed by %s.",
			       db[player].name));
	do_log(LT_WIZ, player, victim, "*** NEWPASSWORD ***");
    }
}

void do_boot(player, name, flag)
     dbref player;
     const char *name;
     int flag;			/* 0, normal boot. 1, descriptor boot */
{
    dbref victim;
    
    if (!Can_Boot(player)) {
	notify(player, "You can't give anyone the boot!");
	return;
    }
    
    if (flag) {
	/* boot by descriptor */
	victim = find_player_by_desc(atoi(name));
	if (victim == NOTHING) {
	    notify(player, "There is no one connected on that descriptor.");
	    return;
	}
    } else {
	/* boot by name */
	init_match(player, name, TYPE_PLAYER);
	match_neighbor();
	match_absolute();
	match_player();
	if ((victim = noisy_match_result()) == NOTHING)
	    return;
    }
    
    if (God(victim)) {
	notify(player, "Thunder booms in the distance.");
	return;
    }
    
    if (victim == player) {
	notify(player, "You're not that limber.");
	return;
    }
    
    if (!Connected(victim) && !God(player)) {
	notify(player, "You can't boot that off!");
    } else {
	do_log(LT_WIZ, player, victim, "*** BOOT ***");
	/* we're ok */
	/* do it */
	/* notify people */
	notify(player, tprintf("You booted %s off!", db[victim].name));
	notify(victim, "You are politely shown to the door.");
	if (!flag)
	    boot_off(victim);
	else
	    boot_desc(atoi(name));
    }
}

void do_chownall(player, name, target)
     dbref player;
     const char *name;
     const char *target;
{
    int i;
    dbref victim;
    dbref n_target;
    int count = 0;
    
    if(!Wizard(player)) {
	notify(player,"Try asking them first!");
	return;
    }
    init_match(player,name,TYPE_PLAYER);
    match_neighbor();
    match_absolute();
    match_player();
    if((victim = noisy_match_result()) == NOTHING)
	return;
    
    if(!target || !*target) {
	n_target = player;
    } else {
	init_match(player, target, TYPE_PLAYER);
	match_neighbor();
	match_absolute();
	match_player();
	if((n_target = noisy_match_result()) == NOTHING)
	    return;
    }
    
    for(i = 0; i <= db_top; i++) {
	if((db[i].owner == victim) && (Typeof(i) != TYPE_PLAYER)) {
	    db[i].owner = n_target;
	    count++;
	}
    }
    
#ifdef QUOTA
    /* change quota (this command is wiz only and we can assume that
     * we intend for the recipient to get all the objects, so we
     * don't do a quota check earlier.
     */
    change_quota(victim, count);
    change_quota(n_target, - count);
#endif
    
    notify(player, tprintf("Ownership changed for %d objects.", count));
}

void do_chzoneall(player, name, target)
     dbref player;
     const char *name;
     const char *target;
{
    int i;
    dbref victim;
    dbref zone;
    int count = 0;
    
    if (!Wizard(player)) {
	notify(player, "You do not have the power to change reality.");
	return;
    }
    
    init_match(player, name, TYPE_PLAYER);
    match_neighbor();
    match_absolute();
    match_player();
    if ((victim = noisy_match_result()) == NOTHING)
	return;
    
    if (!target || !*target) {
	notify(player, "No zone specified.");
	return;
    }
    
    if (!strcasecmp(target, "none"))
	zone = NOTHING;
    else {
	init_match(player, target, TYPE_THING);
	match_neighbor();
	match_possession();
	match_absolute();
	switch (zone = match_result()) {
	  case NOTHING:
	    notify(player, "I can't seem to find that.");
	    return;
	  case AMBIGUOUS:
	    notify(player, "I don't know which one you mean!");
	    return;
	}
	if (Typeof(zone) != TYPE_THING) {
	    notify(player, "Invalid zone object type.");
	    return;
	}
    }
    
    for (i = 0; i <= db_top; i++) {
	if (db[i].owner == victim) {
	    db[i].zone = zone;
	    count++;
	}
    }
    notify(player, tprintf("Zone changed for %d objects.", count));
}

void do_enable(player, param, state)
     dbref player;
     const char *param;
     int state;			/* enable is 1, disable is 0 */
{
    const char *name;
    
    if (!Wizard(player)) {
	notify(player, "Unable to authenticate you, sorry.");
	return;
    }
    
    if (!strcasecmp(param, "logins")) {
	name = "LOGINS";
	options.login_allow = state;
    } else if (!strcasecmp(param, "daytime")) {
	name = "DAYTIME";
	options.daytime = state;
    } else if (!strcasecmp(param, "command_log")) {
	name = "LOG COMMANDS";
	options.log_commands = state;
    } else if (!strcasecmp(param, "huh_log")) {
	name = "LOG HUHS";
	options.log_huhs = state;
    } else if (!strcasecmp(param, "force_log")) {
	name = "LOG FORCES";
	options.log_forces = state;
    } else if (!strcasecmp(param, "wall_log")) {
	name = "LOG WIZWALLS";
	options.log_walls = state;
    } else {
	notify(player, "Unknown parameter.");
	return;
    }
    
    if (state == 0)
	notify(player, "Disabled.");
    else
	notify(player, "Enabled.");

    do_log(LT_WIZ, player, NOTHING, "%s %s",
	    name, (state) ? "ENABLED" : "DISABLED");
}

/*-----------------------------------------------------------------------
 * Nasty management: @kick, examine/debug, and @fixdb
 */

void do_kick(player, num)
     dbref player;
     const char *num;
{
    /* executes <num> commands off the queue immediately */
    
    int i, n;
    
    if (!Wizard(player)) {
	notify(player, "Permission denied.");
	return;
    }
    
    if (!num || !*num) {
	notify(player, "How many commands do you want to execute?");
	return;
    }
    
    n = atoi(num);
    
    if ((n <= 0) || (n > 500)) {
	notify(player, "Number out of range.");
	return;
    }
    
    for (i = 0; i < n; i++) {
	do_top();
    }
    
    notify(player, tprintf("%d commands executed.", n));
}

void do_fixdb(player, name, val, sw_opt)
     dbref player;
     const char *name;
     const char *val;
     int sw_opt;
{
    dbref thing;
    dbref new;
    
    /* do all the checks */
    if (!Wizard(player)) {
	notify(player, "Reality is not yours to warp.");
	return;
    }
    
    if (!name || !*name) {
	notify(player, "You must specify an object.");
	return;
    }
    
    if (!val || !*val) {
	notify(player, "You must specify a value.");
	return;
    }
    
    if (!is_number(val)) {
	notify(player, "The value must be an integer.");
	return;
    }
    
    /* find the object */
    init_match(player, name, NOTYPE);
    match_everything();
    thing = noisy_match_result();
    if (thing == NOTHING)
	return;
    
    /* change the pointers. Do no error checking. */
    
    new = atoi(val);
    
    switch (sw_opt) {
      case 0:
	db[thing].location = new;
	break;
      case 1:
	db[thing].contents = new;
	break;
      case 2:
	db[thing].exits = new;
	break;
      case 3:
	db[thing].next = new;
	break;
    }
    
    notify(player, "Fixed.");
}

void do_debug_examine(player, name)
     dbref player;
     const char *name;
{
    dbref thing;
    
    if (!Hasprivs(player)) {
	notify(player, "Permission denied.");
	return;
    }
    
    /* find it */
    init_match(player, name, NOTYPE);
    match_everything();
    thing = noisy_match_result();
    if (thing == NOTHING)
	return;
    
    if (Flags(thing) & GOING) {
	notify(player, "Garbage is garbage.");
	return;
    }
    
    notify(player, object_header(player, thing));
    notify(player, tprintf("Flags value: %d", Flags(thing)));
    notify(player, tprintf("Toggles value: %d", Toggles(thing)));
    notify(player, tprintf("Powers value: %d", Powers(thing)));
    
    notify(player, tprintf("Next: %d", db[thing].next));
    notify(player, tprintf("Contents: %d", db[thing].contents));
    
    switch (Typeof(thing)) {
      case TYPE_PLAYER:
      case TYPE_THING:
	notify(player, tprintf("Location: %d", db[thing].location));
	notify(player, tprintf("Home: %d", db[thing].exits));
	break;
      case TYPE_EXIT:
	notify(player, tprintf("Destination: %d", db[thing].location));
	notify(player, tprintf("Source: %d", db[thing].exits));
	break;
      case TYPE_ROOM:
	notify(player, tprintf("Drop-to: %d", db[thing].location));
	notify(player, tprintf("Exits: %d", db[thing].exits));
	break;
      default:
	notify(player, "Bad object type.");
    }
}

/*-------------------------------------------------------------------------
 * Powers stuff
 */

static object_flag_type find_power(s)
     const char *s;
{
    object_flag_type p;
    
    p = -1;			/* error value */
    
    if (string_prefix("boot", s))
	p = CAN_BOOT;
    else if (string_prefix("builder", s))
	p = CAN_BUILD;
    else if (string_prefix("chat_privs", s))
	p = CHAT_PRIVS;
    else if (string_prefix("functions", s))
	p = GLOBAL_FUNCS;
    else if (string_prefix("halt", s))
	p = HALT_ANYTHING;
    else if (string_prefix("hide", s))
	p = CAN_HIDE;
    else if (string_prefix("idle", s))
	p = UNLIMITED_IDLE;
    else if (string_prefix("immortal", s))
	p = IMMORTAL;
    else if (string_prefix("login", s))
	p = LOGIN_ANYTIME;
    else if (string_prefix("long_fingers", s))
	p = LONG_FINGERS;
    else if (string_prefix("player_create", s))
	p = CREATE_PLAYER;
    else if (string_prefix("poll", s))
	p = SET_POLL;
    else if (string_prefix("queue", s))
	p = HUGE_QUEUE;
    else if (string_prefix("quotas", s))
	p = CHANGE_QUOTAS;
    else if (string_prefix("search", s))
	p = SEARCH_ALL;
    else if (string_prefix("see_all", s))
	p = SEE_ALL;
    else if (string_prefix("see_queue", s))
	p = PS_ALL;
    else if (string_prefix("tport_anywhere", s))
	p = TEL_ANYWHERE;
    else if (string_prefix("tport_anything", s))
	p = TEL_OTHER;
    else
	p = -1;
    
    return p;
}

void do_power(player, name, power)
     dbref player;
     const char *name;
     const char *power;
{
    object_flag_type pwr;
    char *s;
    dbref thing;
    
    if (!Wizard(player)) {
	notify(player, "Only wizards may grant powers.");
	return;
    }
    
    init_match(player, name, NOTYPE);
    match_everything();
    if ((thing = noisy_match_result()) == NOTHING)
	return;
    
    /* move past the not token if there is one */
    for (s = (char *) power; *s && ((*s == NOT_TOKEN) || isspace(*s)); s++)
	;
    
    if (*s == '\0') {
	notify(player, "You must specify a power.");
	return;
    }
    pwr = find_power(s);
    if (pwr == -1) {
	notify(player, "That is not a power.");
	return;
    }
    
    if (*power == NOT_TOKEN) {
	Powers(thing) &= ~pwr;
	if (!Quiet(player) && !Quiet(thing))
	    notify(player, tprintf("%s - power removed.", Name(thing)));
	do_log(LT_WIZ, player, thing, "Power Removed: %s", power);
    } else {
	Powers(thing) |= pwr;
	if (!Quiet(player) && !Quiet(thing))
	    notify(player, tprintf("%s - power granted.", Name(thing)));
	do_log(LT_WIZ, player, thing, "Power Granted: %s", power);
    }
}

/*----------------------------------------------------------------------------
 * Search functions
 */


void do_search(player, arg1, arg3)
     dbref player;
     const char *arg1;
     char *arg3[];
{
    int flag, is_wizard;
    char *arg2, *restrict_name, *ebuf1, *ebuf2;
    dbref thing, from, to;
    int destitute = 1;
    int eval_search = 0;
    dbref restrict_owner = NOTHING;
    dbref restrict_zone = NOTHING;
    dbref restrict_parent = NOTHING;
    object_flag_type flag_mask, toggle_mask, restrict_type, power_mask;
    char tbuf1[BUFFER_LEN];
    int rcount, ecount, pcount, ocount;
    struct dblist *found = NULL, *ptr = NULL, *rlist = NULL,
    *elist = NULL, *olist = NULL, *plist = NULL;
    int bot = 0;
    int top = db_top;
    
    if (options.daytime) {
	notify(player, "Sorry, that command has been temporarily disabled.");
	return;
    }
    
    /* make sure player has money to do the search */
    if (!payfor(player, FIND_COST)) {
	notify(player, tprintf("Searches cost %d %s.", FIND_COST,
			       ((FIND_COST == 1) ? MONEY : MONIES)));
	return;
    }
    
    /* parse first argument into two */
    if (!arg1 || *arg1 == '\0')
	arg1 = "me";
    arg2 = (char *) index(arg1, ' ');
    if (arg2 != NULL)
	*arg2++ = '\0';		/* arg1, arg2, arg3 */
    else {
	if (!arg3[1] || !*arg3[1])
	    arg2 = (char *) "";	/* arg1 */
	else {
	    arg2 = (char *) arg1;	/* arg2, arg3 */
	    arg1 = (char *) "";
	}
    }
    
    is_wizard = See_All(player);
    
    /* set limits on who we search */
    if (*arg1 == '\0') {
	restrict_owner = is_wizard ? ANY_OWNER : player;
    } else if (arg1[0] == '#') {
	restrict_owner = atoi(&arg1[1]);
	if (restrict_owner < 0 || db_top <= restrict_owner)
	    restrict_owner = NOTHING;
	else if (Typeof(restrict_owner) != TYPE_PLAYER)
	    restrict_owner = NOTHING;
    } else if (strcmp(arg1, "me") == 0) {
	restrict_owner = player;
    } else {
	restrict_owner = lookup_player(arg1);
    }
    
    if (restrict_owner == NOTHING) {
	notify(player, tprintf("%s: No such player.", arg1));
	return;
    }
    
    /* set limits on what we search for */
    flag = flag_mask = power_mask = toggle_mask = 0;
    restrict_name = NULL;
    restrict_type = NOTYPE;
    switch (arg2[0]) {
      case '\0':
	/* the no class requested class  :)  */
	break;
      case 'e':
      case 'E':
	if (string_prefix("exits", arg2)) {
	    restrict_name = (char *) arg3[1];
	    restrict_type = TYPE_EXIT;
	} else if (string_prefix("eval", arg2)) {
	    eval_search = 1;
	} else  {
	    flag = 1;
	}
	break;
      case 'f':
      case 'F':
	if (string_prefix("flags", arg2)) {
	    /*
	     * convert_flags ignores previous values of flag_mask and
	     * restrict_type while setting them
	     */
	    if (arg3[1] && *arg3[1] && 
		!convert_flags(player, arg3[1], &flag_mask, &toggle_mask,
			       &restrict_type))
		return;
	} else
	    flag = 1;
	break;
      case 'n':
      case 'N':
	if (string_prefix("name", arg2))
	    restrict_name = (char *) arg3[1];
	else
	    flag = 1;
	break;
      case 'o':
      case 'O':
	if (string_prefix("objects", arg2)) {
	    restrict_name = (char *) arg3[1];
	    restrict_type = TYPE_THING;
	} else
	    flag = 1;
	break;
      case 'p':
      case 'P':
	switch (arg2[1]) {
	  case 'a':
	  case 'A':
	    if (string_prefix("parent", arg2)) {
		if (arg3[1] && *arg3[1]) {
		    init_match(player, arg3[1], NOTYPE);
		    match_absolute();
		    restrict_parent = match_result();
		}
		if (restrict_parent == NOTHING) {
		    notify(player, "Unknown parent.");
		    return;
		}
	    } else
		flag = 1;
	    break;
	  case 'l':
	  case 'L':
	    if (string_prefix("players", arg2)) {
		restrict_name = (char *) arg3[1];
		if (!arg1 || !*arg1)
		    restrict_owner = ANY_OWNER;
		restrict_type = TYPE_PLAYER;
	    } else
		flag = 1;
	    break;
	  case 'o':
	  case 'O':
	    if (string_prefix("powers", arg2)) {
		power_mask = find_power(arg3[1]);
		if (power_mask == -1) {
		    notify(player, "No such power to search for.");
		    return;
		}
	    } else
		flag = 1;
	    break;
	  default:
	    flag = 1;
	} 
	break;
      case 'r':
      case 'R':
	if (string_prefix("rooms", arg2)) {
	    restrict_name = (char *) arg3[1];
	    restrict_type = TYPE_ROOM;
	} else
	    flag = 1;
	break;
      case 't':
      case 'T':
	if (string_prefix("type", arg2)) {
	    if (!arg3[1] || !*arg3[1])
		break;
	    if (string_prefix("room", arg3[1]))
		restrict_type = TYPE_ROOM;
	    else if (string_prefix("exit", arg3[1]))
		restrict_type = TYPE_EXIT;
	    else if (string_prefix("object", arg3[1]))
		restrict_type = TYPE_THING;
	    else if (string_prefix("player", arg3[1])) {
		if (!arg1 || !*arg1)
		    restrict_owner = ANY_OWNER;
		restrict_type = TYPE_PLAYER;
	    } else {
		notify(player, tprintf("%s: unknown type.", arg3[1]));
		return;
	    }
	} else
	    flag = 1;
	break;
      case 'z':
      case 'Z':
	if (string_prefix("zone", arg2)) {
	    if (arg3[1] && *arg3[1]) {
		init_match(player, arg3[1], TYPE_THING);
		match_absolute();
		restrict_zone = match_result();
	    }
	    if (restrict_zone == NOTHING) {
		notify(player, "Unknown zone.");
		return;
	    }
	} else
	    flag = 1;
	break;
      default:
	flag = 1;
    }
    if (flag) {
	notify(player, tprintf("%s: unknown class.", arg2));
	return;
    }
    /* make sure player is authorized to do requested search */
    if (!is_wizard && restrict_type != TYPE_PLAYER &&
	(restrict_owner == ANY_OWNER || restrict_owner != player)) {
	notify(player, "You need a search warrant to do that!");
	return;
    }
    
    /* find range */
    if (arg3[2] && *arg3[2])
	bot = atoi(arg3[2]);
    if (bot < 0)
	bot = 0;
    if (arg3[3] && *arg3[3])
	top = atoi(arg3[3]) + 1;
    if (top > db_top)
	top = db_top;
    
    /* the search loop here */
    flag = 1;
    for (thing = bot; thing < top; thing++) {
	switch(Typeof(thing)) {
	  case TYPE_PLAYER:
	    found = plist;
	    break;
	  case TYPE_EXIT:
	    found = elist;
	    break;
	  case TYPE_THING:
	    found = olist;
	    break;
	  case TYPE_ROOM:
	    found = rlist;
	    break;
	  default:
	    continue;
	}
	if (Flags(thing) & GOING)
	    continue;
	if ((restrict_owner != ANY_OWNER) && (restrict_owner != Owner(thing)))
	    continue;
	if (!eval_search) {
	    if (restrict_type != NOTYPE && restrict_type != Typeof(thing))
		continue;
	    if ((Flags(thing) & flag_mask) != flag_mask)
		continue;
	    if ((Toggles(thing) & toggle_mask) != toggle_mask)
		continue;
	    if ((Powers(thing) & power_mask) != power_mask)
		continue;
	    if (restrict_name != NULL)
		if (!string_match(db[thing].name, restrict_name))
		    continue;
	    if (restrict_parent != NOTHING)
		if (restrict_parent != Parent(thing))
		    continue;
	    if (restrict_zone != NOTHING)
		if (restrict_zone != getzone(thing))
		    continue;
	} else {
	    ebuf1 = replace_string("##", tprintf("#%d", thing), arg3[1]);
	    ebuf2 = exec(player, player, EV_STRIP | EV_FCHECK | EV_EVAL,
			 ebuf1);
	    free(ebuf1);
	    if (*ebuf2 != '1') {
		free(ebuf2);
		continue;
	    }
	    free(ebuf2);
	}
	if (!found) {
	    flag = 0;
	    destitute = 0;
	    found = listcreate(thing);
	} else
	    listadd(found, thing);
	
	switch(Typeof(thing)) {
	  case TYPE_PLAYER:
	    plist = found;
	    break;
	  case TYPE_EXIT:
	    elist = found;
	    break;
	  case TYPE_THING:
	    olist = found;
	    break;
	  case TYPE_ROOM:
	    rlist = found;
	    break;
	  default:
	    break;
	}
    }
    /* if nothing found matching search criteria */
    if (destitute) {
	notify(player, "Nothing found.");
	return;
    }
    
    /* Walk down the list and print */
    rcount = ecount = pcount = ocount = 0;
    if (restrict_type == TYPE_ROOM || restrict_type == NOTYPE) {
	notify(player, "\nROOMS:");
	for (ptr = rlist; ptr; ptr = ptr->next) {
	    sprintf(tbuf1, "%s [owner: ", object_header(player, ptr->obj));
	    strcat(tbuf1, tprintf("%s]",
				  object_header(player, db[ptr->obj].owner)));
	    notify(player, tbuf1);
	    rcount++;
	}
    }
    if (restrict_type == TYPE_EXIT || restrict_type == NOTYPE) {
	notify(player, "\nEXITS:");
	for (ptr = elist; ptr; ptr = ptr->next) {
	    if(db[ptr->obj].exits == NOTHING)
		from = find_entrance(thing);
	    else
		from = db[ptr->obj].exits;
	    to = db[ptr->obj].location;
	    sprintf(tbuf1, "%s [from ", object_header(player, ptr->obj));
	    strcat(tbuf1, tprintf("%s to ", (from == NOTHING) ? "NOWHERE" :
				  object_header(player, from)));
	    strcat(tbuf1, tprintf("%s]", (to == NOTHING) ? "NOWHERE" :
				  object_header(player, to)));
	    notify(player, tbuf1);
	    ecount++;
	}
    }
    if (restrict_type == TYPE_THING || restrict_type == NOTYPE) {
	notify(player, "\nOBJECTS:");
	for (ptr = olist; ptr; ptr = ptr->next) {
	    sprintf(tbuf1, "%s [owner: ", object_header(player, ptr->obj));
	    strcat(tbuf1, tprintf("%s]",
				  object_header(player, db[ptr->obj].owner)));
	    notify(player, tbuf1);
	    ocount++;
	}
    }
    if ((restrict_type == TYPE_PLAYER) || (restrict_type == NOTYPE)) {
	notify(player, "\nPLAYERS:");
	for (ptr = plist; ptr; ptr = ptr->next) {
	    strcpy(tbuf1, object_header(player, ptr->obj));
	    if (is_wizard) {
		strcat(tbuf1, " [location: ");
		strcat(tbuf1, object_header(player, db[ptr->obj].location));
		strcat(tbuf1, "]");
	    }
	    notify(player, tbuf1);
	    pcount++;
	}
    }
    notify(player, "----------  Search Done  ----------");
    notify(player,
	   tprintf("Totals: Rooms...%d  Exits...%d  Objects...%d  Players...%d",
		   rcount, ecount, ocount, pcount));
    if (plist)
	listfree(plist);
    if (rlist)
	listfree(rlist);
    if(olist)
	listfree(olist);
    if(elist)
	listfree(elist);
}

XFUNCTION(fun_lsearch)
{
    /* takes arguments in the form of: player, class, restriction */
    
    int flag, is_wiz;
    char *who, *class, *restrict, *restrict_name, *bp;
    object_flag_type flag_mask, toggle_mask, restrict_type, power_mask;
    dbref restrict_owner = NOTHING;
    dbref restrict_zone = NOTHING;
    dbref restrict_parent = NOTHING;
    dbref thing;
    int destitute = 1;
    struct dblist *found = NULL, *ptr = NULL, *list = NULL;
    
    if (options.daytime) {
	notify(privs, "Function disabled.");
	strcpy(buff, "#-1");
	return;
    }
    
    if (!payfor(privs, FIND_COST)) {
	notify(privs, "Not enough money to do search.");
	strcpy(buff, "#-1");
	return;
    }
    
    is_wiz = Search_All(privs);
    
    who = args[0];
    class = args[1];
    restrict = args[2];
    
    /* set limits on who we search */
    if (!strcasecmp(who, "all"))
	restrict_owner = is_wiz ? ANY_OWNER : privs;
    else if (!strcasecmp(who, "me"))
	restrict_owner = privs;
    else
	restrict_owner = lookup_player(who);
    
    if (restrict_owner == NOTHING) {
	notify(privs, "No such player.");
	strcpy(buff, "#-1");
	return;
    }
    
    /* set limits on what we search for */
    flag = 0;
    flag_mask = 0;
    toggle_mask = 0;
    power_mask = 0;
    restrict_name = NULL;
    restrict_type = NOTYPE;
    switch (class[0]) {
      case 'e':
      case 'E':
	if (string_prefix("exits", class)) {
	    restrict_name = (char *) restrict;
	    restrict_type = TYPE_EXIT;
	} else
	    flag = 1;
	break;
      case 'f':
      case 'F':
	if (string_prefix("flags", class)) {
	    if (!convert_flags(privs, restrict, &flag_mask, &toggle_mask,
			       &restrict_type)) {
		notify(privs, "Unknown flag.");
		strcpy(buff, "#-1");
		return;
	    }
	} else
	    flag = 1;
	break;
      case 'n':
      case 'N':
	if (string_prefix("name", class))
	    restrict_name = (char *) restrict;
	else if (!string_prefix("none", class))
	    flag = 1;
	break;
      case 'o':
      case 'O':
	if (string_prefix("objects", class)) {
	    restrict_name = (char *) restrict;
	    restrict_type = TYPE_THING;
	} else
	    flag = 1;
	break;
      case 'p':
      case 'P':
	switch (class[1]) {
	  case 'a':
	  case 'A':
	    if (string_prefix("parent", class)) {
		init_match(privs, restrict, NOTYPE);
		match_absolute();
		restrict_parent = match_result();
		if (restrict_parent == NOTHING) {
		    notify(privs, "Unknown parent.");
		    strcpy(buff, "#-1");
		    return;
		}
	    } else
		flag = 1;
	    break;
	  case 'l':
	  case 'L':
	    if (string_prefix("players", class)) {
		restrict_name = (char *) restrict;
		restrict_type = TYPE_PLAYER;
	    } else
		flag = 1;
	    break;
	  case 'o':
	  case 'O':
	    if (string_prefix("powers", class)) {
		power_mask = find_power(restrict);
		if (power_mask == -1) {
		    notify(privs, "Unknown power.");
		    strcpy(buff, "#-1");
		    return;
		}
	    }
	    else
		flag = 1;
	    break;
	  default:
	    flag = 1;
	}
	break;
      case 'r':
      case 'R':
	if (string_prefix("rooms", class)) {
	    restrict_name = (char *) restrict;
	    restrict_type = TYPE_ROOM;
	} else
	    flag = 1;
	break;
      case 't':
      case 'T':
	if (string_prefix("type", class)) {
	    if (!strcasecmp(restrict, "none"))
		break;
	    if (string_prefix("room", restrict))
		restrict_type = TYPE_ROOM;
	    else if (string_prefix("exit", restrict))
		restrict_type = TYPE_EXIT;
	    else if (string_prefix("object", restrict))
		restrict_type = TYPE_THING;
	    else if (string_prefix("player", restrict))
		restrict_type = TYPE_PLAYER;
	    else {
		notify(privs, "Unknown type.");
		strcpy(buff, "#-1");
		return;
	    }
	} else
	    flag = 1;
	break;
      case 'z':
      case 'Z':
	if (string_prefix("zone", class)) {
	    init_match(privs, restrict, TYPE_THING);
	    match_absolute();
	    restrict_zone = match_result();
	    if (restrict_zone == NOTHING) {
		notify(privs, "Unknown zone.");
		strcpy(buff, "#-1");
		return;
	    }
	} else
	    flag = 1;
	break;
      default:
	flag = 1;
    }
    if (flag) {
	notify(privs, "Unknown type.");
	strcpy(buff, "#-1");
	return;
    }
    
    /* check privs */
    if (!is_wiz && (restrict_type != TYPE_PLAYER) &&
	((restrict_owner == ANY_OWNER) || (restrict_owner != privs))) {
	notify(privs, "Permission denied.");
	strcpy(buff, "#-1");
	return;
    }
    
    /* search loop */
    flag = 1;
    for (thing = 0; thing < db_top; thing++) {
	found = list;
	if ((restrict_type != NOTYPE) && (restrict_type != Typeof(thing)))
	    continue;
	if ((Flags(thing) & flag_mask) != flag_mask)
	    continue;
#ifdef DESTROY
	if (Flags(thing) & GOING)
	    continue;
#endif
	if ((Toggles(thing) & toggle_mask) != toggle_mask)
	    continue;
	if ((Powers(thing) & power_mask) != power_mask)
	    continue;
	if ((restrict_owner != ANY_OWNER) && (restrict_owner != db[thing].owner))
	    continue;
	if ((restrict_name != NULL) &&
	    (!string_match(db[thing].name, restrict_name)))
	    continue;
	if ((restrict_parent != NOTHING) && (restrict_parent != Parent(thing)))
	    continue;
	if ((restrict_zone != NOTHING) && (restrict_zone != getzone(thing)))
	    continue;
	if (!found) {
	    flag = 0;
	    destitute = 0;
	    found = listcreate(thing);
	} else
	    listadd(found, thing);
	
	list = found;
    }
    
    /* nothing found matching search criteria */
    if (destitute) {
	notify(privs, "Nothing found.");
	strcpy(buff, "#-1");
	return;
    }
    
    /* go down the list and print into a buffer */
    bp = buff;
    for (ptr = list; ptr; ptr = ptr->next) {
	if (bp != buff)
	    safe_chr(' ', buff, &bp);
	safe_str(tprintf("#%d", ptr->obj), buff, &bp);
    }
    *bp = '\0';
    
    if (list)
	listfree(list);
}

