/* look.c */

#include "copyright.h"

/* commands which look at things */
#include <string.h>

#include "config.h"
#include "db.h"
#include "interface.h"
#include "match.h"
#include "externs.h"
#include "ansi.h"

#if (CHAT_SYSTEM >= 2)
#include "chat.h"
#endif

extern void decompile_flags();	/* from flags.c */

static void look_exits(player, loc, exit_name)
    dbref player;
    dbref loc;
    const char *exit_name;
{
  dbref thing;
  char tbuf1[BUFFER_LEN];
  char *e = tbuf1;

  /* make sure location is a room */
  if ((Typeof(loc) != TYPE_ROOM) || (Flags(loc) & DARK))
    return;
  for (thing = Exits(loc);
       (thing != NOTHING) && (Flags(thing) & DARK);
       thing = db[thing].next) ;
  if (thing == NOTHING)
    return;
  notify(player, exit_name);
  if (Flags(loc) & TRANSPARENT) {
    for (thing = Exits(loc); thing != NOTHING; thing = db[thing].next) {
      if (!(Flags(thing) & DARK)) {
	if (Name(thing)) {
	  strcpy(tbuf1, Name(thing));
	  for (e = tbuf1; *e && (*e != ';'); e++)
	    ;
	  *e = 0;
	  if (Location(thing) == NOTHING)
	    notify(player, tprintf(" %s leads nowhere.", tbuf1));
	  else if (Location(thing) == HOME)
	    notify(player, tprintf(" %s leads home.", tbuf1));
	  else
	    notify(player, tprintf(" %s leads to %s.", tbuf1, 
				   db[Location(thing)].name));
	}
      }
    }				/* end of for loop */
  }				/* if transparent room */
  else {
    for (thing = Exits(loc); thing != NOTHING; thing = db[thing].next)
      if (!(Flags(thing) & DARK))
	/* chop off first exit alias to display */
	{
	  char *s;
	  if (Name(thing) && ((e - tbuf1) < BUFFER_LEN)) {
	    for (s = (char *) Name(thing); *s && (*s != ';') &&
		 (e - tbuf1 < BUFFER_LEN); *e++ = *s++) ;
	    *e++ = ' ';
	    *e++ = ' ';
	  }
	}			/* if !dark */
    *e++ = 0;
    notify(player, tbuf1);
  }				/* else  */
}


static void look_contents(player, loc, contents_name)
    dbref player;
    dbref loc;
    const char *contents_name;
{
  dbref thing;
  dbref can_see_loc;
  /* check to see if he can see the location */
  /*
   * patched so that player can't see in dark rooms even if owned by that
   * player.  (he must use examine command)
   */
  can_see_loc = !Dark(loc);

  /* check to see if there is anything there */
  DOLIST(thing, db[loc].contents) {
    if (can_see(player, thing, can_see_loc)) {
      /* something exists!  show him everything */
      notify(player, contents_name);
      DOLIST(thing, db[loc].contents) {
	if (can_see(player, thing, can_see_loc)) {
	  notify(player, unparse_object(player, thing));
	}
      }
      break;			/* we're done */
    }
  }
}

static void look_atrs(player, thing, mstr)
     dbref player;
     dbref thing;
     const char *mstr;
{
  char fbuf[12];
  char *bp, *r;
  int got = 0;
  ALIST *list = db[thing].list;

  while (list) {
    if (!AL_BAD(list) && Can_Read_Attr(player, thing, AL_ATTR(list)) &&
	((!mstr && (strcmp(AL_NAME(list), "DESCRIBE") != 0)) ||
	 (mstr && wild_match(mstr, AL_NAME(list))))) {
      got = 1;
      r = safe_uncompress(AL_STR(list));
      bp = fbuf;
      if (AL_FLAGS(list) & AF_LOCKED)
	  *bp++ = '+';
      if (AL_FLAGS(list) & AF_NOPROG)
	  *bp++ = '$';
      if (!(AL_FLAGS(list) & AF_ODARK))
	  *bp++ = 'v';
      if (AL_FLAGS(list) & AF_MDARK)
	  *bp++ = 'm';
      if (AL_FLAGS(list) & AF_WIZARD)
	  *bp++ = 'w';
      if (AL_FLAGS(list) & AF_PRIVATE)
	  *bp++ = 'i';
      if (AL_FLAGS(list) & AF_NOCOPY)
	  *bp++ = 'c';
      *bp = '\0';
#ifdef TINY_ATTRS
      if (ShowAnsi(player)) {
	if (Owner(AL_CREATOR(list)) != Owner(thing))
	  notify(player,
		 tprintf("%s%s(#%d%s):%s%s", ANSI_HILITE, AL_NAME(list),
			 Owner(AL_CREATOR(list)), fbuf, ANSI_NORMAL, r));
	else if (*fbuf != '\0')
	  notify(player, tprintf("%s%s(%s):%s%s", ANSI_HILITE, AL_NAME(list),
				 fbuf, ANSI_NORMAL, r));
	else
	  notify(player, tprintf("%s%s:%s%s", ANSI_HILITE, AL_NAME(list),
				 ANSI_NORMAL, r));
      } else {
	if (Owner(AL_CREATOR(list)) != Owner(thing))
	  notify(player, tprintf("%s(#%d%s):%s", AL_NAME(list), 
				 Owner(AL_CREATOR(list)), fbuf, r));
	else if (*fbuf != '\0')
	  notify(player, tprintf("%s(%s):%s", AL_NAME(list), fbuf, r));
	else
	  notify(player, tprintf("%s:%s", AL_NAME(list), r));
      }
#else
      if (ShowAnsi(player)) {
	notify(player,
	       tprintf("%s%s [#%d%s]:%s %s", ANSI_HILITE, AL_NAME(list),
		       db[AL_CREATOR(list)].owner, fbuf, ANSI_NORMAL, r));
      } else {
	notify(player, tprintf("%s [#%d%s]: %s", AL_NAME(list),
			       db[AL_CREATOR(list)].owner, fbuf, r));
      }
#endif
      free(r);
    }
    list = AL_NEXT(list);
  }
  if (!got && mstr)
    notify(player, "No matching attributes.");
}

static void look_simple(player, thing)
    dbref player;
    dbref thing;
{
  Access(thing);
  if (controls(player, thing) || Visual(thing))
    notify(player, unparse_object(player, thing));
  did_it(player, thing, "DESCRIBE", "You see nothing special.", "ODESCRIBE",
         NULL, "ADESCRIBE", NOTHING);
  if (Hasflag(thing, TYPE_EXIT, TRANSPARENT)) {
    if (Location(thing) == HOME)
      look_room(player, db[player].exits, 1);
    else
      look_room(player, Location(thing), 1);
  }
}

void look_room(player, loc, flag)
    dbref player;
    dbref loc;
    int flag;
{
  /* look at a room. Flag value of this function:
   *   0  --  normal look, caused by "look" command.
   *   1  --  remote look, done through a TRANPARENT exit.
   *   2  --  auto-look, caused by moving. Obey TERSE.
   */

  ATTR *s;
  char *tbuf1;
  char tbuf2[BUFFER_LEN];

  if (loc == NOTHING)
    return;

  /* don't give the unparse if looking through Transparent exit */
  if (flag != 1)		
    notify(player, unparse_object(player, loc));
  if (Typeof(loc) != TYPE_ROOM) {
    if ((flag != 2) || !Terse(player)) {
      if ((s = atr_get(loc, "IDESCRIBE")) ||
	  (s = atr_get(loc, "DESCRIBE"))) {
	strcpy(tbuf2, uncompress(s->value));
	tbuf1 = exec(loc, loc, 0, tbuf2);
	notify(player, tbuf1);
	free(tbuf1);
      }
    }
  }
  /* tell him the description */
  else {
    if (flag != 1) {
      if ((flag != 2) || !Terse(player))
	did_it(player, loc, "DESCRIBE", NULL, "ODESCRIBE", NULL, "ADESCRIBE",
	       NOTHING);
      else
	did_it(player, loc, NULL, NULL, "ODESCRIBE", NULL, "ADESCRIBE",
	       NOTHING);
    } else
      did_it(player, loc, "DESCRIBE", NULL, NULL, NULL, NULL, NOTHING);
  }
  /* tell him the appropriate messages if he has the key */
  if (Typeof(loc) == TYPE_ROOM && (flag != 1)) {
    if ((flag == 2) && Terse(player)) {
      if (could_doit(player, loc))
	did_it(player, loc, NULL, NULL, "OSUCCESS", NULL, "ASUCCESS",
	       NOTHING);
      else
	did_it(player, loc, NULL, NULL, "OFAILURE", NULL, "AFAILURE",
	       NOTHING);
    } else
      if (could_doit(player, loc))
	did_it(player, loc, "SUCCESS", NULL, "OSUCCESS", NULL, "ASUCCESS",
	       NOTHING);
      else
	did_it(player, loc, "FAILURE", NULL, "OFAILURE", NULL, "AFAILURE",
	       NOTHING);
  }
  /* tell him the contents */
  look_contents(player, loc, "Contents:");
  if (flag != 1)
    look_exits(player, loc, "Obvious exits:");
}

void do_look_around(player)
    dbref player;
{
  dbref loc;
  if ((loc = Location(player)) == NOTHING)
    return;
  look_room(player, loc, 2);	/* auto-look. Obey TERSE. */
}

void do_look_at(player, name, key)
     dbref player;
     const char *name;
     int key;			/* 0 is normal, 1 is "outside" */
{
  dbref thing;
  dbref loc;

  if (Location(player) == NOTHING)
    return;

  if (key) {			/* look outside */

    /* can't see through opaque objects */
    if ((Typeof(Location(player)) == TYPE_ROOM) ||
	Flags(Location(player)) & OPAQUE) {
      notify(player, "You can't see through that.");
      return;
    }

    loc = Location(Location(player));
    
    if (loc == NOTHING)
      return;

    /* look at location of location */
    if (*name == '\0') {
	look_room(player, loc, 0);
      return;
    }

    init_match_remote(loc, name, NOTYPE);
    match_remote();
    thing = match_result();
    if (thing == NOTHING) {
      notify(player, "I don't see that here.");
      return;
    }
  } else {			/* regular look */
    if (*name == '\0') {
      look_room(player, Location(player), 0);
      return;
    } 

    /* look at a thing in location */
    init_match(player, name, NOTYPE);
    match_nearby();
    if ((thing = match_result()) == NOTHING) {
      thing = parse_match_possessive(player, name);
      if (!GoodObject(thing)) {
	notify(player, "I don't see that here.");
	return;
      }
      if ((Flags(Location(thing)) & OPAQUE) &&
	  (!See_All(player) &&
	   !controls(player, thing) && !controls(player, Location(thing)))) {
	notify(player, "You can't look at that from here.");
	return;
      }
    } else {
      thing = noisy_match_result();
      if (thing == NOTHING)
	return;
    }
  }
  
  /* once we've determined the object to look at, it doesn't matter whether
   * this is look or look/outside.
   */

  /* we need to check for the special case of a player doing 'look here'
   * while inside an object.
   */
  if (Location(player) == thing) {
    look_room(player, thing, 0);
    return;
  }

  switch (Typeof(thing)) {
  case TYPE_ROOM:
    look_room(player, thing, 0);
    /* look_atrs(player, thing); */
    break;
  case TYPE_THING:
  case TYPE_PLAYER:
    look_simple(player, thing);
    /* look_atrs(player,thing); */
    if (!(Flags(thing) & OPAQUE))
      look_contents(player, thing, "Carrying:");
    break;
  default:
    look_simple(player, thing);
    /* look_atrs(player,thing); */
    break;
  }
}

static const char *powers_descriptions(thing)
     dbref thing;
{
    static char buf[BUFFER_LEN];
    object_flag_type p;

    p = Powers(thing);
    strcpy(buf, "Powers:");

    if (p & CAN_BOOT)
	strcat(buf, " Boot");
    if (p & CAN_BUILD)
	strcat(buf, " Builder");
    if (p & CHAT_PRIVS)
	strcat(buf, " Chat_Privs");
    if (p & CREATE_PLAYER)
      strcat(buf, "Player_Create");
    if (p & GLOBAL_FUNCS)
	strcat(buf, " Functions");
    if (p & HALT_ANYTHING)
	strcat(buf, " Halt");
    if (p & CAN_HIDE)
	strcat(buf, " Hide");
    if (p & UNLIMITED_IDLE)
	strcat(buf, " Idle");
    if (p & IMMORTAL)
	strcat(buf, " Immortal");
    if (p & LOGIN_ANYTIME)
	strcat(buf, " Login");
    if (p & LONG_FINGERS)
	strcat(buf, " Long_Fingers");
    if (p & SET_POLL)
	strcat(buf, " Poll");
    if (p & HUGE_QUEUE)
	strcat(buf, " Queue");
    if (p & CHANGE_QUOTAS)
	strcat(buf, " Quotas");
    if (p & SEARCH_ALL)
	strcat(buf, " Search");
    if (p & SEE_ALL)
	strcat(buf, " See_All");
    if (p & PS_ALL)
	strcat(buf, " See_Queue");
    if (p & TEL_ANYWHERE)
	strcat(buf, " Tport_Anywhere");
    if (p & TEL_OTHER)
	strcat(buf, " Tport_Anything");

    return buf;
}

#if (CHAT_SYSTEM >= 2)
static const char *channel_description(player)
     dbref player;
{
  int b, mask;
  static char buf[BUFFER_LEN];
  channel_type c = db[player].channels;

  if (c == 0) {
    strcpy(buf, "Channels: *NONE*");
    return buf;
  }

  strcpy(buf, "Channels:");

  /* walk list of channels */
  mask = 1;
  for (b = 1; b <= 32; b++) {
    if (c & mask)
      sprintf(buf, "%s %s", buf, channel_name(mask));
    mask <<= 1;
  }

  return buf;
}
#endif

void do_examine(player, name, brief)
    dbref player;
    const char *name;
    int brief;
{
  dbref thing;
  ATTR *a;
  dbref content;
  dbref exit_dbref;
  char *real_name = NULL, *attrib_name = NULL;
  char *r, *tp;
  char tbuf[BUFFER_LEN];
  int ok = 0;

  if (*name == '\0') {
    if ((thing = Location(player)) == NOTHING)
      return;
    attrib_name = NULL;
  } else {

    if ((attrib_name = (char *) index(name,'/')) != NULL) {
      *attrib_name = '\0';
      attrib_name++;
    }
    real_name = (char *)name; 
    /* look it up */
    init_match(player, real_name, NOTYPE);
    match_everything();

    /* get result */
    if ((thing = noisy_match_result()) == NOTHING)
      return;
  }
  /*  only look at some of the attributes */
  if (attrib_name && *attrib_name) {
    look_atrs(player, thing, attrib_name);
    return;
  }

  /*  can't examine destructed objects  */
  if (Flags(thing) & GOING) {
    notify(player, "Garbage is garbage.");
    return;
  }

  ok = Can_Examine(player, thing);

#ifdef EX_PUBLIC_ATTRIBS
  if (!ok && !nearby(player, thing)) {
#else
  if (!ok) {
#endif				/* EX_PUBLIC_ATTRIBS */
    /* if it's not examinable and we're not near it, we can only get the
     * name and the owner.
     */
    tp = tbuf;
    safe_str(object_header(player, thing), tbuf, &tp);
    safe_str((char *)" is owned by ", tbuf, &tp);
    safe_str(object_header(player, Owner(thing)), tbuf, &tp);
    *tp = '\0';
    notify(player, tbuf);
    return;
  }

  if (ok) {
    notify(player, object_header(player, thing));
#ifdef FLAGS_ON_EXAMINE
    notify(player, flag_description(player, thing));
#endif
  }

#ifdef EX_PUBLIC_ATTRIBS
  if (!brief) {
    a = atr_get_noparent(thing, "DESCRIBE");
    if (a) {
      r = safe_uncompress(a->value);
      notify(player, r);
      free(r);
    }
  }
#endif

  if (ok) {
    notify(player,
	   tprintf("Owner: %s  Key: %s  %s: %d",
		   db[db[thing].owner].name,
		   unparse_boolexp(player, db[thing].key, 0),
		   MONIES, Pennies(thing)));
    if (Typeof(thing) != TYPE_PLAYER)
      notify(player, tprintf("Parent: %s", 
			     object_header(player, db[thing].parent)));
    notify(player, tprintf("Zone: %s", object_header(player, Zone(thing))));

    if (Typeof(thing) == TYPE_PLAYER)
      notify(player, tprintf("Page Key: %s", 
			     unparse_boolexp(player, db[thing].usekey, 0)));
    else
      notify(player, tprintf("Use Key: %s", 
			     unparse_boolexp(player, db[thing].usekey, 0)));

    if (Typeof(thing) == TYPE_ROOM)
      notify(player, tprintf("Teleport Key: %s", 
			     unparse_boolexp(player, db[thing].enterkey, 0)));
    else
      notify(player, tprintf("Enter Key: %s", 
			     unparse_boolexp(player, db[thing].enterkey, 0)));

    notify(player, powers_descriptions(thing));

#if (CHAT_SYSTEM >= 2)
    if (Typeof(thing) == TYPE_PLAYER)
      notify(player, channel_description(thing));
#endif				/* CHAT_SYSTEM */
  }

#ifdef EX_PUBLIC_ATTRIBS
  if (!brief) {
#else
  if (!brief && ok) {
#endif				/* EX_PUBLIC_ATTRIBS */
    look_atrs(player, thing, NULL);
  }

  /* show contents */
  if ((Contents(thing) != NOTHING) && (ok || !Opaque(thing))) {
    if (Typeof(thing) == TYPE_PLAYER)
      notify(player, "Carrying:");
    else 
      notify(player, "Contents:");
    DOLIST(content, Contents(thing)) {
      if (ok || controls(player, content) || 
	  (!Dark(content) && IS(content, TYPE_PLAYER, PLAYER_CONNECT)))
	notify(player, object_header(player, content));
    }
  }

  if (!ok) {
    /* if not examinable, just show obvious exits and name and owner */
    if (Typeof(thing) == TYPE_ROOM)
      look_exits(player, thing, "Obvious exits:");
    tp = tbuf;
    safe_str(object_header(player, thing), tbuf, &tp);
    safe_str((char *)" is owned by ", tbuf, &tp);
    safe_str(object_header(player, Owner(thing)), tbuf, &tp);
    *tp = '\0';
    notify(player, tbuf);
    return;
  }

  switch (Typeof(thing)) {
  case TYPE_ROOM:
    /* tell him about exits */
    if (db[thing].exits != NOTHING) {
      notify(player, "Exits:");
      DOLIST(exit_dbref, db[thing].exits)
	notify(player, object_header(player, exit_dbref));
    } else
      notify(player, "No exits.");
    /* print dropto if present */
    if (Location(thing) != NOTHING) {
      notify(player,
	     tprintf("Dropped objects go to: %s",
		     object_header(player, Location(thing))));
    }
    break;
  case TYPE_THING:
  case TYPE_PLAYER:
    /* print home */
    notify(player,
	   tprintf("Home: %s",
		   object_header(player, db[thing].exits)));	/* home */
    /* print location if player can link to it */
    if (Location(thing) != NOTHING)
      notify(player,
	     tprintf("Location: %s",
		     object_header(player, Location(thing))));
    break;
  case TYPE_EXIT:
    /* print source */
    switch (db[thing].exits) {
    case NOTHING:
      fprintf(stderr,
	      "*** BLEAH *** Weird exit %s(#%d) in #%d with source NOTHING.\n",
	      Name(thing), thing, Location(thing));
      break;
    case HOME:
      fprintf(stderr,
	      "*** BLEAH *** Weird exit %s(#%d) in #%d with source HOME.\n",
	      Name(thing), thing, Location(thing));
      break;
    default:
      notify(player,
	     tprintf("Source: %s",
		     object_header(player, db[thing].exits)));
      break;
    }
    /* print destination */
    switch (Location(thing)) {
    case NOTHING:
      notify(player, "Destination: *UNLINKED*");
      break;
    case HOME:
      notify(player, "Destination: *HOME*");
      break;
    default:
      notify(player,
	     tprintf("Destination: %s",
		     object_header(player, Location(thing))));
      break;
    }
    break;
  default:
    /* do nothing */
    break;
  }
}

void do_score(player)
    dbref player;
{

  notify(player,
	 tprintf("You have %d %s.",
		 Pennies(player),
		 Pennies(player) == 1 ? MONEY : MONIES));
}

void do_inventory(player)
    dbref player;
{
  dbref thing;
  if ((thing = Contents(player)) == NOTHING) {
    notify(player, "You aren't carrying anything.");
  } else {
    notify(player, "You are carrying:");
    DOLIST(thing, thing) {
      notify(player, unparse_object(player, thing));
    }
  }

  do_score(player);
}

void do_find(player, name, argv)
    dbref player;
    const char *name;
    char *argv[];
{
    dbref i;
    int count = 0;
    int bot = 0;
    int top = db_top;

    if (options.daytime) {
	notify(player, "Sorry, that command has been temporarily disabled.");
	return;
    }

    if (!payfor(player, FIND_COST)) {
	notify(player, tprintf("Finds cost %d %s.", FIND_COST,
			       ((FIND_COST == 1) ? MONEY : MONIES)));
	return;
    } 

    /* determinte range */
    if (argv[1] && *argv[1])
	bot = atoi(argv[1]);
    if (bot < 0)
	bot = 0;
    if (argv[2] && *argv[2])
	top = atoi(argv[2]) + 1;
    if (top > db_top)
	top = db_top;

    for (i = bot; i < top; i++) {
	if ((Typeof(i) != TYPE_EXIT) && controls(player, i) &&
	    (!*name || string_match(db[i].name, name))) {
	    notify(player, object_header(player, i));
	    count++;
	}
    }
    notify(player, tprintf("*** %d objects found ***", count));
}

/* check the current location for bugs */
void do_sweep(player, arg1)
    dbref player;
    const char *arg1;
{
  char tbuf1[BUFFER_LEN];
  char *p;
  dbref here = db[player].location;
  int connect_flag = 0;
  int here_flag = 0;
  int inven_flag = 0;
  int exit_flag = 0;

  if (here == NOTHING)
    return;

  if(arg1 && *arg1) {
    if (string_prefix(arg1, "connected"))
      connect_flag = 1;
    else if (string_prefix(arg1, "here"))
      here_flag = 1;
    else if (string_prefix(arg1, "inventory"))
      inven_flag = 1;
    else if (string_prefix(arg1, "exits"))
      exit_flag = 1;
    else {
      notify(player, "Invalid parameter.");
      return;
    }
  }

  if (!inven_flag && !exit_flag) {
    notify(player, "Listening in ROOM:");

    if(connect_flag) {
      /* only worry about puppet and players who's owner's are connected */
      if (Connected(here) || (Puppet(here) && Connected(Owner(here)))) {
	if(Typeof(here) == TYPE_PLAYER) {
	  notify(player, tprintf("%s is listening", Name(here)));
	} else {
	  notify(player , tprintf("%s [owner: %s] is listening.", 
				  Name(here), db[Owner(here)].name));
	}
      }
    } else {
      if (Hearer(here) || Listener(here)) {
	if (Connected(here))
	  notify(player, tprintf("%s (this room) [speech]. (connected)",
				 Name(here)));
	else
	  notify(player, tprintf("%s (this room) [speech].", Name(here)));
      }
      if (Commer(here))
	notify(player, tprintf("%s (this room) [commands].", Name(here)));
      if (Audible(here))
	notify(player, tprintf("%s (this room) [broadcasting].", 
			       Name(here)));
    }

    for (here = db[here].contents; here != NOTHING; here = db[here].next) {
      if(connect_flag) {
	/* only worry about puppet and players who's owner's are connected */
	if (Connected(here) || (Puppet(here) && Connected(Owner(here)))) {
	  if(Typeof(here) == TYPE_PLAYER) {
	    notify(player, tprintf("%s is listening", Name(here)));
	  } else {
	    notify(player , tprintf("%s [owner: %s] is listening.", 
				    Name(here), db[Owner(here)].name));
	  }
	}
      } else {
	if (Hearer(here) || Listener(here)) {
	  if (Connected(here))
	    notify(player, tprintf("%s [speech]. (connected)", Name(here)));
	  else
	    notify(player, tprintf("%s [speech].", Name(here)));
	} 
	if(Commer(here))
	  notify(player, tprintf("%s [commands].", Name(here)));
      }
    }
  }

  if (!connect_flag && !inven_flag && 
      (Typeof(Location(player)) == TYPE_ROOM)) {
    notify(player, "Listening EXITS:");
    if (db[Location(player)].flags & AUDIBLE) {
      /* listening exits only work if the room is AUDIBLE */
      for (here = db[Location(player)].exits; here != NOTHING;
	   here = db[here].next) {
	if (db[here].flags & AUDIBLE) {
	  strcpy(tbuf1, Name(here));
	  for (p = tbuf1; *p && (*p != ';'); p++)
	    ;
	  *p = '\0';
	  notify(player, tprintf("%s [broadcasting].", tbuf1));
	}
      }
    }
  }

  if (!here_flag && !exit_flag) {
    notify(player, "Listening in your INVENTORY:");

    for (here = Contents(player); here != NOTHING; here = db[here].next) {
      if(connect_flag) {
	/* only worry about puppet and players who's owner's are connected */
	if (Connected(here) || (Puppet(here) && Connected(Owner(here)))) {
	  if(Typeof(here) == TYPE_PLAYER) {
	    notify(player, tprintf("%s is listening", Name(here)));
	  } else {
	    notify(player , tprintf("%s [owner: %s] is listening.", 
				    Name(here), db[Owner(here)].name));
	  }
	}
      } else {
	if (Hearer(here) || Listener(here)) {
	  if (IS(here, TYPE_PLAYER, PLAYER_CONNECT))
	    notify(player, tprintf("%s [speech]. (connected)", Name(here)));
	  else
	    notify(player, tprintf("%s [speech].", Name(here)));
	} 
	if(Commer(here))
	  notify(player, tprintf("%s [commands].", Name(here)));
      }
    }
  }
}

#ifdef PLAYER_LOCATE
void do_whereis(player, name)
    dbref player;
    const char *name;
{
  dbref thing;
  if (*name == '\0') {
    notify(player, "You must specify a valid player name.");
    return;
  }
  if ((thing = lookup_player(name)) == NOTHING) {
    notify(player, "That player does not seem to exist.");
    return;
  }

  if (!Can_Locate(player, thing)) {
    notify(player, "That player wishes to have some privacy.");
    notify(thing, tprintf("%s tried to locate you and failed.",
	   db[player].name));
    return;
  }
  notify(player,
	 tprintf("%s is at: %s.", Name(thing),
		 unparse_object(player, Location(thing))));
  if (!See_All(player))
    notify(thing,
	   tprintf("%s has just located your position.",
		   db[player].name));
  return;

}
#endif /* PLAYER_LOCATE */

void do_entrances(player, where, argv, val)
    dbref player;
    const char *where;
    char *argv[];
    int val;		   /* 0 all, 1 exits, 2 things, 3 players, 4 rooms */
{
  dbref place;
  dbref counter;
  int exc, tc, pc, rc;		/* how many we've found */
  int exd, td, pd, rd;		/* what we're looking for */
  int bot = 0;
  int top = db_top;

  if (options.daytime) {
    notify(player, "Sorry, that command has been temporarily disabled.");
    return;
  }

  if(!payfor(player, FIND_COST)) {
    notify(player, tprintf("You don't have enough %d %s to do that.",
			   FIND_COST,
			   ((FIND_COST == 1) ? MONEY : MONIES)));
    return;
  }

  if(!where || !*where) {
    if((place = db[player].location) == NOTHING)
      return;
  } else {
    init_match(player, where, NOTYPE);
    match_everything();
    if ((place =  noisy_match_result()) == NOTHING)
      return;
  }

  if (!controls(player, place) && !Search_All(player)) {
    notify(player, "Permission denied.");
    return;
  }

  /* figure out what we're looking for */
  switch (val) {
  case 1:
    exd = 1;
    td = pd = rd = 0;
    break;
  case 2:
    td = 1;
    exd = pd = rd = 0;
    break;
  case 3:
    pd = 1;
    exd = td = rd = 0;
    break;
  case 4:
    rd = 1;
    exd = td = pd = 0;
    break;
  default:
    exd = td = pd = rd = 1;
  }

  exc = tc = pc = rc = 0;

  /* determine range */
  if (argv[1] && *argv[1])
    bot = atoi(argv[1]);
  if (bot < 0)
    bot = 0;
  if (argv[2] && *argv[2])
    top = atoi(argv[2]) + 1;
  if (top > db_top)
    top = db_top;
  
  for (counter = bot; counter < top; counter++) {
    if (controls(player, place) || controls(player, counter)) {
      switch(Typeof(counter)) {
      case TYPE_EXIT:
	if (exd) {
	  if(db[counter].location == place) {
	    notify(player, tprintf("%s(#%d) [from: %s(#%d)]", Name(counter),
				   counter, Name(db[counter].exits),
				   db[counter].exits));
	    exc++;
	  }
	}
	break;
      case TYPE_ROOM:
	if (rd) {
	  if(db[counter].location == place) {
	    notify(player, tprintf("%s(#%d) [dropto]", 
				   Name(counter), counter));
	    rc++;
          }
	}
	break;
      case TYPE_THING:
	if (td) {
	  if (db[counter].exits == place) {
	    notify(player, tprintf("%s(#%d) [home]",
				   Name(counter), counter));
	    tc++;
	  }
	}
	break;
      case TYPE_PLAYER:
	if (pd) {
	  if (db[counter].exits == place) {
	    notify(player, tprintf("%s(#%d) [home]", 
				   Name(counter), counter));
	    pc++;
	  }
	}
	break;
      }
    }
  }

  if (!exc && !tc && !pc && !rc) {
    notify(player, "Nothing found.");
    return;
  } else {
    notify(player, "----------  Entrances Done  ----------");
    notify(player,
	  tprintf("Totals: Rooms...%d  Exits...%d  Objects...%d  Players...%d",
		  rc, exc, tc, pc));
    return;
  }
}

void decompile_atrs(player, thing, name, pattern)
     dbref player;
     dbref thing;
     const char *name;
     const char *pattern;
{
  ALIST *ptr;

  for (ptr = db[thing].list; ptr; ptr = AL_NEXT(ptr)) {
    if (AL_BAD(ptr)) continue;
    if (AL_FLAGS(ptr) & AF_DARK) continue;
    if ((AL_FLAGS(ptr) & AF_MDARK) && !See_All(player)) continue;
    if (local_wild_match(pattern, AL_NAME(ptr))) {
      if (atr_match(AL_NAME(ptr)) != NULL)
	notify(player, tprintf("@%s %s=%s", AL_NAME(ptr), name,
			       uncompress(AL_STR(ptr))));
      else
	notify(player, tprintf("&%s %s=%s", AL_NAME(ptr), name,
			       uncompress(AL_STR(ptr))));
    }
  }
}

void do_decompile(player, name)
  dbref player;
  const char *name;
{
  dbref thing;
  const char *object = NULL;
  char tbuf[BUFFER_LEN];
  char *attrib;

  /* @decompile must always have an argument */
  if (*name == '\0') {
    notify(player, "What do you want to @decompile?");
    return;
  }

  attrib = (char *) index(name, '/');
  if (attrib)
    *attrib++ = '\0';

  /* find object */
  init_match(player, name, NOTYPE);
  match_everything();
  if ((thing = noisy_match_result()) == NOTHING)
      return;

  if (!Can_Examine(player, thing)) {
    notify(player, "Permission denied.");
    return;
  }

  if (Flags(thing) & GOING) {
    notify(player, "Garbage is garbage.");
    return;
  }

  /* if we have an attribute arg specified, wild match on it */
  if (attrib && *attrib) {
    decompile_atrs(player, thing, Name(thing), attrib);
    return;
  }

  /* else we have a full decompile */

  /* determine creation and what we call the object */
  switch (Typeof(thing)) {
  case TYPE_PLAYER:
    object = "me";
    break;
  case TYPE_THING:
    object = Name(thing);
    notify(player, tprintf("@create %s", object));
    break;
  case TYPE_ROOM:
    notify(player, tprintf("@dig/teleport %s", Name(thing)));
    object = "here";
    break;
  case TYPE_EXIT:
    object = Name(thing);
    notify(player, tprintf("@open %s", object));
    break;
  }

  if (Mobile(thing)) {
    if (GoodObject(db[thing].exits))
      notify(player, tprintf("@link %s = #%d", object, db[thing].exits));
    else if (db[thing].exits == HOME)
      notify(player, tprintf("@link %s = HOME", object));
  } else {
    if (GoodObject(Location(thing)))
      notify(player, tprintf("@link %s = #%d", object, Location(thing)));
    else if (Location(thing) == HOME)
      notify(player, tprintf("@link %s = HOME", object));
  }

  if (GoodObject(db[thing].zone))
    notify(player, tprintf("@chzone %s = #%d", object, db[thing].zone));
  if (GoodObject(db[thing].parent) && (Typeof(thing) != TYPE_PLAYER))
    notify(player, tprintf("@parent %s=#%d", object, db[thing].parent));

  sprintf(tbuf, "%s", unparse_boolexp(player, db[thing].key, 1));
  if (strcmp(tbuf, "*UNLOCKED*"))
    notify(player, tprintf("@lock %s = %s", object, tbuf));
  sprintf(tbuf, "%s", unparse_boolexp(player, db[thing].enterkey, 1));
  if (strcmp(tbuf, "*UNLOCKED*"))
    notify(player, tprintf("@lock/enter %s = %s", object, tbuf));
  sprintf(tbuf, "%s", unparse_boolexp(player, db[thing].usekey, 1));
  if (strcmp(tbuf, "*UNLOCKED*"))
    notify(player, tprintf("@lock/use %s = %s", object, tbuf));

  decompile_flags(player, thing, object);
  decompile_atrs(player, thing, object, "*");
}
