/*****************************************************************************
	      idled - idle login killer daemon for the Linux OS
       (C)opyright 1994-1995, B. Scott Burkett, All Rights Reserved
 ***************************************************************************** 
	MODULE: idled.c
 ***************************************************************************** 
       HISTORY:	initial coding		12/18/94		bsb
		added idled.conf	12/21/94		bsb

 *****************************************************************************
  
   Copyright (c), 1994-1995, by B. Scott Burkett
   All rights reserved.
  
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:
  
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
   3. All modifications to the source code must be clearly marked as
      such.  Binary redistributions based on modified source code
      must be clearly marked as modified versions in the documentation
      and/or other materials provided with the distribution.
   4. All advertising materials mentioning features or use of this software
      must display the following acknowledgment:
        This product includes software developed by B. Scott Burkett and
        other unpaid contributors.
   5. The name of B. Scott Burkett may not be used to endorse or promote
      products derived from this software without specific prior
      written permission.
  
   THIS SOFTWARE IS PROVIDED BY B. SCOTT BURKETT AND CONTRIBUTORS "AS IS" AND
   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED.  IN NO EVENT SHALL B. SCOTT BURKETT OR CONTRIBUTORS BE LIABLE
   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
   SUCH DAMAGE.
 
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <utmp.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <utime.h>
#include <errno.h>
#include <ctype.h>
#include <regex.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/syslog.h>
#include <sys/ioctl.h>
#include <sys/param.h>

#include "idled.h"

int main(int argc, char *argv[])
{
	Init( argc );
	ParseCmdLine(argc, argv);
	Report();
	Process();
	return(0);
}

/***************************************************************************
 *                                                                         *
 * void ParseCmdLine(int ac, char *av[]);                                  *
 *                                                                         *
 * This function is responsible for parsing the command line and      	   *
 * setting any parameters wihch are configurable from the command line.    *
 *                                                                         *
 ***************************************************************************/

void ParseCmdLine(int ac, char *av[])
{
	int i, j;
	char	*p;

	if(ac==1) {
		printf("\n\ridled " VERSION "\n\r");
		printf( COPYRIGHT "\n\r\n\r");

		printf("USAGE: ");
		printf("idled [options]\n\r");
		printf("             -s(leep delay between checks (default=60))\n\r");
		printf("             -w(arn first - no initial kill!)\n\r");
		printf("\n\rConsult the idled manual page for more information.\n\r\n\r");
		exit(255);
	}

	for(i = 1; i < ac; i++)
	{
		p = av[i];
		if(*p == '-' || *p == '/')
		{
			for(j = 1; *(p + j); j+=strlen(p + 1))
			{
				switch(tolower(*(p + j)))
				{

					case 's': _sleepdelay=atoi(p+2);
						  break;

					case 'd': _debug = TRUE;
						  break;
           					  
            				case 'w': _warnfirst = TRUE;
                      			  	  break;
						  
				}
			}
		}
		else			     /* unknown parameter	     */
			return;
	}

	return;
}

/***************************************************************************
 *                                                                         *
 * void Report(void);                                                      *
 *                                                                         *
 * Displays runtime parameters to syslog                                                                        *
 *                                                                         *
 ***************************************************************************/

void Report(void)
{
	sprintf(buf, "Sleep delay is %d\n", _sleepdelay);
	err_sys(buf);

	if(_warnfirst)
		err_sys("Warn-first mode");
			else
				err_sys("No-warn mode");

}

/***************************************************************************
 *                                                                         *
 * void Init( int ac );                                                    *
 *                                                                         *
 * Responsible for initialization.  Sets signal handlers and sets up for   *
 * daemon operation.                                                       *
 *                                                                         *
 ***************************************************************************/

void Init( int ac )
{
	if( ac == 1 )
		return;

	be_daemon();
	SetSignals();
	err_sys("-----------------------");
	err_sys("Initializing idler daemon");
	err_sys( COPYRIGHT );

	ReadCFGFile();
}

void ReadCFGFile(void)
{
	FILE *cfgfile;
	int  ndx=0;

	if((cfgfile = fopen("/etc/idled.conf", "rt")) == NULL)
	{
		err_sys("Unable to open /etc/idled.conf");
		exit(0);
	}

	while (1)
	{
		fgets(buf, 80, cfgfile);
		if(feof(cfgfile)) break;

		if(buf[0] == '#') continue;
		if(buf[0] == ';') continue;
		if(buf[0] == '%') continue;
		if(buf[0] == '\r') continue;
		if(buf[0] == '\n') continue;
		if(buf[0] == ' ') continue;
		if(buf[0] == '\t') continue;

		sscanf(buf, "%s %s %d %d %c %d", tmpcfg._username,
						 tmpcfg._ttydevice,
						 &tmpcfg._idle,
						 &tmpcfg._grace,
						 &tmpcfg._notifybymail,
						 &tmpcfg._sessionlimit);

		tmpcfg._notifybymail = toupper(tmpcfg._notifybymail);
	
		if(ndx <= WARNED_TABLE_SIZE)
		{
			memcpy( &cfg[ndx], &tmpcfg, sizeof(_CFG));
			ndx++;
		}

	}

	highentry=ndx-1;
	memset(&cfg[ndx], '\0', sizeof(_CFG));
	fclose(cfgfile);

}

/***************************************************************************
 *                                                                         *
 * void SetSignals(void);                                                  *
 *                                                                         *
 * Sets up signal handlers for the daemon. 				   *
 *                                                                         *
 ***************************************************************************/

void SetSignals(void)
{
	signal(SIGHUP, cleanup);
	signal(SIGTERM, cleanup);
	signal(SIGINT, cleanup);
	signal(SIGSEGV, cleanup);
	signal(SIGFPE, cleanup);
	signal(SIGQUIT, cleanup);
}

/***************************************************************************
 *                                                                         *
 * void cleanup(int signum);                                               *
 *                                                                         *
 * Generic signal handler                   			           *
 *                                                                         *
 ***************************************************************************/

void cleanup(int signum)
{
	char tmpbuf[80];

	sprintf(tmpbuf, "Received signal #%d - exiting!", signum);
	err_sys(tmpbuf);
	exit(0);
}

/***************************************************************************
 *                                                                         *
 * void Process(void);                                                     *
 *                                                                         *
 * Main processing loop of the daemon       			           *
 *                                                                         *
 ***************************************************************************/

void Process(void)
{
	int ndx;

	while( 1 )
	{
		sleep(_sleepdelay);
		ReadCFGFile();

		while( 1 )
		{
			userrec = getutent();
	
			if(userrec == NULL) 
				break;
	
			if(userrec->ut_type == USER_PROCESS)
			{
				/* In 1.00a, we just checked the time. */
				/* if( GetTimes() >= _maxidletime ) */

				if ( JudgementDay( &ndx ) )
					Guillotine( ndx );
			}
		
		}

		endutent();
	}

}

int JudgementDay( int *cfgndx )
{
	int ndx=0;
	int _match=FALSE;

	/* Find the entry in the CFG table that matches this user */
	for(ndx=0; ndx<=highentry; ndx++)
	{
		/* Compile regular expression pattern */
		re_comp( cfg[ndx]._username );

		/* Check for user name match */
		if(re_exec( userrec->ut_user))
		{
			/* Got a match, let's check the terminal line */
			/* Compile regular expression pattern */
			re_comp( cfg[ndx]._ttydevice );
			
			/* Check for terminal device match */
			if(re_exec( userrec->ut_line ))
			{
				/* This is the one! */
				_match = TRUE;
				break;
			}
		}
	}

	if(!_match)
		return(FALSE);
	else {
		*cfgndx = ndx;
		return(TRUE);
	}
}

/***************************************************************************
 *                                                                         *
 * time_t GetTimes(void);                                                  *
 *                                                                         *
 * Returns the time difference between the change time of the user's tty   *
 * device and the current time.  This is the idle time (essentially).      *
 *                                                                         *
 ***************************************************************************/

time_t GetTimes(void)
{
	char buf[80];
	struct stat statbuf;
	time_t currtime;

	/* Build device file name */
	sprintf(buf, "/dev/%s", userrec->ut_line);

	/* Stat the device file */ 
	stat(buf, &statbuf);

	/* Get current time */
	currtime = time(NULL);

	/* Calculate Idletime */
	return(currtime-statbuf.st_atime);
}

/***************************************************************************
 *                                                                         *
 * int CheckNoIdleout(void);                                               *
 *                                                                         *
 * Checks /etc/noidleout for tty device and account name.                  *
 * Returns TRUE if a match is found, else FALSE.                           *
 *                                                                         *
 ***************************************************************************/

int CheckNoidleout(void)
{
	char _leavealone=FALSE;
	FILE *tmpfil;
	char device[20];

	if((tmpfil= fopen("/etc/noidleout", "rt")) == NULL) 
		return(0);

	sprintf(device, "/dev/%s", userrec->ut_line);

	while (1)
	{
		fgets(buf, 80, tmpfil);
		if(feof(tmpfil)) break;

		if(!strncmp(buf, device, strlen(device)))
			_leavealone = TRUE;

		if(!strncmp(buf, userrec->ut_user, strlen(userrec->ut_user)))
			_leavealone = TRUE;

		if(_leavealone)		
		{
			fclose(tmpfil);
			return(1);
		}
	}

	fclose(tmpfil);
	return(0);
}

/***************************************************************************
 *                                                                         *
 * void Guillotine( int cfgndx );                                          *
 *                                                                         *
 * Function responsible for issuing warnings and killing processes.        *
 *                                                                         *
 ***************************************************************************/

void Guillotine( int cfgndx )
{
	int ndx, _killnow=FALSE, mins=0;
	time_t   currtime = time(NULL);

	/* Check the criteria */
	if(cfg[cfgndx]._idle == 0)
		return;

	/* Check session time limit, if necessary */
	if( cfg[cfgndx]._sessionlimit != 0)
	{
		/* Calculate in minutes */
		mins = (currtime-userrec->ut_time) / 60;
		if(cfg[cfgndx]._sessionlimit <= mins)
		{
			sprintf(buf, "%s reached session limit of %d minutes",
					userrec->ut_user, mins);
			err_sys(buf);
			SendFile("/etc/idled.limit");
			_killnow = TRUE;			
		}

	}

	/* Has this process already been warned? */
	if((ndx=BeenWarned( userrec->ut_pid )) != -1 && !_killnow)
	{
		if(GetTimes() < cfg[cfgndx]._grace)
			return;

		_killnow=TRUE;
		WarnNoMore( ndx );
	}
	else
	if(!_killnow)
		if(GetTimes() < cfg[cfgndx]._idle )
			return;
	   
	/* Make sure the process is still there before sending any signals! */
	
	/* Issue a warning, or "behead" the process */
	if( !_warnfirst || _killnow )
	{
		if(!ProcActive(userrec->ut_pid))
			return;			
		
		currtime=time(NULL);	
		sprintf(buf, "Killing %s on %s (%s)", userrec->ut_user,
					         userrec->ut_line,
						 ctime(&currtime));
		err_sys(buf);
		MessageKill();

		kill(userrec->ut_pid, SIGHUP);
		sleep(2);

		if(ProcActive(userrec->ut_pid)) {
			kill(userrec->ut_pid, SIGTERM);
			sleep(2);
		}

		if(ProcActive(userrec->ut_pid)) {
			kill(userrec->ut_pid, SIGKILL);
			sleep(2);
		}
		
		if(ProcActive(userrec->ut_pid)) {
			sprintf(buf, "Possible runaway process: USER=%s PID=%d", 
					userrec->ut_user, userrec->ut_pid);
			err_sys(buf);
		}
		else
		{
			/* Ok, notify by mail if necessary */
			if (cfg[cfgndx]._notifybymail == 'Y')
			{
				sprintf(buf, "/bin/mail %s < /etc/idled.mail",
					userrec->ut_user);
				err_sys(buf);
				system(buf);
			}
		}
	}
	else
	{
		if(!ProcActive(userrec->ut_pid))
			return;
			
		currtime=time(NULL);	
		sprintf(buf, "Warning %s on %s (%s)", userrec->ut_user,
					         userrec->ut_line,
						 ctime(&currtime));
		err_sys(buf);
		MessageWarn();
		MarkAsWarned( userrec->ut_pid );
	}
}

/***************************************************************************
 *                                                                         *
 * void MessageWarn(void);                                                 *
 *                                                                         *
 * Sends the /etc/idled.warn file to the user's tty.                       *
 *                                                                         *
 ***************************************************************************/

void MessageWarn(void)
{
	SendFile("/etc/idled.warn");
}

/***************************************************************************
 *                                                                         *
 * void MessageKill(void);                                                 *
 *                                                                         *
 * Sends the /etc/idled.kill file to the user's tty.                       *
 *                                                                         *
 ***************************************************************************/

void MessageKill(void)
{
	SendFile("/etc/idled.kill");
}

/***************************************************************************
 *                                                                         *
 * void be_daemon(void);                                                   *
 *                                                                         *
 * My version of Coherent's bedaemon() libmisc function.  This turns the   *
 * calling process into a daemon.                                          *
 *                                                                         *
 ***************************************************************************/

void be_daemon(void)
{
	int	childpid, fd;

	/* If we were started by init (process 1) from the /etc/inittab file
	   there's no need to detach. */

	if (getppid() != 1)
	{
		/* Ignore the terminal stop signals */

		signal(SIGTTOU, SIG_IGN);
		signal(SIGTTIN, SIG_IGN);
		signal(SIGTSTP, SIG_IGN);
	
		/* If we were not started in the background, fork and
	   	let the parent exit.  This also guarantees the first child
	   	is not a process group leader. */

		if ( (childpid = fork()) < 0)
			err_sys("can't fork first child");
		else if (childpid > 0)
			exit(0);	/* parent */

		/* First child process.
	   	   Disassociate from controlling terminal and process group.
	   	   Ensure the process can't reacquire a new controlling 
		   terminal. */
	
		if (setpgrp() == -1) 
			err_sys("can't change SID/PGID");

  		/* immune from pgrp leader death */
		signal(SIGHUP, SIG_IGN); 

		if ( (childpid = fork()) < 0)
			err_sys("can't fork second child");
		else if (childpid > 0)
			exit(0);	/* first child */

		/* second child
	 	   Close any open files descriptors. */

		for (fd = 0; fd < NOFILE; fd++)
			close(fd);

		/* probably got set to EBADF from a close */
		errno = 0;		

		/* Move the current directory to root, to make sure we
	 	   aren't on a mounted filesystem. */

		chdir("/");

		/* Clear any inherited file mode creation mask. */
		umask(0);

		/* Ignore the status of dead child processes */
	 	signal(SIGCHLD, SIG_IGN);
	}
}

/***************************************************************************
 *                                                                         *
 * void err_sys(char *msg);                                                *
 *                                                                         *
 * Logs the passed msg buffer into the syslog daemon                       *
 *                                                                         *
 ***************************************************************************/

void err_sys(char *msg)
{
	openlog("idled", LOG_PID|LOG_CONS, LOG_DAEMON);
	syslog(LOG_PID|LOG_ERR, msg);
	closelog();
}

/***************************************************************************
 *                                                                         *
 * int BeenWarned(int pid);                                                *
 *                                                                         *
 * Returns the ndx number (0-WARNED_TABLE_SIZE) of a process that has      *
 * already been issued a warning.  Returns -1 if not in the table.         *
 *                                                                         *
 ***************************************************************************/

int BeenWarned(int pid)
{
	int ndx;

	for(ndx=0; ndx<=highentry; ndx++)
	{
		if(warned[ndx].pid == pid)
			return(ndx);
	}

	return(-1);

}

/***************************************************************************
 *                                                                         *
 * void MarkAsWarned(int pid);                                             *
 *                                                                         *
 * Creates an entry in the warned table for the passed process ID #.       *
 * This denotes that the process has been issued a warning.                *
 *                                            				   * 
 ***************************************************************************/

void MarkAsWarned(int pid)
{
	int ndx;

	for(ndx=0; ndx<=highentry; ndx++)
	{
		if(warned[ndx].pid == 0)
		{
			warned[ndx].pid = pid;
			warned[ndx].wflag = TRUE;
			return;
		}
	}

}

/***************************************************************************
 *                                                                         *
 * void WarnNoMore( int ndx );                                             *
 *                                                                         *
 * Resets an entry in the warned table.                                    *
 *                                            				   * 
 ***************************************************************************/

void WarnNoMore( int ndx )
{
	warned[ndx].pid   = 0;
	warned[ndx].wflag = 0;
}

/***************************************************************************
 *                                                                         *
 * void SendFile(char *filename);                                          *
 *                                                                         *
 * Sends the contents of filename to the user's tty device.  This function *
 * is called by MessageWarn() and MessageKill().		           * 
 *                                            				   * 
 ***************************************************************************/

void SendFile(char *filename)
{
	FILE 	*tmp;
	int	pid, status, fd;
	char 	localhost[20], termdevice[30];
	time_t  currtime = time(NULL);
	struct	utimbuf timebuf;
	
	struct	stat statbuf;

	pid = fork();

	if(pid < 0) 
	{
		err_sys("Unable to fork child process");
		return;
	}

	if(pid)
	{
		wait(&status);
		return;
	}
	else
	{
		sprintf(termdevice, "/dev/%s", userrec->ut_line);

		if((tmp = fopen(filename, "rt")) == NULL)
		{
			err_sys("Unable to open warning message file!");
			exit(0);
		}

		gethostname(localhost, sizeof(localhost));

		if((fd = open(termdevice, O_WRONLY)) == -1)
		{
			err_sys("Unable to open user terminal!");	
			exit(1);
		}
		
		sprintf(buf, "\n\r\n\rMessage from idledaemon.%s at %s\n\r\n\r",
			localhost, ctime(&currtime));
		write(fd, buf, strlen(buf));

		while (1)
		{
			fgets(buf, 80, tmp);
			if(feof(tmp)) break;
	
			write(fd, buf, strlen(buf));
	
		}
	
		fclose(tmp);
		fflush(tty);
		close(fd);

		/* Update access time on device file.  We have to explicitly
		   do this, as it seems some (most?) of the Linux calls do not
		   modify the atime member of a file when accessed.  This seems
		   to be the case with the write() system call, at least.
	        */

		currtime=time(NULL);
		stat(termdevice, &statbuf);
		timebuf.actime = currtime;
		timebuf.modtime = statbuf.st_mtime;	/* Same as before */
		utime(termdevice, (struct utimbuf *)&timebuf);

		exit(0);
	}

}

/***************************************************************************
 *                                                                         *
 * int ProcActive(int pid);                                                *
 *                                                                         *
 * Returns TRUE if passed PID is active.  This requires the proc psuedo    *
 * file system to be in place and mounted.                   		   * 
 *                                            				   * 
 ***************************************************************************/

int ProcActive(int pid)
{
	char filename[80];
	struct stat statbuf;

	/* Build proc_fs style filename */
	sprintf(filename, "/proc/%d", pid);

	if(stat(filename, &statbuf))
		return(FALSE);
	else
		return(TRUE);
}

