/********************************************************

	Classes for reading AVIs
	Copyright 2000 Eugene Smith (divx@euro.ru)
	Last modified: 08.07.2000

*********************************************************/


#include <avifmt.h>
#include <default.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include <pthread.h>
#define MAXSTREAMS 2

struct req
{
    offset_t offset;
    int size;
    enum status {
    BUFFER_CLEAN=0,
    BUFFER_WAITING,
    BUFFER_WRITING,
    BUFFER_READY
    };
    status st;
    unsigned id;//number of stream
    unsigned position;//position in stream
    char* memory;
};

class Cache
{
    unsigned m_size;
    unsigned m_used;
    AVIINDEXENTRY* m_tables[MAXSTREAMS];
    unsigned m_lengths[MAXSTREAMS];
    unsigned m_positions[MAXSTREAMS];
    pthread_mutex_t mutex_in, mutex_out;
    pthread_cond_t cond_in, cond_out;    
    pthread_t thread;
    req* req_buf;
    int buffers_busy;
    int m_quit;
    enum status {
    CLEAR,
    ASYNC
    };
    status m_status;    
protected:
    int Prefetch(unsigned id, unsigned position);
    int Update();//removes unused cache entries and requests newer ones
public:
    Cache(unsigned size): m_size(size), m_used(0), m_quit(0), buffers_busy(0), m_status(CLEAR)
    {
	memset(m_tables, 0, sizeof(m_tables));
	req_buf=new req[size];
    }
    int AddStream(unsigned id, AVIINDEXENTRY *table, unsigned position, unsigned length)
    {
        if(id>=MAXSTREAMS)
	{
	    printf("Only %d cached streams are supported\n", MAXSTREAMS);
	    return -1;
	}
	if(position>=length)
	    return -1;
	m_tables [id]=table;
	m_lengths [id]=length;
	m_positions [id]=position;
	return 0;
    }
    ~Cache();
    int Clear();
    int Create(int fd);
    int Read(char* buffer, unsigned id, unsigned pos, unsigned size, unsigned offset=0);
};


class InputStream
{
protected:
    int m_fd;
    offset_t length;
    Cache* cache;
public:
    InputStream(int fd);
    ~InputStream();
    offset_t seek(offset_t offset)
    {
	return lseek(m_fd, offset, SEEK_SET);
    }
    offset_t seek_cur(offset_t offset)
    {
	return lseek(m_fd, offset, SEEK_CUR);
    }
    offset_t pos()
    {
	return lseek(m_fd, 0, SEEK_CUR);
    }
    offset_t len() const
    {
	return length;
    }
    int eof()
    {
	return (pos()>=length);
    }		
    int read_int()
    {
	int i;
	read(m_fd, &i, 4);
	return i;
    }
    int Async() //enter asynchronous mode
    {
	if(cache==0)return -1;
	return cache->Create(m_fd);
    }
    int AddStream(unsigned id, AVIINDEXENTRY *table, unsigned position, unsigned length)
    {
	if(cache==0)return -1;
	return cache->AddStream(id, table, position, length);
    }	
//  int Read(offset_t offset, char* buffer, unsigned size);//asynchronous, obsolete
    int Read(char* buffer, unsigned size);//synchronous
    //from cache:
    int Read(char* buffer, unsigned id, unsigned pos, unsigned size, unsigned offset =0)
    {
	if(cache==0)return -1;
	return cache->Read(buffer, id, pos, size, offset);
    }
    int Clear()
    {
	if(cache==0)return -1;
	return cache->Clear();
    }    
};    
    
class AviStream
{
protected:
    AVIStreamHeader m_header;
public:
    enum StreamType{
    Audio,
    Video,
    Other
    };
};

class AviReadStream : public AviStream
{
friend class AviReadFile;
private:
    unsigned m_id;
    char* m_format;
    unsigned m_format_size;
    unsigned m_lockcount;
    unsigned m_position;
    unsigned m_ch_off;//chunk offset for streams with fixed sample size
    unsigned m_sample;//for streams with fixed sample size: total number of samples sent
    unsigned m_table_type;
    offset_t start_offset;
protected:
    AVIINDEXENTRY* chunk_table;
    unsigned table_size;
private:    
    InputStream* m_is;
//    StreamType m_type;
public:
    AviReadStream():m_table_type(0), m_lockcount(1), m_format(0), chunk_table(0), m_is(0){}
    ~AviReadStream();
    HRESULT Eof() const;
    HRESULT Init(unsigned m_id, InputStream* is);
//    HRESULT Release();
    StreamType GetType() const;
    unsigned Length() const {return m_header.dwLength;}
    HRESULT Seek(unsigned pos);
    double SeekToTime(double time);
    HRESULT ReadSample(char* buffer, unsigned bufsize, unsigned samples=1);
    HRESULT QuerySampleSize(unsigned* size);
    HRESULT SetStreamProperties(offset_t start_offset, AVIINDEXENTRY* chunk_table,
	    unsigned table_size, unsigned table_type);
    HRESULT GetVideoFormatInfo(void* bi);
    HRESULT GetAudioFormatInfo(void* bi, char** ext);
    HRESULT GetFrameFlags(int* flags);
    double GetFrameTime() const {return m_header.dwScale / m_header.dwRate;}
    double GetTime() const;
    double GetLength() const;
};
    

class AviReadFile
{
private:
    int m_fd;
    InputStream* m_is;
    MainAVIHeader m_header;
    AviReadStream* m_streams;
    offset_t chunk_offset;
    AVIINDEXENTRY* chunk_table;
    unsigned table_size;
    void InitChunkTable(offset_t pos);
    void InitAltChunkTable(offset_t pos);
public:
    AviReadFile():m_fd(-1),m_streams(0),chunk_table(0), m_is(0){}
    ~AviReadFile();
    HRESULT OpenFile(const char* name);
    unsigned StreamCount();
    unsigned VideoStreamCount();
    unsigned AudioStreamCount();
    AviReadStream* GetStream(unsigned stream_id);
    AviReadStream* GetStream(unsigned stream_id, AviStream::StreamType type);
    HRESULT GetFileHeader(MainAVIHeader* header);
    HRESULT CloseFile();
};
class AviWriteFile;

class AviWriteStream : public AviStream
{
friend class AviWriteFile;

protected:
    AviWriteFile* m_file;
    int m_status;
    int m_fd;
    enum StreamState
    {
	ValidFormat = 1,
	ValidHeader = 2,
	ValidType = 4,
	ValidStream = 7
    };
    StreamType m_type;
    
    char* m_format;
    unsigned m_forsize;
    int m_ckid;
    HRESULT SetType(StreamType type)
    {
	if(type==Other)
	    return -1;
	m_status|=ValidType;
	m_type=type;
	
	return 0;
    }	 
public:
    AviWriteStream(AviWriteFile* file, int ckid);
    
    ~AviWriteStream();
    //
    // these two should be called before data insertion begins
    //
    //
    // for video streams, specify time in microsecs per frame in frame_rate
    // for audio streams - count of bytes per second
    //   
    HRESULT SetHeader(int handler, int frame_rate, int samplesize=0, int quality=0, int flags=0)
    {
	if(!(m_status & ValidType))
	    return -1;
	
	m_status|=ValidHeader;
//	m_header=header;
	memset(&m_header, 0, sizeof(m_header));
	m_header.fccType=((m_type==Video)?streamtypeVIDEO:streamtypeAUDIO);
	m_header.fccHandler=handler;
	m_header.dwFlags=flags;
	if(m_type==Video)
	{
	    m_header.dwRate=1000000;
	    m_header.dwScale=frame_rate;
	}
	else
	{
	    m_header.dwRate=frame_rate;
	    m_header.dwScale=samplesize;
	}        
	m_header.dwSampleSize=samplesize;
	m_header.dwQuality=quality;
	m_header.dwFlags=flags;
	return 0;
    }	
    HRESULT SetFormat(const char* format, unsigned format_size)
    {
	if(format==0)
	    return -1;
	m_status|=ValidFormat;
	m_format=new char[format_size];
	m_forsize=format_size;
	memcpy(m_format, format, format_size);
	if(m_type==Video)
	{
	    m_header.rcFrame.right=*(int*)(format+4);
	    m_header.rcFrame.bottom=*(int*)(format+8);
	}    
	return 0;
    }	
    StreamType GetType() const
    {
    	return m_type;
    }	    

    HRESULT AddChunk(const char* chunk, unsigned size, unsigned flags=0);

    unsigned GetLength()//0 on error
    {
	if(m_status!=ValidStream)
	    return 0;
	return m_header.dwLength;
    }	    
};            

//
//maintains global file info
//
class AviWriteFile
{
friend class AviWriteStream;
private:
    AviWriteStream** m_streams;
    int m_strcnt;
    MainAVIHeader m_header;
    int m_status;
    AVIINDEXENTRY* m_index;
    int m_indsize;    
protected:
    int m_fd;
public:
    AviWriteFile(): m_streams(0), m_strcnt(0), m_status(0), m_indsize(0), m_index(0) {m_fd=0;}
    HRESULT Create(const char* name, int flags=0);
    AviWriteStream* AddStream(enum AviWriteStream::StreamType);
    HRESULT Close();
protected:
    //
    //called when data is added to stream
    //
    HRESULT AddChunk(offset_t offset, unsigned size, unsigned id, unsigned flags=0);    
};

    