/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { j2s } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { GlobalTestState } from "../harness/harness.js";
import {
  createSimpleTestkudosEnvironmentV3,
  createWalletDaemonWithClient,
  withdrawViaBankV3,
} from "../harness/helpers.js";
import { SyncService } from "../harness/sync.js";

/**
 * Run test for basic, bank-integrated withdrawal.
 */
export async function runWalletBackupBasicTest(t: GlobalTestState) {
  // Set up test environment

  const { commonDb, merchant, walletClient, bankClient, exchange } =
    await createSimpleTestkudosEnvironmentV3(t);

  const sync = await SyncService.create(t, {
    currency: "TESTKUDOS",
    annualFee: "TESTKUDOS:0.5",
    database: commonDb.connStr,
    fulfillmentUrl: "taler://fulfillment-success",
    httpPort: 8089,
    name: "sync1",
    paymentBackendUrl: merchant.makeInstanceBaseUrl(),
    uploadLimitMb: 10,
  });

  await sync.start();
  await sync.pingUntilAvailable();

  await walletClient.call(WalletApiOperation.AddBackupProvider, {
    backupProviderBaseUrl: sync.baseUrl,
    activate: false,
    name: sync.baseUrl,
  });

  {
    const bi = await walletClient.call(WalletApiOperation.GetBackupInfo, {});
    t.assertDeepEqual(bi.providers[0].active, false);
  }

  await walletClient.call(WalletApiOperation.AddBackupProvider, {
    backupProviderBaseUrl: sync.baseUrl,
    activate: true,
    name: sync.baseUrl,
  });

  {
    const bi = await walletClient.call(WalletApiOperation.GetBackupInfo, {});
    t.assertDeepEqual(bi.providers[0].active, true);
  }

  await walletClient.call(WalletApiOperation.RunBackupCycle, {});

  {
    const bi = await walletClient.call(WalletApiOperation.GetBackupInfo, {});
    console.log(bi);
    t.assertDeepEqual(
      bi.providers[0].paymentStatus.type,
      "insufficient-balance",
    );
  }

  await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:10",
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  await walletClient.call(WalletApiOperation.RunBackupCycle, {});

  {
    const bi = await walletClient.call(WalletApiOperation.GetBackupInfo, {});
    console.log(bi);
  }

  await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:5",
  });

  await walletClient.call(WalletApiOperation.RunBackupCycle, {});

  {
    const bi = await walletClient.call(WalletApiOperation.GetBackupInfo, {});
    console.log(bi);
  }

  const backupRecovery = await walletClient.call(
    WalletApiOperation.ExportBackupRecovery,
    {},
  );

  const txs = await walletClient.call(WalletApiOperation.GetTransactions, {});
  console.log(`backed up transactions ${j2s(txs)}`);

  const { walletClient: walletClient2 } = await createWalletDaemonWithClient(
    t,
    {
      name: "w2",
    },
  );

  // Check that the second wallet is a fresh wallet.
  {
    const bal = await walletClient2.call(WalletApiOperation.GetBalances, {});
    t.assertTrue(bal.balances.length === 0);
  }

  await walletClient2.call(WalletApiOperation.ImportBackupRecovery, {
    recovery: backupRecovery,
  });

  await walletClient2.call(WalletApiOperation.RunBackupCycle, {});

  // Check that now the old balance is available!
  {
    const bal = await walletClient2.call(WalletApiOperation.GetBalances, {});
    t.assertTrue(bal.balances.length === 1);
    console.log(bal);
  }

  // Now do some basic checks that the restored wallet is still functional
  {
    const txs = await walletClient2.call(
      WalletApiOperation.GetTransactions,
      {},
    );
    console.log(`restored transactions ${j2s(txs)}`);
    const bal1 = await walletClient2.call(WalletApiOperation.GetBalances, {});

    t.assertAmountEquals(bal1.balances[0].available, "TESTKUDOS:14.1");

    await withdrawViaBankV3(t, {
      walletClient: walletClient2,
      bankClient,
      exchange,
      amount: "TESTKUDOS:10",
    });

    await exchange.runWirewatchOnce();

    await walletClient2.call(
      WalletApiOperation.TestingWaitTransactionsFinal,
      {},
    );

    const txs2 = await walletClient2.call(
      WalletApiOperation.GetTransactions,
      {},
    );
    console.log(`tx after withdraw after restore ${j2s(txs2)}`);

    const bal2 = await walletClient2.call(WalletApiOperation.GetBalances, {});

    t.assertAmountEquals(bal2.balances[0].available, "TESTKUDOS:23.82");
  }
}

runWalletBackupBasicTest.suites = ["wallet", "wallet-backup"];
// See https://bugs.taler.net/n/7598
runWalletBackupBasicTest.experimental = true;
