/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import { HttpStatusCode, TalerError, assertUnreachable } from "@gnu-taler/taler-util";
import { Loading, useTranslationContext } from "@gnu-taler/web-util/browser";
import { VNode, h } from "preact";
import { ErrorLoadingMerchant } from "../../../../components/ErrorLoadingMerchant.js";
import { CreatedSuccessfully } from "../../../../components/notifications/CreatedSuccessfully.js";
import { useOrderDetails } from "../../../../hooks/order.js";
import { NotFoundPageOrAdminCreate } from "../../../notfound/index.js";
import { Entity } from "./index.js";
import { LoginPage } from "../../../login/index.js";

interface Props {
  entity: Entity;
  onConfirm: () => void;
  onCreateAnother?: () => void;
}

export function OrderCreatedSuccessfully({
  entity,
  onConfirm,
  onCreateAnother,
}: Props): VNode {
  const result = useOrderDetails(entity.response.order_id)
  const { i18n } = useTranslationContext();

  if (!result) return <Loading />
  if (result instanceof TalerError) {
    return <ErrorLoadingMerchant error={result} />
  }
  if (result.type === "fail") {
    switch(result.case) {
      case HttpStatusCode.NotFound: {
        return <NotFoundPageOrAdminCreate />
      }
      case HttpStatusCode.BadGateway: {
        return <div>Failed to obtain a response from the exchange</div>;
      }
      case HttpStatusCode.GatewayTimeout: {
        return (
          <div>The merchant's interaction with the exchange took too long</div>
        );
      }
      case HttpStatusCode.Unauthorized: {
        return <LoginPage />
      }
      default: {
        assertUnreachable(result)
      }
    }
  }

  const url = result.body.order_status === "unpaid" ?
    result.body.taler_pay_uri :
    result.body.contract_terms.fulfillment_url

  return (
    <CreatedSuccessfully
      onConfirm={onConfirm}
      onCreateAnother={onCreateAnother}
    >
      <div class="field is-horizontal">
        <div class="field-label is-normal">
          <label class="label">
            <i18n.Translate>Amount</i18n.Translate>
          </label>
        </div>
        <div class="field-body is-flex-grow-3">
          <div class="field">
            <p class="control">
              <input
                class="input"
                readonly
                value={entity.request.order.amount}
              />
            </p>
          </div>
        </div>
      </div>
      <div class="field is-horizontal">
        <div class="field-label is-normal">
          <label class="label">
            <i18n.Translate>Summary</i18n.Translate>
          </label>
        </div>
        <div class="field-body is-flex-grow-3">
          <div class="field">
            <p class="control">
              <input
                class="input"
                readonly
                value={entity.request.order.summary}
              />
            </p>
          </div>
        </div>
      </div>
      <div class="field is-horizontal">
        <div class="field-label is-normal">
          <label class="label">
            <i18n.Translate>Order ID</i18n.Translate>
          </label>
        </div>
        <div class="field-body is-flex-grow-3">
          <div class="field">
            <p class="control">
              <input class="input" readonly value={entity.response.order_id} />
            </p>
          </div>
        </div>
      </div>
      <div class="field is-horizontal">
        <div class="field-label is-normal">
          <label class="label">
            <i18n.Translate>Payment URL</i18n.Translate>
          </label>
        </div>
        <div class="field-body is-flex-grow-3">
          <div class="field">
            <p class="control">
              <input class="input" readonly value={url} />
            </p>
          </div>
        </div>
      </div>
    </CreatedSuccessfully>
  );
}
