/*
 * Copyright © 2001 Red Hat, Inc.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of Red Hat not be used in advertising or
 * publicity pertaining to distribution of the software without specific,
 * written prior permission.  Red Hat makes no representations about the
 * suitability of this software for any purpose.  It is provided "as is"
 * without express or implied warranty.
 *
 * RED HAT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL RED HAT
 * BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN 
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Author:  Owen Taylor, Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 *          Olivier Fourdan: adapted to the "multi-channel" concept
 *	        Benedikt Meurer: Fixed some bugs.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <X11/Xlib.h>
#include <X11/Xmd.h>

#include <glib.h>

#include "mcs-common.h"

McsManagerCheck
mcs_manager_check_running(Display *display, int screen)
{
    Atom selection_atom;
    Atom mc_selection_atom;
    gchar *buffer;
    Window w1, w2;

    buffer = g_strdup_printf("_MCS_S%d", screen);
    mc_selection_atom = XInternAtom(display, buffer, False);
    g_free(buffer);

    buffer = g_strdup_printf("_XSETTINGS_S%d", screen);
    selection_atom = XInternAtom(display, buffer, False);
    g_free(buffer);

    w1 = XGetSelectionOwner(display, selection_atom);
    w2 = XGetSelectionOwner(display, mc_selection_atom);

    if(w1 && w2)
    {
        return MCS_MANAGER_BOTH;
    }
    else if(w1)
    {
        return MCS_MANAGER_STD;
    }
    else if(w2)
    {
        return MCS_MANAGER_MULTI_CHANNEL;
    }
    return MCS_MANAGER_NONE;
}

McsSetting *
mcs_setting_copy(McsSetting *setting)
{
    McsSetting *result;

    g_return_val_if_fail(setting != NULL, NULL);
    g_return_val_if_fail(setting->name != NULL, NULL);
    g_return_val_if_fail(setting->channel_name != NULL, NULL);

    if ((result = g_new0(McsSetting, 1)) == NULL)
        return(NULL);

    if ((result->name = g_strdup(setting->name)) == NULL)
        goto err;

    if ((result->channel_name = g_strdup(setting->channel_name)) == NULL)
        goto err;

    result->type = setting->type;

    switch (setting->type)
    {
        case MCS_TYPE_INT:
            result->data.v_int = setting->data.v_int;
            break;
        case MCS_TYPE_COLOR:
            result->data.v_color = setting->data.v_color;
            break;
        case MCS_TYPE_STRING:
            if (!setting->data.v_string)
                goto err;
 
	    result->data.v_string = g_strdup(setting->data.v_string);
            break;
    }

    result->last_change_serial = setting->last_change_serial;

    return result;

err:
    if(result->name)
        g_free(result->name);

    if(result->channel_name)
        g_free(result->channel_name);

    g_free(result);

    return(NULL);
}

McsList *
mcs_list_copy(McsList *list)
{
    McsList *new;
    McsList *new_node;
    McsList *new_iter;
    McsList *old_iter;

    new = NULL;
    new_iter = NULL;

    for (old_iter = list; old_iter != NULL; old_iter = old_iter->next)
    {
        if ((new_node = g_new(McsList, 1)) == NULL)
            goto error;

        if ((new_node->setting = mcs_setting_copy(old_iter->setting)) == NULL)
        {
            g_free(new_node);
            goto error;
        }

        if(new_iter)
            new_iter->next = new_node;
        else
            new = new_node;

        new_iter = new_node;
    }

    return new;

  error:
    mcs_list_free(new);
    return NULL;
}

int mcs_setting_equal(McsSetting * setting_a, McsSetting * setting_b)
{
    g_return_val_if_fail (setting_a != NULL, 0);
    g_return_val_if_fail (setting_b != NULL, 0);
    g_return_val_if_fail (setting_a->name != NULL, 0);
    g_return_val_if_fail (setting_b->name != NULL, 0);
    g_return_val_if_fail (setting_a->channel_name != NULL, 0);
    g_return_val_if_fail (setting_b->channel_name != NULL, 0);

    if(setting_a->type != setting_b->type)
        return 0;

    if(strcmp(setting_a->name, setting_b->name) != 0)
        return 0;

    if(strcmp(setting_a->channel_name, setting_b->channel_name) != 0)
        return 0;

    switch (setting_a->type)
    {
        case MCS_TYPE_INT:
            return setting_a->data.v_int == setting_b->data.v_int;
        case MCS_TYPE_COLOR:
            return (setting_a->data.v_color.red == setting_b->data.v_color.red && setting_a->data.v_color.green == setting_b->data.v_color.green && setting_a->data.v_color.blue == setting_b->data.v_color.blue && setting_a->data.v_color.alpha == setting_b->data.v_color.alpha);
        case MCS_TYPE_STRING:
	    if (setting_a->data.v_string && setting_b->data.v_string)
	    {
                return strcmp(setting_a->data.v_string, setting_b->data.v_string) == 0;
	    }
	    else
	    {
	        return 0;
	    }
    }
    return 0;
}

void mcs_setting_free(McsSetting * setting)
{
    g_return_if_fail (setting != NULL);

    if((setting->type == MCS_TYPE_STRING) && (setting->data.v_string))
        g_free(setting->data.v_string);

    if(setting->name)
        g_free(setting->name);

    if(setting->channel_name)
        g_free(setting->channel_name);

    g_free(setting);
}

void mcs_list_free(McsList * list)
{
    while(list)
    {
        McsList *next = list->next;

        mcs_setting_free(list->setting);
        g_free(list);

        list = next;
    }
}

McsResult mcs_list_insert(McsList ** list, McsSetting * setting)
{
    McsList *node;
    McsList *iter;
    McsList *last = NULL;

    g_return_val_if_fail (setting != NULL, MCS_FAILED);
    g_return_val_if_fail (setting->name != NULL, MCS_FAILED);
    g_return_val_if_fail (setting->channel_name != NULL, MCS_FAILED);

    node = g_malloc(sizeof *node);
    if(!node)
        return MCS_NO_MEM;
    node->setting = setting;

    iter = *list;
    while(iter)
    {
        int cmp = strcmp(setting->name, iter->setting->name);

        if(cmp < 0)
            break;
        else if(cmp == 0)
        {
            g_free(node);
            return MCS_DUPLICATE_ENTRY;
        }

        last = iter;
        iter = iter->next;
    }

    if(last)
        last->next = node;
    else
        *list = node;

    node->next = iter;

    return MCS_SUCCESS;
}

McsResult mcs_list_delete(McsList ** list, const gchar *name)
{
    McsList *iter;
    McsList *last = NULL;

    g_return_val_if_fail (name != NULL, MCS_FAILED);

    for (iter = *list; iter != NULL; iter = iter->next)
    {
        if(strcmp(name, iter->setting->name) == 0)
        {
            if(last)
                last->next = iter->next;
            else
                *list = iter->next;

            mcs_setting_free(iter->setting);
            g_free(iter);

            return MCS_SUCCESS;
        }

        last = iter;
    }

    return MCS_FAILED;
}

McsSetting *mcs_list_lookup(McsList * list, const gchar *name)
{
    McsList *iter;

    g_return_val_if_fail (name != NULL, NULL);

    iter = list;
    while(iter)
    {
        if(strcmp(name, iter->setting->name) == 0)
            return iter->setting;

        iter = iter->next;
    }

    return NULL;
}

gchar
mcs_byte_order(void)
{
#ifndef G_BYTE_ORDER
    CARD32 myint = 0x01020304;
    return (*(gchar *)&myint == 1) ? MSBFirst : LSBFirst;
#else
    if (G_BYTE_ORDER == G_LITTLE_ENDIAN)
        return(LSBFirst);
    else
        return(MSBFirst);
#endif /* !G_BYTE_ORDER */
}
