/*
 **********************************************************************
 *     midi.c - /dev/midi interface for emu10k1 driver
 *     Copyright 1999, 2000 Creative Labs, Inc.
 *
 **********************************************************************
 *
 *     Date                 Author          Summary of changes
 *     ----                 ------          ------------------
 *     October 20, 1999     Bertrand Lee    base code release
 *
 **********************************************************************
 *
 *     This program is free software; you can redistribute it and/or
 *     modify it under the terms of the GNU General Public License as
 *     published by the Free Software Foundation; either version 2 of
 *     the License, or (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public
 *     License along with this program; if not, write to the Free
 *     Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
 *     USA.
 *
 **********************************************************************
 */

#ifdef MODULE
#define __NO_VERSION__
#include <linux/module.h>
#else
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif

#include "hwaccess.h"
#include "mycommon.h"
#include "icardmid.h"
#include "cardmo.h"
#include "cardmi.h"
#include "midi.h"


static spinlock_t midi_spinlock __attribute((unused));
extern spinlock_t sblive_spinlock;

static int emu10k1_midi_open(struct inode *inode, struct file *file)
{
	int minor = MINOR(inode->i_rdev);
	struct sblive_hw *sb_hw = sblive_devs;
	struct sblive_mididevice *midi_dev;

	DPF("emu10k1_midi_open() called\n");

	/* Check for correct device to open */
	while (sb_hw && sb_hw->midi_num != minor)
	  sb_hw = sb_hw->next;
	if (!sb_hw)
	  return -ENODEV;

	/* Wait for device to become free */
	down(&sb_hw->open_sem);
	while (sb_hw->open_mode & (file->f_mode << FMODE_MIDI_SHIFT)) 
	{
		if (file->f_flags & O_NONBLOCK) 
		{
			up(&sb_hw->open_sem);
			return -EBUSY;
		}
		
		up(&sb_hw->open_sem);
		interruptible_sleep_on(&sb_hw->open_wait);
		
		if (signal_pending(current))
		  return -ERESTARTSYS;
		
		down(&sb_hw->open_sem);
	}

	midi_dev = (struct sblive_mididevice *) kmalloc(sizeof(*midi_dev), GFP_KERNEL);
	if (!midi_dev) 
	{
		DPF(" MIDIDEVICEOBJ alloc fail.");
		return -EINVAL;
	}
	
#ifdef MEMTEST
	DPD("midi.c: kmalloc: [%p]\n", midi_dev);
#endif
	
	midi_dev->sb_hw = sb_hw;
	midi_dev->midiouthandle = 0;
	midi_dev->midiinhandle = 0;
	midi_dev->mistate = MIDIIN_STATE_STOPPED;
	init_waitqueue(&midi_dev->oWait);
	init_waitqueue(&midi_dev->iWait);
	midi_dev->ird = 0;
	midi_dev->iwr = 0;
	midi_dev->icnt = 0;
	midi_dev->pmiHdrList = NULL;

	if (file->f_mode & FMODE_READ) 
	{
		struct midi_openinfo dsCardMidiOpenInfo;
		struct midi_hdr *midihdr1;
		struct midi_hdr *midihdr2;

		dsCardMidiOpenInfo.CallbackFn = midiCallbackFn;
		dsCardMidiOpenInfo.refdata = (u32) midi_dev;
		
		if (sblive_mpuinOpen(sb_hw, &dsCardMidiOpenInfo, &midi_dev->midiinhandle) 
		    != CTSTATUS_SUCCESS) 
		{
			DPF("sblive_mpuinOpen failed.\n");
			kfree(midi_dev);
			return -ENODEV;
		}
		
		/* Add two buffers to receive sysex buffer */
		if (midiInAddBuffer(midi_dev, &midihdr1) != CTSTATUS_SUCCESS) 
		{
			kfree(midi_dev);
			return -ENODEV;
		}
		
		if (midiInAddBuffer(midi_dev, &midihdr2) != CTSTATUS_SUCCESS) 
		{
			osListRemove((struct sblive_list **) &midi_dev->pmiHdrList, (struct sblive_list *) midihdr1);
			kfree(midihdr1->lpData);
			kfree(midihdr1);
			kfree(midi_dev);
			return -ENODEV;
		}
	}
	
	if (file->f_mode & FMODE_WRITE) 
	{
		struct midi_openinfo dsCardMidiOpenInfo;

		dsCardMidiOpenInfo.CallbackFn = midiCallbackFn;
		dsCardMidiOpenInfo.refdata = (u32) midi_dev;
		
		if (sblive_mpuoutOpen(sb_hw, &dsCardMidiOpenInfo, &midi_dev->midiouthandle)
		    != CTSTATUS_SUCCESS) 
		{
			DPF("sblive_mpuoutOpen failed.\n");
			kfree(midi_dev);
			return -ENODEV;
		}
	}

	file->private_data = (void *) midi_dev;

	sb_hw->open_mode |= (file->f_mode << FMODE_MIDI_SHIFT) & (FMODE_MIDI_READ | FMODE_MIDI_WRITE);

	up(&sb_hw->open_sem);

	MOD_INC_USE_COUNT;
	return 0;
}

static void emu10k1_midi_release(struct inode *inode, struct file *file)
{
	struct sblive_mididevice *midi_dev = (struct sblive_mididevice *) file->private_data;
	struct sblive_hw *sb_hw = midi_dev->sb_hw;

	struct wait_queue wait = { current, NULL };

	DPF("emu10k1_midi_release() called\n");
	
	if (file->f_mode & FMODE_WRITE) 
	{
		current->state = TASK_INTERRUPTIBLE;
		add_wait_queue(&midi_dev->oWait, &wait);
		
		for (;;) 
		{
			int status = CTSTATUS_ERROR;

			status = sblive_mpuoutClose(sb_hw, midi_dev->midiouthandle);
			
			if (status == CTSTATUS_STILLPLAYING) 
			{
				if (file->f_flags & O_NONBLOCK) 
				{
					remove_wait_queue(&midi_dev->oWait, &wait);
					current->state = TASK_RUNNING;
					return;
				}
			}
			
			if (status == CTSTATUS_SUCCESS)
			  break;
			
			schedule();
		}
	}
	
	if (file->f_mode & FMODE_READ) 
	{
		struct midi_hdr *midihdr;

		if (midi_dev->mistate == MIDIIN_STATE_STARTED) 
		{
			sblive_mpuinStop(sb_hw, midi_dev->midiinhandle);
			midi_dev->mistate = MIDIIN_STATE_STOPPED;
			DPF("MIDI in stopped\n");
		}
		
		sblive_mpuinReset(sb_hw, midi_dev->midiinhandle);
		sblive_mpuinClose(sb_hw, midi_dev->midiinhandle);
		
		DPF("MIDI in closed\n");
		
		while ((midihdr = (struct midi_hdr *) osListGetNext((struct sblive_list *) midi_dev->pmiHdrList, NULL))) 
		{
			osListRemove((struct sblive_list **) &midi_dev-> pmiHdrList, (struct sblive_list *) midihdr);
			kfree(midihdr->lpData);
			kfree(midihdr);
		}
	}
	
	kfree(midi_dev);

	down(&sb_hw->open_sem);
	sb_hw->open_mode &= ~((file->f_mode << FMODE_MIDI_SHIFT) & (FMODE_MIDI_READ | FMODE_MIDI_WRITE));
	up(&sb_hw->open_sem);
	wake_up(&sb_hw->open_wait);
	
	MOD_DEC_USE_COUNT;
}


static int emu10k1_midi_read(struct inode *ino, struct file *file, char *buffer, int count)
{
	struct sblive_mididevice *midi_dev = (struct sblive_mididevice *) file->private_data;
	ssize_t ret = 0;
	u16 cnt;
	unsigned long flags;

	DPD("emu10k1_midi_read() called, count %d", count);

	if (!access_ok(VERIFY_WRITE, buffer, count))
	  return -EFAULT;

	if (midi_dev->mistate == MIDIIN_STATE_STOPPED) 
	{
		if (sblive_mpuinStart(midi_dev->sb_hw, midi_dev->midiinhandle) 
		    != CTSTATUS_SUCCESS) 
		{
			DPF("sblive_mpuinStart failed.\n");
			return -EINVAL;
		}
		
		midi_dev->mistate = MIDIIN_STATE_STARTED;
	}

	while (count > 0) 
	{
		spin_lock_irqsave(&sblive_spinlock, flags);

		cnt = MIDIIN_BUFLEN - midi_dev->ird;
		
		if (midi_dev->icnt < cnt)
		  cnt = midi_dev->icnt;
		
		spin_unlock_irqrestore(&sblive_spinlock, flags);
		
		if (cnt > count)
		  cnt = count;
		
		if (cnt <= 0) 
		{
			if (file->f_flags & O_NONBLOCK)
			  return ret ? ret : -EAGAIN;
			DPF(" Go to sleep...");
			
			interruptible_sleep_on(&midi_dev->iWait);
			
			if (signal_pending(current))
				return ret ? ret : -ERESTARTSYS;
			
			continue;
		}
		
		if (copy_to_user(buffer, midi_dev->iBuf + midi_dev->ird, cnt)) {
			DPF(" copy_to_user failed.");
			return ret ? ret : -EFAULT;
		}
		
		spin_lock_irqsave(&sblive_spinlock, flags);
		
		midi_dev->ird += cnt;
		midi_dev->ird %= MIDIIN_BUFLEN;
		midi_dev->icnt -= cnt;
		
		spin_unlock_irqrestore(&sblive_spinlock, flags);
		
		count -= cnt;
		buffer += cnt;
		ret += cnt;
	}
	
	return ret;
}


static int emu10k1_midi_write(struct inode *ino, struct file *file, const char *buffer, int count)
{
	struct sblive_mididevice *midi_dev = (struct sblive_mididevice *) file->private_data;
	struct midi_hdr *midihdr;
	ssize_t ret = 0;
	unsigned long flags;

	DPD("emu10k1_midi_write() called, count=%d\n", count);

	if (!access_ok(VERIFY_READ, buffer, count))
	  return -EFAULT;

	spin_lock_irqsave(&sblive_spinlock, flags);

	midihdr = (struct midi_hdr *) kmalloc(sizeof(struct midi_hdr *), GFP_KERNEL);
	if (!midihdr) 
	{
		spin_unlock_irqrestore(&midi_spinlock, flags);
		return -EINVAL;
	}
	
	midihdr->bufferlength = count;
	midihdr->bytesrecorded = 0;
	midihdr->flags = 0;
	midihdr->next = NULL;
	
	midihdr->lpData = (u8 *) kmalloc(count, GFP_KERNEL);
	if (!midihdr->lpData) 
	{
		DPF("midihdr->lpData alloc failed\n");
		kfree(midihdr);
		
		spin_unlock_irqrestore(&midi_spinlock, flags);
		return -EINVAL;
	}
	
	/* FIXME: COPY WITH LOCK HELD IS UNSAFE!!!!!!!!!!!!!!
	 */
	
	if (copy_from_user(midihdr->lpData, buffer, count)) 
	{
		kfree(midihdr->lpData);
		kfree(midihdr);
		
		spin_unlock_irqrestore(&midi_spinlock, flags);
		return ret ? ret : -EFAULT;
	}

	if (sblive_mpuoutAddBuffer(midi_dev->sb_hw, midi_dev->midiouthandle, midihdr) != CTSTATUS_SUCCESS) 
	{
		DPF("sblive_mpuoutAddBuffer failed.\n");
		kfree(midihdr->lpData);
		kfree(midihdr);
		
		spin_unlock_irqrestore(&midi_spinlock, flags);
		return -EINVAL;
	}

	spin_unlock_irqrestore(&sblive_spinlock, flags);

	return count;
}


static int emu10k1_midi_poll(struct inode *inode, struct file *file, int sel_type, select_table *wait)
{
	DPF("emu10k1_midi_poll() called\n");
	return 0;
}


int midiCallbackFn(unsigned long msg, unsigned long refdata, unsigned long param)
{
	struct sblive_mididevice *midi_dev = (struct sblive_mididevice *) refdata;
	u32 *pmsg = (u32 *) param;
	struct midi_hdr *midihdr = NULL;
	unsigned long flags;

	DPF("midiCallbackFn!");
	
	spin_lock_irqsave(&sblive_spinlock, flags);

	switch (msg) 
	{
	case ICARDMIDI_OUTLONGDATA:
		midihdr = (struct midi_hdr *) pmsg[2];
		kfree(midihdr->lpData);
		kfree(midihdr);
		break;

	case ICARDMIDI_INLONGDATA:
		DPF(" ICARDMIDI_INLONGDATA");
		midihdr = (struct midi_hdr *) pmsg[2];
		
		memcpy(&midi_dev->iBuf[midi_dev->iwr], midihdr->lpData, midihdr->bytesrecorded);
		midi_dev->iwr = (midi_dev->iwr + midihdr->bytesrecorded) % MIDIIN_BUFLEN;
		midi_dev->icnt += midihdr->bytesrecorded;
		midi_dev->icnt %= MIDIIN_BUFLEN;
		
		if (midi_dev->mistate == MIDIIN_STATE_STARTED) 
		{
			initMidiHdr(midihdr);
			sblive_mpuinAddBuffer(midi_dev->sb_hw->card_mpuin, midi_dev->midiinhandle, midihdr);
			wake_up(&midi_dev->iWait);
		}
		break;

	case ICARDMIDI_INDATA:
		DPF(" ICARDMIDI_INDATA");
		{
			u8 *pBuf = (u8 *) & pmsg[1];
			u16 i, bytesvalid = pmsg[2];

			for (i = 0; i < bytesvalid; i++)
			  midi_dev->iBuf[midi_dev->iwr++] = pBuf[i];
			midi_dev->icnt += bytesvalid;
			midi_dev->icnt %= MIDIIN_BUFLEN;
		}
		
		wake_up(&midi_dev->iWait);
		break;

	default:		/* Unknown message */
		DPF("Unknown message\n");
		return CTSTATUS_ERROR;
	}

	spin_unlock_irqrestore(&sblive_spinlock, flags);
	
	return CTSTATUS_SUCCESS;
}


/* Local functions */
int midiInAddBuffer(struct sblive_mididevice *midi_dev, struct midi_hdr **midihdrptr)
{
	struct midi_hdr *midihdr;

	midihdr = (struct midi_hdr *) kmalloc(sizeof(struct midi_hdr *), GFP_KERNEL);
	if (!midihdr)
	  return -EINVAL;
	
	initMidiHdr(midihdr);
	
	midihdr->lpData = (u8 *) kmalloc(MIDIIN_BUFLEN, GFP_KERNEL);	
	if (!midihdr->lpData) 
	{
		DPF("midihdr->lpData alloc fail.");
		kfree(midihdr);
		return CTSTATUS_ERROR;
	}
	
	if (sblive_mpuinAddBuffer(midi_dev->sb_hw->card_mpuin, midi_dev->midiinhandle, midihdr) != CTSTATUS_SUCCESS) 
	{
		DPF("sblive_mpuinAddBuffer failed.\n");
		kfree(midihdr->lpData);
		kfree(midihdr);
		return CTSTATUS_ERROR;
	}
	
	*midihdrptr = midihdr;
	osListAttach((struct sblive_list **) &midi_dev->pmiHdrList, (struct sblive_list *) midihdr);

	return CTSTATUS_SUCCESS;
}

void initMidiHdr(struct midi_hdr * midihdr)
{
	midihdr->bufferlength = MIDIIN_BUFLEN;
	midihdr->bytesrecorded = 0;
	midihdr->flags = 0;
	midihdr->next = NULL;
}

/* End of local functions */


int midi_init(struct sblive_hw *sb_hw)
{
	/* Initialize CardMpuOut struct */
	sb_hw->card_mpuout = kmalloc(sizeof(struct sblive_mpuout), GFP_KERNEL);
	if (!sb_hw->card_mpuout) 
	{
		printk(KERN_WARNING "alloc struct sblive_mpuout: out of memory\n");
		return CTSTATUS_ERROR;
	}
	
#ifdef MEMTEST
	DPD("kmalloc: [%p]\n", sb_hw->card_mpuout);
#endif
	
	memset(sb_hw->card_mpuout, 0, sizeof(struct sblive_mpuout));

	sblive_mpuoutInit(sb_hw->card_mpuout, sb_hw);

	/* Initialize CardMpuIn struct */
	sb_hw->card_mpuin = kmalloc(sizeof(struct sblive_mpuin), GFP_KERNEL);
	if (!sb_hw->card_mpuin) 
	{
		printk(KERN_WARNING "alloc CARDMIDIOBJ: out of memory\n");
		return CTSTATUS_ERROR;
	}
	
#ifdef MEMTEST
	DPD("kmalloc: [%p]\n", sb_hw->card_mpuin);
#endif
	
	memset(sb_hw->card_mpuin, 0, sizeof(struct sblive_mpuin));

	sblive_mpuinInit(sb_hw->card_mpuin, sb_hw);

	return CTSTATUS_SUCCESS;
}

int midi_exit(struct sblive_hw *sb_hw)
{
	sblive_mpuoutExit(sb_hw);
	sblive_mpuinExit(sb_hw);
	kfree(sb_hw->card_mpuout);
	kfree(sb_hw->card_mpuin);
	
#ifdef MEMTEST
	DPD("kfree: [%p]\n", sb_hw->card_mpuout);
	DPD("kfree: [%p]\n", sb_hw->card_mpuin);
#endif
	
	sb_hw->card_mpuout = NULL;
	sb_hw->card_mpuin = NULL;

	return CTSTATUS_SUCCESS;
}


/* MIDI file operations */
struct file_operations emu10k1_midi_fops = 
{
	NULL,
	&emu10k1_midi_read,
	&emu10k1_midi_write,
	NULL,			/* readdir */
	&emu10k1_midi_poll,
	NULL,			/* ioctl */
	NULL,			/* mmap */
	&emu10k1_midi_open,
	&emu10k1_midi_release,
	NULL,			/* fsync */
	NULL,			/* fasync */
	NULL,			/* check_media_change */
};
