/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Author: Charles Kerr <charles@rebelbase.com>
 *
 * Copyright (C) 2000, 2001  Pan Development Team <pan@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*********************
**********************  Includes
*********************/

#include <config.h>

#include <libgnomeui/libgnomeui.h>

#include <pan/base/base-prefs.h>
#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/util-file.h>

#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/queue.h>
#include <pan/task-bodies.h>
#include <pan/task-save.h>
#include <pan/save-ui.h>
#include <pan/text.h>
#include <pan/util.h>

/*********************
**********************  Private Function Prototypes
*********************/

static gchar * last_save_as_path = NULL;

static gboolean append_groupname = FALSE;

static gboolean back_of_queue = TRUE;

static gboolean save_attachments = TRUE;
static gboolean use_subject = FALSE;
static gboolean use_encoded = TRUE;
static gboolean use_custom = FALSE;

static gboolean save_bodies = FALSE;
static gboolean use_message_id = FALSE;
static gboolean use_subject_for_body = FALSE;
static gboolean use_custom_for_body = FALSE;

/*********************
**********************  BEGINNING OF SOURCE
*********************/

typedef struct
{
	GtkWidget * default_dir_rb;
	GtkWidget * group_default_dir_rb;
	GtkWidget * user_specified_rb;
	GtkWidget * user_specified_entry;
	GtkWidget * append_groupname_tb;

	GtkWidget * body_filename_off_rb;
	GtkWidget * body_filename_message_id_rb;
	GtkWidget * body_filename_subject_rb;
	GtkWidget * body_filename_custom_rb;
	GtkWidget * body_filename_custom_entry;

	GtkWidget * decode_filename_encoded_rb;
	GtkWidget * decode_filename_subject_rb;
	GtkWidget * decode_filename_custom_rb;
	GtkWidget * decode_filename_custom_entry;
	GtkWidget * decode_filename_off_rb;

	GtkWidget * front_of_queue_rb;
	GtkWidget * back_of_queue_rb;

	Group  *group;

	GSList *gslists_of_unowned_articles;
}
SaveAsDialogStruct;

#define IS_ACTIVE(w) gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w))

static void
save_attachment_as_dialog_clicked_cb (GnomeDialog      * dialog,
                                      int                button,
                                      gpointer           user_data)
{
	SaveAsDialogStruct* save = (SaveAsDialogStruct*)user_data;
	GSList * l;
	gchar * path;
	gchar * custom_body_fname = NULL;
	gchar * custom_attachment_fname = NULL;
	debug_enter ("save_attachment_as_dialog_clicked_cb");

	/* cancel button */
	if (button==1)
	{
		pan_lock();
		gtk_widget_destroy (GTK_WIDGET(dialog));
		pan_unlock();
		return;
	}

	/**
	***  Get Path
	**/

	path = NULL;
	if (IS_ACTIVE(save->user_specified_rb)) {
		path = gtk_editable_get_chars (GTK_EDITABLE(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY (save->user_specified_entry))), 0, -1);
		g_strstrip (path);
		if (path!=NULL && *path=='~')
			replace_gstr (&path, g_strconcat (g_get_home_dir(), path+1, NULL));
	}
	else if ((save->group_default_dir_rb!=NULL) && IS_ACTIVE(save->group_default_dir_rb)) {
		path = g_strdup(save->group->download_dir);
	}
	else {
		path = g_strdup (download_dir);
	}

	if (IS_ACTIVE(save->append_groupname_tb))
		replace_gstr (&path, g_strdup_printf ("%s%c%s", path, G_DIR_SEPARATOR, save->group->name));

	/**
	***  Get Body Filename
	**/

	save_bodies = !IS_ACTIVE(save->body_filename_off_rb);
	use_message_id = IS_ACTIVE(save->body_filename_message_id_rb);
	use_subject_for_body = IS_ACTIVE(save->body_filename_subject_rb);
	use_custom_for_body = IS_ACTIVE(save->body_filename_custom_rb);
	custom_body_fname = gtk_editable_get_chars (GTK_EDITABLE(save->body_filename_custom_entry), 0, -1);

	/**
	***  Get Attachment Filenames
	**/

	save_attachments = !IS_ACTIVE(save->decode_filename_off_rb);
	use_encoded = IS_ACTIVE(save->decode_filename_encoded_rb);
	use_subject = IS_ACTIVE(save->decode_filename_subject_rb);
	use_custom = IS_ACTIVE(save->decode_filename_custom_rb);
	custom_attachment_fname = gtk_editable_get_chars (GTK_EDITABLE(save->decode_filename_custom_entry), 0, -1);

	/**
	***  Queue 'em
	**/

	for (l=save->gslists_of_unowned_articles; l!=NULL; l=l->next)
	{
		GSList * articles = (GSList*) l->data;
		const Article * article = ARTICLE(articles->data);
		const gchar * subject = article_get_subject (article);
		PanObject * task;

		task = task_save_new (save->group, subject, articles);

		if (save_attachments)
		{
			const gchar * filename = NULL;
			if (use_subject)
				filename = subject;
			else if (use_custom)
				filename = custom_attachment_fname;
			task_save_set_attachments (TASK_SAVE(task), path, filename);
		}

		if (save_bodies)
		{
			const gchar * filename = NULL;
			if (use_custom_for_body)
				filename = custom_body_fname;
			else if (use_message_id)
				filename = article_get_message_id (article);
			else if (use_subject_for_body)
				filename = subject;
			task_save_set_bodies (TASK_SAVE(task), path, filename);
		}


		task_save_validate_and_queue_or_self_destruct (TASK_SAVE(task), back_of_queue?-1:0);
	}

	/* cleanup */
	g_free (path);
	g_free (custom_body_fname);
	g_free (custom_attachment_fname);
	pan_lock();
	gtk_widget_destroy (GTK_WIDGET(dialog));
	pan_unlock();
	debug_exit ("save_attachment_as_dialog_clicked_cb");
}

static void
priority_toggled_cb (GtkToggleButton    * togglebutton,
                     gpointer              user_data)
{
	SaveAsDialogStruct *save = (SaveAsDialogStruct*) user_data;
	gboolean b = gtk_toggle_button_get_active (togglebutton);
	back_of_queue = togglebutton==GTK_TOGGLE_BUTTON(save->back_of_queue_rb) ? b : !b;
}

static void
append_groupname_toggled_cb (GtkToggleButton   *  togglebutton,
                             gpointer             user_data)
{
	append_groupname = gtk_toggle_button_get_active(togglebutton);
}


static void
save_attachment_as_dialog_destroy_cb (GtkObject    * o,
                                      gpointer       user_data)
{
	GSList * l;
	SaveAsDialogStruct *save = (SaveAsDialogStruct*) user_data;

	/* free the save-as-dialog utility structure */
	for (l=save->gslists_of_unowned_articles; l!=NULL; l=l->next)
	{
		GSList* l2 = (GSList*) l->data;
		/*g_slist_foreach (l2, (GFunc)article_free, NULL);*/
		g_slist_free (l2);
	}

	/* cleanup */
	g_slist_free (save->gslists_of_unowned_articles);
	g_free (save);
}

static void
user_specified_path_changed_cb (GtkEditable   * editable,
                                gpointer        user_data)
{
	replace_gstr (&last_save_as_path,
	              gtk_editable_get_chars (GTK_EDITABLE(editable), 0, -1));
}

static gint
entry_tickled_cb (GtkWidget * w, gpointer * event, gpointer user_data)
{
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(user_data), TRUE);
	return FALSE;
}

static void
create_save_attachment_as_dialog (Group * group,
                                  GSList * gslists_of_unowned_articles)
{
	gchar * pch;
	GtkWidget * w;
	GtkWidget * hbox;
	GtkWidget * vbox;
	GtkWidget * frame;
	GtkWidget * entry;
	GtkWidget * dialog;
	SaveAsDialogStruct * data;

	data = g_new0 (SaveAsDialogStruct, 1);
	data->group = group;
	data->gslists_of_unowned_articles = gslists_of_unowned_articles;

	pan_lock ();

	/* create & parent the dialog */
	dialog = gnome_dialog_new (_("Pan: Save Articles As"),
				   GNOME_STOCK_BUTTON_OK,
				   GNOME_STOCK_BUTTON_CANCEL,
				   NULL);
	gnome_dialog_set_default (GNOME_DIALOG(dialog), 0); /* ok */

	/* "Location" pane */
	frame = gtk_frame_new (_("Location"));
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
	vbox = gtk_vbox_new (FALSE, 0);

	/* global default directory */
	w = gtk_radio_button_new_with_label (NULL, download_dir);
	gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
	data->default_dir_rb = w;

	/* group default directory */
	if (is_nonempty_string (group->download_dir)) {
		w = gtk_radio_button_new_with_label_from_widget (
			GTK_RADIO_BUTTON(w), group->download_dir);
		gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, 0);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
		data->group_default_dir_rb = w;
	}

	/* user-specified directory */
	hbox = gtk_hbox_new (FALSE, 0);
	w = gtk_radio_button_new_from_widget (GTK_RADIO_BUTTON(w));
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	data->user_specified_rb = w;
	entry = gnome_file_entry_new ("Save As", _("Download Directory"));
	gtk_signal_connect (
		GTK_OBJECT(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(entry))),
		"focus_in_event",
		GTK_SIGNAL_FUNC(entry_tickled_cb),
		data->user_specified_rb); 
	gtk_signal_connect (
		GTK_OBJECT(gnome_file_entry_gtk_entry(GNOME_FILE_ENTRY(entry))),
		"button_press_event",
		GTK_SIGNAL_FUNC(entry_tickled_cb),
		data->user_specified_rb); 
	gnome_file_entry_set_directory (GNOME_FILE_ENTRY (entry), TRUE);
	gtk_widget_set_usize (GTK_WIDGET(entry), 400, 0);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w),
				      last_save_as_path!=NULL);
	if (!is_nonempty_string(last_save_as_path))
		replace_gstr (&last_save_as_path, g_strdup(download_dir));
	gtk_entry_set_text (GTK_ENTRY(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (entry))), last_save_as_path);
	gtk_box_pack_start (GTK_BOX(hbox), entry, TRUE, TRUE, 0);
	data->user_specified_entry = entry;
	gtk_box_pack_start (GTK_BOX(vbox), hbox, FALSE, FALSE, 0);


	/* append group name to the path? */
	pch = g_strdup_printf (_("Add ``/%s'' to path"), group->name);
	w = gtk_check_button_new_with_label (pch);
	g_free (pch);
	data->append_groupname_tb = w;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), append_groupname);
	gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, GNOME_PAD_SMALL);

	/* add "Location" to the dialog */
	gtk_container_add (GTK_CONTAINER(frame), vbox);
	gtk_box_pack_start (
		GTK_BOX (GNOME_DIALOG (dialog)->vbox), frame, TRUE, TRUE, 0);



	/***
	****  Message Pane
	***/


	frame = gtk_frame_new (_("Article Text"));
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
	vbox = gtk_vbox_new (TRUE, 0);

	/* don't save bodies */
	w = gtk_radio_button_new_with_label (NULL, _("Don't Save Article Text"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !save_bodies);
	gtk_box_pack_start (GTK_BOX(vbox), w, TRUE, TRUE, 0);
	data->body_filename_off_rb = w;

	/* use message-id */
	w = gtk_radio_button_new_with_label_from_widget (
		GTK_RADIO_BUTTON(w), _("Use Message-ID"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), use_message_id);
	gtk_box_pack_start (GTK_BOX(vbox), w, TRUE, TRUE, 0);
	data->body_filename_message_id_rb = w;

	/* use message-id */
	w = gtk_radio_button_new_with_label_from_widget (
		GTK_RADIO_BUTTON(w), _("Use Subject"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), use_subject_for_body);
	gtk_box_pack_start (GTK_BOX(vbox), w, TRUE, TRUE, 0);
	data->body_filename_subject_rb = w;

	/* use custom */
	hbox = gtk_hbox_new (FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (
		GTK_RADIO_BUTTON(w), _("Use Custom Filename"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), use_custom_for_body);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	data->body_filename_custom_rb = w;
	w = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX(hbox), w, TRUE, TRUE, 0);
	data->body_filename_custom_entry = w;
	gtk_signal_connect (GTK_OBJECT(w), "focus_in_event",
	                    GTK_SIGNAL_FUNC(entry_tickled_cb),
	                    data->body_filename_custom_rb); 
	gtk_signal_connect (GTK_OBJECT(w), "button_press_event",
	                    GTK_SIGNAL_FUNC(entry_tickled_cb),
	                    data->body_filename_custom_rb); 
	gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

	/* add "Bodies" to the dialog */
	gtk_container_add (GTK_CONTAINER(frame), vbox);
	gtk_box_pack_start (
		GTK_BOX (GNOME_DIALOG (dialog)->vbox), frame, TRUE, TRUE, 0);

	/***
	****  Attachments Pane
	***/

	frame = gtk_frame_new (_("Attachments"));
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
	vbox = gtk_vbox_new (TRUE, 0);

	/* don't save attachments */
	w = gtk_radio_button_new_with_label (NULL, _("Don't Save Attachments"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), !save_attachments);
	gtk_box_pack_start (GTK_BOX(vbox), w, TRUE, TRUE, 0);
	data->decode_filename_off_rb = w;

	/* use encoded filename */
	w = gtk_radio_button_new_with_label_from_widget (
		GTK_RADIO_BUTTON(w), _("Use Encoded Filename"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), use_encoded);
	gtk_box_pack_start (GTK_BOX(vbox), w, TRUE, TRUE, 0);
	data->decode_filename_encoded_rb = w;

	/* use subject */
	w = gtk_radio_button_new_with_label_from_widget (
		GTK_RADIO_BUTTON(w), _("Use Article Subject"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), use_subject);
	gtk_box_pack_start (GTK_BOX(vbox), w, TRUE, TRUE, 0);
	data->decode_filename_subject_rb = w;

	/* user-specified filenames */
	hbox = gtk_hbox_new (FALSE, 0);
	w = gtk_radio_button_new_with_label_from_widget (
		GTK_RADIO_BUTTON(w), _("Use Custom Filename"));
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(w), use_custom);
	gtk_box_pack_start (GTK_BOX(hbox), w, FALSE, FALSE, 0);
	data->decode_filename_custom_rb = w;
	w = gtk_entry_new ();
	gtk_box_pack_start (GTK_BOX(hbox), w, TRUE, TRUE, 0);
	data->decode_filename_custom_entry = w;
	gtk_signal_connect (GTK_OBJECT(w), "focus_in_event",
	                    GTK_SIGNAL_FUNC(entry_tickled_cb),
	                    data->decode_filename_custom_rb); 
	gtk_signal_connect (GTK_OBJECT(w), "button_press_event",
	                    GTK_SIGNAL_FUNC(entry_tickled_cb),
	                    data->decode_filename_custom_rb); 
	gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

	/* add "Filename" to the dialog */
	gtk_container_add (GTK_CONTAINER(frame), vbox);
	gtk_box_pack_start (
		GTK_BOX (GNOME_DIALOG (dialog)->vbox), frame, TRUE, TRUE, 0);

	/***
	****  Priority Pane
	***/

	/* "Priority" pane */
	frame = gtk_frame_new (_("Priority"));
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
	vbox = gtk_vbox_new (TRUE, 0);

	/* front of queue */
	w = gtk_radio_button_new_with_label (NULL, _("Add Task to the Front of the Task Manager"));
	gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), !back_of_queue);
	data->front_of_queue_rb = w;

	/* back of queue (default) */
	w = gtk_radio_button_new_with_label_from_widget (
		GTK_RADIO_BUTTON(w), _("Add Task to the Back of the Task Manager"));
	gtk_box_pack_start (GTK_BOX(vbox), w, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), back_of_queue);
	data->back_of_queue_rb = w;

	/* add "Priority" to the dialog */
	gtk_container_add (GTK_CONTAINER(frame), vbox);
	gtk_box_pack_start (
		GTK_BOX (GNOME_DIALOG (dialog)->vbox), frame, TRUE, TRUE, 0);




	/* prep the dialog */
	gtk_signal_connect (GTK_OBJECT(dialog), "clicked",
		GTK_SIGNAL_FUNC(save_attachment_as_dialog_clicked_cb),
		data);
	gtk_signal_connect (GTK_OBJECT(dialog), "destroy",
		GTK_SIGNAL_FUNC(save_attachment_as_dialog_destroy_cb),
		data); 
	gtk_signal_connect (GTK_OBJECT(data->append_groupname_tb), "toggled",
		append_groupname_toggled_cb, data);
	gtk_signal_connect (GTK_OBJECT(data->front_of_queue_rb), "toggled",
		priority_toggled_cb, data);
	gtk_signal_connect (GTK_OBJECT(data->back_of_queue_rb), "toggled",
		priority_toggled_cb, data);
	gtk_signal_connect (
		GTK_OBJECT(gnome_file_entry_gtk_entry (GNOME_FILE_ENTRY (entry))), "changed",
		user_specified_path_changed_cb, NULL);

	gnome_dialog_set_parent (GNOME_DIALOG(dialog), GTK_WINDOW(Pan.window));
	gtk_widget_show_all (dialog);
	pan_unlock ();
}



void
save_attachment_as (Group* group,
		    GSList* gslists_of_unowned_articles)
{
	create_save_attachment_as_dialog (group,
					  gslists_of_unowned_articles);
}
