#include <libart_lgpl/art_misc.h>
#include <libart_lgpl/art_svp.h>
#include <libart_lgpl/art_vpath.h>
#include <libart_lgpl/art_svp_vpath.h>
#include <libart_lgpl/art_rgb_svp.h>
#include <libart_lgpl/art_rgb.h>

#define WIDTH 100
#define HEIGHT 100
#define BYTES_PER_PIXEL 3 /* 24 packed rgb bits */
#define ROWSTRIDE (WIDTH*BYTES_PER_PIXEL)


static ArtSVP *make_path (void);
static unsigned char *render_path (const ArtSVP *path);
static void save_buffer (const unsigned char *buffer, const char *filename);


#include <gdk-pixbuf/gdk-pixbuf.h>
#include <png.h>
#include <stdio.h>

/**
 * pixbuf_save_to_file:
 * @pixbuf: pixel buffer to save.
 * @filename: file name to save the pixel buffer into.
 *
 * Saves the pixel buffer in the given filename under 
 * the .png format. Returns TRUE is succesful and FALSE
 * otherwise.
 *
 * Copyright is to Iain Holmes and Eazel, inc for this function.
 * It was stolen from nautilus-gdk-pixbuf-extensions.c but
 * was originally coming from gnome-iconedit by Iain Holmes
 * It was hacked by Ramiro Estrugo for Eazel, inc.
 */
static gboolean
pixbuf_save_to_file (const GdkPixbuf *pixbuf, const char *file_name)
{
	FILE *handle;
  	char *buffer;
	gboolean has_alpha;
	int width, height, depth, rowstride;
  	guchar *pixels;
  	png_structp png_ptr;
  	png_infop info_ptr;
  	png_text text[2];
  	int i;

	g_return_val_if_fail (pixbuf != NULL, FALSE);
	g_return_val_if_fail (file_name != NULL, FALSE);
	g_return_val_if_fail (file_name[0] != '\0', FALSE);

        handle = fopen (file_name, "wb");
        if (handle == NULL) {
        	return FALSE;
	}

	png_ptr = png_create_write_struct (PNG_LIBPNG_VER_STRING, NULL, NULL, NULL);
	if (png_ptr == NULL) {
		fclose (handle);
		return FALSE;
	}

	info_ptr = png_create_info_struct (png_ptr);
	if (info_ptr == NULL) {
		png_destroy_write_struct (&png_ptr, (png_infopp)NULL);
		fclose (handle);
	    	return FALSE;
	}

	if (setjmp (png_ptr->jmpbuf)) {
		png_destroy_write_struct (&png_ptr, &info_ptr);
		fclose (handle);
		return FALSE;
	}

	png_init_io (png_ptr, handle);

        has_alpha = gdk_pixbuf_get_has_alpha (pixbuf);
	width = gdk_pixbuf_get_width (pixbuf);
	height = gdk_pixbuf_get_height (pixbuf);
	depth = gdk_pixbuf_get_bits_per_sample (pixbuf);
	pixels = gdk_pixbuf_get_pixels (pixbuf);
	rowstride = gdk_pixbuf_get_rowstride (pixbuf);

	png_set_IHDR (png_ptr, info_ptr, width, height,
			depth, PNG_COLOR_TYPE_RGB_ALPHA,
			PNG_INTERLACE_NONE,
			PNG_COMPRESSION_TYPE_DEFAULT,
			PNG_FILTER_TYPE_DEFAULT);

	/* Some text to go with the png image */
	text[0].key = "Title";
	text[0].text = (char *) file_name;
	text[0].compression = PNG_TEXT_COMPRESSION_NONE;
	text[1].key = "Software";
	text[1].text = "Nautilus Thumbnail";
	text[1].compression = PNG_TEXT_COMPRESSION_NONE;
	png_set_text (png_ptr, info_ptr, text, 2);

	/* Write header data */
	png_write_info (png_ptr, info_ptr);

	/* if there is no alpha in the data, allocate buffer to expand into */
	if (has_alpha) {
		buffer = NULL;
	} else {
		buffer = g_malloc(4 * width);
	}
	
	/* pump the raster data into libpng, one scan line at a time */	
	for (i = 0; i < height; i++) {
		if (has_alpha) {
			png_bytep row_pointer = pixels;
			png_write_row (png_ptr, row_pointer);
		} else {
			/* expand RGB to RGBA using an opaque alpha value */
			int x;
			char *buffer_ptr = buffer;
			char *source_ptr = pixels;
			for (x = 0; x < width; x++) {
				*buffer_ptr++ = *source_ptr++;
				*buffer_ptr++ = *source_ptr++;
				*buffer_ptr++ = *source_ptr++;
				*buffer_ptr++ = 255;
			}
			png_write_row (png_ptr, (png_bytep) buffer);		
		}
		pixels += rowstride;
	}
	
	png_write_end (png_ptr, info_ptr);
	png_destroy_write_struct (&png_ptr, &info_ptr);
	
	g_free (buffer);
		
	fclose (handle);
	return TRUE;
}

static unsigned char *
render_path (const ArtSVP *path)
{
  art_u8 *buffer = NULL;
  art_u32 color = (0xFF << 24) | (0x00 <<16) | (0x00<<8) | (0xFF) ; /* RRGGBBAA */
    

  buffer = art_new (art_u8, WIDTH*HEIGHT*BYTES_PER_PIXEL);
  art_rgb_run_alpha (buffer, 0xFF, 0xFF, 0xFF, 0xFF, WIDTH*HEIGHT);
  art_rgb_svp_alpha (path, 0, 0, WIDTH, HEIGHT, 
		     color, buffer, ROWSTRIDE, NULL);

  return (unsigned char *) buffer;
}

static ArtSVP *
make_path (void)
{
  ArtVpath *vec = NULL;
  ArtSVP *svp = NULL;
  
  vec = art_new (ArtVpath, 10);
  vec[0].code = ART_MOVETO;
  vec[0].x = 0;
  vec[0].y = 0;
  vec[1].code = ART_LINETO;
  vec[1].x = 0;
  vec[1].y = 10;
  vec[2].code = ART_LINETO;
  vec[2].x = 10;
  vec[2].y = 10;
  vec[3].code = ART_LINETO;
  vec[3].x = 10;
  vec[3].y = 0;
  vec[4].code = ART_END;

  svp = art_svp_from_vpath (vec);

  return svp;
}


static void
save_buffer (const unsigned char *buffer, const char *filename)
{
  GdkPixbuf *pixbuf;

  pixbuf = gdk_pixbuf_new_from_data (buffer,
				     GDK_COLORSPACE_RGB, 
				     FALSE, 8,
				     WIDTH, HEIGHT,
				     ROWSTRIDE,
				     NULL, NULL);

  pixbuf_save_to_file (pixbuf, filename);
 
  gdk_pixbuf_unref (pixbuf);
}


int main (int argc, char *argv[])
{
  ArtSVP *path;
  char *buffer;

  path = make_path ();

  buffer = render_path (path);

  save_buffer (buffer, "foo.png");

  return 0;
}







