/*
 * Copyright (C) 2002-2003 the xine-project
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: infobar.c,v 1.10 2003/03/24 00:13:55 guenter Exp $
 *
 * a nice black information display area
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "infobar.h"

struct infobar_s {

  GtkWidget            *w;

  PangoLayout          *pl;
  PangoFontDescription *pfd;

  GdkGC                *black_gc, *blue_gc, *white_gc;

  gchar                *line1;
  gchar                *line2;
};

static void paint_bar (infobar_t *bar) {

  gdk_draw_rectangle (bar->w->window, bar->black_gc, TRUE,
		      0, 0, 
		      bar->w->allocation.width, bar->w->allocation.height);

  pango_layout_set_text (bar->pl, bar->line1, -1);

  gdk_draw_layout (bar->w->window, bar->blue_gc, 7, 2, bar->pl);

  pango_layout_set_text (bar->pl, bar->line2, -1);
  gdk_draw_layout (bar->w->window, bar->blue_gc, 7, 15, bar->pl);
}

static gboolean expose_event_cb (GtkWidget *widget, 
				 GdkEventExpose *event, 
				 gpointer data) {

  infobar_t *bar = (infobar_t *) data;

  gdk_window_clear_area (widget->window,
                         event->area.x, event->area.y,
                         event->area.width, event->area.height);

  gdk_gc_set_clip_rectangle (widget->style->fg_gc[widget->state],
                             &event->area);

  paint_bar (bar);

  gdk_gc_set_clip_rectangle (widget->style->fg_gc[widget->state],
                             NULL);
  return TRUE;
}

void infobar_line1 (infobar_t *bar, const char *format, ...) {

  va_list argp;

  va_start (argp, format);

  vsnprintf (bar->line1, 1024, format, argp);

  va_end (argp);

  normalize_to_ascii (bar->line1);

  paint_bar (bar);
}

void infobar_line2 (infobar_t *bar, const char *format, ...){

  va_list argp;

  va_start (argp, format);

  vsnprintf (bar->line2, 1024, format, argp);

  va_end (argp);

  normalize_to_ascii (bar->line2);

  paint_bar (bar);
}

static void alloc_color(GtkWidget *window, GdkGC *gc, gint r, gint g, gint b) {

  GdkColor *color=malloc(sizeof(GdkColor));

  if (gc) {
    color->red = r; color->green = g; color->blue = b;
    if (!gdk_colormap_alloc_color (gtk_widget_get_colormap (window), color, FALSE, TRUE)) {
      printf ("failed to alloc color (%d/%d/%d)\n",color->red,color->green,color->blue);
      exit(1);
    }
    gdk_gc_set_foreground (gc, color);
  }
}


infobar_t *create_infobar (GtkWidget **w) {

  infobar_t *bar;

  bar = (infobar_t *) malloc (sizeof (infobar_t));

  bar->w = gtk_drawing_area_new ();
  gtk_drawing_area_size (GTK_DRAWING_AREA(bar->w), 250, 30);

  bar->line1 = malloc(1025);
  bar->line2 = malloc(1025);

  *w = bar->w;

  return bar;
}

void postinit_infobar (GtkWidget *window, infobar_t *bar) {

  g_signal_connect (bar->w, "expose_event", 
		    G_CALLBACK (expose_event_cb), bar);

  bar->black_gc = gdk_gc_new (window->window);
  alloc_color (window, bar->black_gc, 0, 0, 0);

  bar->white_gc = gdk_gc_new (window->window);
  alloc_color (window, bar->white_gc, 65535, 65535, 65535);

  bar->blue_gc = gdk_gc_new (window->window);
  alloc_color (window, bar->blue_gc, 20000, 20000, 65535);

  /* 
   * pango stuff (antialiased font rendering
   */

  bar->pfd = pango_font_description_new ();
  pango_font_description_set_family (bar->pfd, "Sans");
  pango_font_description_set_weight (bar->pfd, PANGO_WEIGHT_BOLD);
  pango_font_description_set_size (bar->pfd, 8*PANGO_SCALE);

  bar->pl = gtk_widget_create_pango_layout (bar->w, NULL);

  pango_layout_set_font_description (bar->pl, bar->pfd);
}
