/***************************************************************************
                          mediacontrol.cpp  -  main file of the applet
                             -------------------
    begin                : Tue Apr 25 11:53:11 CEST 2000
    copyright            : (C) 2000-2003 by Stefan Gehn
    email                : sgehn@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "mediacontrol.h"
#include "mediacontrol.moc"

#include "mediacontrolconfig.h"
#include "configfrontend.h"
#include "xmmsInterface.h"
#include "noatunInterface.h"
#include "jukInterface.h"

#include <qfile.h>
#include <qdragobject.h>
#include <qtooltip.h>
#include <qstyle.h>
#include <qpainter.h>
#include <qpopupmenu.h>

#include <kapplication.h>
#include <kipc.h>
#include <kglobal.h>
#include <kiconloader.h>
#include <knotifyclient.h>
#include <kprocess.h>
#include <dcopclient.h>

#define MC_BUTTONSIZE 18 // TODO: Might become dynamical for bigger panels

extern "C"
{
	KPanelApplet *init( QWidget *parent, const QString &configFile)
	{
		KGlobal::locale()->insertCatalogue("mediacontrol");
		return new MediaControl(configFile, KPanelApplet::Normal,
			KPanelApplet::About | KPanelApplet::Preferences, parent, "mediacontrol");
	}
}

// ====================================================================================

class MediaControlToolTip : public QToolTip
{
	public:
		MediaControlToolTip(QWidget *widget, PlayerInterface *pl_obj) : QToolTip(widget), mWidget(widget), mPlayer(pl_obj) {};

	protected:
		virtual void maybeTip(const QPoint &pt)
		{
			QRect rc( mWidget->rect());
			if (rc.contains(pt))
			{
				tip ( rc, mPlayer->getTrackTitle() );
			}
		}
	private:
		QWidget *mWidget;
		PlayerInterface *mPlayer;
};

// ====================================================================================

MediaControl::MediaControl(const QString &configFile, Type t, int actions, QWidget *parent, const char *name)
	: KPanelApplet(configFile, t, actions, parent, name), DCOPObject("MediaControl"),
		_instance(new KInstance("mediacontrol"))
{
	_player = 0L;
	_prefsDialog = 0L;

	_configFrontend = new ConfigFrontend( config() );
	// My own dcopclient
	_dcopClient = new DCOPClient();
	_dcopClient->registerAs("mediacontrol", false);

	setAcceptDrops(true);

	prev_button  = new TrayButton ( this, "PREVIOUS" );
	play_button  = new TrayButton ( this, "PLAY" );
	pause_button = new TrayButton ( this, "PAUSE" );
	stop_button  = new TrayButton ( this, "STOP" );
	next_button  = new TrayButton ( this, "NEXT" );
	time_slider  = new QSlider (QSlider::Horizontal, this, "time_slider" );
	time_slider->setRange(0,0);
	time_slider->setValue(0);
	time_slider->setTracking( false );

	// request notification of changes in icon style
	kapp->addKipcEventMask(KIPC::IconChanged);
	connect(kapp, SIGNAL(iconChanged(int)), this, SLOT(slotIconChanged()));

	reparseConfig();

	rmbMenu = new QPopupMenu(this, "RMB Menu");
	rmbMenu->insertItem(SmallIcon("configure"), i18n("Configure MediaControl..."), this, SLOT(preferences()));
	rmbMenu->insertItem(i18n("About MediaControl"), this, SLOT(about()));
}

MediaControl::~MediaControl()
{
	delete _player;
	delete _configFrontend;
	delete _dcopClient;
        KGlobal::locale()->removeCatalogue("mediacontrol");
}

// Drag-n-Drop stuff ==================================================================

void MediaControl::dragEnterEvent(QDragEnterEvent* event)
{
	_player->dragEnterEvent(event);  // Just pass dnd to the playerInterface
}

void MediaControl::dropEvent(QDropEvent* event)
{
	_player->dropEvent(event);  // Just pass dnd to the playerInterface
}

// ====================================================================================


void MediaControl::setSliderPosition(int len ,int time)
{
	time_slider->blockSignals ( true );
	if( orientation() == Vertical)
	  time = len - time;

	if ( lastLen != len )
	  time_slider->setRange(0,len);
	lastLen = len;

	if ( lastTime != time )
	  time_slider->setValue ( time );
	lastTime = time;

	time_slider->blockSignals ( false );
}

void MediaControl::enableAll(void)
{
	prev_button->setDisabled(false);
	play_button->setDisabled(false);
	pause_button->setDisabled(false);
	stop_button->setDisabled(false);
	next_button->setDisabled(false);
	time_slider->setDisabled(false);
}

void MediaControl::disableAll(void)
{
	prev_button->setDisabled(true);
	play_button->setDisabled(true);
	pause_button->setDisabled(true);
	stop_button->setDisabled(true);
	next_button->setDisabled(true);
	time_slider->setDisabled(true);
}

void MediaControl::slotIconChanged()
{
	if(!_configFrontend->useCustomTheme())
	{
		prev_button->setPixmap ( SmallIcon("player_start.png") );
		play_button->setPixmap ( SmallIcon("player_play.png") );
		pause_button->setPixmap ( SmallIcon("player_pause.png") );
		stop_button->setPixmap ( SmallIcon("player_stop.png") );
		next_button->setPixmap ( SmallIcon("player_end.png") );
	}
}

// Dialogs ============================================================================

void MediaControl::preferences(void)
{
	if ( _prefsDialog )
	{
		_prefsDialog->raise();
	}
	else
	{
		_prefsDialog = new MediaControlConfig ( _configFrontend );
		connect ( _prefsDialog, SIGNAL(closing()), this, SLOT(slotClosePrefsDialog()) );
		connect ( _prefsDialog, SIGNAL(destroyed()), this, SLOT(slotPrefsDialogClosing()) );
		connect ( _prefsDialog, SIGNAL(configChanged()), this, SLOT(slotConfigChanged()) );
	}
}

void MediaControl::slotConfigChanged( void )
{
	reparseConfig();
}

void MediaControl::slotClosePrefsDialog()
{
	delete _prefsDialog;
}

void MediaControl::slotPrefsDialogClosing()
{
	if ( _prefsDialog )
		_prefsDialog = 0L;
}

void MediaControl::about(void)
{

	KAboutData aboutData( "MediaControl", I18N_NOOP("MediaControl"), MEDIACONTROL_VERSION,
		I18N_NOOP("A small control-applet for various media players"),
		KAboutData::License_GPL_V2, "(c) 2001-2003 Stefan Gehn",
		0, "http://metz.gehn.net");

	aboutData.addAuthor("Stefan Gehn", I18N_NOOP("Main Developer"), "sgehn@gmx.net",
		"http://metz.gehn.net");
	aboutData.addAuthor("Robbie Ward", I18N_NOOP("Initial About-Dialog"), "wardy@robbieward.co.uk",
		"http://www.robbieward.co.uk");

	aboutData.addCredit("Sascha Hoffman", I18N_NOOP("Button-Pixmaps"), "tisch.sush@gmx.de", 0);
	aboutData.addCredit("Christian Hoffman", I18N_NOOP("Button-Pixmaps"), "tisch.crix@gmx.de",
		"http://www.crixensgfxcorner.de.vu/");
	aboutData.addCredit("Ulrik Mikaelsson", I18N_NOOP("Fix for Noatun-Support"), "rawler@rsn.bth.se", 0);
	aboutData.addCredit("Anthony J Moulen", I18N_NOOP("Fix for Vertical Slider"), "ajmoulen@moulen.org", 0);

	KAboutApplication mediacontrolabout(&aboutData);

	mediacontrolabout.setCaption( i18n("About MediaControl") );
	mediacontrolabout.exec();
}

// Fixing the orientation problem in qslider.
void MediaControl::adjustTime(int time)
{
	if(orientation() == Vertical)
		emit(newJumpToTime(lastLen - time));
	else
		emit(newJumpToTime(time));
}

// Config Stuff =======================================================================

void MediaControl::reparseConfig()
{
//	kdDebug() << "reparseConfig();" << endl;
	_configFrontend->reparseConfiguration();

	if ( _player != 0L ) // make sure there is no player-object
	{
		_player->disconnect();	// disconnect from all things

		time_slider->disconnect();
		prev_button->disconnect();
		play_button->disconnect();
		pause_button->disconnect();
		stop_button->disconnect();
		next_button->disconnect();

		delete slider_tooltip; // tooltip depends on _player : delete it before _player gets deleted
		slider_tooltip = 0L;

		delete _player;
		_player = 0L;
	}

	lastLen = -1;
	lastTime = -1;

	QString playerString = _configFrontend->player();

#ifdef HAVE_XMMS
	if ( playerString == "XMMS" )
	{
//		kdDebug() << "player = XMMS" << endl;
		_player = new XmmsInterface ();
		time_slider->setSteps((_configFrontend->mouseWheelSpeed()*1000),
			(_configFrontend->mouseWheelSpeed()*1000));
	}
	// player is noatun or something wrong was written to the config
#endif
	if ( playerString == "Noatun")
        {
//		kdDebug() << "player = Noatun" << endl;
		_player = new NoatunInterface();
		time_slider->setSteps((_configFrontend->mouseWheelSpeed()),
			(_configFrontend->mouseWheelSpeed()));
	}
        if ( playerString == "JuK")
        {
                kdDebug() << "player = JuK" << endl;
                _player = new JuKInterface();
		time_slider->setSteps((_configFrontend->mouseWheelSpeed()),
			(_configFrontend->mouseWheelSpeed()));
        }

	//  this signal gets emitted by a playerInterface when the player's playtime changed
	connect(_player, SIGNAL(newSliderPosition(int,int)), SLOT(setSliderPosition(int,int)));

	connect(_player, SIGNAL(playerStarted()), SLOT(enableAll()));
	connect(_player, SIGNAL(playerStopped()), SLOT(disableAll()));

	// do we use our icons or the default ones from KDE?
	if(_configFrontend->useCustomTheme())
	{
		// load theme
		QString skindir = locate("data", "mediacontrol/"+_configFrontend->theme()+"/");

		// the user has to take care if all pixmaps are there, we only check for one of them
		if (QFile(skindir+"play.png").exists())
		{
			prev_button->setPixmap ( QPixmap( locate("data",skindir+"prev.png") ) );
			play_button->setPixmap ( QPixmap(locate("data",skindir+"play.png")) );
			pause_button->setPixmap ( QPixmap(locate("data",skindir+"pause.png")) );
			stop_button->setPixmap ( QPixmap(locate("data",skindir+"stop.png")) );
			next_button->setPixmap ( QPixmap(locate("data",skindir+"next.png")) );
		}
		else // icon-theme is invalid or not there
		{
			KNotifyClient::event("warning",
				i18n("There was trouble loading theme %1. Please choose a different theme.").arg(skindir));

			// default to kde-icons, they have to be installed :)
			slotIconChanged();

			// and open prefs-dialog
			preferences();
		}
	}
	else // KDE default-icons, assuming that these icons exist!
	{
		// sets icons from kde
		slotIconChanged();
	}

	slider_tooltip = new MediaControlToolTip(time_slider, _player);

	connect(prev_button, SIGNAL(clicked()), _player, SLOT(prev()));
	connect(play_button, SIGNAL(clicked()), _player, SLOT(play()));
	connect(pause_button, SIGNAL(clicked()), _player, SLOT(pause()));
	connect(stop_button, SIGNAL(clicked()), _player, SLOT(stop()));
	connect(next_button, SIGNAL(clicked()), _player, SLOT(next()));

	connect(time_slider, SIGNAL(sliderPressed()), _player, SLOT(sliderStartDrag()));
	connect(time_slider, SIGNAL(sliderReleased()), _player, SLOT(sliderStopDrag()));
	connect(time_slider, SIGNAL(valueChanged(int)), this, SLOT(adjustTime(int)));
	connect(this, SIGNAL(newJumpToTime(int)), _player, SLOT(jumpToTime(int)));
}

// Widget Placement ===================================================================

// kicker wants to know what width we need for a given height
// (this is called when being a HORIZONTAL panel)
int MediaControl::widthForHeight(int height) const
{
//	kdDebug() << "kicker height: " << height << endl;
//	kdDebug() << "slider needs: " << time_slider->minimumSizeHint().height() << endl;

	// slider height + button height
	if ( height >= (time_slider->minimumSizeHint().height()+MC_BUTTONSIZE) )
	{	// slider UNDER buttons
		// (5 * button width + spaces between them);
		return (5*MC_BUTTONSIZE+10);
	}
	else
	{	// slider ASIDE buttons
		// (5 * button width + spaces between them) * 2 [size of slider = size of all buttons]
		return ((5*MC_BUTTONSIZE+10)*2);
	}
}

// kicker wants to know what height we need for a given width
// (this is called when being a VERTICAL panel)
int MediaControl::heightForWidth(int width) const
{
//	kdDebug() << "kicker width: " << width << endl;

	// slider height + button height
	if ( width >= (time_slider->minimumSizeHint().width()+MC_BUTTONSIZE) )
	{	// slider ASIDE icons
		// (5 * button width + spaces between them);
		return (5*MC_BUTTONSIZE+10);
	}
	else
	{	// slider UNDER buttons
		// (5 * button width + spaces between them) * 2
		// because the size of the slider = the size of all buttons
		return ((5*MC_BUTTONSIZE+10)*2);
	}
}

void MediaControl::mousePressEvent(QMouseEvent* e)
{
	rmbMenu->popup(e->globalPos());
}

// Danger: Weird Code ahead! ;))
void MediaControl::resizeEvent( QResizeEvent* )
{
//	kdDebug() << "resizeEvent()" << endl;
	int w = width();
	int h = height();
	if ( orientation() == Vertical )
	{		// ====== VERTICAL =================================================
		time_slider->setOrientation(QSlider::Vertical);
		int slider_width = time_slider->minimumSizeHint().width();
		// some styles need more space for sliders than avilable in very small panels :(
		if ( slider_width > w ) slider_width = w;

		// that width would be needed to put the slider aside the buttons
		if ( w >= (slider_width+MC_BUTTONSIZE) )
		{	// Slider ASIDE icons
			int applet_space = (w - (slider_width+MC_BUTTONSIZE) ) / 2;
			if ( applet_space < 0 )
				applet_space = 0;

			prev_button->setGeometry  ( applet_space, 1, MC_BUTTONSIZE, MC_BUTTONSIZE );
			play_button->setGeometry  ( applet_space, 3+MC_BUTTONSIZE, MC_BUTTONSIZE, MC_BUTTONSIZE );
			pause_button->setGeometry ( applet_space, 5+2*MC_BUTTONSIZE, MC_BUTTONSIZE, MC_BUTTONSIZE );
			stop_button->setGeometry  ( applet_space, 7+3*MC_BUTTONSIZE, MC_BUTTONSIZE, MC_BUTTONSIZE );
			next_button->setGeometry  ( applet_space, 9+4*MC_BUTTONSIZE, MC_BUTTONSIZE, MC_BUTTONSIZE );
			time_slider->setGeometry  ( applet_space+MC_BUTTONSIZE, 1, slider_width, 5*MC_BUTTONSIZE+8 );
		}
		else
		{	// Slider UNDER Icons
			int slider_space = (w - slider_width)/2;
			int button_space = (w - MC_BUTTONSIZE)/2;

			prev_button->setGeometry  ( button_space, 1 , MC_BUTTONSIZE, MC_BUTTONSIZE );
			play_button->setGeometry  ( button_space, 3+MC_BUTTONSIZE, MC_BUTTONSIZE, MC_BUTTONSIZE );
			pause_button->setGeometry ( button_space, 5+2*MC_BUTTONSIZE, MC_BUTTONSIZE, MC_BUTTONSIZE );
			stop_button->setGeometry  ( button_space, 7+3*MC_BUTTONSIZE, MC_BUTTONSIZE, MC_BUTTONSIZE );
			next_button->setGeometry  ( button_space, 9+4*MC_BUTTONSIZE, MC_BUTTONSIZE, MC_BUTTONSIZE );
			time_slider->setGeometry  ( slider_space, 11+5*MC_BUTTONSIZE, slider_width, 5*MC_BUTTONSIZE+8 );
		}
	}
	else	// ====== HORIZONTAL ===============================================
	{
		time_slider->setOrientation(QSlider::Horizontal);
		int slider_height = time_slider->minimumSizeHint().height();
		// some styles need more space for sliders than avilable in very small panels :(
		if ( slider_height > h ) slider_height = h;

		// that h would be needed to put the slider under the buttons
		if ( h >= (slider_height+MC_BUTTONSIZE) )
		{	// Slider UNDER icons
			int applet_space = (h-(slider_height+MC_BUTTONSIZE))/2;
			if ( applet_space < 0 )
				applet_space = 0;

			prev_button->setGeometry  ( 1 , applet_space, MC_BUTTONSIZE, MC_BUTTONSIZE );
			play_button->setGeometry  ( 3+MC_BUTTONSIZE, applet_space, MC_BUTTONSIZE, MC_BUTTONSIZE );
			pause_button->setGeometry ( 5+2*MC_BUTTONSIZE, applet_space, MC_BUTTONSIZE, MC_BUTTONSIZE );
			stop_button->setGeometry  ( 7+3*MC_BUTTONSIZE, applet_space, MC_BUTTONSIZE, MC_BUTTONSIZE );
			next_button->setGeometry  ( 9+4*MC_BUTTONSIZE, applet_space, MC_BUTTONSIZE, MC_BUTTONSIZE );
			time_slider->setGeometry  ( 1, applet_space + MC_BUTTONSIZE, 5*MC_BUTTONSIZE+8, slider_height );
		}
		else
		{	// Slider ASIDE Icons
			int slider_space = (h - slider_height)/2;
			int button_space = (h - MC_BUTTONSIZE)/2;

			prev_button->setGeometry  ( 1 , button_space, MC_BUTTONSIZE, MC_BUTTONSIZE );
			play_button->setGeometry  ( 3+MC_BUTTONSIZE, button_space, MC_BUTTONSIZE, MC_BUTTONSIZE );
			pause_button->setGeometry ( 5+2*MC_BUTTONSIZE, button_space, MC_BUTTONSIZE, MC_BUTTONSIZE );
			stop_button->setGeometry  ( 7+3*MC_BUTTONSIZE, button_space, MC_BUTTONSIZE, MC_BUTTONSIZE );
			next_button->setGeometry  ( 9+4*MC_BUTTONSIZE, button_space, MC_BUTTONSIZE, MC_BUTTONSIZE );
			time_slider->setGeometry  ( 11+5*MC_BUTTONSIZE, slider_space, 5*MC_BUTTONSIZE+8, slider_height );
		}
	}
}

// Our Button ========================================================================

void TrayButton::drawButton(QPainter *p)
{
	QPixmap bg( size() );
	QPainter pbg;

	pbg.begin(&bg);
	pbg.fillRect(rect(), colorGroup().brush(QColorGroup::Background));
	if ( isDown() || isOn() )
	{	// Draw shapes to indicate the down state.
		pbg.setPen(Qt::black);
		pbg.drawLine(0, 0, width()-1, 0);
		pbg.drawLine(0, 0, 0, height()-1);
		pbg.setPen(colorGroup().light());
		pbg.drawLine(0, height()-1, width()-1, height()-1);
		pbg.drawLine(width()-1, 0, width()-1, height()-1);
	}
	pbg.end();

	p->drawPixmap(0,0,bg); // draw the background

	// draw icon
	if(!pixmap()->isNull())
	{
		QRect br(1, 1, width()-2, height()-2);
		int dx = (br.width() - pixmap()->width()) / 2;
		int dy = (br.height() - pixmap()->height()) / 2;
		p->drawPixmap(br.x() + dx, br.y() + dy, *pixmap());
	}
}
