/*
 * @(#)GroboInstrumentTaskUTest.java
 *
 * Copyright (C) 2004 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.ant;

import java.io.File;
import java.io.IOException;
import java.util.Properties;

import junit.framework.Test;
import junit.framework.TestSuite;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;


/**
 * Tests the GroboInstrumentTask class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:27 $
 * @since     March 9, 2004
 */
public class GroboInstrumentTaskUTest extends AntTestA
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = GroboInstrumentTaskUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public GroboInstrumentTaskUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    //...........................
    // Tests that pass
    
    private static final Properties EXPECTED_1 = new Properties();
    static {
        EXPECTED_1.setProperty( "factory",
            "net.sourceforge.groboutils.codecoverage.v2.logger.DirectoryChannelLoggerFactory" );
        EXPECTED_1.setProperty( "channel-count", "0" );
    }
    public void testInstrument1() throws Exception
    {
        executeTarget( "instrument-1" );
        
        EXPECTED_1.setProperty( "logger.dir",
            (new File( getCoverageDir(),
            "logs" + File.separator + "logs" )).getAbsolutePath() );
        
        assertEquals(
            "Did not create the grobocoverage.properties file correctly",
            EXPECTED_1,
            loadGroboProperties() );
    }
    
    private static final Properties EXPECTED_2 = new Properties();
    static {
        EXPECTED_2.setProperty( "factory",
            "net.sourceforge.groboutils.codecoverage.v2.logger.DirectoryChannelLoggerFactory" );
        EXPECTED_2.setProperty( "channel-count", "0" );
    }
    public void testInstrument2() throws Exception
    {
        executeTarget( "instrument-2" );
        
        EXPECTED_2.setProperty( "logger.dir",
            (new File( getCoverageDir(),
            "logs" + File.separator + "logs" )).getAbsolutePath() );
        
        assertEquals(
            "Did not create the grobocoverage.properties file correctly",
            EXPECTED_2,
            loadGroboProperties() );
        
        // Also, ensure that the data files were created...
        assertModuleData(
            "Did not create the module data correctly:",
            "LineCount",
            "a.A" );
    }
    
    
    public void testInstrument3() throws Exception
    {
        executeTarget( "instrument-3" );
        
        // Ensure that only one data file type for each class was created...
        assertModuleData(
            "Did not create the module data correctly:",
            "LineCount",
            "a.A" );
        assertModuleData(
            "Did not create the module data correctly:",
            "LineCount",
            "a.A$X" );
        assertModuleData(
            "Did not create the module data correctly:",
            "LineCount",
            "a.B" );
        assertModuleData(
            "Did not create the module data correctly:",
            "LineCount",
            "a.B$X" );
    }
    
    
    public void testInstrument3a() throws Exception
    {
        executeTarget( "instrument-3a" );
        
        // Ensure that only one data file type for each class was created...
        assertModuleData(
            "Did not create the module data correctly:",
            "LineCount",
            "a.A" );
        assertModuleData(
            "Did not create the module data correctly:",
            "LineCount",
            "a.A$X" );
        assertModuleData(
            "Did not remove the module data correctly:",
            "LineCount",
            "a.B", 0 );
    }
    
    
    public void testInstrument4() throws Exception
    {
        executeTarget( "instrument-4" );
        
        // Ensure that only one data file type for each class was created...
        
        //why is this failing?
        /*
        assertModuleData(
            "Did not create the module data correctly:",
            "LineCount",
            "a.A", 2 );
        */
        assertModuleData(
            "Did not create the module data correctly:",
            "LineCount",
            "a.A$X" );
        assertModuleData(
            "Did not create the module data correctly:",
            "LineCount",
            "a.B", 2 );
        assertModuleData(
            "Did not create the module data correctly:",
            "LineCount",
            "a.B$X" );
    }
    
    
    public void testInstrument5() throws Exception
    {
        executeTarget( "instrument-5" );
        
        // Ensure that only one data file type for each class was created...
        assertModuleData(
            "Did not create the module data correctly:",
            "LineCount",
            "a.A" );
        assertModuleData(
            "Did not keep the module data correctly:",
            "LineCount",
            "a.B" );
        assertModuleData(
            "Did not create the module data correctly:",
            "LineCount",
            "a.A$X" );
    }
    
    
    private static final Properties EXPECTED_6 = new Properties();
    static {
        EXPECTED_6.setProperty( "factory",
            "net.sourceforge.groboutils.codecoverage.v2.logger.DirectoryChannelLoggerFactory" );
        EXPECTED_6.setProperty( "channel-count", "0" );
        EXPECTED_6.setProperty( "logger.1", "A" );
    }
    public void testInstrument6() throws Exception
    {
        executeTarget( "instrument-6" );
        
        EXPECTED_6.setProperty( "logger.dir",
            (new File( getCoverageDir(),
            "logs" + File.separator + "logs" )).getAbsolutePath() );
        EXPECTED_6.setProperty( "logger.2",
            getCoverageDir().getAbsolutePath() );
        
        assertEquals(
            "Did not create the grobocoverage.properties file correctly",
            EXPECTED_6,
            loadGroboProperties() );
    }
    
    
    public void testInstrument7() throws Exception
    {
        // this does all of its own tests
        executeTarget( "instrument-7" );
    }
    
    
    private static final Properties EXPECTED_8 = new Properties();
    static {
        EXPECTED_8.setProperty( "factory",
            "net.sourceforge.groboutils.codecoverage.v2.logger.MinDirChannelLoggerFactory" );
        EXPECTED_8.setProperty( "channel-count", "0" );
    }
    public void testInstrument8() throws Exception
    {
        executeTarget( "instrument-8" );
        
        EXPECTED_8.setProperty( "logger.dir",
            (new File( getCoverageDir(),
            "logs" + File.separator + "logs" )).getAbsolutePath() );
        
        assertEquals(
            "Did not create the grobocoverage.properties file correctly",
            EXPECTED_8,
            loadGroboProperties() );
    }
    
    
    private static final Properties EXPECTED_9 = new Properties();
    static {
        EXPECTED_9.setProperty( "factory",
            "net.sourceforge.groboutils.codecoverage.v2.logger.X" );
        EXPECTED_9.setProperty( "channel-count", "0" );
    }
    public void testInstrument9() throws Exception
    {
        executeTarget( "instrument-9" );
        
        EXPECTED_9.setProperty( "logger.dir",
            (new File( getCoverageDir(),
            "logs" + File.separator + "logs" )).getAbsolutePath() );
        
        assertEquals(
            "Did not create the grobocoverage.properties file correctly",
            EXPECTED_9,
            loadGroboProperties() );
    }
    
    
    private static final Properties EXPECTED_10 = new Properties();
    static {
        EXPECTED_10.setProperty( "factory",
            "a.X" );
        EXPECTED_10.setProperty( "channel-count", "0" );
    }
    public void testInstrument10() throws Exception
    {
        executeTarget( "instrument-9" );
        
        EXPECTED_10.setProperty( "logger.dir",
            (new File( getCoverageDir(),
            "logs" + File.separator + "logs" )).getAbsolutePath() );
        
        assertEquals(
            "Did not create the grobocoverage.properties file correctly",
            EXPECTED_10,
            loadGroboProperties() );
    }
    
    
    
    //...........................
    // Tests that fail
    
    public void testFInstrument1() throws Exception
    {
        expectBuildExceptionContaining( "f-instrument-1",
            "did not fail on no logdir specified",
            "Attribute 'logdir' was never set.");
    }
    
    
    public void testFInstrument2() throws Exception
    {
        expectBuildExceptionContaining( "f-instrument-2",
            "did not fail on no destdir specified",
            "Attribute 'destdir' was never set.");
    }
    
    
    public void testFInstrument3() throws Exception
    {
        expectBuildExceptionContaining( "f-instrument-3",
            "did not fail on no loggerprop value specified",
            "No value or location given for loggerprop key \"3\".");
    }
    
    
    public void testFInstrument4() throws Exception
    {
        expectBuildExceptionContaining( "f-instrument-4",
            "did not fail on no loggerprop key specified",
            "No key given for loggerprop.");
    }
    
    
    public void testFInstrument5() throws Exception
    {
        expectBuildExceptionContaining( "f-instrument-5",
            "did not fail on no loggerprop key specified",
            "No key given for loggerprop.");
    }
    
    
    public void testFInstrument6() throws Exception
    {
        expectBuildExceptionContaining( "f-instrument-6",
            "did not fail on no loggerprop key specified",
            "No key given for loggerprop.");
    }
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    protected void assertEquals( String text, Properties expected,
            Properties actual )
    {
        PropertyCheckUtil.assertEquals( text, expected, actual );
    }
    

    protected Properties loadGroboProperties()
            throws IOException
    {
        return loadProperties(
            (new File( getCoverageDir(),
                "classes" + File.separator + "grobocoverage.properties")).
            getAbsolutePath() );
    }
    
    
    protected Properties loadProperties( String file )
            throws IOException
    {
        return PropertyCheckUtil.loadProperties( file );
    }
    
    
    protected File getCoverageDir()
    {
        return new File( getProjectDir(), "instrument" +
            File.separator + "coverage" );
    }
    
    
    protected void assertModuleData( String text, String moduleName,
            String className )
    {
        StringBuffer errors = new StringBuffer( text );
        errors.append( "(for class " ).append( className ).
            append( " in module " ).append( moduleName ).
            append( ")" );
        boolean hasErrors = false;
        String prefix = className + "-";
        File parent = new File( getCoverageDir(), "logs" +
            File.separator + "data" + File.separator + moduleName );
        String contents[] = parent.list();
        boolean classdata = false;
        boolean classmarks = false;
        for (int i = 0; i < contents.length; ++i)
        {
            if (contents[i].startsWith( prefix ))
            {
                if (contents[i].endsWith( ".classdata.txt" ))
                {
                    System.out.println("Found class data file "+contents[i]);
                    if (classdata)
                    {
                        errors.append( "; too many classdata files" );
                        hasErrors = true;
                    }
                    else
                    {
                        classdata = true;
                    }
                }
                if (contents[i].endsWith( ".classmarks.txt" ))
                {
                    System.out.println("Found class marks file "+contents[i]);
                    if (classmarks)
                    {
                        errors.append( "; too many classmark files" );
                        hasErrors = true;
                    }
                    else
                    {
                        classmarks = true;
                    }
                }
            }
        }
        
        if (!classdata)
        {
            errors.append( "; did not create classdata file" );
            hasErrors = true;
        }
        if (!classmarks)
        {
            errors.append( "; did not create classmarks file" );
            hasErrors = true;
        }
        assertFalse( errors.toString(), hasErrors );
    }
    
    
    /**
     * Yes, it's duplication of code, but the first one will be used
     * more often, and gives a clearer error message set.
     */
    protected void assertModuleData( String text, String moduleName,
            String className, int count )
    {
        String baseErrorText = text + "(for class " + className +
            " in module " + moduleName + ") ";
        String prefix = className + "-";
        File parent = new File( getCoverageDir(), "logs" +
            File.separator + "data" + File.separator + moduleName );
        String contents[] = parent.list();
        int classdata = 0;
        int classmarks = 0;
        for (int i = 0; i < contents.length; ++i)
        {
            if (contents[i].startsWith( prefix ))
            {
                if (contents[i].endsWith( ".classdata.txt" ))
                {
                    ++classdata;
                }
                if (contents[i].endsWith( ".classmarks.txt" ))
                {
                    ++classmarks;
                }
            }
        }
        
        assertEquals(
            baseErrorText + "incorrect number of classdata files.",
            count,
            classdata );
        assertEquals(
            baseErrorText + "incorrect number of classmarks files.",
            count,
            classmarks );
    }
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
        configureProject( "instrument.xml" );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        executeTarget( "test-teardown" );
        
        super.tearDown();
    }
}

