/* classes: h_files */

#ifndef OBJECTSH
#define OBJECTSH

/*	Copyright (C) 1996 Free Software Foundation, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307 USA
 *
 * As a special exception, the Free Software Foundation gives permission
 * for additional uses of the text contained in its release of GUILE.
 *
 * The exception is that, if you link the GUILE library with other files
 * to produce an executable, this does not by itself cause the
 * resulting executable to be covered by the GNU General Public License.
 * Your use of that executable is in no way restricted on account of
 * linking the GUILE library code into it.
 *
 * This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU General Public License.
 *
 * This exception applies only to the code released by the
 * Free Software Foundation under the name GUILE.  If you copy
 * code from other Free Software Foundation releases into a copy of
 * GUILE, as the General Public License permits, the exception does
 * not apply to the code that you add in this way.  To avoid misleading
 * anyone as to the status of such modified files, you must delete
 * this exception notice from them.
 *
 * If you write modifications of your own for GUILE, it is your choice
 * whether to permit this exception to apply to your modifications.
 * If you do not wish that, delete this exception notice.  */


/* This file and objects.c contains those minimal pieces of the Guile
 * Object Oriented Programming System which need to be included in
 * libguile.
 *
 * {Objects and structs}
 *
 * Objects are currently based upon structs.  Although the struct
 * implementation will change thoroughly in the future, objects will
 * still be based upon structs.
 */

#include "libguile/__scm.h"
#include "libguile/struct.h"



/* {Class flags}
 *
 * These are used for efficient identification of instances of a
 * certain class or its subclasses when traversal of the inheritance
 * graph would be too costly.
 */
#define SCM_CLASS_FLAGS(class) (SCM_STRUCT_DATA (class)[scm_struct_i_tag])
#define SCM_OBJ_CLASS_FLAGS(obj)\
(SCM_STRUCT_VTABLE_DATA (obj)[scm_struct_i_tag])
#define SCM_SET_CLASS_FLAGS(c, f) (SCM_CLASS_FLAGS (c) |= (f))
#define SCM_CLEAR_CLASS_FLAGS(c, f) (SCM_CLASS_FLAGS (c) &= ~(f))
#define SCM_CLASSF_MASK (0xFF << 24)

/* Operator classes need to be identified in the evaluator. */
#define SCM_CLASSF_OPERATOR	(1L << 30)
/* Entities also have SCM_CLASSF_OPERATOR set in their vtable. */
#define SCM_CLASSF_ENTITY	(1L << 29)

#define SCM_I_OPERATORP(obj)\
((SCM_OBJ_CLASS_FLAGS (obj) & SCM_CLASSF_OPERATOR) != 0)
#define SCM_OPERATOR_CLASS(obj)\
((struct scm_metaclass_operator *) SCM_STRUCT_DATA (obj))
#define SCM_OBJ_OPERATOR_CLASS(obj)\
((struct scm_metaclass_operator *) SCM_STRUCT_VTABLE_DATA (obj))
#define SCM_OPERATOR_PROC_0(obj) (SCM_OBJ_OPERATOR_CLASS (obj)->proc0)
#define SCM_OPERATOR_PROC_1(obj) (SCM_OBJ_OPERATOR_CLASS (obj)->proc1)
#define SCM_OPERATOR_PROC_2(obj) (SCM_OBJ_OPERATOR_CLASS (obj)->proc2)
#define SCM_OPERATOR_PROC_3(obj) (SCM_OBJ_OPERATOR_CLASS (obj)->proc3)

#define SCM_I_ENTITYP(obj)\
((SCM_OBJ_CLASS_FLAGS (obj) & SCM_CLASSF_ENTITY) != 0)
#define SCM_ENTITY(obj) ((scm_entity*) SCM_STRUCT_DATA (obj))
#define SCM_ENTITY_PROC_0(obj) (SCM_ENTITY (obj)->proc0)
#define SCM_ENTITY_PROC_1(obj) (SCM_ENTITY (obj)->proc1)
#define SCM_ENTITY_PROC_2(obj) (SCM_ENTITY (obj)->proc2)
#define SCM_ENTITY_PROC_3(obj) (SCM_ENTITY (obj)->proc3)

/* {Operator classes}
 *
 * Instances of operator classes can work as operators, i. e., they
 * can be applied to arguments just as if they were ordinary
 * procedures.
 *
 * For instances of operator classes, the procedures to be applied are
 * stored in four dedicated slots in the associated class object.
 * Which one is selected depends on the number of arguments in the
 * application.
 *
 * If zero arguments are passed, the first will be selected.
 * If one argument is passed, the second will be selected.
 * If two arguments are passed, the third will be selected.
 * If three or more arguments are passed, the fourth will be selected.
 *
 * This is complicated and may seem gratuitous but has to do with the
 * architecture of the evaluator.  Using only one procedure would
 * result in a great deal less efficient application, loss of
 * tail-recursion and would be difficult to reconcile with the
 * debugging evaluator.
 *
 * Also, using this "forked" application in low-level code has the
 * advantage of speeding up some code.  An example is method dispatch
 * for generic operators applied to few arguments.  On the user level,
 * the "forked" application will be hidden by mechanisms in the GOOPS
 * package.
 *
 * Operator classes have the metaclass <operator-metaclass>.
 *
 * An example of an operator class is the class <tk-command>.
 */
#define SCM_METACLASS_STANDARD_LAYOUT "pwpw"
struct scm_metaclass_standard {
  SCM layout;
  SCM vcell;
  SCM vtable;
  SCM print;
  SCM direct_supers;
  SCM direct_slots;
};

#define SCM_METACLASS_OPERATOR_LAYOUT "pwpwpopopopo"
struct scm_metaclass_operator {
  SCM layout;
  SCM vcell;
  SCM vtable;
  SCM print;
  SCM direct_supers;
  SCM direct_slots;
  SCM proc0;
  SCM proc1;
  SCM proc2;
  SCM proc3;
};

/* {Entity classes}
 *
 * For instances of entity classes (entities), the procedures to be
 * applied are stored in the instance itself rather than in the class
 * object as is the case for instances of operator classes (see above).
 *
 * An example of an entity class is the class of generic methods.
 */
#define SCM_ENTITY_LAYOUT "popopopo"
typedef struct scm_entity {
  SCM proc0;
  SCM proc1;
  SCM proc2;
  SCM proc3;
} scm_entity;

extern SCM scm_metaclass_standard;
extern SCM scm_metaclass_operator;

extern SCM scm_set_object_procedure_x (SCM obj, SCM procs);
extern SCM scm_make_class_object (SCM metaclass, SCM layout);
extern SCM scm_make_subclass_object (SCM class, SCM layout);

extern void scm_init_objects SCM_P ((void));

#endif /* OBJECTSH */
