/* atmsigd.c - ATM signaling demon */

/* Written 1995-1997 by Werner Almesberger, EPFL-LRC */


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <limits.h>
#include <sys/types.h>
#include <linux/atm.h>

#include "atm.h"
#include "atmd.h"
#include "qlib.h"

#include "io.h"
#include "proto.h"
#include "saal.h"
#include "trace.h"


#define COMPONENT "SIGD"
#define CONFIG_FILE "/etc/atmsigd.conf"


#ifdef MEM_DEBUG
extern int mpr(void);
extern int mcheck (void (*)(int));
#endif

extern int yyparse(void);
extern FILE *yyin;

int net = 0;
int debug = 0;
int pretty = A2T_PRETTY | A2T_NAME;
int sig_pcr = -1; /* obsolete */
const char *sig_qos = NULL;
const char *dump_dir = NULL;


/* ------------------------------ SAAL relays ------------------------------ */


static SAAL_DSC saal;


static void q_estab_conf(void *user_data,void *uu_data,int uu_length)
{
    saal_okay();
}


static void q_rel_ind(void *user_data,void *uu_data,int uu_length)
{
    saal_failure();
    saal_estab_req(&saal,NULL,0);
}


static void q_restart(void *user_data,void *uu_data,int uu_length,int ind)
{
    saal_failure();
    if (!ind) saal_okay(); /* actually, ind should probably never be zero */
}


void from_net(void *buffer,int size)
{
    saal_pdu(&saal,buffer,size);
}


void to_signaling(void *msg,int size)
{
    trace_q2931("TO NETWORK",msg,size);
    saal_send(&saal,msg,size);
}


static void q_data_ind(void *user_data,void *data,int length)
{
    trace_q2931("FROM NETWORK",data,length);
    to_q2931(data,length);
}


static void q_cpcs_send(void *user_data,void *data,int length)
{
    to_net(data,length);
}


static SAAL_USER_OPS ops = {
    NULL, /* no q_estab_ind - 5.5.6.9 says 5.5.6.11 and 5.5.6.11 says "may" */
    q_estab_conf,
    q_rel_ind,
    NULL, /* no q_rel_conf - what to do ? */
    q_restart,
    q_data_ind,
    NULL, /* no q_unitdata */
    q_cpcs_send
};


/* -------------------------------- signals -------------------------------- */


static volatile int got_usr1 = 0,got_usr2 = 0;


static void dump_sap(FILE *file,const char *label,struct sockaddr_atmsvc *sap)
{
    struct atm_blli *blli;
    int i,length;

    if (!sap) return;
    fprintf(file,"  %s ",label);
    if (*sap->sas_addr.pub)
	fprintf(file,"%s%s",sap->sas_addr.pub,*sap->sas_addr.prv ? "+":
	  "\n    ");
    if (*sap->sas_addr.prv) {
	for (i = 0; i < ATM_ESA_LEN; i++)
	    fprintf(file,"%02X",sap->sas_addr.prv[i]);
	fprintf(file,"\n    ");
    }
    if (sap->sas_addr.bhli.hl_type) {
	length =
#ifdef UNI30
	   sap->sas_addr.bhli.hl_type == ATM_HL_HLP ? 4 :
#endif
	  sap->sas_addr.bhli.hl_type == ATM_HL_VENDOR ? 7 :
	  sap->sas_addr.bhli.hl_length;
	fprintf(file,"BHLI %d[%d] = { ",sap->sas_addr.bhli.hl_type,length);
	for (i = 0; i < length; i++)
	    fprintf(file,"%s%02X",i ? "," : "",sap->sas_addr.bhli.hl_info[i]);
	fprintf(file," }\n    ");
    }
    for (blli = sap->sas_addr.blli; blli; blli = blli->next) {
	fprintf(file,"BLLI");
	if (blli->l2_proto) {
	    fprintf(file,"  L2 %d",blli->l2_proto);
	    switch (blli->l2_proto) {
		case ATM_L2_X25_LL:
		case ATM_L2_X25_ML:
		case ATM_L2_HDLC_ARM:
		case ATM_L2_HDLC_NRM:
		case ATM_L2_HDLC_ABM:
		case ATM_L2_Q922:
		case ATM_L2_ISO7776:
		    fprintf(file," mode=%d, window=%d\n    ",
		      blli->l2.itu.mode,blli->l2.itu.window);
		    break;
		case ATM_L2_USER:
		    fprintf(file," user=0x%x\n    ",blli->l2.user);
		    break;
		default:
		    fprintf(file,"\n    ");
	    }
	}
	if (blli->l3_proto) {
	    fprintf(file,"  L3 %d",blli->l3_proto);
	    switch (blli->l3_proto) {
		case ATM_L3_X25:
		case ATM_L3_ISO8208:
		case ATM_L3_X223:
		    fprintf(file," mode=%d, def=%d, pack=%d\n    ",
		      blli->l3.itu.mode,blli->l3.itu.def_size,
		      blli->l3.itu.window);
		    break;
		case ATM_L3_TR9577:
		    fprintf(file," ipi=0x%x",blli->l3.tr9577.ipi);
		    if (blli->l3.tr9577.ipi != NLPID_IEEE802_1_SNAP)
			fprintf(file,"\n    ");
		    else {
			for (i = 0; i < 5; i++)
			    fprintf(file,"%s%02X",i ? "," : " ",
			      blli->l3.tr9577.snap[i]);
			fprintf(file,"\n    ");
		    }
		    break;
		case ATM_L3_USER:
		    fprintf(file," user=0x%x\n    ",blli->l3.user);
		    break;
		default:
		    fprintf(file,"\n    ");
	    }
	}
    }
    fprintf(file,"\n");
}


static void dump_status(FILE *file,const char *banner)
{
    SOCKET *walk;

    if (sockets) fprintf(file,"%s\n\n",banner);
    for (walk = sockets; walk; walk = walk->next) {
	fprintf(file,"0x%lx: %s, CR 0x%06lX, PVC %d.%d.%d\n",walk->id,
	  state_name[walk->state],walk->call_ref,walk->pvc.sap_addr.itf,
	  walk->pvc.sap_addr.vpi,walk->pvc.sap_addr.vci);
        dump_sap(file,"local ",walk->local);
        dump_sap(file,"remote",walk->remote);
    }
}


static void dump_trace(FILE *file,const char *banner)
{
    static int busy = 0;
    char *trace;

    if (busy++) abort();
    trace = get_trace();
    if (trace) {
	fprintf(file,"%s\n\n",banner);
	fprintf(file,"%s",trace);
    }
    busy--;
}


void poll_signals(void)
{
    static status_num = 0,trace_num = 0;
    char path[PATH_MAX+1];
    FILE *file;

    if (got_usr1) {
	got_usr1 = 0;
	if (!dump_dir) file = stderr;
	else {
	    sprintf(path,"atmsigd.%d.status.%d",getpid(),status_num++);
	    if ((file = fopen(path,"w")))
		diag(COMPONENT,DIAG_INFO,"Dumping to %s",path);
	    else {
		perror(path);
		file = stderr;
	    }
	}
	dump_status(file,"Status dump (on SIGUSR1)");
	if (file != stderr) (void) fclose(file);
    }
    if (got_usr2) {
	pid_t pid;

	got_usr2 = 0;
	if (!dump_dir) file = stderr;
	else {
	    sprintf(path,"atmsigd.%d.trace.%d",getpid(),trace_num++);
	    if ((file = fopen(path,"w")))
		diag(COMPONENT,DIAG_INFO,"Dumping to %s",path);
	    else {
		perror(path);
		file = stderr;
	    }
	}
	if (!(pid = fork()))
	     dump_trace(file,"Message trace (on SIGUSR2)");
	else if (pid < 0) perror("fork");
	if (file != stderr) (void) fclose(file);
	if (!pid) exit(0);
    }
}


static void handle_signal(int sig)
{
    switch (sig) {
	case SIGUSR1:
	    got_usr1 = 1;
	    break;
	case SIGUSR2:
	    got_usr2 = 1;
	    break;
	default:
	    break;
    }
}


static void setup_signals(void)
{
    struct sigaction act;

    (void) signal(SIGCHLD,SIG_IGN); /* reap children automatially */
    act.sa_handler = handle_signal;
    sigemptyset(&act.sa_mask);
    act.sa_flags = 0;
    if (sigaction(SIGUSR1,&act,NULL) < 0) {
	perror("sigaction");
	exit(1);
    }
    if (sigaction(SIGUSR2,&act,NULL) < 0) {
	perror("sigaction");
	exit(1);
    }
}


/* ------------------------------- main ...  ------------------------------- */


static void trace_on_exit(int status,void *dummy)
{
    char path[PATH_MAX+1];
    FILE *file;

    if (!status) return;
    if (dump_dir) file = stderr;
    else {
	sprintf(path,"atmsigd.%d.trace.exit",getpid());
	if (!(file = fopen(path,"w"))) {
	    perror(path);
	    file = stderr;
	}
    }
    dump_trace(file,"Message trace (after error exit)");
    if (file != stderr) (void) fclose(file);
}


static void usage(const char *name)
{
    fprintf(stderr,"usage: %s [ -b ] [ -c config_file ] [ -d ] [ -D dump_dir ]"
      " [ -l logfile ] [ -n ] [ -N ] [ -q qos ] [ -t trace_length ] "
      "[ [itf.]vpi.vci ]\n",name);
    exit(1);
}


int main(int argc,char **argv)
{
    const char *config_file;
    char *end;
    int c,background;

#ifdef MEM_DEBUG
    if (getenv("MCHECK")) mcheck(0);
    if (mpr() < 0) return 1;
#endif
    set_application("atmsigd");
    config_file = CONFIG_FILE;
    dump_dir = NULL;
    background = 0;
    memset(&signaling_pvc,0,sizeof(signaling_pvc));
    signaling_pvc.sap_addr.vci = 5;
    /* 1st pass to get the -c option */
    while ((c = getopt(argc,argv,"bc:dD:l:nNP:q:t:")) != EOF)
	if (c == 'c') config_file = optarg;
    if (!(yyin = fopen(config_file,"r")))
	diag(COMPONENT,DIAG_WARN,"%s not found. - Using defaults.",config_file);
    else if (yyparse()) {
	    diag(COMPONENT,DIAG_FATAL,"Error in config file. - Aborting.");
	    return 1;
	}
    /* process all other options but -c */
    optind = 0;
    while ((c = getopt(argc,argv,"bc:dD:l:nNP:q:t:")) != EOF)
	switch (c) {
	    case 'b':
		background = 1;
		break;
	    case 'c':
		/* already handled */
		break;
	    case 'd':
		set_verbosity(NULL,DIAG_DEBUG);
		set_verbosity("QMSG",DIAG_INFO);
		set_verbosity("SSCOP",DIAG_INFO);
		debug = 1;
		/*q_dump = 1;*/
		break;
	    case 'D':
		dump_dir = optarg;
		if (!trace_size) trace_size = DEFAULT_TRACE_SIZE;
		break;
	    case 'l':
		set_logfile(optarg);
		break;
	    case 'n':
		pretty = A2T_PRETTY;
		break;
	    case 'N':
		net = 1;
		break;
	    case 'q':
		if (sig_pcr != -1) usage(argv[0]);
		sig_qos = optarg;
		break;
	    case 'P': /* obsolete */
		if (sig_qos) usage(argv[0]);
		sig_pcr = strtol(optarg,&end,0);
		if (*end) usage(argv[0]);
		break;
	    case 't':
		trace_size = strtol(optarg,&end,0);
		if (*end) usage(argv[0]);
		break;
	    default:
		usage(argv[0]);
	}
    if (optind == argc-1) {
	if (text2atm(argv[optind],(struct sockaddr *) &signaling_pvc,
	  sizeof(signaling_pvc),T2A_PVC) < 0)
	    diag(COMPONENT,DIAG_FATAL,"text2atm \"%s\": failed",argv[optind]);
    }
    else if (optind != argc) usage(argv[0]);
    diag(COMPONENT,DIAG_INFO,"Linux ATM signaling "
#ifdef UNI30
      "UNI 3.0"
#endif
#ifdef UNI31
      "UNI 3.1"
#ifdef ALLOW_UNI30
      "+3.0compat"
#endif
#endif
      "/AAL5, version " VERSION);
    if (dump_dir)
	if (chdir(dump_dir) < 0)
	    diag(COMPONENT,DIAG_ERROR,"chdir %s: %s",dump_dir,strerror(errno));
    diag(COMPONENT,DIAG_INFO,"Acting as %s side",net ? "NETWORK" : "USER");
    if (open_all()) return 1;
    init_current_time();
    init_addr();
    q_start();
    start_saal(&saal,&ops,NULL);
    saal_estab_req(&saal,NULL,0);
    setup_signals();
    if (background) {
	pid_t pid;

	pid = fork();
	if (pid < 0)
	    diag(COMPONENT,DIAG_FATAL,"fork: %s",strerror(errno));
	if (pid) {
	    diag(COMPONENT,DIAG_DEBUG,"Backgrounding (PID %d)",pid);
	    exit(0);
	}
    }
    (void) on_exit(trace_on_exit,NULL);
    poll_loop();
    close_all();
    stop_saal(&saal);
    return 0;
}
