<!-- This Source Code Form is subject to the terms of the Mozilla Public
   - License, v. 2.0. If a copy of the MPL was not distributed with this
   - file, You can obtain one at http://mozilla.org/MPL/2.0/. -->

# Localization #

The SDK supports localization of strings appearing in:

* [your main add-on's JavaScript code](dev-guide/tutorials/l10n.html#Using Localized Strings in JavaScript) 
* [HTML files packaged with your add-on](dev-guide/tutorials/l10n.html#Using Localized Strings in HTML)
* [the `title` and `description` fields of your add-on's preferences](dev-guide/tutorials/l10n.html#Using Localized Strings in Preferences).

It doesn't, yet, support localization of CSS or content scripts.

## Localized Strings ##

Translated strings are kept in a directory called "locale" under your
main add-on directory, one file for each locale. The files:

* use the [`.properties` format](http://en.wikipedia.org/wiki/.properties)
* are named "xx-YY.properties", where "xx-YY" is the [name of the locale](https://wiki.mozilla.org/L10n:Locale_Codes) in question
* contain one entry for each string you want to localize, consisting
of an identifier for the string and its translation in that locale,
in the format `identifier=translation`.

Suppose your add-on contains a single localizable string,
represented in English as "Hello!", and you want to supply US English
and French French localizations.

You'd add two files to the "locale" directory:

<pre>
my-addon/
         data
         lib
         locale/
                en-US.properties
                fr-FR.properties
</pre>

"en.US.properties" contains this:

<pre>
hello_id= Hello!
</pre>

"fr.FR.properties" contains this:

<pre>
hello_id= Bonjour !
</pre>

Now whenever your JavaScript or HTML asks the localization system for
the translation of the `hello_id` identifier, it will get the correct
translation for the current locale.

## Using Localized Strings in HTML ##

To reference localized strings from HTML, add a `data-l10n-id` attribute to
the HTML tag where you want the localized string to appear, and assign
the identifier to it:

<pre class="brush: html">
&lt;html&gt;
  &lt;body&gt;
    &lt;h1 data-l10n-id="hello_id"&gt;&lt;/h1>&gt;
  &lt;/body&gt;
&lt;/html&gt;
</pre>

Then you can use this HTML file to build your interface, for example
inside a panel:

    var hello = require("sdk/panel").Panel({
      height: 75,
      width: 150,
      contentURL: require("sdk/self").data.url("my-panel.html")
    });

    var widget = require("sdk/widget").Widget({
      id: "mozilla-link",
      label: "Mozilla website",
      contentURL: "http://www.mozilla.org/favicon.ico",
      panel: hello
    });

<img class="image-right" src="static-files/media/screenshots/l10n-html-enUS.png"
alt="Example of panel containing US English text">

<img class="image-right" src="static-files/media/screenshots/l10n-html-frFR.png"
alt="Example of panel containing French French text">

Given locale files for "en-US" and "fr-FR" which provide translations
of `hello_id`, the panel will now display "Hello!" or "Bonjour !", according
to the current locale.

The translation is inserted into the node which has the `data-l10n-id`
attribute set. Any previously existing content is just replaced.

The string is inserted as text, so you can't insert HTML using a statement
like:

<pre>
hello_id= &lt;blink&gt;Hello!&lt;/blink&gt;
</pre>

## Using Localized Strings in JavaScript

To reference localized strings from your main add-on code, you do this:

    var _ = require("sdk/l10n").get;
    console.log(_("hello_id!"));

<span class="aside">Assigning to "`_`" in particular is not required, but
is a convention from the
[gettext](https://www.gnu.org/software/gettext/gettext.html) tools
and will make it possible to work with existing tools that expect "`_`"
to indicate localizable strings.</span>

1. Import the `l10n` module, and assign its `get` function to
"`_`" (underscore).
2. Wrap all references to localizable strings with the `_()`
function.

If you run it you'll see the expected output for the current locale:

<pre>
info: Hello!
</pre>

<pre>
info: Bonjour !
</pre>

Note that because you can't `require()` modules in content scripts,
you can't yet reference localized strings from content scripts.

### Plurals ###

The `l10n` module supports plural forms. Different languages have
different rules for the formation of plurals. For example,
English has two forms: a singular form for "one", and a plural form
for "everything else, including zero":

<pre>
one tomato
no tomatoes
two tomatoes
</pre>

But Russian has different forms for numbers ending in 1 (except 11),
numbers ending in 2-4 (except 12-14) and other numbers:

<pre>
один помидор     // one tomato
два помидора     // two tomatoes
пять помидоров   // five tomatoes
</pre>

The SDK uses the [Unicode CLDR](http://cldr.unicode.org/index) data
to describe the different plural forms used by different languages.

#### Unicode CLDR Plural Forms ####

The Unicode CLDR project defines a scheme for describing a particular
language's plural rules. In this scheme a language maps each distinct
range of numbers on to one of up to six forms, identified by the
following categories: *zero*, *one*, *two*, *few*, *many*, and *other*.

English has two forms, which can be described by mapping "1" to "one"
and "everything else" to "other":

<pre>
one   → n is 1;
other → everything else
</pre>

Russian uses four forms, that can be described as follows:

<pre>
one   → n mod 10 is 1 and n mod 100 is not 11;
few   → n mod 10 in 2..4 and n mod 100 not in 12..14;
many  → n mod 10 is 0 or n mod 10 in 5..9 or n mod 100 in 11..14;
other → everything else
</pre>

Plural rules for all languages can be found in the CLDR
[Language Plural Rules](http://unicode.org/repos/cldr-tmp/trunk/diff/supplemental/language_plural_rules.html)
page (although this table is out of date compared to the
[CLDR XML source](http://unicode.org/repos/cldr/trunk/common/supplemental/plurals.xml)).

#### Plural Forms in the SDK ####

In the code, you supply an extra parameter alongside the identifier,
describing how many items there are:

    var _ = require("sdk/l10n").get;
    console.log(_("tomato_id"));
    console.log(_("tomato_id", 1));
    console.log(_("tomato_id", 2));
    console.log(_("tomato_id", 5));
    console.log(_("tomato_id", .5));

In the `.properties` file for each language you can define a different
localization for each plural form possible in that language, using the
CLDR keywords. So in English we could have two plural localizations
(note that the "other" category does **not** take the CLDR keyword):

<pre>
# en-US translations
tomato_id[one]= %d tomato
tomato_id= %d tomatoes
</pre>

In Russian we could have four plural localizations:

<pre>
# ru-RU translations
tomato_id[one]= %d помидор
tomato_id[few]= %d помидора
tomato_id[many]= %d помидоров
tomato_id= %d помидоры
</pre>

The localization module itself understands the CLDR definitions for each
language, enabling it to map between, for example, "2" in the code and
"few" in the `ru-RU.properties` file. Then it retrieves and returns
the localization appropriate for the count you supplied.

### Placeholders ###

The `l10n` module supports placeholders, allowing you to
insert a string which should not be localized into one which is.
The following "en-US" and "fr-FR" ".properties" files include
placeholders:

<pre>
# en-US translations
hello_id= Hello %s!
</pre>

<pre>
# fr-FR translations
hello_id= Bonjour %s !
</pre>

To use placeholders, supply the placeholder string after the identifier:

    var _ = require("sdk/l10n").get;
    console.log(_("hello_id", "Bob"));
    console.log(_("hello_id", "Alice"));

In the "en-US" locale, this gives us:

<pre>
info: Hello Bob!
info: Hello Alice!
</pre>

In "fr-FR" we get:

<pre>
info: Bonjour Bob !
info: Bonjour Alice !
</pre>

### Ordering Placeholders ###

When a localizable string can take two or more placeholders, translators
can define the order in which placeholders are inserted, without affecting
the code.

Primarily, this is important because different languages have different
rules for word order. Even within the same language, though, translators
should have the freedom to define word order.

For example, suppose we want to include a localized string naming a
person's home town. There are two placeholders: the name of the person
and the name of the home town:

    var _ = require("sdk/l10n").get;
    console.log(_("home_town_id", "Bob", "London"));

An English translator might want to choose between the following:

<pre>
"&lt;town_name> is &lt;person_name>'s home town."
</pre>

<pre>
"&lt;person_name>'s home town is &lt;town_name>"
</pre>

To choose the first option, the `.properties` file can order the
placeholders as follows:

<pre>
home_town_id= %2s is %1s's home town.
</pre>

This gives us the following output:

<pre>
info: London is Bob's home town.
</pre>

## Using Localized Strings in Preferences ##

By including a
[`"preferences"` structure in your add-on's "package.json" file](modules/sdk/simple-prefs.html ), you can define
preferences for your add-on that the user can see and edit
using Firefox's
[Add-ons Manager](https://support.mozilla.org/en-US/kb/Using%20extensions%20with%20Firefox#w_how-to-change-extension-settings).

Preferences have mandatory `title` and optional `description` fields.
These are strings which appear alongside the preference in the Add-ons
Manager, to help explain to the user what the preference means.

* To provide the localized form of the preference title, include an
entry in your "properties" file whose identifier is the preference
name followed by `_title`, and whose value is the localized title.

* To provide the localized form of the preference description, include
an entry in your "properties" file whose identifier is the preference
name followed by `_description`, and whose value is the localized description.

For example, suppose your "package.json" defines a single preference:

<pre>
{
    "preferences": [
        {
            "type": "string", 
            "name": "monster_name", 
            "value": "Gerald",
            "title": "Name"
        }
    ], 
    "name": "monster-builder", 
    "license": "MPL 2.0", 
    "author": "me", 
    "version": "0.1", 
    "fullName": "Monster Builder", 
    "id": "monster-builder@me.org", 
    "description": "Build your own monster"
}
</pre>

In your "en-US.properties" file, include these two items:

<pre>
monster_name_title= Name
monster_name_description= What is the monster's name?
</pre>

In your "fr-FR.properties" file, include the French translation:

<pre>
monster_name_title= Nom
monster_name_description= Quel est le nom du monstre ?
</pre>

Now when the browser's locale is set to "en-US", users see this
in the Add-ons Manager:

<img class="image-center" src="static-files/media/screenshots/preference-us.png" alt="screenshot of US preference localization">

When the browser's locale is set to "fr-FR", they see this:

<img class="image-center" src="static-files/media/screenshots/preference-french.png" alt="screenshot of French preference localization">

The `menulist` and the `radio` preference types have options.
The `label` attribute of each option is displayed to the user.
If the locale file has a entry with the value of the `label` attribute
prefixed with "{name}_options." as its key, where {name} is the name of 
the preference, its value is used as a localized label.

## Using Identifiers ##

If the localization system can't find an entry for a particular identifier
using the current locale, then it just returns the identifier itself.

This has the nice property that you can write localizable, fully
functional add-ons without having to write any locale files. You can just
use the default language strings as your identifier, and subsequently supply
`.properties` files for all the additional locales you want to support.

For example, in the case above you could use "Hello!" as the identifier, and
just have one `.properties` file for the "fr-FR" locale:

<pre>
Hello!= Bonjour !
</pre>

Then when the locale is "en-US", the system would fail to find a `.properties`
file, and return "Hello!".

However, this approach makes it difficult to maintain an add-on which
has many localizations, because you're using the default language strings
both as user interface strings and as keys to look up your translations.
This means that if you want to change the wording of a string in the default
language, or fix a typo, then you break all your locale files.

## Locale Updater ##

The [locale updater](https://github.com/downloads/ochameau/locale-updater/locale-updater.xpi)
add-on makes it easier to update locale files. Once you've installed it,
open the Add-on Manager, and you'll see a see a new button labeled
"Update l10n" next to each add-on you've installed:

<img class="align-center" src="static-files/media/screenshots/locale-updater.png"
alt="Add-on manager with locale updater installed" />

Click the button and you'll be prompted for a new `.properties` file
for that add-on. If you provide a new file, the add-on's locale data
will be updated with the new file.

## <a name="limitations">Limitations</a> ##

The current localization support is a first step towards full support,
and contains a number of limitations.

* There's no support for content scripts or CSS files: at
the moment, you can only localize strings appearing in JavaScript files
that can `require()` SDK modules and in HTML.

* The set of locale files is global across an add-on. This means that
a module isn't able to override a more general translation: so a module
`informal.js` can't specify that "hello_id" occurring in its code
should be localized to "Hi!".

* The SDK tools compile the locale files into a JSON format when
producing an XPI. This means that translators can't localize an add-on
given the XPI alone, but must be given access to the add-on source.

* The add-on developer must manually assemble the set of localizable
strings that make up the locale files. In a future release we'll add
a command to `cfx` that scans the add-on for localizable strings and
builds a template `.properties` file listing all the strings that need
to be translated.

