/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <config.h>

#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <pan/base/argset.h>
#include <pan/base/debug.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/log.h>

#include <pan/globals.h>
#include <pan/gui.h>
#include <pan/prefs.h>
#include <pan/util.h>

/***
****
****  GUI
****
***/

typedef struct
{
	GtkWindow * window;
	GtkMessageType type;
	char * text;
}
IdleDialogStruct;

static int
pan_dialog_idle (gpointer user_data)
{
	GtkWidget * w;
	IdleDialogStruct * info = (IdleDialogStruct*) user_data;

	/* build & show the dialog */
	pan_lock();
	if (!GTK_IS_WINDOW(info->window)) /* make info->window is alive */
		info->window = GTK_WINDOW(Pan.window);
	w = gtk_message_dialog_new (GTK_WINDOW(info->window),
	                            GTK_DIALOG_DESTROY_WITH_PARENT,
	                            info->type,
				    GTK_BUTTONS_CLOSE,
				    "%s", info->text);
	g_signal_connect_swapped (GTK_OBJECT(w), "response",
	                          G_CALLBACK (gtk_widget_destroy),
	                          GTK_OBJECT(w));
	gtk_widget_show_all (w);
	pan_unlock();

	/* cleanup */
	g_free (info->text);
	g_free (info);
	return 0;
}

static void
pan_dialog (char * i_own_it_str, GtkMessageType type, GtkWindow * window)
{
	IdleDialogStruct * info = g_new (IdleDialogStruct, 1);
	info->text = i_own_it_str;
	info->type = type;
	info->window = window;
	gui_queue_add (pan_dialog_idle, info);
}
void
pan_info_dialog (const char *format, ...)
{
	va_list args;
	char *str = NULL;
	g_return_if_fail (format != NULL);
	va_start (args, format);
	str = g_strdup_vprintf (format, args);
	va_end (args);
	g_warning (str);

	pan_dialog (str, GTK_MESSAGE_INFO, GTK_WINDOW(Pan.window));
}

void
pan_error_dialog_parented (gpointer parent, const char *format, ...)
{
	va_list args;
	char *str = NULL;
	g_return_if_fail (format != NULL);
	va_start (args, format);
	str = g_strdup_vprintf (format, args);
	va_end (args);
	g_warning (str);

	pan_dialog (str, GTK_MESSAGE_ERROR, GTK_WINDOW(parent));
}

void
pan_error_dialog (const char *format, ...)
{
	va_list args;
	char *str = NULL;
	g_return_if_fail (format != NULL);
	va_start (args, format);
	str = g_strdup_vprintf (format, args);
	va_end (args);
	g_warning (str);

	pan_dialog (str, GTK_MESSAGE_ERROR, GTK_WINDOW(Pan.window));
}

/***
****
****  IDLE FUNC
****
***/

/**
 * Thus spoke the GTK FAQ: "Callbacks require a bit of attention.
 * Callbacks from GTK+ (signals) are made within the GTK+ lock. However
 * callbacks from GLib (timeouts, IO callbacks, and idle functions) are
 * made outside of the GTK+ lock. So, within a signal handler you do not
 * need to call gdk_threads_enter(), but within the other types of
 * callbacks, you do."
 *
 * pan_timeout_add() and run_in_main_thread_idle() are wrappers around
 * glib-level callbacks that ensure that pan_lock() works properly
 * to give them a gdk_threads_enter() lock.
 */

static gboolean main_thread_is_in_glib_callback = FALSE;

static gint
pan_timeout_wrapper (gpointer data)
{
	ArgSet * argset = (ArgSet*) data;
	GtkFunction func = (GtkFunction) argset_get (argset, 0);
	gpointer user_data = (gpointer) argset_get (argset, 1);
	gint retval;

	/* sanity clause */
	pan_warn_if_fail (g_thread_self() == Pan.main_t);

	/* call the timer func */
	main_thread_is_in_glib_callback = TRUE;
	retval = (*func)(user_data);
	main_thread_is_in_glib_callback = FALSE;

	/* cleanup */
	if (!retval)
		argset_free (argset);
	return retval;
}
guint
pan_timeout_add (guint32 interval, GtkFunction func, gpointer arg)
{
	return gtk_timeout_add (interval, pan_timeout_wrapper, argset_new2 (func, arg));
}

/***
****
****  LOCKING
****
***/

static const GThread * has_lock_thr = NULL;

void
pan_lock_from (const gchar * file, const gchar * func, int line)
{
	static const gchar * last_file = NULL;
	static const gchar * last_func = NULL;
	static int last_line = -1;
	const GThread * thr = g_thread_self ();

	/**
	 * If pan_lock() is called from the main thread while it has a GUI lock
	 * (typically from a gtk signals, like a button press signal etc.)
	 * then we don't need to lock.
	 *
	 * However if pan_lock() is called from a worker thread, or the main
	 * thread inside a glib idle function (via run_in_main_thread_nolock())
	 * then we _do_ need to obtain a gtk lock.
	 */
	if (thr==Pan.main_t && !main_thread_is_in_glib_callback)
	{
		debug4 (DEBUG_LOCK,"mainthread %p attempted unnecessary lock from %s:%d (%s)", thr, file, line, func);
	}
	else if (thr == has_lock_thr)
	{
		g_error ("thread %p attempted double lock!\nfirst lock was in %s:%d (%s),\nnow trying for another one from %s:%d (%s)",
			thr,
			last_file, last_line, last_func,
			file, line, func);
	}
	else
	{
	       	/* if (thr==Pan.main_t && main_thread_is_in_glib_callback)
			odebug3 ("idle func %s:%d (%s) getting a pan lock", file, line, func);*/

		gdk_threads_enter();
		last_file = file;
		last_func = func;
		last_line = line;
		has_lock_thr = thr;
		debug3 (DEBUG_LOCK,"thread %p entered gdk_threads from %s %d", thr, file, line);
	}
}

void
pan_unlock_from (const gchar* file, const gchar * func, int line)
{
	const GThread* thr = g_thread_self ();

	if (thr==Pan.main_t && !has_lock_thr)
	{
		debug4 (DEBUG_LOCK,"mainthread %p attempted unneccessary unlock from %s:%d (%s)", thr, file, line, func);
	}
	else if (has_lock_thr != thr)
	{
		g_error ("thread %p attempted to remove a lock it didn't have from %s:%d (%s)", thr, file, line, func);
	}
	else
	{
		has_lock_thr = NULL;
		gdk_threads_leave();
		debug4 (DEBUG_LOCK,"thread %p left gdk_threads from %s:%d (%s)", g_thread_self(), file, line, func);
	}
}

/***
****
****  UNSORTED
****
***/

void
get_date_display_string (time_t date, const char * fmt, char * fillme, int len)
{
	struct tm tm_date;

	/* sanity clause */
	g_return_if_fail (is_nonempty_string(fmt));
	g_return_if_fail (fillme != NULL);
	*fillme = '\0';

	/* get the date string */
	pan_localtime_r (&date, &tm_date);

	/* try to put the date in the buffer */
	if (!strftime (fillme, len, fmt, &tm_date))
		*fillme = '\0';
}

static gboolean
try_to_execute_url (const gchar * template, const gchar * url)
{
	gboolean retval;
	char * str;

	/* sanity clause */
	g_return_val_if_fail (is_nonempty_string(url), FALSE);
	if (template == NULL)
		template = "netscape %s";

	/* add the URL to the template */
	if (strstr(template,"%s"))
		str = pan_substitute (template, "%s", url);
	else
		str = g_strdup_printf ("%s %s", template, url);
	replace_gstr (&str, pan_substitute(str, "%%%%", "%%"));

	/* execute it */
	g_message ("%s", str);
	retval = g_spawn_command_line_async (str, NULL);

	/* cleanup */
	g_free (str);
	return retval;
}


void
pan_url_show (const char * url)
{
	const char * cmd;

	/* sanity clause */
	g_return_if_fail (is_nonempty_string(url));

	if (is_nonempty_string (external_web_browser))
		cmd = external_web_browser;
	else
		cmd = g_getenv ("BROWSER");

	if (is_nonempty_string (cmd))
	{
		int i;
		char ** tries;
		gboolean result = FALSE;
		
		tries = g_strsplit (external_web_browser, ":", -1);
		for (i=0; !result && tries[i]!=NULL; ++i)
			result = try_to_execute_url (tries[i], url);

		g_strfreev (tries);
	}
}


/***
****  Menu Building
***/

static char*
menu_translate(const char* path, gpointer data)
{
	return gettext (path);
}

GtkWidget*
menubar_create (GtkWidget            * window,
                GtkItemFactoryEntry  * entries,
                guint                  entries_qty,
                const char           * path,
                gpointer               data)
{
	GtkItemFactory *factory;
	GtkAccelGroup *accel_group;

	accel_group = gtk_accel_group_new ();
	gtk_window_add_accel_group (GTK_WINDOW (window), accel_group);
	g_object_unref (accel_group);

	factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, path, accel_group);
	gtk_item_factory_set_translate_func(factory, menu_translate, NULL, NULL);
	gtk_item_factory_create_items(factory, entries_qty, entries, data);

	return gtk_item_factory_get_widget (factory, path);
}

GtkWidget*
menu_create_items (GtkItemFactoryEntry   * entries,
                   guint                   entries_qty,
                   const char            * path,
                   GtkItemFactory       ** factory,
                   gpointer                data)
{
	*factory = gtk_item_factory_new(GTK_TYPE_MENU, path, NULL);
	gtk_item_factory_set_translate_func(*factory, menu_translate, NULL, NULL);
	gtk_item_factory_create_items(*factory, entries_qty, entries, data);

	return gtk_item_factory_get_widget(*factory, path);
}

void
menu_set_sensitive (GtkItemFactory   * ifactory,
                    const char       * path,
                    gboolean           sensitive)
{
	GtkWidget *widget;

	g_return_if_fail(ifactory != NULL);

	widget = gtk_item_factory_get_item (ifactory, path);
	if (widget == NULL)
		g_error ("couldn't find menu item \"%s\"", path);

	gtk_widget_set_sensitive (widget, sensitive);
}

void
menu_set_checked (GtkItemFactory   * ifactory,
                  const char       * path,
                  gboolean           active)
{
	GtkCheckMenuItem * w;

	g_return_if_fail (ifactory != NULL);

	w = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (ifactory, path));
	if (w == NULL)
		g_error ("couldn't find menu item \"%s\"", path);
	else if (!!w->active != !!active)
		gtk_check_menu_item_set_active (w, active);
}

gboolean
menu_is_checked (GtkItemFactory   * ifactory,
                 const char       * path)
{
	GtkWidget * w;

	g_return_val_if_fail (ifactory!=NULL, FALSE);
	w = gtk_item_factory_get_item (ifactory, path);
	g_return_val_if_fail (GTK_IS_CHECK_MENU_ITEM(w), FALSE);

	return gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM(w));
}

void
menu_set_insensitive_all(GtkMenuShell * menu_shell)
{
	GList * cur;

	for (cur = menu_shell->children; cur != NULL; cur = cur->next)
		gtk_widget_set_sensitive(GTK_WIDGET(cur->data), FALSE);
}

void
pan_gtk_entry_set_text (GtkWidget            * w,
                        const char           * text)
{
	char * pch;

	g_return_if_fail (GTK_IS_ENTRY(w));
	pch = pan_header_to_utf8 (text, -1, NULL);
	gtk_entry_set_text (GTK_ENTRY(w), pch ? pch : "");
	g_free (pch);
}

void
pan_hig_frame_new        (const char           * title,
                          GtkWidget           ** setme_container,
                          GtkWidget           ** setme_vbox)
{
	char * pch;
	GtkWidget * frame;
	GtkWidget * h;
	GtkWidget * v;
	GtkWidget * w;

	pch = g_strdup_printf ("<b>%s</b>", title);
	frame = gtk_frame_new (pch);
	g_free (pch);
	gtk_frame_set_shadow_type (GTK_FRAME(frame), GTK_SHADOW_NONE);
	gtk_label_set_use_markup (GTK_LABEL(gtk_frame_get_label_widget (GTK_FRAME(frame))), TRUE);

	h = gtk_hbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(frame), h);

	w = gtk_alignment_new (0.0f, 0.0f, 0.0f, 0.0f);
	gtk_widget_set_usize (w, 12u, 0u);
	gtk_box_pack_start (GTK_BOX(h), w, FALSE, FALSE, 0);

	v = gtk_vbox_new (FALSE, GUI_PAD_SMALL);
	gtk_box_pack_start (GTK_BOX(h), v, TRUE, TRUE, 0);

	*setme_container = frame;
	*setme_vbox = v;
}

/*
 * Check for sig delims
 */

int
pan_is_signature_delimitor (const char * line, const int line_len)
{
	switch (line_len)
	{
		case 2: if (!strncmp (line,"--"   ,2)) return SIG_NON_STANDARD;
					break;
		
		case 3: if (!strncmp (line,"--\r" ,3)) return SIG_NON_STANDARD;
				if (!strncmp (line,"-- "  ,3)) return SIG_STANDARD;
				break;
		
		case 4: if (!strncmp (line,"-- \r",4)) return SIG_STANDARD;
		
		default:
			return SIG_NONE;
	}
	return SIG_NONE;
}

gboolean
pan_remove_signature (char * body)
{
	const char * march = body;
	char * line = NULL;
	char * sig_point = NULL;
	int sig_type = SIG_NONE;
	int lines_below = 0;
	int line_len = 0;

	/*iterate through the text, line by line*/
	while (get_next_token_run (march, '\n', &march, (const char **) &line, &line_len))
	{
		int st;
		st = pan_is_signature_delimitor (line, line_len);
		if (st == SIG_STANDARD || st == SIG_NON_STANDARD)
		{
			sig_type = st;
			sig_point = line;
			lines_below = 0;
			
		} else 
		{
			if (sig_point)
				lines_below++;
		}
	}
	if (sig_type == SIG_NONE) return FALSE;
	if (sig_point == NULL) return FALSE;
	if (sig_type == SIG_STANDARD)
	{
		*sig_point = '\0';
		return TRUE;
	}
	
	
	/*if we have a non-standard sig, make sure it's the last one and that
	 * there are less than SIG_THRESHOLD lines
	 */
	if (sig_type == SIG_NON_STANDARD && lines_below <= SIG_THRESHOLD )
	{
		*sig_point = '\0';
		return TRUE;
	}
	
	return FALSE;
}
