/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* libwps
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2006, 2007 Andrew Ziem
 * Copyright (C) 2004 Marc Maurer (uwog@uwog.net)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 */

#include <stdlib.h>
#include <string.h>

#include <algorithm>
#include <cmath>
#include <sstream>
#include <stack>
#include <utility>

#include <librevenge-stream/librevenge-stream.h>

#include "libwps_internal.h"
#include "libwps_tools_win.h"

#include "WKSContentListener.h"
#include "WKSSubDocument.h"

#include "WPSCell.h"
#include "WPSEntry.h"
#include "WPSFont.h"
#include "WPSHeader.h"
#include "WPSOLEParser.h"
#include "WPSOLEStream.h"
#include "WPSPageSpan.h"
#include "WPSStream.h"
#include "WPSStringStream.h"

//#include "QuattroChart.h"
#include "QuattroGraph.h"
#include "QuattroSpreadsheet.h"

#include "Quattro.h"

using namespace libwps;

//! Internal: namespace to define internal class of QuattroParser
namespace QuattroParserInternal
{
//! the font of a QuattroParser
struct Font final : public WPSFont
{
	//! constructor
	explicit Font(libwps_tools_win::Font::Type type) : WPSFont(), m_type(type)
	{
	}
	//! font encoding type
	libwps_tools_win::Font::Type m_type;
};

//! Internal: the subdocument of a WPS4Parser
class SubDocument final : public WKSSubDocument
{
public:
	//! constructor for a text entry
	SubDocument(RVNGInputStreamPtr const &input, QuattroParser &pars, bool header) :
		WKSSubDocument(input, &pars), m_header(header) {}
	//! destructor
	~SubDocument() final {}

	//! operator==
	bool operator==(std::shared_ptr<WPSSubDocument> const &doc) const final
	{
		if (!doc || !WKSSubDocument::operator==(doc))
			return false;
		auto const *sDoc = dynamic_cast<SubDocument const *>(doc.get());
		if (!sDoc) return false;
		return m_header == sDoc->m_header;
	}

	//! the parser function
	void parse(std::shared_ptr<WKSContentListener> &listener, libwps::SubDocumentType subDocumentType) final;
	//! a flag to known if we need to send the header or the footer
	bool m_header;
};

void SubDocument::parse(std::shared_ptr<WKSContentListener> &listener, libwps::SubDocumentType)
{
	if (!listener.get())
	{
		WPS_DEBUG_MSG(("QuattroParserInternal::SubDocument::parse: no listener\n"));
		return;
	}
	if (!dynamic_cast<WKSContentListener *>(listener.get()))
	{
		WPS_DEBUG_MSG(("QuattroParserInternal::SubDocument::parse: bad listener\n"));
		return;
	}

	QuattroParser *pser = m_parser ? dynamic_cast<QuattroParser *>(m_parser) : nullptr;
	if (!pser)
	{
		listener->insertCharacter(' ');
		WPS_DEBUG_MSG(("QuattroParserInternal::SubDocument::parse: bad parser\n"));
		return;
	}
	pser->sendHeaderFooter(m_header);
}

//! the state of QuattroParser
struct State
{
	//! constructor
	explicit State(libwps_tools_win::Font::Type fontType, char const *password)
		: m_fontType(fontType)
		, m_version(-1)
		, m_actualSheet(-1)
		, m_fontsList()
		, m_colorsList()
		, m_idToFileNameMap()
		, m_idToFieldMap()
		, m_pageSpan()
		, m_actPage(0)
		, m_numPages(0)
		, m_headerString()
		, m_footerString()
		, m_password(password)
		, m_isEncrypted(false)
		, m_isDecoded(false)
	{
	}
	//! return the default font style
	libwps_tools_win::Font::Type getDefaultFontType() const
	{
		if (m_fontType != libwps_tools_win::Font::UNKNOWN)
			return m_fontType;
		return libwps_tools_win::Font::WIN3_WEUROPE;
	}

	//! returns a color corresponding to an id
	bool getColor(int id, WPSColor &color) const;
	//! returns a default font (Courier12) with file's version to define the default encoding */
	WPSFont getDefaultFont() const
	{
		WPSFont res;
		res.m_name="Times New Roman";
		res.m_size=12;
		return res;
	}

	//! the user font type
	libwps_tools_win::Font::Type m_fontType;
	//! the file version
	int m_version;
	//! the actual sheet
	int m_actualSheet;
	//! the font list
	std::vector<Font> m_fontsList;
	//! the color list
	std::vector<WPSColor> m_colorsList;
	//! map id to filename
	std::map<int, librevenge::RVNGString> m_idToFileNameMap;
	//! map id to field
	std::map<int, std::pair<librevenge::RVNGString,WKSContentListener::FormulaInstruction> >m_idToFieldMap;
	//! the actual document size
	WPSPageSpan m_pageSpan;
	int m_actPage /** the actual page*/, m_numPages /* the number of pages */;
	//! the header string
	librevenge::RVNGString m_headerString;
	//! the footer string
	librevenge::RVNGString m_footerString;
	//! the password (if known)
	char const *m_password;
	//! true if the file is encrypted
	bool m_isEncrypted;
	//! true if the main stream has been decoded
	bool m_isDecoded;
private:
	State(State const &)=delete;
	State &operator=(State const &)=delete;
};

bool State::getColor(int id, WPSColor &color) const
{
	if (m_colorsList.empty())
	{
		auto *THIS=const_cast<State *>(this);
		static const uint32_t quattroColorMap[]=
		{
			0xFFFFFF, 0xC0C0C0, 0x808080, 0x000000,
			0xFF0000, 0x00FF00, 0x0000FF, 0x00FFFF,
			0xFF00FF, 0xFFFF00, 0x800080, 0x000080,
			0x808000, 0x008000, 0x800000, 0x008080
		};
		for (int i=0; i<int(WPS_N_ELEMENTS(quattroColorMap)); ++i)
			THIS->m_colorsList.push_back(WPSColor(quattroColorMap[i]));
	}
	if (id < 0 || id >= int(m_colorsList.size()))
	{
		WPS_DEBUG_MSG(("QuattroParserInternal::State::getColor(): unknown Quattro Pro color id: %d\n",id));
		return false;
	}
	color = m_colorsList[size_t(id)];
	return true;
}

}

// constructor, destructor
QuattroParser::QuattroParser(RVNGInputStreamPtr &input, WPSHeaderPtr &header,
                             libwps_tools_win::Font::Type encoding, char const *password)
	: WKSParser(input, header)
	, m_listener()
	, m_state(new QuattroParserInternal::State(encoding, password))
	  //, m_chartParser(new QuattroChart(*this))
	, m_graphParser(new QuattroGraph(*this))
	, m_spreadsheetParser(new QuattroSpreadsheet(*this))
{
}

QuattroParser::~QuattroParser()
{
}

int QuattroParser::version() const
{
	return m_state->m_version;
}

libwps_tools_win::Font::Type QuattroParser::getDefaultFontType() const
{
	return m_state->getDefaultFontType();
}

librevenge::RVNGString QuattroParser::getFileName(int fId) const
{
	auto it = m_state->m_idToFileNameMap.find(fId);
	if (it!=m_state->m_idToFileNameMap.end())
		return it->second;
	WPS_DEBUG_MSG(("QuattroParser::getFileName: can not find %d name\n", fId));
	return "";
}

bool QuattroParser::getField(int fId, librevenge::RVNGString &text,
                             WKSContentListener::FormulaInstruction &instr) const
{
	auto it = m_state->m_idToFieldMap.find(fId);
	if (it!=m_state->m_idToFieldMap.end())
	{
		text=it->second.first;
		instr=it->second.second;
		return true;
	}
	WPS_DEBUG_MSG(("QuattroParser::getField: can not find %d field\n", fId));
	return false;
}

//////////////////////////////////////////////////////////////////////
// interface with QuattroGraph
//////////////////////////////////////////////////////////////////////
bool QuattroParser::sendGraphics(int sheetId, Vec2i const &cell) const
{
	return m_graphParser->sendGraphics(sheetId, cell);
}


//////////////////////////////////////////////////////////////////////
// interface with QuattroSpreadsheet
//////////////////////////////////////////////////////////////////////

bool QuattroParser::getColor(int id, WPSColor &color) const
{
	return m_state->getColor(id, color);
}

bool QuattroParser::getFont(int id, WPSFont &font, libwps_tools_win::Font::Type &type) const
{
	if (id < 0 || id>=(int)m_state->m_fontsList.size())
	{
		WPS_DEBUG_MSG(("QuattroParser::getFont: can not find font %d\n", id));
		return false;
	}
	auto const &ft=m_state->m_fontsList[size_t(id)];
	font=ft;
	type=ft.m_type;
	return true;
}

// main function to parse the document
void QuattroParser::parse(librevenge::RVNGSpreadsheetInterface *documentInterface)
{
	RVNGInputStreamPtr input=getInput();
	if (!input)
	{
		WPS_DEBUG_MSG(("QuattroParser::parse: does not find main ole\n"));
		throw (libwps::ParseException());
	}

	if (!checkHeader(nullptr)) throw(libwps::ParseException());

	bool ok=false;
	try
	{
		ascii().setStream(input);
		ascii().open("MN0");
		if (checkHeader(nullptr) && readZones())
			m_listener=createListener(documentInterface);
		if (m_listener)
		{
			//m_chartParser->setListener(m_listener);
			m_graphParser->setListener(m_listener);
			m_spreadsheetParser->setListener(m_listener);

			m_graphParser->updateState();
			m_spreadsheetParser->updateState();

			m_listener->startDocument();
			int numSheet=m_spreadsheetParser->getNumSpreadsheets();
			if (numSheet==0) ++numSheet;
			for (int i=0; i<numSheet; ++i)
				m_spreadsheetParser->sendSpreadsheet(i, m_graphParser->getGraphicCellsInSheet(i));
			m_listener->endDocument();
			m_listener.reset();
			ok = true;
		}
	}
	catch (...)
	{
		WPS_DEBUG_MSG(("QuattroParser::parse: exception catched when parsing MN0\n"));
		throw (libwps::ParseException());
	}

	ascii().reset();
	if (!ok)
		throw(libwps::ParseException());
}

std::shared_ptr<WKSContentListener> QuattroParser::createListener(librevenge::RVNGSpreadsheetInterface *interface)
{
	std::vector<WPSPageSpan> pageList;
	WPSPageSpan ps(m_state->m_pageSpan);
	int numSheet=m_spreadsheetParser->getNumSpreadsheets();
	if (numSheet<=0) numSheet=1;
	if (!m_state->m_headerString.empty())
	{
		WPSSubDocumentPtr subdoc(new QuattroParserInternal::SubDocument
		                         (getInput(), *this, true));
		ps.setHeaderFooter(WPSPageSpan::HEADER, WPSPageSpan::ALL, subdoc);
	}
	if (!m_state->m_footerString.empty())
	{
		WPSSubDocumentPtr subdoc(new QuattroParserInternal::SubDocument
		                         (getInput(), *this, false));
		ps.setHeaderFooter(WPSPageSpan::FOOTER, WPSPageSpan::ALL, subdoc);
	}
	ps.setPageSpan(numSheet);
	pageList.push_back(ps);
	return std::shared_ptr<WKSContentListener>(new WKSContentListener(pageList, interface));
}

////////////////////////////////////////////////////////////
// low level
////////////////////////////////////////////////////////////
// read the header
////////////////////////////////////////////////////////////
bool QuattroParser::checkHeader(WPSHeader *header, bool strict)
{
	m_state.reset(new QuattroParserInternal::State(m_state->m_fontType, m_state->m_password));
	std::shared_ptr<WPSStream> mainStream(new WPSStream(getInput(), ascii()));
	if (!checkHeader(mainStream, strict))
		return false;
	if (header)
	{
		header->setMajorVersion(m_state->m_version);
		header->setCreator(libwps::WPS_QUATTRO_PRO);
		header->setKind(libwps::WPS_SPREADSHEET);
		header->setIsEncrypted(m_state->m_isEncrypted);
	}
	return true;
}

bool QuattroParser::checkHeader(std::shared_ptr<WPSStream> stream, bool strict)
{
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;

	if (!stream->checkFilePosition(12))
	{
		WPS_DEBUG_MSG(("QuattroParser::checkHeader: file is too short\n"));
		return false;
	}

	input->seek(0,librevenge::RVNG_SEEK_SET);
	auto firstOffset = (int) libwps::readU8(input);
	auto type = (int) libwps::read8(input);
	f << "FileHeader:";
	if (firstOffset == 0 && type == 0)
		m_state->m_version=1000;
	else
	{
		WPS_DEBUG_MSG(("QuattroParser::checkHeader: find unexpected first data\n"));
		return false;
	}
	auto val=(int) libwps::read16(input);
	if (val==2)
	{
		// version
		val=(int) libwps::readU16(input);
		if (val==0x1001)
		{
			m_state->m_version=1001;
			f << "quattropro[wb1],";
		}
		else if (val==0x1002)
		{
			m_state->m_version=1002;
			f << "quattropro[wb2],";
		}
		else
		{
			WPS_DEBUG_MSG(("QuattroParser::checkHeader: find unknown file version\n"));
			return false;
		}
	}
	else
	{
		WPS_DEBUG_MSG(("QuattroParser::checkHeader: header contain unexpected size field data\n"));
		return false;
	}
	input->seek(0, librevenge::RVNG_SEEK_SET);
	if (strict)
	{
		for (int i=0; i < 6; ++i)
		{
			if (!readZone(stream)) return false;
			if (m_state->m_isEncrypted) break;
		}
	}
	ascFile.addPos(0);
	ascFile.addNote(f.str().c_str());

	return true;
}

bool QuattroParser::readZones()
{
	m_graphParser->cleanState();
	m_spreadsheetParser->cleanState();

	std::shared_ptr<WPSStream> stream(new WPSStream(getInput(), ascii()));
	RVNGInputStreamPtr &input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	input->seek(0, librevenge::RVNG_SEEK_SET);
	while (true)
	{
		if (!stream->checkFilePosition(input->tell()+4))
			break;
		if (!readZone(stream))
			break;
		if (m_state->m_isEncrypted && !m_state->m_isDecoded)
			throw(libwps::PasswordException());
	}

	//
	// look for ending
	//
	long pos = input->tell();
	if (!stream->checkFilePosition(pos+4))
	{
		WPS_DEBUG_MSG(("QuattroParser::readZones: cell header is too short\n"));
		return m_spreadsheetParser->getNumSpreadsheets()>0;
	}
	auto type = (int) libwps::readU16(input); // 1
	auto length = (int) libwps::readU16(input);
	if (length)
	{
		WPS_DEBUG_MSG(("QuattroParser::readZones: parse breaks before ending\n"));
		ascFile.addPos(pos);
		ascFile.addNote("Entries(BAD):###");
		return m_spreadsheetParser->getNumSpreadsheets()>0;
	}

	ascFile.addPos(pos);
	if (type != 1)
	{
		WPS_DEBUG_MSG(("QuattroParser::readZones: odd end cell type: %d\n", type));
		ascFile.addNote("Entries(BAD):###");
		return m_spreadsheetParser->getNumSpreadsheets();
	}
	ascFile.addNote("Entries(EndSpreadsheet)");

	if (version()==1002)
		readOLEZones(stream);
	return m_spreadsheetParser->getNumSpreadsheets();
}

bool QuattroParser::readOLEZones(std::shared_ptr<WPSStream> &stream)
{
	if (!stream)
		return false;
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;
	f << "Entries(OLEData)[header]:";
	long pos = input->tell();
	long endPos = stream->m_eof;
	if (!stream->checkFilePosition(pos+18))
	{
		WPS_DEBUG_MSG(("QuattroParser::readOLEZones: the zone seems to short\n"));
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return false;
	}
	for (int i=0; i<4; ++i)   // 0
	{
		auto val=int(libwps::read16(input));
		if (val) f << "f" << i << "=" << val << ",";
	}
	auto sSz=long(libwps::readU32(input));
	librevenge::RVNGString text; // QPW$ExtendedStorage$6.0
	if (sSz<=0 || sSz > endPos-input->tell()-6 || !readCString(stream,text,sSz))
	{
		WPS_DEBUG_MSG(("QuattroParser::readOLEZones: can not read header's type\n"));
		f << "##sSz,";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return true;
	}
	f << "type=" << text.cstr() << ",";
	ascFile.addPos(pos);
	ascFile.addNote(f.str().c_str());
	while (input->tell()+6<=endPos)
	{
		pos=input->tell();
		f.str("");
		f << "OLEData:";
		auto type=int(libwps::read16(input));
		sSz=long(libwps::readU32(input));
		if (sSz < 0 || sSz > endPos-pos-6 || type<1 || type>2 || (sSz==0 && type==2))
		{
			input->seek(pos, librevenge::RVNG_SEEK_SET);
			break;
		}
		if (type==1)
		{
			if (sSz)
				f << "###sz=" << sSz << ",";
			f << "end,";
			ascFile.addPos(pos);
			ascFile.addNote(f.str().c_str());
			return true;
		}
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		unsigned long numRead;
		const unsigned char *data=input->read(static_cast<unsigned long>(sSz), numRead);
		if (data && long(numRead)==sSz)
		{
			auto ole=libwps_OLE::getOLEInputStream(std::make_shared<WPSStringStream>(data, unsigned(numRead)));
			if (!ole)
			{
				WPS_DEBUG_MSG(("QuattroParser::readOLEZones::readOLE: oops, can not decode the ole\n"));
			}
			else
			{
				/* normally /_Date_XXX/ Where Date~1507005964 and XXX is an hexadecimal data.
				   Moreever, a file can be saved many times, so must read /_Date_XXX/ to retrieve the correspondance
				 */
				std::map<std::string,size_t> dirToIdMap;
				WPSOLEParser oleParser("", [&dirToIdMap](std::string const &dir)
				{
					if (dirToIdMap.find(dir)==dirToIdMap.end())
						dirToIdMap[dir]=dirToIdMap.size();
					return int(dirToIdMap.find(dir)->second);
				});
				oleParser.parse(ole);
				ascFile.skipZone(pos+6, pos+6+sSz-1);

				auto objectMap=oleParser.getObjectsMap();
				std::map<librevenge::RVNGString,WPSEmbeddedObject> nameToObjectsMap;
				for (auto it : dirToIdMap)
				{
					for (int wh=0; wh<2; ++wh)
					{
						std::string name=it.first+"/"+(wh==0 ? "LinkInfo" : "BOlePart");
						RVNGInputStreamPtr cOle(ole->getSubStreamByName(name.c_str()));
						if (!cOle)
						{
							WPS_DEBUG_MSG(("QuattroParser::readOLEZones::readOLE: oops, can not find link info for dir %s\n", name.c_str()));
							continue;
						}
						libwps::DebugFile asciiFile(cOle);
						asciiFile.open(libwps::Debug::flattenFileName(name));
						if (wh==1)
							readOleBOlePart(std::make_shared<WPSStream>(cOle,asciiFile));
						else
						{
							librevenge::RVNGString linkName;
							if (readOleLinkInfo(std::make_shared<WPSStream>(cOle,asciiFile),linkName) && !linkName.empty())
							{
								if (objectMap.find(int(it.second))==objectMap.end())
								{
									WPS_DEBUG_MSG(("QuattroParser::readOLEZones::readOLE: oops, can not find embedded data for %s\n", name.c_str()));
									continue;
								}
								nameToObjectsMap[linkName]=objectMap.find(int(it.second))->second;
							}
						}
					}
				}
				if (!nameToObjectsMap.empty())
					m_graphParser->storeObjects(nameToObjectsMap);
			}
		}
		else
		{
			WPS_DEBUG_MSG(("QuattroParser::readOLEZones::readOLE: I can not find the data\n"));
			input->seek(pos, librevenge::RVNG_SEEK_SET);
			break;
		}
		input->seek(pos+6+sSz, librevenge::RVNG_SEEK_SET);
	}
	if (input->tell()<endPos)
	{
		WPS_DEBUG_MSG(("QuattroParser::readOLEZones: find extra data\n"));
		ascFile.addPos(input->tell());
		ascFile.addNote("OLEData:###extra");
	}
	return true;
}

bool QuattroParser::readZone(std::shared_ptr<WPSStream> &stream)
{
	if (!stream)
		return false;
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;
	long pos = input->tell();
	auto id = (int) libwps::readU16(input);
	auto sz = (long) libwps::readU16(input);
	if (sz<0 || !stream->checkFilePosition(pos+4+sz))
	{
		WPS_DEBUG_MSG(("QuattroParser::readZone: size is bad\n"));
		input->seek(pos, librevenge::RVNG_SEEK_SET);
		return false;
	}

	if (id&0x8000)
	{
		// very rare, unsure what this means ; is it possible to have
		// other flags here ?
		WPS_DEBUG_MSG(("QuattroParser::readZone: find type[8] flags\n"));
		ascFile.addPos(pos);
		ascFile.addNote("#flag8000,");
		id &= 0x7fff;
	}

	if (id>=0x800) // I never found anything biggest than 47d, so must be ok
	{
		input->seek(pos, librevenge::RVNG_SEEK_SET);
		return false;
	}

	if (sz>=0xFF00 && stream->checkFilePosition(pos+4+sz+4))
	{
		input->seek(pos+4+sz, librevenge::RVNG_SEEK_SET);
		if (libwps::readU16(input)==0x10f)
		{
			// incomplete block, we must rebuild it
			input->seek(pos, librevenge::RVNG_SEEK_SET);
			unsigned long numRead;
			const unsigned char *data=input->read(4+static_cast<unsigned long>(sz), numRead);
			if (data && long(numRead)==4+sz)
			{
				ascFile.skipZone(pos,pos+4+sz-1);
				auto newInput=std::make_shared<WPSStringStream>(data, unsigned(numRead));
				bool ok=true;
				while (true)
				{
					long actPos=input->tell();
					if (!stream->checkFilePosition(actPos+4) || libwps::readU16(input)!=0x10f)
					{
						input->seek(actPos, librevenge::RVNG_SEEK_SET);
						break;
					}
					auto extraSize=long(libwps::readU16(input));
					if (!stream->checkFilePosition(actPos+4+extraSize))
					{
						ok=false;
						break;
					}
					ascFile.addPos(actPos);
					ascFile.addNote("Entries(ExtraData):");
					if (!extraSize)
						break;
					data=input->read(static_cast<unsigned long>(extraSize), numRead);
					if (!data || long(numRead)!=extraSize)
					{
						ok=false;
						break;
					}
					newInput->append(data, unsigned(numRead));
					ascFile.skipZone(actPos+4,actPos+4+extraSize-1);
				}
				if (ok)
				{
					std::stringstream s;
					static int complexDataNum=0;
					s << "Data" << ++complexDataNum;
					auto newStream=std::make_shared<WPSStream>(newInput);
					newStream->m_ascii.open(s.str());
					newStream->m_ascii.setStream(newInput);
					readZone(newStream);
					return true;
				}
			}
			WPS_DEBUG_MSG(("QuattroParser::readZone: can not reconstruct a zone\n"));
			ascFile.addPos(pos);
			ascFile.addNote("Entries(###Bad):");
			input->seek(pos+4+sz, librevenge::RVNG_SEEK_SET);
			return true;
		}
	}
	f << "Entries(Zone" << std::hex << id << std::dec << "A):";
	bool ok = true, isParsed = false, needWriteInAscii = false;
	int val;
	input->seek(pos, librevenge::RVNG_SEEK_SET);
	switch (id)
	{
	case 0:
		if (sz!=2) break;
		f.str("");
		f << "version=" << std::hex << libwps::readU16(input) << std::dec << ",";
		isParsed=needWriteInAscii=true;
		break;
	case 0x1: // EOF
		ok = false;
		break;
	// boolean
	case 0x2: // Calculation mode 0 or FF
	case 0x3: // Calculation order
	case 0x4: // Split window type
	case 0x5: // Split window syn: never seen
	case 0x24: // protected if FF
	case 0x27: // print setup ?
	case 0x29: // label format 22|27|5e (spreadsheet): never seen
	case 0x2f: // always 1
	case 0x30: // print page break 0|ff (spreadsheet)
	case 0x31: // cursor/location 1|2: never seen
	case 0x38: // lock: never seen
	case 0x99: // with 0
	case 0xf4:
	case 0xf7:
	case 0xfa:
	case 0x101: // with 0
	case 0x102: // with 0
	case 0x109: // with 0
	case 0x10a: // with 0
	case 0x111: // with 0|3|4
	case 0x132: // with 0
	case 0x133: // with 0
	case 0x137: // with 0|1
		f.str("");
		switch (id)
		{
		case 0x30:
			f << "Entries(PageSetup)[nopgbreak]:";
			break;
		case 0xf4: // sometimes with size 5
			f << "Entries(PageSetup)[fittopage]:";
			break;
		case 0xf7: // USEME
			f << "Entries(PageSetup)[portrait]:";
			break;
		case 0xfa:
			f << "Entries(PageSetup)[centerblock]:";
			break;
		default:
			f << "Entries(Byte" << std::hex << id << std::dec << "Z):";
			break;
		}
		if (sz!=1)
		{
			f << "##";
			break;
		}
		input->seek(pos+4, librevenge::RVNG_SEEK_SET);
		val=(int) libwps::readU8(input);
		if (id==0x29)
			f << "val=" << std::hex << val << std::dec << ",";
		else if (id==0x31)
		{
			if (val!=1) f << val << ",";
		}
		else
		{
			if (val==1) f << "true,";
			else if (val) f << "#val=" << val << ",";
		}
		isParsed=needWriteInAscii=true;
		break;
	case 0x4c: // with 0 or 3=encrypted file?
	case 0x4d: // with 0
	case 0xc9: // with [026]{00,01,c8}
	case 0xf5:
	case 0xf6: // seems related to paper format(checkme)
	case 0x12d: // with 1
	case 0x136: // with 0
	case 0x26a: // with 1
		f.str("");
		switch (id)
		{
		case 0xf5:
			f << "Entries(PageSetup)[scale]:";
			break;
		case 0x26a:
			f << "Entries(SlideShow)[6a]:";
			break;
		default:
			f << "Entries(Int" << std::hex << id << std::dec << "Z):";
			break;
		}
		if (sz!=2)
		{
			f << "##";
			break;
		}
		input->seek(pos+4, librevenge::RVNG_SEEK_SET);
		val=(int) libwps::readU16(input);
		if (val) f << "f0=" << std::hex << val << std::dec << ",";
		break;
	case 0x12e: // with 1|d
	case 0x260: // with 5,1
		f.str("");
		switch (id)
		{
		case 0x260:
			f << "Entries(SlideShow)[60]:";
			break;
		default:
			f << "Entries(Long" << std::hex << id << std::dec << "Z):";
			break;
		}
		if (sz!=4)
		{
			f << "##";
			break;
		}
		input->seek(pos+4, librevenge::RVNG_SEEK_SET);
		for (int i=0; i<2; ++i)
		{
			val=(int) libwps::readU16(input);
			if (val) f << "f" << i << "=" << std::hex << val << std::dec << ",";
		}
		break;
	case 0x6: // active worksheet range
		ok = m_spreadsheetParser->readSheetSize(stream);
		isParsed = true;
		break;
	case 0xb: // named range
		readFieldName(stream);
		isParsed=true;
		break;
	case 0xc: // blank cell
	case 0xd: // integer cell
	case 0xe: // floating cell
	case 0xf: // label cell
	case 0x10: // formula cell
	case 0x33:  // value of string formula
		// case 10e:  seems relative to cell with formula: list of dependency? can have some text
		ok = m_spreadsheetParser->readCell(stream);
		isParsed=true;
		break;

	case 0x18: // data table range
	case 0x19: // query range
	case 0x1b: // sort range
	case 0x1c: // fill range
	case 0x1d: // primary sort key range
	case 0x20: // distribution range
	case 0x23: // secondary sort key range
	case 0x66:
	case 0x67:
	case 0x69:
	case 0x9f:
	case 0xa0:
	case 0xa1:
	case 0xb7:
		readRangeList(stream);
		isParsed=true;
		break;
	// 0x1a: range or ???

	// checkme this nodes appear two times, even/odd page ?
	case 0x25: // footer
	case 0x26: // header
		readHeaderFooter(stream, id==0x26);
		isParsed = true;
		break;
	case 0x28: // print margin: one by spreadsheet
		if (sz!=12) break;
		input->seek(pos+4, librevenge::RVNG_SEEK_SET);
		f.str("");
		f << "Entries(PrintMargin):";
		f << "margs=[";
		for (int i=0; i<4; ++i)   // LRTB
			f << float(libwps::read16(input))/20.f << ",";
		f << "],";
		f << "hf[height]=[";
		for (int i=0; i<2; ++i)   // header, footer height
			f << float(libwps::read16(input))/20.f << ",";
		f << "],";
		isParsed=needWriteInAscii=true;
		break;

	case 0x4b:
		f.str("");
		f << "Entries(Password):";
		m_state->m_isEncrypted=true;
		if (sz==20)
		{
			m_state->m_isEncrypted=true;
			input->seek(pos+4, librevenge::RVNG_SEEK_SET);
			f.str("");
			uint16_t fileKey(libwps::readU16(input));
			f << "Entries(Password):pass=" << std::hex << fileKey << std::dec << ",";
			f << "len=" << int(libwps::readU16(input)) << ",";
			isParsed = needWriteInAscii = true;
			std::vector<uint8_t> keys;
			keys.resize(16);
			for (auto &k : keys) k=uint8_t(libwps::readU8(input));
			// to check users password:
			//   call libwps::encodeLotusPassword(m_state->m_password, key, lotusKeys, someDefValues);
			//   and check if  int16_t(key<<8|key>>8)==fileKey
			if (!m_state->m_isDecoded)
			{
				auto newInput=decodeStream(input, keys);
				if (newInput)
				{
					// let's replace the current input by the decoded input
					m_state->m_isDecoded=true;
					stream->m_input=newInput;
					stream->m_ascii.setStream(newInput);
				}
			}
		}
		if (!m_state->m_isDecoded)
		{
			WPS_DEBUG_MSG(("QuattroParser::parse: can not decode the file\n"));
		}
		break;
	case 0x96:
		readCellPosition(stream);
		isParsed=true;
		break;
	case 0x97: // name use to refer to some external spreadsheet
		readFileName(stream);
		isParsed=true;
		break;

	case 0xca:
	case 0xcb:
		isParsed=m_spreadsheetParser->readBeginEndSheet(stream, m_state->m_actualSheet);
		break;
	case 0xcc:
		isParsed=m_spreadsheetParser->readSheetName(stream);
		break;
	case 0xce:
		ok = m_spreadsheetParser->readCellStyle(stream);
		isParsed = true;
		break;
	case 0xcf:
		// also 0xfc?
		isParsed=readFontDef(stream);
		break;
	case 0xd0:
		isParsed=readStyleName(stream);
		break;
	case 0xd6:
		isParsed = m_spreadsheetParser->readRowSize(stream);
		break;
	case 0xd8:
		isParsed = m_spreadsheetParser->readColumnSize(stream);
		break;
	// case 0xdc: hidden row
	// 110: seems to contain also some font name
	case 0xe3:
	case 0xe4: // find with 4 QFinance
	case 0xe5: // find with 29 WKDAY
	case 0xe6:   // alway with 4 and TMonth' 'D', 'YYYY
	{
		f.str("");
		if (id==0xe3)
			f << "Entries(GroupName):";
		else if (id==0xe6)
			f << "Entries(UserFormat):";
		else
			f << "Entries(DLLIdToFunc)[F" << id-0xe3 << "]:";
		if (sz<3) break;
		input->seek(pos+4, librevenge::RVNG_SEEK_SET);
		val=int(libwps::readU16(input));
		if (id==0xe3)
			f << "sheet" << (val&0xff) << "=>" << (val>>8) << ",";
		else
			f << "id=" << val << ",";
		librevenge::RVNGString text;
		if (!readCString(stream, text,sz-2))
			f << "###";
		else
		{
			if (id==0xe4 || id==0xe5)
				m_spreadsheetParser->addDLLIdName(val, text, id==0xe4);
			else
				m_spreadsheetParser->addUserFormat(val, text);
			f << text.cstr() << ",";
		}
		break;
	}
	case 0xe8:
		isParsed=readColorList(stream);
		break;
	case 0xfc:
		isParsed=readPageSetup(stream);
		break;
	case 0x197: // one by sheet
		isParsed=m_spreadsheetParser->readZone197(stream);
		break;
	// case 0xfe: some counter
	case 0x259:
	case 0x25e: // followed by 260 and 26a
	{
		f.str("");
		if (id==0x259)
			f << "Entries(GraphName):";
		else
			f << "Entries(SlideShow):";
		librevenge::RVNGString text;
		input->seek(pos+4, librevenge::RVNG_SEEK_SET);
		if (!readCString(stream,text,sz))
			f << "###";
		else
			f << text.cstr() << ",";
		break;
	}
	case 0x25a:
	case 0x25b:
	case 0x25c:
	case 0x25f:
	case 0x31f:
		f.str("");
		switch (id)
		{
		case 0x25a:
			f << "Entries(GraphName)[end]:";
			break;
		case 0x25b: // end/begin of ?
			f << "Entries(ZoneBBegEnd)[begin]:";
			break;
		case 0x25c:
			f << "Entries(ZoneBBegEnd)[end]:";
			break;
		case 0x25f:
			f << "Entries(SlideShow)[end]:";
			break;
		case 0x31f:
			f << "Entries(Zone341)[end]:";
			break;
		default:
			f << "Entries(Null" << std::hex << id << std::dec << "):";
			break;
		}
		if (sz!=0)
			f << "###";
		break;
	case 0x321:
	case 0x322:
		isParsed=m_graphParser->readBeginEnd(stream, m_state->m_actualSheet);
		break;
	// case 0x324: a list of XX:20?: seems relative to graph/chart
	case 0x341: // maybe chart ?
		isParsed = readZone341(stream);
		break;
	case 0x335:
	case 0x337:
	case 0x33f:
	case 0x342:
	case 0x343:
	case 0x345:
	case 0x349:
	case 0x34a:
	case 0x34e:
	case 0x34f:
	case 0x351:
	case 0x35d:
	case 0x36d:
		isParsed = m_graphParser->readDialogUnknown(stream);
		break;
	case 0x35e:
		isParsed = m_graphParser->readDialog(stream);
		break;
	case 0x381: // frame wb2
		isParsed = m_graphParser->readFrameOLE(stream);
		break;
	case 0x382: // only wb2?
		isParsed = m_graphParser->readImage(stream);
		break;
	case 0x383: // only wb2?
		isParsed = m_graphParser->readBitmap(stream);
		break;
	case 0x384:
		isParsed = m_graphParser->readChart(stream);
		break;
	case 0x385:
		isParsed = m_graphParser->readFrame(stream);
		break;
	case 0x386:
		isParsed = m_graphParser->readButton(stream);
		break;
	case 0x38b:
		isParsed = m_graphParser->readOLEData(stream);
		break;
	default:
		break;
	}

	if (!ok)
	{
		input->seek(pos, librevenge::RVNG_SEEK_SET);
		return false;
	}
	if (isParsed)
	{
		if (needWriteInAscii)
		{
			ascFile.addPos(pos);
			ascFile.addNote(f.str().c_str());
		}
		input->seek(pos+4+sz, librevenge::RVNG_SEEK_SET);
		return true;
	}

	if (sz && input->tell()!=pos && input->tell()!=pos+4+sz)
		ascFile.addDelimiter(input->tell(),'|');
	input->seek(pos+4+sz, librevenge::RVNG_SEEK_SET);
	ascFile.addPos(pos);
	ascFile.addNote(f.str().c_str());
	return true;
}

////////////////////////////////////////////////////////////
//   generic
////////////////////////////////////////////////////////////

bool QuattroParser::readCString(std::shared_ptr<WPSStream> stream, librevenge::RVNGString &string, long maxSize)
{
	RVNGInputStreamPtr input = stream->m_input;
	long pos = input->tell();
	string.clear();
	if (!stream->checkFilePosition(pos+maxSize))
	{
		WPS_DEBUG_MSG(("QuattroParser::readCString: string's size seems bad\n"));
		return false;
	}
	auto const fontType = getDefaultFontType();
	for (long i=0; i<maxSize; ++i)
	{
		auto c = (unsigned char) libwps::readU8(input);
		if (c == '\0') break;
		libwps::appendUnicode
		((uint32_t)libwps_tools_win::Font::unicode((unsigned char)c,fontType), string);
	}
	return true;
}

bool QuattroParser::readFieldName(std::shared_ptr<WPSStream> stream)
{
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;

	long pos = input->tell();
	auto type = (long)(libwps::readU16(input)&0x7fff);
	if (type != 0xb)
	{
		WPS_DEBUG_MSG(("QuattroParser::readFieldName: not a zoneB type\n"));
		return false;
	}
	auto sz = (long) libwps::readU16(input);
	long endPos=pos+4+sz;
	f << "Entries(FldNames):";
	if (sz < 10)
	{
		WPS_DEBUG_MSG(("QuattroParser::readFieldName: size seems bad\n"));
		f << "###";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return true;
	}
	auto id=int(libwps::readU16(input));
	f << "id=" << id << ",";
	auto val=int(libwps::readU8(input)); // always 1?
	if (val!=1) f << "f0=" << val << ",";
	librevenge::RVNGString name;
	auto sSz=int(libwps::readU8(input));
	if (10+sSz>sz || !readCString(stream,name,sSz))
	{
		f << "##sSz,";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return true;
	}
	else if (!name.empty())
		f << name.cstr() << ',';
	val=int(libwps::readU8(input)); // 0-4: 0 maybe bad
	if (val) f << "f1=" << val << ",";
	auto fl=int(libwps::readU8(input));
	int N=(fl&0x10) ? 2 : 1;
	if (fl&0xef) f << "fl=" << std::hex << (fl&0xef) << std::dec << ",";
	if (input->tell()+4*N!=endPos)
	{
		WPS_DEBUG_MSG(("QuattroParser::readFieldName: can not find the cell(s)\n"));
		f << "###";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return true;
	}
	WKSContentListener::FormulaInstruction instr;
	instr.m_type = N==1 ? instr.F_Cell : instr.F_CellList;
	f << "cell=";
	for (int i=0; i<N; ++i)
	{
		auto col=int(libwps::readU8(input));
		auto sheet=int(libwps::readU8(input)); // checkme
		auto row=int(libwps::readU16(input));
		instr.m_positionRelative[i]=Vec2b(false,false);
		instr.m_position[i]=Vec2i(col,row);
		instr.m_sheetId[i]=sheet;
		f << col << "x" << row;
		if (sheet) f << "[" << sheet << "]";
		if (i+1==N)
			f << ",";
		else
			f << ":";
	}
	if (m_state->m_idToFieldMap.find(id)!=m_state->m_idToFieldMap.end())
	{
		WPS_DEBUG_MSG(("QuattroParser::readFieldName: oops a field with id=%d already exists\n", id));
	}
	else
		m_state->m_idToFieldMap[id]=std::pair<librevenge::RVNGString,WKSContentListener::FormulaInstruction>(name, instr);
	ascFile.addPos(pos);
	ascFile.addNote(f.str().c_str());
	return true;
}

bool QuattroParser::readFileName(std::shared_ptr<WPSStream> stream)
{
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;
	long pos = input->tell();
	auto type = (int)(libwps::readU16(input)&0x7fff);

	if (type != 0x97)
	{
		WPS_DEBUG_MSG(("QuattroParser::readFileName: not a font zone\n"));
		return false;
	}
	auto sz = (long)libwps::readU16(input);
	f << "Entries(FileName):";
	if (sz<3)
	{
		WPS_DEBUG_MSG(("QuattroParser::readFileName: seems very short\n"));
		f << "###";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return true;
	}
	auto id=(int)libwps::readU16(input);
	f << "id=" << id << ",";
	librevenge::RVNGString name;
	if (!readCString(stream,name,sz-2))
	{
		f << "##name,";
	}
	else if (m_state->m_idToFileNameMap.find(id)!=m_state->m_idToFileNameMap.end())
	{
		WPS_DEBUG_MSG(("QuattroParser::readFileName: id=%d already found\n", id));
		f << "##duplicated,";
	}
	else if (!name.empty())
		m_state->m_idToFileNameMap[id]=name;
	if (!name.empty())
		f << name.cstr() << ',';
	if (input->tell()!=pos+4+sz) ascFile.addDelimiter(input->tell(),'|');
	ascFile.addPos(pos);
	ascFile.addNote(f.str().c_str());
	return true;
}

bool QuattroParser::readFontDef(std::shared_ptr<WPSStream> stream)
{
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;
	long pos = input->tell();
	auto type = int(libwps::readU16(input)&0x7fff);

	if (type != 0xcf)
	{
		WPS_DEBUG_MSG(("QuattroParser::readFontDef: not a font zone\n"));
		return false;
	}
	auto sz = (long)libwps::readU16(input);
	f << "Entries(FontDef)[F" << m_state->m_fontsList.size() << "]:";
	QuattroParserInternal::Font font(getDefaultFontType());
	if (sz!=0x24)
	{
		WPS_DEBUG_MSG(("QuattroParser::readFontDef: seems very bad\n"));
		f << "###";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		m_state->m_fontsList.push_back(font);
		return true;
	}
	auto fSize = (int) libwps::readU16(input);
	if (fSize >= 1 && fSize <= 50)
		font.m_size=double(fSize);
	else
		f << "###fSize=" << fSize << ",";
	auto flags = (int)libwps::readU16(input);
	uint32_t attributes = 0;
	if (flags & 1) attributes |= WPS_BOLD_BIT;
	if (flags & 2) attributes |= WPS_ITALICS_BIT;
	if (flags & 4) attributes |= WPS_UNDERLINE_BIT;
	if (flags & 0x20) attributes |= WPS_STRIKEOUT_BIT;

	font.m_attributes=attributes;
	flags &= 0xffd8;
	if (flags)
		f << "##fl=" << std::hex << flags << std::dec << ",";
	librevenge::RVNGString name;
	if (!readCString(stream,name,22))
	{
		f << "##name,";
	}
	else
		font.m_name=name;
	input->seek(pos+30, librevenge::RVNG_SEEK_SET);
	auto val= (int) libwps::readU16(input); // 1|5a: color
	if (val) f << "color?=" << val << ",";
	for (int i=0; i<2; ++i)   // 33, 1
	{
		val=(int) libwps::readU8(input);
		if (val) f << "fl" << i << "=" << val << ",";
	}
	ascFile.addDelimiter(input->tell(),'|'); // junk?
	m_state->m_fontsList.push_back(font);

	f << font;
	ascFile.addPos(pos);
	ascFile.addNote(f.str().c_str());
	return true;
}

bool QuattroParser::readColorList(std::shared_ptr<WPSStream> stream)
{
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;
	long pos = input->tell();
	auto type = int(libwps::readU16(input)&0x7fff);

	if (type != 0xe8)
	{
		WPS_DEBUG_MSG(("QuattroParser::readColorList: not a font zone\n"));
		return false;
	}
	auto sz = (long)libwps::readU16(input);
	f << "Entries(ColorList):";
	if (sz<0x40 || (sz%4))
	{
		WPS_DEBUG_MSG(("QuattroParser::readColorList: seems very bad\n"));
		f << "###";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return true;
	}
	auto N=size_t(sz/4);
	m_state->m_colorsList.resize(N);
	for (auto &c: m_state->m_colorsList)
	{
		uint8_t cols[4];
		for (auto &co : cols) co=uint8_t(libwps::readU8(input));
		c=WPSColor(cols[0],cols[1],cols[2],cols[3]);
		f << c << ",";
	}
	ascFile.addPos(pos);
	ascFile.addNote(f.str().c_str());
	return true;
}

bool QuattroParser::readStyleName(std::shared_ptr<WPSStream> stream)
{
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;
	long pos = input->tell();
	auto type = int(libwps::readU16(input));
	f << "Entries(StyleName):";
	if (type&0x8000)
	{
		f << "#type[8],";
		type &= 0x7fff;
	}

	if (type != 0xd0)
	{
		WPS_DEBUG_MSG(("QuattroParser::readStyleName: not a font zone\n"));
		return false;
	}
	auto sz = (long)libwps::readU16(input);
	if (sz<4)
	{
		WPS_DEBUG_MSG(("QuattroParser::readStyleName: seems very bad\n"));
		f << "###";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return true;
	}
	auto id=(int) libwps::readU16(input);
	f << "St" << id << ",";
	auto val=(int) libwps::readU16(input);
	if (val!=id) f << "f0=" << val << ",";
	if (sz!=4)   // no name seems ok
	{
		librevenge::RVNGString name;
		if (!readCString(stream,name,sz-4))
		{
			f << "##name,";
		}
		else if (!name.empty())
			f << name.cstr() << ",";
		if (input->tell()!=pos+4+sz)
			ascFile.addDelimiter(input->tell(),'|');
	}
	ascFile.addPos(pos);
	ascFile.addNote(f.str().c_str());
	return true;
}

// ----------------------------------------------------------------------
// Header/Footer
// ----------------------------------------------------------------------
void QuattroParser::sendHeaderFooter(bool header)
{
	if (!m_listener)
	{
		WPS_DEBUG_MSG(("QuattroParser::sendHeaderFooter: can not find the listener\n"));
		return;
	}

	m_listener->setFont(m_state->getDefaultFont());
	auto const &text = header ? m_state->m_headerString : m_state->m_footerString;
	m_listener->insertUnicodeString(text);
}

bool QuattroParser::readHeaderFooter(std::shared_ptr<WPSStream> stream, bool header)
{
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;
	long pos = input->tell();
	auto type = int(libwps::readU16(input)&0x7fff);
	if (type != 0x0026 && type != 0x0025)
	{
		WPS_DEBUG_MSG(("QuattroParser::readHeaderFooter: not a header/footer\n"));
		return false;
	}
	auto sz = (long)libwps::readU16(input);
	long endPos = pos+4+sz;

	f << "Entries(" << (header ? "HeaderText" : "FooterText") << "):";
	librevenge::RVNGString text;
	if (!readCString(stream,text,sz))
	{
		f << "##sSz,";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return true;
	}
	if (!text.empty())
	{
		if (header)
			m_state->m_headerString=text;
		else
			m_state->m_footerString=text;
	}
	if (!text.empty())
		f << text.cstr();
	if (input->tell()!=endPos)
		ascFile.addDelimiter(input->tell(), '|');
	ascFile.addPos(pos);
	ascFile.addNote(f.str().c_str());

	return true;
}

////////////////////////////////////////////////////////////
//   Unknown
////////////////////////////////////////////////////////////

/* some spreadsheet zones 0:18, 0:19, 0:20, 0:27, 0:2a */
bool QuattroParser::readRangeList(std::shared_ptr<WPSStream> stream)
{
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;

	long pos = input->tell();
	auto type = long(libwps::readU16(input)&0x7fff);
	int extraSize=0;
	int N=0;
	switch (type)
	{
	case 0x18:
	case 0x19:
		N=3;
		extraSize=2;
		break;
	case 0x1b:
		N=1;
		extraSize=4;
		break;
	case 0x1c:
		N=1;
		break;
	case 0x1d:
		N=1;
		extraSize=2;
		break;
	case 0x20:
		N=2;
		break;
	case 0x23:
		N=1;
		extraSize=2;
		break;
	case 0x66:
		N=2;
		break;
	case 0x67:
		N=3;
		extraSize=2;
		break;
	case 0x69:
		N=5;
		break;
	case 0x9f:
	case 0xa0:
	case 0xa1:
		N=1;
		extraSize=2;
		break;
	case 0xb7:
		N=2;
		extraSize=18;
		break;
	default:
		WPS_DEBUG_MSG(("QuattroParser::readRangeList: unexpected type ???\n"));
		return false;
	}
	auto sz = (long) libwps::readU16(input);

	f << "Entries(RangeList" << std::hex << type << std::dec << ")]:";
	if (sz != 10*N+extraSize)
	{
		WPS_DEBUG_MSG(("QuattroParser::readRangeList: the zone size seems too bad\n"));
		f << "###";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return true;
	}
	for (int i=0; i<N; ++i)
	{
		f << "cell" << i << "=[";
		auto val=int(libwps::readU8(input)); // 0-4: 0 maybe bad
		if (val) f << "f0=" << val << ",";
		auto fl=int(libwps::readU8(input));
		if (fl&0x10) f << "x2,";
		if (fl&0xef) f << "fl=" << std::hex << (fl&0xef) << std::dec << ",";
		for (int j=0; j<2; ++j)
		{
			auto col=int(libwps::readU8(input));
			auto sheet=int(libwps::readU8(input)); // checkme
			auto row=int(libwps::readU16(input));
			f << col << "x" << row;
			if (sheet) f << "[" << sheet << "]";
			if (j==1)
				f << ",";
			else
				f << ":";
		}
		f << "],";
	}
	for (int i=0; i<extraSize/2; ++i)
	{
		auto val=(int) libwps::read16(input);
		if (val) f << "f" << i << "=" << val << ",";
	}

	ascFile.addPos(pos);
	ascFile.addNote(f.str().c_str());

	return true;
}

bool QuattroParser::readCellPosition(std::shared_ptr<WPSStream> stream)
{
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;
	long pos = input->tell();
	auto type = int(libwps::readU16(input)&0x7fff);

	if (type != 0x96)
	{
		WPS_DEBUG_MSG(("QuattroParser::readCellPosition: not a cell position zone\n"));
		return false;
	}
	auto sz = (long)libwps::readU16(input);
	f << "Entries(CellPosition):";
	if (sz%6)
	{
		WPS_DEBUG_MSG(("QuattroParser::readCellPosition: size seems very bad\n"));
		f << "###";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return true;
	}
	auto N=int(sz/6);
	for (int i=0; i<N; ++i)
	{
		int cellPos[3]; // col, rowMin, rowMax
		for (auto &c : cellPos) c=(int) libwps::readU16(input);
		f <<  "C" << cellPos[0] << "[" << cellPos[1] << "->" << cellPos[2] << "],";
	}
	ascFile.addPos(pos);
	ascFile.addNote(f.str().c_str());
	return true;
}

bool QuattroParser::readPageSetup(std::shared_ptr<WPSStream> stream)
{
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;
	long pos = input->tell();
	auto type = int(libwps::readU16(input)&0x7fff);

	if (type != 0xfc)
	{
		WPS_DEBUG_MSG(("QuattroParser::readPageSetup: not a cell position zone\n"));
		return false;
	}
	auto sz = (long)libwps::readU16(input);
	f << "Entries(PageSetup):";
	if (sz!=0x24)
	{
		WPS_DEBUG_MSG(("QuattroParser::readPageSetup: size seems very bad\n"));
		f << "###";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return true;
	}
	// header/footer font(useme)
	f << "font=[";
	f << "sz=" << libwps::read16(input) << ",";
	auto val=int(libwps::readU16(input));
	if (val) f << "attr=" << std::hex << val << std::dec << ",";
	librevenge::RVNGString name;
	if (!readCString(stream, name,32))
	{
		f << "##string,";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return true;
	}
	if (!name.empty())
		f << "name=" << name.cstr() << ",";
	f << "],";
	input->seek(pos+4+4+32, librevenge::RVNG_SEEK_SET);
	ascFile.addPos(pos);
	ascFile.addNote(f.str().c_str());
	return true;
}

bool QuattroParser::readZone341(std::shared_ptr<WPSStream> stream)
{
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;
	long pos = input->tell();
	auto type = int(libwps::readU16(input)&0x7fff);
	if (type != 0x341)
	{
		WPS_DEBUG_MSG(("QuattroParser::readZone341: not a cell position zone\n"));
		return false;
	}
	auto sz = (long)libwps::readU16(input);
	long endPos=pos+4+sz;
	f << "Entries(Zone341):";
	if (sz<75)
	{
		WPS_DEBUG_MSG(("QuattroParser::readZone341: size seems very bad\n"));
		f << "###";
		ascFile.addPos(pos);
		ascFile.addNote(f.str().c_str());
		return true;
	}
	ascFile.addDelimiter(input->tell(),'|');
	input->seek(pos+4+75, librevenge::RVNG_SEEK_SET);
	ascFile.addPos(pos);
	ascFile.addNote(f.str().c_str());

	while (input->tell()+4<=endPos)
	{
		pos=input->tell();
		bool end=(libwps::readU16(input)&0x7fff)==0x31f;
		input->seek(pos,librevenge::RVNG_SEEK_SET);
		if (!readZone(stream) || input->tell()>endPos)
		{
			WPS_DEBUG_MSG(("QuattroParser::readZone341: find extra data\n"));
			ascFile.addPos(pos);
			ascFile.addNote("Zone341:###extra");
			return true;
		}
		if (end)
		{
			if (input->tell()<endPos)
			{
				ascFile.addPos(input->tell());
				ascFile.addNote("_");
			}
			return true;
		}
	}
	if (sz!=75)
	{
		WPS_DEBUG_MSG(("QuattroParser::readZone341: oops, does not find end zone\n"));
	}
	return true;
}

////////////////////////////////////////////////////////////
//   ole stream
////////////////////////////////////////////////////////////
bool QuattroParser::readOleLinkInfo(std::shared_ptr<WPSStream> stream, librevenge::RVNGString &link)
{
	if (!stream || !stream->checkFilePosition(4))
	{
		WPS_DEBUG_MSG(("QuattroParser::readLinkInfo: unexpected zone\n"));
		return false;
	}
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;
	f << "Entries(LinkInfo):";
	int val=libwps::readU8(input);
	if (val!=0x53)
		f << "f0=" << std::hex << val << std::dec << ",";
	val=libwps::readU16(input); // 2 or 3
	if (val) f << "f1=" << val << ",";
	if (!readCString(stream, link, stream->m_eof-3))
	{
		WPS_DEBUG_MSG(("QuattroParser::readLinkInfo: can not read the link\n"));
		f << "##link,";
		ascFile.addPos(0);
		ascFile.addNote(f.str().c_str());
		return false;
	}
	if (!link.empty())
		f << "link=" << link.cstr() << ",";
	ascFile.addPos(0);
	ascFile.addNote(f.str().c_str());
	return true;
}

bool QuattroParser::readOleBOlePart(std::shared_ptr<WPSStream> stream)
{
	if (!stream || !stream->checkFilePosition(20))
	{
		WPS_DEBUG_MSG(("QuattroParser::readOleBOlePart: unexpected zone\n"));
		return false;
	}
	RVNGInputStreamPtr input = stream->m_input;
	libwps::DebugFile &ascFile=stream->m_ascii;
	libwps::DebugStream f;
	f << "Entries(BOlePart):";
	for (int i=0; i<5; ++i)   // f0=1, f1=f2=small int(often 1), f3=f4=small int(often 1)
	{
		auto val=int(libwps::read32(input));
		if (val!=1) f << "f" << i << "=" << val << ",";
	}
	ascFile.addPos(0);
	ascFile.addNote(f.str().c_str());
	return true;
}

////////////////////////////////////////////////////////////
//   decode
////////////////////////////////////////////////////////////
RVNGInputStreamPtr QuattroParser::decodeStream(RVNGInputStreamPtr input, std::vector<uint8_t> const &key)
{
	if (!input || key.size()!=16)
	{
		WPS_DEBUG_MSG(("QuattroParser::decodeStream: the arguments seems bad\n"));
		return RVNGInputStreamPtr();
	}
	long actPos=input->tell();
	input->seek(0,librevenge::RVNG_SEEK_SET);
	librevenge::RVNGBinaryData data;
	if (!libwps::readDataToEnd(input, data) || !data.getDataBuffer())
	{
		WPS_DEBUG_MSG(("QuattroParser::decodeStream: can not read the original input\n"));
		return RVNGInputStreamPtr();
	}
	auto *buf=const_cast<unsigned char *>(data.getDataBuffer());
	auto endPos=long(data.size());
	input->seek(actPos,librevenge::RVNG_SEEK_SET);
	uint32_t d7=0;
	std::stack<long> stack;
	stack.push(endPos);
	while (!input->isEnd() && !stack.empty())
	{
		long pos=input->tell();
		if (pos+4>stack.top()) break;
		auto id=int(libwps::readU16(input)&0x7fff);
		auto sSz=int(libwps::readU16(input));
		if (pos+4+sSz>stack.top())
		{
			input->seek(pos,librevenge::RVNG_SEEK_SET);
			break;
		}
		if (id==0x341 && sSz>75)
		{
			// special case a container with header of size 75 + different subzones
			stack.push(pos+4+sSz);
			sSz=75; // transform only the header
		}
		for (int i=0; i<sSz; ++i)
		{
			auto c=uint8_t(libwps::readU8(input));
			c=(c^key[(d7++)&0xf]);
			buf[pos+4+i]=uint8_t((c>>5)|(c<<3));
		}
		// main zone ends with zone 1, zone 341 ends with zone 31f
		if (id==(stack.size()==1 ? 1 : 0x31f))
		{
			input->seek(stack.top(),librevenge::RVNG_SEEK_SET);
			stack.pop();
		}
	}
	if (input->tell()!=endPos)
	{
		WPS_DEBUG_MSG(("QuattroParser::decodeStream: can not decode the end of the file, data may be bad %lx %lx\n", static_cast<unsigned long>(input->tell()), static_cast<unsigned long>(endPos)));
	}
	RVNGInputStreamPtr res(new WPSStringStream(data.getDataBuffer(), static_cast<unsigned int>(endPos)));
	res->seek(actPos, librevenge::RVNG_SEEK_SET);
	return res;
}
/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
