#! /usr/bin/python3
import os
import sys
import generator_utils as gen

template = """// @{generatedby}@
/* ///////////////////////// The MPI Bugs Initiative ////////////////////////

  Origin: @{origin}@

  Description: @{shortdesc}@
    @{longdesc}@

   Version of MPI: Conforms to MPI 1.1, does not require MPI 2 implementation

BEGIN_MPI_FEATURES
  P2P!basic: @{p2pfeature}@
  P2P!nonblocking: @{ip2pfeature}@
  P2P!persistent: @{persfeature}@
  COLL!basic: Lacking
  COLL!nonblocking: Lacking
  COLL!persistent: Lacking
  COLL!tools: Yes
  RMA: Lacking
END_MPI_FEATURES

BEGIN_MBI_TESTS
  $ mpirun -np 2 ${EXE}
  | @{outcome}@
  | @{errormsg}@
END_MBI_TESTS
//////////////////////       End of MBI headers        /////////////////// */

#include <mpi.h>
#include <stdio.h>
#include <stdlib.h>


int main(int argc, char **argv) {
  int nprocs = -1;
  int rank = -1;
  int src=0, dest=1;
  int stag = 0, rtag = 0;
  int buff_size = 1;

  MPI_Init(&argc, &argv);
  MPI_Comm_size(MPI_COMM_WORLD, &nprocs);
  MPI_Comm_rank(MPI_COMM_WORLD, &rank);
  printf("Hello from rank %d \\n", rank);

  if (nprocs < 2)
    printf("MBI ERROR: This test needs at least 2 processes to produce a bug!\\n");

  MPI_Datatype type = MPI_INT;
  MPI_Comm newcom;
  MPI_Comm_split(MPI_COMM_WORLD, 0, nprocs - rank, &newcom);
  @{change_com}@
  @{change_srcdest}@

  @{init1}@
  @{init2}@
  if (rank == 0) {
    @{operation1}@ /* MBIERROR1 */
    @{start1}@
    @{fini1}@
  }else if (rank == 1) {
    @{operation2}@ /* MBIERROR2 */
    @{start2}@
    @{fini2}@
  }
  @{free1}@
  @{free2}@

  if(newcom != MPI_COMM_NULL && newcom != MPI_COMM_WORLD)
    MPI_Comm_free(&newcom);

  MPI_Finalize();
  printf("Rank %d finished normally\\n", rank);
  return 0;
}
"""


for p1 in gen.send + gen.isend + gen.psend:
    for p2 in gen.recv + gen.irecv + gen.precv:
        patterns = {}
        patterns = {'p1': p1, 'p2': p2}
        patterns['origin'] = "MBI"
        patterns['generatedby'] = f'DO NOT EDIT: this file was generated by {os.path.basename(sys.argv[0])}. DO NOT EDIT.'
        patterns['p2pfeature'] = 'Yes' if p1 in gen.send or p2 in gen.recv  else 'Lacking'
        patterns['ip2pfeature'] = 'Yes' if p1 in gen.isend or p2 in gen.irecv  else 'Lacking'
        patterns['persfeature'] = 'Yes' if p1 in gen.psend or p2 in gen.precv  else 'Lacking'
        patterns['p1'] = p1
        patterns['p2'] = p2
        patterns['init1'] = gen.init[p1]("1")
        patterns['init2'] = gen.init[p2]("2")
        patterns['start1'] = gen.start[p1]("1")
        patterns['start2'] = gen.start[p2]("2")
        patterns['fini1'] = gen.fini[p1]("1")
        patterns['fini2'] = gen.fini[p2]("2")
        patterns['operation1'] = gen.operation[p1]("1") #send
        patterns['operation2'] = gen.operation[p2]("2") #recv
        patterns['free1'] = gen.free[p1]("1")
        patterns['free2'] = gen.free[p2]("2")
        patterns['change_srcdest'] = ""
        patterns['change_com'] = ""

        # Generate the incorrect matching
        replace = patterns.copy()
        replace['shortdesc'] = 'Point to point @{p1}@ and @{p2}@ have a communicator mismatch'
        replace['longdesc'] = 'Process 1 uses newcom as the communicator while process 0 uses MPI_COMM_WORLD.'
        replace['outcome'] = 'ERROR: CommunicatorMatching'
        replace['errormsg'] = 'P2P Communicator mismatch. @{p1}@ at @{filename}@:@{line:MBIERROR1}@ and @{p2}@ at @{filename}@:@{line:MBIERROR2}@ have newcom or MPI_COMM_WORLD as a communicator.'
        replace['change_com'] = 'if (rank==0)\n    newcom = MPI_COMM_WORLD; /* MBIERROR */'
        gen.make_file(template, f'ParamMatching_Com_{p1}_{p2}_nok.c', replace)

        # Generate the code with an invalid communicator
        replace = patterns.copy()
        replace['shortdesc'] = 'Point to point @{p1}@ and @{p2}@ have an invalid communicator'
        replace['longdesc'] = 'Point to point @{p1}@ and @{p2}@ have an invalid communicator.'
        replace['outcome'] = 'ERROR: InvalidCommunicator'
        replace['errormsg'] = 'Invalid Communicator. @{p1}@ at @{filename}@:@{line:MBIERROR1}@ and @{p2}@ at @{filename}@:@{line:MBIERROR2}@ use a communicator that is freed line @{line:MBIERROR}@.'
        replace['change_com'] = 'MPI_Comm_free(&newcom);  /* MBIERROR */'
        gen.make_file(template, f'InvalidParam_Com_{p1}_{p2}_nok.c', replace)

        #  Generate the code with an invalid communicator ==> TO CHECK
        #replace = patterns.copy()
        #replace['shortdesc'] = 'Point to point @{p1}@ and @{p2}@ have an invalid communicator'
       # replace['longdesc'] = 'Point to point @{p1}@ and @{p2}@ have an invalid communicator.'
       # replace['outcome'] = 'ERROR: InvalidCommunicator'
       # replace['errormsg'] = 'Invalid Communicator. @{p1}@ at @{filename}@:@{line:MBIERROR1}@ and @{p2}@ at @{filename}@:@{line:MBIERROR2}@ use different communicators'
       # replace['origin'] = "MPI-Corrbench"
       # replace['change_com'] = ""
       # gen.make_file(template, f'InvalidParam_Com_{p1}_{p2}_nok.c', replace)

        # Generate the code with an invalid dest
        replace = patterns.copy()
        replace['origin'] = "MBI"
        replace['shortdesc'] = 'Point to point @{p1}@ has an invalid argument'
        replace['longdesc'] = 'Point to point @{p1}@ and @{p2}@ have an invalid communicator.'
        replace['outcome'] = 'ERROR: InvalidSrcDest'
        replace['errormsg'] = 'InvalidSrcDest. @{p1}@ at @{filename}@:@{line:MBIERROR1}@ performs a send with a dest not in communicator (dest is changed line @{line:MBIERROR}@).'
        replace['change_com'] = ""
        replace['change_srcdest'] = 'dest=4; /* MBIERROR */'
        gen.make_file(template, f'InvalidParam_Dest_{p1}_{p2}_nok.c', replace)

        # Generate the code with an invalid src
        replace = patterns.copy()
        replace['shortdesc'] = 'Point to point @{p2}@ has an invalid argument'
        replace['longdesc'] = 'Point to point @{p1}@ and @{p2}@ have an invalid communicator.'
        replace['outcome'] = 'ERROR: InvalidSrcDest'
        replace['errormsg'] = 'InvalidSrcDest. @{p2}@ at @{filename}@:@{line:MBIERROR2}@ performs a recv with a negative integer as source (src is changed line @{line:MBIERROR}@).'
        replace['change_srcdest'] = 'src=-1; /* MBIERROR */'
        gen.make_file(template, f'InvalidParam_Src_{p1}_{p2}_nok.c', replace)
