// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "components/autofill/core/browser/geo/country_data.h"

#include <utility>

#include "base/memory/singleton.h"
#include "base/stl_util.h"
#include "components/strings/grit/components_strings.h"
#include "third_party/icu/source/common/unicode/locid.h"

namespace autofill {
namespace {

struct StaticCountryData {
  char country_code[3];
  CountryData country_data;
};

// Alias definitions record for CountryData requests.  A request for
// |country_code_alias| is served with the |CountryData| for
// |country_code_target|.
struct StaticCountryCodeAliasData {
  char country_code_alias[3];
  char country_code_target[3];
};

// Alias definitions.
const StaticCountryCodeAliasData kCountryCodeAliases[] = {{"UK", "GB"}};

// Maps country codes to localized label string identifiers. Keep this sorted
// by country code.
// This list is comprized of countries appearing in both
// //third_party/icu/source/data/region/en.txt and
// //third_party/libaddressinput/src/cpp/src/region_data_constants.cc.
const StaticCountryData kCountryData[] = {
    // clang-format off
  {"AC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"AD", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PARISH,
           ADDRESS_REQUIRES_ADDRESS_LINE_1_ONLY } },
  {"AE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_EMIRATE,
           ADDRESS_REQUIRES_STATE } },
  {"AF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"AG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_ADDRESS_LINE_1_ONLY } },
  {"AI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"AL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"AM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"AO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"AQ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"AR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"AS", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"AT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"AU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"AW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"AX", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"AZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"BA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"BB", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PARISH,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"BD", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"BE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"BF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"BG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"BH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"BI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"BJ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"BL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"BM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"BN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"BO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"BQ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"BR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"BS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_ISLAND,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"BT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"BV", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"BW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"BY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"BZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"CA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"CC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"CD", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"CF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"CG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"CH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"CI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"CK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"CL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"CM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"CN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"CO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"CR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"CS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_ADDRESS_LINE_1_ONLY } },
  {"CV", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_ISLAND,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"CW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"CX", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"CY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"CZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"DE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"DJ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"DK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"DM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"DO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"DZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"EC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"EE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"EG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"EH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"ER", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"ES", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"ET", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"FI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"FJ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"FK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"FM", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"FO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"FR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"GA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"GB", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_COUNTY,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"GD", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"GE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"GF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"GG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"GH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"GI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_ADDRESS_LINE_1_ONLY } },
  {"GL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"GM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"GN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"GP", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"GQ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"GR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"GS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"GT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"GU", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"GW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"GY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"HK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_AREA,
           ADDRESS_REQUIRES_STATE } },
  {"HM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"HN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"HR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"HT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"HU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"ID", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"IE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_COUNTY,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"IL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"IM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"IN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"IO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"IQ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"IR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"IS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"IT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"JE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"JM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PARISH,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"JO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"JP", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PREFECTURE,
           ADDRESS_REQUIRES_STATE_ZIP } },
  {"KE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"KG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"KH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"KI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_ISLAND,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"KM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"KN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_ISLAND,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"KP", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"KR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"KW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"KY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_ISLAND,
           ADDRESS_REQUIRES_STATE } },
  {"KZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"LA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"LB", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"LC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"LI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"LK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"LR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"LS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"LT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"LU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"LV", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"LY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"MA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"MC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"MD", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"ME", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"MF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"MG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"MH", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"MK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"ML", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"MM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"MN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"MO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_ADDRESS_LINE_1_ONLY } },
  {"MP", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"MQ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"MR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"MS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"MT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"MU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"MV", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"MW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"MX", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"MY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"MZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"NA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"NC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"NE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"NF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"NG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"NI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_DEPARTMENT,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"NL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"NO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"NP", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"NR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_DISTRICT,
           ADDRESS_REQUIRES_STATE } },
  {"NU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"NZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"OM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"PA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"PE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"PF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_ISLAND,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"PG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"PH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"PK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"PL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"PM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"PN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"PR", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"PS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"PT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"PW", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"PY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"QA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"RE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"RO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"RS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"RU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"RW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"SA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"SB", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"SC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_ISLAND,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"SE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"SG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_ZIP } },
  {"SH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"SI", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"SJ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"SK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"SL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"SM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_ZIP } },
  {"SN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"SO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"SR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"SS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"ST", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"SV", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"SX", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"SZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"TA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"TC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"TD", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"TF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"TG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"TH", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"TJ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"TK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"TL", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"TM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"TN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"TO", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"TR", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"TT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"TV", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_ISLAND,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"TW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_COUNTY,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"TZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"UA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"UG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"UM", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"US", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"UY", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"UZ", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"VA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"VC", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"VE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_STATE } },
  {"VG", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_ADDRESS_LINE_1_ONLY } },
  {"VI", { IDS_AUTOFILL_FIELD_LABEL_ZIP_CODE,
           IDS_AUTOFILL_FIELD_LABEL_STATE,
           ADDRESS_REQUIRES_CITY_STATE_ZIP } },
  {"VN", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"VU", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"WF", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"WS", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"XK", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"YE", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"YT", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"ZA", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY_ZIP } },
  {"ZM", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
  {"ZW", { IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
           IDS_AUTOFILL_FIELD_LABEL_PROVINCE,
           ADDRESS_REQUIRES_CITY } },
    // clang-format on
};

// GetCountryCodes and GetCountryData compute the data for CountryDataMap
// based on |kCountryData|.
std::vector<std::string> GetCountryCodes() {
  std::vector<std::string> country_codes;
  country_codes.reserve(base::size(kCountryData));
  for (const auto& static_data : kCountryData) {
    country_codes.push_back(static_data.country_code);
  }
  return country_codes;
}

std::map<std::string, CountryData> GetCountryDataMap() {
  std::map<std::string, CountryData> country_data;
  // Add all the countries we have explicit data for.
  for (const auto& static_data : kCountryData) {
    country_data.insert(
        std::make_pair(static_data.country_code, static_data.country_data));
  }

  // Add any other countries that ICU knows about, falling back to default data
  // values.
  for (const char* const* country_pointer = icu::Locale::getISOCountries();
       *country_pointer; ++country_pointer) {
    std::string country_code = *country_pointer;
    if (!country_data.count(country_code)) {
      CountryData data = {IDS_AUTOFILL_FIELD_LABEL_POSTAL_CODE,
                          IDS_AUTOFILL_FIELD_LABEL_PROVINCE};
      country_data.insert(
          std::make_pair(std::move(country_code), std::move(data)));
    }
  }
  return country_data;
}

std::map<std::string, std::string> GetCountryCodeAliasMap() {
  std::map<std::string, std::string> country_code_aliases;
  // Create mappings for the aliases defined in |kCountryCodeAliases|.
  for (const auto& static_alias_data : kCountryCodeAliases) {
    // Insert the alias.
    country_code_aliases.insert(
        std::make_pair(std::string(static_alias_data.country_code_alias),
                       std::string(static_alias_data.country_code_target)));
  }
  return country_code_aliases;
}

}  // namespace

// static
CountryDataMap* CountryDataMap::GetInstance() {
  return base::Singleton<CountryDataMap>::get();
}

CountryDataMap::CountryDataMap()
    : country_data_(GetCountryDataMap()),
      country_code_aliases_(GetCountryCodeAliasMap()),
      country_codes_(GetCountryCodes()) {}

CountryDataMap::~CountryDataMap() = default;

bool CountryDataMap::HasCountryData(const std::string& country_code) const {
  return country_data_.count(country_code) > 0;
}

const CountryData& CountryDataMap::GetCountryData(
    const std::string& country_code) const {
  auto lookup = country_data_.find(country_code);
  if (lookup != country_data_.end())
    return lookup->second;
  // If there is no entry for country_code return the entry for the US.
  return country_data_.find("US")->second;
}

bool CountryDataMap::HasCountryCodeAlias(
    const std::string& country_code_alias) const {
  return country_code_aliases_.count(country_code_alias) > 0;
}

const std::string CountryDataMap::GetCountryCodeForAlias(
    const std::string& country_code_alias) const {
  auto lookup = country_code_aliases_.find(country_code_alias);
  if (lookup != country_code_aliases_.end()) {
    DCHECK(HasCountryData(lookup->second));
    return lookup->second;
  }
  return std::string();
}

}  // namespace autofill
