/*
** ypserver.c			YP server routines.
**
** Copyright (c) 1993 Signum Support AB, Sweden
**
** This file is part of the NYS YP Server.
**
** The NYS YP Server is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** The NYS YP Server is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
** 
** You should have received a copy of the GNU General Public
** License along with the NYS YP Server; see the file COPYING.  If
** not, write to the Free Software Foundation, Inc., 675 Mass Ave,
** Cambridge, MA 02139, USA.
**
** Author: Peter Eriksson <pen@signum.se>
*/

#include <stdio.h>
#include <fcntl.h>
#include <dirent.h>
#include <string.h>
#include <gdbm.h>
#include <sys/stat.h>
#include "yp.h"


#define _PATH_YP   "/var/yp"
char *path_ypdb = _PATH_YP;

int debug_flag = 1;


static void Gperror(char *str)
{
    fprintf(stderr, "%s: GDBM Error Code #%d\n", gdbm_errno);
}


void *ypproc_null_2_svc(void *dummy, struct svc_req *rqstp)
{
    static int foo;

    
    if (debug_flag)
	fprintf(stderr, "ypproc_null()\n");

    return (void *) &foo;
}

/*
** Return 1 if the name is a valid domain name served by us, else 0.
*/
static int is_valid_domain(char *name)
{
    char buf[1025];
    struct stat sbuf;
    
    
    sprintf(buf, "%s/%s", path_ypdb, name);
    if (stat(buf, &sbuf) < 0 || !S_ISDIR(sbuf.st_mode))
	return 0;

    return 1;
}


bool_t *ypproc_domain_2_svc(domainname *name, struct svc_req *rqstp)
{
    static bool_t result;


    if (debug_flag)
	fprintf(stderr, "ypproc_domain(\"%s\")\n", *name);

    if (is_valid_domain(*name))
	result = TRUE;
    else
	result = FALSE;
    
    if (debug_flag)
	fprintf(stderr, "\t-> %s.\n",
		(result == TRUE ? "Ok" : "Not served by us"));
    
    return &result;
}


bool_t *ypproc_domain_nonack_2_svc(domainname *name, struct svc_req *rqstp)
{
    static bool_t result;

    if (debug_flag)
	fprintf(stderr, "ypproc_domain_nonack(\"%s\")\n", *name);
    
    if (!is_valid_domain(*name))
    {
	if (debug_flag)
	    fprintf(stderr, "\t-> Not a valid domain.\n");

	/* Bail out and don't return any RPC value */
	return NULL;
    }
    
    if (debug_flag)
	fprintf(stderr, "\t-> OK.\n");
    
    result = TRUE;
    return &result;
}


static GDBM_FILE open_database(char *domain, char *map)
{
    GDBM_FILE dbp;
    char buf[1025];

    
    sprintf(buf, "%s/%s/%s", path_ypdb, domain, map);
    
    dbp = gdbm_open(buf, 0, O_RDONLY, 0, NULL);

    if (debug_flag && dbp == NULL)
	Gperror("gdbm_open()");
    
    return dbp;
}


/*
** Get the DateTimeModified value for a certain map database
*/
static unsigned long get_dtm(char *domain, char *map)
{
    struct stat sbuf;
    char buf[1025];

    sprintf(buf, "%s/%s/%s", path_ypdb, domain, map);
    if (stat(buf, &sbuf) < 0)
	return 0;
    else
	return (unsigned long) sbuf.st_mtime;
}


ypresp_val *ypproc_match_2_svc(ypreq_key *key, struct svc_req *rqstp)
{
    static ypresp_val result;

    if (debug_flag)
    {
	fprintf(stderr, "ypproc_match():\n");
	fprintf(stderr, "\t\tdomainname = \"%s\"\n",
		key->domain);
	fprintf(stderr, "\t\tmapname = \"%s\"\n",
		key->map);
	fprintf(stderr, "\t\tkeydat = \"%.*s\"\n",
		key->key.keydat_len,
		key->key.keydat_val);
    }

    result.val.valdat_len = 0;
    if (result.val.valdat_val)
    {
	free(result.val.valdat_val);
	result.val.valdat_val = NULL;
    }
    
    if (key->domain[0] == '\0' || key->map[0] == '\0')
	result.stat = YP_BADARGS;
    else if (!is_valid_domain(key->domain))
	result.stat = YP_NODOM;
    else
    {
	datum rdat, qdat;
	
	GDBM_FILE dbp = open_database(key->domain, key->map);
	if (dbp == NULL)
	    result.stat = YP_NOMAP;
	else
	{
	    qdat.dsize = key->key.keydat_len;
	    qdat.dptr = key->key.keydat_val;

	    rdat = gdbm_fetch(dbp, qdat);
	    if (rdat.dptr == NULL)
	    {
		if (debug_flag)
		    Gperror("gdbm_fetch()");
		result.stat = YP_NOKEY;
	    }
	    else
	    {
		result.stat = YP_TRUE;
		result.val.valdat_len = rdat.dsize;
		result.val.valdat_val = rdat.dptr;
	    }
	    gdbm_close(dbp);
	}
    }
	
    return &result;
}

ypresp_key_val *ypproc_first_2_svc(ypreq_key *key, struct svc_req *rqstp)
{
    static ypresp_key_val result;


    if (debug_flag)
    {
	fprintf(stderr, "ypproc_match():\n");
	fprintf(stderr, "\tdomainname = \"%s\"\n", key->domain);
	fprintf(stderr, "\tmapname = \"%s\"\n", key->map);
	fprintf(stderr, "\tkeydat = \"%.*s\"\n", key->key.keydat_len,
		key->key.keydat_val);
    }

    result.key.keydat_len = 0;
    if (result.key.keydat_val)
    {
	free(result.key.keydat_val);
	result.key.keydat_val = NULL;
    }
    
    result.val.valdat_len = 0;
    if (result.val.valdat_val)
    {
	free(result.val.valdat_val);
	result.val.valdat_val = NULL;
    }

    if (key->map[0] == '\0' || key->domain[0] == '\0')
	result.stat = YP_BADARGS;
    else if (!is_valid_domain(key->domain))
	result.stat = YP_NODOM;
    else
    {
	datum dkey, dval;
	
	GDBM_FILE dbp = open_database(key->domain, key->map);
	if (dbp == NULL)
	    result.stat = YP_NOMAP;
	else
	{
	    dkey = gdbm_firstkey(dbp);
	    if (dkey.dptr == NULL)
		result.stat = YP_NOKEY;
	    else
	    {
		dval = gdbm_fetch(dbp, dkey);
		
		result.stat = YP_TRUE;
		
		result.key.keydat_len = dkey.dsize;
		result.key.keydat_val = dkey.dptr;
		
		result.val.valdat_len = dval.dsize;
		result.val.valdat_val = dval.dptr;
	    }
	}
	gdbm_close(dbp);
    }
    
    return &result;
}


ypresp_key_val *ypproc_next_2_svc(ypreq_key *key, struct svc_req *rqstp)
{
    static ypresp_key_val result;


    if (debug_flag)
    {
	fprintf(stderr, "ypproc_match():\n");
	fprintf(stderr, "\tdomainname = \"%s\"\n", key->domain);
	fprintf(stderr, "\tmapname = \"%s\"\n", key->map);
	fprintf(stderr, "\tkeydat = \"%.*s\"\n", key->key.keydat_len,
		key->key.keydat_val);
    }

    result.key.keydat_len = 0;
    if (result.key.keydat_val)
    {
	free(result.key.keydat_val);
	result.key.keydat_val = NULL;
    }
    
    result.val.valdat_len = 0;
    if (result.val.valdat_val)
    {
	free(result.val.valdat_val);
	result.val.valdat_val = NULL;
    }
    
    if (key->map[0] == '\0' || key->domain[0] == '\0')
	result.stat = YP_BADARGS;
    else if (!is_valid_domain(key->domain))
	result.stat = YP_NODOM;
    else
    {
	datum dkey, dval;


	GDBM_FILE dbp = open_database(key->domain, key->map);
	if (dbp == NULL)
	    result.stat = YP_NOMAP;
	else
	{
	    dkey.dsize = key->key.keydat_len;
	    dkey.dptr  = key->key.keydat_val;
	    
	    dkey = gdbm_nextkey(dbp, dkey);
	    if (dkey.dptr == NULL)
	    {
		if (debug_flag)
		    Gperror("gdbm_nextkey()");
		
		result.stat = YP_NOMORE;
	    }
	    else
	    {
		dval = gdbm_fetch(dbp, dkey);
		if (dval.dptr == NULL)
		{
		    if (debug_flag)
			Gperror("gdbm_fetch()");
		    
		    result.stat = YP_BADDB;
		}
		else
		{
		    result.stat = YP_TRUE;
		    
		    result.key.keydat_len = dkey.dsize;
		    result.key.keydat_val = dkey.dptr;
		    
		    result.val.valdat_len = dval.dsize;
		    result.val.valdat_val = dval.dptr;
		}
	    }
	    
	    gdbm_close(dbp);
	}
    }

    return &result;
}

static void print_ypmap_parms(struct ypmap_parms *pp)
{
    fprintf(stderr, "\t\tdomain   = \"%s\"\n", pp->domain);
    fprintf(stderr, "\t\tmap      = \"%s\"\n", pp->map);
    fprintf(stderr, "\t\tordernum = %u\n", pp->ordernum);
    fprintf(stderr, "\t\tpeer     = \"%s\"\n", pp->peer);
}


ypresp_xfr *ypproc_xfr_2_svc(ypreq_xfr *xfr, struct svc_req *rqstp)
{
    static ypresp_xfr result;

    if (debug_flag)
    {
	fprintf(stderr, "ypproc_xfr_2_svc():\n\tmap_parms:\n");
	print_ypmap_parms(&xfr->map_parms);
	fprintf(stderr, "\t\ttransid = %u\n", xfr->transid);
	fprintf(stderr, "\t\tprog = %u\n", xfr->prog);
	fprintf(stderr, "\t\tport = %u\n", xfr->port);
	fprintf(stderr, "\t-> Request denied.\n");
    }

    result.transid = xfr->transid;
    result.xfrstat = YPXFR_REFUSED;
    
    return &result;
}


void *ypproc_clear_2_svc(void *dummy, struct svc_req *rqstp)
{
    static int foo;

    if (debug_flag)
    	fprintf(stderr, "ypproc_clear_2_svc()\n");

    return (void *) &foo;
}



ypresp_all *ypproc_all_2_svc(ypreq_nokey *nokey, struct svc_req *rqstp)
{
    static ypresp_all result;
    
    
    if (debug_flag)
    {
	fprintf(stderr, "ypproc_all_2_svc():\n");
	fprintf(stderr, "\t\tdomain = \"%s\"\n", nokey->domain);
	fprintf(stderr, "\t\tmap = \"%s\"\n", nokey->map);
	fprintf(stderr, "\t-> Request denied (not yet implemented).\n");
    }
    
    result.more = 0;
    result.ypresp_all_u.val.stat = YP_YPERR;
    result.ypresp_all_u.val.key.keydat_len = 0;
    result.ypresp_all_u.val.key.keydat_val = NULL;
    result.ypresp_all_u.val.val.valdat_len = 0;
    result.ypresp_all_u.val.val.valdat_val = NULL;
    
    return &result;
}


ypresp_master *ypproc_master_2_svc(ypreq_nokey *nokey, struct svc_req *rqstp)
{
    static ypresp_master result;

    
    if (debug_flag)
    {
	fprintf(stderr, "ypproc_master_2_svc():\n");
	fprintf(stderr, "\t\tdomain = \"%s\"\n", nokey->domain);
	fprintf(stderr, "\t\tmap = \"%s\"\n", nokey->map);
    }

    if (result.peer)
    {
	free(result.peer);
	result.peer = NULL;
    }

    if (nokey->domain[0] == '\0')
	result.stat = YP_BADARGS;
    else if (!is_valid_domain(nokey->domain))
	result.stat = YP_NODOM;
    else
    {
	GDBM_FILE dbp = open_database(nokey->domain, nokey->map);
	if (dbp == NULL)
	    result.stat = YP_NOMAP;
	else
	{
	    datum key, val;

	    key.dsize = sizeof("YP_MASTER_NAME")-1;
	    key.dptr = "YP_MASTER_NAME";
	    
	    val = gdbm_fetch(dbp, key);
	    if (val.dptr == NULL)
	    {
		/* No YP_MASTER_NAME record in map? Assume we are Master */
		static char hostbuf[1025];

		gethostname(hostbuf, sizeof(hostbuf)-1);
		result.peer = strdup(hostbuf);
	    }
	    else
		result.peer = val.dptr;

	    result.stat = YP_TRUE;
	    gdbm_close(dbp);
	}
    }

    if (result.peer == NULL)
	result.peer = strdup("");

    if (debug_flag)
	fprintf(stderr, "\t-> Peer = \"%s\"\n", result.peer);
    
    return &result;
}


ypresp_order *ypproc_order_2_svc(ypreq_nokey *nokey, struct svc_req *rqstp)
{
    static ypresp_order result;

    if (debug_flag)
    {
	fprintf(stderr, "ypproc_order_2_svc():\n");
	fprintf(stderr, "\t\tdomain = \"%s\"\n", nokey->domain);
	fprintf(stderr, "\t\tmap = \"%s\"\n", nokey->map);
    }

    result.ordernum = 0;

    if (nokey->domain[0] == '\0')
	result.stat = YP_BADARGS;
    else if (!is_valid_domain(nokey->domain))
	result.stat = YP_NODOM;
    else
    {
	GDBM_FILE dbp = open_database(nokey->domain, nokey->map);
	if (dbp == NULL)
	    result.stat = YP_NOMAP;
	else
	{
	    datum key, val;

	    key.dsize = sizeof("YP_LAST_MODIFIED")-1;
	    key.dptr = "YP_LAST_MODIFIED";
	    
	    val = gdbm_fetch(dbp, key);
	    if (val.dptr == NULL)
	    {
		/* No YP_LAST_MODIFIED record in map? Use DTM timestamp.. */
		result.ordernum = get_dtm(nokey->domain, nokey->map);
	    }
	    else
	    {
		result.ordernum = atoi(val.dptr);
		free(val.dptr);
	    }

	    result.stat = YP_TRUE;
	    gdbm_close(dbp);
	}
    }

    if (debug_flag)
	fprintf(stderr, "-> Order # %d\n", result.ordernum);
    
    return &result;
}


static void free_maplist(ypmaplist *mlp)
{
    ypmaplist *next;

    while (mlp != NULL)
    {
	next = mlp->next;
	free(mlp->map);
	free(mlp);
	mlp = next;
    }
}

static int add_maplist(ypmaplist **mlhp, char *map)
{
    ypmaplist *mlp;

    mlp = malloc(sizeof(*mlp));
    if (mlp == NULL)
	return -1;

    mlp->map = strdup(map);
    if (mlp->map == NULL)
    {
	free(mlp);
	return -1;
    }

    mlp->next = *mlhp;
    *mlhp = mlp;

    return 0;
}

ypresp_maplist *ypproc_maplist_2_svc(domainname *name, struct svc_req *rqstp)
{
    static ypresp_maplist result;
    

    if (debug_flag)
    {
	fprintf(stderr, "ypproc_maplist_2_svc():\n");
	fprintf(stderr, "\t\tdomain = \"%s\"\n", *name);
    }

    if (result.maps)
	free_maplist(result.maps);
    
    result.maps = NULL;

    if ((*name)[0] == '\0')
	result.stat = YP_BADARGS;
    else if (!is_valid_domain(*name))
	result.stat = YP_NODOM;
    else
    {
	DIR *dp;

	dp = opendir(path_ypdb);
	if (dp == NULL)
	{
	    if (debug_flag)
		perror("opendir");
	    
	    result.stat = YP_BADDB;
	}
	else
	{
	    struct dirent *dep;
	    
	    while (dep = readdir(dp))
		if (add_maplist(&result.maps, dep->d_name) < 0)
		{
		    result.stat = YP_YPERR;
		    break;
		}
	}

	closedir(dp);
    }
    
    return &result;
}
