/*
** nis_if.c                Cover routines for NIS+
**
** Copyright (c) 1993 Signum Support AB, Sweden
**
** This file is part of the NYS Library.
**
** The NYS Library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Library General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** The NYS Library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Library General Public License for more details.
** 
** You should have received a copy of the GNU Library General Public
** License along with the NYS Library; see the file COPYING.LIB.  If
** not, write to the Free Software Foundation, Inc., 675 Mass Ave,
** Cambridge, MA 02139, USA.
**
** Author: Peter Eriksson <pen@signum.se>
*/

#include "config.h"

#ifdef ENABLE_NISEMU

#include <stdio.h>
#include <rpc/rpc.h>
#include "rpcsvc/nis.h"
#include "nis_conf.h"


extern char *strstr();


static int nissetup_flag = 0;


static int nis_setup(void)
{
    if (nissetup_flag)
	return 0;
    
    if (nisconf_set())
	return -1;

    nissetup_flag = 1;
    return 0;
}


int nis_get_default_domain(char **outdomain)
{
    char buf[1025];
    
    
    if (nis_setup())
	return -1;

    if (_nis_config->domainname)
    {
	*outdomain = strdup(_nis_config->domainname);
	return 0;
    }

    if (getdomainname(buf, sizeof(buf)) == 0)
    {
	*outdomain = strdup(buf);
	return 0;
    }

    return -1;
}


nis_name nis_mkfullname(nis_name name, char **domain)
{
    static char buf[1024];
    int len;
    char *ddom;
    
    
    len = strlen(name);
    strcpy(buf, name);
    
    if (len == 0)
	return buf;

    if (name[len-1] == '.')
    {
	/* Fully specified domainname */

	if (domain)
	{
	    char *cp;
	    int i;

	    
	    i = sizeof(".org_dir.");
	    cp = strstr(name, ".org_dir.");
	    if (cp == NULL)
	    {
		i = sizeof(".group_dir.");
		cp = strstr(name, ".group_dir.");
	    }

	    if (cp == NULL)
		*domain = "<unknown>";
	    else
		*domain = cp + i - 1;
	}
	return buf;
    }
    
    if (nis_get_default_domain(&ddom) == 0)
    {
	strcat(buf, ".");
	strcat(buf, ddom);
	
	if (domain)
	    *domain = ddom;
	
         return buf;
    }
    else 
	return buf; /* XXX or should we return NULL and fail? */
}



static nis_result *
  do_niscall(char *domain,
	     struct nis_result * (*cfh)(void *req, CLIENT *clh),
	     void *req)
			     
{
    struct nis_result *res;
    int nresp_flag = 0;
    CLIENT *clh;
    char *address;
    
    
    if (nis_setup())
	return NULL;

    clh = nisconf_domain2client(domain, NULL, &address);
    if (clh == NULL)
	return NULL;
				
    while ((res = (*cfh)(req, clh)) == NULL)
    {
	nresp_flag = 1;
	fprintf(stderr,
    "NIS: server \"%s\" not responding for domain \"%s\", still trying...\n",
		address,
		domain);
    }

    if (nresp_flag == 1)
	fprintf(stderr, "NIS: server \"%s\" OK.\n",
		address);

    return res;
		
}


nis_result *nis_lookup(nis_name name, u_long flags)
{
    struct ns_request nsreq;
    char *domain;

    
    if (nis_setup())
	return NULL;

    nsreq.ns_name = nis_mkfullname(name, &domain);
    nsreq.ns_object.ns_object_len = 0;
    nsreq.ns_object.ns_object_val = NULL;
	
    return do_niscall(domain, &nis_lookup_3, &nsreq);
}


nis_result *nis_first_entry(nis_name name)
{
    struct ib_request ibreq;
    char *domain;

    
    if (nis_setup() == -1)
	return NULL;

    ibreq.ibr_name = nis_mkfullname(name, &domain);
    ibreq.ibr_flags = 0;

    ibreq.ibr_srch.ibr_srch_len = 0;
    ibreq.ibr_srch.ibr_srch_val = NULL;
    
    ibreq.ibr_obj.ibr_obj_len = 0;
    ibreq.ibr_obj.ibr_obj_val = NULL;
    
    ibreq.ibr_cbhost.ibr_cbhost_len = 0;
    ibreq.ibr_cbhost.ibr_cbhost_val = NULL;

    ibreq.ibr_bufsize = 0;
    
    ibreq.ibr_cookie.n_len = 0;
    ibreq.ibr_cookie.n_bytes = NULL;
    
    return do_niscall(domain, &nis_ibfirst_3, &ibreq);
}


nis_result *nis_next_entry(nis_name name, netobj *cookie)
{
    struct ib_request ibreq;
    char *domain;
    

    if (nis_setup() == -1)
	return NULL;

    ibreq.ibr_name = nis_mkfullname(name, &domain);
    ibreq.ibr_flags = 0;

    ibreq.ibr_srch.ibr_srch_len = 0;
    ibreq.ibr_srch.ibr_srch_val = NULL;
    
    ibreq.ibr_obj.ibr_obj_len = 0;
    ibreq.ibr_obj.ibr_obj_val = NULL;
    
    ibreq.ibr_cbhost.ibr_cbhost_len = 0;
    ibreq.ibr_cbhost.ibr_cbhost_val = NULL;

    ibreq.ibr_bufsize = 0;
    
    ibreq.ibr_cookie = *cookie;

    return do_niscall(domain, &nis_ibnext_3, &ibreq);
}

#endif /* ENABLE_NISEMU */
