#!/usr/bin/perl -w
# based on fl10.mpg
# http://bmrc.berkeley.edu/ftp/pub/multimedia/mpeg/mpeg2/conformance-bitstreams/audio/mpeg1/compliance/

use strict;
use integer;

my $n;
use Test::More tests => ($n = 49) * 9 + 2;
BEGIN { use_ok("MPEG::Audio::Frame") };

for (1 .. $n){
	isa_ok(my $frame = MPEG::Audio::Frame->read(*DATA), "MPEG::Audio::Frame", "frame $_");
	ok($frame->mpeg1, "frame is MPEG1");
	ok($frame->layer2, "frame is layer II");
	ok($frame->has_crc, "frame has CRC");
	ok(!$frame->broken, "frame isn't broken");
	is($frame->bitrate, 192, "bitrate");
	is($frame->sample, 32000, "sample rate");
	ok((($_-1) % 5 == 0) ? $frame->stereo : $frame->joint_stereo, "stereo xor joint stereo");
	is($frame->modext, (0, 0, 3, 2, 1)[$_ % 5], "modext");
}
is(MPEG::Audio::Frame->read(*DATA), undef, "nothing else in the file");

__DATA__
 #wUwwfffDD3m$I"I$    PÜ9˜Ϭӏ4͸ۍ۝~y88y8	~~}}}}}{{l[űm>|ϟ>kZֵe877z&މu{EVűl[mmϟ>|浭kZ=u'Oa((_̗,BWCr+ڰS`W}h@3)Φo.5ˣxHI"Hqyy^Ev]n$_-"E*y糥
aB}}f?_Wvd]-\ǹ,hpmNJJ^񍭫"kmȱx&8&7fMٓL54i@yӛqөm
8)!p1sssse=i"ZB'FwＣyEdY -СqÈ0^b˛@MBPFIvw1nf\%lia˅qZP 2kZ6C_ʹgg8(~#%	qQ%8q{p4ϟF3ǆYˋ~G2ߑ3f3i|&eYm/TKʗRrU%IkuVդL%*{Ycncgg"Jgni LYČr;wx^E"FrZ.gNcf2n:GiGdͩ3jef^eD " @Ɂm]Wmv[VL"E)k9pGU3UUUffUUUI$nFH UUUU}-}WUgy]s<<<qǜ]w|<<s_^
V򧼩fnzӭ-ЊÕJw:Э

ZKSeng2aί{u}c;uԛI
̬Ӻ'typ	{n@K‵WR\'wl=:<.*-"ҷ+lk'0Si<鮪*kX*5rr7K^स"ѝHK5EhI`n4Hҭ*9%[$$̗4wd.˶n/ 	Zr5jlMUqJd=@(
6
-#;U0"©I~-IRvkfQ5
tb%첬2޷O_>kO[h7xAyqoNH:u)S-leYO=hT,z& UipaK
fγnfA,RblԭJnϬMMh<S)&ZqJ$'Q!8;}IRR+(ꎰik$3tf*yjiv5IF=nG#7Qk~5+_SzޮN/jr1,Y9s1cQPOt[𶭭FWjA708n϶oA{ q)(1#UMR2IxI-x!AI

F9c2(*l}TM*SJqq4Lx	ҰиYPjh`ef3UfffffeU$m UUUUU UUUUUUygsqs<s>ϼs<<sqsC^f4ԱD#>4Phvs?o*'TOR֊Yu]CJ6S/".vB)ιseLʝ!:uuևq!
Q5	9Y'J[35Bf0jW,+4Vh=Vz94rh=эj2FL:I:SaS٨MD+i][Dk:t:i9r&%yChƒPG]P:FR2̬ʤLXёAqۄfp8D^W]v6:2f?4q
wPkCD333={Y31<bk1BcV:Y'tVUE FFm0'[=ܹ\C;wN5'ZN]3G#B
!׀f`$׀xTn.#/]QZM9sgσgk6m^=y. wiXM%bh4u<y9c9·܍sPi4P\ N EmC]vKVwl$-v-K=j^2	.2*ʬL Apc+:ur"]f2x{9Uդ5)A^>7HԦn,
4U5BjE$]>|#%)Y7wUdճ9kBo.f|u3\ 
+"={KZC⇲1db9qvSr6)4M P,f3ffwfffkmFҕV9睹۞yy89y8㮸9zw}߮x8ypa㇮+]a%TJrbeէXXڱGN2q2,8Qbsgf;;5V	&6Ll'Og[s]"CU
lGPee½ŒͱPۍ^8/v^9s1n[U)iӏ{DˢV%(Z9pCJ"!\v8rpIRC{kJ֕V,6̰jNuվ-_ANw
s6'?:T1ʃ+lYbSc9ssvl\ZX;meyM؀ʍlYUY} EQii5l7CkK4E|e";d).=ygT=z&F,YEMi+T5PZ&H@ֳ#
Sͱbem(Qjj֙wF쵡-hUvx9*I:u
ohdXl9iV Bv@
VҴݙY+tVϋX'V!iA{D}
5XibɗWh9#0fA7Fr
Ý~1S$Iҙ2>[2̧O0,(XP
FN'pN)p̻B0#mx'4;9M iqNZ%f^EgYk:V]({pt9+>OS⥷Zqo
ݦMzK5
مTs)j2ԁ֕q @UDDfggfm$Vꪪ9xz߭9|y8xy9x8yz9iIvUiIvVSw̧Os;.HZtAP%[\:hWyg3BLH^$^Y-Y-{ϝ-=Ȝ[ҞUYIѳ]M׍(4ӥp568in8inQ(JPF֍έI$ƝNp*UHhaw^!e-Zvҷ:&BBT^E.V-s-LA1+5cŝywEw[N]g:Yq:Yqe_))F(Rq#ꍌFKF?IU#ǅ#obm5Fj-j
>OʩtfZMk4#5%dWh˒'Ә˒'Ƀ	̂θr;v;kbׇrwi5QmYGsU;
pV6pV68q5[YV5Tc†['~5nK|;#g[NcNbqF:udI([zp*^Ҙ"7{[hmDh+b^iY}?3EZfQfQ	G/bER0TT1@r錥еŗ䑭xe䑭xer)LƙT844um+d%(*[d5geIm+cE5HE5H)Frh@a}ɐvƩ=>jr0dGC:{uWFȄ'D]  7D"DUUUfDDDmmI"I  

S8뮺y:y8yyy8y뮶c"" ∈s`-qxvYeb+|aX[VVѤjW+VXM<y孄PlfQ^½U*n^Vifa^hZy	HoYqZ'c#x4
[4E+Lt'bvfhmD7IMmƣR9G
JT>c
R5{*hcw [hݝ2Fld)`"LuMj4VPztc)De9'qN<&FQ]U؝MJ&u]P%6ar9f8MVv*V))\H?Ò".t\ `$&BDH0YVUcIR]
Zּyx[M*TZִ޻7[&r)X
`bXG}_T*eiZc1j7I-m۵#vGݤ	5dY%YeZl$vd+e7ōFhi4קMJX	y)[^Jy`+_{WWvh7Տ&Aյo3m*G#^Uj[x"D !kyh{sZ8Uhf/R)afMuuIRVqBй$&#j7v,uE".\,Fg{Xoql[$IjZN'DmrZm8PVZ/ϱƋ3ԩSe2&a1$ṊT-fWAБHxK
D	I$J-`pQD3UefUfUUUI$IH URUUWUWP}g]s<sq]q<oq<ǜyq<w|w<xr~ĪÓ%Qcʚ!s9BXud$Y
2ifzWzgxT%rcQ^8Ys}uT"F7¥1u|j/*8<OX.$\HF4-2#"DKȒ"K-`*ӿ^I
,HPwĥIJndejՎX@ԁɬ$NYݝ%r[3^pٟC5F
27k֬jR&:LIid1hpJ+:7Jn2+g)2W1=2W&L;ûA#Usw6uW%sD4G|U\gH$%	^W%
SW{m'oD"FafZjn)G:s=4J;#IE-t!Fȱݶ&6tlΌ-Uڡ-g|<yYݞ"b%%bVMee#DuAgg	+;N
+*RTKv	V i̬Ȭ/z.Z6JV_<
F
9Ӳ3#)h={ǩ{EDTXi\ ôE9
g|Ns/3s!KnݞػB5nlѳg=P\[0ض#h湮aCIInO޿G*{26j]mU1S{Yi^ڍsF0#)ģC^Hg5R/D^UUUaV\bu,ɫ*han+
8#"kRv((`^33UffffffU㍤UUUUPUUUUUg<^׼os<|{ϼusw<|ss<<C:j$Qk l|oޭ!ZEk,S #B4E1I]"1,(3ɂ	 dRHl7xa7JnILy

0d)FLo;θ,-WFbF7
>MO!ݭ	D}&L"XDU"Mt*¨9-ޒ\l3s9mjT(-\[b%K
&M=RzBr<$UmaUb@dv]%M]ɼKZv5k>}*Ebų':$I˲D.Ԝ-}Hm뽸n3jI4J$٦aêTAʄ,6]S3kcFMRa_g4Ej܅[-̽{QA] %\"C9*
$N/鬅뛖9?k(s7,qkbH;
vOL%K>74;MTnr˟
ܽµE[;>X$#.O\\6mݛ/m^ڻ3BгlkP"mֆ:ޚ!x^6/ꎨK5=r8FpsY̳R310dnÚŜ2vA)O2S̫oTK$:CC4$Iei1fG̎{2*4lB+3EnJ{0G9Ժ[5.bҥ*VnկR6l9t=Eԋ)rS
c1]3
--$8cs&PDDewffffktUVjih*9ߞw߮|89:߭:߭zx;9箹}y箌e|vEbOJʬ-j\k2%N0PX7rJ;͓,rAW!A.]8.]*˲[9}|(sԨw6HE?$$٨HܫĸE]^%-lمj㣃r*UeA %sm{V$SO	b<q`<66#iI}=$5뙤IEc#jy}#F@HIaAbpm(\;AGRQ9n8?B<)FOiӴvRƥ`icА#h^̇!^\89"NEBHiUJ$	UR}8sW"8j&kb[B/md8+-4|7icnܶmۋZeg
>d|)S̶$Mٲ십aa;^cStUD1-
'O7D.I-Z&NăT7
6[J,UY͠uD4|R!␹q[MAVjҕ/3g%'g{oRD
*`>w[2ۅG-B>5NXȫB4,YՕ*a4)^D30<*[kJ8jqTFl"E{lP4Rꥅ*"TCLPNەCDX*v>QUc)[ɱD1$LI-v={TBƉxP$%*ʒ[i:@@#UDfwwvm[d꾪:;yy9箹ߝ㮹yߎ89箹xx|圬l`ӎU~ܖ9,k۾MZp%a-mJ7('3x&Vv*bl[c֙bƴjeE
T
GveSTy$QzrUѕCtE=*&[;/2.F\1yr2RAc*Kg_R݆e#h[ܚ2k!kY˓C -Ϊ溹÷5S9Ξ=k"9,lfKxe0&e㲩L2(Nh_@b@tDbm=Z5Rz+΅Zr<'RB&rS]\.+^ZЉ4"KByfFbmqhA[IM
[}ޙ,<"qS@|Ar
ά:Uʏq5[y>]̂<e
*^+0I(#IxIDK3gjl8q-~jkZe/ֱ`1׳8s2ZK<5wéa)lWߑM4d
2m%ZJml "`UfbENxa$J̌&(A})eڑg(Ԇ]MT_&EVq[\j<4j)`51nyp:εKm)+S`Tg	˯V CK>K>*Uqm[H9v5S=s7eQ
jlmOZsJSծkKkD  D33UTUfDDCFmHIP  
ZZy뮺~8ߝy瞷~88y89}x8@kA5	m$2Zej՝Y72nԓ<<
&DR)h4HeJֵq$Ҏ q	4<y,rˌmSvݤ,݇cm6E'#̉DlcJFs <q.߬YxYEGvMt)lZXr8Q]Fj%J!Q 
/"Bä:	84v<3v2L8hqvQ(N')vpքImNmts#F7YR"X\7EƛMG$N&R)I9ЃZBo/"ȃEv1C	Lb13'AbD?ԑ$[BWHRe2]kceLa+
SJc0a]r9!vmXzd$IjPZem6Ii76:d%5·@p(>5.m;Ӻ1-˻#vFčn{GQٖea[->Z16q
$\jåKr*I6hsygZֳHXk茣2T6m6%mk`ׄN5)*|TLDYE,RKcy7ǉzkZ[űl*SiD)NkTrME4\Q7o'xR*!aCi[K%Ɂ`R°iDQU:N&m>| NjRSdv`;YƅM46[*nC,2bд6M-p$ۍiWm	ŊRkkPp33UUffUUTDI$nHRWURUUUUUUWg]so<]u]s}w<w<<]uoqw|sg'g=a4uGGpl+Q#F-[k4Nʬeɺnf٠Hj*j]23[F*Xhr/ʼWx(IIK`|&]+
f#HyVjh5]%sLk0^ȁʫ*Jԫ;2*#5D
Hݳ33:xݷbCL$XMC.F4<p1sIXarw'Nr-P˖Ѵ"յ#RwMt5dy5FN
2v9ZhqfLYY)2Q2@U5KD:ض7mK$\ol&T2&1XZYZ7TnqHXTwږ&	 WטEfn:[(7VM\%[B,.R\w{<M:2d62rbJkl>n`܏obd}{ ^*2\+jVK9LŜ%q\FѴ=ͬ[o
2C`TUG4mwevLd&aXP#JrԵ k'2Foh}EVmZU*ru7S;hvЊd@c:#C."h`<	O yIVWA>J¬*jj]5\HJ8]WRI$ʻ*溻器v=E,[*1/Xzv)ӝ9Sin[D4EvPR^6J `Ύ33UffffffU$&UUUUz}UUWцqq<}<{|<q<]q]w{[u<q2p2AjqZDR5k"EjxI:Ku+	S6I;fY$Z,2´+3-f[ŷl9nQܣȩS:cJ,̀:c%J٨7CUH7UX7xFdfJ5'JNo=!ByǑy"eYGUQ\j rK4oiIEtVkfnx:s(vPLz V&o6b,<iҞevi]vvij%^3gU6mTx38T *'lm
`ɲ$-i$KB-MT<TS8jiӮ]'O
KN\ح#bjj{dɤhm	l(b*vgiA҃r:ý{իmʜgxX44@F[(x6eLɒW2,
'
P.L".0R ={
P(̳/1	J͑ΠipΡƍZ>#D(QXh׋)b1Ur@qfUhUh*M.:;+Rϭ[&.M۷dI&Lu{JļJӷ2) Xy&9yv(|yvP(b(*4,hV}3ff͉ihve*BWP|sp67o%7^h莈*f
nvUElA}k6..Y^vHQP#/f3fUffff[mVjꪾ9y{x9箸<yx;=8yxy箹箸pp$I$Lt3eӛ9ӑH^"WdevC"C[Ӑ\Ӑ\<]IS13A&17Xnms-;]<ӊFEpB¥h_h_D2xd\.\'ONuXDZìhĈW1`B}j
d"55ذ.R,ٽ36c		5rmM5IK

V'L3Sr)2)֔޴tIsRmID\)>㙖ulex6vȍ:^qS^qSN
0OiS37&k)SE¾3=X;dhõ$'j57V^7V^TŊ,Qsy9݇PnBdϚU9{zVˤc'KBh5gR1-+Q-+Q]#@	X|.T]rfJ=PHjRRPsh/v;v;/|tgFKΕ+ԗ/4h
&!aU"ݖ/B[kfJ2J2EO>*yqYW H,0jE<1sK@ȒP6T1A1B\,tHQ5kVSQ5sבdbczw3 O5ؚc`^h^iuڻ՚٭֓*򮧥OLɱ-[54t/ڷml[:@hwDfgvmY#*j9yy}:9|x<z:99{:9:u9t7޵9t7ޖ!!Apr崯5D&J$	˔YXĤXV%+8Ʒ+8ƶ?R?R3E
{aH!9[!;;0Q}LW&,@\riJ}ܡc}ܡ`kk0_J>O|EKah~20=Lĵ㧦`61P.w]7Q29ƙ29ƀ8 8	zuۏ.;k%7u0TFx`RB;?uZyytftd6XlA ]Фvښ>Ktu@P2XRsB"i1(]cWNcWN717*	c|8M0,O_EJ.i37E#XE#XYYH44쥒*RfV:\`J
ٔu1Xi]zI]zI@$@#;w݃dT.X3Bңy`FMWM3´CwnunwnuRRlgnhS;s"qN݆ziw%2ajɚK	K;0sfR8Ƿ"eqvOjCbqqZZ+MCsE:CZTFxru6SJQBk٤J<XOH<XOH7KK=	y-KQ1y+gȎ=5#65XjHX   "3DUTfUUDD$mlIP  

88߮;9㎺8y:뮺x׹Ps5T7I$FqNV6]]a0v]W]t&v;#Ś3n715۬>Mu9Ky,c1(x~gvѴ]WWSFX!эӉG#!/u( %xg^*pMHRzӄyNGqVk5m]fmSIr1sym2g\[LM%YsH}gu!Л8R%YmXiZw;UսoPh2MˌSi kg>Jג.uMӕ9Va&_.fNT&tЇa6
N',I))``H|!b|WŃ ʄHEBFB)2hvl
K
zmJiل9چ6UYm$J![Ws]5ѕZ.w3lX&F].2#qm'6_WhɉvbF$C,lxNxTFsz^#
g͆ŚͲ٥b=hkT.sdoKE^UZngӍ 
"/Zkx2f:j].(ZkL3n'bXeIKU]5gwwwٝo[V]W[ugӔ'g3eB"7kiFVR-qYG-P2^d]rw$/[fFfѴD&N+UjLƄjgerhvXs2 +c7xEc6H-GHU*
bQ+Yspf3UUfffUUDI#F* UUUUUU}UUUUUQuqs<<<|w<<sq<<e杍ycC	(LȂ2)TK2zBWE8$JĬK'fZsUTӝV$᫉.ZRЩ*2/"E*g59#*2hJJp\5HÜ0Lmu\gW3zƋ4Xvjq.+F`M[;C:(A`/
laE'|0PFTtfUg+8Kj6Se*UTx"3f9S= 9x#ﮊ4mU*]K9apfs63Yͺ]z/xo,sR()S;19IsRԩ*J "HoiwnƳ$,.ʭqr0;*ıhη)a[S%Mmo{'`ǫ2-u	P!H%K1d
@#:L7"B4l-!
Jci3ݩLѬ2;W}L=3Qf@p7iXQ8tZs:VKFZ/Q
ж+2e7oTKY|_°&IIUdɳtd(Df%%/)|*򯘧1KjՐJQV=cP3BЬ5@ RHl+9
Eo+m,ΎMtS<39:N3
NbD 勔BG,%N%"S0s
#JFK*N'-JԤJ`8#H^6^
կ&-5 `	gU3UffffUfD$mUWUUUUUUWU)q<q]q]q]q<<<<sϜ<o\u燯
j#<HB5pH-mZ9r<#(
hM"J[lCGph)|1Fg
ƿ})SћJ7[%̦mE$FO^'#P;DK3fut ί^Iv6*BmXV6=κ㮹
OօIō(r52j(KkN*
^(]ݨG.7j)QѸ0B*5#5+A.b5'A5sWDq)2Ez6nYs@;EΒ/",G&YFuVe|-qkQrJN$">pmOb2Ԅ!*:ӭAD"EL-[+Z2-[:0t{-rȓjVZcZksx^(&Lz4k9҃F^
Q8c<2#3JX%H%|,3YC%e,K:<!̜J"EIsh-[82%YW)rtHBg2,ԋ5+TI0S)'ROQ=eZʥZXf)D3
Zlը:+Z=׌4Ou%o̲S%6mlێKft\:u;1nnK񔖾mD۪02r4MtO*gϭ{Ce31:8uST+20c:n"	6 P@pfDUfwfvfkeGF*k꾪3;߾﮹9;8ߞ|z8뎹yx}x㮸xkV3\j
nEId](X%24diۆm)ڀ^Fȑt[2A:~~G=e4#u5

4E&Q)o_7V,Y7N),S*1-c2G=:_a)Zq\iID"l1flm,B%o[)K+q\Y_-(
*=3ԧ)NW!}_E1B}tNV-$+hNʆm^yRbzcVڶн8SDّ6^=蟬1*awHoPg>!(e	[ZB%jJ1.i:fLԉЅ&{YtXޣ.\,Ս!
{<)kOLsBŬZD$9RnG𪲘"F]YJċZn)#qK2Gi;OkQ9s^/ fTTeU-˰M9pĨt_7'9&
oVh]KyƞrAIVgSAimNH5[r+j\Nm#l +:ՓAÃuXݡF'Cbzq0FKK1.C*kЖ'KBA,8Qg9 .ssԑ2.G43*Lʕ«nޖEHIfF]wuC
KU@-mZIk5ok9:kV:u[)tIkйČȦI%K{D @kLDDwwwmij98y9箸xy9:y:{yy89m9(iD]VxkgG(c>%WHwMo)ȢrK()I5\Es\Hh.k!uV %ګU˺[Xa~Xt|cr.)LRZym8a/`aO,$P%ayZX&
R_Lg(~4Lhnr[wXϜhֲJXLmstx;uqPK.w*r585.މDΉc{Xkc"bdG.+dEa5iH]](YQC҅&bzߎwl-OznAһHEM]AdAF}r1ÑqV䘹&PۅN:jgbcM+Φc&|J>[+"NTi*u^I3izegy3jH`! ^4[	,
&&v[~K#8Y-k٨ZjMg^X/,S&>/zʽ .F
<.5x}fNco+q0b*_zrC+0m\i䚠^wx[VuNOuH~2
dU&UgVK&ZEF]Hd}x4mTҦc,:{d	} KeY1ndiR]ǂn禷T&``u
xp  f3DDUUfDUCFmlI   


Z0}xyyy}yy8y㎺㍒@{$1>hJMneUUVAa~a(Ȓ#$kR	K̩A0ΥBȉM\sz,ϛ歫ljsh1;\^qFdyc1p8m8%r !5BFk`+a	AG*&E\HO[gXؚT.˳Ml6[*մJQ{-[+ZȦ851)0ʲE4S*UZ7
(1Qndxd1im6cCOAÌg!o gvwit5Ffَ#d$PtUu*)Mku=rtzzCw7rjK8BpjڭR,)40Il&odR&[,\HKb	3UbZVNVUY%Ywz}\p 8
s\d$dj4+"1;UbwN4cA`ҳ0:ѪʳZe}_B!@[rI!t1*Bmfk4wgzֵFQjGxQTO%Bk"m4dͦլ֩Qnn("PEf-K,Άa+
J3g+:1-vmƲm6b2dj9ZkU3h+|f
И	N$M-n*"͏l[ĜEisD%&Gs#Zǂ0ŏa:dLs1sK8}P
-lѷ%qrI-tov pxf3DUfeUUUDI$HJ UURUPUU/o<u\s]u<ם|s<<>oϜ\uu}osǜf]
d~C\s̬0JԭPaD6XPkܽʪjCi գ5!y>ej9ü^www2twIc+TJ")+qWal0cBд+hKgJA6);)
pNK
l &>6N|b)JД%FR4u]T-m* SdlUbMQv+0{H+{6hj8ơԵ(7]8DƼ>chg>Ù¯Nzr(p,3:fuU:YȌ*":SIBPpvoW!6LZl񙘢[<ff(kf3Y'sTygi\һSLʔE<S*aXTUJҮ86H\Vnkiۤ 9c["ta&eX晥h 5cX3
&IbK	McbgSe2fj &slf<bq	T]=5N3@䯫<1#"U֮g޲942)Qd.tQ	
wWuETX#,]WS,YEcl3YbT$͛6kͶb<iQJ`	/X%pJ3ĝ\θ?$C$8"д@"3-+?"OI
n3[53;ŝ^Vj)e?OuB=7.L:4#MD5WUMsV4M0S#Ȋ#e`6L$z
 `]f3UffffUUUY$q-WUUUUUUUUWQs<ss<<q]s<u<=s|wqg˔=~\i{M,̄HD)&H{MN!EETUݛ=*0/+K IJ@?Y"bߝTrW(ɫ`Axe`| bF%Zwcv3TՁ1L01%AS bkoY2߬zɖAѶh\jɈ75u/RN'kM86!1#k+7Y^qe]ݬIc(׵cgڝTIi*=֤n9]mVj?6W,jSU$bF.\7Nju&E4S"J$a)
BQ#Ol`KۭYܮjeG ifknސ!GN"-h[e؝.KIυ1lpq"'1#@Kq [M>hkt@5k^(QhO$KUuWOC
(]ZS
w"6ֻ;aAI؛
Їnje&Qi
(Qa:Bt#&FNխY؝MdYǗs߂8E+\(@!FPPˌ"R% *@H*RTC:Ü;;B<8Hh4^";_Y","Wuk: [طsAt\Wu~ViiVb
-9f.,fsW3M*mJ:*sTٮ]YN96vjoLΓDкh"G(.jI ܅p^y]\D;	"-Yٝ!	2%] Pdf3ffffffimFUVjj6
Þxy箺y;뾸:88y箸:yxxT6..8T6..8mGn4DDG;%pB)NR^"h:T8RS[4?)?)5h~5O(P2HfkJ3s(S=̣E&[RtmjNFjNFd'WB._)>Rae	םFw[a0 z2j8SISIɢ
Hc8cǮp,DR$I6@8YۗͤL8(GZd
ܮ:5Yܮ:5Y_5c64ͩZ*4TjocǗRn՝+).Ø5vJs[MDj&W%r;w-r"gaC{lTLBK3<63z4ⲗR%%]15ipzmf1EKRŘ,ie;0L/0L,hr9c	!2'$رF#;G$&&bKƪ~N2Ι2"~WvMų[++W:t$/)j˲;Ҵ<*9m-wD:t:tC6|'cU*X#PFӽ;VHpc<m##2UqT-lFFf1k17Cw=nz^Yh8pZ4=>5T4$G5[qj+qj({{wWu+lf9s(XI׷F˺2J#d6N
@@|DfwwfmJ۩Vj+J蓮7yx8ߞx{z뮸㮹y8㮹xd~9>d~9>RxIdGȏ{=(<$g
p<U0Ӧc*~i5ưsH:b,W,WxO9<>p|RVUF/T'|HxB)ZBZLaPkQ`mI֫֫UTrKժUe++Iz}>[[ʝF*XG5?uYSA)SA)Ͳt:ӣESuUlZ-SIZKraS}V$l?l54#l54"רn^E#F:bg䶙h,)VkR,zӕKJ㖂
%%Eq
A)iֶm\Sn%SDirRëf]LET-RXb-RXHx/!ྦྷӯUHd$p6tZ]`AχLeԦ^S-C-DtQS;ǶGlo.)[6\reJ4\&]jNe6Z0fY_jfY_jEC AU Xގ@gN)ԣ4ufX(Wy5vx'mfw9K.9K.%LOOuY&':-n]̑f-

R4$
JǶn_JǶnt"eș3%JdPy82)ۖ\H&tiauyAG;AG;hRU0͙6&rfuPS}VT2E1$^؃iR%N8PR$  w3DDDDUDD3nFmm"I   

Z0ȟr'yy88瞺8yyyl'd(!aba?Ql4ͱZV'FM3N9I"^!)x:$7ƙ!me֭uY[ĭ+%ej.˳xp8~I$YDTrMV<f}34ۥBn&`EbQ%al[HIZV(ڒ\8i8pƶfO#42|	'ƲCF4bGQԭ+XVXUPqvpH4MR)(PFdJ~ż׳-潎:K-KWeHR qWF&t݈\Tq˕$XWjsMPSjuMףQgzet\76sgBз=%.J%ԲZ=TM&XJTP"@a:Id&IaGڕGjZ)L.*uHv;1br7!Ff7:3@?HddFEwIcafil[e1aFihN'bJYѣB+}XՓcVLT
ҢE
V5	iOE"q4V(!	kE&LcZ,)6I=Y$=d&B84$-d%	EDTuvj5fm6#Nq޽!KZZYG)יyx]"H$3AeiZbq2ioTJ&N;R$q!0Ķ(n[QmDz5^V uvYe(+JWJ$Yr[իZr<xB pj3DUUfUUDDFm  WU-UUUUUUWц6k<]s<u<<<]s<<|sX^zX^z)+rouDTBZ͌LwqLֵ<<+KJTi`M77y:Rm&d!#6m\plWUeYp\C۩T	Y̤ڋY̤Sҥ8*ҍ(it)Br"kND5%2h	ڕrЅrЀ
YcF1{Z8Ba8Ne<OƲG#N U7DJܛAgFAgF#	I8ДA&Z=U
άx#Ns߷'%g."BpBp.I~O$_a鞚5F͜6jVձxMazLQHqئѐoƎZoƎIiYVIj*MDG}g2d<=X?:kn7FsfL
wF\g;wF\g;p̂-7"ӉxX6sg4*B.
YeAl[slil]54Rk&PEPE
(Q&ҍ%l:E]=odmخOlik	MRII@׬*
w*
w6m#fM2*f[(AglvgelVifyQ`0\pɮ^]]cOEf4]1385\JT*jfJTȲaYdZ.\imD ޲-:oZ?R-:oZ?[/0rj7VugHQ8YR(4֥s3dI4z6E~ `]3UfUffUDUIiWU}U}UUUUUQ$om[sw|q}|w]q<<ss<':I':ItwP82nd[UWBÛ&;(̖jߛ4ߛ4ĢA5El5\L@9]6ɹ+]WTH5Lķ4e]D(fʶ'ʶ'2֪\eT
,Vj*UhIܵ-JԤIBe<k)tA{CΗ۵CΗ|Zj$4+Bˬlv>{Jmʌh.sػ)|:$	<	=Sq ALέR2%jb+KTr7"+#"kqԪYAsG PAsG Q3DBdfUQ39V/\JtQ5ڶnC)Mu^w#rn#rn!h2fC4d̥*SMm[6\<{V1FSJȺve`Jͣ8XX|)xR-#=C8Ì*8TiLsյdd-vq.q.Z;aa"-5OT-;CIb%4L.B+ʼX_ {ZUK։UKמn=bKW6lN9 e`#H
3Q:Xii&Da@LhVl.ˌ_HSf;㲻+p6Aݰ9bxbeT0FlfQΣ+xr7kTUvUI3GcqhPKDUUfffUY,TF*ꪪO?9y玸zyy<㮹zx|y箹80Eq0Eq,`K"%T"ԌH!CصSJm
9!Le?ESoWCm&(m&(ʊ]2W}
0{*U}"EDʥH3H&%/%/9$x	'Vn2W%ujjԔs(Lr	|q~Z8ۊ 2iY/]/]B#ŭPk`M+2ZͬHёskA !EF(gtqk$tqk&ޭ@Ϧi]و[ҏ*c⬦@*
DDAVSU
Q)Rgvp<oz/Kb;Id1Ln+m\o-\o+~Pߥ"LN]uթkL֔i~υV6%f<9H2&let"5=$=$wDp+@]
{)C$0E2e0icVr2#M|.d28!8!̆ is! \1i7yGMN	%ŔZָ9uBiK?%
ֽ3Q=3QuV)'B=UIЋA*"TĩQȹ&+s',fW#FKp"1Kp"1;b#+؈1cYb);-xY,BHJ]ym+kcݯCaCaht`@'΢E<Ⱥ&tL]r5m07qRxUj@iUffvf[mI"UVꪪQ5y8xy:뮹89箹yx98FiiǷ¶=&9bm.)OvUcPN(a"DFU	F}&VqUoaUoa?5D[NEj׸ͰAZ	ԉYr^ #T)4#T)4L^Wi*bMS}t\א]`cDB`i\خC0sWZ;Z:;ǟx<df&CF䒶@%ulJz4D]/~JqJ7ؿ=pô=p÷ޭĮmiG*UU^Z֬mQ1H]tP󧥭xᄾ~xᄾ~Zvo*NtӌY&1(K^&SiRn9ָeZpm 
R
R㩅dDGL*	eY^8q(B m{8q*d4M6N
OYK:	٦	٦ҿo펕!9H={RK'G.yG%6Q54
vr Mvr Y4$Jɦ"P+9ș9ܧ1q)mIX5\Q(ҕQ(ҕ/c
{
:O&,<4lzΛacCFyryCS<fAhfAhI@EN.kJ]k0yke$J	j\88
o^zue*ORc߫!åHsS[zMR+Jжi$  3DDUDUD33mi$$I   
AMy88㎺yyyy9+}	Z!w'Ĩ[ωPU+HJ<G\-vmvhLnSE):Gs'81]-T-V[wgp*c눸!fhBQܒI%r].Ri JZϚ	kl7pepn -^0Z7+Jñ(,6%HkMUFIG&[׮rB5u{yk7^!Ie̒e[QTa'
RTm
(2
6J%,Z	"{<x%+1n#;3dFP$M'5Z-T>2JPֵtǛU>}+'rw#Phmxk*JY+unMQ(E-$TRgaL3׺qcSc¦Ȑl1ow
en
N4e
@4m*CKZ֨
@+a$IfZ%1b8w.T$SI$Q&iQIkLu( %%q+pB;Q] 66 *.J:#?O9nDb)GʖK-@[ǆLJֹUO%}yol z7v[yz^jmI\ʩL6fRnZjZ鞨oL͛v68,HJ.raqM3
dj5bV7zZֵkvq[ww LA2	[QE4G[j5I$Wy2c&aBkXƵ) pIDDDUUUUD3Fۑ( R}}UU}UUU*jنsϜ<<]ssϜqy<]q<<[!+_8sg3fḕeYYeUYUiǘQDP-T*D)rW`XX$I,}K"/ʂ3o6oF*¨;$9a`[Y$2:/bb1ض1H1بs6*F9=+Ji&ReQUvqF#,#u,^2$dY?#B3,ʴXUT@4JˌL9Kn)WpƑv6G1]$HYCPԲ{''{9s&ib,u]MSVW+B\wDK7=	eJ<YET`>U7lِ+znʮNi,!ʙM"wJ闣up| <cP]ݪ6#b*xۍWqc29Mw*;yq|Dޕ+в+ɽ=S.¡K2?ݷz{1p6I">;;=Q!OyHFuMr-LͶh'%h~_iI|__3-'jpI8j+)ʙr};*U"Hd4@i2,AʭcJ[2ZBrϴvgv]lE6)'IF b].N:LUVLI$) q(uzV''I焳.,˲EKf/R4cj&TMD'
Ѵh;+oԴοf`13DUffUUUU9$mPURUUUUUUUU]g\cq<s<<q]qqq<<[{Y{UK^K^ČH 3g6vͥ~F8xwH}WYVTn<mJ"V^K+^\3u
3u
l[)U\vnib"jfW&"7Qjd6Gd-$+$+$ePsNMUGTvhfVuhiqM-.ye53.Yqχ-q{{֝i4Nb&!z$BeY`!U0 V)uWԀ>@|;@I;@Ifk%dVCgbafZuVUe63cmE/RaDٍ-[*TnRTnRm[]efMk"E֝j%]Z\W	ky$)ֵЅ||-3N]-3NXs5bhV\QIR}'zԍD'+3q<7!n(>Z]5*\j7&S7&S$rGJ$*J)ڝhc8(>ƞÞmk$OjRjRfh0uZ5ja1|WuwZ5%Y)-l΃S%)+Lw֏sww֏sw]UeYM6m{&avgf*!g瀕8OvB{X,`ݖ`g@BG@BGړ8Zxgk4Q`jYkFkjF7-3M5ҝiMtHoHoEqf0aajiڝ%QN!},Qq PůfUUfffVKlHUC*knv( o8z:y9<:x9zy9<T/&f:/&f7~^cF4jA>)S8M!-4\{
gZ/M(E쩷9MCO<CCO<TP2ĶKe139s:*UM.Ij"LIstUpv@1~v:H$:H$9)T眥UUUj=jvmKikQ#Np\LA,cZr/Zr/B"苬ɴJ*Urtul22f~ejX+-5[W3Q :jT pݦ pݦeךA^g{cf6cuR}%PifU5 	o,g::*P;SqD),RXġBt$9tvUuyM8*{.Ivvv|;<ugR-"\`fL̘H.AUW4ܰ(
͊(
͊QJE+θ]Å
-lZ-$ZNI%''TTRS1e\*tm#lkhݼkhݸRceIHXBpVm,"ϋ]pCpCfnEęEęNA}8U332J9s!FBC;R.WDbBXKVlkuvuveRmR+tVsǧO$D#;FM= 	qBkXm5iC2?C2?,\^۝	Кb՝;9s]:4#4斅F!Z{,U!v@?UUfvVmHەV(*ꪨ+1u]Dx899x::߭z9x㮸8<x|0	9.	9.DwЉ(ȼaCj:0*7Rt:quSGV֩*V~~!yWx7?[lV[l
JDBXUi6qD 䲹k-yk-y5אXSk yɜ]+gZn.ʮd@ؖVΕߤ4ߤ
<CdIkꦉb-lTG}mxAiyD	͇D	͝TL;riR`id'XFrk䧒S!PovFѪhS5S5VV纬J9Ϛlh:-qόBRTBX	JDQHeRrqwrqwL??gVu>{{oh
M@W^lU^:1Fr1Fr
Yzl2/˭CsWiZ@V+hf( B
 B
1)bSAYe_tK
*?U4aօ]aTC	
e䴓RtRkm
V]U5Mp=f,"5(ĕiW=ȏTU1`o`of3gayc 1*mBw(LRePԣi2N)8]mЫWЫDE<wgv\^NAPk<T:u]%()=ư  "3DUDDD3m$HI   
ZP5Myzx{8yyy㎺62S062S1P'湳#2:γD-}nZu:Q Ù2zָ\$~쩘~쩘i..Rm6 ^W'IqѴm5Y>}>RW9b0vF#b0vF#1ǐkA+G)ʬ+	ņ$m6hVZ!9uiĺuiHr,\pkdXX0L3%j3I&q%ʕӤ1E""_<3 15dM8<Zm-~rS-ұYF׶JBeN==L@cq
Tfm^7T5Z7Qg4ɮ\˖)9R.>g]N>g]LKy'[J7AP
°^Wd+q8i3f7it
TWt
TWk|<Fѵ[4q2L}yJE*m(zczKUFKUGuKuSd=L9hoۉĵn'j'SVB	TD)TD<2#c1kםynOcdY(ArjU'67~fܹ5(\lZY
BR0S3L&/i2r)ZIÃZQ/YQ/YR-X$[%`c	$e| @>xio#qQϟmh0c pǎ"""DUUUUDI$* PU-UUWUUUUUW<i,نq<s<<s|ws<<q<<)TVV(QtV
J\5DKļ/kt"^i)
fR*bl$ONMONMFK/M^
d#Hab-鞙G?%%IV5>J!KTٱ(0:Xu0:Xv}p%o^HeJKtX5*-)-)N&tQQa,XYujС7kfB(p\gb5
D$
U	I럢럢5$`kFIl*N\%:ͬ<1B/$/Nb=ԲdԲd7Ocn[)͜L
 -5"(uރP" " S-^Zo7#0nir\W<N=C5Hm䬆m䬆Zڒ458Czw{dH`MT&rd^ƤP``ȿ{ ~A:u9[%cCD4%C-,$?¬YF[-	<	<+c/\V&Ho]ٝX!baox]x^*w+%-w"*BiMBiN贰iaϸp*-Bi&J8dVհ4?9hj]-~-~-|'N#Kdɟ>]UΜꇨxr'X2ܷsbZֲU;h!  `""DUUUfU$m U-UWUUUUsvqfo{u]s;q<<<<<<|\˂.e2x4k3_[IqQ+W.EQYLy7gX3]pqɀKL/e,}zmCd6эe 
ޝ%gxUkjQ](/G<J9_MP5&iEDS]5<SM(ʌŻ7b()wM;![`[=k&&bU-Ro{Xepz)rc!ܼl2	^dϯ2
h%kD1.fcջlC變.SR3S2,S+468[M٤UlGz<$m2Sit9yX57w#.FfRe
dZՀ	RR@U[UH$Lu_BgvӶ:b*!B)".+",㯕jҕZ-AcÙN,qZw@`]eRiRDZ54SH[)gjv"As%Wp)8fl6r3eq
m߽z$뎲3#.z\JZ̺o9Z6#Q{(ze;-mZjՍcmؖIdI6"b&"b&5B]-&l̄~=74Q詸nt^L;#LĬYHNɌU9Ýk!IgMöESu7*3y!v]؝IIȜɹXEWuwUnlGHP Pl3f"3DUfkuHҕVyL?yzx:뮷ߞy|yy箹}xyU^,Tbe4}BHe4}BI昷^gρ!vjt[!:Q,
y@YST'Pи0	$!d	Է!d	Է*U.RݻiŲzM"UTNA
s֏6]h.YWʸm;-m;-5d|SV@&c!HLǡ]`:TmyZS$P%BjFGyd -TTî:ըT)	Bn;)DHh¶)hbR`]0TႡ19"Nk$;2vh52{,=L8FX\`h%1+Y5NZv͆C

C
)-Qb)ӵZ(zK*D)	%Y(G\=	]˚\֥W9YW9Y&8o3>%.K&rALfEwfbh9ViXc``M]ޱEmmxcG(X]R!%4gq>W
%W
+>ѺsTe@ʌrǓr..GBdWNH5QkX5Ԏ5Ԕ2h_C&fI5/"zMAU@1|bY!«6ldLec1b1bf8oMeby5Q6dM{!KAS:f6wa77f6``|nUe^qN4ʕOZ5B#@@3D#Ffmd꾫|P@989x}ߞx:yx{߽玸9=xmr9gAgYgAgT&`hLF:a/bٯK!Y:akq4')Nt7?d}Y2%b@]Nab@]Naqnc `BD]\(6g֣GCvp\jtxKsxÀsx2ze,;(&\j5 J\463,LI
ğ8EQQJJ=FþzzsYE\i5rR#粗2S%k!N>ܾ!N>ܴRЩ$+RB%,j)Υ&HW'..c{J2(dq44雿^T7~FjggFd
mp!~/{/W3<y*b-ņv[ޛņv[ t!AB%	3.iT͸zΔ3gMgZA5~5~MoSާr]c5	 jf4LTUJL}1Z+avFWiFWiB*q0Taf+M-[]Ǖi	
Mkح]ᅼloWffAFhU,7^)t2Nnڀ[<BCuiaR9:rtw$ow$oӋ}% Iq{tg=[JDsq͉mQ.nE'dohͷKt	666ߘǒk'1$GMX}i"Z]:2UK.3`kr  "3w  3CDIFI"IP  
8(60wz;yyy8㎺yyE!b:B	7 wyʌTyʌTҎ؛Zڶ P+Ju:#
Ȏ,Z,cZI$
;FE\HFE\H
Km9Z<k܇"ȲS<ѴL 8ru(k[M&j[fT6eN"d"dal%~iPLmʲL(9r %:4{CsǴ79/!!s@59६kV?FE1*_~ U(z|엧%%`z nHư1`.jkٶu:˾}$*/rhv@kEkEے䰁ıUD`6Ԋ@7j˷_>6ҷ|aȿ__֡hhBsVnð#)
(R+jN&×)%';["DKãl:U_U_9N}M&4&?i#YjD~˛-kXqY<APS
0;;m[zVY*Vp,6Q^-h֧l׫6zj3,&3C3,&3RJ쒻'"5/FYeoi߿|۲TJ*ڝPu
y(My(D%&kXUv"nԪV,ojBZШTJ<ݨ3ٵ|W|W19viZ2\Wm[YlբDBs p"3U  ""DDdHRU}UUUU}RUU]uju<<<<<<Ϝs<ss<<|7x=-Kǆv*Eǆv*DȗdKվ|3p+
ĳK1EщdY#(ˀ5;GkwyCCTjҀCTj҉{.ｗzD
6,RѥL)mbsU]ĳQ\E-z}p4}p4-drJ.**8n&kM+-4``nGs7N2eQTQm5VhCm6ږ]Cx8J[EMU E _@E _R:ܩQT
Tr@AS:Q,+PFj(Im2)ZRVdR&jվ{2վ{2WN_X @
oU)ZE$I2NCJB
z`=;wgmwgm|\dD}R(nftJTU6rJ#d#zcXĵr{8ĵr{)2̹z^6hi[u1LT&tF"Pz`;]̜2:L~ne~ne
7D8G#~r;s;Nmٖ۠XyAȫZf㙻
+1K
+1@B!4gYq!.]CшbѪNKF̵23#LS[1]1]"|D	>4%fYaTUEo6:#@9uneXDnaF߶*߶)KlʥyJujWU۹jf|G°"ȴHdl[-3!h`s33UU  "DmG,UU}UUUU-RW\u]sMNSgZֲ<<qu<]s]qs;<{ӄ8CMo_޳kw޳kad7~ɸnfhRT)2L'[	&=1RҵQa [`6b-A$ʽ桥{
桥{Ѵ
iqODSZ<8-r-l.&Kؽlg_&&Jd={;	8kƾGo1ꫀ2"]q!6MAP[5yAP[5x%!%2KFUY(8+A4B$*ǆbC$6HEJ̐d̐dֺuU(\z'PoSZpiF2OT+OT+h9lFrٺ<=^̲ضůӗLμqS\y6ޱ㹻>Tu)6/u6/w0>`	B
*Үt6J^`t%[-.-WUQ,ɬP:݇݇4flh8ف
iB/mU "ʦl4mC<ʾ욲vvb}'9|޷m *9RU2/<~O"2"20Khaô;kضU^j23+%-|GVA)̼k>pv(pv(gq7|	
$PtJGmAhBD3mj)aϘ~mЈԍX2		󤛓7&%,`FkVHR\+]XҮX#q$|: Ppp3DT  96lC꾪y7~@D;y랸㮼8x㮹x}:<㎸DN96u@u@dSd?IuH9q&1"&8(ڄ@Յ*BM8H8H @1 @1+Z╭2S)_ɡ.eA9ЇJDWPa䯙,r)7]jj筈F5%uF5%t*&}C1z-5%Ź}UEs26-D0mj>pj>g
iVUƆZBTA:dZNL(vr%%$4$5
io~4L!wT)^ؕzU͕YA+VlU%_3c_3c'qh/'qh/3s
&+RN
d	)06XwzI{GRFFIvMD,)vMD,':;Z5S
cjDn(V}Rӱd1dv9dv;ootȺsHdo_6VejD <(TW~R7SdHHg&g&daLmm`pK9ws	4*R;o
ƫb(*
y:
y1sBD6]6]{U*Lakc
WnWo@{At7AV0Z]U͹cXWVWV㉹x2㉹x~L"?^&>%e]9۫91:bq@3DDw'-+(+ꪪ랹9瞹mEL3<;ߞy9x뎹箻z{8ߎ'pYw*5[]FgZ[bMYD18,WaU@]`8tOUKzmZ0a)@<'g!>01BU$3[a3wRQC[2e]fAhm_AbkpkLڠc)twTVz4t/j(RPtVqxaEyG.:7 jj)!Lg^ĉj+ݣ{-@Ed$6qe-FK=O`;P]Iʵg,Un!g5-Z(QN"\lTgOl>r;$#HP/HzwJ((]!RB!fmYXd͉cq5dtW+Ӈ<+|N 	Q&BLۑ"5f^Gt@:Ub)[yĳT$U*Xybm67gec5NiӇ+jc6)cv	AS!4h֧:YQ4Hm-+n:2KJ!-̩K<nS =<A	ƽ;#ԁ9K9:ݭ[F# =arg ˫Oe#ڦe5m%xѾ@"SV,Q
3E5z^%wuu}!j+EEaU6+{%v"T('a): 33DUf     [nFۑP__Z瞷y1L瞺㎺yy8뮷~8y8)"jg_ųagg9fmFҹ\"GG]ֵ2mF$jF1yt^yt^m$)Z#4D!@EKRfňrlZ9|gPu ͮ]dY8ƝdY8 deq.KA%
q"U#pI̥se](ѵiQR8\8]K)}u(mۍm[Tj3DRV2IFDd|9QV#$V-$V-fC(P$iV1˂IrS|FV`r)N:dE+[MM/8Lu	\MFQP `
Q"Fn'ko|XŉfhF&"b%{Jx&tmx&tms|vRuYqp9=ۍݢJZ{QȳuX$I9uy9u{)|e/1HqEQE"j6ͳzSG;vkmfΩ:]U7V
 &M1,','r^VA:We٨I$ӵRY,akX͘6iI*W{wTZ"+YZ"+GdT슛ko{FsWP@:RKyyաZDĤDĔj8ҍGer%.cn[q4\n:TZJVa)֝o^4sG܅E5܅E6US#+[9m6)v!*F>e&kmkZ p3D3ff    II#\UPPUUUUUP}UUss<t]d2q]u<<s<}o[|s]u<@.x"EO)w9u|s=ZlcfͶ\H@oYհ~-sOQ#
X5J.]'eh啢Ri?]p9J)qb4.IPed$Zmfŵk<1_TѬFf{m EHqөm(B!VJg͏!.{%K)|`G
*NU"ڤX9kZϋuۯ)7kļ<yUWqyFUx`M7؀to!?MiZhbFtKuF&!j5fVnC0x^.3ׇf\gLakk[Zńtӊ o
ݷ}2{S>Sڧ	FN:-;藂m/r@YHh)լZ41Hf%*Lf8YF31qS8%K)~gyem(Yg:,D%(dЮB*.v(]l؅5q;5	k0/vҕ:bDvqcdMG_\6.γ]1HHd'Od2Tpa|o2LE7iYJݻ&+kU&aR$E=yL
qTĭW)\}Ŕ19bg#,L)LcQ9$X=UgxGw(*7v{sv{\^afM)xBo 9jՉԃ)F75îRhj `33DffD̙   PUUPPUUUUUUWW>qϜs<<.dg)Ċq<s<]u]oq=sq
P;ӽ@dɦ Af(NlP>
R^)	/6E6DktmpXu^#J8My&hI#G
f5gCqTr9m6bԌbԍЂ^Ɯ;\oV|J$kƽ-ۥU^гڶNj
	<D"~4-'*T=Yؓ	SmlPqMjJ>*U|2)eSoyH7$O))u5/AS&D^>:oBOR(U/_L6mmiu)R  "$2)eLFҽr鮛(ҍ':NuZ2=Pz.'ZNuѺ$[$[&pfZh%y{K:9ß],Y6,.
'El"L
L
t6q_f*qbnnUh᜚F9O\]qmn'| NƚM}uuɱZm.gVޕrzJLZ1c3bE҈Z4uH(h
SkBl"[j}YMP<ЫΌQ+SnD5j-ÔFktddr	RB
CFjX	NF`]BS%wKkN9u
D
D.,*5#EFy)zXPJt6li|(8q[ ?|؟x,!k P?DDDfwwD  -9Ꚗꪨ*;yy
JCRzc{߽޹9y~x뎹箹t(nhΫVnx]}yv>]aD@AD@al۰"smI=J#9bol3|xF7=F8!7\FP7\FPz_{X!"LgkϽD%TnٓY=zLvSV0ՓtSS1ISS1Iϙs5E7&}djl\2[aV¨F9ξoCh%EE[EE[I
 h- ]u^K۫HeO&s|7DM
iebXwQ}FwQ}Nuo߽S)u>ܘQ54=(-[^bM^[ں~^{bzz(CBRuGrI"&܉] A:f
Rf9r9N}f-QV-QWׂ<+'!]L杁;Au3g3@Vղ4<a_7&7&@񾢘\ïk\Vz
VNn<FgGX±]cb@/&Iz;z/X4ACpL*fvz<͙\[2d=QB	tL`V5Ull-l->wK
vIRHmctzDg=2nd꽺mby99&zڔ(p)^yH*:uR[Ԗۭ2~
Pbu''0
