package Bencher::Scenario::Serializers;

use 5.010001;
use strict;
use utf8;
use warnings;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2021-09-07'; # DATE
our $DIST = 'Bencher-Scenario-Serializers'; # DIST
our $VERSION = '0.161'; # VERSION

our $scenario = {
    summary => 'Benchmark Perl data serialization modules',
    participants => [
        {
            tags => ['json', 'serialize'],
            module => 'JSON::PP',
            function => 'encode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::PP',
            function => 'decode_json',
            code_template => 'state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::Tiny',
            function => 'encode_json',
            code_template => 'JSON::Tiny::encode_json(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::Tiny',
            function => 'decode_json',
            code_template => 'JSON::Tiny::decode_json(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'Cpanel::JSON::XS',
            function => 'encode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'Cpanel::JSON::XS',
            function => 'decode_json',
            code_template => 'state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            module => 'JSON::MaybeXS',
            function => 'encode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::MaybeXS',
            function => 'decode_json',
            code_template => 'state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Decode::Regexp::from_json(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            module => 'JSON::Parser::Regexp',
            code_template => 'JSON::Parser::Regexp->new->decode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'PERLANCAR::JSON::Match::match_json(<data>)',
            include_by_default => 0,
        },
        {
            tags => ['json', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'JSON::Decode::Marpa::from_json(<data>)',
        },
        {
            name => 'Pegex::JSON',
            tags => ['json', 'deserialize'],
            module => 'Pegex::JSON',
            code_template => 'state $obj = Pegex::JSON->new; $obj->load(<data>);',
        },
        {
            tags => ['json', 'serialize'],
            fcall_template => 'JSON::Create::create_json(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'JSON::Parse::parse_json(<data>)',
        },
        {
            tags => ['json', 'serialize'],
            fcall_template => 'MarpaX::ESLIF::ECMA404->encode(<data>)',
        },
        {
            tags => ['json', 'deserialize'],
            fcall_template => 'MarpaX::ESLIF::ECMA404->decode(<data>)',
        },

        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Old::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Old::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::Syck::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::Syck::Load(<data>)',
        },
        {
            tags => ['yaml', 'serialize'],
            fcall_template => 'YAML::XS::Dump(<data>)',
        },
        {
            tags => ['yaml', 'deserialize'],
            fcall_template => 'YAML::XS::Load(<data>)',
        },

        {
            tags => ['binary', 'storable', 'serialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::freeze(<data>)',
        },
        {
            tags => ['binary', 'storable', 'deserialize', 'cant_handle_scalar'],
            fcall_template => 'Storable::thaw(<data>)',
        },

        {
            tags => ['binary', 'sereal', 'serialize'],
            fcall_template => 'Sereal::encode_sereal(<data>)',
        },
        {
            tags => ['binary', 'sereal', 'deserialize'],
            fcall_template => 'Sereal::decode_sereal(<data>)',
        },

        {
            name => 'Data::MessagePack::pack',
            tags => ['binary', 'msgpack', 'serialize'],
            module => 'Data::MessagePack',
            function => 'pack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->pack(<data>)',
        },
        {
            name => 'Data::MessagePack::unpack',
            tags => ['binary', 'msgpack', 'deserialize'],
            module => 'Data::MessagePack',
            function => 'unpack',
            code_template => 'state $obj = Data::MessagePack->new; $obj->unpack(<data>)',
        },

        {
            name => 'eval()',
            tags => ['perl', 'deserialize'],
            code_template => 'eval(<data>)',
        },
        {
            name => 'Data::Undump',
            tags => ['perl', 'deserialize'],
            fcall_template => 'Data::Undump::undump(<data>)',
        },
        {
            name => 'Data::Undump::PPI',
            tags => ['perl', 'deserialize'],
            fcall_template => 'Data::Undump::PPI::Undump(<data>)',
        },

    ],

    # XXX: add more datasets (larger data, etc)
    datasets => [
        {
            name => 'undef',
            summary => 'undef',
            args => {data=>undef},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A non-Unicode string 1024 characters/bytes long',
            args => {data=>'a' x 1024},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'str1k',
            summary => 'A Unicode string 1024 characters (3072-bytes) long',
            args => {data=>'我爱你爱你一辈子' x 128},
            tags => ['serialize', 'unicode'],
            include_participant_tags => ['serialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>[1..10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_100',
            summary => 'A 100-element array containing ints',
            args => {data=>[1..100]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>[1..1000]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_str1k_10',
            summary => 'A 10-element array containing 1024-characters/bytes-long non-Unicode strings',
            args => {data=>[('a' x 1024) x 10]},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'array_ustr1k_10',
            summary => 'A 10-element array containing 1024-characters-long (3072-bytes long) Unicode strings',
            args => {data=>[('我爱你爱你一辈子' x 128) x 10]},
            tags => ['serialize', 'json'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'hash_int_10',
            summary => 'A 10-key hash {1=>0, ..., 10=>0}',
            args => {data=>{map {$_=>0} 1..10}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_100',
            summary => 'A 100-key hash {1=>0, ..., 100=>0}',
            args => {data=>{map {$_=>0} 1..100}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },
        {
            name => 'hash_int_1000',
            summary => 'A 1000-key hash {1=>0, ..., 1000=>0}',
            args => {data=>{map {$_=>0} 1..1000}},
            tags => ['serialize'],
            include_participant_tags => ['serialize'],
        },

        {
            name => 'json:null',
            summary => 'null',
            args => {data=>'null'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:num',
            summary => 'A single number (-1.23)',
            args => {data=>-1.23},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },
        {
            name => 'json:str1k',
            summary => 'A non-Unicode (ASCII) string 1024-characters/bytes long',
            args => {data=>'"' . ('a' x 1024) . '"'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
            exclude_participant_tags => ['cant_handle_scalar'],
        },

        {
            name => 'json:array_int_10',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..10).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_100',
            summary => 'A 10-element array containing ints',
            args => {data=>'['.join(',',1..100).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_int_1000',
            summary => 'A 1000-element array containing ints',
            args => {data=>'['.join(',',1..1000).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:array_str1k_10',
            summary => 'A 10-element array containing 1024-characters/bytes-long non-Unicode strings',
            args => {data=>'['.join(',',(('"'.('a' x 1024).'"') x 10)).']'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },

        {
            name => 'json:hash_int_10',
            summary => 'A 10-key hash {"1":0, ..., "10":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..10).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_100',
            summary => 'A 100-key hash {"1":0, ..., "100":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..100).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },
        {
            name => 'json:hash_int_1000',
            summary => 'A 1000-key hash {"1":0, ..., "1000":0}',
            args => {data=>'{'.join(',', map {qq("$_":0)} 1..1000).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['json & deserialize'],
        },

        {
            name => 'sereal:hash_int_100',
            summary => 'A 100-key hash {1=>0, ..., 100=>0}',
            args => {data=>"=\xF3rl\3\0(*db26\0b63\0b95\0b35\0b13\0b41\0b33\0b73\0b84\0b85\0b94\0b24\0b12\0b72\0b99\0b71\0a6\0b64\0b70\0b50\0b83\0b68\0a8\0b15\0a9\0a5\0b67\0b25\0b10\0a4\0b56\0b89\0b16\0b90\0b66\0b59\0b29\0b54\0b44\0b27\0b77\0b81\0b32\0b37\0b74\0b65\0b36\0b11\0b18\0b86\0a7\0b17\0b21\0b14\0b28\0b47\0b20\0b76\0b98\0b40\0b91\0b75\0b97\0b31\0b55\0b80\0b19\0b92\0b82\0b43\0b30\0b78\0b57\0b38\0b23\0a3\0b69\0b88\0b61\0b51\0b39\0b42\0b58\0b93\0a2\0b62\0a1\0b79\0b34\0b45\0b87\0c100\0b96\0b22\0b49\0b60\0b52\0b53\0b46\0b48\0"},
            tags => ['deserialize'],
            include_participant_tags => ['sereal & deserialize'],
        },

        {
            name => 'perl:hash_int_100',
            summary => 'A 100-key hash {1=>0, ..., 100=>0}',
            args => {data=>'{'.join(',', map {qq($_=>0)} 1..100).'}'},
            tags => ['deserialize'],
            include_participant_tags => ['perl & deserialize'],
        },
    ],
};

1;
# ABSTRACT: Benchmark Perl data serialization modules

__END__

=pod

=encoding UTF-8

=head1 NAME

Bencher::Scenario::Serializers - Benchmark Perl data serialization modules

=head1 VERSION

This document describes version 0.161 of Bencher::Scenario::Serializers (from Perl distribution Bencher-Scenario-Serializers), released on 2021-09-07.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher -m Serializers

To run module startup overhead benchmark:

 % bencher --module-startup -m Serializers

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 DESCRIPTION

Packaging a benchmark script as a Bencher scenario makes it convenient to include/exclude/add participants/datasets (either via CLI or Perl code), send the result to a central repository, among others . See L<Bencher> and L<bencher> (CLI) for more details.

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<JSON::PP> 4.06

L<JSON::Tiny> 0.58

L<JSON::XS> 4.03

L<Cpanel::JSON::XS> 4.26

L<JSON::MaybeXS> 1.004003

L<JSON::Decode::Regexp> 0.101

L<JSON::Parser::Regexp> 0.18

L<JSON::Decode::Marpa> 0.02

L<Pegex::JSON> 0.31

L<JSON::Create> 0.34

L<JSON::Parse> 0.61

L<MarpaX::ESLIF::ECMA404> 0.014

L<YAML> 1.30

L<YAML::Old> 1.23

L<YAML::Syck> 1.34

L<YAML::XS> 0.83

L<Storable> 3.23

L<Sereal> 4.018

L<Data::MessagePack> 1.01

L<Data::Undump> 0.15

L<Data::Undump::PPI> 0.06

=head1 BENCHMARK PARTICIPANTS

=over

=item * JSON::PP::encode_json (perl_code) [json, serialize]

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->encode(<data>)



=item * JSON::PP::decode_json (perl_code) [json, deserialize]

Code template:

 state $json = JSON::PP->new->allow_nonref; $json->decode(<data>)



=item * JSON::Tiny::encode_json (perl_code) [json, serialize]

Code template:

 JSON::Tiny::encode_json(<data>)



=item * JSON::Tiny::decode_json (perl_code) [json, deserialize]

Code template:

 JSON::Tiny::decode_json(<data>)



=item * JSON::XS::encode_json (perl_code) [json, serialize]

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * JSON::XS::decode_json (perl_code) [json, deserialize]

Code template:

 state $json = JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * Cpanel::JSON::XS::encode_json (perl_code) [json, serialize]

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->encode(<data>)



=item * Cpanel::JSON::XS::decode_json (perl_code) [json, deserialize]

Code template:

 state $json = Cpanel::JSON::XS->new->allow_nonref; $json->decode(<data>)



=item * JSON::MaybeXS::encode_json (perl_code) [json, serialize]

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->encode(<data>)



=item * JSON::MaybeXS::decode_json (perl_code) [json, deserialize]

Code template:

 state $json = JSON::MaybeXS->new(allow_nonref=>1); $json->decode(<data>)



=item * JSON::Decode::Regexp::from_json (perl_code) [json, deserialize]

Function call template:

 JSON::Decode::Regexp::from_json(<data>)



=item * JSON::Parser::Regexp (perl_code) [json, deserialize]

Code template:

 JSON::Parser::Regexp->new->decode(<data>)



=item * JSON::Decode::Marpa::from_json (perl_code) [json, deserialize, cant_handle_scalar]

Function call template:

 JSON::Decode::Marpa::from_json(<data>)



=item * Pegex::JSON (perl_code) [json, deserialize]

Code template:

 state $obj = Pegex::JSON->new; $obj->load(<data>);



=item * JSON::Create::create_json (perl_code) [json, serialize]

Function call template:

 JSON::Create::create_json(<data>)



=item * JSON::Parse::parse_json (perl_code) [json, deserialize]

Function call template:

 JSON::Parse::parse_json(<data>)



=item * MarpaX::ESLIF::ECMA404::encode (perl_code) [json, serialize]

Function call template:

 MarpaX::ESLIF::ECMA404->encode(<data>)



=item * MarpaX::ESLIF::ECMA404::decode (perl_code) [json, deserialize]

Function call template:

 MarpaX::ESLIF::ECMA404->decode(<data>)



=item * YAML::Dump (perl_code) [yaml, serialize]

Function call template:

 YAML::Dump(<data>)



=item * YAML::Load (perl_code) [yaml, deserialize]

Function call template:

 YAML::Load(<data>)



=item * YAML::Old::Dump (perl_code) [yaml, serialize]

Function call template:

 YAML::Old::Dump(<data>)



=item * YAML::Old::Load (perl_code) [yaml, deserialize]

Function call template:

 YAML::Old::Load(<data>)



=item * YAML::Syck::Dump (perl_code) [yaml, serialize]

Function call template:

 YAML::Syck::Dump(<data>)



=item * YAML::Syck::Load (perl_code) [yaml, deserialize]

Function call template:

 YAML::Syck::Load(<data>)



=item * YAML::XS::Dump (perl_code) [yaml, serialize]

Function call template:

 YAML::XS::Dump(<data>)



=item * YAML::XS::Load (perl_code) [yaml, deserialize]

Function call template:

 YAML::XS::Load(<data>)



=item * Storable::freeze (perl_code) [binary, storable, serialize, cant_handle_scalar]

Function call template:

 Storable::freeze(<data>)



=item * Storable::thaw (perl_code) [binary, storable, deserialize, cant_handle_scalar]

Function call template:

 Storable::thaw(<data>)



=item * Sereal::encode_sereal (perl_code) [binary, sereal, serialize]

Function call template:

 Sereal::encode_sereal(<data>)



=item * Sereal::decode_sereal (perl_code) [binary, sereal, deserialize]

Function call template:

 Sereal::decode_sereal(<data>)



=item * Data::MessagePack::pack (perl_code) [binary, msgpack, serialize]

Code template:

 state $obj = Data::MessagePack->new; $obj->pack(<data>)



=item * Data::MessagePack::unpack (perl_code) [binary, msgpack, deserialize]

Code template:

 state $obj = Data::MessagePack->new; $obj->unpack(<data>)



=item * eval() (perl_code) [perl, deserialize]

Code template:

 eval(<data>)



=item * Data::Undump (perl_code) [perl, deserialize]

Function call template:

 Data::Undump::undump(<data>)



=item * Data::Undump::PPI (perl_code) [perl, deserialize]

Function call template:

 Data::Undump::PPI::Undump(<data>)



=back

=head1 BENCHMARK DATASETS

=over

=item * undef [serialize]

undef.

=item * num [serialize]

A single number (-1.23).

=item * str1k [serialize]

A non-Unicode string 1024 charactersE<sol>bytes long.

=item * str1k [serialize, unicode]

A Unicode string 1024 characters (3072-bytes) long.

=item * array_int_10 [serialize]

A 10-element array containing ints.

=item * array_int_100 [serialize]

A 100-element array containing ints.

=item * array_int_1000 [serialize]

A 1000-element array containing ints.

=item * array_str1k_10 [serialize]

A 10-element array containing 1024-charactersE<sol>bytes-long non-Unicode strings.

=item * array_ustr1k_10 [serialize, json]

A 10-element array containing 1024-characters-long (3072-bytes long) Unicode strings.

=item * hash_int_10 [serialize]

A 10-key hash {1=E<gt>0, ..., 10=E<gt>0}.

=item * hash_int_100 [serialize]

A 100-key hash {1=E<gt>0, ..., 100=E<gt>0}.

=item * hash_int_1000 [serialize]

A 1000-key hash {1=E<gt>0, ..., 1000=E<gt>0}.

=item * json:null [deserialize]

null.

=item * json:num [deserialize]

A single number (-1.23).

=item * json:str1k [deserialize]

A non-Unicode (ASCII) string 1024-charactersE<sol>bytes long.

=item * json:array_int_10 [deserialize]

A 10-element array containing ints.

=item * json:array_int_100 [deserialize]

A 10-element array containing ints.

=item * json:array_int_1000 [deserialize]

A 1000-element array containing ints.

=item * json:array_str1k_10 [deserialize]

A 10-element array containing 1024-charactersE<sol>bytes-long non-Unicode strings.

=item * json:hash_int_10 [deserialize]

A 10-key hash {"1":0, ..., "10":0}.

=item * json:hash_int_100 [deserialize]

A 100-key hash {"1":0, ..., "100":0}.

=item * json:hash_int_1000 [deserialize]

A 1000-key hash {"1":0, ..., "1000":0}.

=item * sereal:hash_int_100 [deserialize]

A 100-key hash {1=E<gt>0, ..., 100=E<gt>0}.

=item * perl:hash_int_100 [deserialize]

A 100-key hash {1=E<gt>0, ..., 100=E<gt>0}.

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.34.0 >>, CPU: I<< Intel(R) Core(TM) i5-7200U CPU @ 2.50GHz (2 cores) >>, OS: I<< GNU/Linux Ubuntu version 20.04 >>, OS kernel: I<< Linux version 5.3.0-64-generic >>.

Benchmark with C<< bencher -m Serializers --include-participant-tags serialize >>:

 #table1#
 {dataset=>"array_int_10"}
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | p_tags                                          | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | YAML::Dump                     | yaml, serialize                                 |      4100 |   240     |                 0.00% |             44100.70% | 6.9e-07 |      20 |
 | YAML::Old::Dump                | yaml, serialize                                 |      4100 |   240     |                 0.62% |             43827.69% | 6.4e-07 |      20 |
 | JSON::PP::encode_json          | json, serialize                                 |     71000 |    14     |              1614.24% |              2478.44% | 8.5e-08 |      21 |
 | MarpaX::ESLIF::ECMA404::encode | json, serialize                                 |     76000 |    13     |              1753.36% |              2284.90% |   6e-08 |      20 |
 | JSON::Tiny::encode_json        | json, serialize                                 |     93300 |    10.7   |              2166.16% |              1850.47% | 3.3e-09 |      20 |
 | YAML::Syck::Dump               | yaml, serialize                                 |    100000 |     9.8   |              2375.98% |              1685.18% | 1.6e-08 |      21 |
 | YAML::XS::Dump                 | yaml, serialize                                 |    148460 |     6.736 |              3507.55% |              1125.23% | 2.2e-11 |      20 |
 | Storable::freeze               | binary, storable, serialize, cant_handle_scalar |    331790 |     3.014 |              7962.66% |               448.21% | 2.6e-11 |      20 |
 | JSON::MaybeXS::encode_json     | json, serialize                                 |   1300000 |     0.78  |             30930.46% |                42.44% | 5.9e-09 |      21 |
 | Sereal::encode_sereal          | binary, sereal, serialize                       |   1400000 |     0.7   |             34807.79% |                26.62% | 8.3e-10 |      20 |
 | Cpanel::JSON::XS::encode_json  | json, serialize                                 |   1500000 |     0.69  |             35297.75% |                24.87% | 7.6e-10 |      24 |
 | JSON::XS::encode_json          | json, serialize                                 |   1700000 |     0.6   |             40118.51% |                 9.90% | 8.3e-10 |      20 |
 | JSON::Create::create_json      | json, serialize                                 |   1700000 |     0.6   |             40553.09% |                 8.73% | 3.2e-09 |      25 |
 | Data::MessagePack::pack        | binary, msgpack, serialize                      |   1800000 |     0.55  |             44100.70% |                 0.00% | 8.3e-10 |      20 |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                            Rate  Y:D yaml, serialize  YO:D yaml, serialize  JP:e_j json, serialize  MEE:e json, serialize  JT:e_j json, serialize  YS:D yaml, serialize  YX:D yaml, serialize  S:f binary, storable, serialize, cant_handle_scalar  JM:e_j json, serialize  S:e_s binary, sereal, serialize  CJX:e_j json, serialize  JX:e_j json, serialize  JC:c_j json, serialize  DM:p binary, msgpack, serialize 
  Y:D yaml, serialize                                     4100/s                   --                    0%                    -94%                   -94%                    -95%                  -95%                  -97%                                                 -98%                    -99%                             -99%                     -99%                    -99%                    -99%                             -99% 
  YO:D yaml, serialize                                    4100/s                   0%                    --                    -94%                   -94%                    -95%                  -95%                  -97%                                                 -98%                    -99%                             -99%                     -99%                    -99%                    -99%                             -99% 
  JP:e_j json, serialize                                 71000/s                1614%                 1614%                      --                    -7%                    -23%                  -29%                  -51%                                                 -78%                    -94%                             -95%                     -95%                    -95%                    -95%                             -96% 
  MEE:e json, serialize                                  76000/s                1746%                 1746%                      7%                     --                    -17%                  -24%                  -48%                                                 -76%                    -94%                             -94%                     -94%                    -95%                    -95%                             -95% 
  JT:e_j json, serialize                                 93300/s                2142%                 2142%                     30%                    21%                      --                   -8%                  -37%                                                 -71%                    -92%                             -93%                     -93%                    -94%                    -94%                             -94% 
  YS:D yaml, serialize                                  100000/s                2348%                 2348%                     42%                    32%                      9%                    --                  -31%                                                 -69%                    -92%                             -92%                     -92%                    -93%                    -93%                             -94% 
  YX:D yaml, serialize                                  148460/s                3462%                 3462%                    107%                    92%                     58%                   45%                    --                                                 -55%                    -88%                             -89%                     -89%                    -91%                    -91%                             -91% 
  S:f binary, storable, serialize, cant_handle_scalar   331790/s                7862%                 7862%                    364%                   331%                    255%                  225%                  123%                                                   --                    -74%                             -76%                     -77%                    -80%                    -80%                             -81% 
  JM:e_j json, serialize                               1300000/s               30669%                30669%                   1694%                  1566%                   1271%                 1156%                  763%                                                 286%                      --                             -10%                     -11%                    -23%                    -23%                             -29% 
  S:e_s binary, sereal, serialize                      1400000/s               34185%                34185%                   1900%                  1757%                   1428%                 1300%                  862%                                                 330%                     11%                               --                      -1%                    -14%                    -14%                             -21% 
  CJX:e_j json, serialize                              1500000/s               34682%                34682%                   1928%                  1784%                   1450%                 1320%                  876%                                                 336%                     13%                               1%                       --                    -13%                    -13%                             -20% 
  JX:e_j json, serialize                               1700000/s               39900%                39900%                   2233%                  2066%                   1683%                 1533%                 1022%                                                 402%                     30%                              16%                      14%                      --                      0%                              -8% 
  JC:c_j json, serialize                               1700000/s               39900%                39900%                   2233%                  2066%                   1683%                 1533%                 1022%                                                 402%                     30%                              16%                      14%                      0%                      --                              -8% 
  DM:p binary, msgpack, serialize                      1800000/s               43536%                43536%                   2445%                  2263%                   1845%                 1681%                 1124%                                                 447%                     41%                              27%                      25%                      9%                      9%                               -- 
 
 Legends:
   CJX:e_j json, serialize: p_tags=json, serialize participant=Cpanel::JSON::XS::encode_json
   DM:p binary, msgpack, serialize: p_tags=binary, msgpack, serialize participant=Data::MessagePack::pack
   JC:c_j json, serialize: p_tags=json, serialize participant=JSON::Create::create_json
   JM:e_j json, serialize: p_tags=json, serialize participant=JSON::MaybeXS::encode_json
   JP:e_j json, serialize: p_tags=json, serialize participant=JSON::PP::encode_json
   JT:e_j json, serialize: p_tags=json, serialize participant=JSON::Tiny::encode_json
   JX:e_j json, serialize: p_tags=json, serialize participant=JSON::XS::encode_json
   MEE:e json, serialize: p_tags=json, serialize participant=MarpaX::ESLIF::ECMA404::encode
   S:e_s binary, sereal, serialize: p_tags=binary, sereal, serialize participant=Sereal::encode_sereal
   S:f binary, storable, serialize, cant_handle_scalar: p_tags=binary, storable, serialize, cant_handle_scalar participant=Storable::freeze
   Y:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Dump
   YO:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Old::Dump
   YS:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Syck::Dump
   YX:D yaml, serialize: p_tags=yaml, serialize participant=YAML::XS::Dump

 #table2#
 {dataset=>"array_int_100"}
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | p_tags                                          | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | YAML::Old::Dump                | yaml, serialize                                 |     540   | 1.9       |                 0.00% |             58661.01% |   4e-06 |      22 |
 | YAML::Dump                     | yaml, serialize                                 |     540   | 1.9       |                 0.24% |             58521.30% | 5.6e-06 |      20 |
 | JSON::PP::encode_json          | json, serialize                                 |    9810   | 0.102     |              1727.38% |              3115.58% | 5.3e-08 |      20 |
 | JSON::Tiny::encode_json        | json, serialize                                 |   12946.7 | 0.07724   |              2310.76% |              2337.44% | 2.1e-11 |      20 |
 | YAML::Syck::Dump               | yaml, serialize                                 |   16000   | 0.062     |              2919.66% |              1845.95% | 2.4e-07 |      20 |
 | YAML::XS::Dump                 | yaml, serialize                                 |   21369.1 | 0.0467966 |              3879.08% |              1376.75% | 2.5e-11 |      20 |
 | MarpaX::ESLIF::ECMA404::encode | json, serialize                                 |   21000   | 0.047     |              3896.41% |              1370.34% | 2.8e-07 |      20 |
 | Storable::freeze               | binary, storable, serialize, cant_handle_scalar |  144520   | 0.0069194 |             26810.79% |               118.35% | 2.6e-11 |      20 |
 | Cpanel::JSON::XS::encode_json  | json, serialize                                 |  170000   | 0.0059    |             31719.94% |                84.67% | 2.7e-08 |      20 |
 | JSON::MaybeXS::encode_json     | json, serialize                                 |  180000   | 0.0055    |             33509.95% |                74.83% | 3.5e-08 |      20 |
 | JSON::XS::encode_json          | json, serialize                                 |  247140   | 0.0040463 |             45919.66% |                27.69% | 2.2e-11 |      20 |
 | Data::MessagePack::pack        | binary, msgpack, serialize                      |  303520   | 0.0032946 |             56418.52% |                 3.97% | 2.7e-11 |      21 |
 | Sereal::encode_sereal          | binary, sereal, serialize                       |  307000   | 0.00325   |             57132.72% |                 2.67% | 1.6e-09 |      22 |
 | JSON::Create::create_json      | json, serialize                                 |  315570   | 0.0031689 |             58661.01% |                 0.00% | 2.4e-11 |      20 |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                            Rate  YO:D yaml, serialize  Y:D yaml, serialize  JP:e_j json, serialize  JT:e_j json, serialize  YS:D yaml, serialize  MEE:e json, serialize  YX:D yaml, serialize  S:f binary, storable, serialize, cant_handle_scalar  CJX:e_j json, serialize  JM:e_j json, serialize  JX:e_j json, serialize  DM:p binary, msgpack, serialize  S:e_s binary, sereal, serialize  JC:c_j json, serialize 
  YO:D yaml, serialize                                     540/s                    --                   0%                    -94%                    -95%                  -96%                   -97%                  -97%                                                 -99%                     -99%                    -99%                    -99%                             -99%                             -99%                    -99% 
  Y:D yaml, serialize                                      540/s                    0%                   --                    -94%                    -95%                  -96%                   -97%                  -97%                                                 -99%                     -99%                    -99%                    -99%                             -99%                             -99%                    -99% 
  JP:e_j json, serialize                                  9810/s                 1762%                1762%                      --                    -24%                  -39%                   -53%                  -54%                                                 -93%                     -94%                    -94%                    -96%                             -96%                             -96%                    -96% 
  JT:e_j json, serialize                               12946.7/s                 2359%                2359%                     32%                      --                  -19%                   -39%                  -39%                                                 -91%                     -92%                    -92%                    -94%                             -95%                             -95%                    -95% 
  YS:D yaml, serialize                                   16000/s                 2964%                2964%                     64%                     24%                    --                   -24%                  -24%                                                 -88%                     -90%                    -91%                    -93%                             -94%                             -94%                    -94% 
  MEE:e json, serialize                                  21000/s                 3942%                3942%                    117%                     64%                   31%                     --                    0%                                                 -85%                     -87%                    -88%                    -91%                             -92%                             -93%                    -93% 
  YX:D yaml, serialize                                 21369.1/s                 3960%                3960%                    117%                     65%                   32%                     0%                    --                                                 -85%                     -87%                    -88%                    -91%                             -92%                             -93%                    -93% 
  S:f binary, storable, serialize, cant_handle_scalar   144520/s                27359%               27359%                   1374%                   1016%                  796%                   579%                  576%                                                   --                     -14%                    -20%                    -41%                             -52%                             -53%                    -54% 
  CJX:e_j json, serialize                               170000/s                32103%               32103%                   1628%                   1209%                  950%                   696%                  693%                                                  17%                       --                     -6%                    -31%                             -44%                             -44%                    -46% 
  JM:e_j json, serialize                                180000/s                34445%               34445%                   1754%                   1304%                 1027%                   754%                  750%                                                  25%                       7%                      --                    -26%                             -40%                             -40%                    -42% 
  JX:e_j json, serialize                                247140/s                46856%               46856%                   2420%                   1808%                 1432%                  1061%                 1056%                                                  71%                      45%                     35%                      --                             -18%                             -19%                    -21% 
  DM:p binary, msgpack, serialize                       303520/s                57570%               57570%                   2995%                   2244%                 1781%                  1326%                 1320%                                                 110%                      79%                     66%                     22%                               --                              -1%                     -3% 
  S:e_s binary, sereal, serialize                       307000/s                58361%               58361%                   3038%                   2276%                 1807%                  1346%                 1339%                                                 112%                      81%                     69%                     24%                               1%                               --                     -2% 
  JC:c_j json, serialize                                315570/s                59857%               59857%                   3118%                   2337%                 1856%                  1383%                 1376%                                                 118%                      86%                     73%                     27%                               3%                               2%                      -- 
 
 Legends:
   CJX:e_j json, serialize: p_tags=json, serialize participant=Cpanel::JSON::XS::encode_json
   DM:p binary, msgpack, serialize: p_tags=binary, msgpack, serialize participant=Data::MessagePack::pack
   JC:c_j json, serialize: p_tags=json, serialize participant=JSON::Create::create_json
   JM:e_j json, serialize: p_tags=json, serialize participant=JSON::MaybeXS::encode_json
   JP:e_j json, serialize: p_tags=json, serialize participant=JSON::PP::encode_json
   JT:e_j json, serialize: p_tags=json, serialize participant=JSON::Tiny::encode_json
   JX:e_j json, serialize: p_tags=json, serialize participant=JSON::XS::encode_json
   MEE:e json, serialize: p_tags=json, serialize participant=MarpaX::ESLIF::ECMA404::encode
   S:e_s binary, sereal, serialize: p_tags=binary, sereal, serialize participant=Sereal::encode_sereal
   S:f binary, storable, serialize, cant_handle_scalar: p_tags=binary, storable, serialize, cant_handle_scalar participant=Storable::freeze
   Y:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Dump
   YO:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Old::Dump
   YS:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Syck::Dump
   YX:D yaml, serialize: p_tags=yaml, serialize participant=YAML::XS::Dump

 #table3#
 {dataset=>"array_int_1000"}
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | p_tags                                          | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | YAML::Old::Dump                | yaml, serialize                                 |        53 |   19      |                 0.00% |             64425.97% |   9e-05 |      20 |
 | YAML::Dump                     | yaml, serialize                                 |        54 |   19      |                 0.74% |             63952.79% | 5.4e-05 |      20 |
 | JSON::PP::encode_json          | json, serialize                                 |       900 |    1      |              1519.37% |              3884.64% | 1.5e-05 |      23 |
 | JSON::Tiny::encode_json        | json, serialize                                 |      1300 |    0.77   |              2338.55% |              2546.08% | 2.4e-06 |      21 |
 | YAML::Syck::Dump               | yaml, serialize                                 |      1700 |    0.6    |              3004.02% |              1978.79% |   5e-06 |      20 |
 | YAML::XS::Dump                 | yaml, serialize                                 |      2260 |    0.442  |              4149.89% |              1418.30% | 2.7e-07 |      20 |
 | MarpaX::ESLIF::ECMA404::encode | json, serialize                                 |      2700 |    0.36   |              5061.21% |              1150.21% | 1.1e-06 |      22 |
 | Storable::freeze               | binary, storable, serialize, cant_handle_scalar |     18000 |    0.057  |             32850.30% |                95.83% | 1.1e-07 |      20 |
 | Cpanel::JSON::XS::encode_json  | json, serialize                                 |     20000 |    0.051  |             36533.69% |                76.14% | 2.2e-07 |      21 |
 | JSON::MaybeXS::encode_json     | json, serialize                                 |     24000 |    0.042  |             44545.23% |                44.53% | 5.3e-08 |      20 |
 | JSON::XS::encode_json          | json, serialize                                 |     26000 |    0.039  |             48290.13% |                33.35% | 5.2e-08 |      21 |
 | JSON::Create::create_json      | json, serialize                                 |     30900 |    0.0324 |             57886.15% |                11.28% | 1.3e-08 |      20 |
 | Sereal::encode_sereal          | binary, sereal, serialize                       |     34000 |    0.029  |             63771.57% |                 1.02% | 6.7e-08 |      20 |
 | Data::MessagePack::pack        | binary, msgpack, serialize                      |     34400 |    0.0291 |             64425.97% |                 0.00% | 1.3e-08 |      21 |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                          Rate  YO:D yaml, serialize  Y:D yaml, serialize  JP:e_j json, serialize  JT:e_j json, serialize  YS:D yaml, serialize  YX:D yaml, serialize  MEE:e json, serialize  S:f binary, storable, serialize, cant_handle_scalar  CJX:e_j json, serialize  JM:e_j json, serialize  JX:e_j json, serialize  JC:c_j json, serialize  DM:p binary, msgpack, serialize  S:e_s binary, sereal, serialize 
  YO:D yaml, serialize                                    53/s                    --                   0%                    -94%                    -95%                  -96%                  -97%                   -98%                                                 -99%                     -99%                    -99%                    -99%                    -99%                             -99%                             -99% 
  Y:D yaml, serialize                                     54/s                    0%                   --                    -94%                    -95%                  -96%                  -97%                   -98%                                                 -99%                     -99%                    -99%                    -99%                    -99%                             -99%                             -99% 
  JP:e_j json, serialize                                 900/s                 1800%                1800%                      --                    -23%                  -40%                  -55%                   -64%                                                 -94%                     -94%                    -95%                    -96%                    -96%                             -97%                             -97% 
  JT:e_j json, serialize                                1300/s                 2367%                2367%                     29%                      --                  -22%                  -42%                   -53%                                                 -92%                     -93%                    -94%                    -94%                    -95%                             -96%                             -96% 
  YS:D yaml, serialize                                  1700/s                 3066%                3066%                     66%                     28%                    --                  -26%                   -40%                                                 -90%                     -91%                    -93%                    -93%                    -94%                             -95%                             -95% 
  YX:D yaml, serialize                                  2260/s                 4198%                4198%                    126%                     74%                   35%                    --                   -18%                                                 -87%                     -88%                    -90%                    -91%                    -92%                             -93%                             -93% 
  MEE:e json, serialize                                 2700/s                 5177%                5177%                    177%                    113%                   66%                   22%                     --                                                 -84%                     -85%                    -88%                    -89%                    -91%                             -91%                             -91% 
  S:f binary, storable, serialize, cant_handle_scalar  18000/s                33233%               33233%                   1654%                   1250%                  952%                  675%                   531%                                                   --                     -10%                    -26%                    -31%                    -43%                             -48%                             -49% 
  CJX:e_j json, serialize                              20000/s                37154%               37154%                   1860%                   1409%                 1076%                  766%                   605%                                                  11%                       --                    -17%                    -23%                    -36%                             -42%                             -43% 
  JM:e_j json, serialize                               24000/s                45138%               45138%                   2280%                   1733%                 1328%                  952%                   757%                                                  35%                      21%                      --                     -7%                    -22%                             -30%                             -30% 
  JX:e_j json, serialize                               26000/s                48617%               48617%                   2464%                   1874%                 1438%                 1033%                   823%                                                  46%                      30%                      7%                      --                    -16%                             -25%                             -25% 
  JC:c_j json, serialize                               30900/s                58541%               58541%                   2986%                   2276%                 1751%                 1264%                  1011%                                                  75%                      57%                     29%                     20%                      --                             -10%                             -10% 
  DM:p binary, msgpack, serialize                      34400/s                65192%               65192%                   3336%                   2546%                 1961%                 1418%                  1137%                                                  95%                      75%                     44%                     34%                     11%                               --                               0% 
  S:e_s binary, sereal, serialize                      34000/s                65417%               65417%                   3348%                   2555%                 1968%                 1424%                  1141%                                                  96%                      75%                     44%                     34%                     11%                               0%                               -- 
 
 Legends:
   CJX:e_j json, serialize: p_tags=json, serialize participant=Cpanel::JSON::XS::encode_json
   DM:p binary, msgpack, serialize: p_tags=binary, msgpack, serialize participant=Data::MessagePack::pack
   JC:c_j json, serialize: p_tags=json, serialize participant=JSON::Create::create_json
   JM:e_j json, serialize: p_tags=json, serialize participant=JSON::MaybeXS::encode_json
   JP:e_j json, serialize: p_tags=json, serialize participant=JSON::PP::encode_json
   JT:e_j json, serialize: p_tags=json, serialize participant=JSON::Tiny::encode_json
   JX:e_j json, serialize: p_tags=json, serialize participant=JSON::XS::encode_json
   MEE:e json, serialize: p_tags=json, serialize participant=MarpaX::ESLIF::ECMA404::encode
   S:e_s binary, sereal, serialize: p_tags=binary, sereal, serialize participant=Sereal::encode_sereal
   S:f binary, storable, serialize, cant_handle_scalar: p_tags=binary, storable, serialize, cant_handle_scalar participant=Storable::freeze
   Y:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Dump
   YO:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Old::Dump
   YS:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Syck::Dump
   YX:D yaml, serialize: p_tags=yaml, serialize participant=YAML::XS::Dump

 #table4#
 {dataset=>"array_str1k_10"}
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | p_tags                                          | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | MarpaX::ESLIF::ECMA404::encode | json, serialize                                 |      1100 |   940     |                 0.00% |             50377.69% | 5.6e-06 |      20 |
 | YAML::Dump                     | yaml, serialize                                 |      3000 |   340     |               177.43% |             18094.45% | 1.1e-06 |      20 |
 | YAML::Old::Dump                | yaml, serialize                                 |      3100 |   320     |               190.41% |             17281.69% | 3.7e-07 |      20 |
 | YAML::XS::Dump                 | yaml, serialize                                 |      8300 |   120     |               678.37% |              6385.08% | 1.6e-07 |      21 |
 | JSON::PP::encode_json          | json, serialize                                 |     10000 |    70     |              1214.24% |              3740.83% | 8.8e-07 |      20 |
 | JSON::Tiny::encode_json        | json, serialize                                 |     25000 |    41     |              2204.34% |              2090.55% | 5.3e-08 |      20 |
 | YAML::Syck::Dump               | yaml, serialize                                 |     25300 |    39.6   |              2271.87% |              2028.18% | 1.3e-08 |      20 |
 | JSON::Create::create_json      | json, serialize                                 |     32000 |    31     |              2898.93% |              1583.19% | 2.3e-07 |      20 |
 | Cpanel::JSON::XS::encode_json  | json, serialize                                 |     46000 |    22     |              4236.16% |              1064.11% |   8e-08 |      20 |
 | JSON::XS::encode_json          | json, serialize                                 |     51000 |    19     |              4718.61% |               947.56% | 2.6e-08 |      21 |
 | JSON::MaybeXS::encode_json     | json, serialize                                 |     53705 |    18.62  |              4938.08% |               901.92% | 2.3e-11 |      21 |
 | Storable::freeze               | binary, storable, serialize, cant_handle_scalar |    243000 |     4.12  |             22670.12% |               121.68% | 1.4e-09 |      29 |
 | Sereal::encode_sereal          | binary, sereal, serialize                       |    508500 |     1.967 |             47602.13% |                 5.82% | 2.7e-11 |      20 |
 | Data::MessagePack::pack        | binary, msgpack, serialize                      |    540000 |     1.9   |             50377.69% |                 0.00% | 2.6e-09 |      33 |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                           Rate  MEE:e json, serialize  Y:D yaml, serialize  YO:D yaml, serialize  YX:D yaml, serialize  JP:e_j json, serialize  JT:e_j json, serialize  YS:D yaml, serialize  JC:c_j json, serialize  CJX:e_j json, serialize  JX:e_j json, serialize  JM:e_j json, serialize  S:f binary, storable, serialize, cant_handle_scalar  S:e_s binary, sereal, serialize  DM:p binary, msgpack, serialize 
  MEE:e json, serialize                                  1100/s                     --                 -63%                  -65%                  -87%                    -92%                    -95%                  -95%                    -96%                     -97%                    -97%                    -98%                                                 -99%                             -99%                             -99% 
  Y:D yaml, serialize                                    3000/s                   176%                   --                   -5%                  -64%                    -79%                    -87%                  -88%                    -90%                     -93%                    -94%                    -94%                                                 -98%                             -99%                             -99% 
  YO:D yaml, serialize                                   3100/s                   193%                   6%                    --                  -62%                    -78%                    -87%                  -87%                    -90%                     -93%                    -94%                    -94%                                                 -98%                             -99%                             -99% 
  YX:D yaml, serialize                                   8300/s                   683%                 183%                  166%                    --                    -41%                    -65%                  -67%                    -74%                     -81%                    -84%                    -84%                                                 -96%                             -98%                             -98% 
  JP:e_j json, serialize                                10000/s                  1242%                 385%                  357%                   71%                      --                    -41%                  -43%                    -55%                     -68%                    -72%                    -73%                                                 -94%                             -97%                             -97% 
  JT:e_j json, serialize                                25000/s                  2192%                 729%                  680%                  192%                     70%                      --                   -3%                    -24%                     -46%                    -53%                    -54%                                                 -89%                             -95%                             -95% 
  YS:D yaml, serialize                                  25300/s                  2273%                 758%                  708%                  203%                     76%                      3%                    --                    -21%                     -44%                    -52%                    -52%                                                 -89%                             -95%                             -95% 
  JC:c_j json, serialize                                32000/s                  2932%                 996%                  932%                  287%                    125%                     32%                   27%                      --                     -29%                    -38%                    -39%                                                 -86%                             -93%                             -93% 
  CJX:e_j json, serialize                               46000/s                  4172%                1445%                 1354%                  445%                    218%                     86%                   80%                     40%                       --                    -13%                    -15%                                                 -81%                             -91%                             -91% 
  JX:e_j json, serialize                                51000/s                  4847%                1689%                 1584%                  531%                    268%                    115%                  108%                     63%                      15%                      --                     -1%                                                 -78%                             -89%                             -90% 
  JM:e_j json, serialize                                53705/s                  4948%                1725%                 1618%                  544%                    275%                    120%                  112%                     66%                      18%                      2%                      --                                                 -77%                             -89%                             -89% 
  S:f binary, storable, serialize, cant_handle_scalar  243000/s                 22715%                8152%                 7666%                 2812%                   1599%                    895%                  861%                    652%                     433%                    361%                    351%                                                   --                             -52%                             -53% 
  S:e_s binary, sereal, serialize                      508500/s                 47688%               17185%                16168%                 6000%                   3458%                   1984%                 1913%                   1476%                    1018%                    865%                    846%                                                 109%                               --                              -3% 
  DM:p binary, msgpack, serialize                      540000/s                 49373%               17794%                16742%                 6215%                   3584%                   2057%                 1984%                   1531%                    1057%                    900%                    880%                                                 116%                               3%                               -- 
 
 Legends:
   CJX:e_j json, serialize: p_tags=json, serialize participant=Cpanel::JSON::XS::encode_json
   DM:p binary, msgpack, serialize: p_tags=binary, msgpack, serialize participant=Data::MessagePack::pack
   JC:c_j json, serialize: p_tags=json, serialize participant=JSON::Create::create_json
   JM:e_j json, serialize: p_tags=json, serialize participant=JSON::MaybeXS::encode_json
   JP:e_j json, serialize: p_tags=json, serialize participant=JSON::PP::encode_json
   JT:e_j json, serialize: p_tags=json, serialize participant=JSON::Tiny::encode_json
   JX:e_j json, serialize: p_tags=json, serialize participant=JSON::XS::encode_json
   MEE:e json, serialize: p_tags=json, serialize participant=MarpaX::ESLIF::ECMA404::encode
   S:e_s binary, sereal, serialize: p_tags=binary, sereal, serialize participant=Sereal::encode_sereal
   S:f binary, storable, serialize, cant_handle_scalar: p_tags=binary, storable, serialize, cant_handle_scalar participant=Storable::freeze
   Y:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Dump
   YO:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Old::Dump
   YS:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Syck::Dump
   YX:D yaml, serialize: p_tags=yaml, serialize participant=YAML::XS::Dump

 #table5#
 {dataset=>"array_ustr1k_10"}
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | p_tags                                          | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | MarpaX::ESLIF::ECMA404::encode | json, serialize                                 |       580 |   1.7     |                 0.00% |             46575.45% | 2.2e-06 |      20 |
 | YAML::Dump                     | yaml, serialize                                 |      1400 |   0.72    |               139.39% |             19397.80% | 2.7e-06 |      21 |
 | YAML::Old::Dump                | yaml, serialize                                 |      1400 |   0.71    |               142.87% |             19118.43% |   2e-06 |      20 |
 | JSON::PP::encode_json          | json, serialize                                 |      2000 |   0.4     |               299.65% |             11579.07% | 1.2e-05 |      21 |
 | JSON::Tiny::encode_json        | json, serialize                                 |      3000 |   0.34    |               412.71% |              9003.61% | 6.8e-07 |      21 |
 | JSON::XS::encode_json          | json, serialize                                 |      5000 |   0.2     |               689.10% |              5815.02% | 5.8e-06 |      20 |
 | YAML::XS::Dump                 | yaml, serialize                                 |      4700 |   0.21    |               710.65% |              5657.77% | 4.3e-07 |      20 |
 | Cpanel::JSON::XS::encode_json  | json, serialize                                 |      5100 |   0.2     |               775.99% |              5228.31% | 1.1e-06 |      20 |
 | JSON::MaybeXS::encode_json     | json, serialize                                 |      5000 |   0.2     |               789.91% |              5144.96% | 2.2e-06 |      30 |
 | YAML::Syck::Dump               | yaml, serialize                                 |      9690 |   0.103   |              1579.13% |              2679.75% | 5.2e-08 |      21 |
 | JSON::Create::create_json      | json, serialize                                 |     19000 |   0.054   |              3129.93% |              1345.09% | 1.6e-07 |      20 |
 | Storable::freeze               | binary, storable, serialize, cant_handle_scalar |    190000 |   0.0054  |             32049.36% |                45.18% | 8.3e-09 |      20 |
 | Sereal::encode_sereal          | binary, sereal, serialize                       |    260000 |   0.0039  |             44684.90% |                 4.22% | 6.7e-09 |      20 |
 | Data::MessagePack::pack        | binary, msgpack, serialize                      |    269000 |   0.00371 |             46575.45% |                 0.00% | 1.6e-09 |      21 |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                           Rate  MEE:e json, serialize  Y:D yaml, serialize  YO:D yaml, serialize  JP:e_j json, serialize  JT:e_j json, serialize  YX:D yaml, serialize  JX:e_j json, serialize  CJX:e_j json, serialize  JM:e_j json, serialize  YS:D yaml, serialize  JC:c_j json, serialize  S:f binary, storable, serialize, cant_handle_scalar  S:e_s binary, sereal, serialize  DM:p binary, msgpack, serialize 
  MEE:e json, serialize                                   580/s                     --                 -57%                  -58%                    -76%                    -80%                  -87%                    -88%                     -88%                    -88%                  -93%                    -96%                                                 -99%                             -99%                             -99% 
  Y:D yaml, serialize                                    1400/s                   136%                   --                   -1%                    -44%                    -52%                  -70%                    -72%                     -72%                    -72%                  -85%                    -92%                                                 -99%                             -99%                             -99% 
  YO:D yaml, serialize                                   1400/s                   139%                   1%                    --                    -43%                    -52%                  -70%                    -71%                     -71%                    -71%                  -85%                    -92%                                                 -99%                             -99%                             -99% 
  JP:e_j json, serialize                                 2000/s                   325%                  79%                   77%                      --                    -15%                  -47%                    -50%                     -50%                    -50%                  -74%                    -86%                                                 -98%                             -99%                             -99% 
  JT:e_j json, serialize                                 3000/s                   399%                 111%                  108%                     17%                      --                  -38%                    -41%                     -41%                    -41%                  -69%                    -84%                                                 -98%                             -98%                             -98% 
  YX:D yaml, serialize                                   4700/s                   709%                 242%                  238%                     90%                     61%                    --                     -4%                      -4%                     -4%                  -50%                    -74%                                                 -97%                             -98%                             -98% 
  JX:e_j json, serialize                                 5000/s                   750%                 259%                  254%                    100%                     70%                    4%                      --                       0%                      0%                  -48%                    -73%                                                 -97%                             -98%                             -98% 
  CJX:e_j json, serialize                                5100/s                   750%                 259%                  254%                    100%                     70%                    4%                      0%                       --                      0%                  -48%                    -73%                                                 -97%                             -98%                             -98% 
  JM:e_j json, serialize                                 5000/s                   750%                 259%                  254%                    100%                     70%                    4%                      0%                       0%                      --                  -48%                    -73%                                                 -97%                             -98%                             -98% 
  YS:D yaml, serialize                                   9690/s                  1550%                 599%                  589%                    288%                    230%                  103%                     94%                      94%                     94%                    --                    -47%                                                 -94%                             -96%                             -96% 
  JC:c_j json, serialize                                19000/s                  3048%                1233%                 1214%                    640%                    529%                  288%                    270%                     270%                    270%                   90%                      --                                                 -90%                             -92%                             -93% 
  S:f binary, storable, serialize, cant_handle_scalar  190000/s                 31381%               13233%                13048%                   7307%                   6196%                 3788%                   3603%                    3603%                   3603%                 1807%                    900%                                                   --                             -27%                             -31% 
  S:e_s binary, sereal, serialize                      260000/s                 43489%               18361%                18105%                  10156%                   8617%                 5284%                   5028%                    5028%                   5028%                 2541%                   1284%                                                  38%                               --                              -4% 
  DM:p binary, msgpack, serialize                      269000/s                 45722%               19307%                19037%                  10681%                   9064%                 5560%                   5290%                    5290%                   5290%                 2676%                   1355%                                                  45%                               5%                               -- 
 
 Legends:
   CJX:e_j json, serialize: p_tags=json, serialize participant=Cpanel::JSON::XS::encode_json
   DM:p binary, msgpack, serialize: p_tags=binary, msgpack, serialize participant=Data::MessagePack::pack
   JC:c_j json, serialize: p_tags=json, serialize participant=JSON::Create::create_json
   JM:e_j json, serialize: p_tags=json, serialize participant=JSON::MaybeXS::encode_json
   JP:e_j json, serialize: p_tags=json, serialize participant=JSON::PP::encode_json
   JT:e_j json, serialize: p_tags=json, serialize participant=JSON::Tiny::encode_json
   JX:e_j json, serialize: p_tags=json, serialize participant=JSON::XS::encode_json
   MEE:e json, serialize: p_tags=json, serialize participant=MarpaX::ESLIF::ECMA404::encode
   S:e_s binary, sereal, serialize: p_tags=binary, sereal, serialize participant=Sereal::encode_sereal
   S:f binary, storable, serialize, cant_handle_scalar: p_tags=binary, storable, serialize, cant_handle_scalar participant=Storable::freeze
   Y:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Dump
   YO:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Old::Dump
   YS:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Syck::Dump
   YX:D yaml, serialize: p_tags=yaml, serialize participant=YAML::XS::Dump

 #table6#
 {dataset=>"hash_int_10"}
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | p_tags                                          | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | YAML::Dump                     | yaml, serialize                                 |      3000 |   340     |                 0.00% |             19527.58% | 9.1e-07 |      20 |
 | YAML::Old::Dump                | yaml, serialize                                 |      3000 |   330     |                 0.99% |             19335.96% | 1.2e-06 |      20 |
 | MarpaX::ESLIF::ECMA404::encode | json, serialize                                 |     43000 |    23     |              1362.30% |              1242.24% | 1.1e-07 |      20 |
 | JSON::PP::encode_json          | json, serialize                                 |     44000 |    23     |              1402.10% |              1206.67% | 2.7e-08 |      20 |
 | JSON::Tiny::encode_json        | json, serialize                                 |     53000 |    19     |              1677.78% |              1004.05% | 5.3e-08 |      20 |
 | YAML::Syck::Dump               | yaml, serialize                                 |     66000 |    15     |              2144.51% |               774.47% | 2.7e-08 |      20 |
 | YAML::XS::Dump                 | yaml, serialize                                 |     73000 |    14     |              2384.77% |               689.92% | 2.7e-08 |      20 |
 | Storable::freeze               | binary, storable, serialize, cant_handle_scalar |    230000 |     4.3   |              7698.65% |               151.68% | 6.7e-09 |      20 |
 | Data::MessagePack::pack        | binary, msgpack, serialize                      |    485000 |     2.06  |             16291.30% |                19.74% | 7.6e-10 |      24 |
 | Cpanel::JSON::XS::encode_json  | json, serialize                                 |    510000 |     1.9   |             17296.72% |                12.82% | 2.5e-09 |      20 |
 | JSON::XS::encode_json          | json, serialize                                 |    500000 |     2     |             17759.94% |                 9.90% | 2.2e-08 |      20 |
 | Sereal::encode_sereal          | binary, sereal, serialize                       |    544000 |     1.84  |             18316.30% |                 6.58% |   7e-10 |      28 |
 | JSON::MaybeXS::encode_json     | json, serialize                                 |    577600 |     1.731 |             19437.27% |                 0.46% | 2.8e-11 |      20 |
 | JSON::Create::create_json      | json, serialize                                 |    600000 |     2     |             19527.58% |                 0.00% | 3.5e-08 |      20 |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                           Rate  Y:D yaml, serialize  YO:D yaml, serialize  MEE:e json, serialize  JP:e_j json, serialize  JT:e_j json, serialize  YS:D yaml, serialize  YX:D yaml, serialize  S:f binary, storable, serialize, cant_handle_scalar  DM:p binary, msgpack, serialize  JX:e_j json, serialize  JC:c_j json, serialize  CJX:e_j json, serialize  S:e_s binary, sereal, serialize  JM:e_j json, serialize 
  Y:D yaml, serialize                                    3000/s                   --                   -2%                   -93%                    -93%                    -94%                  -95%                  -95%                                                 -98%                             -99%                    -99%                    -99%                     -99%                             -99%                    -99% 
  YO:D yaml, serialize                                   3000/s                   3%                    --                   -93%                    -93%                    -94%                  -95%                  -95%                                                 -98%                             -99%                    -99%                    -99%                     -99%                             -99%                    -99% 
  MEE:e json, serialize                                 43000/s                1378%                 1334%                     --                      0%                    -17%                  -34%                  -39%                                                 -81%                             -91%                    -91%                    -91%                     -91%                             -92%                    -92% 
  JP:e_j json, serialize                                44000/s                1378%                 1334%                     0%                      --                    -17%                  -34%                  -39%                                                 -81%                             -91%                    -91%                    -91%                     -91%                             -92%                    -92% 
  JT:e_j json, serialize                                53000/s                1689%                 1636%                    21%                     21%                      --                  -21%                  -26%                                                 -77%                             -89%                    -89%                    -89%                     -90%                             -90%                    -90% 
  YS:D yaml, serialize                                  66000/s                2166%                 2100%                    53%                     53%                     26%                    --                   -6%                                                 -71%                             -86%                    -86%                    -86%                     -87%                             -87%                    -88% 
  YX:D yaml, serialize                                  73000/s                2328%                 2257%                    64%                     64%                     35%                    7%                    --                                                 -69%                             -85%                    -85%                    -85%                     -86%                             -86%                    -87% 
  S:f binary, storable, serialize, cant_handle_scalar  230000/s                7806%                 7574%                   434%                    434%                    341%                  248%                  225%                                                   --                             -52%                    -53%                    -53%                     -55%                             -57%                    -59% 
  DM:p binary, msgpack, serialize                      485000/s               16404%                15919%                  1016%                   1016%                    822%                  628%                  579%                                                 108%                               --                     -2%                     -2%                      -7%                             -10%                    -15% 
  JX:e_j json, serialize                               500000/s               16900%                16400%                  1050%                   1050%                    850%                  650%                  600%                                                 114%                               3%                      --                      0%                      -5%                              -7%                    -13% 
  JC:c_j json, serialize                               600000/s               16900%                16400%                  1050%                   1050%                    850%                  650%                  600%                                                 114%                               3%                      0%                      --                      -5%                              -7%                    -13% 
  CJX:e_j json, serialize                              510000/s               17794%                17268%                  1110%                   1110%                    900%                  689%                  636%                                                 126%                               8%                      5%                      5%                       --                              -3%                     -8% 
  S:e_s binary, sereal, serialize                      544000/s               18378%                17834%                  1150%                   1150%                    932%                  715%                  660%                                                 133%                              11%                      8%                      8%                       3%                               --                     -5% 
  JM:e_j json, serialize                               577600/s               19541%                18964%                  1228%                   1228%                    997%                  766%                  708%                                                 148%                              19%                     15%                     15%                       9%                               6%                      -- 
 
 Legends:
   CJX:e_j json, serialize: p_tags=json, serialize participant=Cpanel::JSON::XS::encode_json
   DM:p binary, msgpack, serialize: p_tags=binary, msgpack, serialize participant=Data::MessagePack::pack
   JC:c_j json, serialize: p_tags=json, serialize participant=JSON::Create::create_json
   JM:e_j json, serialize: p_tags=json, serialize participant=JSON::MaybeXS::encode_json
   JP:e_j json, serialize: p_tags=json, serialize participant=JSON::PP::encode_json
   JT:e_j json, serialize: p_tags=json, serialize participant=JSON::Tiny::encode_json
   JX:e_j json, serialize: p_tags=json, serialize participant=JSON::XS::encode_json
   MEE:e json, serialize: p_tags=json, serialize participant=MarpaX::ESLIF::ECMA404::encode
   S:e_s binary, sereal, serialize: p_tags=binary, sereal, serialize participant=Sereal::encode_sereal
   S:f binary, storable, serialize, cant_handle_scalar: p_tags=binary, storable, serialize, cant_handle_scalar participant=Storable::freeze
   Y:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Dump
   YO:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Old::Dump
   YS:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Syck::Dump
   YX:D yaml, serialize: p_tags=yaml, serialize participant=YAML::XS::Dump

 #table7#
 {dataset=>"hash_int_100"}
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | p_tags                                          | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | YAML::Dump                     | yaml, serialize                                 |       350 |     2.8   |                 0.00% |             17095.20% | 1.1e-05 |      20 |
 | YAML::Old::Dump                | yaml, serialize                                 |       360 |     2.8   |                 0.05% |             17086.54% | 1.8e-05 |      22 |
 | JSON::PP::encode_json          | json, serialize                                 |      4800 |     0.21  |              1262.46% |              1162.07% | 4.8e-07 |      20 |
 | JSON::Tiny::encode_json        | json, serialize                                 |      5800 |     0.17  |              1523.54% |               959.12% | 4.3e-07 |      25 |
 | MarpaX::ESLIF::ECMA404::encode | json, serialize                                 |      6900 |     0.14  |              1846.36% |               783.45% | 9.1e-07 |      20 |
 | YAML::Syck::Dump               | yaml, serialize                                 |      7200 |     0.14  |              1922.10% |               750.36% |   4e-07 |      23 |
 | YAML::XS::Dump                 | yaml, serialize                                 |      7880 |     0.127 |              2120.06% |               674.54% | 5.3e-08 |      20 |
 | Sereal::encode_sereal          | binary, sereal, serialize                       |     47000 |     0.021 |             13061.48% |                30.65% |   6e-08 |      20 |
 | Data::MessagePack::pack        | binary, msgpack, serialize                      |     50000 |     0.02  |             13916.13% |                22.68% | 2.7e-08 |      20 |
 | Storable::freeze               | binary, storable, serialize, cant_handle_scalar |     51000 |     0.019 |             14358.29% |                18.93% | 2.6e-08 |      21 |
 | JSON::MaybeXS::encode_json     | json, serialize                                 |     50000 |     0.02  |             15151.79% |                12.74% | 4.5e-07 |      20 |
 | JSON::XS::encode_json          | json, serialize                                 |     50000 |     0.02  |             15369.22% |                11.16% | 2.3e-07 |      20 |
 | Cpanel::JSON::XS::encode_json  | json, serialize                                 |     57000 |     0.018 |             15819.70% |                 8.01% | 1.1e-07 |      20 |
 | JSON::Create::create_json      | json, serialize                                 |     61000 |     0.016 |             17095.20% |                 0.00% |   6e-08 |      20 |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                          Rate  Y:D yaml, serialize  YO:D yaml, serialize  JP:e_j json, serialize  JT:e_j json, serialize  MEE:e json, serialize  YS:D yaml, serialize  YX:D yaml, serialize  S:e_s binary, sereal, serialize  DM:p binary, msgpack, serialize  JM:e_j json, serialize  JX:e_j json, serialize  S:f binary, storable, serialize, cant_handle_scalar  CJX:e_j json, serialize  JC:c_j json, serialize 
  Y:D yaml, serialize                                    350/s                   --                    0%                    -92%                    -93%                   -95%                  -95%                  -95%                             -99%                             -99%                    -99%                    -99%                                                 -99%                     -99%                    -99% 
  YO:D yaml, serialize                                   360/s                   0%                    --                    -92%                    -93%                   -95%                  -95%                  -95%                             -99%                             -99%                    -99%                    -99%                                                 -99%                     -99%                    -99% 
  JP:e_j json, serialize                                4800/s                1233%                 1233%                      --                    -19%                   -33%                  -33%                  -39%                             -90%                             -90%                    -90%                    -90%                                                 -90%                     -91%                    -92% 
  JT:e_j json, serialize                                5800/s                1547%                 1547%                     23%                      --                   -17%                  -17%                  -25%                             -87%                             -88%                    -88%                    -88%                                                 -88%                     -89%                    -90% 
  MEE:e json, serialize                                 6900/s                1899%                 1899%                     49%                     21%                     --                    0%                   -9%                             -85%                             -85%                    -85%                    -85%                                                 -86%                     -87%                    -88% 
  YS:D yaml, serialize                                  7200/s                1899%                 1899%                     49%                     21%                     0%                    --                   -9%                             -85%                             -85%                    -85%                    -85%                                                 -86%                     -87%                    -88% 
  YX:D yaml, serialize                                  7880/s                2104%                 2104%                     65%                     33%                    10%                   10%                    --                             -83%                             -84%                    -84%                    -84%                                                 -85%                     -85%                    -87% 
  S:e_s binary, sereal, serialize                      47000/s               13233%                13233%                    899%                    709%                   566%                  566%                  504%                               --                              -4%                     -4%                     -4%                                                  -9%                     -14%                    -23% 
  DM:p binary, msgpack, serialize                      50000/s               13900%                13900%                    950%                    750%                   600%                  600%                  535%                               5%                               --                      0%                      0%                                                  -5%                     -10%                    -19% 
  JM:e_j json, serialize                               50000/s               13900%                13900%                    950%                    750%                   600%                  600%                  535%                               5%                               0%                      --                      0%                                                  -5%                     -10%                    -19% 
  JX:e_j json, serialize                               50000/s               13900%                13900%                    950%                    750%                   600%                  600%                  535%                               5%                               0%                      0%                      --                                                  -5%                     -10%                    -19% 
  S:f binary, storable, serialize, cant_handle_scalar  51000/s               14636%                14636%                   1005%                    794%                   636%                  636%                  568%                              10%                               5%                      5%                      5%                                                   --                      -5%                    -15% 
  CJX:e_j json, serialize                              57000/s               15455%                15455%                   1066%                    844%                   677%                  677%                  605%                              16%                              11%                     11%                     11%                                                   5%                       --                    -11% 
  JC:c_j json, serialize                               61000/s               17399%                17399%                   1212%                    962%                   775%                  775%                  693%                              31%                              25%                     25%                     25%                                                  18%                      12%                      -- 
 
 Legends:
   CJX:e_j json, serialize: p_tags=json, serialize participant=Cpanel::JSON::XS::encode_json
   DM:p binary, msgpack, serialize: p_tags=binary, msgpack, serialize participant=Data::MessagePack::pack
   JC:c_j json, serialize: p_tags=json, serialize participant=JSON::Create::create_json
   JM:e_j json, serialize: p_tags=json, serialize participant=JSON::MaybeXS::encode_json
   JP:e_j json, serialize: p_tags=json, serialize participant=JSON::PP::encode_json
   JT:e_j json, serialize: p_tags=json, serialize participant=JSON::Tiny::encode_json
   JX:e_j json, serialize: p_tags=json, serialize participant=JSON::XS::encode_json
   MEE:e json, serialize: p_tags=json, serialize participant=MarpaX::ESLIF::ECMA404::encode
   S:e_s binary, sereal, serialize: p_tags=binary, sereal, serialize participant=Sereal::encode_sereal
   S:f binary, storable, serialize, cant_handle_scalar: p_tags=binary, storable, serialize, cant_handle_scalar participant=Storable::freeze
   Y:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Dump
   YO:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Old::Dump
   YS:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Syck::Dump
   YX:D yaml, serialize: p_tags=yaml, serialize participant=YAML::XS::Dump

 #table8#
 {dataset=>"hash_int_1000"}
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | participant                    | p_tags                                          | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | YAML::Old::Dump                | yaml, serialize                                 |        30 |    30     |                 0.00% |             13875.67% |   0.00053 |      20 |
 | YAML::Dump                     | yaml, serialize                                 |        35 |    28     |                 9.29% |             12687.44% | 7.2e-05   |      20 |
 | JSON::PP::encode_json          | json, serialize                                 |       420 |     2.4   |              1197.65% |               977.00% | 2.2e-05   |      20 |
 | JSON::Tiny::encode_json        | json, serialize                                 |       560 |     1.8   |              1636.48% |               704.83% | 6.2e-06   |      20 |
 | MarpaX::ESLIF::ECMA404::encode | json, serialize                                 |       600 |     2     |              1824.54% |               626.18% | 3.3e-05   |      21 |
 | YAML::Syck::Dump               | yaml, serialize                                 |       730 |     1.4   |              2164.90% |               517.06% | 2.7e-06   |      20 |
 | YAML::XS::Dump                 | yaml, serialize                                 |       750 |     1.3   |              2215.93% |               503.46% | 5.6e-06   |      20 |
 | Sereal::encode_sereal          | binary, sereal, serialize                       |      3490 |     0.287 |             10661.97% |                29.86% | 2.7e-07   |      20 |
 | JSON::XS::encode_json          | json, serialize                                 |      4000 |     0.3   |             11240.75% |                23.23% | 5.3e-06   |      21 |
 | JSON::MaybeXS::encode_json     | json, serialize                                 |      3900 |     0.26  |             11889.01% |                16.57% | 1.5e-06   |      20 |
 | Data::MessagePack::pack        | binary, msgpack, serialize                      |      4100 |     0.24  |             12630.07% |                 9.78% | 4.8e-07   |      20 |
 | Storable::freeze               | binary, storable, serialize, cant_handle_scalar |      4300 |     0.23  |             13246.87% |                 4.71% | 4.3e-07   |      20 |
 | Cpanel::JSON::XS::encode_json  | json, serialize                                 |      4300 |     0.23  |             13305.65% |                 4.25% | 6.9e-07   |      20 |
 | JSON::Create::create_json      | json, serialize                                 |      4500 |     0.22  |             13875.67% |                 0.00% | 2.7e-07   |      20 |
 +--------------------------------+-------------------------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                         Rate  YO:D yaml, serialize  Y:D yaml, serialize  JP:e_j json, serialize  MEE:e json, serialize  JT:e_j json, serialize  YS:D yaml, serialize  YX:D yaml, serialize  JX:e_j json, serialize  S:e_s binary, sereal, serialize  JM:e_j json, serialize  DM:p binary, msgpack, serialize  S:f binary, storable, serialize, cant_handle_scalar  CJX:e_j json, serialize  JC:c_j json, serialize 
  YO:D yaml, serialize                                   30/s                    --                  -6%                    -92%                   -93%                    -94%                  -95%                  -95%                    -99%                             -99%                    -99%                             -99%                                                 -99%                     -99%                    -99% 
  Y:D yaml, serialize                                    35/s                    7%                   --                    -91%                   -92%                    -93%                  -95%                  -95%                    -98%                             -98%                    -99%                             -99%                                                 -99%                     -99%                    -99% 
  JP:e_j json, serialize                                420/s                 1150%                1066%                      --                   -16%                    -25%                  -41%                  -45%                    -87%                             -88%                    -89%                             -90%                                                 -90%                     -90%                    -90% 
  MEE:e json, serialize                                 600/s                 1400%                1300%                     19%                     --                     -9%                  -30%                  -35%                    -85%                             -85%                    -87%                             -88%                                                 -88%                     -88%                    -89% 
  JT:e_j json, serialize                                560/s                 1566%                1455%                     33%                    11%                      --                  -22%                  -27%                    -83%                             -84%                    -85%                             -86%                                                 -87%                     -87%                    -87% 
  YS:D yaml, serialize                                  730/s                 2042%                1900%                     71%                    42%                     28%                    --                   -7%                    -78%                             -79%                    -81%                             -82%                                                 -83%                     -83%                    -84% 
  YX:D yaml, serialize                                  750/s                 2207%                2053%                     84%                    53%                     38%                    7%                    --                    -76%                             -77%                    -80%                             -81%                                                 -82%                     -82%                    -83% 
  JX:e_j json, serialize                               4000/s                 9900%                9233%                    700%                   566%                    500%                  366%                  333%                      --                              -4%                    -13%                             -19%                                                 -23%                     -23%                    -26% 
  S:e_s binary, sereal, serialize                      3490/s                10352%                9656%                    736%                   596%                    527%                  387%                  352%                      4%                               --                     -9%                             -16%                                                 -19%                     -19%                    -23% 
  JM:e_j json, serialize                               3900/s                11438%               10669%                    823%                   669%                    592%                  438%                  400%                     15%                              10%                      --                              -7%                                                 -11%                     -11%                    -15% 
  DM:p binary, msgpack, serialize                      4100/s                12400%               11566%                    900%                   733%                    650%                  483%                  441%                     25%                              19%                      8%                               --                                                  -4%                      -4%                     -8% 
  S:f binary, storable, serialize, cant_handle_scalar  4300/s                12943%               12073%                    943%                   769%                    682%                  508%                  465%                     30%                              24%                     13%                               4%                                                   --                       0%                     -4% 
  CJX:e_j json, serialize                              4300/s                12943%               12073%                    943%                   769%                    682%                  508%                  465%                     30%                              24%                     13%                               4%                                                   0%                       --                     -4% 
  JC:c_j json, serialize                               4500/s                13536%               12627%                    990%                   809%                    718%                  536%                  490%                     36%                              30%                     18%                               9%                                                   4%                       4%                      -- 
 
 Legends:
   CJX:e_j json, serialize: p_tags=json, serialize participant=Cpanel::JSON::XS::encode_json
   DM:p binary, msgpack, serialize: p_tags=binary, msgpack, serialize participant=Data::MessagePack::pack
   JC:c_j json, serialize: p_tags=json, serialize participant=JSON::Create::create_json
   JM:e_j json, serialize: p_tags=json, serialize participant=JSON::MaybeXS::encode_json
   JP:e_j json, serialize: p_tags=json, serialize participant=JSON::PP::encode_json
   JT:e_j json, serialize: p_tags=json, serialize participant=JSON::Tiny::encode_json
   JX:e_j json, serialize: p_tags=json, serialize participant=JSON::XS::encode_json
   MEE:e json, serialize: p_tags=json, serialize participant=MarpaX::ESLIF::ECMA404::encode
   S:e_s binary, sereal, serialize: p_tags=binary, sereal, serialize participant=Sereal::encode_sereal
   S:f binary, storable, serialize, cant_handle_scalar: p_tags=binary, storable, serialize, cant_handle_scalar participant=Storable::freeze
   Y:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Dump
   YO:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Old::Dump
   YS:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Syck::Dump
   YX:D yaml, serialize: p_tags=yaml, serialize participant=YAML::XS::Dump

 #table9#
 {dataset=>"num"}
 +--------------------------------+----------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | p_tags                     | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+----------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | YAML::Old::Dump                | yaml, serialize            |     20000 |   51      |                 0.00% |             24465.64% | 1.7e-07 |      20 |
 | YAML::Dump                     | yaml, serialize            |     20000 |   50      |                 1.91% |             24006.26% |   2e-07 |      20 |
 | MarpaX::ESLIF::ECMA404::encode | json, serialize            |    100000 |   10      |               404.45% |              4769.83% | 1.9e-07 |      21 |
 | YAML::Syck::Dump               | yaml, serialize            |    200000 |    5      |               894.97% |              2369.00% |   7e-08 |      20 |
 | JSON::Tiny::encode_json        | json, serialize            |    240000 |    4.2    |              1124.80% |              1905.68% | 6.7e-09 |      20 |
 | JSON::PP::encode_json          | json, serialize            |    270000 |    3.7    |              1277.26% |              1683.67% | 6.7e-09 |      20 |
 | YAML::XS::Dump                 | yaml, serialize            |    339000 |    2.95   |              1621.21% |              1327.23% | 8.3e-10 |      20 |
 | Cpanel::JSON::XS::encode_json  | json, serialize            |   1800000 |    0.57   |              8845.91% |               174.60% | 1.4e-09 |      29 |
 | JSON::MaybeXS::encode_json     | json, serialize            |   1840000 |    0.543  |              9267.82% |               162.23% | 2.1e-10 |      20 |
 | JSON::XS::encode_json          | json, serialize            |   1967000 |    0.5084 |              9900.67% |               145.64% | 2.2e-11 |      20 |
 | JSON::Create::create_json      | json, serialize            |   2200000 |    0.46   |             11049.43% |               120.33% | 4.4e-09 |      20 |
 | Sereal::encode_sereal          | binary, sereal, serialize  |   3400000 |    0.294  |             17168.15% |                42.26% | 1.1e-10 |      20 |
 | Data::MessagePack::pack        | binary, msgpack, serialize |   4830000 |    0.207  |             24465.64% |                 0.00% |   1e-10 |      21 |
 +--------------------------------+----------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                        Rate  YO:D yaml, serialize  Y:D yaml, serialize  MEE:e json, serialize  YS:D yaml, serialize  JT:e_j json, serialize  JP:e_j json, serialize  YX:D yaml, serialize  CJX:e_j json, serialize  JM:e_j json, serialize  JX:e_j json, serialize  JC:c_j json, serialize  S:e_s binary, sereal, serialize  DM:p binary, msgpack, serialize 
  YO:D yaml, serialize               20000/s                    --                  -1%                   -80%                  -90%                    -91%                    -92%                  -94%                     -98%                    -98%                    -99%                    -99%                             -99%                             -99% 
  Y:D yaml, serialize                20000/s                    2%                   --                   -80%                  -90%                    -91%                    -92%                  -94%                     -98%                    -98%                    -98%                    -99%                             -99%                             -99% 
  MEE:e json, serialize             100000/s                  409%                 400%                     --                  -50%                    -57%                    -63%                  -70%                     -94%                    -94%                    -94%                    -95%                             -97%                             -97% 
  YS:D yaml, serialize              200000/s                  919%                 900%                   100%                    --                    -15%                    -26%                  -40%                     -88%                    -89%                    -89%                    -90%                             -94%                             -95% 
  JT:e_j json, serialize            240000/s                 1114%                1090%                   138%                   19%                      --                    -11%                  -29%                     -86%                    -87%                    -87%                    -89%                             -93%                             -95% 
  JP:e_j json, serialize            270000/s                 1278%                1251%                   170%                   35%                     13%                      --                  -20%                     -84%                    -85%                    -86%                    -87%                             -92%                             -94% 
  YX:D yaml, serialize              339000/s                 1628%                1594%                   238%                   69%                     42%                     25%                    --                     -80%                    -81%                    -82%                    -84%                             -90%                             -92% 
  CJX:e_j json, serialize          1800000/s                 8847%                8671%                  1654%                  777%                    636%                    549%                  417%                       --                     -4%                    -10%                    -19%                             -48%                             -63% 
  JM:e_j json, serialize           1840000/s                 9292%                9108%                  1741%                  820%                    673%                    581%                  443%                       4%                      --                     -6%                    -15%                             -45%                             -61% 
  JX:e_j json, serialize           1967000/s                 9931%                9734%                  1866%                  883%                    726%                    627%                  480%                      12%                      6%                      --                     -9%                             -42%                             -59% 
  JC:c_j json, serialize           2200000/s                10986%               10769%                  2073%                  986%                    813%                    704%                  541%                      23%                     18%                     10%                      --                             -36%                             -55% 
  S:e_s binary, sereal, serialize  3400000/s                17246%               16906%                  3301%                 1600%                   1328%                   1158%                  903%                      93%                     84%                     72%                     56%                               --                             -29% 
  DM:p binary, msgpack, serialize  4830000/s                24537%               24054%                  4730%                 2315%                   1928%                   1687%                 1325%                     175%                    162%                    145%                    122%                              42%                               -- 
 
 Legends:
   CJX:e_j json, serialize: p_tags=json, serialize participant=Cpanel::JSON::XS::encode_json
   DM:p binary, msgpack, serialize: p_tags=binary, msgpack, serialize participant=Data::MessagePack::pack
   JC:c_j json, serialize: p_tags=json, serialize participant=JSON::Create::create_json
   JM:e_j json, serialize: p_tags=json, serialize participant=JSON::MaybeXS::encode_json
   JP:e_j json, serialize: p_tags=json, serialize participant=JSON::PP::encode_json
   JT:e_j json, serialize: p_tags=json, serialize participant=JSON::Tiny::encode_json
   JX:e_j json, serialize: p_tags=json, serialize participant=JSON::XS::encode_json
   MEE:e json, serialize: p_tags=json, serialize participant=MarpaX::ESLIF::ECMA404::encode
   S:e_s binary, sereal, serialize: p_tags=binary, sereal, serialize participant=Sereal::encode_sereal
   Y:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Dump
   YO:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Old::Dump
   YS:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Syck::Dump
   YX:D yaml, serialize: p_tags=yaml, serialize participant=YAML::XS::Dump

 #table10#
 {dataset=>"str1k"}
 +--------------------------------+--------------------+----------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | ds_tags            | p_tags                     | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+--------------------+----------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | MarpaX::ESLIF::ECMA404::encode | serialize, unicode | json, serialize            |      5000 |   200     |                 0.00% |             74352.92% | 1.6e-06 |      21 |
 | MarpaX::ESLIF::ECMA404::encode | serialize          | json, serialize            |      9300 |   110     |                86.76% |             39765.85% | 9.1e-07 |      20 |
 | YAML::Old::Dump                | serialize, unicode | yaml, serialize            |     10000 |    97     |               107.20% |             35832.18% | 3.5e-07 |      20 |
 | YAML::Dump                     | serialize, unicode | yaml, serialize            |     10000 |    96     |               107.54% |             35774.49% | 2.4e-07 |      20 |
 | YAML::Dump                     | serialize          | yaml, serialize            |     17000 |    58     |               242.47% |             21640.13% | 1.3e-07 |      22 |
 | YAML::Old::Dump                | serialize          | yaml, serialize            |     18000 |    57     |               254.11% |             20925.62% | 9.9e-08 |      23 |
 | JSON::PP::encode_json          | serialize, unicode | json, serialize            |     25200 |    39.6   |               405.51% |             14628.21% | 1.2e-08 |      24 |
 | JSON::Tiny::encode_json        | serialize, unicode | json, serialize            |     28000 |    36     |               461.07% |             13169.74% | 6.7e-08 |      20 |
 | YAML::XS::Dump                 | serialize, unicode | yaml, serialize            |     45000 |    22     |               793.86% |              8229.34% | 2.7e-08 |      20 |
 | JSON::MaybeXS::encode_json     | serialize, unicode | json, serialize            |     49000 |    21     |               872.02% |              7559.58% | 1.1e-07 |      20 |
 | Cpanel::JSON::XS::encode_json  | serialize, unicode | json, serialize            |     52000 |    19     |               950.18% |              6989.53% | 3.2e-08 |      22 |
 | JSON::XS::encode_json          | serialize, unicode | json, serialize            |     57000 |    18     |              1043.39% |              6411.59% | 5.3e-08 |      20 |
 | YAML::Syck::Dump               | serialize, unicode | yaml, serialize            |     69000 |    14     |              1287.86% |              5264.57% | 2.6e-08 |      21 |
 | YAML::XS::Dump                 | serialize          | yaml, serialize            |     75000 |    13     |              1405.75% |              4844.57% | 2.7e-08 |      20 |
 | JSON::PP::encode_json          | serialize          | json, serialize            |    140000 |     7.3   |              2639.11% |              2618.15% | 1.3e-08 |      20 |
 | YAML::Syck::Dump               | serialize          | yaml, serialize            |    100000 |     7     |              2699.21% |              2559.78% | 8.3e-08 |      27 |
 | JSON::Tiny::encode_json        | serialize          | json, serialize            |    170000 |     5.8   |              3333.25% |              2068.59% | 6.7e-09 |      20 |
 | JSON::Create::create_json      | serialize, unicode | json, serialize            |    180000 |     5.4   |              3578.30% |              1924.11% | 1.5e-08 |      21 |
 | JSON::Create::create_json      | serialize          | json, serialize            |    300000 |     3     |              5916.70% |              1137.44% | 3.5e-08 |      20 |
 | Cpanel::JSON::XS::encode_json  | serialize          | json, serialize            |    400000 |     3     |              7599.15% |               867.03% | 9.3e-08 |      25 |
 | JSON::MaybeXS::encode_json     | serialize          | json, serialize            |    450000 |     2.2   |              8824.22% |               734.28% | 7.5e-09 |      20 |
 | JSON::XS::encode_json          | serialize          | json, serialize            |    521000 |     1.92  |             10326.16% |               614.10% | 8.3e-10 |      20 |
 | Sereal::encode_sereal          | serialize, unicode | binary, sereal, serialize  |   2580000 |     0.388 |             51548.43% |                44.15% |   2e-10 |      23 |
 | Sereal::encode_sereal          | serialize          | binary, sereal, serialize  |   2790000 |     0.359 |             55686.02% |                33.46% | 2.1e-10 |      20 |
 | Data::MessagePack::pack        | serialize, unicode | binary, msgpack, serialize |   3400000 |     0.3   |             67702.65% |                 9.81% | 3.1e-10 |      20 |
 | Data::MessagePack::pack        | serialize          | binary, msgpack, serialize |   3720000 |     0.269 |             74352.92% |                 0.00% |   9e-11 |      29 |
 +--------------------------------+--------------------+----------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                           Rate  MEE:e json, serialize serialize, unicode  MEE:e json, serialize serialize  YO:D yaml, serialize serialize, unicode  Y:D yaml, serialize serialize, unicode  Y:D yaml, serialize serialize  YO:D yaml, serialize serialize  JP:e_j json, serialize serialize, unicode  JT:e_j json, serialize serialize, unicode  YX:D yaml, serialize serialize, unicode  JM:e_j json, serialize serialize, unicode  CJX:e_j json, serialize serialize, unicode  JX:e_j json, serialize serialize, unicode  YS:D yaml, serialize serialize, unicode  YX:D yaml, serialize serialize  JP:e_j json, serialize serialize  YS:D yaml, serialize serialize  JT:e_j json, serialize serialize  JC:c_j json, serialize serialize, unicode  JC:c_j json, serialize serialize  CJX:e_j json, serialize serialize  JM:e_j json, serialize serialize  JX:e_j json, serialize serialize  S:e_s binary, sereal, serialize serialize, unicode  S:e_s binary, sereal, serialize serialize  DM:p binary, msgpack, serialize serialize, unicode  DM:p binary, msgpack, serialize serialize 
  MEE:e json, serialize serialize, unicode               5000/s                                        --                             -44%                                     -51%                                    -52%                           -71%                            -71%                                       -80%                                       -82%                                     -89%                                       -89%                                        -90%                                       -91%                                     -93%                            -93%                              -96%                            -96%                              -97%                                       -97%                              -98%                               -98%                              -98%                              -99%                                                -99%                                       -99%                                                -99%                                       -99% 
  MEE:e json, serialize serialize                        9300/s                                       81%                               --                                     -11%                                    -12%                           -47%                            -48%                                       -64%                                       -67%                                     -80%                                       -80%                                        -82%                                       -83%                                     -87%                            -88%                              -93%                            -93%                              -94%                                       -95%                              -97%                               -97%                              -98%                              -98%                                                -99%                                       -99%                                                -99%                                       -99% 
  YO:D yaml, serialize serialize, unicode               10000/s                                      106%                              13%                                       --                                     -1%                           -40%                            -41%                                       -59%                                       -62%                                     -77%                                       -78%                                        -80%                                       -81%                                     -85%                            -86%                              -92%                            -92%                              -94%                                       -94%                              -96%                               -96%                              -97%                              -98%                                                -99%                                       -99%                                                -99%                                       -99% 
  Y:D yaml, serialize serialize, unicode                10000/s                                      108%                              14%                                       1%                                      --                           -39%                            -40%                                       -58%                                       -62%                                     -77%                                       -78%                                        -80%                                       -81%                                     -85%                            -86%                              -92%                            -92%                              -93%                                       -94%                              -96%                               -96%                              -97%                              -98%                                                -99%                                       -99%                                                -99%                                       -99% 
  Y:D yaml, serialize serialize                         17000/s                                      244%                              89%                                      67%                                     65%                             --                             -1%                                       -31%                                       -37%                                     -62%                                       -63%                                        -67%                                       -68%                                     -75%                            -77%                              -87%                            -87%                              -90%                                       -90%                              -94%                               -94%                              -96%                              -96%                                                -99%                                       -99%                                                -99%                                       -99% 
  YO:D yaml, serialize serialize                        18000/s                                      250%                              92%                                      70%                                     68%                             1%                              --                                       -30%                                       -36%                                     -61%                                       -63%                                        -66%                                       -68%                                     -75%                            -77%                              -87%                            -87%                              -89%                                       -90%                              -94%                               -94%                              -96%                              -96%                                                -99%                                       -99%                                                -99%                                       -99% 
  JP:e_j json, serialize serialize, unicode             25200/s                                      405%                             177%                                     144%                                    142%                            46%                             43%                                         --                                        -9%                                     -44%                                       -46%                                        -52%                                       -54%                                     -64%                            -67%                              -81%                            -82%                              -85%                                       -86%                              -92%                               -92%                              -94%                              -95%                                                -99%                                       -99%                                                -99%                                       -99% 
  JT:e_j json, serialize serialize, unicode             28000/s                                      455%                             205%                                     169%                                    166%                            61%                             58%                                        10%                                         --                                     -38%                                       -41%                                        -47%                                       -50%                                     -61%                            -63%                              -79%                            -80%                              -83%                                       -85%                              -91%                               -91%                              -93%                              -94%                                                -98%                                       -99%                                                -99%                                       -99% 
  YX:D yaml, serialize serialize, unicode               45000/s                                      809%                             400%                                     340%                                    336%                           163%                            159%                                        80%                                        63%                                       --                                        -4%                                        -13%                                       -18%                                     -36%                            -40%                              -66%                            -68%                              -73%                                       -75%                              -86%                               -86%                              -90%                              -91%                                                -98%                                       -98%                                                -98%                                       -98% 
  JM:e_j json, serialize serialize, unicode             49000/s                                      852%                             423%                                     361%                                    357%                           176%                            171%                                        88%                                        71%                                       4%                                         --                                         -9%                                       -14%                                     -33%                            -38%                              -65%                            -66%                              -72%                                       -74%                              -85%                               -85%                              -89%                              -90%                                                -98%                                       -98%                                                -98%                                       -98% 
  CJX:e_j json, serialize serialize, unicode            52000/s                                      952%                             478%                                     410%                                    405%                           205%                            200%                                       108%                                        89%                                      15%                                        10%                                          --                                        -5%                                     -26%                            -31%                              -61%                            -63%                              -69%                                       -71%                              -84%                               -84%                              -88%                              -89%                                                -97%                                       -98%                                                -98%                                       -98% 
  JX:e_j json, serialize serialize, unicode             57000/s                                     1011%                             511%                                     438%                                    433%                           222%                            216%                                       120%                                       100%                                      22%                                        16%                                          5%                                         --                                     -22%                            -27%                              -59%                            -61%                              -67%                                       -70%                              -83%                               -83%                              -87%                              -89%                                                -97%                                       -98%                                                -98%                                       -98% 
  YS:D yaml, serialize serialize, unicode               69000/s                                     1328%                             685%                                     592%                                    585%                           314%                            307%                                       182%                                       157%                                      57%                                        50%                                         35%                                        28%                                       --                             -7%                              -47%                            -50%                              -58%                                       -61%                              -78%                               -78%                              -84%                              -86%                                                -97%                                       -97%                                                -97%                                       -98% 
  YX:D yaml, serialize serialize                        75000/s                                     1438%                             746%                                     646%                                    638%                           346%                            338%                                       204%                                       176%                                      69%                                        61%                                         46%                                        38%                                       7%                              --                              -43%                            -46%                              -55%                                       -58%                              -76%                               -76%                              -83%                              -85%                                                -97%                                       -97%                                                -97%                                       -97% 
  JP:e_j json, serialize serialize                     140000/s                                     2639%                            1406%                                    1228%                                   1215%                           694%                            680%                                       442%                                       393%                                     201%                                       187%                                        160%                                       146%                                      91%                             78%                                --                             -4%                              -20%                                       -26%                              -58%                               -58%                              -69%                              -73%                                                -94%                                       -95%                                                -95%                                       -96% 
  YS:D yaml, serialize serialize                       100000/s                                     2757%                            1471%                                    1285%                                   1271%                           728%                            714%                                       465%                                       414%                                     214%                                       200%                                        171%                                       157%                                     100%                             85%                                4%                              --                              -17%                                       -22%                              -57%                               -57%                              -68%                              -72%                                                -94%                                       -94%                                                -95%                                       -96% 
  JT:e_j json, serialize serialize                     170000/s                                     3348%                            1796%                                    1572%                                   1555%                           900%                            882%                                       582%                                       520%                                     279%                                       262%                                        227%                                       210%                                     141%                            124%                               25%                             20%                                --                                        -6%                              -48%                               -48%                              -62%                              -66%                                                -93%                                       -93%                                                -94%                                       -95% 
  JC:c_j json, serialize serialize, unicode            180000/s                                     3603%                            1937%                                    1696%                                   1677%                           974%                            955%                                       633%                                       566%                                     307%                                       288%                                        251%                                       233%                                     159%                            140%                               35%                             29%                                7%                                         --                              -44%                               -44%                              -59%                              -64%                                                -92%                                       -93%                                                -94%                                       -95% 
  JC:c_j json, serialize serialize                     300000/s                                     6566%                            3566%                                    3133%                                   3100%                          1833%                           1800%                                      1220%                                      1100%                                     633%                                       600%                                        533%                                       500%                                     366%                            333%                              143%                            133%                               93%                                        80%                                --                                 0%                              -26%                              -36%                                                -87%                                       -88%                                                -90%                                       -91% 
  CJX:e_j json, serialize serialize                    400000/s                                     6566%                            3566%                                    3133%                                   3100%                          1833%                           1800%                                      1220%                                      1100%                                     633%                                       600%                                        533%                                       500%                                     366%                            333%                              143%                            133%                               93%                                        80%                                0%                                 --                              -26%                              -36%                                                -87%                                       -88%                                                -90%                                       -91% 
  JM:e_j json, serialize serialize                     450000/s                                     8990%                            4899%                                    4309%                                   4263%                          2536%                           2490%                                      1700%                                      1536%                                     900%                                       854%                                        763%                                       718%                                     536%                            490%                              231%                            218%                              163%                                       145%                               36%                                36%                                --                              -12%                                                -82%                                       -83%                                                -86%                                       -87% 
  JX:e_j json, serialize serialize                     521000/s                                    10316%                            5629%                                    4952%                                   4900%                          2920%                           2868%                                      1962%                                      1775%                                    1045%                                       993%                                        889%                                       837%                                     629%                            577%                              280%                            264%                              202%                                       181%                               56%                                56%                               14%                                --                                                -79%                                       -81%                                                -84%                                       -85% 
  S:e_s binary, sereal, serialize serialize, unicode  2580000/s                                    51446%                           28250%                                   24900%                                  24642%                         14848%                          14590%                                     10106%                                      9178%                                    5570%                                      5312%                                       4796%                                      4539%                                    3508%                           3250%                             1781%                           1704%                             1394%                                      1291%                              673%                               673%                              467%                              394%                                                  --                                        -7%                                                -22%                                       -30% 
  S:e_s binary, sereal, serialize serialize           2790000/s                                    55610%                           30540%                                   26919%                                  26640%                         16055%                          15777%                                     10930%                                      9927%                                    6028%                                      5749%                                       5192%                                      4913%                                    3799%                           3521%                             1933%                           1849%                             1515%                                      1404%                              735%                               735%                              512%                              434%                                                  8%                                         --                                                -16%                                       -25% 
  DM:p binary, msgpack, serialize serialize, unicode  3400000/s                                    66566%                           36566%                                   32233%                                  31900%                         19233%                          18900%                                     13100%                                     11900%                                    7233%                                      6900%                                       6233%                                      5900%                                    4566%                           4233%                             2333%                           2233%                             1833%                                      1700%                              900%                               900%                              633%                              540%                                                 29%                                        19%                                                  --                                       -10% 
  DM:p binary, msgpack, serialize serialize           3720000/s                                    74249%                           40792%                                   35959%                                  35587%                         21461%                          21089%                                     14621%                                     13282%                                    8078%                                      7706%                                       6963%                                      6591%                                    5104%                           4732%                             2613%                           2502%                             2056%                                      1907%                             1015%                              1015%                              717%                              613%                                                 44%                                        33%                                                 11%                                         -- 
 
 Legends:
   CJX:e_j json, serialize serialize: ds_tags=serialize p_tags=json, serialize participant=Cpanel::JSON::XS::encode_json
   CJX:e_j json, serialize serialize, unicode: ds_tags=serialize, unicode p_tags=json, serialize participant=Cpanel::JSON::XS::encode_json
   DM:p binary, msgpack, serialize serialize: ds_tags=serialize p_tags=binary, msgpack, serialize participant=Data::MessagePack::pack
   DM:p binary, msgpack, serialize serialize, unicode: ds_tags=serialize, unicode p_tags=binary, msgpack, serialize participant=Data::MessagePack::pack
   JC:c_j json, serialize serialize: ds_tags=serialize p_tags=json, serialize participant=JSON::Create::create_json
   JC:c_j json, serialize serialize, unicode: ds_tags=serialize, unicode p_tags=json, serialize participant=JSON::Create::create_json
   JM:e_j json, serialize serialize: ds_tags=serialize p_tags=json, serialize participant=JSON::MaybeXS::encode_json
   JM:e_j json, serialize serialize, unicode: ds_tags=serialize, unicode p_tags=json, serialize participant=JSON::MaybeXS::encode_json
   JP:e_j json, serialize serialize: ds_tags=serialize p_tags=json, serialize participant=JSON::PP::encode_json
   JP:e_j json, serialize serialize, unicode: ds_tags=serialize, unicode p_tags=json, serialize participant=JSON::PP::encode_json
   JT:e_j json, serialize serialize: ds_tags=serialize p_tags=json, serialize participant=JSON::Tiny::encode_json
   JT:e_j json, serialize serialize, unicode: ds_tags=serialize, unicode p_tags=json, serialize participant=JSON::Tiny::encode_json
   JX:e_j json, serialize serialize: ds_tags=serialize p_tags=json, serialize participant=JSON::XS::encode_json
   JX:e_j json, serialize serialize, unicode: ds_tags=serialize, unicode p_tags=json, serialize participant=JSON::XS::encode_json
   MEE:e json, serialize serialize: ds_tags=serialize p_tags=json, serialize participant=MarpaX::ESLIF::ECMA404::encode
   MEE:e json, serialize serialize, unicode: ds_tags=serialize, unicode p_tags=json, serialize participant=MarpaX::ESLIF::ECMA404::encode
   S:e_s binary, sereal, serialize serialize: ds_tags=serialize p_tags=binary, sereal, serialize participant=Sereal::encode_sereal
   S:e_s binary, sereal, serialize serialize, unicode: ds_tags=serialize, unicode p_tags=binary, sereal, serialize participant=Sereal::encode_sereal
   Y:D yaml, serialize serialize: ds_tags=serialize p_tags=yaml, serialize participant=YAML::Dump
   Y:D yaml, serialize serialize, unicode: ds_tags=serialize, unicode p_tags=yaml, serialize participant=YAML::Dump
   YO:D yaml, serialize serialize: ds_tags=serialize p_tags=yaml, serialize participant=YAML::Old::Dump
   YO:D yaml, serialize serialize, unicode: ds_tags=serialize, unicode p_tags=yaml, serialize participant=YAML::Old::Dump
   YS:D yaml, serialize serialize: ds_tags=serialize p_tags=yaml, serialize participant=YAML::Syck::Dump
   YS:D yaml, serialize serialize, unicode: ds_tags=serialize, unicode p_tags=yaml, serialize participant=YAML::Syck::Dump
   YX:D yaml, serialize serialize: ds_tags=serialize p_tags=yaml, serialize participant=YAML::XS::Dump
   YX:D yaml, serialize serialize, unicode: ds_tags=serialize, unicode p_tags=yaml, serialize participant=YAML::XS::Dump

 #table11#
 {dataset=>"undef"}
 +--------------------------------+----------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                    | p_tags                     | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +--------------------------------+----------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | YAML::Dump                     | yaml, serialize            |     23000 |    44     |                 0.00% |             27821.63% | 1.7e-07 |      20 |
 | YAML::Old::Dump                | yaml, serialize            |     23000 |    44     |                 0.80% |             27600.29% | 9.5e-08 |      25 |
 | MarpaX::ESLIF::ECMA404::encode | json, serialize            |    120000 |     8.2   |               438.15% |              5088.46% | 1.3e-08 |      20 |
 | YAML::Syck::Dump               | yaml, serialize            |    240000 |     4.1   |               963.06% |              2526.53% | 2.2e-08 |      20 |
 | JSON::Tiny::encode_json        | json, serialize            |    450000 |     2.22  |              1880.06% |              1310.14% | 7.8e-10 |      23 |
 | JSON::PP::encode_json          | json, serialize            |    549000 |     1.82  |              2314.83% |              1056.26% | 8.3e-10 |      20 |
 | YAML::XS::Dump                 | yaml, serialize            |    598000 |     1.67  |              2529.80% |               961.74% | 8.3e-10 |      20 |
 | Sereal::encode_sereal          | binary, sereal, serialize  |   3770000 |     0.266 |             16468.29% |                68.52% |   1e-10 |      23 |
 | Data::MessagePack::pack        | binary, msgpack, serialize |   5600000 |     0.18  |             24407.23% |                13.93% | 4.2e-10 |      20 |
 | JSON::MaybeXS::encode_json     | json, serialize            |   5800000 |     0.17  |             25256.72% |                10.12% | 4.1e-10 |      21 |
 | Cpanel::JSON::XS::encode_json  | json, serialize            |   5900000 |     0.17  |             26036.06% |                 6.83% | 3.1e-10 |      20 |
 | JSON::XS::encode_json          | json, serialize            |   6000000 |     0.17  |             26232.04% |                 6.04% | 4.2e-10 |      20 |
 | JSON::Create::create_json      | json, serialize            |   6000000 |     0.2   |             27821.63% |                 0.00% | 5.5e-09 |      20 |
 +--------------------------------+----------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                        Rate  Y:D yaml, serialize  YO:D yaml, serialize  MEE:e json, serialize  YS:D yaml, serialize  JT:e_j json, serialize  JP:e_j json, serialize  YX:D yaml, serialize  S:e_s binary, sereal, serialize  JC:c_j json, serialize  DM:p binary, msgpack, serialize  JM:e_j json, serialize  CJX:e_j json, serialize  JX:e_j json, serialize 
  Y:D yaml, serialize                23000/s                   --                    0%                   -81%                  -90%                    -94%                    -95%                  -96%                             -99%                    -99%                             -99%                    -99%                     -99%                    -99% 
  YO:D yaml, serialize               23000/s                   0%                    --                   -81%                  -90%                    -94%                    -95%                  -96%                             -99%                    -99%                             -99%                    -99%                     -99%                    -99% 
  MEE:e json, serialize             120000/s                 436%                  436%                     --                  -50%                    -72%                    -77%                  -79%                             -96%                    -97%                             -97%                    -97%                     -97%                    -97% 
  YS:D yaml, serialize              240000/s                 973%                  973%                   100%                    --                    -45%                    -55%                  -59%                             -93%                    -95%                             -95%                    -95%                     -95%                    -95% 
  JT:e_j json, serialize            450000/s                1881%                 1881%                   269%                   84%                      --                    -18%                  -24%                             -88%                    -90%                             -91%                    -92%                     -92%                    -92% 
  JP:e_j json, serialize            549000/s                2317%                 2317%                   350%                  125%                     21%                      --                   -8%                             -85%                    -89%                             -90%                    -90%                     -90%                    -90% 
  YX:D yaml, serialize              598000/s                2534%                 2534%                   391%                  145%                     32%                      8%                    --                             -84%                    -88%                             -89%                    -89%                     -89%                    -89% 
  S:e_s binary, sereal, serialize  3770000/s               16441%                16441%                  2982%                 1441%                    734%                    584%                  527%                               --                    -24%                             -32%                    -36%                     -36%                    -36% 
  JC:c_j json, serialize           6000000/s               21900%                21900%                  3999%                 1949%                   1010%                    810%                  735%                              33%                      --                             -10%                    -15%                     -15%                    -15% 
  DM:p binary, msgpack, serialize  5600000/s               24344%                24344%                  4455%                 2177%                   1133%                    911%                  827%                              47%                     11%                               --                     -5%                      -5%                     -5% 
  JM:e_j json, serialize           5800000/s               25782%                25782%                  4723%                 2311%                   1205%                    970%                  882%                              56%                     17%                               5%                      --                       0%                      0% 
  CJX:e_j json, serialize          5900000/s               25782%                25782%                  4723%                 2311%                   1205%                    970%                  882%                              56%                     17%                               5%                      0%                       --                      0% 
  JX:e_j json, serialize           6000000/s               25782%                25782%                  4723%                 2311%                   1205%                    970%                  882%                              56%                     17%                               5%                      0%                       0%                      -- 
 
 Legends:
   CJX:e_j json, serialize: p_tags=json, serialize participant=Cpanel::JSON::XS::encode_json
   DM:p binary, msgpack, serialize: p_tags=binary, msgpack, serialize participant=Data::MessagePack::pack
   JC:c_j json, serialize: p_tags=json, serialize participant=JSON::Create::create_json
   JM:e_j json, serialize: p_tags=json, serialize participant=JSON::MaybeXS::encode_json
   JP:e_j json, serialize: p_tags=json, serialize participant=JSON::PP::encode_json
   JT:e_j json, serialize: p_tags=json, serialize participant=JSON::Tiny::encode_json
   JX:e_j json, serialize: p_tags=json, serialize participant=JSON::XS::encode_json
   MEE:e json, serialize: p_tags=json, serialize participant=MarpaX::ESLIF::ECMA404::encode
   S:e_s binary, sereal, serialize: p_tags=binary, sereal, serialize participant=Sereal::encode_sereal
   Y:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Dump
   YO:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Old::Dump
   YS:D yaml, serialize: p_tags=yaml, serialize participant=YAML::Syck::Dump
   YX:D yaml, serialize: p_tags=yaml, serialize participant=YAML::XS::Dump


Benchmark with C<< bencher -m Serializers --include-participant-tags deserialize >>:

 #table12#
 {dataset=>"json:array_int_10"}
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                     | p_tags                                | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | JSON::Decode::Marpa::from_json  | json, deserialize, cant_handle_scalar |      1000 |   900     |                 0.00% |            157174.18% | 1.3e-05 |      20 |
 | Pegex::JSON                     | json, deserialize                     |      1800 |   550     |                64.14% |             95717.41% | 1.1e-06 |      20 |
 | JSON::Parser::Regexp            | json, deserialize                     |      6400 |   160     |               472.94% |             27350.49% | 2.1e-07 |      20 |
 | MarpaX::ESLIF::ECMA404::decode  | json, deserialize                     |     12000 |    85     |               955.15% |             14805.37% | 2.1e-07 |      20 |
 | JSON::PP::decode_json           | json, deserialize                     |     34000 |    29     |              2984.51% |              4998.84% | 3.9e-08 |      21 |
 | JSON::Decode::Regexp::from_json | json, deserialize                     |     44100 |    22.7   |              3862.21% |              3869.35% |   2e-08 |      20 |
 | JSON::Tiny::decode_json         | json, deserialize                     |     52700 |    19     |              4630.37% |              3224.77% | 6.7e-09 |      20 |
 | JSON::MaybeXS::decode_json      | json, deserialize                     |   1560000 |     0.641 |            139987.39% |                12.27% | 6.3e-10 |      20 |
 | Cpanel::JSON::XS::decode_json   | json, deserialize                     |   1590000 |     0.63  |            142281.47% |                10.46% | 2.1e-10 |      20 |
 | JSON::XS::decode_json           | json, deserialize                     |   1600000 |     0.62  |            144694.33% |                 8.62% | 8.3e-10 |      20 |
 | JSON::Parse::parse_json         | json, deserialize                     |   1800000 |     0.57  |            157174.18% |                 0.00% | 6.2e-10 |      20 |
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                      Rate  JDM:f_j json, deserialize, cant_handle_scalar  P:J json, deserialize  JP:R json, deserialize  MEE:d json, deserialize  JP:d_j json, deserialize  JDR:f_j json, deserialize  JT:d_j json, deserialize  JM:d_j json, deserialize  CJX:d_j json, deserialize  JX:d_j json, deserialize  JP:p_j json, deserialize 
  JDM:f_j json, deserialize, cant_handle_scalar     1000/s                                             --                   -38%                    -82%                     -90%                      -96%                       -97%                      -97%                      -99%                       -99%                      -99%                      -99% 
  P:J json, deserialize                             1800/s                                            63%                     --                    -70%                     -84%                      -94%                       -95%                      -96%                      -99%                       -99%                      -99%                      -99% 
  JP:R json, deserialize                            6400/s                                           462%                   243%                      --                     -46%                      -81%                       -85%                      -88%                      -99%                       -99%                      -99%                      -99% 
  MEE:d json, deserialize                          12000/s                                           958%                   547%                     88%                       --                      -65%                       -73%                      -77%                      -99%                       -99%                      -99%                      -99% 
  JP:d_j json, deserialize                         34000/s                                          3003%                  1796%                    451%                     193%                        --                       -21%                      -34%                      -97%                       -97%                      -97%                      -98% 
  JDR:f_j json, deserialize                        44100/s                                          3864%                  2322%                    604%                     274%                       27%                         --                      -16%                      -97%                       -97%                      -97%                      -97% 
  JT:d_j json, deserialize                         52700/s                                          4636%                  2794%                    742%                     347%                       52%                        19%                        --                      -96%                       -96%                      -96%                      -97% 
  JM:d_j json, deserialize                       1560000/s                                        140305%                 85703%                  24860%                   13160%                     4424%                      3441%                     2864%                        --                        -1%                       -3%                      -11% 
  CJX:d_j json, deserialize                      1590000/s                                        142757%                 87201%                  25296%                   13392%                     4503%                      3503%                     2915%                        1%                         --                       -1%                       -9% 
  JX:d_j json, deserialize                       1600000/s                                        145061%                 88609%                  25706%                   13609%                     4577%                      3561%                     2964%                        3%                         1%                        --                       -8% 
  JP:p_j json, deserialize                       1800000/s                                        157794%                 96391%                  27970%                   14812%                     4987%                      3882%                     3233%                       12%                        10%                        8%                        -- 
 
 Legends:
   CJX:d_j json, deserialize: p_tags=json, deserialize participant=Cpanel::JSON::XS::decode_json
   JDM:f_j json, deserialize, cant_handle_scalar: p_tags=json, deserialize, cant_handle_scalar participant=JSON::Decode::Marpa::from_json
   JDR:f_j json, deserialize: p_tags=json, deserialize participant=JSON::Decode::Regexp::from_json
   JM:d_j json, deserialize: p_tags=json, deserialize participant=JSON::MaybeXS::decode_json
   JP:R json, deserialize: p_tags=json, deserialize participant=JSON::Parser::Regexp
   JP:d_j json, deserialize: p_tags=json, deserialize participant=JSON::PP::decode_json
   JP:p_j json, deserialize: p_tags=json, deserialize participant=JSON::Parse::parse_json
   JT:d_j json, deserialize: p_tags=json, deserialize participant=JSON::Tiny::decode_json
   JX:d_j json, deserialize: p_tags=json, deserialize participant=JSON::XS::decode_json
   MEE:d json, deserialize: p_tags=json, deserialize participant=MarpaX::ESLIF::ECMA404::decode
   P:J json, deserialize: p_tags=json, deserialize participant=Pegex::JSON

 #table13#
 {dataset=>"json:array_int_100"}
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                     | p_tags                                | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Pegex::JSON                     | json, deserialize                     |       410 |    2.4    |                 0.00% |             56872.59% | 1.2e-05 |      20 |
 | JSON::Decode::Marpa::from_json  | json, deserialize, cant_handle_scalar |       420 |    2.4    |                 1.45% |             56058.30% | 7.5e-06 |      22 |
 | JSON::Parser::Regexp            | json, deserialize                     |       510 |    2      |                23.29% |             46110.67% | 3.4e-06 |      20 |
 | MarpaX::ESLIF::ECMA404::decode  | json, deserialize                     |      1700 |    0.59   |               310.19% |             13789.24% | 4.1e-06 |      21 |
 | JSON::PP::decode_json           | json, deserialize                     |      3400 |    0.29   |               729.87% |              6765.24% | 6.4e-07 |      20 |
 | JSON::Decode::Regexp::from_json | json, deserialize                     |      4900 |    0.204  |              1083.18% |              4715.19% | 1.5e-07 |      22 |
 | JSON::Tiny::decode_json         | json, deserialize                     |      7200 |    0.14   |              1642.69% |              3169.24% |   2e-07 |      22 |
 | Cpanel::JSON::XS::decode_json   | json, deserialize                     |    210000 |    0.0047 |             51043.01% |                11.40% | 2.8e-08 |      21 |
 | JSON::Parse::parse_json         | json, deserialize                     |    230000 |    0.0044 |             54531.53% |                 4.29% | 1.3e-08 |      20 |
 | JSON::MaybeXS::decode_json      | json, deserialize                     |    230000 |    0.0043 |             56191.85% |                 1.21% |   2e-08 |      20 |
 | JSON::XS::decode_json           | json, deserialize                     |    240000 |    0.0042 |             56872.59% |                 0.00% | 1.5e-08 |      20 |
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                     Rate  P:J json, deserialize  JDM:f_j json, deserialize, cant_handle_scalar  JP:R json, deserialize  MEE:d json, deserialize  JP:d_j json, deserialize  JDR:f_j json, deserialize  JT:d_j json, deserialize  CJX:d_j json, deserialize  JP:p_j json, deserialize  JM:d_j json, deserialize  JX:d_j json, deserialize 
  P:J json, deserialize                             410/s                     --                                             0%                    -16%                     -75%                      -87%                       -91%                      -94%                       -99%                      -99%                      -99%                      -99% 
  JDM:f_j json, deserialize, cant_handle_scalar     420/s                     0%                                             --                    -16%                     -75%                      -87%                       -91%                      -94%                       -99%                      -99%                      -99%                      -99% 
  JP:R json, deserialize                            510/s                    19%                                            19%                      --                     -70%                      -85%                       -89%                      -93%                       -99%                      -99%                      -99%                      -99% 
  MEE:d json, deserialize                          1700/s                   306%                                           306%                    238%                       --                      -50%                       -65%                      -76%                       -99%                      -99%                      -99%                      -99% 
  JP:d_j json, deserialize                         3400/s                   727%                                           727%                    589%                     103%                        --                       -29%                      -51%                       -98%                      -98%                      -98%                      -98% 
  JDR:f_j json, deserialize                        4900/s                  1076%                                          1076%                    880%                     189%                       42%                         --                      -31%                       -97%                      -97%                      -97%                      -97% 
  JT:d_j json, deserialize                         7200/s                  1614%                                          1614%                   1328%                     321%                      107%                        45%                        --                       -96%                      -96%                      -96%                      -97% 
  CJX:d_j json, deserialize                      210000/s                 50963%                                         50963%                  42453%                   12453%                     6070%                      4240%                     2878%                         --                       -6%                       -8%                      -10% 
  JP:p_j json, deserialize                       230000/s                 54445%                                         54445%                  45354%                   13309%                     6490%                      4536%                     3081%                         6%                        --                       -2%                       -4% 
  JM:d_j json, deserialize                       230000/s                 55713%                                         55713%                  46411%                   13620%                     6644%                      4644%                     3155%                         9%                        2%                        --                       -2% 
  JX:d_j json, deserialize                       240000/s                 57042%                                         57042%                  47519%                   13947%                     6804%                      4757%                     3233%                        11%                        4%                        2%                        -- 
 
 Legends:
   CJX:d_j json, deserialize: p_tags=json, deserialize participant=Cpanel::JSON::XS::decode_json
   JDM:f_j json, deserialize, cant_handle_scalar: p_tags=json, deserialize, cant_handle_scalar participant=JSON::Decode::Marpa::from_json
   JDR:f_j json, deserialize: p_tags=json, deserialize participant=JSON::Decode::Regexp::from_json
   JM:d_j json, deserialize: p_tags=json, deserialize participant=JSON::MaybeXS::decode_json
   JP:R json, deserialize: p_tags=json, deserialize participant=JSON::Parser::Regexp
   JP:d_j json, deserialize: p_tags=json, deserialize participant=JSON::PP::decode_json
   JP:p_j json, deserialize: p_tags=json, deserialize participant=JSON::Parse::parse_json
   JT:d_j json, deserialize: p_tags=json, deserialize participant=JSON::Tiny::decode_json
   JX:d_j json, deserialize: p_tags=json, deserialize participant=JSON::XS::decode_json
   MEE:d json, deserialize: p_tags=json, deserialize participant=MarpaX::ESLIF::ECMA404::decode
   P:J json, deserialize: p_tags=json, deserialize participant=Pegex::JSON

 #table14#
 {dataset=>"json:array_int_1000"}
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                     | p_tags                                | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | JSON::Parser::Regexp            | json, deserialize                     |      37.6 |    26.6   |                 0.00% |             84221.25% | 2.1e-05 |      20 |
 | Pegex::JSON                     | json, deserialize                     |      46   |    22     |                21.72% |             69172.09% | 4.6e-05 |      20 |
 | JSON::Decode::Marpa::from_json  | json, deserialize, cant_handle_scalar |      49   |    20     |                30.91% |             64310.72% | 5.2e-05 |      20 |
 | MarpaX::ESLIF::ECMA404::decode  | json, deserialize                     |     140   |     7.2   |               267.33% |             22855.10% | 1.8e-05 |      20 |
 | JSON::PP::decode_json           | json, deserialize                     |     310   |     3.2   |               719.14% |             10193.82% | 4.2e-06 |      20 |
 | JSON::Decode::Regexp::from_json | json, deserialize                     |     380   |     2.6   |               917.14% |              8190.03% | 1.1e-05 |      20 |
 | JSON::Tiny::decode_json         | json, deserialize                     |     760   |     1.3   |              1923.12% |              4067.88% |   2e-06 |      20 |
 | JSON::Parse::parse_json         | json, deserialize                     |   29000   |     0.035 |             76749.22% |                 9.72% | 5.2e-08 |      21 |
 | JSON::XS::decode_json           | json, deserialize                     |   31000   |     0.033 |             81340.82% |                 3.54% | 5.3e-08 |      20 |
 | Cpanel::JSON::XS::decode_json   | json, deserialize                     |   31000   |     0.032 |             82617.13% |                 1.94% | 5.3e-08 |      20 |
 | JSON::MaybeXS::decode_json      | json, deserialize                     |   32000   |     0.032 |             84221.25% |                 0.00% | 5.3e-08 |      20 |
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                    Rate  JP:R json, deserialize  P:J json, deserialize  JDM:f_j json, deserialize, cant_handle_scalar  MEE:d json, deserialize  JP:d_j json, deserialize  JDR:f_j json, deserialize  JT:d_j json, deserialize  JP:p_j json, deserialize  JX:d_j json, deserialize  CJX:d_j json, deserialize  JM:d_j json, deserialize 
  JP:R json, deserialize                          37.6/s                      --                   -17%                                           -24%                     -72%                      -87%                       -90%                      -95%                      -99%                      -99%                       -99%                      -99% 
  P:J json, deserialize                             46/s                     20%                     --                                            -9%                     -67%                      -85%                       -88%                      -94%                      -99%                      -99%                       -99%                      -99% 
  JDM:f_j json, deserialize, cant_handle_scalar     49/s                     33%                    10%                                             --                     -64%                      -84%                       -87%                      -93%                      -99%                      -99%                       -99%                      -99% 
  MEE:d json, deserialize                          140/s                    269%                   205%                                           177%                       --                      -55%                       -63%                      -81%                      -99%                      -99%                       -99%                      -99% 
  JP:d_j json, deserialize                         310/s                    731%                   587%                                           525%                     125%                        --                       -18%                      -59%                      -98%                      -98%                       -99%                      -99% 
  JDR:f_j json, deserialize                        380/s                    923%                   746%                                           669%                     176%                       23%                         --                      -50%                      -98%                      -98%                       -98%                      -98% 
  JT:d_j json, deserialize                         760/s                   1946%                  1592%                                          1438%                     453%                      146%                       100%                        --                      -97%                      -97%                       -97%                      -97% 
  JP:p_j json, deserialize                       29000/s                  75900%                 62757%                                         57042%                   20471%                     9042%                      7328%                     3614%                        --                       -5%                        -8%                       -8% 
  JX:d_j json, deserialize                       31000/s                  80506%                 66566%                                         60506%                   21718%                     9596%                      7778%                     3839%                        6%                        --                        -3%                       -3% 
  CJX:d_j json, deserialize                      31000/s                  83025%                 68650%                                         62400%                   22400%                     9900%                      8025%                     3962%                        9%                        3%                         --                        0% 
  JM:d_j json, deserialize                       32000/s                  83025%                 68650%                                         62400%                   22400%                     9900%                      8025%                     3962%                        9%                        3%                         0%                        -- 
 
 Legends:
   CJX:d_j json, deserialize: p_tags=json, deserialize participant=Cpanel::JSON::XS::decode_json
   JDM:f_j json, deserialize, cant_handle_scalar: p_tags=json, deserialize, cant_handle_scalar participant=JSON::Decode::Marpa::from_json
   JDR:f_j json, deserialize: p_tags=json, deserialize participant=JSON::Decode::Regexp::from_json
   JM:d_j json, deserialize: p_tags=json, deserialize participant=JSON::MaybeXS::decode_json
   JP:R json, deserialize: p_tags=json, deserialize participant=JSON::Parser::Regexp
   JP:d_j json, deserialize: p_tags=json, deserialize participant=JSON::PP::decode_json
   JP:p_j json, deserialize: p_tags=json, deserialize participant=JSON::Parse::parse_json
   JT:d_j json, deserialize: p_tags=json, deserialize participant=JSON::Tiny::decode_json
   JX:d_j json, deserialize: p_tags=json, deserialize participant=JSON::XS::decode_json
   MEE:d json, deserialize: p_tags=json, deserialize participant=MarpaX::ESLIF::ECMA404::decode
   P:J json, deserialize: p_tags=json, deserialize participant=Pegex::JSON

 #table15#
 {dataset=>"json:array_str1k_10"}
 +---------------------------------+---------------------------------------+-----------+------------+-----------------------+-----------------------+-----------+---------+
 | participant                     | p_tags                                | rate (/s) | time (ms)  | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +---------------------------------+---------------------------------------+-----------+------------+-----------------------+-----------------------+-----------+---------+
 | JSON::Parser::Regexp            | json, deserialize                     |      14   | 69         |                 0.00% |            455081.62% |   0.00012 |      20 |
 | JSON::Decode::Marpa::from_json  | json, deserialize, cant_handle_scalar |     110   |  8.9       |               684.39% |             57930.25% | 3.6e-05   |      20 |
 | JSON::PP::decode_json           | json, deserialize                     |     220   |  4.6       |              1401.52% |             30214.66% | 1.3e-05   |      20 |
 | Pegex::JSON                     | json, deserialize                     |     930   |  1.1       |              6357.21% |              6949.20% | 3.4e-06   |      22 |
 | JSON::Tiny::decode_json         | json, deserialize                     |    2090   |  0.479     |             14414.32% |              3036.09% | 2.1e-07   |      20 |
 | MarpaX::ESLIF::ECMA404::decode  | json, deserialize                     |    4000   |  0.25      |             27485.82% |              1550.06% | 6.4e-07   |      20 |
 | JSON::Decode::Regexp::from_json | json, deserialize                     |   26000   |  0.038     |            181187.15% |               151.08% | 5.3e-08   |      20 |
 | JSON::Parse::parse_json         | json, deserialize                     |   34730.5 |  0.0287931 |            241271.38% |                88.58% | 2.4e-11   |      20 |
 | JSON::MaybeXS::decode_json      | json, deserialize                     |   53843   |  0.018573  |            374097.91% |                21.64% | 2.5e-11   |      20 |
 | Cpanel::JSON::XS::decode_json   | json, deserialize                     |   53979   |  0.018526  |            375042.07% |                21.34% | 4.3e-11   |      20 |
 | JSON::XS::decode_json           | json, deserialize                     |   65500   |  0.0153    |            455081.62% |                 0.00% |   5e-09   |      35 |
 +---------------------------------+---------------------------------------+-----------+------------+-----------------------+-----------------------+-----------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                      Rate  JP:R json, deserialize  JDM:f_j json, deserialize, cant_handle_scalar  JP:d_j json, deserialize  P:J json, deserialize  JT:d_j json, deserialize  MEE:d json, deserialize  JDR:f_j json, deserialize  JP:p_j json, deserialize  JM:d_j json, deserialize  CJX:d_j json, deserialize  JX:d_j json, deserialize 
  JP:R json, deserialize                              14/s                      --                                           -87%                      -93%                   -98%                      -99%                     -99%                       -99%                      -99%                      -99%                       -99%                      -99% 
  JDM:f_j json, deserialize, cant_handle_scalar      110/s                    675%                                             --                      -48%                   -87%                      -94%                     -97%                       -99%                      -99%                      -99%                       -99%                      -99% 
  JP:d_j json, deserialize                           220/s                   1400%                                            93%                        --                   -76%                      -89%                     -94%                       -99%                      -99%                      -99%                       -99%                      -99% 
  P:J json, deserialize                              930/s                   6172%                                           709%                      318%                     --                      -56%                     -77%                       -96%                      -97%                      -98%                       -98%                      -98% 
  JT:d_j json, deserialize                          2090/s                  14305%                                          1758%                      860%                   129%                        --                     -47%                       -92%                      -93%                      -96%                       -96%                      -96% 
  MEE:d json, deserialize                           4000/s                  27500%                                          3460%                     1739%                   340%                       91%                       --                       -84%                      -88%                      -92%                       -92%                      -93% 
  JDR:f_j json, deserialize                        26000/s                 181478%                                         23321%                    12005%                  2794%                     1160%                     557%                         --                      -24%                      -51%                       -51%                      -59% 
  JP:p_j json, deserialize                       34730.5/s                 239540%                                         30810%                    15876%                  3720%                     1563%                     768%                        31%                        --                      -35%                       -35%                      -46% 
  JM:d_j json, deserialize                         53843/s                 371407%                                         47819%                    24667%                  5822%                     2479%                    1246%                       104%                       55%                        --                         0%                      -17% 
  CJX:d_j json, deserialize                        53979/s                 372349%                                         47940%                    24729%                  5837%                     2485%                    1249%                       105%                       55%                        0%                         --                      -17% 
  JX:d_j json, deserialize                         65500/s                 450880%                                         58069%                    29965%                  7089%                     3030%                    1533%                       148%                       88%                       21%                        21%                        -- 
 
 Legends:
   CJX:d_j json, deserialize: p_tags=json, deserialize participant=Cpanel::JSON::XS::decode_json
   JDM:f_j json, deserialize, cant_handle_scalar: p_tags=json, deserialize, cant_handle_scalar participant=JSON::Decode::Marpa::from_json
   JDR:f_j json, deserialize: p_tags=json, deserialize participant=JSON::Decode::Regexp::from_json
   JM:d_j json, deserialize: p_tags=json, deserialize participant=JSON::MaybeXS::decode_json
   JP:R json, deserialize: p_tags=json, deserialize participant=JSON::Parser::Regexp
   JP:d_j json, deserialize: p_tags=json, deserialize participant=JSON::PP::decode_json
   JP:p_j json, deserialize: p_tags=json, deserialize participant=JSON::Parse::parse_json
   JT:d_j json, deserialize: p_tags=json, deserialize participant=JSON::Tiny::decode_json
   JX:d_j json, deserialize: p_tags=json, deserialize participant=JSON::XS::decode_json
   MEE:d json, deserialize: p_tags=json, deserialize participant=MarpaX::ESLIF::ECMA404::decode
   P:J json, deserialize: p_tags=json, deserialize participant=Pegex::JSON

 #table16#
 {dataset=>"json:hash_int_10"}
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                     | p_tags                                | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | JSON::Parser::Regexp            | json, deserialize                     |       440 |  2.3      |                 0.00% |            221113.17% | 2.2e-05 |      20 |
 | JSON::Decode::Marpa::from_json  | json, deserialize, cant_handle_scalar |       900 |  1        |               107.55% |            106483.02% | 2.1e-05 |      20 |
 | Pegex::JSON                     | json, deserialize                     |      1400 |  0.73     |               210.35% |             71178.04% | 3.3e-06 |      20 |
 | MarpaX::ESLIF::ECMA404::decode  | json, deserialize                     |      5000 |  0.2      |              1122.42% |             17996.26% | 2.2e-06 |      20 |
 | JSON::PP::decode_json           | json, deserialize                     |      7800 |  0.13     |              1666.87% |             12420.08% | 6.4e-07 |      20 |
 | JSON::Decode::Regexp::from_json | json, deserialize                     |     21000 |  0.049    |              4571.43% |              4635.45% | 5.3e-08 |      20 |
 | JSON::Tiny::decode_json         | json, deserialize                     |     27000 |  0.037    |              6105.88% |              3464.58% | 5.3e-08 |      20 |
 | Cpanel::JSON::XS::decode_json   | json, deserialize                     |    795000 |  0.00126  |            180204.25% |                22.69% | 4.3e-10 |      20 |
 | JSON::MaybeXS::decode_json      | json, deserialize                     |    799000 |  0.00125  |            181122.71% |                22.07% |   4e-10 |      22 |
 | JSON::XS::decode_json           | json, deserialize                     |    930000 |  0.0011   |            211074.16% |                 4.75% | 1.7e-09 |      20 |
 | JSON::Parse::parse_json         | json, deserialize                     |    974900 |  0.001026 |            221113.17% |                 0.00% | 2.2e-11 |      20 |
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                     Rate  JP:R json, deserialize  JDM:f_j json, deserialize, cant_handle_scalar  P:J json, deserialize  MEE:d json, deserialize  JP:d_j json, deserialize  JDR:f_j json, deserialize  JT:d_j json, deserialize  CJX:d_j json, deserialize  JM:d_j json, deserialize  JX:d_j json, deserialize  JP:p_j json, deserialize 
  JP:R json, deserialize                            440/s                      --                                           -56%                   -68%                     -91%                      -94%                       -97%                      -98%                       -99%                      -99%                      -99%                      -99% 
  JDM:f_j json, deserialize, cant_handle_scalar     900/s                    129%                                             --                   -27%                     -80%                      -87%                       -95%                      -96%                       -99%                      -99%                      -99%                      -99% 
  P:J json, deserialize                            1400/s                    215%                                            36%                     --                     -72%                      -82%                       -93%                      -94%                       -99%                      -99%                      -99%                      -99% 
  MEE:d json, deserialize                          5000/s                   1049%                                           400%                   265%                       --                      -35%                       -75%                      -81%                       -99%                      -99%                      -99%                      -99% 
  JP:d_j json, deserialize                         7800/s                   1669%                                           669%                   461%                      53%                        --                       -62%                      -71%                       -99%                      -99%                      -99%                      -99% 
  JDR:f_j json, deserialize                       21000/s                   4593%                                          1940%                  1389%                     308%                      165%                         --                      -24%                       -97%                      -97%                      -97%                      -97% 
  JT:d_j json, deserialize                        27000/s                   6116%                                          2602%                  1872%                     440%                      251%                        32%                        --                       -96%                      -96%                      -97%                      -97% 
  CJX:d_j json, deserialize                      795000/s                 182439%                                         79265%                 57836%                   15773%                    10217%                      3788%                     2836%                         --                        0%                      -12%                      -18% 
  JM:d_j json, deserialize                       799000/s                 183899%                                         79900%                 58300%                   15900%                    10300%                      3820%                     2860%                         0%                        --                      -12%                      -17% 
  JX:d_j json, deserialize                       930000/s                 208990%                                         90809%                 66263%                   18081%                    11718%                      4354%                     3263%                        14%                       13%                        --                       -6% 
  JP:p_j json, deserialize                       974900/s                 224071%                                         97365%                 71050%                   19393%                    12570%                      4675%                     3506%                        22%                       21%                        7%                        -- 
 
 Legends:
   CJX:d_j json, deserialize: p_tags=json, deserialize participant=Cpanel::JSON::XS::decode_json
   JDM:f_j json, deserialize, cant_handle_scalar: p_tags=json, deserialize, cant_handle_scalar participant=JSON::Decode::Marpa::from_json
   JDR:f_j json, deserialize: p_tags=json, deserialize participant=JSON::Decode::Regexp::from_json
   JM:d_j json, deserialize: p_tags=json, deserialize participant=JSON::MaybeXS::decode_json
   JP:R json, deserialize: p_tags=json, deserialize participant=JSON::Parser::Regexp
   JP:d_j json, deserialize: p_tags=json, deserialize participant=JSON::PP::decode_json
   JP:p_j json, deserialize: p_tags=json, deserialize participant=JSON::Parse::parse_json
   JT:d_j json, deserialize: p_tags=json, deserialize participant=JSON::Tiny::decode_json
   JX:d_j json, deserialize: p_tags=json, deserialize participant=JSON::XS::decode_json
   MEE:d json, deserialize: p_tags=json, deserialize participant=MarpaX::ESLIF::ECMA404::decode
   P:J json, deserialize: p_tags=json, deserialize participant=Pegex::JSON

 #table17#
 {dataset=>"json:hash_int_100"}
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | participant                     | p_tags                                | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | JSON::Parser::Regexp            | json, deserialize                     |        35 |   28      |                 0.00% |            258511.85% |   0.00011 |      21 |
 | JSON::Decode::Marpa::from_json  | json, deserialize, cant_handle_scalar |       200 |    5      |               499.33% |             43050.33% |   6e-05   |      20 |
 | Pegex::JSON                     | json, deserialize                     |       240 |    4.2    |               570.43% |             38474.27% | 1.7e-05   |      21 |
 | MarpaX::ESLIF::ECMA404::decode  | json, deserialize                     |       700 |    2      |              1795.53% |             13543.27% | 1.7e-05   |      20 |
 | JSON::PP::decode_json           | json, deserialize                     |      1900 |    0.52   |              5377.87% |              4621.03% | 6.4e-07   |      20 |
 | JSON::Decode::Regexp::from_json | json, deserialize                     |      2200 |    0.46   |              6053.54% |              4102.65% | 9.1e-07   |      20 |
 | JSON::Tiny::decode_json         | json, deserialize                     |      3300 |    0.31   |              9179.44% |              2686.93% | 4.3e-07   |      20 |
 | Cpanel::JSON::XS::decode_json   | json, deserialize                     |     77000 |    0.013  |            218953.33% |                18.06% | 5.2e-08   |      21 |
 | JSON::MaybeXS::decode_json      | json, deserialize                     |     77900 |    0.0128 |            221925.34% |                16.48% | 6.4e-09   |      22 |
 | JSON::XS::decode_json           | json, deserialize                     |     91000 |    0.011  |            258240.22% |                 0.11% | 1.3e-08   |      20 |
 | JSON::Parse::parse_json         | json, deserialize                     |     91000 |    0.011  |            258511.85% |                 0.00% | 1.3e-08   |      20 |
 +---------------------------------+---------------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                    Rate  JP:R json, deserialize  JDM:f_j json, deserialize, cant_handle_scalar  P:J json, deserialize  MEE:d json, deserialize  JP:d_j json, deserialize  JDR:f_j json, deserialize  JT:d_j json, deserialize  CJX:d_j json, deserialize  JM:d_j json, deserialize  JX:d_j json, deserialize  JP:p_j json, deserialize 
  JP:R json, deserialize                            35/s                      --                                           -82%                   -85%                     -92%                      -98%                       -98%                      -98%                       -99%                      -99%                      -99%                      -99% 
  JDM:f_j json, deserialize, cant_handle_scalar    200/s                    459%                                             --                   -15%                     -60%                      -89%                       -90%                      -93%                       -99%                      -99%                      -99%                      -99% 
  P:J json, deserialize                            240/s                    566%                                            19%                     --                     -52%                      -87%                       -89%                      -92%                       -99%                      -99%                      -99%                      -99% 
  MEE:d json, deserialize                          700/s                   1300%                                           150%                   110%                       --                      -74%                       -77%                      -84%                       -99%                      -99%                      -99%                      -99% 
  JP:d_j json, deserialize                        1900/s                   5284%                                           861%                   707%                     284%                        --                       -11%                      -40%                       -97%                      -97%                      -97%                      -97% 
  JDR:f_j json, deserialize                       2200/s                   5986%                                           986%                   813%                     334%                       13%                         --                      -32%                       -97%                      -97%                      -97%                      -97% 
  JT:d_j json, deserialize                        3300/s                   8932%                                          1512%                  1254%                     545%                       67%                        48%                        --                       -95%                      -95%                      -96%                      -96% 
  CJX:d_j json, deserialize                      77000/s                 215284%                                         38361%                 32207%                   15284%                     3900%                      3438%                     2284%                         --                       -1%                      -15%                      -15% 
  JM:d_j json, deserialize                       77900/s                 218650%                                         38962%                 32712%                   15525%                     3962%                      3493%                     2321%                         1%                        --                      -14%                      -14% 
  JX:d_j json, deserialize                       91000/s                 254445%                                         45354%                 38081%                   18081%                     4627%                      4081%                     2718%                        18%                       16%                        --                        0% 
  JP:p_j json, deserialize                       91000/s                 254445%                                         45354%                 38081%                   18081%                     4627%                      4081%                     2718%                        18%                       16%                        0%                        -- 
 
 Legends:
   CJX:d_j json, deserialize: p_tags=json, deserialize participant=Cpanel::JSON::XS::decode_json
   JDM:f_j json, deserialize, cant_handle_scalar: p_tags=json, deserialize, cant_handle_scalar participant=JSON::Decode::Marpa::from_json
   JDR:f_j json, deserialize: p_tags=json, deserialize participant=JSON::Decode::Regexp::from_json
   JM:d_j json, deserialize: p_tags=json, deserialize participant=JSON::MaybeXS::decode_json
   JP:R json, deserialize: p_tags=json, deserialize participant=JSON::Parser::Regexp
   JP:d_j json, deserialize: p_tags=json, deserialize participant=JSON::PP::decode_json
   JP:p_j json, deserialize: p_tags=json, deserialize participant=JSON::Parse::parse_json
   JT:d_j json, deserialize: p_tags=json, deserialize participant=JSON::Tiny::decode_json
   JX:d_j json, deserialize: p_tags=json, deserialize participant=JSON::XS::decode_json
   MEE:d json, deserialize: p_tags=json, deserialize participant=MarpaX::ESLIF::ECMA404::decode
   P:J json, deserialize: p_tags=json, deserialize participant=Pegex::JSON

 #table18#
 {dataset=>"json:hash_int_1000"}
 +---------------------------------+---------------------------------------+-----------+------------+-----------------------+-----------------------+-----------+---------+
 | participant                     | p_tags                                | rate (/s) | time (ms)  | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +---------------------------------+---------------------------------------+-----------+------------+-----------------------+-----------------------+-----------+---------+
 | JSON::Parser::Regexp            | json, deserialize                     |      3    | 300        |                 0.00% |            224003.01% |   0.0034  |      20 |
 | JSON::Decode::Marpa::from_json  | json, deserialize, cant_handle_scalar |     23    |  43        |               651.15% |             29734.84% |   0.00011 |      20 |
 | Pegex::JSON                     | json, deserialize                     |     24    |  42        |               660.60% |             29364.14% | 9.7e-05   |      20 |
 | MarpaX::ESLIF::ECMA404::decode  | json, deserialize                     |     56    |  18        |              1724.51% |             12182.89% | 3.9e-05   |      20 |
 | JSON::Decode::Regexp::from_json | json, deserialize                     |    140    |   7.2      |              4358.82% |              4926.07% | 6.5e-05   |      20 |
 | JSON::PP::decode_json           | json, deserialize                     |    170    |   6        |              5284.49% |              4062.01% |   4e-05   |      20 |
 | JSON::Tiny::decode_json         | json, deserialize                     |    320    |   3.1      |             10304.19% |              2053.97% | 1.1e-05   |      21 |
 | JSON::Parse::parse_json         | json, deserialize                     |   5900    |   0.17     |            189507.39% |                18.19% | 8.5e-07   |      20 |
 | Cpanel::JSON::XS::decode_json   | json, deserialize                     |   6100    |   0.16     |            197942.04% |                13.16% | 2.7e-07   |      20 |
 | JSON::MaybeXS::decode_json      | json, deserialize                     |   6100    |   0.16     |            198173.71% |                13.03% | 2.1e-07   |      20 |
 | JSON::XS::decode_json           | json, deserialize                     |   6933.95 |   0.144218 |            224003.01% |                 0.00% |   2e-11   |      20 |
 +---------------------------------+---------------------------------------+-----------+------------+-----------------------+-----------------------+-----------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                                      Rate  JP:R json, deserialize  JDM:f_j json, deserialize, cant_handle_scalar  P:J json, deserialize  MEE:d json, deserialize  JDR:f_j json, deserialize  JP:d_j json, deserialize  JT:d_j json, deserialize  JP:p_j json, deserialize  CJX:d_j json, deserialize  JM:d_j json, deserialize  JX:d_j json, deserialize 
  JP:R json, deserialize                               3/s                      --                                           -85%                   -86%                     -94%                       -97%                      -98%                      -98%                      -99%                       -99%                      -99%                      -99% 
  JDM:f_j json, deserialize, cant_handle_scalar       23/s                    597%                                             --                    -2%                     -58%                       -83%                      -86%                      -92%                      -99%                       -99%                      -99%                      -99% 
  P:J json, deserialize                               24/s                    614%                                             2%                     --                     -57%                       -82%                      -85%                      -92%                      -99%                       -99%                      -99%                      -99% 
  MEE:d json, deserialize                             56/s                   1566%                                           138%                   133%                       --                       -60%                      -66%                      -82%                      -99%                       -99%                      -99%                      -99% 
  JDR:f_j json, deserialize                          140/s                   4066%                                           497%                   483%                     150%                         --                      -16%                      -56%                      -97%                       -97%                      -97%                      -97% 
  JP:d_j json, deserialize                           170/s                   4900%                                           616%                   600%                     200%                        19%                        --                      -48%                      -97%                       -97%                      -97%                      -97% 
  JT:d_j json, deserialize                           320/s                   9577%                                          1287%                  1254%                     480%                       132%                       93%                        --                      -94%                       -94%                      -94%                      -95% 
  JP:p_j json, deserialize                          5900/s                 176370%                                         25194%                 24605%                   10488%                      4135%                     3429%                     1723%                        --                        -5%                       -5%                      -15% 
  CJX:d_j json, deserialize                         6100/s                 187400%                                         26775%                 26150%                   11150%                      4400%                     3650%                     1837%                        6%                         --                        0%                       -9% 
  JM:d_j json, deserialize                          6100/s                 187400%                                         26775%                 26150%                   11150%                      4400%                     3650%                     1837%                        6%                         0%                        --                       -9% 
  JX:d_j json, deserialize                       6933.95/s                 207918%                                         29715%                 29022%                   12381%                      4892%                     4060%                     2049%                       17%                        10%                       10%                        -- 
 
 Legends:
   CJX:d_j json, deserialize: p_tags=json, deserialize participant=Cpanel::JSON::XS::decode_json
   JDM:f_j json, deserialize, cant_handle_scalar: p_tags=json, deserialize, cant_handle_scalar participant=JSON::Decode::Marpa::from_json
   JDR:f_j json, deserialize: p_tags=json, deserialize participant=JSON::Decode::Regexp::from_json
   JM:d_j json, deserialize: p_tags=json, deserialize participant=JSON::MaybeXS::decode_json
   JP:R json, deserialize: p_tags=json, deserialize participant=JSON::Parser::Regexp
   JP:d_j json, deserialize: p_tags=json, deserialize participant=JSON::PP::decode_json
   JP:p_j json, deserialize: p_tags=json, deserialize participant=JSON::Parse::parse_json
   JT:d_j json, deserialize: p_tags=json, deserialize participant=JSON::Tiny::decode_json
   JX:d_j json, deserialize: p_tags=json, deserialize participant=JSON::XS::decode_json
   MEE:d json, deserialize: p_tags=json, deserialize participant=MarpaX::ESLIF::ECMA404::decode
   P:J json, deserialize: p_tags=json, deserialize participant=Pegex::JSON

 #table19#
 {dataset=>"json:null"}
 +---------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                     | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +---------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Pegex::JSON                     |      2900 |   340     |                 0.00% |            432868.50% | 1.4e-06 |      20 |
 | JSON::Parser::Regexp            |     27000 |    37     |               813.86% |             47278.00% | 5.3e-08 |      20 |
 | MarpaX::ESLIF::ECMA404::decode  |     34000 |    29     |              1060.03% |             37223.78% | 1.2e-07 |      20 |
 | JSON::Tiny::decode_json         |    210000 |     4.8   |              7018.46% |              5982.34% | 8.3e-09 |      20 |
 | JSON::PP::decode_json           |    310000 |     3.2   |             10427.64% |              4012.68% |   2e-08 |      20 |
 | JSON::Decode::Regexp::from_json |    450000 |     2.2   |             15130.12% |              2742.84% | 3.3e-09 |      20 |
 | JSON::MaybeXS::decode_json      |   7940000 |     0.126 |            269739.33% |                60.45% | 5.7e-11 |      20 |
 | Cpanel::JSON::XS::decode_json   |   8200000 |     0.12  |            277179.29% |                56.15% | 2.1e-10 |      20 |
 | JSON::XS::decode_json           |   9500000 |     0.11  |            322725.10% |                34.12% | 4.2e-10 |      20 |
 | JSON::Parse::parse_json         |  13000000 |     0.079 |            432868.50% |                 0.00% | 1.5e-10 |      21 |
 +---------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                 Rate      P:J    JP:R   MEE:d  JT:d_j  JP:d_j  JDR:f_j  JM:d_j  CJX:d_j  JX:d_j  JP:p_j 
  P:J          2900/s       --    -89%    -91%    -98%    -99%     -99%    -99%     -99%    -99%    -99% 
  JP:R        27000/s     818%      --    -21%    -87%    -91%     -94%    -99%     -99%    -99%    -99% 
  MEE:d       34000/s    1072%     27%      --    -83%    -88%     -92%    -99%     -99%    -99%    -99% 
  JT:d_j     210000/s    6983%    670%    504%      --    -33%     -54%    -97%     -97%    -97%    -98% 
  JP:d_j     310000/s   10525%   1056%    806%     49%      --     -31%    -96%     -96%    -96%    -97% 
  JDR:f_j    450000/s   15354%   1581%   1218%    118%     45%       --    -94%     -94%    -95%    -96% 
  JM:d_j    7940000/s  269741%  29265%  22915%   3709%   2439%    1646%      --      -4%    -12%    -37% 
  CJX:d_j   8200000/s  283233%  30733%  24066%   3900%   2566%    1733%      5%       --     -8%    -34% 
  JX:d_j    9500000/s  308990%  33536%  26263%   4263%   2809%    1900%     14%       9%      --    -28% 
  JP:p_j   13000000/s  430279%  46735%  36608%   5975%   3950%    2684%     59%      51%     39%      -- 
 
 Legends:
   CJX:d_j: participant=Cpanel::JSON::XS::decode_json
   JDR:f_j: participant=JSON::Decode::Regexp::from_json
   JM:d_j: participant=JSON::MaybeXS::decode_json
   JP:R: participant=JSON::Parser::Regexp
   JP:d_j: participant=JSON::PP::decode_json
   JP:p_j: participant=JSON::Parse::parse_json
   JT:d_j: participant=JSON::Tiny::decode_json
   JX:d_j: participant=JSON::XS::decode_json
   MEE:d: participant=MarpaX::ESLIF::ECMA404::decode
   P:J: participant=Pegex::JSON

 #table20#
 {dataset=>"json:num"}
 +---------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                     | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +---------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Pegex::JSON                     |      3100 |  320      |                 0.00% |             46984.68% | 6.9e-07 |      20 |
 | MarpaX::ESLIF::ECMA404::decode  |     14000 |   70      |               358.01% |             10180.19% |   1e-07 |      21 |
 | JSON::Parser::Regexp            |     22000 |   44      |               618.07% |              6457.08% | 5.3e-08 |      20 |
 | JSON::Tiny::decode_json         |    150000 |    6.6    |              4730.25% |               874.79% | 1.3e-08 |      20 |
 | JSON::PP::decode_json           |    152000 |    6.56   |              4769.40% |               866.95% | 3.3e-09 |      21 |
 | JSON::Decode::Regexp::from_json |    290000 |    3.5    |              9070.48% |               413.44% | 6.7e-09 |      20 |
 | JSON::MaybeXS::decode_json      |   1300000 |    0.77   |             41326.32% |                13.66% | 1.7e-09 |      20 |
 | Cpanel::JSON::XS::decode_json   |   1310000 |    0.766  |             41606.00% |                12.90% | 4.2e-10 |      20 |
 | JSON::Parse::parse_json         |   1452000 |    0.6887 |             46271.06% |                 1.54% | 2.4e-11 |      20 |
 | JSON::XS::decode_json           |   1470000 |    0.678  |             46984.68% |                 0.00% | 2.1e-10 |      20 |
 +---------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                Rate     P:J   MEE:d   JP:R  JT:d_j  JP:d_j  JDR:f_j  JM:d_j  CJX:d_j  JP:p_j  JX:d_j 
  P:J         3100/s      --    -78%   -86%    -97%    -97%     -98%    -99%     -99%    -99%    -99% 
  MEE:d      14000/s    357%      --   -37%    -90%    -90%     -95%    -98%     -98%    -99%    -99% 
  JP:R       22000/s    627%     59%     --    -85%    -85%     -92%    -98%     -98%    -98%    -98% 
  JT:d_j    150000/s   4748%    960%   566%      --      0%     -46%    -88%     -88%    -89%    -89% 
  JP:d_j    152000/s   4778%    967%   570%      0%      --     -46%    -88%     -88%    -89%    -89% 
  JDR:f_j   290000/s   9042%   1900%  1157%     88%     87%       --    -78%     -78%    -80%    -80% 
  JM:d_j   1300000/s  41458%   8990%  5614%    757%    751%     354%      --       0%    -10%    -11% 
  CJX:d_j  1310000/s  41675%   9038%  5644%    761%    756%     356%      0%       --    -10%    -11% 
  JP:p_j   1452000/s  46364%  10064%  6288%    858%    852%     408%     11%      11%      --     -1% 
  JX:d_j   1470000/s  47097%  10224%  6389%    873%    867%     416%     13%      12%      1%      -- 
 
 Legends:
   CJX:d_j: participant=Cpanel::JSON::XS::decode_json
   JDR:f_j: participant=JSON::Decode::Regexp::from_json
   JM:d_j: participant=JSON::MaybeXS::decode_json
   JP:R: participant=JSON::Parser::Regexp
   JP:d_j: participant=JSON::PP::decode_json
   JP:p_j: participant=JSON::Parse::parse_json
   JT:d_j: participant=JSON::Tiny::decode_json
   JX:d_j: participant=JSON::XS::decode_json
   MEE:d: participant=MarpaX::ESLIF::ECMA404::decode
   P:J: participant=Pegex::JSON

 #table21#
 {dataset=>"json:str1k"}
 +---------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                     | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +---------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | JSON::Parser::Regexp            |       150 | 6.9       |                 0.00% |            445739.55% |   1e-05 |      20 |
 | JSON::PP::decode_json           |      2210 | 0.452     |              1421.30% |             29206.57% | 2.1e-07 |      20 |
 | Pegex::JSON                     |      2600 | 0.38      |              1702.24% |             24638.06% |   2e-06 |      20 |
 | JSON::Tiny::decode_json         |     20300 | 0.0494    |             13842.37% |              3097.73% | 1.3e-08 |      20 |
 | MarpaX::ESLIF::ECMA404::decode  |     21000 | 0.047     |             14598.21% |              2933.29% | 6.7e-08 |      20 |
 | JSON::Decode::Regexp::from_json |    240000 | 0.0041    |            167571.35% |               165.90% | 6.7e-09 |      20 |
 | JSON::Parse::parse_json         |    349150 | 0.0028641 |            240240.49% |                85.50% | 2.1e-11 |      20 |
 | JSON::MaybeXS::decode_json      |    533900 | 0.001873  |            367430.08% |                21.31% | 2.9e-11 |      21 |
 | Cpanel::JSON::XS::decode_json   |    536000 | 0.00186   |            369133.12% |                20.75% | 7.3e-10 |      26 |
 | JSON::XS::decode_json           |    647700 | 0.001544  |            445739.55% |                 0.00% | 5.1e-11 |      20 |
 +---------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

               Rate     JP:R  JP:d_j     P:J  JT:d_j  MEE:d  JDR:f_j  JP:p_j  JM:d_j  CJX:d_j  JX:d_j 
  JP:R        150/s       --    -93%    -94%    -99%   -99%     -99%    -99%    -99%     -99%    -99% 
  JP:d_j     2210/s    1426%      --    -15%    -89%   -89%     -99%    -99%    -99%     -99%    -99% 
  P:J        2600/s    1715%     18%      --    -87%   -87%     -98%    -99%    -99%     -99%    -99% 
  JT:d_j    20300/s   13867%    814%    669%      --    -4%     -91%    -94%    -96%     -96%    -96% 
  MEE:d     21000/s   14580%    861%    708%      5%     --     -91%    -93%    -96%     -96%    -96% 
  JDR:f_j  240000/s  168192%  10924%   9168%   1104%  1046%       --    -30%    -54%     -54%    -62% 
  JP:p_j   349150/s  240813%  15681%  13167%   1624%  1541%      43%      --    -34%     -35%    -46% 
  JM:d_j   533900/s  368292%  24032%  20188%   2537%  2409%     118%     52%      --       0%    -17% 
  CJX:d_j  536000/s  370867%  24201%  20330%   2555%  2426%     120%     53%      0%       --    -16% 
  JX:d_j   647700/s  446791%  29174%  24511%   3099%  2944%     165%     85%     21%      20%      -- 
 
 Legends:
   CJX:d_j: participant=Cpanel::JSON::XS::decode_json
   JDR:f_j: participant=JSON::Decode::Regexp::from_json
   JM:d_j: participant=JSON::MaybeXS::decode_json
   JP:R: participant=JSON::Parser::Regexp
   JP:d_j: participant=JSON::PP::decode_json
   JP:p_j: participant=JSON::Parse::parse_json
   JT:d_j: participant=JSON::Tiny::decode_json
   JX:d_j: participant=JSON::XS::decode_json
   MEE:d: participant=MarpaX::ESLIF::ECMA404::decode
   P:J: participant=Pegex::JSON

 #table22#
 {dataset=>"perl:hash_int_100"}
 +-------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant       | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Data::Undump::PPI |        41 |    24     |                 0.00% |            205899.40% | 4.7e-05 |      20 |
 | eval()            |     16000 |     0.063 |             38777.09% |               429.87% | 2.1e-07 |      20 |
 | Data::Undump      |     84000 |     0.012 |            205899.40% |                 0.00% | 1.3e-08 |      21 |
 +-------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

           Rate     DU:P     e   D:U 
  DU:P     41/s       --  -99%  -99% 
  e     16000/s   37995%    --  -80% 
  D:U   84000/s  199900%  425%    -- 
 
 Legends:
   D:U: participant=Data::Undump
   DU:P: participant=Data::Undump::PPI
   e: participant=eval()

 #table23#
 {dataset=>"sereal:hash_int_100"}
 +-----------------------+---------------------+-------------+-----------------------------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant           | dataset             | ds_tags     | p_tags                      | perl | rate (/s) | time (μs) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-----------------------+---------------------+-------------+-----------------------------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Sereal::decode_sereal | sereal:hash_int_100 | deserialize | binary, sereal, deserialize | perl |    120000 |      8.34 |                 0.00% |                 0.00% | 3.3e-09 |      20 |
 +-----------------------+---------------------+-------------+-----------------------------+------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

         Rate     
     120000/s  -- 
 
 Legends:
   : dataset=sereal:hash_int_100 ds_tags=deserialize p_tags=binary, sereal, deserialize participant=Sereal::decode_sereal perl=perl


Benchmark module startup overhead (C<< bencher -m Serializers --module-startup >>):

 #table24#
 +------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | participant            | time (ms) | mod_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | JSON::Decode::Marpa    |     110   |              94   |                 0.00% |               570.99% |   0.00022 |      20 |
 | MarpaX::ESLIF::ECMA404 |      92   |              76   |                14.98% |               483.55% |   0.00013 |      20 |
 | Data::Undump::PPI      |      66.2 |              50.2 |                59.18% |               321.52% | 4.8e-05   |      21 |
 | JSON::Parser::Regexp   |      51   |              35   |               107.40% |               223.52% |   0.00018 |      20 |
 | JSON::Tiny             |      33   |              17   |               223.58% |               107.36% | 4.4e-05   |      20 |
 | JSON::PP               |      29.1 |              13.1 |               262.44% |                85.13% | 2.7e-05   |      20 |
 | Pegex::JSON            |      28   |              12   |               270.48% |                81.12% | 5.7e-05   |      20 |
 | JSON::MaybeXS          |      25   |               9   |               322.92% |                58.65% | 4.8e-05   |      20 |
 | YAML                   |      23   |               7   |               352.70% |                48.22% |   0.00013 |      20 |
 | Storable               |      23   |               7   |               354.09% |                47.77% | 3.9e-05   |      20 |
 | Sereal                 |      23   |               7   |               354.51% |                47.63% | 4.3e-05   |      21 |
 | YAML::Old              |      23   |               7   |               365.71% |                44.08% |   5e-05   |      20 |
 | YAML::XS               |      21   |               5   |               399.83% |                34.24% | 5.1e-05   |      20 |
 | JSON::Parse            |      21   |               5   |               402.72% |                33.47% | 4.7e-05   |      20 |
 | JSON::XS               |      20.4 |               4.4 |               416.87% |                29.82% | 1.5e-05   |      20 |
 | Cpanel::JSON::XS       |      20   |               4   |               423.28% |                28.23% | 4.2e-05   |      20 |
 | YAML::Syck             |      19   |               3   |               443.58% |                23.44% | 2.5e-05   |      20 |
 | JSON::Decode::Regexp   |      19   |               3   |               466.88% |                18.37% |   3e-05   |      20 |
 | JSON::Create           |      19   |               3   |               467.08% |                18.32% | 3.8e-05   |      20 |
 | Data::Undump           |      18   |               2   |               471.53% |                17.40% | 2.6e-05   |      20 |
 | Data::MessagePack      |      18   |               2   |               474.41% |                16.81% | 2.1e-05   |      20 |
 | perl -e1 (baseline)    |      16   |               0   |               570.99% |                 0.00% |   5e-05   |      20 |
 +------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+


Formatted as L<Benchmark.pm|Benchmark> result:

                            Rate  JSON::Decode::Marpa  MarpaX::ESLIF::ECMA404  Data::Undump::PPI  JSON::Parser::Regexp  JSON::Tiny  JSON::PP  Pegex::JSON  JSON::MaybeXS  YAML  Storable  Sereal  YAML::Old  YAML::XS  JSON::Parse  JSON::XS  Cpanel::JSON::XS  YAML::Syck  JSON::Decode::Regexp  JSON::Create  Data::Undump  Data::MessagePack  perl -e1 (baseline) 
  JSON::Decode::Marpa      9.1/s                   --                    -16%               -39%                  -53%        -70%      -73%         -74%           -77%  -79%      -79%    -79%       -79%      -80%         -80%      -81%              -81%        -82%                  -82%          -82%          -83%               -83%                 -85% 
  MarpaX::ESLIF::ECMA404  10.9/s                  19%                      --               -28%                  -44%        -64%      -68%         -69%           -72%  -75%      -75%    -75%       -75%      -77%         -77%      -77%              -78%        -79%                  -79%          -79%          -80%               -80%                 -82% 
  Data::Undump::PPI       15.1/s                  66%                     38%                 --                  -22%        -50%      -56%         -57%           -62%  -65%      -65%    -65%       -65%      -68%         -68%      -69%              -69%        -71%                  -71%          -71%          -72%               -72%                 -75% 
  JSON::Parser::Regexp    19.6/s                 115%                     80%                29%                    --        -35%      -42%         -45%           -50%  -54%      -54%    -54%       -54%      -58%         -58%      -60%              -60%        -62%                  -62%          -62%          -64%               -64%                 -68% 
  JSON::Tiny              30.3/s                 233%                    178%               100%                   54%          --      -11%         -15%           -24%  -30%      -30%    -30%       -30%      -36%         -36%      -38%              -39%        -42%                  -42%          -42%          -45%               -45%                 -51% 
  JSON::PP                34.4/s                 278%                    216%               127%                   75%         13%        --          -3%           -14%  -20%      -20%    -20%       -20%      -27%         -27%      -29%              -31%        -34%                  -34%          -34%          -38%               -38%                 -45% 
  Pegex::JSON             35.7/s                 292%                    228%               136%                   82%         17%        3%           --           -10%  -17%      -17%    -17%       -17%      -25%         -25%      -27%              -28%        -32%                  -32%          -32%          -35%               -35%                 -42% 
  JSON::MaybeXS           40.0/s                 340%                    268%               164%                  104%         32%       16%          12%             --   -7%       -7%     -7%        -7%      -16%         -16%      -18%              -19%        -24%                  -24%          -24%          -28%               -28%                 -36% 
  YAML                    43.5/s                 378%                    300%               187%                  121%         43%       26%          21%             8%    --        0%      0%         0%       -8%          -8%      -11%              -13%        -17%                  -17%          -17%          -21%               -21%                 -30% 
  Storable                43.5/s                 378%                    300%               187%                  121%         43%       26%          21%             8%    0%        --      0%         0%       -8%          -8%      -11%              -13%        -17%                  -17%          -17%          -21%               -21%                 -30% 
  Sereal                  43.5/s                 378%                    300%               187%                  121%         43%       26%          21%             8%    0%        0%      --         0%       -8%          -8%      -11%              -13%        -17%                  -17%          -17%          -21%               -21%                 -30% 
  YAML::Old               43.5/s                 378%                    300%               187%                  121%         43%       26%          21%             8%    0%        0%      0%         --       -8%          -8%      -11%              -13%        -17%                  -17%          -17%          -21%               -21%                 -30% 
  YAML::XS                47.6/s                 423%                    338%               215%                  142%         57%       38%          33%            19%    9%        9%      9%         9%        --           0%       -2%               -4%         -9%                   -9%           -9%          -14%               -14%                 -23% 
  JSON::Parse             47.6/s                 423%                    338%               215%                  142%         57%       38%          33%            19%    9%        9%      9%         9%        0%           --       -2%               -4%         -9%                   -9%           -9%          -14%               -14%                 -23% 
  JSON::XS                49.0/s                 439%                    350%               224%                  150%         61%       42%          37%            22%   12%       12%     12%        12%        2%           2%        --               -1%         -6%                   -6%           -6%          -11%               -11%                 -21% 
  Cpanel::JSON::XS        50.0/s                 450%                    359%               231%                  154%         64%       45%          39%            25%   14%       14%     14%        14%        5%           5%        2%                --         -5%                   -5%           -5%           -9%                -9%                 -19% 
  YAML::Syck              52.6/s                 478%                    384%               248%                  168%         73%       53%          47%            31%   21%       21%     21%        21%       10%          10%        7%                5%          --                    0%            0%           -5%                -5%                 -15% 
  JSON::Decode::Regexp    52.6/s                 478%                    384%               248%                  168%         73%       53%          47%            31%   21%       21%     21%        21%       10%          10%        7%                5%          0%                    --            0%           -5%                -5%                 -15% 
  JSON::Create            52.6/s                 478%                    384%               248%                  168%         73%       53%          47%            31%   21%       21%     21%        21%       10%          10%        7%                5%          0%                    0%            --           -5%                -5%                 -15% 
  Data::Undump            55.6/s                 511%                    411%               267%                  183%         83%       61%          55%            38%   27%       27%     27%        27%       16%          16%       13%               11%          5%                    5%            5%            --                 0%                 -11% 
  Data::MessagePack       55.6/s                 511%                    411%               267%                  183%         83%       61%          55%            38%   27%       27%     27%        27%       16%          16%       13%               11%          5%                    5%            5%            0%                 --                 -11% 
  perl -e1 (baseline)     62.5/s                 587%                    475%               313%                  218%        106%       81%          75%            56%   43%       43%     43%        43%       31%          31%       27%               25%         18%                   18%           18%           12%                12%                   -- 
 
 Legends:
   Cpanel::JSON::XS: mod_overhead_time=4 participant=Cpanel::JSON::XS
   Data::MessagePack: mod_overhead_time=2 participant=Data::MessagePack
   Data::Undump: mod_overhead_time=2 participant=Data::Undump
   Data::Undump::PPI: mod_overhead_time=50.2 participant=Data::Undump::PPI
   JSON::Create: mod_overhead_time=3 participant=JSON::Create
   JSON::Decode::Marpa: mod_overhead_time=94 participant=JSON::Decode::Marpa
   JSON::Decode::Regexp: mod_overhead_time=3 participant=JSON::Decode::Regexp
   JSON::MaybeXS: mod_overhead_time=9 participant=JSON::MaybeXS
   JSON::PP: mod_overhead_time=13.1 participant=JSON::PP
   JSON::Parse: mod_overhead_time=5 participant=JSON::Parse
   JSON::Parser::Regexp: mod_overhead_time=35 participant=JSON::Parser::Regexp
   JSON::Tiny: mod_overhead_time=17 participant=JSON::Tiny
   JSON::XS: mod_overhead_time=4.4 participant=JSON::XS
   MarpaX::ESLIF::ECMA404: mod_overhead_time=76 participant=MarpaX::ESLIF::ECMA404
   Pegex::JSON: mod_overhead_time=12 participant=Pegex::JSON
   Sereal: mod_overhead_time=7 participant=Sereal
   Storable: mod_overhead_time=7 participant=Storable
   YAML: mod_overhead_time=7 participant=YAML
   YAML::Old: mod_overhead_time=7 participant=YAML::Old
   YAML::Syck: mod_overhead_time=3 participant=YAML::Syck
   YAML::XS: mod_overhead_time=5 participant=YAML::XS
   perl -e1 (baseline): mod_overhead_time=0 participant=perl -e1 (baseline)

To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Bencher-Scenario-Serializers>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Bencher-Scenario-Serializers>.

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 CONTRIBUTING


To contribute, you can send patches by email/via RT, or send pull requests on
GitHub.

Most of the time, you don't need to build the distribution yourself. You can
simply modify the code, then test via:

 % prove -l

If you want to build the distribution (e.g. to try to install it locally on your
system), you can install L<Dist::Zilla>,
L<Dist::Zilla::PluginBundle::Author::PERLANCAR>, and sometimes one or two other
Dist::Zilla plugin and/or Pod::Weaver::Plugin. Any additional steps required
beyond that are considered a bug and can be reported to me.

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2021, 2017, 2016, 2015 by perlancar <perlancar@cpan.org>.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Bencher-Scenario-Serializers>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=cut
