#include <stdio.h>
#include <string.h>
#include "wdefs.h"
#include "wutils.h"
#include "wquery.h"
#include "wreply.h"


static StrVal rfmts[] =
{
  { "abridged", R_ABRIDGED },
  { "full",     R_FULL },
  { "format",   R_FULL_FMT },
  { "handle",   R_HANDLE },
  { "mime",     R_MIME },
  { "pointer",  R_POINT },
  { "summary",  R_SUM },

  { (const char *)0, NO_VAL }
};

/* Global response state */

Val response = R_FULL;


/*  
 *
 *
 *                              Internal routines
 *
 *
 */  

static int get_hits(w)
  WItem *w;
{
  int hits = 0;

  if (w)
  {
    hits++;
    while ((w = w->next))
    {
      if (w->format == R_FULL_FMT) hits++;
    }
  }

  return hits;
}


static WItem *fullparse(ifp, w)
  FILE *ifp;
  WItem *w;
{
  WItem *l = w;
  char line[128];

  while (wGetLine(ifp, line, sizeof line))
  {
    const char *end;
    char *name, *str;

    d3fprintf(stderr, "fullparse: read [%s].\n", line);
    if (line[0] == ' ')
    {
      /* Should be attribute/value pair */

      if ( ! getNameStr(line, &name, &str, &end))
      {
        goto failed;
      }
      else
      {
        WItem *t = wNewWItem();

        if ( ! t)
        {
          free(name); free(str);
          d2fprintf(stderr, "fullparse: wNewWItem() failed.\n");
          goto failed;
        }
        else
        {
          t->type = T_REPLY;
          t->format = R_FULL;
          t->lhs = name;
          t->rhs = str;
          l->next = t;
          l = t;
        }
      }
    }
    else if (line[0] == '#')
    {
      /*  
       *  Should be one of:
       *  
       *    "# <template-name> <handle>"
       *  
       *    "# END"
       */      

      if ( ! (name = get_name(line+1, &end)))
      {
        d2fprintf(stderr, "fullparse: get_name() failed on [%s].\n", line+1);
        goto failed;
      }
      else
      {
        if (strcasecmp(name, "end") == 0)
        {
          free(name);
          return w;
        }
        else
        {
          if ( ! (str = get_name(end, &end)))
          {
            free(name);
            d2fprintf(stderr, "fullparse: get_name() failed on [%s].\n", end);
            goto failed;
          }
          else
          {
            WItem *t = wNewWItem();

            if ( ! t)
            {
              free(name); free(str);
              d2fprintf(stderr, "fullparse: wNewWItem() failed.\n");
              goto failed;
            }
            else
            {
              t->type = T_REPLY;
              t->format = R_FULL_FMT;
              t->lhs = name;
              t->rhs = str;
              l->next = t;
              l = t;
            }
          }
        }
      }
    }
    else
    {
      /* Unknown line type */
      d2fprintf(stderr, "fullparse: unknown line type in [%s].\n", line);
      goto failed;
    }
  }

  d2fprintf(stderr, "fullparse: missing [# end] line.\n");

failed:
  wFreeList(w);
  return (WItem *)0;
}


/*  
 *
 *
 *                              External routines
 *
 *
 */  


char *wGetLine(ifp, line, len)
  FILE *ifp;
  char *line;
  int len;
{
  int c;
  int count = 0;
  int cr = 0;

  if (len > 0)
  {
    while (1)
    {
      c = getc(ifp);
      if (c == EOF)
      {
        return (char *)0;
      }
      if (count == len - 1)
      {
        line[count] = '\0';
        ungetc(c, ifp);
        return line;
      }
      else
      {
        switch (c)
        {
        case EOF:
          line[count] = '\0';
          return line;

        case CR:
          line[count++] = c;
          cr = 1;
          if (count == len - 1)
          {
            line[count] = '\0';
            return line;
          }
          break;

        case LF:
          line[count++] = c;
          if (cr)
          {
            line[count-2] = '\0';
            return line;
          }
          break;

        default:
          cr = 0;
          line[count++] = c;
          break;
        }
      }
    }
  }
  return (char *)0;
}


void wReplyPrint(w)
  const WItem *w;
{
  if (w && w->next)
  {
    const char *fstr;
    int hits = atoi(w->lhs);
    
    fstr = getStr(rfmts, w->format);
    if (fstr)
    {
      printf("# %s %d\n", fstr, hits);
      while ((w = w->next))
      {
        /* bug: doesn't handle "summary" + check for bad values? */
        printf("%c %s %s\n", w->format == R_FULL_FMT ? '#' : ' ', w->lhs, w->rhs);
      }
      printf("# end\n");
    }
  }
}


WItem *wReplyParse(ifp)
  FILE *ifp;
{
  Val fval;
  WItem *w = (WItem *)0;
  char line[128];
  char *name;
  char *number;
  const char *end;

  /*  
   *  Skip over leading junk lines
   */  
  do
  {
    if ( ! wGetLine(ifp, line, sizeof line))
    {
      return (WItem *)0;
    }
    d3fprintf(stderr, "wReplyParse: read in skip loop [%s].\n", line);
  }
  while (line[0] != '#');
  d3fprintf(stderr, "wReplyParse: stopped skipping on [%s].\n", line);

  /*  
   *  Should be "# <reply-type> <hits>"
   */  
  if ( ! (name = get_name(line+1, &end)))
  {
    d2fprintf(stderr, "wReplyParse: get_name() failed on [%s].\n", line+1);
    return (WItem *)0;
  }
  else
  {
    if ( ! (number = get_number(end, &end)))
    {
      free(name);
      d2fprintf(stderr, "wReplyParse: get_number() failed on [%s].\n", end);
      return (WItem *)0;
    }
  }

  fval = getVal(rfmts, name);
  if (fval == NO_VAL)
  {
    free(name); free(number);
    d2fprintf(stderr, "wReplyParse: unknown reply format [%s].\n", name);
    return (WItem *)0;
  }

  w = wNewWItem();
  if ( ! w)
  {
    d2fprintf(stderr, "wReplyParse: wNewWItem() failed for header item.\n");
    return (WItem *)0;
  }
  w->type = T_REPLY;
  w->format = F_HEAD;  
  w->lhs = name;
  w->rhs = number;

  if (fval == R_FULL)
  {
    free(name);
    return fullparse(ifp, w);
  }
  else
  {
    d2fprintf(stderr, "wReplyParse: don't handle %s replies, yet.\n", name);
    free(name); free(number);
    wFreeList(w);
    return (WItem *)0;
  }
}


/*  
 *  Given a list of items write a reply to the specified file descriptor.  If
 *  `w->rhs' doesn't seem to contain the number of hits, then we scan the
 *  list to produce the proper count.
 */    

int wReply(w, ofp)
  const WItem *w;
  FILE *ofp;
{
  extern Val response;
  int r = 0;
  char *tmp_str;
  char out_buf[OUT_BUFSIZ];
  char hold_buf[HOLD_BUFSIZ];

  ptr_check(w, "w", "wReply", 0);
  ptr_check(ofp, "ofp", "wReply", 0);

  if (w->type != T_REPLY)
  {
    d2fprintf(stderr, "wReply: header type is not T_REPLY.\n");
  }
  else
  {
    if ( ! w->lhs)
    {
      d2fprintf(stderr, "wReply: lhs in header is NULL; should contain reply format.\n");
    }
    else
    {
      int hits = w->rhs ? atoi(w->rhs) : -1;

      if (hits < 0)
      {
        hits = get_hits(w->next);
      }

      if(hits > FULL_LIMIT)
	 response = R_ABRIDGED;


      switch(response){

	    case R_FULL:
	       tmp_str = "FULL";
	       break;
	    case R_ABRIDGED:
	       tmp_str = "ABRIDGED";
	       break;
      }


      fprintf(ofp, "# %s %d %s", tmp_str, hits, CRLF);
      d3fprintf(stderr, "wReply: wrote [# %s %d].\n", w->lhs, hits);

      while ((w = w->next))
      {
	int len;

	if(w -> type == T_SYSTEM){
          fprintf(ofp, "%s%s", w->lhs, CRLF);
	  continue;
	}

	len = 0;
	hold_buf[0] = '\0';

	switch(response){

	  case R_FULL:
	    if(w->format == R_FULL_FMT)
	    {
	      fprintf(ofp, "# %s %s %s", w->lhs, w->rhs, CRLF);
              d3fprintf(stderr, "wReply: wrote [# %s %s].\n", w->lhs, w->rhs);
	    }
	    else
	    {
              sprintf(hold_buf, " %s: %s", w->lhs, w->rhs);
              d3fprintf(stderr, "wReply: wrote [ %s %s].\n", w->lhs, w->rhs);
	    }
	    break;

	  case R_ABRIDGED:
	    if(w->format == R_FULL_FMT)
	    {
	      sprintf(hold_buf, " %s: %s", w -> lhs, w->rhs);
              d3fprintf(stderr, "wReply: wrote [# %s %s].\n", w->lhs, w->rhs);
	    }
	  default:
	    break;
	}

	/* Handle continuation lines */

	if((len = strlen(hold_buf)) > MAX_LINE){
	   char *buf_end;
	   char *ptr = hold_buf;
	   char a[MAX_LINE];

	   buf_end = ptr + len;

	   strncpy(a, ptr, MAX_LINE - 2);
	   a[MAX_LINE - 2] = '\0';
	   fprintf(ofp, "%s%s", a, CRLF);

	   for(ptr += MAX_LINE - 2; ptr < buf_end; ptr += MAX_LINE - 4){
	      strncpy(a, ptr, MAX_LINE - 4);
	      a[MAX_LINE - 4] = '\0';
	      fprintf(ofp, " +%-s%s", a, CRLF);
	   }
	}
	else
	   if(len)
	      fprintf(ofp, "%s%s", hold_buf, CRLF);

      }

      fprintf(ofp, "# END %s", CRLF);
      d3fprintf(stderr, "wReply: wrote [# END].\n");
      fflush(ofp);
      r = 1;
    }
  }

  return r;
}
