#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include "typedef.h"
#include "db_files.h"
#include "error.h"
#include "master.h"
#include "archie_strings.h"
#include "archie_catalogs.h"


/*
 * read_catalogs_list: read the catalogs file (by default
 * "~archie/etc/catalogs.cf") into the internal system list.  format of
 * file:  <catalog name> <catalogs database pathname> <expand>
 */

status_t read_catalogs_list( catalogs_file, catalogs_list, max_cat)
   file_info_t *catalogs_file;
   catalogs_list_t *catalogs_list;
   int max_cat;

{
   char input_buf[2048];
   char *iptr, *tmp_ptr;
   int lineno;
   char **attriblist, **aptr;
   pathname_t attributes;
   pathname_t typestring, dirstring, accesstr;
   catalogs_list_t *curr_cat;
   int count;

   /* catalogs_list must be zeroed out before being passed */

   ptr_check(catalogs_file, file_info_t, "read_catalogs_list", ERROR);
   ptr_check(catalogs_list, catalogs_list_t, "read_catalogs_list", ERROR);

   if(catalogs_file -> filename[0] == '\0'){

      /* generate default catalogs filename */

      sprintf(catalogs_file -> filename, "%s/%s/%s", get_archie_home(), DEFAULT_ETC_DIR, DEFAULT_CATALOGS_FILENAME);
   }

   if(access(catalogs_file -> filename, R_OK | F_OK) == -1){

      catalogs_list -> cat_name[0] = '\0';
      return(A_OK);
   }

   if(open_file(catalogs_file, O_RDONLY) == ERROR){

      error(A_ERR, "read_catalogs_list", "Can't open catalogs file %s", catalogs_file -> filename);
      return(ERROR);
   }

   count = 0;

   for(iptr = input_buf, lineno = 1; fgets(iptr, input_buf + sizeof(input_buf) - iptr, catalogs_file -> fp_or_dbm.fp) != (char *) NULL; lineno++){

      if((tmp_ptr = strstr(input_buf,CONTINUATION_LINE)) != NULL){
	 *tmp_ptr = '\0';
	 iptr = tmp_ptr;
	 continue;
      }
	 
      /* Recognise comments */

      if((tmp_ptr = strchr(input_buf,COMMENT_CHAR)) != NULL)
	 *tmp_ptr = '\0';

      curr_cat = &catalogs_list[count];
      attributes[0] = '\0';

      if(sscanf(input_buf, "%s %s %s %s", curr_cat -> cat_name, typestring, accesstr, dirstring) < 2){

	 error(A_ERR, "read_catalogs_list", "Error in catalogs file %s, line %d. Insufficient arguments", catalogs_file -> filename, lineno);
	 continue;
      }

      aptr = str_sep(typestring, ',');

      if(strcasecmp(aptr[0], "template") == 0){

	 curr_cat -> cat_type = CAT_TEMPLATE;

	 strcpy(curr_cat -> cat_aux, aptr[1]);
      }

      if(aptr)
	 free_opts(aptr);

      aptr = str_sep(accesstr, ',');

      if(strcasecmp(aptr[0], "wais") == 0){

	 curr_cat -> cat_access = CATA_WAIS;

	 if(strcasecmp(aptr[1], "expand") == 0)
	    curr_cat -> cat_expand = 1;
      }

      if(aptr)
	 free_opts(aptr);

      if(dirstring[0] == '/')
	 strcpy(curr_cat -> cat_db, dirstring);
      else
	 strcpy(curr_cat -> cat_db, master_db_filename(dirstring));


      count++;

   }

   catalogs_list[count].cat_name[0] = '\0';

   return(A_OK);
}

catalogs_list_t *find_in_catalogs(dbname, catalog_list)
   char *dbname;
   catalogs_list_t *catalog_list;
{
   catalogs_list_t *cat;

   ptr_check(dbname, char, "find_in_catalogs", (catalogs_list_t *) NULL);
   ptr_check(catalog_list, catalogs_list_t, "find_in_catalogs", (catalogs_list_t *) NULL);

   for(cat = catalog_list; cat -> cat_name[0] != '\0'; cat++){

      if(strcasecmp(dbname, cat -> cat_name) == 0)
	 return(cat);
   }

   return((catalogs_list_t *) NULL);

}
   
