/*
 * dfile.c - FreeBSD file processing functions for lsof
 */


/*
 * Copyright 1994 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */

#ifndef lint
static char copyright[] =
"@(#) Copyright 1994 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dfile.c,v 1.2 97/10/23 12:20:29 abe Exp $";
#endif


#include "lsof.h"


/*
 * ck_file_arg() - check file arguments
 */

int
ck_file_arg(i, ac, av)
	int i;			/* first file argument index */
	int ac;			/* argument count */
	char *av[];		/* argument vector */
{
	unsigned char ad, an;
	short err = 0;
	char *fnm, *fsnm, *path;
	unsigned char fsmty;
	int ftype, j;
	MALLOC_S l;
	struct mounts *mp;
	static struct mounts **mmp = (struct mounts **)NULL;
	static unsigned char *mmpty = (unsigned char *)NULL;
	int mx, nm;
	static int nma = 0;
	struct stat sb;
	struct sfile *sfp;

#if	defined(HASPROCFS)
	struct procfsid *pfi;
	short pfsnl = -1;
	pid_t pid;
	char *pr;
#endif	/* defined(HASPROCFS) */

	for (; i < ac; i++) {
	    if ((path = Readlink(av[i])) == NULL) {
		err = 1;
		continue;
	    }
	/*
	 * Remove extra terminating `/'.
	 *
	 * Check for file system argument.
	 */
	    for (ftype = 1, mp = Mtab, nm = 0; mp; mp = mp->next) {
		if (strcmp(mp->dir, path) == 0)
		    fsmty = 1;
	        else if (strcmp(mp->fsname, path) == 0)
		    fsmty = 2;
		else
		    continue;
		ftype = 0;
		if (nm >= nma) {
		    nma += 5;
		    l = (MALLOC_S)(nma * sizeof(struct mounts *));
		    if (mmp)
			mmp = (struct mounts **)realloc((MALLOC_P *)mmp, l);
		    else
			mmp = (struct mounts **)malloc(l);
		    if (!mmp) {
			(void) fprintf(stderr,
			    "%s: no space for mount pointers\n", Pn);
			Exit(1);
		    }
		    l = (MALLOC_S)(nma * sizeof(unsigned char));
		    if (mmpty)
			mmpty = (unsigned char *)realloc((MALLOC_P *)mmpty, l);
		    else
			mmpty = (unsigned char *)malloc(l);
		    if (!mmpty) {
			(void) fprintf(stderr,
			    "%s: no space for mount pointer types\n", Pn);
			Exit(1);
		    }
		}
		mmp[nm] = mp;
		mmpty[nm++] = fsmty;
	    }
	/*
	 * Loop through the file system matches.  If there were none, make one
	 * pass through the loop, using simply the path name.
	 */
	    mx = 0;
	    do {
	        if (ftype) {

		/*
		 * For a non-file system path, use the path as the file name
		 * and set a NULL file system name.
		 */
		    fnm = path;
		    fsnm = NULL;
		/*
		 * Stat the path to obtain its characteristics.
		 */
		    if (statsafely(fnm, &sb) != 0) {
			(void) fprintf(stderr, "%s: status error on %s: %s\n",
			    Pn, fnm, strerror(errno));
		        err = 1;
		        break;
		    }
		} else {

		/*
		 * Derive file name, file system name, and a stat(2) buffer
		 * for a mount point.
		 */
		    mp = mmp[mx];
		    if (mmpty[mx] == 1) {
			fnm = path;
			fsnm = mp->fsname;
		    } else {
			fnm = mp->dir;
			fsnm = path;
		    }
		    mx++;

#if	defined(HASPROCFS)
		    if (mp == Mtprocfs) {
			Procsrch = 1;
			continue;
		    }
#endif	/* defined(HASPROCFS) */

		    sb.st_dev = mp->dev;
		    sb.st_rdev = mp->rdev;
		    sb.st_ino = mp->inode;
		    sb.st_mode = mp->mode;
		}
	    /*
	     * Allocate an sfile structure and fill in the type, inode,
	     * find-flag and linkages.
	     */
		if ((sfp = (struct sfile *)malloc(sizeof(struct sfile)))
		== NULL) {
		    (void) fprintf(stderr, "%s: no space for files\n", Pn);
		    Exit(1);
		}
		sfp->next = Sfile;
		Sfile = sfp;
		sfp->type = ftype;
		sfp->i = sb.st_ino;
		sfp->f = 0;
	    /*
	     * Store the file name and file system name pointers in the sfile
	     * structure, allocating space as necessary.
	     */
		if (fnm == NULL || fnm == path) {
		    sfp->name = fnm;
		    an = 0;
		} else {
		    if ((sfp->name = (char *)malloc((MALLOC_S)(strlen(fnm)+1)))
		    == NULL) {
			(void) fprintf(stderr,
			    "%s: no space for file name %s\n", Pn, fnm);
			Exit(1);
		    }
		    (void) strcpy(sfp->name, fnm);
		    an = 1;
		}
		if (fsnm == NULL || fsnm == path) {
		    sfp->devnm = fsnm;
		    ad = 0;
		} else {
		    if ((sfp->devnm=(char *)malloc((MALLOC_S)(strlen(fsnm)+1)))
		    == NULL) {
			(void) fprintf(stderr,
			    "%s: no space for file system name %s\n", Pn, fsnm);
			Exit(1);
		    }
		    (void) strcpy(sfp->devnm, fsnm);
		    ad = 1;
		}
		if ((sfp->aname = (char *)malloc((MALLOC_S)(strlen(av[i]) + 1)))
		== NULL) {
		    (void) fprintf(stderr,
			"%s: no space for argument file name %s\n", Pn, av[i]);
		    Exit(1);
		}
		(void) strcpy(sfp->aname, av[i]);
	    /*
	     * Save the stat() buffer mode value in the sfile structure.
	     * Use st_rdev if the mode value is S_IFBLK or S_IFCHR; otherwise
	     * use st_dev.
	     */
		sfp->mode = sb.st_mode & S_IFMT;
		if (sfp->mode == S_IFBLK || sfp->mode == S_IFCHR)

#if	defined(CKFA_EXPDEV)
		    sfp->dev = rdev;
		else
		    sfp->dev = dev;
#else	/* defined(CKFA_EXPDEV) */
		    sfp->dev = sb.st_rdev;
		else
		    sfp->dev = sb.st_dev;
#endif	/* defined(CKFA_EXPDEV) */

#if	defined(HASPROCFS)
	    /*
	     * See if this is an individual member of a proc file system.
	     */
		if (Mtprocfs == NULL || Procsrch)
		    continue;
		if (pfsnl == -1)
		    pfsnl = strlen(Mtprocfs->dir);
		if (! pfsnl)
		    continue;
		if (strncmp(Mtprocfs->dir, path, pfsnl) != 0)
		    continue;
		if (path[pfsnl] != '/')
		    continue;

# if	!defined(HASPINODEN)
		for (j = pfsnl+1; path[j]; j++) {
		    if ( ! isdigit(path[j]))
			break;
		}
		if (path[j] || (j - pfsnl - 1) != PNSIZ)
		    continue;
# endif	/* !defined(HASPINODEN) */

		if ((pfi = (struct procfsid *)malloc((MALLOC_S)
			    sizeof(struct procfsid)))
		== NULL) {
		    (void) fprintf(stderr, "%s: no space for %s ID: %s\n",
			Pn, Mtprocfs->dir, path);
		    Exit(1);
		}

# if	defined(HASPINODEN)
		pfi->pid = 0;
		pfi->inode = (unsigned long)sb.st_ino;
# else	/* !defined(HASPINODEN) */
		pfi->pid = atoi(&path[pfsnl+1]);
# endif	/* defined(HASPINODEN) */

		pfi->next = Procfsid;
		Procfsid = pfi;
	    /*
	     * Abandon the Sfile entry, lest it be used in is_file_named().
	     */
		Sfile = sfp->next;
		(void) free((FREE_P *)sfp->aname);
		if (ad)
		    (void) free((FREE_P *)sfp->devnm);
		if (an)
		    (void) free((FREE_P *)sfp->name);
		(void) free((FREE_P *)sfp);
#endif	/* HASPROCFS */

	    } while (mx < nm);
	}
	return((int)err);
}
