/* Copyright (c) 1995,1996 NEC Corporation.  All rights reserved.            */
/*                                                                           */
/* The redistribution, use and modification in source or binary forms of     */
/* this software is subject to the conditions set forth in the copyright     */
/* document ("COPYRIGHT") included with this distribution.                   */

/* This file has all the function to do tcp proxying itself.  The only one   */
/* that is visible to the outside world should be HandleTcpConnection.       */
#include "socks5p.h"
#include "daemon.h"
#include "proxy.h"
#include "flow.h"
#include "log.h"
#include "msg.h"

#ifndef INACTIVITY_TIMEOUT
#define INACTIVITY_TIMEOUT 15*60 /* How much inactivity will I tolerate???   */
#endif

static int FlowSetup(S5Packet *buf) {
    char *olddata = buf->data;

    if (buf->data == NULL) {
	buf->data = malloc(GENERICBUFSIZE);
	if (buf->data) S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(10), 0, "Flow Setup: Allocated Buffer");
	buf->len  = GENERICBUFSIZE;
	buf->off  = 0;
    }

    if (buf->len == buf->off) {
	buf->data = realloc(olddata = buf->data, buf->len += GENERICBUFSIZE);
	if (buf->data) S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(10), 0, "Flow Setup: Grew Buffer");
    }

    if (buf->data == NULL) {
	S5LogUpdate(S5LogDefaultHandle, S5_LOG_WARNING, 0, "Flow Setup: Couldn't allocate buffer space");
	if (!olddata) free(olddata);
	return -1;
    }

    return 0;
}

int S5TcpFlowRecv(S5IOInfo *iio, S5IOInfo *oio, S5Packet *packet, int *dir) {
    S5IOHandle fdsbits = ((iio->fd > oio->fd)?iio->fd:oio->fd)+1;
    double timerm = (double)INACTIVITY_TIMEOUT;
    fd_set fds, bu;
    S5IOInfo *io;
    char *string;
    int n;

    FD_ZERO(&bu);
    if (*dir & S5_DIRECTION_OUT) FD_SET(iio->fd, &bu);
    if (*dir & S5_DIRECTION_IN)  FD_SET(oio->fd, &bu);

    if (FlowSetup(packet) < 0) {
	return -1;
    }

    for (fds = bu ; ; fds = bu) {
	struct timeval tout = { INACTIVITY_TIMEOUT, 0 };

	if (!FD_ISSET(iio->fd,  &fds) && !FD_ISSET(oio->fd, &fds)) {
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_WARNING, 0, "Flow Recv: Neither file descriptor is set");
	    return -1;
	}

	switch (select(fdsbits, &fds, NULL,NULL, &tout)) {
	    case -1:
		if (ISSOCKETERROR(EINTR)) continue;
		S5LogUpdate(S5LogDefaultHandle, S5_LOG_WARNING, 0, "Flow Recv: Select failed: %m");
		return -1;
	    case 0:
		S5LogUpdate(S5LogDefaultHandle, S5_LOG_WARNING, 0, "Flow Recv: Select failed: Inactivity timeout");
		return -1;
	}

	if (FD_ISSET(iio->fd, &fds)) {
	    *dir = S5_DIRECTION_OUT;
	    string = "client";
	    io = iio;
	} else if (FD_ISSET(oio->fd, &fds)) {
	    *dir = S5_DIRECTION_IN;
	    string = "server";
	    io = oio;
	} else {
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_WARNING, 0, "Flow Recv: Invalid file descriptor set");
	    return -1;
	}

#define RECV_IOFLAGS S5_IOFLAGS_TIMED|S5_IOFLAGS_RESTART

	S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(10), 0, "Flow Recv: Reading from %s socket", string);
	switch ((n = S5IORecv(io->fd, io, packet->data + packet->off, packet->len - packet->off, 0, RECV_IOFLAGS, &timerm))) {
	    case -1: 
		S5LogUpdate(S5LogDefaultHandle, S5_LOG_WARNING,  0, "Flow Recv: %s Read failed: %m", string);
		return -1;
	    case 0:
		S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(0), 0, "Flow Recv: %s closed connection", string);
		return 0;
	    default:
		S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(10), 0, "Flow Recv: Read %d bytes from %s socket", n, string);
		packet->off += n;
		return n;
	}
    }
}

int S5TcpFlowSend(S5IOInfo *iio, S5IOInfo *oio, S5Packet *packet, int *dir) {
    double timerm = (double)INACTIVITY_TIMEOUT;
    S5IOInfo *io;
    char *string;
    int n;

    switch (*dir) {
	case S5_DIRECTION_OUT:
	    string = "server";
	    io = oio;
	    break;
	case S5_DIRECTION_IN:
	    string = "client";
	    io = iio;
	    break;
	default:
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_WARNING, 0, "Flow Send: Invalid direction: %d", *dir);
	    return -1;
    }

    S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(10), 0, "Flow Send: Writing %d bytes to %s socket", packet->off, string);    

#define SEND_IOFLAGS S5_IOFLAGS_TIMED|S5_IOFLAGS_RESTART|S5_IOFLAGS_NBYTES

    switch ((n = S5IOSend(io->fd, io, packet->data, packet->off, 0, SEND_IOFLAGS, &timerm))) {
	case -1: 
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_WARNING, 0, "Flow Send: %s Write failed: %m", string);
	    return -1;
	case 0:
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(0), 0, "Flow Send: %s closed connection", string);
	    return 0;
	default:
	    S5LogUpdate(S5LogDefaultHandle, S5_LOG_DEBUG(10), 0, "Flow Send: Wrote %d bytes to %s", n, string);
	    packet->off -= n;
	    return n;
    }
}


