/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1993.  The Regents of the University of California.  All rights   */
/*   reserved.                                                             */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XFTP shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XFTP for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <string.h>
#include <sys/param.h>
#include <Xm/SelectioB.h>
#include <Xm/MessageB.h>
#include <Xm/TextF.h>
#include <Xm/Text.h>
#ifdef LLNL
#include <netinet/in.h>
#include <netdb.h>
#endif
#include "rconnect.h"
#include "xftp.h"

#define FTPCTRLPORT 21

#define NONE	0
#define YES		1
#define NO		2

static char passwd[MAXPASSWD+1];
static Widget w_passwordDialog;
static int passwd_response;
static Widget w_accountDialog;
static int account_response;

extern XtAppContext app;
extern struct rconnect_st rconnect;
extern struct st_host_info hinfo[];
extern int add_to_cache_ok;
extern int use_last_dir;

void cb_password_ok();
void cb_password_cancel();
void cb_password();
void cb_account_ok();
void cb_account_cancel();
char *strtok();
char *encrypt_password();
char *decrypt_password();
#ifdef LLNL
struct hostent *gethostbyname();
char * inet_ntoa();
#endif


/*
 * cb_connect - Callback that uses information in remote connection dialog
 *              to attempt a control connection to a remote FTP server.
 */
void
cb_connect(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	char *username;
	char *hostname;
	int host;
	int retval;
	int i;
	int len;
	char *host_part;
	int port_part;

	/* Hide "Remote Connect" dialog */
	XtUnmanageChild(rconnect.w_dialog);

	/* Get host identifier */
	XtVaGetValues(rconnect.w_dialog, XmNuserData, &host, NULL);

	/* Get user name and make sure it's non-null */
	username = XmTextFieldGetString(rconnect.w_userName);
	len = strlen(username);
	for (i=0; i<len; i++)
		if (username[i] != ' ')
			break;
	if (i == len) {
		warning_error("Unable to log onto FTP server.  No user name.");
		return;
	}

	/* Get host name / port address and check format */
	hostname = XmTextFieldGetString(rconnect.w_hostName);
	switch (parse_hostname(hostname, &host_part, &port_part)) {
	case -1:
		warning_error("Unable to log onto FTP server.  No host name.");
		return;
	case -2:
		warning_error("Unable to log onto FTP server.  Bad port format.");
		return;
	case 0:
		XtFree(host_part);
	}

	/* Add user name to cache */
	if (add_to_cache_ok)
		add_to_cache(USER, hostname, username);

	/* Do it */
	retval = do_rconnect(host, hostname, username, NULL, NULL);

	/* Free up memory */
	XtFree(hostname);
	XtFree(username);

	/* Well, are we connected? */
	if (retval < 0)
		return;

	/* This might take some time */
	use_busy_cursor();

	/* Successfully logged in.  Update the display */
	if (update_dir_displays(host, False) < 0) {
		restore_prev_cursor();
		lost_connection(host);
		return;
	}
	update_xfer_controls();
	update_dirname_menu(host);
	update_dirpath(host);
	update_dir_cache(host);
	update_hostname(host);
	update_host_controls(host);
	restore_prev_cursor();
}


/*
 * do_rconnect - Establish connection to remote host.   Use supplied
 *               password and account if non-null, otherwise prompt
 *               user for them.  Returns 0 if successful, else -1.
 *               If a password is supplied, it must be encrypted.
 */
do_rconnect(host, hostname, username, password, account)
int host;
char *hostname;
char *username;
char *password;
char *account;
{
	char *acct = NULL;
	Widget w_passwd;
	Widget w_account;
	char buf[100];
	char *wd;
	int retval;
	struct sl_struct *list;
	char msg[MAXPATHLEN+40];
	char *host_part;
	int port_part;
	char *decrypted;
#ifdef LLNL
	struct in_addr host_inet_addr;
	struct hostent *hostptr;
	char *ptr;
#endif

	/* This might take some time */
	use_busy_cursor();

	/* Initialize password to null string */
	passwd[0] = '\0';

	/* Open connection with FTP server */
	if (parse_hostname(hostname, &host_part, &port_part))
		fatal_error("Bug in do_rconnect()");
	retval = ftp_open(host, host_part, port_part);
	XtFree(host_part);
	if (retval < 0) {
		sprintf(buf, "Unable to connect to %s", hostname);
		warning_error(buf);
		restore_prev_cursor();
		return -1;
	}

	/* Try to login user */
	switch (ftp_user(host, username)) {
	case 0:
		goto logged_on;
	case 1:
		break;
	default:
		warning_error("Unable to log onto FTP server");
		close(hinfo[host].ctrl_fd);
		restore_prev_cursor();
		return -1;
	}

	/* Password required */
	if (password) {   /* Use last password */
		decrypted = decrypt_password(password);
		strcpy(passwd, decrypted);
		bzero(decrypted, strlen(decrypted));
		XtFree(decrypted);
	} else {   /* Get password from user */
		create_password_dialog(rconnect.w_dialog);
		w_passwd = XmSelectionBoxGetChild(w_passwordDialog, XmDIALOG_TEXT);
		XmTextSetString(w_passwd, "");
		passwd_response = NONE;
		XtManageChild(w_passwordDialog);
		use_normal_cursor();
		add_dialog_to_list(w_passwordDialog);
		XmProcessTraversal(w_passwd, XmTRAVERSE_CURRENT);
		XmProcessTraversal(w_passwd, XmTRAVERSE_CURRENT);
		while (passwd_response == NONE)
			XtAppProcessEvent(app, XtIMAll);
		restore_prev_cursor();
		if (passwd_response == NO) {
			close(hinfo[host].ctrl_fd);
			restore_prev_cursor();
			return -1;
		}
	}

	/* Send password to FTP server */
	retval = ftp_password(host, passwd);
	switch (retval) {
	case 0:
		goto logged_on;
	case 1:
		break;
	default:
		warning_error("Unable to log onto FTP server");
		close(hinfo[host].ctrl_fd);
		restore_prev_cursor();
		bzero(passwd, MAXPASSWD);
		return -1;
	}

	/* Account required.  Get it from user */
	if (account)   /* Use last account */
		acct = XtNewString(account);
	else {    /* Get account from user */
		create_account_dialog(rconnect.w_dialog);
		w_account = XmSelectionBoxGetChild(w_accountDialog, XmDIALOG_TEXT);
		XmTextSetString(w_account, "");
		account_response = NONE;
		XtManageChild(w_accountDialog);
		add_dialog_to_list(w_accountDialog);
		use_normal_cursor();
		XmProcessTraversal(w_account, XmTRAVERSE_CURRENT);
		XmProcessTraversal(w_account, XmTRAVERSE_CURRENT);
		while (account_response == NONE)
			XtAppProcessEvent(app, XtIMAll);
		restore_prev_cursor();
		if (account_response == NO) {
			close(hinfo[host].ctrl_fd);
			restore_prev_cursor();
			return -1;
		}
		acct = XmTextGetString(w_account);
	}

	/* Send account to FTP server */
	if (ftp_account(host, acct) < 0) {
		warning_error("Unable to log onto FTP server");
		XtFree(acct);
		close(hinfo[host].ctrl_fd);
		restore_prev_cursor();
		return -1;
	}

logged_on:

	/* Determine current working directory */
	if (use_last_dir) {
		retrieve_cache(DIRECTORY, hostname, &list, False);
		if (list->nentries > 0) {
			if (remote_cd(host, list->entries[0], False, True) != 0) {
				sprintf(msg, "Unable to change directory to %s",
					list->entries[0]);
				warning_error(msg);
			}
		}
		release_string_list(list);
	}
	if (remote_pwd(host, &wd) < 0) {
		warning_error("Unable to get working directory.  Will log out");
		close(hinfo[host].ctrl_fd);
		restore_prev_cursor();
		if (acct)
			XtFree(acct);
		return -1;
	}

	/* Success */
	hinfo[host].type = REMOTE;
	XtFree(hinfo[host].wd);
	hinfo[host].wd = wd;
	XtFree(hinfo[host].hostname);
	hinfo[host].hostname = XtNewString(hostname);
	XtFree(hinfo[host].username);
	hinfo[host].username = XtNewString(username);
    if (hinfo[host].move_wd) {
        XtFree(hinfo[host].move_wd);
        hinfo[host].move_wd = NULL;
    }
	if (hinfo[host].password)
		XtFree(hinfo[host].password);
	hinfo[host].password = encrypt_password(passwd);
	bzero(passwd, MAXPASSWD);
	if (hinfo[host].account)
		XtFree(hinfo[host].account);
	hinfo[host].account = acct;

#ifdef LLNL  /* Have to special-case SCF Archive */
	hinfo[host].scf_archive = False;
    if ((host_inet_addr.s_addr = (u_long)inet_addr(hostname)) == -1) {
        if ((hostptr = gethostbyname(hostname)) == NULL) {
			restore_prev_cursor();
			return 0;
		}
        host_inet_addr.s_addr =
			((struct in_addr*)hostptr->h_addr_list[0])->s_addr;
    }
	ptr = inet_ntoa(host_inet_addr);
	if (!strcmp(ptr, "130.106.127.42") || !strcmp(ptr, "130.106.127.43") ||
		!strcmp(ptr, "130.106.204.42") || !strcmp(ptr, "130.106.204.43") ||
		!strcmp(ptr, "130.106.205.43"))
		hinfo[host].scf_archive = True;
#endif

	restore_prev_cursor();
	return 0;
}


/*
 * create_password_dialog - Create dialog that prompts user for password.
 */
create_password_dialog(w_parent)
Widget w_parent;
{
	static int initialized = False;
	int i;
	Arg args[1];
	Widget widget;

    /* Create dialog only once */
    if (initialized)
        return;
    initialized = True;

	/* Create prompt dialog to get password */
	i = 0;
	XtSetArg(args[i], XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL); i++;
	w_passwordDialog = XmCreatePromptDialog(w_parent, "password", args, i);
	XtAddCallback(w_passwordDialog, XmNokCallback, cb_password_ok, NULL);
	XtAddCallback(w_passwordDialog, XmNcancelCallback, cb_password_cancel,NULL);

	/* Intercept characters written to dialog's textfield widget */
	widget = XmSelectionBoxGetChild(w_passwordDialog, XmDIALOG_TEXT);
	XtAddCallback(widget, XmNmodifyVerifyCallback, cb_password, NULL);
	XtAddCallback(widget, XmNmotionVerifyCallback, cb_password, NULL);
	XtVaSetValues(widget, XmNverifyBell, False, NULL);

	/* Don't display dialog's "Help" button */
	widget = XmSelectionBoxGetChild(w_passwordDialog, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(widget);

	/* Add callback for the WM_DELETE_WINDOW protocol */
	add_wm_delete_window_cb(w_passwordDialog, cb_password_cancel, NULL);

	/* Kludge to add title bar under OLWM */
	AddOLWMDialogFrame(w_passwordDialog);
}


/*
 * cb_password_ok - Callback that notes that the user has finished typing
 *                  in his/her password.
 */
void
cb_password_ok(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	passwd_response = YES;
}


/*
 * cb_password_cancel - Callback that notes that the user wishes to abort
 *                      connection process (while in password dialog).
 */
void
cb_password_cancel(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	passwd_response = NO;
	XtUnmanageChild(w_passwordDialog);
}


/*
 * cb_password - Callback that individually processes each password 
 *               character as it is typed in.
 */
void
cb_password(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	XmTextVerifyCallbackStruct *cbs = (XmTextVerifyCallbackStruct *)call_data;
	int i;
	int passwd_len = strlen(passwd);

	/* Assure that insertion cursor is always at end of password */
	if (cbs->reason == XmCR_MOVING_INSERT_CURSOR) {
		cbs->doit = False;
		return;
	}

	/* Allow deletions only at end of password */
	if (cbs->text->ptr == NULL) {
		if (passwd_len > 0)
			passwd[--passwd_len] = '\0';
		return;
	}

	/* Add character(s) to end of password */
	for (i=0; i<cbs->text->length; i++)
		if (passwd_len < MAXPASSWD) {
			passwd[passwd_len++] = cbs->text->ptr[i];
			passwd[passwd_len] = '\0';
			cbs->text->ptr[i] = '%';
		} else {
			cbs->doit = False;
			beep();
		}
}


/*
 * create_account_dialog - Creates the "Get Account" dialog.
 */
create_account_dialog(w_parent)
Widget w_parent;
{
	static int initialized = False;
	int i;
	Arg args[1];
	Widget widget;

    /* Create dialog only once */
    if (initialized)
        return;
    initialized = True;

	/* Create prompt dialog to get account */
	i = 0;
	XtSetArg(args[i], XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL); i++;
	w_accountDialog = XmCreatePromptDialog(w_parent, "account", args, i);
	XtAddCallback(w_accountDialog, XmNokCallback, cb_account_ok, NULL);
	XtAddCallback(w_accountDialog, XmNcancelCallback, cb_account_cancel, NULL);

	/* Don't display dialog's "Help" button */
	widget = XmSelectionBoxGetChild(w_accountDialog, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(widget);

	/* Add callback for the WM_DELETE_WINDOW protocol */
	add_wm_delete_window_cb(w_accountDialog, cb_account_cancel, NULL);

	/* Kludge to add title bar under OLWM */
	AddOLWMDialogFrame(w_accountDialog);
}


/*
 * cb_account_ok - Callback that notes that the user has finished typing
 *                 in his/her account.
 */
void
cb_account_ok(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	account_response = YES;
}


/*
 * cb_account_cancel - Callback that notes that the user wishes to abort
 *                     connection process (while in account dialog).
 */
void
cb_account_cancel(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	account_response = NO;
	XtUnmanageChild(w_accountDialog);
}


/*
 * parse_hostname - Parse "hostname" into its host name and port address
 *                  components.  "hostname" can either consist of a blank-
 *                  delimited host name / port address pair, or just a
 *                  host name.  If the port address part is not given,
 *                  the default control connection port will be used.
 *                  If successful, "host_part" is set to point to a
 *                  string (which the caller can free by calling XtFree()),
 *                  "port_part" is set to the specified port, and a value
 *                  of 0 is returned.  If no host name is given, -1 is
 *                  returned.  If an invalid port designator is given,
 *                  -2 is returned.
 */
parse_hostname(hostname, host_part, port_part)
char *hostname;
char **host_part;
int *port_part;
{
    char *temp;
    char *hp;
    char *pp;
    int port;
    int len;
    int i;

    /* Make copy of hostname */
    temp = XtNewString(hostname);

    /* Get host part */
    if ((hp = strtok(temp, " ")) == NULL) {
        XtFree(temp);
        return -1;
    }

    /* Get port part */
    if ((pp = strtok(NULL, " ")) == NULL)
        port = FTPCTRLPORT;
    else {
        port = 0;
        len = strlen(pp);
        for (i=0; i<len; i++)
            if (pp[i] >= '0' && pp[i] <= '9')
                port = 10*port+(pp[i]-'0');
            else {
                XtFree(temp);
                return -2;
            }
        if (strtok(NULL, " ") != NULL) {
            XtFree(temp);
            return -2;
        }
    }

    /* Everything looks good */
    *host_part = XtNewString(hp);
    XtFree(temp);
    *port_part = port;
    return 0;
}

