/*
 * TODO:    !!!
 *  - In wildcard hostname matching, allow *.dom.ain to match dom.ain
 *  - makefdlist listens for packets even on ports where no instances are
 *	registered; maybe fanouts should have usage count of instances
 *	which are live & receiving; makefdlist skips those with usecount==0.
 *  - If instance has partner with fixed port, and second session tries to
 *      grab it, refuse.  This could come from instance of relay with
 *	wildcarded LHS and specified localport in RHS
 *  - Map addresses back to names in syslog lines. (?)
 *  - Why doesn't FIONBIO work?
 *  - There may be some confusion with ownport.  Is this being handled right?
 *
 * TODO: (maybe)
 *  - Add comparison options for ports: >=23, <1024, etc.
 *  - Allow filtering on destination of packets from encapsulated ports.
 *  - Allow config lines to share collections of hosts, to avoid repeating
 *	multiple services allowed from multiple hosts?
 *  - Doesn't check DNS consistency if hostname == "*".  Is this OK?
 *  - Add dump-state command (SIGUSR1?)
 *
 * Copyright 1993 Tom Fitzgerald <fitz@wang.com>; permission to use, modify
 * and distribute this source is granted provided only that this notice is
 * retained intact.
 *
 * Parts of this program were drawn from tcp-wrapper by Wietse Venema.
 *
 * Version 0.2:  Probably barely beta quality.
 */

/*
 * Terminology:
 * A fanout is a structure controlling access to a single unencapsulated
 *	local port.  All "relay" lines which share access to a localport (on
 *	either side) share a fanout structure.  On receiving a packet, the
 *	fanout decides which config line or instance is to get the packet.
 *	Fanout is created from config file and has generics (also created from
 *	config file) and instances (created on the fly) hanging from it, for
 *	all config file directives and connections through this port.
 *	Fanout owns the socket for the port.
 * An encaps defines a single local port mentioned in "encapsulate" lines from
 *	the config file.  Has generics hanging from it, one for each
 *	"encapsulate" directive, specifying the wildcarded name or addr that
 *	can use the encapsulator.  Also has instances, connections set up
 *	through the specified localport.  Created only from config file.
 * A plex (multiplexor) is either an encaps or a fanout; it controls a single
 *	local port of either kind.
 * A generic is a single remote host/port specification.  The spec can be
 *	wildcarded ("*.dom.ain" or address/mask), but even if it's not, it is
 *	still treated as a generic until a packet arrives, to handle multi-
 *	homed hosts.  A generic hanging from an encaps has no partner, since
 *	the partner is determined by the incoming packet.  A generic hanging
 *	from a fanout has a partner which is a template of the far side of
 *	the connection for each instance created from the generic.
 * An instance is a single exactly matched remote address/port, local port
 *	set.  It receives packets and is matched with a partner instance which
 *	is where the packets are sent out to.  This is the one assoc type
 *	which is created on the fly.
 * An assoc (association) is any one of the above; half of a conversation.
 */


#include <stdio.h>
#include <sys/types.h>
#include <signal.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <syslog.h>
#include <poll.h>
#include <string.h>
#ifdef sun
#include <sys/filio.h>
#endif
#ifdef AIX
#include <sys/ioctl.h>
#endif
#include <pwd.h>
#include "udprelay.h"

#define FALSE 0
#define TRUE  1

#ifndef INADDR_NONE
#define INADDR_NONE ((long) 0xffffffff)
#endif

extern char *malloc ();

#define Debug		1
#define LogInstances	1
#ifdef _NFILE
#define MaxSockets	_NFILE		/* Max no. of open sockets */
#else
#define MaxSockets	60
#endif
#define InitPktBufLen	2048

/*
 * Return TRUE if both addresses point to the same address/port pair
 */
#define sameaddr(a1,a2)	((a1)->sin_addr.s_addr == (a2)->sin_addr.s_addr \
			  && (a1)->sin_port == (a2)->sin_port)

typedef struct assoc {
	struct assoc	*next;		/* Global chain */
	struct assoc	*partner;
	struct assoc	*generics;	/* Used fanout or encaps */
	struct assoc	*instances;	/* Used if fanout or encaps */
	struct assoc	*nextinst;	/* Used if generic or instance */
	struct assoc	*plex;		/* Fanout/encaps for this localport */
	SockIn		sin;		/* Far end of connection */
	struct in_addr	addrmask;
	char		*hostname;	/* Hostname, may include wildcard */
	u_short		localport;	/* 0 if "any" and not yet open */
	int		socket;		/* -1 if not open; copy if instance */
	int		fanout;		/* 1 if this just runs a localport */
	int		receiver;	/* 1 means this assoc receives pkts */
	int		encaps;		/* 1 if pkts are encapsulated */
	int		transient;	/* 1 if session will time out */
	int		ownport;	/* 0 if port is copy of plex's */
	time_t		lastaccess;	/* timestamp */
} Assoc;

char		*pktbuf = NULL;
int		pktbuflen = InitPktBufLen;	/* Packet buffer size */

Assoc		*assoclist = NULL;	/* Global chain of assocs */

struct pollfd	fdlist [MaxSockets];
Assoc		*fdlistassocs [MaxSockets];
int		fdlistlen;
int		idletime = 1800;	/* Session timeout: 30 minutes */


#if Debug
int			debug = 0;
#define	DBPRINTF(n,X)	if (debug >= n) printf X
#else
#define	DBPRINTF
#endif


/*
 * Re-allocate packet buffer for bigger packets (use smallest power of 2
 * that's at least as large as len).  If len == 0, allocate initial buffer.
 * Return TRUE if ok, FALSE if failed.
 */
int makepktbuf (len)
	int		len;
{
	char		*newbuf;

	if (len > 0) {
		while (pktbuflen < len)
			pktbuflen <<= 1;
		syslog (LOG_WARNING, "Boosting buf size to %d\n", pktbuflen);
	}
	if ((newbuf = malloc (pktbuflen)) == NULL) {
		syslog (LOG_ERR, "malloc (%d) failed, out of memory", len);
		return (FALSE);
	}
	if (pktbuf != NULL)
		free (pktbuf);
	pktbuf = newbuf;
	return (TRUE);
}


/*
 * Allocate an assoc
 */
Assoc *getassoc ()
{
	Assoc		*assoc;

	assoc = (Assoc *) malloc (sizeof (Assoc));
	if (assoc == NULL)
		return (NULL);
	bzero ((char *) assoc, sizeof (Assoc));
	assoc->socket		= -1;
	assoc->sin.sin_family	= AF_INET;
	assoc->next		= assoclist;	/* Thread it on global list */
	assoclist = assoc;
	return (assoc);
}


/*
 * Delete an instance assoc, removing it from the global chain and from the
 * list of instances hanging off the same plex.
 */
void deleteassoc (assoc)
	Assoc		*assoc;
{
	Assoc		**prev;

	/*
	 * Remove from global chain
	 */
	for (prev = &assoclist; *prev != assoc; prev = &(*prev)->next)
		;
	*prev = assoc->next;
	/*
	 * Remove from chain of instances hanging off the same multiplex
	 */
	if (assoc->plex != NULL) {
		for (prev = &assoc->plex->instances; *prev != assoc;
		  prev = &(*prev)->nextinst)
			;
		*prev = assoc->nextinst;
	}
	free (assoc);
}


/*
 * Return the fanout associated with a particular local port
 */
Assoc *findfanout (port)
	u_short		port;
{
	Assoc		*assoc;

	for (assoc = assoclist; assoc != NULL; assoc = assoc->next)
		if (assoc->fanout && assoc->localport == port)
			break;
	return (assoc);
}


/*
 * Rebuild the socket list from a (potentially new) set of assocs.
 * Sets fdlist, fdlistassocs and fdlistlen
 */
void makefdlist ()
{
	int		j;
	Assoc		*assoc;

	j = 0;
	for (assoc = assoclist; assoc != NULL; assoc = assoc->next) {
		/* !!! This test picks up fanouts with no receiving
		   instances or generics; should filter these out. */
		if (assoc->ownport && assoc->socket >= 0
		  && (assoc->fanout || assoc->encaps || assoc->receiver)) {
			fdlist [j].fd = assoc->socket;
			fdlist [j].events = POLLIN;
			fdlistassocs [j] = assoc;
			DBPRINTF (2, ("polling socket %d, assoc 0x%x\n",
			  assoc->socket, assoc));
			++j;
		}
	}
	if (j == 0) {
		syslog (LOG_ERR, "No sockets to listen on, exiting.");
		exit (1);
	}
	fdlistlen = j;
}


/*
 * Locate any instances that have not been used in the last idletime seconds
 * and delete them and their partners.  Returns number of milliseconds to
 * next timeout, or -1 if no timeouts pending.
 */
int checktimeouts ()
{
	Assoc		*assoc, *next, *partner;
	time_t		curtime, nexttime;
	int		receivergone;

	curtime = time (NULL);
	nexttime = -1;
	receivergone = FALSE;
	for (assoc = assoclist; assoc != NULL; assoc = next) {
		next = assoc->next;
		if (!assoc->transient)
			continue;		/* Don't timeout this one */
		if (assoc->lastaccess + idletime <= curtime) {
			partner = assoc->partner;
			DBPRINTF (1, ("Timed out assoc 0x%x with socket %d\n",
			  assoc, assoc->socket));
			DBPRINTF (1, ("...and partner 0x%x with socket %d\n",
			  partner, partner->socket));
			if (partner->ownport && partner->socket >= 0) {
				if (partner->receiver)
					receivergone = TRUE;
				DBPRINTF (1, ("closing socket %d\n",
				  partner->socket));
				close (partner->socket);
			}
			deleteassoc (partner);
			/*
			 * Note that "partner" might have been "next"
			 */
			next = assoc->next;
			deleteassoc (assoc);
		}
		else if (nexttime == -1
		  || assoc->lastaccess + idletime < nexttime)
			nexttime = assoc->lastaccess + idletime;
	}
	if (receivergone)
		makefdlist ();
	if (nexttime != -1)
		nexttime = (nexttime - curtime) * 1000;
	return (nexttime);
}


/*
 * Open socket - return TRUE for success, FALSE for failure
 */
int opensocket (assoc)
	Assoc		*assoc;
{
	int		sock, one, len;
	SockIn		myaddr;

	DBPRINTF (1, ("opensocket assoc=0x%x, localport=%d\n", assoc,
	  htons (assoc->localport)));
	if ((sock = socket (AF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) {
		syslog (LOG_ERR, "socket: %m");
		return (FALSE);			/* Probably out of sockets */
	}
	one = 1;
	/*
	 * Set non-blocking, just in case.  (We'll flag an error if we try
	 * to read from a socket with nothing available).
	 */
	if (ioctl (sock, FIONBIO, &one) < 0)
		syslog (LOG_ERR, "ioctl (%d, FIONBIO, %d), %m", sock, one);
		/* If error, continue anyway.... */
	bzero (&myaddr, sizeof (myaddr));
	myaddr.sin_family = AF_INET;
	myaddr.sin_port = assoc->localport;
	myaddr.sin_addr.s_addr = INADDR_ANY;
	if (bind (sock, &myaddr, sizeof (myaddr)) < 0) {
		syslog (LOG_ERR, "bind, %m");
		exit (1);
	}
	if (assoc->localport == INADDR_ANY) {
		len = sizeof (myaddr);
		if (getsockname (sock, &myaddr, &len) < 0) {
			syslog (LOG_ERR, "sockname, %m");
			exit (1);
		}
		assoc->localport = myaddr.sin_port;
		DBPRINTF (1, ("opensocket: localport bound to port %d\n",
		  ntohs (assoc->localport)));
	}
	DBPRINTF (1, ("opened socket %d for assoc 0x%x\n", sock, assoc));
	assoc->socket = sock;
	assoc->ownport = TRUE;
	return (TRUE);
}


int fixhostname (assoc)
	Assoc		*assoc;
{
	struct hostent	*hp;

	if ((hp = gethostbyname (assoc->hostname)) == NULL) {
		syslog (LOG_WARNING, "failed to gethostbyname (%s)",
		  assoc->hostname);
		return (FALSE);
	}
	memcpy ((char *) &assoc->sin.sin_addr, hp->h_addr_list [0],
	  sizeof (assoc->sin.sin_addr));
	return (TRUE);
}


char *skipwhite (cp)
	char		*cp;
{
	while (*cp == ' ' || *cp == '\t')
		++cp;
	return (cp);
}

char *findeos (cp)
	char		*cp;
{
	while (*cp != ' ' && *cp != '\t' && *cp != '\0')
		++cp;
	return (cp);
}


void parsehostname (lscanp, assoc, lno)
	char		**lscanp;
	Assoc		*assoc;
	int		lno;
{
	char		*lscan, *end;
	struct hostent	*host;

	lscan = *lscanp;
	DBPRINTF (2, ("parsehostname ('%s', 0x%x, %d)\n", lscan, assoc, lno));
	end = findeos (lscan);
	if (*end)
		*end++ = '\0';
	assoc->sin.sin_addr.s_addr = inet_addr (lscan);
	if (assoc->sin.sin_addr.s_addr == INADDR_NONE) {
		assoc->sin.sin_addr.s_addr = INADDR_ANY;
		/*
		 * Must be text hostname, copy it.
		 */
		assoc->hostname = malloc (strlen (lscan) + 1);
		if (assoc->hostname == NULL) {
			syslog (LOG_ERR, "parsehostname: out of memory");
			exit (1);
		}
		strcpy (assoc->hostname, lscan);
		DBPRINTF (2, ("parsehostname, hostname= '%s'\n", lscan));
		*lscanp = end;
		return;
	}
	/*
	 * Got an IP address, check for mask
	 */
	lscan = skipwhite (end);
	end = findeos (lscan);
	if (end - lscan != 4 || strncmp (lscan, "mask", 4) != 0) {
		assoc->addrmask.s_addr = INADDR_NONE;
		DBPRINTF (2, ("parsehostname got addr=%s, no mask\n",
		  inet_ntoa (assoc->sin.sin_addr)));
		*lscanp = lscan;
		return;				/* No mask */
	}
	lscan = skipwhite (end);		/* Get mask value */
	end = findeos (lscan);
	if (*end)
		*end++ = '\0';
	assoc->addrmask.s_addr = inet_addr (lscan);
	DBPRINTF (2, ("parsehostname got addr=%s, mask=%s\n",
	  inet_ntoa (assoc->sin.sin_addr), lscan));
	*lscanp = end;
	return;
}


/*
 * Return a UDP portnumber pointed at by *lscanp, or INADDR_ANY if it
 * matches wildstr.  Adjusts lscanp to point after end of scanned string.
 */
u_short getportnum (lscanp, wildstr, lno)
	char		**lscanp;
	char		*wildstr;
	int		lno;
{
	struct servent	*servent;
	char		*lscan, *end;

	lscan = skipwhite (*lscanp);
	end = findeos (lscan);
	*end++ = '\0';
	DBPRINTF (2, ("getportnum ('%s', '%s', %d)\n", lscan, wildstr, lno));
	*lscanp = end;
	if (wildstr != NULL && strcmp (lscan, wildstr) == 0)
		return (INADDR_ANY);
	else if ((servent = getservbyname (lscan, "udp")) != NULL)
		return (servent->s_port);
	else if (!isdigit (*lscan)) {
		syslog (LOG_ERR, "Bad port number on line %d: %s", lno, lscan);
		exit (1);
	}
	return (htons (atoi (lscan)));
}


/*
 * Scan a single hostname, remote-port, local-port triplet.  Return pointer
 * to end of string.
 */
char *parsehost (lscan, assoc, lno)
	char		*lscan;
	Assoc		*assoc;
	int		lno;
{
	DBPRINTF (2, ("parsehost ('%s', 0x%x, %d)\n", lscan, assoc, lno));
	parsehostname (&lscan, assoc, lno);
	assoc->sin.sin_port = getportnum (&lscan, "*", lno);
	assoc->localport = getportnum (&lscan, "any", lno);
	DBPRINTF (2, (
	  "parsehost: name=%s, host=%s, remport=%d, localport=%d\n",
	  assoc->hostname, inet_ntoa (assoc->sin.sin_addr),
	  ntohs (assoc->sin.sin_port), ntohs (assoc->localport)));
	return skipwhite (lscan);
}


/*
 * Parse a config file "relay" command line.  lscan points after the "relay".
 * Creates a generic assoc and partner, hooked into the appropriate fanouts.
 * Exits on error.
 */
void parserelay (lscan, lno)
	char		*lscan;
	int		lno;
{
	Assoc		*assoc, *partner, *fanout, **scan;
	int		oneway;
	char		*end;

	lscan = skipwhite (lscan);
	if ((assoc = getassoc ()) == NULL || (partner = getassoc ()) == NULL) {
		syslog (LOG_ERR, "Out of memory, exiting");
		exit (1);
	}
	lscan = parsehost (lscan, assoc, lno);
	if (assoc->localport == INADDR_ANY) {
		syslog (LOG_ERR, "Missing localport on line %d", lno);
		exit (1);
	}
	if ((fanout = findfanout (assoc->localport)) == NULL) {
		DBPRINTF (2, ("Making new fanout for port %d\n",
		  ntohs (assoc->localport)));
		if ((fanout = getassoc ()) == NULL) {
			syslog (LOG_ERR, "Out of memory, exiting");
			exit (1);
		}
		fanout->fanout = TRUE;
		fanout->localport = assoc->localport;
		if (!opensocket (fanout))
			exit (1);
	}
	/*
	 * Append to end of generics list, so list gets searched in the
	 * order in which things were specified in the config file
	 */
	for (scan = &fanout->generics; *scan; scan = &(*scan)->nextinst)
		;
	*scan = assoc;
	assoc->plex = fanout;
	assoc->receiver = TRUE;
	/*
	 * Same thing again, for the partner.  (Even though the partner can't
	 * be wildcarded, we delay resolving the IP address until instance
	 * creation time, so that if the partner's IP address is changed,
	 * udprelay catches the change without needing to be restarted.)
	 */
	assoc->partner = partner;
	lscan = parsehost (lscan, partner, lno);
	partner->partner = assoc;
	if (partner->sin.sin_port == INADDR_ANY) {
		syslog (LOG_ERR, "Missing 2nd port number, line %d", lno);
		exit (1);
	}
	if (partner->addrmask.s_addr != INADDR_ANY
	  && partner->addrmask.s_addr != INADDR_NONE) {
		syslog (LOG_ERR, "Can't use mask on relay RHS, line %d", lno);
		exit (1);
	}
	oneway = FALSE;
	lscan = skipwhite (lscan);
	if (*lscan) {
		end = findeos (lscan);
		if (*end != '\0')
			*end++ = '\0';
		if (strcmp (lscan, "oneway") != 0) {
			syslog (LOG_ERR, "Can't parse line %d", lno);
			exit (1);
		}
		oneway = TRUE;
		lscan = end;
	}
	if (*lscan) {
		syslog (LOG_ERR, "Too much stuff on line %d", lno);
		exit (1);
	}
	/* !!! If LHS is wildcarded and localport2 is specified, then what?
	  can only allow a single session at a time to use the relay (how
	  do we notify others that session is in use?  ICMP?), or can force
	  oneway */
	if (partner->localport != INADDR_ANY) {
		if ((fanout = findfanout (partner->localport)) == NULL) {
			DBPRINTF (2, ("Making new fanout for port %d\n",
			  ntohs (partner->localport)));
			if ((fanout = getassoc ()) == NULL) {
				syslog (LOG_ERR, "Out of memory, exiting");
				exit (1);
			}
			fanout->fanout = TRUE;
			fanout->localport = partner->localport;
			if (!opensocket (fanout))
				exit (1);
		}
		/*
		 * Point plex back to the fanout so instances of this generic
		 * get the right plex pointer; however, we leave it off the
		 * list of generics for the plex so that it doesn't get found
		 * for uninstantiated connections.  (Sessions can be initiated
		 * from right-to-left).
		 */
		partner->plex = fanout;
	}
	if (!oneway)
		partner->receiver = TRUE;
}


void parseencaps (lscan, lno)
	char		*lscan;
	int		lno;
{
	Assoc		*assoc, *encaps;

	lscan = skipwhite (lscan);
	if ((assoc = getassoc ()) == NULL) {
		syslog (LOG_ERR, "Out of memory, exiting");
		exit (1);
	}
	parsehostname (&lscan, assoc, lno);
	assoc->localport = getportnum (&lscan, NULL, lno);
	if (assoc->localport == INADDR_ANY) {
		syslog (LOG_ERR, "Missing localport on line %d", lno);
		exit (1);
	}
	for (encaps = assoclist; encaps != NULL; encaps = encaps->next) {
		if (encaps->encaps && encaps->localport == assoc->localport)
			break;
	}
	if (encaps == NULL) {
		DBPRINTF (2, ("Making new encaps for port %d\n",
		  ntohs (assoc->localport)));
		if ((encaps = getassoc ()) == NULL) {
			syslog (LOG_ERR, "Out of memory, exiting");
			exit (1);
		}
		encaps->encaps = TRUE;
		encaps->localport = assoc->localport;
		if (!opensocket (encaps))
			exit (1);
	}
	assoc->nextinst = encaps->generics;
	encaps->generics = assoc;
	assoc->plex = encaps;
	lscan = skipwhite (lscan);
	if (*lscan) {
		syslog (LOG_ERR, "Too much stuff on line %d", lno);
		exit (1);
	}
	assoc->receiver = TRUE;
}


/*
 * Read configuration file and build initial associations
 */
void readconfig (fname)
	char		*fname;
{
	FILE		*configfile;
	char		line [128], *lscan, *end;
	int		lno;

	configfile = fopen (fname, "r");
	if (configfile == NULL) {
		syslog (LOG_ERR, "opening config file, %m");
		exit (1);
	}
	lno = 0;
	while (fgets (line, sizeof (line), configfile) != NULL) {
		++lno;
		if (strlen (line) >= sizeof (line) - 1) {
			syslog (LOG_ERR, "Config file line %d too long", lno);
			exit (1);
		}
		lscan = strchr (line, '#');
		if (lscan != NULL)
			*lscan = '\0';
		else
			line [strlen (line) - 1] = '\0';	/* Nuke \n */
		lscan = skipwhite (line);
		if (*lscan == '\0')		/* Skip blank lines */
			continue;
		end = findeos (lscan);
		if (*end != '\0')
			*end++ = '\0';
		if (strcmp (lscan, "relay") == 0)
			parserelay (end, lno);
		else if (strcmp (lscan, "encapsulate") == 0)
			parseencaps (end, lno);
		else {
			syslog (LOG_ERR, "Bad config command on line %d", lno);
			exit (1);
		}
	}
	fclose (configfile);
}


/*
 * Return TRUE if the specified sockaddr matches the wildcarded assoc; must
 * match both the address/mask in the assoc, the possibly wildcarded
 * hostname, and the port number.  Code shamelessly stolen from tcpd, thanks
 * be to Wietse Venema.
 */
int validate (assoc, addr)
	Assoc		*assoc;
	SockIn		*addr;
{
	struct hostent	*hp;
	char		namebuf [512], *cp;
	int		mismatch, i;
	u_int		nblen;

	DBPRINTF (2, ("validate (assoc=0x%x, addr=%s/port=%d)\n", assoc,
	  inet_ntoa (addr->sin_addr), ntohs (addr->sin_port)));
	/*
	 * Check IP address and port number
	 */
	if ((addr->sin_addr.s_addr & assoc->addrmask.s_addr)
	  != assoc->sin.sin_addr.s_addr) {
		DBPRINTF (2, ("Failed mask test, addr=%s, mask=%x\n", 
		  inet_ntoa (assoc->sin.sin_addr),
		  ntohl (assoc->addrmask.s_addr)));
		return FALSE;
	}
	if (assoc->sin.sin_port != INADDR_ANY
	  && assoc->sin.sin_port != addr->sin_port) {
		DBPRINTF (2, ("Mismatched ports, assoc.port=%d, have %d\n", 
		  ntohs (assoc->sin.sin_port), ntohs (addr->sin_port)));
		return FALSE;
	}
	/*
	 * If we have a pure wildcard hostname, we match.
	 */
	if (assoc->hostname == NULL || strcmp (assoc->hostname, "*") == 0) {
		DBPRINTF (2, ("Wildcard, good enough\n"));
		return TRUE;
	}
	/*
	 * Get hostname corresponding to the address
	 */
	if ((hp = gethostbyaddr ((char *) &addr->sin_addr,
	  sizeof (addr->sin_addr), AF_INET)) == NULL) {
		syslog (LOG_WARNING, "failed to gethostbyaddr (%s)",
		  inet_ntoa (addr->sin_addr));
		return (FALSE);
	}
	/*
	 * Save the hostname, the call to gethostbyname will overwrite it
	 */
	strncpy (namebuf, hp->h_name, sizeof (namebuf) - 1);
	namebuf [sizeof (namebuf) - 1] = '\0';
	/*
	 * Match it to the wildcarded hostname (case-insensitively)
	 * !!! Should allow *.dom.ain to match dom.ain
	 */
	if (assoc->hostname [0] == '*') {
		nblen = strlen (namebuf);
		cp = &assoc->hostname [1];	/* Pattern to match (w/o *) */
		mismatch = (nblen < strlen (cp))
		  || strcasecmp (cp, namebuf + nblen - strlen (cp)) != 0;
	}
	else {
		/* Name not wildcarded, match it exactly */
		mismatch = strcasecmp (assoc->hostname, namebuf) != 0;
	}
	if (mismatch) {
		DBPRINTF (2, ("Mismatched wildcarded names: %s != %s\n",
		  assoc->hostname, namebuf));
		return (FALSE);
	}
	/*
	 * Look it up in the forward DNS map, and make sure it has the address
	 * the packet came in from
	 */
	if ((hp = gethostbyname (namebuf)) == NULL) {
		syslog (LOG_WARNING, "failed to gethostbyname (%s)", namebuf);
		return (FALSE);
	}
	for (i = 0; hp->h_addr_list [i]; i++) {
		if (memcmp (hp->h_addr_list [i], (caddr_t) &addr->sin_addr,
		  sizeof (addr->sin_addr)) == 0) {
			DBPRINTF (2, ("Validated ok\n"));
			return (TRUE);		/* matched */
		}
	}
	/*
	 * The host name does not map to the original host address. Perhaps
	 * someone has compromised a name server. More likely someone botched
	 * it, but that could be dangerous, too.
	 */
	syslog (LOG_WARNING, "Host name/address mismatch: %s != %s",
	  inet_ntoa (addr->sin_addr), hp->h_name);
	return (FALSE);
}


Assoc *makeinstance (assoc, fromaddr)
	Assoc		*assoc;		/* Fanout/encaps assoc */
	SockIn		*fromaddr;
{
	Assoc		*instance, *partner;

	DBPRINTF (1, ("makeinstance (0x%x) plex=0x%x\n", assoc, assoc->plex));
	if ((instance = getassoc ()) == NULL) {
		syslog (LOG_ERR, "Out of memory");
		return (NULL);
	}
	if ((partner = getassoc ()) == NULL) {
		deleteassoc (instance);
		syslog (LOG_ERR, "Out of memory");
		return (NULL);
	}
	DBPRINTF (1, ("creating instance 0x%x, partner 0x%x\n",
	  instance, partner));
	instance->partner = partner;
	instance->sin = *fromaddr;
	instance->plex = assoc->plex;
	instance->socket = assoc->plex->socket;		/* clone socket */
	instance->localport = assoc->localport;
	instance->receiver = TRUE;
	instance->transient = TRUE;
	partner->partner = instance;
	/*
	 * Thread onto list of instances for this fanout/encaps
	 */
	instance->nextinst = assoc->plex->instances;
	assoc->plex->instances = instance;
	DBPRINTF (1, ("makeinstance: done\n"));
	return (instance);
}


/*
 * Send a packet out to the outgoing half of an assoc pair.  pkt points to
 * the packet, but *must* have space available before it, in its buffer,
 * to hold an encapsulating wrapper if necessary.
 */
void sendtopartner (assoc, pkt, pktlen, fromaddr, rcvport)
	Assoc		*assoc;
	char		*pkt;
	int		pktlen;
	SockIn		*fromaddr;
	u_short		rcvport;
{
	UdpEncaps	*wrapper;

	/*
	 * If socket isn't open yet, open it
	 */
	if (assoc->socket < 0) {
		if (opensocket (assoc)) {
			if (assoc->receiver)
				makefdlist ();	/* Rebuild socket list */
		}
		else {
			syslog (LOG_WARNING, "Packet dropped");
			return;
		}
	}
	/*
	 * If sending to encapsulated socket, add wrapper
	 */
	if (assoc->plex != NULL && assoc->plex->encaps) {
		DBPRINTF (2, ("Encapsulating\n"));
		pkt -= sizeof (UdpEncaps);
		pktlen += sizeof (UdpEncaps);
		wrapper = (UdpEncaps *) pkt;
		bzero (pkt, sizeof (UdpEncaps));
		wrapper->version = UdpRelayVersion;
		wrapper->contents = UdpRelayData;
		wrapper->sin = *fromaddr;
		wrapper->proxyport = rcvport;
	}
	if (sendto (assoc->socket, pkt, pktlen, 0, &assoc->sin,
	  sizeof (SockIn)) != pktlen)
		syslog (LOG_ERR, "sendto failed, %m");
	DBPRINTF (2, (
	  "packet sent to assoc=0x%x socket=%d addr=%s port=%d localport=%d\n",
	  assoc, assoc->socket, inet_ntoa (assoc->sin.sin_addr),
	  ntohs (assoc->sin.sin_port), ntohs (assoc->localport)));
}


void fwdpacketfrom (assoc)
	Assoc		*assoc;
{
	Assoc		*instance, *partner, *fano, *generic;
	int		maxpktlen, pktlen, fromaddrlen;
	SockIn		fromaddr;
	char		*pkt;
	UdpEncaps	*wrapper;

	DBPRINTF (2, ("Reading from assoc 0x%x, socket %d\n", assoc,
	  assoc->socket));
	fromaddrlen = sizeof (fromaddr);
	/*
	 * Get packet in buffer, with room before it to encapsulate it if
	 * necessary
	 */
	pkt = pktbuf + sizeof (UdpEncaps);
	maxpktlen = pktbuflen - sizeof (UdpEncaps);
	if ((pktlen = recvfrom (assoc->socket, pkt, maxpktlen, 0, &fromaddr,
	  &fromaddrlen)) <= 0) {
		syslog (LOG_ERR, "recvfrom failed, pktlen=%d, %m", pktlen);
		return;
	}
	DBPRINTF (2, ("pkt rcvd from addr=%s, port=%d, pktlen=%d\n",
	  inet_ntoa (fromaddr.sin_addr), ntohs (fromaddr.sin_port), pktlen));
	if (pktlen > maxpktlen) {
		syslog (LOG_WARNING, "Dropped pkt (pktlen=%d, maxpktlen=%d)",
		  pktlen, maxpktlen);
		(void) makepktbuf (pktlen);	/* Boost pkt buffer size */
		return;				/* Drop this packet */
	}
	if (assoc->encaps) {			/* Encapsulated packet? */
		DBPRINTF (2, ("De-encapsulating\n"));
		if (pktlen < sizeof (UdpEncaps)) {
			syslog (LOG_WARNING, "Pkt missing encapsulation");
			return;
		}
		wrapper = (UdpEncaps *) pkt;
		pkt += sizeof (UdpEncaps);
		pktlen -= sizeof (UdpEncaps);
		if (wrapper->version != UdpRelayVersion
		  || wrapper->contents != UdpRelayData) {
			syslog (LOG_WARNING, "Bad UDP encapsulation");
			return;
		}
		/*
		 * Find existing instance (source/dest pair).  If one doesn't
		 * exist, validate pkt against list of generics, and create
		 * new instance.
		 */
		for (instance = assoc->instances; instance != NULL;
		  instance = instance->nextinst) {
			partner = instance->partner;
#if 0
			if (sameaddr (&instance->sin, &fromaddr)
			  && sameaddr (&partner->sin, &wrapper->sin)
			  && (partner->localport == wrapper->proxyport
			    || wrapper->proxyport == INADDR_ANY))
#endif
			/*
			 * Exit loop if we find an instance whose source/dest
			 * addresses and proxyport match (or proxyport == 0),
			 * or proxyport is non-0 and dest address matches -
			 * this is an attempt to take over an existing session.
			 */
			if (sameaddr (&partner->sin, &wrapper->sin)
			  && ((wrapper->proxyport == INADDR_ANY
			      && sameaddr (&instance->sin, &fromaddr))
			    || partner->localport == wrapper->proxyport))
				break;
		}
		if (instance == NULL) {
			/*
			 * No existing path for this packet - create one.
			 * First see if it matches any specified "encaps" rule.
			 */
			for (generic = assoc->generics; generic != NULL;
			  generic = generic->nextinst) {
				if (validate (generic, &fromaddr))
					break;
			}
			if (!generic) {
				syslog (LOG_WARNING,
				  "pkt from unapproved source: %s on port %d",
				  inet_ntoa (fromaddr.sin_addr),
				  ntohs (assoc->localport));
				syslog (LOG_WARNING,
				  "... sent to: %s, port %d, localport %d",
				  inet_ntoa (wrapper->sin.sin_addr),
				  ntohs (wrapper->sin.sin_port),
				  ntohs (wrapper->proxyport));
				return;
			}
			instance = makeinstance (generic, &fromaddr);
			if (instance == NULL)
				return;
#if LogInstances
			syslog (LOG_INFO, "creating new instance from %s/%d",
			  inet_ntoa (fromaddr.sin_addr),
			  ntohs (fromaddr.sin_port));
			syslog (LOG_INFO, "... to %s/%d (localport %d)",
			  inet_ntoa (wrapper->sin.sin_addr),
			  ntohs (wrapper->sin.sin_port),
			  ntohs (wrapper->proxyport));
#endif
			partner = instance->partner;
			/* !!! If client specifies localport, but no fanout
			   is available for it, maybe should create one?  Does
			   no functional good but conserves sockets. */
			if (wrapper->proxyport != INADDR_ANY &&
			  (fano = findfanout (wrapper->proxyport)) != NULL) {
				/*
				 * Port is slaved to wildcard - thread on list
				 * of ports under the fanout.
				 */
				DBPRINTF (1, ("Hook on fano 0x%x\n", fano));
				partner->plex = fano;
				partner->nextinst = fano->instances;
				fano->instances = partner;
				partner->socket = fano->socket;
			}
			partner->sin	    = wrapper->sin;
			partner->localport  = wrapper->proxyport;
			partner->receiver   = wrapper->twoway;
		}
		else {
			if (!sameaddr (&instance->sin, &fromaddr)) {
				syslog (LOG_WARNING,
				  "Can't override existing session");
				/* !!! Should send back error */
				return;
			}
			DBPRINTF (2, ("mapped to instance 0x%x, %s/%d\n",
			  instance, inet_ntoa (instance->sin.sin_addr),
			  ntohs (instance->sin.sin_port)));
		}
		assoc = instance;
		assoc->lastaccess = time (NULL);
		assoc->partner->lastaccess = assoc->lastaccess;
	}
	else if (assoc->fanout) {		/* Wildcarded source? */
		for (instance = assoc->instances; instance != NULL;
		  instance = instance->nextinst) {
			if (sameaddr (&instance->sin, &fromaddr))
				break;
		}
		if (instance == NULL) {
			for (generic = assoc->generics; generic != NULL;
			  generic = generic->nextinst) {
				if (validate (generic, &fromaddr))
					break;
			}
			if (generic == NULL) {
				syslog (LOG_WARNING,
				  "pkt from unapproved source: %s on port %d",
				  inet_ntoa (fromaddr.sin_addr),
				  ntohs (assoc->localport));
				return;
			}
			if (!generic->receiver) {
				syslog (LOG_WARNING,
				  "Pkt rcvd at wrong end of 1-way link: %s/%d",
				  inet_ntoa (fromaddr.sin_addr),
				  ntohs (assoc->localport));
				return;
			}
			instance = makeinstance (generic, &fromaddr);
			if (instance == NULL)
				return;
			partner = instance->partner;
			partner->sin       = generic->partner->sin;
			partner->localport = generic->partner->localport;
			partner->receiver  = generic->partner->receiver;
			partner->hostname  = generic->partner->hostname;
			if (partner->sin.sin_addr.s_addr == INADDR_ANY) {
				if (!partner->hostname
				  || partner->hostname [0] == '*') {
					DBPRINTF (2, ("Send generic\n"));
					syslog (LOG_WARNING, "Send generic");
					deleteassoc (instance);
					deleteassoc (partner);
					return;
				}
				if (!fixhostname (partner)) {
					deleteassoc (instance);
					deleteassoc (partner);
					return;
				}
			}
#if LogInstances
			syslog (LOG_INFO, "creating new instance from %s/%d",
			  inet_ntoa (fromaddr.sin_addr),
			  ntohs (fromaddr.sin_port));
			syslog (LOG_INFO, "... to %s/%d (localport %d)",
			  inet_ntoa (partner->sin.sin_addr),
			  ntohs (partner->sin.sin_port),
			  ntohs (partner->localport));
#endif
			if (generic->partner->plex != NULL) {
				/*
				 * Port is slaved to wildcard - thread on list
				 * of ports under the fanout.
				 */
				partner->plex = generic->partner->plex;
				partner->nextinst = partner->plex->instances;
				partner->plex->instances = partner;
				partner->socket = partner->plex->socket;
			}
		}
#if Debug
		else {
			DBPRINTF (2, ("mapped to instance 0x%x, %s/%d\n",
			  instance, inet_ntoa (instance->sin.sin_addr),
			  ntohs (instance->sin.sin_port)));
		}
#endif
		assoc = instance;
		assoc->lastaccess = time (NULL);
		assoc->partner->lastaccess = assoc->lastaccess;
	}
	sendtopartner (assoc->partner, pkt, pktlen, &fromaddr,
	  assoc->localport);
}


int main (argc, argv)
	int		argc;
	char		*argv [];
{
	extern char *	optarg;
	int		pollval, i, opt, timeout;
	char		*configname = "/etc/udprelay.conf";

	while ((opt = getopt (argc, argv, "d:f:")) != -1) {
	    switch (opt) {
#if Debug
		case 'd':
			debug = atoi (optarg);
			idletime = 20;		/* Short timeouts */
			setbuf (stdout, NULL);
			break;
#endif
		case 'f':
			configname = optarg;
			break;
		default:
#if Debug
			printf ("usage: udprelay [ -d ] [ -f configfile ]\n");
#else
			printf ("usage: udprelay [ -f configfile ]\n");
#endif
			exit (1);
	    }
	}
	openlog ("udprelay", LOG_PID, LOG_DAEMON);
	readconfig (configname);
	/*
	 * If we're running as root, we've not got all the reserved ports
	 * we deserve to get, so lose root privs.
	 */
#ifdef NOBODY
	if (getuid () == 0) {
		struct passwd  *pw;
		pw = getpwnam (NOBODY);
		if (pw == NULL) {
			syslog (LOG_ERR, "User ID '%s' not found", NOBODY);
			exit (1);
		}
		setgid (pw->pw_gid);
		setuid (pw->pw_uid);
	}
#endif
	/*
	 * Create initial packet buffer; die if no memory available
	 */
	if (!makepktbuf (0))
		exit (1);
	makefdlist ();		/* Build list of sockets to poll */
	while (TRUE) {
		timeout = checktimeouts ();
		DBPRINTF (2, ("polling %d sockets, timeout %d\n", fdlistlen,
		  timeout));
		pollval = poll (fdlist, (u_long) fdlistlen, timeout);
		DBPRINTF (2, ("returned from poll\n"));
		if (pollval < 0) {
			syslog (LOG_ERR, "poll failed, pollval=%d, %m",
			  pollval);
			exit (1);
		}
		if (pollval == 0)
			continue;	/* Timeout only */
		for (i = 0; i < fdlistlen; ++i) {
			if ((fdlist [i].revents & POLLIN) != 0)
				fwdpacketfrom (fdlistassocs [i]);
			if ((fdlist [i].revents & ~POLLIN) != 0) {
				syslog (LOG_ERR,
				  "Error on polled socket, revent=0x%x",
				  fdlist [i].revents);
			}
		}
	}
	/*NOTREACHED*/
}

