/*
 * Copyright (c) 1988 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that: (1) source distributions retain this entire copyright
 * notice and comment, and (2) distributions including binaries display
 * the following acknowledgement:  ``This product includes software
 * developed by the University of California, Berkeley and its contributors''
 * in the documentation or other materials provided with the distribution
 * and in all advertising materials mentioning features or use of this
 * software. Neither the name of the University nor the names of its
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
/*
 * Copyright (c) 1992 Purdue University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by Purdue University.  The name of the University may not be used
 * to endorse or promote products derived * from this software without
 * specific prior written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * Note: this copyright applies to portions of this software developed
 * at Purdue beyond the software covered by the original copyright.
 */

#ifndef lint
static char sccsid[] = "@(#)uucplock.c	5.5 (Berkeley) 6/1/90";
#endif /* not lint */

#include <sys/types.h>
#include <sys/file.h>
#include <sys/dir.h>
#include <errno.h>

extern char *sprintf();

/* 
 * uucp style locking routines
 * return: 0 - success
 * 	  -1 - failure
 */
#if	1
#define	_PATH_LOCKDIRNAME	"/var/spool/locks"

static int uu_dolock(), uu_lockactive(), chown_uucp();

/*
 * These versions of uu_lock and uu_unlock are designed to be compatible
 * with the way that the Sun uucp does locking..
 */
#define	MAXPIDLEN	10
uu_lock(ttyname)
	char *ttyname;
{
	char pid[MAXPIDLEN+2],
	     tfile[sizeof(_PATH_LOCKDIRNAME) + MAXNAMLEN],
	     lfile[sizeof(_PATH_LOCKDIRNAME) + MAXNAMLEN];
	    
	(void)sprintf(pid, "%*d\n", MAXPIDLEN, getpid());
	(void)sprintf(tfile, "%s/LTMP.%d", _PATH_LOCKDIRNAME, getpid());
	(void)sprintf(lfile, "%s/LCK..%s", _PATH_LOCKDIRNAME, ttyname);

	/*
	 * If the lock fails, check to see if it is currently locked
	 * by a valid process.  If not, try the lock again.
	 */
	if (uu_dolock(pid, tfile, lfile)) {
	    if (uu_lockactive(lfile))
		return -1;
	    else if (uu_dolock(pid, tfile, lfile))
		return -1;
	}
	return 0;
}

static
uu_lockactive(lfile)
char *lfile;
{
    int lf;
    char pid_str[MAXPIDLEN+2];
    int pid, r;

    /*
     * Make sure the file is readable and contains a valid PID.
     * If not, and we can unlink it, then return success.
     */
    lf = open(lfile, O_RDONLY);
    if (lf < 0)
	return (errno == ENOENT) ? 0 : unlink(lfile);

    r = read(lf, pid_str, MAXPIDLEN+1);
    (void)close(lf);
    if (r != MAXPIDLEN+1)
	return unlink(lfile);
    pid = atoi(pid_str);
    if (pid <= 0)
	return unlink(lfile);

    /*
     * We have a found a seemingly valid pid.
     * Make sure the process is still around.
     */
    return (kill(pid, 0) == 0 || errno == EPERM) ? -1 : unlink(lfile);
}

#define	DEFUUCPUID	4
#define	DEFUUCPGID	8

#include <pwd.h>
static
chown_uucp(fd)
int fd;
{
    static int uuid = -1, ugid;
    struct passwd *pw;

    if (uuid < 0)
	if (pw = getpwnam("uucp")) {
	    uuid = pw->pw_uid;
	    ugid = pw->pw_gid;
	}
	else {
	    uuid = DEFUUCPUID;
	    uuid = DEFUUCPGID;
	}
    return fchown(fd, uuid, ugid);
}

static
uu_dolock(pid, tfile, lfile)
char *pid,
     *tfile,
     *lfile;
{
    int tf;

    tf = open(tfile, O_WRONLY|O_CREAT|O_TRUNC, 0444);
    if (tf < 0) {
	(void)unlink(tfile);
	return -1;
    }

    (void)write(tf, pid, MAXPIDLEN+1);
    (void)fchmod(tf, 0444);
    (void)chown_uucp(tf);
    (void)close(tf);

    if (link(tfile, lfile) < 0) {
	(void)unlink(tfile);
	return -1;
    }
    (void)unlink(tfile);
    return 0;
}

uu_unlock(ttyname)
char *ttyname;
{
    char lfile[sizeof(_PATH_LOCKDIRNAME) + MAXNAMLEN];

    (void)sprintf(lfile, "%s/LCK..%s", _PATH_LOCKDIRNAME, ttyname);
    return unlink(lfile);
}
#else
#define	_PATH_LOCKDIRNAME	"/usr/spool/uucp/LCK..%s"

/*
 * These routines must be for older style UUCP's.
 */
uu_lock(ttyname)
	char *ttyname;
{
	extern int errno;
	int fd, pid;
	char tbuf[sizeof(_PATH_LOCKDIRNAME) + MAXNAMLEN];
	long lseek();

	(void)sprintf(tbuf, _PATH_LOCKDIRNAME, ttyname);
	fd = open(tbuf, O_RDWR|O_CREAT|O_EXCL, 0660);
	if (fd < 0) {
		/*
		 * file is already locked
		 * check to see if the process holding the lock still exists
		 */
		fd = open(tbuf, O_RDWR, 0);
		if (fd < 0) {
			/*perror("lock open");*/
			return(-1);
		}
		if (read(fd, (char *)&pid, sizeof pid) != sizeof pid) {
			(void)close(fd);
			/*perror("lock read");*/
			return(-1);
		}

		if (kill(pid, 0) == 0 || errno != ESRCH) {
			(void)close(fd);	/* process is still running */
			return(-1);
		}
		/*
		 * The process that locked the file isn't running, so
		 * we'll lock it ourselves
		 */
		if (lseek(fd, 0L, L_SET) < 0) {
			(void)close(fd);
			/*perror("lock lseek");*/
			return(-1);
		}
		/* fall out and finish the locking process */
	}
	pid = getpid();
	if (write(fd, (char *)&pid, sizeof(pid)) != sizeof(pid)) {
		(void)close(fd);
		(void)unlink(tbuf);
		/*perror("lock write");*/
		return(-1);
	}
	(void)close(fd);
	return(0);
}

uu_unlock(ttyname)
	char *ttyname;
{
	char tbuf[sizeof(_PATH_LOCKDIRNAME) + MAXNAMLEN];

	(void)sprintf(tbuf, _PATH_LOCKDIRNAME, ttyname);
	return(unlink(tbuf));
}
#endif
