/* osdep_linux.c: Key engine/PF_KEY socket OS dependencies for Linux
   Many parts were derived from af_unix.c. See the GPLv2 or that source
   file for license details on those parts. */

#include <linux/module.h>

#include <netkey/osdep_linux.h>

#if 0
#include <linux/kernel.h>
#include <linux/major.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/stat.h>
#include <linux/socket.h>
#include <linux/un.h>
#include <linux/fcntl.h>
#include <linux/termios.h>
#include <linux/socket.h>
#include <linux/sockios.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/fs.h>
#include <linux/malloc.h>

#include <asm/segment.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/proc_fs.h>
#endif /* 0 */

#include <net/sock.h>
#include <linux/random.h>
#include <linux/netdevice.h>
#include <linux/errno.h>

#include <netkey/key.h>
#include <netkey/key_debug.h>

#define min(a,b)	(((a)<(b))?(a):(b))

static struct sockaddr key_addr = { AF_KEY, };
static struct sock *key_socket_list = NULL;

volatile struct sock *key_last_sendup;

long random(void)
{
  long rval;
  get_random_bytes(&rval, sizeof(rval));
  return rval;
}

long time_seconds(void)
{
  struct timeval t;
  do_gettimeofday(&t);
  return t.tv_sec;
}

int key_sendup(struct socket *s, struct key_msghdr *km)
{
  struct sk_buff *skb;
  struct sock *sk = (struct sock *)s->data;
  int error = 0, size = km->key_msglen;

#if 0
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_sendup(). size=%d\n", size));
  DPRINTF(IDL_MAJOR_EVENT, ("key_last_sendup = %08x\n", (unsigned)key_last_sendup));
  key_last_sendup = sk;
  DPRINTF(IDL_MAJOR_EVENT, ("key_last_sendup = %08x\n", (unsigned)key_last_sendup));
/* should this be non-blocking? */
  if (!sk)
    DPRINTF(IDL_MAJOR_EVENT, ("NULL sk = lossage\n"));
  DPRINTF(IDL_MAJOR_EVENT, ("sk = %08x, sk->wmem_alloc=%d + size=%d < sk->sndbuf = %d\n", (unsigned)sk, 
			    sk->wmem_alloc, size, sk->sndbuf));
  if (!(skb = sock_alloc_send_skb(sk, size, 1024, 0, &error))) {
    DPRINTF(IDL_ERROR, ("key_sendup: sock_alloc_send_skb(sk, size=%d, 1024, 0, &error->%d) failed!\n",
	size, error));
    return 0;
  }

  if (skb_tailroom(skb) != size) {
    DPRINTF(IDL_ERROR, ("key_sendup: discrepency: skb_tailroom=%d != size=%d\n",
	skb_tailroom(skb), size));
    return 0;
  }

  skb->sk = sk;
  skb->free = 1;
  memcpy(skb_put(skb, size), km, size);

  DPRINTF(IDL_MAJOR_EVENT, ("key_sendup: skb=%08x\n", (unsigned)skb));
  sock_queue_rcv_skb(sk, skb);
#else /* 0 */
  if (!(skb = alloc_skb(size, GFP_ATOMIC))) {
    DPRINTF(IDL_ERROR, ("key_sendup: alloc_skb failed!\n"));
    return 0;
  }
  
  skb->free = 1;
  skb->h.iph = skb->ip_hdr = skb_put(skb, size);
  memcpy(skb->ip_hdr, km, size);

  if ((error = sock_queue_rcv_skb(sk, skb)) < 0) {
    DPRINTF(IDL_ERROR, ("key_sendup: sock_queue_rcv_skb failed (error = %d)!\n", error));
    skb->sk = NULL;
    kfree_skb(skb, FREE_READ);
  }
#endif /* 0 */

  DPRINTF(IDL_MAJOR_EVENT, ("key_sendup: returning successfully\n"));
  return 1;
}

int my_addr(SOCKADDR *sa)
{
  switch (sa->sa_family) {
  case AF_INET:
    return (ip_chk_addr(((struct sockaddr_in *)sa)->sin_addr.s_addr) == IS_MYADDR);
  default:
    return 0;
  }
}

/*
 *	Note: Sockets may not be removed _during_ an interrupt or net_bh
 *	handler using this technique. They can be added although we do not
 *	use this facility.
 */
 
static void key_remove_socket(struct sock *sk)
{
  struct sock **s;
	
  cli();
  s = &key_socket_list;

  while(*s) {
    if (*s == sk) {
      *s = sk->next;
      sti();
      return;
    }
    s = &((*s)->next);
  }
  sti();
}

static void key_insert_socket(struct sock *sk)
{
  cli();
  sk->next = key_socket_list;
  key_socket_list = sk;
  sti();
}

static int key_einval(void)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_einval\n"));
  return -EINVAL;
}

static void key_statechange(struct sock *sk)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_statechange\n"));
  if(!sk->dead)
    wake_up_interruptible(sk->sleep);
}

static void key_dataready(struct sock *sk, int len)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_dataready(sk, len=%d)\n", len));
  if(!sk->dead) {
    wake_up_interruptible(sk->sleep);
    sock_wake_async(sk->socket, 1);
  }
}

static void key_writespace(struct sock *sk)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_writespace\n"));
  if(!sk->dead) {
    wake_up_interruptible(sk->sleep);
    sock_wake_async(sk->socket, 2);
  }
}

static int key_create(struct socket *sock, int protocol)
{
  struct sock *sk;

/*  MOD_INC_USE_COUNT; */

  DPRINTF(IDL_MAJOR_EVENT, ("PF_KEY: Entering key_create(*, %d)\n", protocol));
  if (protocol && (protocol != PF_KEY))
    return -EPROTONOSUPPORT;

  if (!(sk = (struct sock *)kmalloc(sizeof(*sk), GFP_KERNEL)))
    return -ENOMEM;

  if (sock->type != SOCK_RAW) {
    kfree_s(sk, sizeof(*sk));
    return -ESOCKTNOSUPPORT;
  }

  memset(sk, 0, sizeof(*sk));

  sk->type = sock->type;
  init_timer(&sk->timer); /* ? */
  /*  skb_queue_head_init(&sk->write_queue);*/
  skb_queue_head_init(&sk->receive_queue);
  skb_queue_head_init(&sk->back_log);

  sk->protocol = protocol;

  sk->rmem_alloc=0;
  sk->wmem_alloc=0;
  sk->dead=0;
  sk->next=NULL;
  sk->broadcast=0;
  sk->rcvbuf=SK_RMEM_MAX;
  sk->sndbuf=SK_WMEM_MAX;
  sk->allocation=GFP_KERNEL;
  sk->inuse=0;
  sk->bsdism=0;
  sk->debug=0;
  sk->prot=NULL;
  sk->err=0;
  sk->localroute=0;
  sk->send_head=NULL;
  sk->state=TCP_CLOSE;
  sk->priority=SOPRI_NORMAL;
  sk->ack_backlog=0;
  sk->shutdown=0;
  sk->state_change = key_statechange;
  sk->data_ready = key_dataready;
  sk->write_space = key_writespace;
  sk->error_report = key_statechange;
  sk->mtu=4096;
  sk->socket=sock;
  sock->data=(void *)sk;
  sk->sleep=sock->wait;
  sk->zapped=0;

  key_insert_socket(sk);
  return 0;
};

static int key_release(struct socket *sock, struct socket *peer)
{
  struct sock *sk = sock->data;
  struct sk_buff *skb;

  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_release\n"));

  if (!sk)
    return 0;

  sk->state_change(sk);
  sk->dead = 1;
  
  while ((skb = skb_dequeue(&sk->receive_queue)))
      kfree_skb(skb, FREE_WRITE);

  key_remove_socket(sk);
	
  kfree_s(sk,sizeof(*sk));
	
  return 0;
}

static int key_getname(struct socket *sock, struct sockaddr *uaddr, int *uaddr_len, int peer)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_getname\n"));
  memcpy(uaddr, &key_addr, *uaddr_len = sizeof(key_addr));
  return 0;
}

static int key_select(struct socket *sock,  int sel_type, select_table *wait)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_select\n"));
  return datagram_select(sock->data,sel_type,wait);
}

static int key_shutdown(struct socket *sock, int mode)
{
  struct sock *sk=(struct sock *)sock->data;

  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_shutdown\n"));
  
  if (mode & SEND_SHUTDOWN) {
    sk->shutdown |= SEND_SHUTDOWN;
    sk->state_change(sk);
  }
  if (mode & RCV_SHUTDOWN) {
    sk->shutdown |= RCV_SHUTDOWN;
    sk->state_change(sk);
  }
/*  MOD_DEC_USE_COUNT;*/
  return 0;
}

/* We ignore msg->msg_name */
static int key_sendmsg(struct socket *sock, struct msghdr *msg, int len, int nonblock, int flags)
{
  struct sock *sk = sock->data;
  int error = 0;
  struct key_msghdr *km;

  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_sendmsg\n"));
  
  if (sk->err)
    return sock_error(sk);

  if (flags || msg->msg_accrights)
    return -EINVAL;
  
  if (len > 100)
    return -EMSGSIZE;

  if (!(km = (struct key_msghdr *)kmalloc(len, GFP_KERNEL)))
    return -ENOMEM;

  memcpy_fromiovec((void *)km, msg->msg_iov, len);

  DPRINTF(IDL_MAJOR_EVENT, ("Do stuff here\n"));

  if (len < sizeof(*km) || len != km->key_msglen) {
    DPRINTF(IDL_CRITICAL,("keyout: Invalid length field/length mismatch!\n"));
    error = EINVAL;
    goto flush;
  }

  error = key_parse(&km, sock, NULL);
flush:
  km->key_errno = error;

  key_sendup(sock, km);

  KFREE(km);

  return len;
}

static int key_recvmsg(struct socket *sock, struct msghdr *msg, int size, int noblock, int flags, int *addr_len)
{
  struct sock *sk=sock->data;
  struct sk_buff *skb;
  int num;
  struct iovec *iov=msg->msg_iov;

  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_recvmsg\n"));
  
  if (flags)
    return -EINVAL;
  
  if (addr_len)
    *addr_len = sizeof(struct sockaddr);
  
  if (sk->err)
    return sock_error(sk);

  if (msg->msg_name) {
    struct sockaddr *sa = msg->msg_name;
    memset(sa, 0, sizeof(*sa));
#ifdef SA_LEN
    sa->sa_len = sizeof(sa);
#endif /* SA_LEN */
    sa->sa_family = AF_KEY;
  }

  if (size > iov->iov_len)
    size = iov->iov_len;

  if (size > 100)
    size = 100;

  DPRINTF(IDL_MAJOR_EVENT, ("In key_recvmsg, before loop\n"));
  
  while(1) {
    DPRINTF(IDL_MAJOR_EVENT, ("In key_recvmsg, in the loop\n"));
    if (!(skb = skb_dequeue(&sk->receive_queue))) {
      DPRINTF(IDL_MAJOR_EVENT, ("In key_recvmsg, after skb_dequeue\n"));
      if (sk->shutdown & RCV_SHUTDOWN)
	return 0;
      if (noblock)
	return -EAGAIN;
      if (current->signal & ~current->blocked)
	return -ERESTARTSYS;
      cli();
      if (!skb_peek(&sk->receive_queue)) {
	sk->socket->flags |= SO_WAITDATA;
        DPRINTF(IDL_MAJOR_EVENT, ("In key_recvmsg, settting interruptible_sleep_on\n"));
	interruptible_sleep_on(sk->sleep);
	sk->socket->flags &= ~SO_WAITDATA;
      }
      sti();
      DPRINTF(IDL_MAJOR_EVENT, ("In key_recvmsg, continue\n"));
      continue;
    }
  }

  DPRINTF(IDL_MAJOR_EVENT, ("In key_recvmsg, after loop\n"));
  
  num = min(skb->len, size);
  memcpy_tofs(iov->iov_base, skb->data, num);
  
  if (!(flags & MSG_PEEK))
    skb_pull(skb, num);

  if (skb->len)
    skb_queue_head(&sk->receive_queue, skb);
  else
    kfree_skb(skb, FREE_WRITE);

  DPRINTF(IDL_MAJOR_EVENT, ("Leaving key_recvmsg, num=%d\n", num));

  return num;
}

static int key_dup(void)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_dup\n"));
  return -EINVAL;
}

static int key_bind(void)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_bind\n"));
  return -EINVAL;
}

static int key_connect(void)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_connect\n"));
  return -EINVAL;
}

static int key_socketpair(void)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_socketpair\n"));
  return -EINVAL;
}

static int key_accept(void)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_accept\n"));
  return -EINVAL;
}

static int key_ioctl(void)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_ioctl\n"));
  return -EINVAL;
}

static int key_listen(void)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_listen\n"));
  return -EINVAL;
}

static int key_getsockopt(struct socket *socket, int level, int optname,
char *optval, int *optlen)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_getsockopt\n"));

  if (level != SOL_SOCKET)
    return -EOPNOTSUPP;

  return sock_getsockopt((struct sock *)socket->data, level,
			 optname, optval, optlen);
}

static int key_setsockopt(struct socket *socket, int level, int optname,
char *optval, int optlen)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_setsockopt\n"));

  if (level != SOL_SOCKET)
    return -EOPNOTSUPP;

  return sock_setsockopt((struct sock *)socket->data, level,
			 optname, optval, optlen);
}

static int key_fcntl(void)
{
  DPRINTF(IDL_MAJOR_EVENT, ("Entering key_fcntl\n"));
  return -EINVAL;
}

static struct proto_ops key_proto_ops = {
  PF_KEY,
  key_create,
  key_dup,      /* (void *)key_einval, */
  key_release,
  key_bind,     /* (void *)key_einval, */
  key_connect,  /* (void *)key_einval, */
  key_socketpair,/*(void *)key_einval, */
  key_accept,   /* (void *)key_einval, */
  key_getname,
  key_select,
  key_ioctl,    /* (void *)key_einval, */
  key_listen,   /* (void *)key_einval, */
  key_shutdown,
  key_setsockopt, /* (void *)key_einval, */
  key_getsockopt, /* (void *)key_einval, */
  key_fcntl,    /* (void *)key_einval, */
  key_sendmsg,
  key_recvmsg
};

void key_init(void)
{
  printk("NRL key management engine, Linux version 0.01\n");
  sock_register(key_proto_ops.family, &key_proto_ops);
  key_inittables();
};

#ifdef MODULE
void init_module(void)
{
  key_init();
};

void cleanup_module(void)
{
  key_freetables();
  sock_unregister(key_proto_ops.family);
};
#else /* MODULE */
void key_proto_init(struct protocol *p)
{
  key_init();
};
#endif /* MODULE */
