/***************************************
  $Revision: 1.18 $

  which_keytypes:  Determine which keys to look for.
  
  This is based on the existing Perl code. 

  Authors: ottrey, marek

  ******************/ /******************
  Copyright (c) 1999                              RIPE NCC
 
  All Rights Reserved
  
  Permission to use, copy, modify, and distribute this software and its
  documentation for any purpose and without fee is hereby granted,
  provided that the above copyright notice appear in all copies and that
  both that copyright notice and this permission notice appear in
  supporting documentation, and that the name of the author not be
  used in advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.
  
  THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
  AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
  DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
  OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
  ***************************************/
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <libgen.h>
#include <glib.h>

#include "bitmask.h"
#include "memwrap.h"

#define  WK_IMPL
#include "which_keytypes.h"
#include <regex.h>


#define DOMAINNAME "^[ ]*[a-zA-Z0-9-]*(\\.[a-zA-Z0-9-]+)*[ ]*$"
/* add a constraint: there must be at least one character in the domain name
   because the TLD must not be composed of digits only */
#define DOMAINALPHA  "[a-zA-Z]"

#define VALIDIP6PREFIX "^[0-9A-F:]*:[0-9A-F:/]*$"     /* at least one colon */
/* "^[0-9A-F]{1,4}(:[0-9A-F]{1,4}){7}$"*/

#define NET "^[0-9]{1,3}.[0-9]{1,3}.[0-9]{1,3}.[0-9]{1,3}$"

#define ASNUM "^AS[1-9]+[0-9]{0,4}$"

#define ASRANGE "^AS[0-9]{1,5}[ ]*([-][ ]*AS[0-9]{1,5}){0,1}$"   /* [ ]*(-[ ]*AS[0-9]+)?   */

#define NETNAME "^[A-Z][A-Z0-9-]*$"

#define MAINTAINER "^[A-Z][A-Z0-9-]*$"

#define LIMERICK "^LIM-[A-Z0-9-]+$"

#define KEYCERT "^PGPKEY-[0-9A-F]{8}$"

#define ROUTESETNAME "^RS-[A-Z0-9_-]*$"

#define ASSETNAME "^AS-[A-Z0-9_-]*$"

#define AUTONICPREFIXREGULAR "^AUTO-"

#define IPRANGE "^[0-9]{1,3}(\\.[0-9]{1,3}){0,3}[ ]*-[ ]*[0-9]{1,3}(\\.[0-9]{1,3}){0,3}$"

#define IPADDRESS "^[0-9.]+$"

#define IPPREFIX "^[0-9.]+/[0-9]+$"

#define PEERINGSET "^PRNG-"

#define FILTERSET  "^FLTR-"

#define RTRSET     "^RTRS-"

#define NICHANDLE "^[A-Z0-9-]+$"

/*
  XXX This seems to be the same as the Perl code.  But I don't see where a " " is allowed for.
  I.e. Perl -> ^[a-zA-Z][\w\-\.\'\|\`]*$
  Does \w include [ ;:,?/}{()+*#] ?
#define NAME_B "^[a-zA-Z][a-zA-Z_0-9.'|`-]*$"
*/
#define NAME_B "^[a-zA-Z][a-zA-Z_0-9.'|`;:,?/}{()+*#&-]*$"

#define VALIDIP4PREFIX

#define EMAIL "^[.a-zA-Z0-9-]*@[a-zA-Z0-9-]*(\\.[a-zA-Z0-9-]+)*$"

/*****************************************************************/

static int perform_regex_test(const char *pattern, char *string)
{

  int match = 0;

  /* These are not used, since REG_NOSUB is specified in regcomp() */
  size_t nmatch = 0;
  regmatch_t pmatch[1];
  regex_t re;
  int err;

  if( (err = regcomp(&re, pattern, REG_EXTENDED|REG_NOSUB)) != 0) 
    {
      char erbuf[2048];      
      regerror(err,&re,erbuf,sizeof(erbuf));
      die;
    }
  if (regexec(&re, string, nmatch, pmatch, 0))
    match = 0;
  else
    match = 1;

  regfree(&re);

  return(match);

} 

/*
 I split the isname up into isname_a & isname_b.  And created isname_ab to join them together.
  - So I can test it properly.  -ottrey
 */
#if 0
static int isname_a(char *string) {
    return perform_regex_test(AUTONICPREFIXREGULAR, string);
}

static int isname_b(char *string) {
    return perform_regex_test(NAME_B, string);
}

static int isname_ab(char *string) {
    return (isname_a(string) || isname_b(string));
}
#endif 

static int wk_is_name(char *key) {
    /* Everything matches to name */
    return 1;
} /* wk_is_name() */

/***************************************************************/

static int isdomname(char *string) {
    return (    perform_regex_test(DOMAINNAME, string)
	     && perform_regex_test(DOMAINALPHA, string));
}

static int wk_is_domain(char *key) {
    return isdomname(key);
} /* wk_is_domname() */

static int wk_is_iprange(char *key) {
  return perform_regex_test(IPRANGE, key);
} /* wk_is_iprange() */

static int wk_is_hostname(char *key) {
  /* XXX Why is there a hostname & a domainname? */
  /* Answer - hostname can be a domainname or an IP */
    return (isdomname(key) || wk_is_iprange(key));
} /* wk_is_hostname() */

/* WK_to_string() */
/*++++++++++++++++++++++++++++++++++++++
  Convert the which keytypes bitmap into a string.

  mask_t wk The which keytypes mask to be converted.

  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
char *WK_to_string(mask_t wk) {

  return MA_to_string(wk, Keytypes);

} /* WK_to_string() */

/* WK_new() */
/*++++++++++++++++++++++++++++++++++++++
  Create a new which keytypes bitmap.

  char *key The key to be examined.

  More:
  +html+ <PRE>
  Authors:
        ottrey
  +html+ </PRE><DL COMPACT>
  +html+ <DT>Online References:
  +html+ <DD><UL>
  +html+ </UL></DL>

  ++++++++++++++++++++++++++++++++++++++*/
mask_t WK_new(char *key) {
  mask_t wk; 

  wk = MA_new(MA_END);

  MA_set(&wk, WK_NAME,         wk_is_name(key));
  MA_set(&wk, WK_NIC_HDL,      perform_regex_test(NICHANDLE, key));
  MA_set(&wk, WK_EMAIL,        perform_regex_test(EMAIL, key));
  MA_set(&wk, WK_MNTNER,       perform_regex_test(MAINTAINER, key));
  MA_set(&wk, WK_KEY_CERT,     perform_regex_test(KEYCERT, key));
  MA_set(&wk, WK_IPADDRESS,    perform_regex_test(IPADDRESS, key));
  MA_set(&wk, WK_IPRANGE,      wk_is_iprange(key));
  MA_set(&wk, WK_IPPREFIX,     perform_regex_test(IPPREFIX, key));
  MA_set(&wk, WK_IP6PREFIX,    perform_regex_test(VALIDIP6PREFIX, key));
  MA_set(&wk, WK_NETNAME,      perform_regex_test(NETNAME, key));
  MA_set(&wk, WK_NET6NAME,     perform_regex_test(NETNAME, key));
  MA_set(&wk, WK_AUTNUM,       perform_regex_test(ASNUM, key));
  MA_set(&wk, WK_ASSETNAME,    perform_regex_test(ASSETNAME, key));
  MA_set(&wk, WK_ROUTESETNAME, perform_regex_test(ROUTESETNAME, key));
  MA_set(&wk, WK_DOMAIN,       wk_is_domain(key));
  MA_set(&wk, WK_HOSTNAME,     wk_is_hostname(key));
  MA_set(&wk, WK_LIMERICK,     perform_regex_test(LIMERICK, key));
  MA_set(&wk, WK_ASRANGE,      perform_regex_test(ASRANGE, key));
  MA_set(&wk, WK_PEERINGSET,   perform_regex_test(PEERINGSET, key));
  MA_set(&wk, WK_FILTERSET,    perform_regex_test(FILTERSET, key));
  MA_set(&wk, WK_RTRSET,       perform_regex_test(RTRSET, key));
  
  return wk;

} /* WK_new() */
