/** ORACLE -- a general-purpose question answering program
 **
 **  => Before running Oracle:
 **	1) set up the "Adjustable #define's" below
 **	2) create Oracle's directories, and write several initial questions
 **	   (see "Oracle question files" below)
 **
 ** Unix implementation by Adlai Waksman, University of Pennsylvania (Fall '88)
 **    <waksman@eniac.seas.upenn.edu>, <waksman@grasp.cis.upenn.edu>
 **---------------------------------------------------------------------------
 ** Copyright (C) 1988 A. Waksman
 ** This program may be freely distributed, ported, and modified as long as
 ** it is not sold or used for profit or commercial advantage, and this notice
 ** remains intact.
 **---------------------------------------------------------------------------
 **
 ** ...inspired by a VMS .COM file at the Johns Hopkins University
 **    designed and created by Erica Liebman, 
 **    expanded and improved by Eric Staubly, Adlai Waksman, and Ken Arromdee
 **
 ** ...which was in turn inspired by a program at Bell Labs...
 ** ...whose origins are shrouded in the mists of antiquity...
 ***************************************************************************
 ** This program is relatively bug-free, but "Nothing is foolproof because
 ** fools are so ingenious."  (--Eric S. and others)
 ** >>>>>>>> It is distributed "AS IS", WITH NO WARRANTY OF ANY KIND. <<<<<<<<
 ***************************************************************************
 ** This program is very 4.2bsd/Ultrix specific with things like
 ** file locking and mail sending.  Feel free to port it to other systems.
 **
 ** RUN IT SETUID/GID AT YOUR OWN RISK!  Here at Penn we keep the
 ** question directory writable to everyone, but hidden under a couple of
 ** execute-only directories, and things work fine.
 ** DO NOT SET THE 'T' (01000) BIT on the directory.  Oracle needs to be
 ** able to rename and delete files in the question directory that do not
 ** belong to the user.
 **----------------------------------------------------------------
 ** There's plenty of room for improvement; please send me your modifications
 ** so that other people can take advantage of them as well.
 **----------------------------------------------------------------
 ** Enjoy!			--AW
 **/

#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/dir.h>
#include <sys/stat.h>
#include <ctype.h>
#include <pwd.h>
#include <sys/signal.h>
#include <sys/errno.h>

/** Adjustable #define's:
 **	VERSION		string to print after "Welcome to Oracle"
 **	QFIL_DIR	directory where question files are stored -
 **			everyone must have read/write/execute permission
 **			(program chdir's to this directory)
 **	TEMP_DIR	directory where temporary mail-message files
 **			are created.  Again, this must be unprotected.
 **	OWNER_CHECK	If defined, oracle won't use question files
 **			owned by the user (so they won't get their own
 **			questions).
 **	MODE_CHECK	If defined, oracle won't use question files
 **			writable by the user (so they won't get their own
 **			questions; this does allow people to "plant" read-
 **			only files for themselves).
 **	GUARDIAN	If defined (as a string), oracle will use this
 **			address for a Bcc: line (so the guardian can get
 **			copies of questions and answers).
 **	MOTD		If defined (as a string), oracle will print the
 **			"message-of-the-day" contained in this filename.
 **/

#define VERSION  "(v1.1)"

#define QFIL_DIR "/usr/games/lib/oracle"
#define TEMP_DIR "/usr/games/lib/oracle"
#define OWNER_CHECK
/* #define GUARDIAN "waksman" */
#define MOTD	"/usr/games/lib/oracle/.motd"


/****************************************************************

		***************************
		** Oracle question files **
		***************************

Oracle must be given a set of initial questions before people can use
it.  Create several (4 to 8) question files (q0001, q0002, q0003, etc.)
in QFIL_DIR, with your username on the first line of each, for example:

	waksman
	Why is the sky blue?
	(And don't say "because it reflects the sea".)

Make sure the question files are readable by everyone, and that QFIL_DIR
has full access (0777 or a=rwx).

Answers will eventually come to you in the mail; this shows that Oracle
is working and people are using it.

Oracle requires the user to answer a question ("to compare one of its
brilliant ... answers to one of yours") before being allowed to ask one
of his/her own.  This is how the Oracle gets its answers!  It mails
these users' answers to the people who originally asked the questions.

A queue of questions is stored in the directory QFIL_DIR, each question
in a separate file.  Filenames are numbered: q0001, q0002, q0003, etc.
Oracle asks the user the question in the lowest-numbered file, and 
stores the user's new question in a new file whose number is higher
than existing question numbers.  The number of questions remains
constant -- unless you add or remove question files yourself -- but
the filename's numbers keep increasing.
[Renaming and a lock file are used, so several people can run Oracle
 at the same time.]

The first line of a question file gives the username to which to mail
the answer.  Everything after the first line is the question itself.

Having more questions in the queue allows a user to have more questions
"pending" for the Oracle to consider, but it will take longer for people
to get their answers back.

(If the queue contains no questions, or only the user's own questions,
Oracle will exit with "The Oracle is very busy now; please try again
later.)

****************************************************************/

/* --- parameters for filenames and the question queue --- */

#define Q_PREFIX "q"		/* prefix for "qnnn" queued question files */
#define OLDQ_FMT "r%ld"		/* OLDQ_FMT and NEWQ_FMT take a PID */
#define NEWQ_FMT "w%ld"
#define TEMP_FMT "%s/orM%ld"	/* takes a directory string and a PID */

#define NAMESIZE  64	/* filenames' significant (for us) length */
#define MIN_WIDTH  4	/* minimum width for nnn in "qnnn" filenames */
#define DEF_WIDTH  8	/* default width for numeric field */

/* --- definitions for lock file handling --- */

#define RETRY_LIMIT 30	/* for q.directory lock file */
#define RETRY_DELAY  1

#define BEGIN_CRITICAL	{ old_sigmask = sigblock(~0); open_lockfile(); }
#define END_CRITICAL	{ sigsetmask(old_sigmask); close_lockfile(); }

/* ------ */

static char IdEnTiFy[] = "Oracle - A. Waksman @upenn 1988";

int old_sigmask = 0;

#define equal(str1,str2)  (!strcmp(str1,str2))

char oldq_name[NAMESIZE], newq_name[NAMESIZE], temp_name[255];
char login_name[NAMESIZE];
int login_uid;
FILE *newq;
short brief_flag = 0;


main(argc,argv)
     int argc;
     char **argv;
{

  brief_flag = (argc > 1) && equal(argv[1], "-b");
  init();
  welcome();

  grab_old_q();
#ifdef DEBUG
fprintf(stderr,"oldq_name=%s\n", oldq_name);
#endif
  get_answer();

  get_question(newq);
  send_answer();
  normal_wrapup();
}


errchk(stat, string)
     int stat;
     char *string;
{
  if (stat < 0) {
    perror(string);
    exit(-stat);
  }
}

errchkcln(stat, string)
     int stat;
     char *string;
{
  if (stat < 0) {
    perror(string);
    cleanup();
  }
}



init()
{
  long pid = getpid();
  char *getlogin();
  register char *n = getlogin();
  struct passwd *p;

  login_uid = getuid();

  if ((!n || !*n) && (p=getpwuid(login_uid)))
    n = p->pw_name;
  if (!n) {
    fprintf(stderr, "You don't exist.  Boy, am I confused!\n");
    exit(666);
  }
  else if (!*n) {
    fprintf(stderr, "You have no name.  Boy, am I confused!\n");
    exit(667);
  }
  strcpy(login_name, n);

  sprintf(oldq_name, OLDQ_FMT, pid);
  sprintf(newq_name, NEWQ_FMT, pid);
  sprintf(temp_name, TEMP_FMT, TEMP_DIR, pid);

/* set umask and directory BEFORE doing file ops */

  (void) umask(0022);
  errchk(chdir(QFIL_DIR), QFIL_DIR);

/* open new qfil now so user won't get error msg in middle */

  if ( ! (newq=fopen(newq_name, "w")) ) {	
    perror(newq_name);
    exit(1);
  }

}

welcome()
{
  FILE *motd;
  int c;

  printf("Welcome to Oracle %s\n", VERSION);
  if (!brief_flag) {
    printf("(AAW88 - thanks to EJL et al.)\n");
#ifdef MOTD
    if ((motd = fopen(MOTD, "r")) != NULL) {
      while ((c = getc(motd)) != EOF)
	putchar(c);
      fclose(motd);
    }
#endif
  }
  printf("\n\n");
}

grab_old_q()
{
  char *old;

  BEGIN_CRITICAL;
  getoldfilename(Q_PREFIX, &old);
  if (! *old) {
    printf("The Oracle is very busy now; please try again later.\n");
    unlink(newq_name);
    exit(2);
  }
  errchk(rename(old, oldq_name), "rename");
  setup_cleanup();
  END_CRITICAL;
}


get_answer()
{
  register FILE *q, *a;
  register char c;
  register short ok;

  if ( ! (q=fopen(oldq_name, "r")) ) {
    perror(oldq_name);
    exit(1);
  }
  if ( ! (a=fopen(temp_name, "w")) ) {
    perror(temp_name);
    exit(1);
  }

#ifdef GUARDIAN
  fprintf(a, "Bcc: %s\n", GUARDIAN);
#endif
  fprintf(a, "From: \"The Oracle\"\nTo: ");
  while (((c=getc(q)) != '\n') && c!=EOF)
    putc(c,a);
  fprintf(a, "\nSubject: Oracle answer\n\n");

  if ( ! brief_flag) {
    printf("You must first earn the privilege of asking the great Oracle\n");
    printf("a question.  Obviously you are of an inferior intellect;\n");
    printf("however, the Oracle wishes to compare one of its brilliant\n");
    printf("and perfect answers to one of yours.\n\n");
  }
  while ((c=getc(q)) != EOF) {
    putchar(c);
    putc(c,a);
  }
  fclose(q);
  fprintf(a, "\nThe Oracle has answered your question as follows:\n\n");

  printf("\nThat is the question you are to answer.\n");
  if (!brief_flag) {
    printf("Ponder it carefully for a few moments, then type a creative,\n");
    printf("interesting answer.\n");
    printf("When you are finished, type a control-D on a line by itself.\n");
  }
  putchar('\n');
  fflush(stdout);

  ok = 0;
  while ((c=getchar()) != EOF) {
    putc(c,a);
    ok = ok || (isascii(c) && isalnum(c));
  }
/*  putc('\n',a);	/*REMOVE?*/
  fclose(a);
  printf("---\n"); fflush(stdout);

  printf("\nThe Oracle is considering your answer...\n\n");
  if (!ok) {
    if (!brief_flag) sleep(3);
    printf("The Oracle has considered your answer and has found it to be\n");
    printf("unacceptable.  (Hint: no good answer could be that short.)\n");
    printf("Please try to give a better answer next time.\n");
    cleanup();
  }
}






get_question(newq)
     FILE *newq;
{
  register char c;
  register short ok;

  fprintf(newq, "%s\n", login_name);

  if (!brief_flag) {
    sleep(3);
    printf("The Oracle has considered your answer and has found it to be\n");
    printf("satisfactory.\n");
  }
  printf("You may now ask the Oracle a question.\n");
  if (!brief_flag) {
    printf("Type it in as before, ending your question with control-D\n");
    printf("on a line by itself.\n");
  }
  putchar('\n');
  fflush(stdout);
  ok = 0;
  while ((c=getchar()) != EOF) {
    putc(c,newq);
    ok = ok || (isascii(c) && isalnum(c));
  }
/*  putc('\n',newq);	/*REMOVE?*/
  fclose(newq); newq=NULL; /*(so cleanup() won't try to reclose it)*/
  printf("---\n"); fflush(stdout);

  if (!ok) {
    sleep(1);
    printf("The Oracle has found your question exceptionally trivial\n");
    printf("and has answered it as follows:\n\n            ==> MU <==\n\n\n");
    printf("Thank you for consulting the Oracle (and go ahead and ask\n");
    printf("more difficult questions!).\n");
    cleanup();
  }
  printf("\nOne moment please...\n");
}


send_answer()
{
  int maildesc;

  errchkcln(maildesc=open(temp_name, O_RDONLY), temp_name);
  if ( ! vfork() ) {
    dup2(maildesc, 0);
    close(maildesc);
    execl("/usr/lib/sendmail", "sendmail", "-oi", "-t", 0);
    /* --- problem if the exec didn't work --- */
    fprintf(stderr,
	    "ERROR: Can't run the mailer -- please report this to %s.\n",
#ifdef GUARDIAN
	    GUARDIAN
#else
	    "the Guardian of Oracle"
#endif
	    ); 
    _exit(999);
  }
  close(maildesc);
}

normal_wrapup()
{
  char *new;
  
#ifdef DEBUG
fprintf(stderr, "in normal_wrapup\n");
#endif
  BEGIN_CRITICAL;
  getnewfilename(Q_PREFIX, &new);
  errchk(rename(newq_name, new), newq_name);
  unlink(oldq_name);
  unsetup_cleanup();
  END_CRITICAL;
  printf("Thank you for consulting the Oracle.\n");
  printf("The Oracle will mail you your answer within a few days.\n");
  cleanup();
}

cleanup()
{
  char *new;

#ifdef DEBUG
fprintf(stderr, "in cleanup\n");
#endif
  if (newq) { fclose(newq); newq=0; }
#ifndef DEBUG
  unlink(newq_name);
  unlink(temp_name);
#endif
  if (access(oldq_name, F_OK) >= 0) {	/* r$$ exists; put back on queue */
    BEGIN_CRITICAL;
    getnewfilename(Q_PREFIX, &new);
    errchk(rename(oldq_name, new), oldq_name);
    END_CRITICAL;
  }
  exit(0);
}


int lock_fd = -1;
char lock_name[] = "lock";

open_lockfile()
{
  register int retries=0;

  while ((lock_fd=open(lock_name, O_RDONLY|O_CREAT|O_BLKANDSET|O_NDELAY, 0666))
	 < 0  &&  errno == EWOULDBLOCK) {
    if (retries++ > RETRY_LIMIT) {
      fputs("ERROR: lock file busy\n", stderr);
      exit(-1);
    }
    sleep(RETRY_DELAY);
  }
  if (lock_fd < 0) {
    perror("ERROR: lock file");
    exit(-1);
  }
}

close_lockfile()
{
  if (lock_fd >= 0)
    close(lock_fd);
  unlink(lock_name);
}
  
setup_cleanup()
{
  struct sigvec sv;
  register int s;

  sv.sv_mask = ~0;			/* let cleanup() catch "all" signals */
  sv.sv_handler = cleanup;
  sv.sv_onstack = 0;
  for (s=1; s<=NSIG; s++) sigvec(s, &sv, 0);

  sv.sv_handler = SIG_DFL;	/* don't catch URG, TSTP, CONT, CHLD, TTIN,
				   TTOU, IO, WINCH */
  sigvec(SIGURG, &sv, 0);
  sigvec(SIGWINCH, &sv, 0);
  for (s=18; s<=23; s++) sigvec(s, &sv, 0);
}

unsetup_cleanup()
{
  struct sigvec sv;
  register int s;

  sv.sv_handler = SIG_DFL;
  for (s=1; s<=NSIG; s++) sigvec(s, &sv, 0);
}



/********************* QUEUE DIRECTORY/FILENAME ROUTINES ******************/

char first[NAMESIZE], last[NAMESIZE];
char next[NAMESIZE], nextfmt[NAMESIZE];


getoldfilename(prefix, fp)
     char *prefix, **fp;
{
  DIR *dirp;
  struct direct *dp;

  first[0] = first[NAMESIZE-1] = '\0';
  if ( ! prefix )   prefix = "";

  if ((dirp=opendir(".")) == NULL) {
    perror("OLD scan");
    exit(1);
  }
  while ((dp=readdir(dirp)) != NULL) {
    if (prefixok(dp->d_name, prefix)) {
      if (!first[0] || strcmp(first,dp->d_name) > 0)
#ifdef MODE_CHECK
	if (access(dp->d_name, W_OK) < 0)
#endif
#ifdef OWNER_CHECK
	if (owner_ok(dp->d_name))
#endif
	strncpy(first, dp->d_name, NAMESIZE-1);
    }
  }
  *fp = first;
}

#ifdef OWNER_CHECK
owner_ok(file)		/* file is not ok if it belongs to the user */
     char *file;
{
  struct stat statbuf;

  if (stat(file,&statbuf) < 0)  return(1);	/* ??? - assume ok */
  if (statbuf.st_uid == login_uid)  return(0);
  return(1);
}
#endif


getnewestexisting(prefix, np)
     char *prefix, **np;
{
  DIR *dirp;
  struct direct *dp;

  last[0] = last[NAMESIZE-1] = '\0';
  if ( ! prefix )   prefix = "";

  if ((dirp=opendir(".")) == NULL) {
    perror("NEW scan");
    exit(1);
  }
  while ((dp=readdir(dirp)) != NULL) {
    if (prefixok(dp->d_name, prefix)) {
      if (!last[0] || strcmp(last,dp->d_name) < 0)
	strncpy(last, dp->d_name, NAMESIZE-1);
    }
  }
  *np = last;
}


getnewfilename(prefix, np)
     char *prefix, **np;
{
  register int n=0, width;
  char *last, *l2, *rest;

  getnewestexisting(prefix, &last);

  l2 = last + strlen(prefix);
  n = strtol(l2, &rest, 10);
  n++;
  if ((width = rest-l2) < MIN_WIDTH)
    width = DEF_WIDTH;
  sprintf(nextfmt, "%%s%%0%dd", width);
  sprintf(next, nextfmt, prefix, n);

  *np = next;
}


prefixok(name, pre)
     char *name, *pre;
{
  for ( ; *name && *pre ; name++, pre++) {
    if (*name != *pre)
      return(0);
  }
  return(*pre == '\0');
}
