
/****************************************************************************

    PROGRAM: Plan_Map.c - version 1.2

    PURPOSE: Prints the galaxy map of VGA PLANETS v3.0

    COMPILER: Borland C++ 3.1 for windows. (however, this is plain C-code)

    AUTHOR: Andre van den Bos, July-August 1993.
	    Enschede, the Netherlands.
	    bos@cs.utwente.nl

****************************************************************************/


#include <windows.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include "drivinit.h"
#include "commdlg.h"
#include "plan_map.h"

/*------------------------------------------------------------------------*/
/*    Constants                                                           */
/*------------------------------------------------------------------------*/

#define NUM_PLANETS       500   // PLANETS has a fixed number of planets
#define PLANETNAME_LENGTH 20    // As in file PLANET.NM
#define NUM_RACES         11    // There are 11 races

#define MAX_GALAXY        10000 // Max size of galaxy incl. borders
#define GALAXY_BORDER     10    // Size of galaxy borders (no planets there)
#define MIN_GALAXYSIZE    100   // Minimum galaxy size

#define PLANET_AREA_X (wGalaxyXTo-wGalaxyXFrom)
#define PLANET_AREA_Y (wGalaxyYTo-wGalaxyYFrom)

WORD wGalaxyXFrom = 1000;       // Default planets galaxy size
WORD wGalaxyYFrom = 1000;
WORD wGalaxyXTo   = 3000;
WORD wGalaxyYTo   = 3000;

#define MAX_GALAXYNAME    200   // Max. galaxy name incl. 0 terminator
#define DEFAULT_GALAXY_NAME "Standard galaxy created by HOST.EXE."

/* Constants for the scroll bars. In galaxy units */
#define SCROLL_STEP       10

/* For the print dialog window */
#define WM_SETPAGES       200

/* Length of the race names (long form, plural form, adjective form) */
#define RACE_LENGTH(j) (((j)==0)?30:(((j)==1)?20:12))

/* Maximum zoom factor and ID for update-variables in window message */
#define MAXZOOM    36
#define IDM_UPDATE 300

/* Maximum number of ships */
#define MAXSHIP    500

/*------------------------------------------------------------------------*/
/*    TYPE definitions                                                    */
/*------------------------------------------------------------------------*/

typedef struct
{  char szPlanetName[PLANETNAME_LENGTH+1];
   WORD xCoord, yCoord;
   BYTE Char1;           // Contains a character of the galaxy name
   BYTE bUnplaced;       // Planet has not been placed yet
   int  ID;              // ID of planet
   UINT uiAlign;         // Alignment of planet name
} PLANET;

typedef struct
{  int  bNames;             // Display names
   int  bIDs;               // Display planet numbers
   int  bGrid;              // Display grid
   int  bMarkSheets;        // Display the printer sheets (green line)
   int  bXNumbers;          // if bGrid display horizontal numbers
   int  bYNumbers;          // if bGrid display vertical numbers
   WORD iGridSize[MAXZOOM]; // In planet-units for each iZoom
   int  iZoom;              // 1 planet unit = iZoom twips
   long xOffset, yOffset;   // Bottom-left orgin in planets-units
   char szText[40];         // Bottom text for printing entire galaxy map

   PRINTDLG     *pPd;       // Used to pass data struct to ZoomDlgProc
} DISPLAY_OPTIONS;

typedef struct
{  PLANET planet;           // Planet name and coordinates
   LONG   dx;               // Offset from delta-checksum
   BOOL   bOK;              // Planet member data is valid
} INFO_BAR_DATA;

typedef struct              // PLANET.HST
{ WORD wID;                 // ship's ID
  WORD wRace;               // owner of ship
  char sCode[3];            // Friendly code; three characters
  WORD wWarp;
  WORD w2, w3;
  WORD xCoord;              // Location of ship (xCoord, yCoord)
  WORD yCoord;
  WORD wDriveLevel;         // Tech level of engines
  WORD wShipType;           // Type of ship (12=super transport) (11=smaller)
  WORD wBeamLevel;          // Tech level of beams
  WORD wBeamWeapons;        // Number of ..
  WORD wFighterBays;        // Number of ..
  WORD wTorpLevel;          // Torp level
  WORD wTorps;              // Number of torps
  WORD wTorpsBays;          // Number of torp bays / fighters
  WORD wMission;
  WORD wEnemy;
  WORD wUnkown[2];
  WORD wCrew;               // Number of crew members on ship
  WORD wColonists;          // Number of colonists on ship
  char sName[20];           // Name of ship
  WORD wFuel, wTri, wDur, wMol;
  WORD wSupplies;           // Supplies on ship
  WORD wUnkown2[15];
  WORD wMoney;
} SHIPDATA;

/*------------------------------------------------------------------------*/
/*   Global variables                                                     */
/*------------------------------------------------------------------------*/

char      szNamesFile[]  = "PLANET.NM";  // File with planet names
char      szCoordsFile[] = "XYPLAN.DAT"; // File with planet coordinates
char      szRaceFile[]   = "RACE.NM";    // File with race names
char      szShipFile[]   = "SHIP.HST";   // File with initial ship-data
char      szAppName[]    = "Plan_Map";   // Name of class
char      szAppDescr[]   = "Galaxy map of VGA PLANETS v3.0";

HFONT     hfFont = 0;     	       // Selected font (init empty handle)
COLORREF  crColor      = RGB(0,0,0);   // Black planet names by default
COLORREF  crGridColor  = RGB(0,0,128); // Color of grid
COLORREF  crSheetColor = RGB(0,128,0); // Color to mark the sheets 
HINSTANCE hInst;		       // current instance
GLOBALHANDLE hPlanets = NULL;          // Handle to planet data
GLOBALHANDLE hShips   = NULL;          // Handle to ship data
int          iShipCount = 0;           // Number of ships in data file
int          iNewShipID = 0;           // Next ID for new ship

/* Information bar variables */
HWND            hDlgInfo=0;            // Handle to info bar dialog box
INFO_BAR_DATA   BarInfo;               // Data in info bar

/* Printer variables */
BOOL    bAbort;                        // FALSE if user cancels printing
HWND    hAbortDlgWnd;
FARPROC lpAbortDlg, lpAbortProc;
PRINTDLG pd;
WORD    xSheet, ySheet;                // Size of printer sheet in mm

/* Variable to store race names (1KB) */
char szRaceNames[NUM_RACES][3][31];   // Max. 30 + '\0' for race names

char szBeam[][20] =
{ "No beams", "Laser",           "X-Ray Laser", "Plasma Bolt",
  "Blaster",  "Positron Beam",   "Disruptor",   "Heavy Blaster",
  "Phaser",   "Heavy Disruptor", "Heavy Phaser" };

char szTorp[][20] =
{ "No torps",      "Mark 1 Photon", "Proton torp",   "Mark 2 Photon",
  "Gamma bomb",    "Mark 3 Photon", "Mark 4 Photon", "Mark 5 Photon",
  "Mark 6 Photon", "Mark 7 Photon", "Mark 8 Phoont" };

char szDrive[][30] =
{ "StarDrive 1",   "StarDrive 2", "StarDrive 3", "SuperStarDrive 4",
  "NovaDrive 5",   "HeavyNove Drive 6", "Quantum Drive 7",
  "Hyper Drive 8", "Transwarp Drive" };

/*------------------------------------------------------------------------*/
/*    Function prototypes                                                 */
/*------------------------------------------------------------------------*/

long FAR PASCAL WndProc     (HWND, WORD, WORD, LONG);
BOOL FAR PASCAL InfoDlgProc (HWND, WORD, WORD, LONG);

/*------------------------------------------------------------------------*/
/*    Main window procedure                                               */
/*------------------------------------------------------------------------*/

int PASCAL WinMain( HANDLE hInstance, HANDLE hPrevInstance,
		    LPSTR  lpszCmdLine, int nCmdShow )
{
   HWND     hwnd;
   MSG      msg;
   WNDCLASS wndclass;

   hInst = hInstance;

   if (!hPrevInstance)
   {	wndclass.style		= CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
	wndclass.lpfnWndProc	= (WNDPROC) WndProc;
	wndclass.cbClsExtra	= 0;
	wndclass.cbWndExtra	= 0;
	wndclass.hInstance	= hInstance;
	wndclass.hIcon		= LoadIcon(hInstance, "Plan_Map");
	wndclass.hCursor	= NULL;
	wndclass.hbrBackground  = GetStockObject(WHITE_BRUSH);
	wndclass.lpszMenuName   = szAppName;
	wndclass.lpszClassName	= szAppName;	

	RegisterClass(&wndclass);
   }


   hwnd = CreateWindow( szAppName, szAppDescr,
			WS_OVERLAPPEDWINDOW | WS_VSCROLL | WS_HSCROLL |
			WS_CLIPCHILDREN,
			CW_USEDEFAULT, CW_USEDEFAULT,
			CW_USEDEFAULT, CW_USEDEFAULT,
			NULL, NULL, hInstance, NULL);


   ShowWindow(hwnd, nCmdShow);
   UpdateWindow(hwnd);

   /* Create info bar (modeless dialog box) */
   hDlgInfo = CreateDialog( hInst, "InfoDlgBox", hwnd,
   			    MakeProcInstance(InfoDlgProc, hInst));

   while (GetMessage(&msg, NULL, 0, 0))
   {
      if ((hDlgInfo == 0) || !IsDialogMessage(hDlgInfo, &msg))
      {
         TranslateMessage(&msg);
	 DispatchMessage(&msg);
      };
   };
   return msg.wParam;
}

/*------------------------------------------------------------------------*/
/*    Planet the planet names such that no overlapping occures            */
/*------------------------------------------------------------------------*/


#define CHECKDIS 50 /* planets closer than distance affect place */

void OrderPlanetNames( GLOBALHANDLE hPlanets )
{
    PLANET FAR* planet;   // First planet in array
    PLANET FAR* planet1;
    PLANET FAR* planet2;
    LONG   dis, disMin, xDif, yDif;
    UINT   uiAlign;
    int    i,j;
    HCURSOR hOldCursor;

    hOldCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));
    randomize();
    planet1 = planet = (PLANET FAR*) GlobalLock(hPlanets);
    for (i=0; i<NUM_PLANETS; i++)
    {  planet2 = planet;
       uiAlign = TA_CENTER | TA_TOP;     // Default alignment
       disMin  = CHECKDIS*CHECKDIS;
       for (j=0; j<NUM_PLANETS; j++)
       { if ( j!=i )
         {  /* Fussy: assume length of planet name = 10X its height */
            xDif = ((LONG) planet1->xCoord - planet2->xCoord)/5;
	    yDif = (LONG) planet1->yCoord - planet2->yCoord;
	    dis  = xDif*xDif + yDif*yDif;
	    if ( dis < disMin )
	    {   disMin = dis;
		uiAlign=((xDif<0)?TA_RIGHT:TA_LEFT)|((yDif<0)?TA_TOP:TA_BOTTOM);
	    };

            /* Check duplicate name and correct if required */
	    if ( 0 == _fstrcmp(planet1->szPlanetName, planet2->szPlanetName))
	       _fstrcat( planet2->szPlanetName, " I" );

	    /* Check duplicate planet locations and correct if required */
	    if ( (planet1->xCoord==planet2->xCoord) &&
		 (planet1->yCoord==planet2->yCoord))
	    {  planet2->xCoord+=rand()%100-50; // New random location
               planet2->yCoord+=rand()%100-50; // Pray if this is a good location
            };
	 }; 
         planet2++;
       };
       planet1->uiAlign = uiAlign;
       planet1++;
    };
    GlobalUnlock(hPlanets);
    SetCursor(hOldCursor);
};


/*------------------------------------------------------------------------*/
/*    Write VGA PLANETS data into files                                   */
/*------------------------------------------------------------------------*/

int WritePlanets( GLOBALHANDLE hPlanets )
{
   PLANET FAR* planets;
   SHIPDATA FAR* ship;
   SHIPDATA LocalShip;
   char sName[max(31,PLANETNAME_LENGTH+1)];
   struct { WORD x, y, dummy; } iCoords;
   FILE *fNames;
   FILE *fCoords;
   FILE *fRaces;
   FILE *fShips;
   int i,j,r;

   // Open the PLANET files
   if ((fNames  = fopen( szNamesFile,  "wb" ))== NULL )
      return 1;
   if ((fCoords = fopen( szCoordsFile, "wb")) == NULL )
      { fclose(fNames); return 1; }
   if ((fRaces  = fopen( szRaceFile,   "wb" ))== NULL )
      { fclose(fNames); fclose(fCoords); return 1; }
   if ((fShips = fopen( szShipFile,    "w+b" )) == NULL )
      { fclose(fNames); fclose(fCoords); fclose(fRaces); return 1; }

   // Write to files
   planets = (PLANET FAR*) GlobalLock(hPlanets);   
   for (i=0; i<NUM_PLANETS; i++)
   {
      iCoords.x     = planets->xCoord;
      iCoords.y     = planets->yCoord;
      iCoords.dummy = (WORD) planets->Char1 + ((WORD)planets->bUnplaced<<8);

      _fstrcpy( sName, planets->szPlanetName );
      for (j=strlen(sName); j<PLANETNAME_LENGTH; j++)
	 sName[j] = ' ';

      fwrite( sName, PLANETNAME_LENGTH, 1, fNames );
      fwrite( &iCoords, sizeof(WORD),   3, fCoords );

      planets++;
   };
   GlobalUnlock(hPlanets);

   /* Write race names */
   for (r=0; r<3; r++)
      for (i=0; i<NUM_RACES; i++)
      {  
          strcpy( sName, szRaceNames[i][r] );
          for (j=strlen(sName); j<RACE_LENGTH(r); j++)
	      sName[j] = ' ';
	  fwrite( sName, RACE_LENGTH(r), 1, fRaces );
      };

   /* Write ship-data */
   ship = (SHIPDATA FAR *) GlobalLock(hShips);
   fseek( fShips, (LONG) sizeof(WORD), SEEK_SET );   // Overwrite at beginning of file
   for (i=0; i<MAXSHIP; i++, ship++)
   {  _fmemcpy( &LocalShip, ship, sizeof(SHIPDATA)); // Get ship in low mem.
      fwrite( &LocalShip, sizeof(SHIPDATA), 1, fShips );
   };
   GlobalUnlock(hShips); 

   // Close files
   fclose(fCoords);
   fclose(fNames);
   fclose(fRaces);
   fclose(fShips);
   return 0;
};



/*------------------------------------------------------------------------*/
/*    Read VGA PLANETS data files into memory                             */
/*------------------------------------------------------------------------*/

int ReadPlanets( GLOBALHANDLE hPlanets )
/* Read planet names and coordinates from data files into structure */
{
   PLANET   FAR* planets;
   SHIPDATA FAR* ship;
   SHIPDATA LocalShip;
   char sName[max(31,PLANETNAME_LENGTH+1)];
   struct { WORD x, y, dummy; } iCoords;
   FILE *fNames;
   FILE *fCoords;
   FILE *fRaces;
   FILE *fShips;
   int i,j,r, dummy;

   // Open the PLANET files
   if ((fNames  = fopen( szNamesFile,  "rb" )) == NULL )
      return 1;
   if ((fCoords = fopen( szCoordsFile, "rb" )) == NULL )
      { fclose(fNames); return 1; }
   if ((fRaces  = fopen( szRaceFile,   "rb" )) == NULL )
      { fclose(fNames); fclose(fCoords); return 1; }
   if ((fShips = fopen( szShipFile,    "rb" )) == NULL )
      { fclose(fNames); fclose(fCoords); fclose(fRaces); return 1; }

   // Read files
   planets = (PLANET FAR*) GlobalLock(hPlanets);   
   for (i=0; i<NUM_PLANETS; i++)
   {
      fread( sName,  PLANETNAME_LENGTH,  1, fNames );
      fread( &iCoords, sizeof(WORD),     3, fCoords );  // Read 3 words

      j=PLANETNAME_LENGTH;   // Place 0-terminator + remove last blanks
      do sName[j--]=0;
      while ((j>=0) && (sName[j]==' '));
      _fstrcpy( planets->szPlanetName, sName );

      planets->xCoord    = iCoords.x;
      planets->yCoord    = iCoords.y;
      planets->Char1     = iCoords.dummy & 0xFF;
      planets->bUnplaced = iCoords.dummy >> 8;
      planets->ID        = i+1;

      if (iCoords.dummy)
        switch (i)   // First four planets contain galaxy size
        {            // galaxy size = 0 -> Galaxy created by MASTER.EXE
	   case 0: wGalaxyXFrom  = (WORD) planets->Char1;      break;
	   case 1: wGalaxyXFrom += (WORD) planets->Char1 << 8; break;
	   case 2: wGalaxyXTo    = (WORD) planets->Char1;      break;
	   case 3: wGalaxyXTo   += (WORD) planets->Char1 << 8; break;
	   case 4: wGalaxyYFrom  = (WORD) planets->Char1;      break;
	   case 5: wGalaxyYFrom += (WORD) planets->Char1 << 8; break;
	   case 6: wGalaxyYTo    = (WORD) planets->Char1;      break;
	   case 7: wGalaxyYTo   += (WORD) planets->Char1 << 8; break;
        };

      planets++;
   };
   GlobalUnlock(hPlanets);

   /* Read race file */
   for (r=0; r<3; r++)
      for (i=0; i<NUM_RACES; i++)
      {  
	  fread( sName, j=RACE_LENGTH(r), 1, fRaces );  // Read race name    
          do sName[j--]=0;
          while ((j>=0) && (sName[j]==' '));               
          strcpy( szRaceNames[i][r], sName );
      };

   /* Read ships and dummy ships into high memory */
   ship       = (SHIPDATA FAR*) GlobalLock(hShips);
   fread( &dummy, sizeof(WORD), 1, fShips);
   iShipCount = 0;
   for (i=0; i<MAXSHIP; i++, ship++)
   {   fread( &LocalShip, sizeof(SHIPDATA), 1, fShips );
       _fmemcpy( ship, &LocalShip, sizeof(SHIPDATA)); // Copy ship into far memory
       if ( ship->wRace )  // Otherwise it's a dummy ship
       {   iShipCount++;
           if ( iNewShipID<=ship->wID)   // Update global variable for next ship ID
	      iNewShipID = ship->wID+1;
       };
   };
   GlobalUnlock(hShips); 

   // Close files
   fclose(fRaces);
   fclose(fCoords);
   fclose(fNames);
   fclose(fShips);
   return 0;
};


/*------------------------------------------------------------------------*/
/*    Display galaxy                                                      */
/*------------------------------------------------------------------------*/

void XYMap( HDC hdc, GLOBALHANDLE hPlanets, DISPLAY_OPTIONS *pDisOp,
            DWORD wMaxX, DWORD wMaxY )
{
    PLANET FAR* planet;
    LONG  i,x,y, x1, y1;
    int   nPage;
    char  szBuffer[80];
    DWORD dwSize;
    HFONT hOldFont;

    hOldFont = SelectObject(hdc, hfFont );
    SetTextColor(hdc, crColor);

    planet = (PLANET FAR*) GlobalLock(hPlanets);

    for (i=0; i<NUM_PLANETS; i++)
    { if ( !(planet->bUnplaced) ) // Don't display unplaced planets
      {
	/* One logical unit equals 1/1440 inch                    */
	/* pDisOp->iZoom 10 means 1 planet unit equals 1/144 inch */
        /* The galaxy is 2000x2000 units -> 13x13 inch map        */
        x = (LONG)(planet->xCoord - pDisOp->xOffset) * pDisOp->iZoom;
	y = (LONG)(planet->yCoord - pDisOp->yOffset) * pDisOp->iZoom;

	if (( x <= wMaxX ) && ( planet->xCoord > pDisOp->xOffset ) &&
	    ( y <= wMaxY ) && ( planet->yCoord > pDisOp->yOffset ))
	{
	    /* Display planet name & Nr if enabled */
	    wsprintf( szBuffer, "%s%s",
		      pDisOp->bNames?planet->szPlanetName:"",
		      (pDisOp->bIDs & pDisOp->bNames)?"/":"");
	    if ( pDisOp->bIDs )
		 wsprintf( szBuffer+strlen(szBuffer), "%d", planet->ID );
	    if ( *szBuffer)  
            {   SetTextAlign(hdc, planet->uiAlign);
		TextOut(hdc,x,y,szBuffer,_fstrlen(szBuffer));
            };

            /* Draw planet as a small rectangle */
	    Rectangle(hdc,x-20,y-20,x+20, y+20);
	};
      };
      /* Next planet */
      planet++;
    }; 
    GlobalUnlock(hPlanets);

    /* Display grid. iGridSize in planet units */
    if ( pDisOp->bGrid )
    {   SelectObject(hdc, CreatePen(PS_DOT, 0, crGridColor)); 
	SetTextColor(hdc, crGridColor);
	for (i=0; i<=MAX_GALAXY; i+=pDisOp->iGridSize[pDisOp->iZoom-1])
	{
	   x = (i - pDisOp->xOffset) * pDisOp->iZoom;
	   y = (i - pDisOp->yOffset) * pDisOp->iZoom;	   
	   dwSize = GetTextExtent( hdc, szBuffer, wsprintf(szBuffer, "%d", i));

	   if ((x<=wMaxX) && (i>=pDisOp->xOffset) && (x>LOWORD(dwSize)))
	   {   MoveTo( hdc, x, 0     );
	       LineTo( hdc, x, wMaxY-HIWORD(dwSize) );
	       if ( (x>360) && (pDisOp->bXNumbers) )
	       {  SetTextAlign(hdc, TA_CENTER | TA_TOP );
	          TextOut(hdc, x, wMaxY, szBuffer, strlen(szBuffer));
               };
	   };
	   if ((y<=wMaxY-HIWORD(dwSize)) && (i>=pDisOp->yOffset))
	   {   if ( (y<wMaxY-360) && (pDisOp->bYNumbers) )
	       {  MoveTo( hdc, LOWORD(dwSize),  y );
	          LineTo( hdc, wMaxX,           y );
	          SetTextAlign(hdc, TA_LEFT | TA_BASELINE );
		  TextOut(hdc, 0, y-HIWORD(dwSize)/3, szBuffer, strlen(szBuffer));
	       } else
	       {  MoveTo( hdc, 0,     y );
		  LineTo( hdc, wMaxX, y );
               }
	   };
	};
	DeleteObject(SelectObject(hdc, GetStockObject(BLACK_PEN)));       
    };

    /* Display sheets */
    if ((pDisOp->bMarkSheets) && xSheet && ySheet )
    {   SelectObject(hdc, CreatePen(PS_SOLID, 0, crSheetColor));
        SetTextColor(hdc, crSheetColor);
	x1 = (xSheet*56-1440)/pDisOp->iZoom;  // Sheet size in planet units
	y1 = (ySheet*56-2880)/pDisOp->iZoom;
        nPage = 0;
	for (i=wGalaxyXFrom; i<wGalaxyXTo; i+=x1)  // Global variables!
	{   x = (i - pDisOp->xOffset) * pDisOp->iZoom;
            nPage++;
	    if ((x<=wMaxX) && (i>=pDisOp->xOffset))
	    {   SetTextAlign(hdc, TA_LEFT | TA_TOP );
		TextOut( hdc, x, wMaxY-1440, szBuffer,
			 wsprintf(szBuffer, "%d", nPage));
		MoveTo( hdc, x, 0     );
		LineTo( hdc, x, wMaxY );
            };
	};
        nPage=0;
	for (i=wGalaxyYFrom; i<wGalaxyYTo; i+=y1)
	{   y = (i - pDisOp->yOffset) * pDisOp->iZoom;
            nPage++;
	    if ( (y<wMaxY-360) && (pDisOp->bYNumbers) )
	       {  SetTextAlign(hdc, TA_CENTER | TA_BOTTOM );
		  TextOut( hdc, 1440, y, szBuffer,
			   wsprintf(szBuffer, "%d", nPage));
		  MoveTo( hdc, 0,     y );
		  LineTo( hdc, wMaxX, y );
	       }
        };
        DeleteObject(SelectObject(hdc, GetStockObject(BLACK_PEN))); 
    };

    // Print page number 1/8" below galaxy map; only visible when printing
    SetTextAlign( hdc, TA_LEFT | TA_TOP );
    TextOut( hdc, 0, -180, pDisOp->szText, _fstrlen(pDisOp->szText));

    SelectObject(hdc, hOldFont);
};

/*------------------------------------------------------------------------*/
/*    Printer support procedures                                          */
/*------------------------------------------------------------------------*/

HDC GetPrinterDC2(void)
/* Update globals xSheet and ySheet and returns a handle to printer */
{
     char szPrinter[80];
     char *szDevice, *szDriver, *szOutput;
     HDC  hPrn;
     LPDEVNAMES lpDevNames;
     LPSTR      lpszDriverName;
     LPSTR      lpszDeviceName;
     LPSTR      lpszPortName;
     
     if ( pd.hDevNames )   /* User selected printer */
     {   lpDevNames = (LPDEVNAMES) GlobalLock(pd.hDevNames);
         lpszDriverName = (LPSTR) lpDevNames + lpDevNames->wDriverOffset;
         lpszDeviceName = (LPSTR) lpDevNames + lpDevNames->wDeviceOffset;
         lpszPortName   = (LPSTR) lpDevNames + lpDevNames->wOutputOffset;
	 hPrn = CreateDC(lpszDriverName, lpszDeviceName, lpszPortName, NULL);
	 GlobalUnlock(pd.hDevNames);
	 xSheet = GetDeviceCaps( hPrn, HORZSIZE );
	 ySheet = GetDeviceCaps( hPrn, VERTSIZE );
     }
     else  /* Default printer from win.ini */
     {
         GetProfileString("windows", "device", ",,,", szPrinter, 80);

         if (((szDevice = strtok (szPrinter, ",")) != 0) &&
	     ((szDriver = strtok (NULL,      ",")) != 0) &&
	     ((szOutput = strtok (NULL,      ",")) != 0))
	     {  hPrn  = CreateDC(szDriver, szDevice, szOutput, NULL);
	        xSheet = GetDeviceCaps( hPrn, HORZSIZE );
	        ySheet = GetDeviceCaps( hPrn, VERTSIZE );
	     }
	     else
		hPrn = xSheet = ySheet = 0;  // No printer no sheets
     }
     return hPrn;

}


/****************************************************************************

    FUNCTION: GetPrinterDC()

    PURPOSE:  Get hDc for current device on current output port according to
              info in WIN.INI.

    COMMENTS:

        Searches WIN.INI for information about what printer is connected, and
        if found, creates a DC for the printer.

        returns
            hDC > 0 if success
            hDC = 0 if failure

****************************************************************************/

HDC GetPrinterDC(HWND hwnd)
{

    HDC         hDC;
    LPDEVMODE   lpDevMode = NULL;
    LPDEVNAMES  lpDevNames;
    LPSTR       lpszDriverName;
    LPSTR       lpszDeviceName;
    LPSTR       lpszPortName;

    if (!PrintDlg((LPPRINTDLG)&pd))
	 return(NULL);

    if (pd.hDC)
	hDC = pd.hDC;
    else
      {
	if (!pd.hDevNames)
            return(NULL);

	lpDevNames = (LPDEVNAMES)GlobalLock(pd.hDevNames);
        lpszDriverName = (LPSTR)lpDevNames + lpDevNames->wDriverOffset;
        lpszDeviceName = (LPSTR)lpDevNames + lpDevNames->wDeviceOffset;
        lpszPortName   = (LPSTR)lpDevNames + lpDevNames->wOutputOffset;
	GlobalUnlock(pd.hDevNames);

	if (pd.hDevMode)
	    lpDevMode = (LPDEVMODE)GlobalLock(pd.hDevMode);

        hDC = CreateDC(lpszDriverName, lpszDeviceName, lpszPortName, (LPSTR)lpDevMode);

	if (pd.hDevMode && lpDevMode)
            GlobalUnlock(pd.hDevMode);
      }

    xSheet = GetDeviceCaps( hDC, HORZSIZE );
    ySheet = GetDeviceCaps( hDC, VERTSIZE );
    return(hDC);
}

/****************************************************************************

    FUNCTION: AbortProc()

    PURPOSE:  Processes messages for the Abort Dialog box

****************************************************************************/

int FAR PASCAL AbortProc(HDC hPr, int Code)
{
    MSG msg;

    if (!hAbortDlgWnd)              /* If the abort dialog isn't up yet */
        return(TRUE);

    /* Process messages intended for the abort dialog box */
    while (!bAbort && PeekMessage(&msg, NULL, NULL, NULL, TRUE))
	if (!IsDialogMessage(hAbortDlgWnd, &msg))
	{
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }

    /* bAbort is TRUE (return is FALSE) if the user has aborted */

    return (!bAbort);
}

/****************************************************************************

    FUNCTION: AbortDlg(HWND, unsigned, WORD, LONG)

    PURPOSE:  Processes messages for printer abort dialog box

    MESSAGES:

        WM_INITDIALOG - initialize dialog box
        WM_COMMAND    - Input received

    COMMENTS

        This dialog box is created while the program is printing, and allows
        the user to cancel the printing process.

****************************************************************************/

int FAR PASCAL AbortDlg(HWND hDlg, WORD msg, WORD wParam, LONG lParam)
{
    switch(msg) {

        /* Watch for Cancel button, RETURN key, ESCAPE key, or SPACE BAR */

	case WM_COMMAND:
	    if ( wParam == WM_SETPAGES )
	    {  SetWindowText( GetDlgItem(hDlg, IDC_PRINTPAGE), (LPSTR) lParam);
	       InvalidateRect( GetDlgItem(hDlg, IDC_PRINTPAGE), NULL, TRUE);
               return TRUE;
	    }
	    else return (bAbort = TRUE);

        case WM_INITDIALOG:

            /* Set the focus to the Cancel box of the dialog */

            SetFocus(GetDlgItem(hDlg, IDCANCEL));
            return (TRUE);
        }
    return (FALSE);
}


/*------------------------------------------------------------------------*/
/*    Main printing procedure                                             */
/*------------------------------------------------------------------------*/

void PrintPage( HWND hwnd, GLOBALHANDLE hPlanets, DISPLAY_OPTIONS *pDisOp )
{
   WORD            x, y, yOrg, Org;
   char            szBuffer[80];
   int             Status, iPages, cx, cy, xPage, yPage;
   int             iCount = 0;  // Simple page counter
   HCURSOR         hSaveCursor;
   LONG            i, j;
   DISPLAY_OPTIONS DisOp = *pDisOp;
   DOCINFO         DocInfo =
   		   { sizeof(DOCINFO), "VGA Planets 3.0 galaxy map", NULL };

   hSaveCursor = SetCursor(LoadCursor(NULL, IDC_WAIT));

   DisOp.bMarkSheets = FALSE;  // Don't PRINT page borders

   /* Determine number of pages required for a complete galaxy map */
   cx = LOWORD(1+((LONG)PLANET_AREA_X*DisOp.iZoom)/((LONG)xSheet*56-1440));
   cy = LOWORD(1+((LONG)PLANET_AREA_Y*DisOp.iZoom)/((LONG)ySheet*56-2880));
   pd.nFromPage = pd.nMinPage = 11;      // Meaning page (1,1)
   pd.nToPage   = pd.nMaxPage = cx*10+cy;// Meaning page (cx,cy); assume cx<10
   
   if ( !GetPrinterDC(hwnd) )
   {   SetCursor(hSaveCursor);      /* Remove the hourglass */
       return;
   };

   if ( !(pd.Flags & (PD_SELECTION | PD_PAGENUMS)) )
   {  pd.nFromPage = pd.nMinPage;
      pd.nToPage   = pd.nMaxPage;
   };

   if ( (pd.nFromPage/10 > pd.nToPage/10) ||
	(pd.nFromPage%10 > pd.nToPage%10) )
   {  MessageBox( hwnd, "Invalid range.\nNote: page 12 means page (1,2).",
                  szAppDescr, MB_OK | MB_ICONEXCLAMATION );
      return;
   };

   lpAbortDlg =  MakeProcInstance(AbortDlg, hInst);
   lpAbortProc = MakeProcInstance(AbortProc, hInst);

   /* Define the abort function */
   bAbort       = FALSE;                   // Clears the abort flag
   hAbortDlgWnd = NULL;                    // Abort window not up yet
   SetAbortProc( pd.hDC, lpAbortProc );

   if ( StartDoc( pd.hDC, &DocInfo )<=0 )
   {
   	MessageBox(hwnd, "Unable to start print job", NULL, MB_OK | MB_ICONHAND);
        FreeProcInstance(lpAbortDlg);
        FreeProcInstance(lpAbortProc);
	DeleteDC(pd.hDC);
	SetCursor(hSaveCursor);      /* Remove the hourglass */
        return;
   }

   x = GetDeviceCaps(pd.hDC, HORZSIZE)*56-1440; // 1mm = 56 Twips + .5" margins
   y = GetDeviceCaps(pd.hDC, VERTSIZE)*56-2880; // 1" margins
   Org = wGalaxyXFrom-720/DisOp.iZoom;         // .5" in planet units
   yOrg = GetDeviceCaps(pd.hDC, VERTRES)-
	  GetDeviceCaps(pd.hDC, LOGPIXELSY)/2;  // 0.5" from bottom left
   iPages = (1+pd.nToPage%10 - pd.nFromPage%10)*
	    (1+pd.nToPage/10 - pd.nFromPage/10);

   /* Create the Abort dialog box (modeless) */
   hAbortDlgWnd = CreateDialog(hInst, "AbortDlg", hwnd, lpAbortDlg);
   ShowWindow (hAbortDlgWnd, SW_NORMAL);

   /* Disable the main window to avoid reentrancy problems */
   EnableWindow(hwnd, FALSE);
   SetCursor(hSaveCursor);      /* Remove the hourglass */

   xPage = yPage = 0;  // Page numbers
   if ( !(pd.Flags & PD_SELECTION) )
   {   /* Entire galaxy or part of it */
       for (i=0; i<PLANET_AREA_X; i+=x/DisOp.iZoom)
       {      xPage++; yPage=0;
       	      for (j=0; j<PLANET_AREA_Y; j+=y/DisOp.iZoom)
	      {   yPage++;
		  if ( (xPage>=pd.nFromPage/10) && (xPage<=pd.nToPage/10) &&
		       (yPage>=pd.nFromPage%10) && (yPage<=pd.nToPage%10) )
                  {
		     wsprintf( szBuffer, "Printing page %d of %d.", ++iCount, iPages);
	             SendMessage( hAbortDlgWnd, WM_COMMAND, WM_SETPAGES, (LONG) (LPSTR) szBuffer ); 
		     StartPage( pd.hDC );
		     SetMapMode(pd.hDC, MM_TWIPS);
		     SetViewportOrg( pd.hDC, 0, yOrg );

		     /* Effective print area: (x,y), use 1/2" overlap at top,
		        bottom, left & right of page */
		     DisOp.xOffset   = Org+i;
		     DisOp.yOffset   = Org+j;
		     DisOp.bYNumbers = (xPage==pd.nFromPage/10);
		     DisOp.bXNumbers = (yPage==pd.nToPage%10);
		     wsprintf( DisOp.szText, "Page (%d,%d)", xPage, yPage );
		     XYMap(pd.hDC, hPlanets, &DisOp, x+1440, y+1440 );

		     Status = EndPage( pd.hDC );
		     if (Status < 0 || bAbort)    // Stop printing when error
			break;
                  };
	      };
        };
   }
   else
   {   /* One page, orgin bottom left of windows-screen */
	  SendMessage( hAbortDlgWnd, WM_COMMAND, WM_SETPAGES,
	               (LONG) (LPSTR) "Printing page 1 of 1.");
	  StartPage( pd.hDC );
	  SetMapMode(pd.hDC, MM_TWIPS);
	  SetViewportOrg( pd.hDC, 0, yOrg );
	  XYMap(pd.hDC, hPlanets, &DisOp, x, y );
	  Status = EndPage( pd.hDC );
   };

   if (Status >= 0 && !bAbort)   // End document if no error
     EndDoc( pd.hDC );
   EnableWindow(hwnd, TRUE);

   DestroyWindow(hAbortDlgWnd);
   FreeProcInstance(lpAbortDlg);
   FreeProcInstance(lpAbortProc);
   DeleteDC(pd.hDC);
};


/*------------------------------------------------------------------------*/
/*    Zoom dialog proc                                                    */
/*------------------------------------------------------------------------*/

BOOL FAR PASCAL ZoomDlgProc( HWND hDlg, WORD message,WORD wParam, LONG lParam)
{
  static DISPLAY_OPTIONS* pDisOp;
  static WORD wWidth, wHeight;  // Paper size (printer) in 1/10 Inch
  char   szBuffer[40];
  int    iSizeX, iSizeY;

  switch(message)
  {
    case WM_INITDIALOG:

	pDisOp = (DISPLAY_OPTIONS*) lParam;

	if ((xSheet!=0) && (ySheet!=0))
        {
	   wWidth  = 0.5+xSheet/2.54;                   // 1mm = 1/25.4"
	   wHeight = 0.5+ySheet/2.54;
	}
	else
           wWidth = wHeight = 0;
	
	/* Initialize scroll bars */
	SetScrollRange(GetDlgItem(hDlg,ID_SCROLL), SB_CTL, 1, MAXZOOM, FALSE);
	SetScrollPos  (GetDlgItem(hDlg,ID_SCROLL), SB_CTL, pDisOp->iZoom, TRUE);

        SendMessage(hDlg, WM_COMMAND, IDM_UPDATE, 0L);
	return FALSE;

    case WM_VSCROLL:
        switch(wParam)
	{  case SB_LINEUP:        pDisOp->iZoom--;     break;
	   case SB_LINEDOWN:      pDisOp->iZoom++;     break;
	   case SB_PAGEUP:        pDisOp->iZoom -= 4;  break;
	   case SB_PAGEDOWN:      pDisOp->iZoom += 4;  break;
	   case SB_THUMBPOSITION: pDisOp->iZoom = LOWORD(lParam); break;
	};
	pDisOp->iZoom=max(1,min(pDisOp->iZoom,MAXZOOM));

	if ( pDisOp->iZoom != GetScrollPos(GetDlgItem(hDlg,ID_SCROLL), SB_CTL))
	{  SetScrollPos(GetDlgItem(hDlg,ID_SCROLL), SB_CTL, pDisOp->iZoom, TRUE);
           InvalidateRect(GetParent(hDlg), NULL, TRUE);
           SendMessage(hDlg, WM_COMMAND, IDM_UPDATE, 0L);
	}
	return FALSE;

    case WM_COMMAND:
	switch(wParam)
	{
	  case IDM_UPDATE:  // No control, update text in dialog box

              /* Update size of the map */
	      iSizeX = (DWORD) PLANET_AREA_X*pDisOp->iZoom/144;  // in 1/10" units
              iSizeY = (DWORD) PLANET_AREA_Y*pDisOp->iZoom/144;
	      wsprintf( szBuffer, "Galaxy map %d x %d", iSizeX/10, iSizeY/10);
	      SetWindowText(GetDlgItem(hDlg,ID_MAPSIZE), szBuffer);
	      InvalidateRect(GetDlgItem(hDlg,ID_MAPSIZE), NULL, TRUE);

	      /* Update required printer sheets */
	      if (wWidth)
	      	wsprintf( szBuffer, "%d.%d x %d.%d",
			wWidth/10, wWidth%10, wHeight/10, wHeight%10);
              else strcpy( szBuffer, "- x -");
	      SetWindowText(GetDlgItem(hDlg, ID_PAPERSIZE), szBuffer );
	      InvalidateRect(GetDlgItem(hDlg, ID_PAPERSIZE), NULL, TRUE);

	      /* Per sheet 1" top & bottom margins + .5" left & right margins */
	      if ( (wWidth>10) && (wHeight>20))  
		 wsprintf( szBuffer, "%d x %d",
		   LOWORD(1+((LONG)PLANET_AREA_X*pDisOp->iZoom)/((LONG)xSheet*56-1440)),
		   LOWORD(1+((LONG)PLANET_AREA_Y*pDisOp->iZoom)/((LONG)ySheet*56-2880)));
              else strcpy( szBuffer, "- x -");
	      SetWindowText(GetDlgItem(hDlg, ID_PAGES), szBuffer);
	      InvalidateRect(GetDlgItem(hDlg, ID_PAGES), NULL, TRUE);

	      return TRUE;
	       
          case IDCANCEL:
          case IDOK:
	      EndDialog( hDlg, FALSE );
	      return TRUE;
        };
  };
  return FALSE;
};
 

/*------------------------------------------------------------------------*/
/*    GotoCoord                                                           */
/*                                                                        */
/*    Place the planet coordinates (wCoordX, wCoordY) in the center of the*/
/*    the screen. Move the mouse the that location too. Update the bars.  */
/*------------------------------------------------------------------------*/

void GotoCoord( HWND hwnd, WORD wCoordX, WORD wCoordY,
		DISPLAY_OPTIONS* pDisOp )
/* Moves orgin in pDisOp such that wCoordX and wCoordY in center of screen */
{  RECT  rect;
   POINT point;
   HDC   hdc   = GetDC(hwnd);
   LONG  i;

   GetClientRect( hwnd, &rect );
     
   SetMapMode(hdc, MM_TWIPS);
   DPtoLP( hdc, (LPPOINT) &rect, 2 );                 // Move map
   i = (LONG)wCoordX - rect.right/(2*pDisOp->iZoom);
   pDisOp->xOffset = max(wGalaxyXFrom, min(i, wGalaxyXTo));
   i = (LONG)wCoordY - rect.top  /(2*pDisOp->iZoom);
   pDisOp->yOffset = max(wGalaxyYFrom, min(i, wGalaxyYTo));
   ReleaseDC( hwnd, hdc );

   // Calculate the new mouse position
   point.x = pDisOp->iZoom*(wCoordX-pDisOp->xOffset); // To twips
   point.y = pDisOp->iZoom*(wCoordY-pDisOp->yOffset);
   LPtoDP( hdc, &point, 1 );
   ClientToScreen( hwnd, &point);
   SetCursorPos( point.x, point.y);

   SetScrollPos  (hwnd, SB_VERT, wGalaxyYTo-pDisOp->yOffset+wGalaxyYFrom, TRUE);
   SetScrollPos  (hwnd, SB_HORZ, pDisOp->xOffset, TRUE);

   MessageBeep(0);
   InvalidateRect(hwnd, NULL, TRUE);
};

/*------------------------------------------------------------------------*/
/*    Find dialog proc                                                    */
/*------------------------------------------------------------------------*/

BOOL FAR PASCAL FindDlgProc( HWND hDlg, WORD message,WORD wParam, LONG lParam)
{
  static DISPLAY_OPTIONS* pDisOp;
  static HWND hwndList, hwndCoords;  // Child windows
  static PLANET SelPlanet;           // Selected planet
  PLANET FAR* planet;             
  WORD   n, i;
  BOOL   bTranslated;
  BOOL   bFound;
  char   szSelName[40]; // Name of selected planet
  char   szBuffer[40]; 

  switch(message)
  {
    case WM_INITDIALOG:

	pDisOp     = (DISPLAY_OPTIONS*) lParam;
	hwndList   = GetDlgItem(hDlg, ID_LISTBOX);
        hwndCoords = GetDlgItem(hDlg, ID_COORDS);

	/* Fill the list box */
        bFound = FALSE;
	planet = (PLANET FAR*) GlobalLock( hPlanets );
	for (i=0; i<NUM_PLANETS; i++)
	{  if (!planet->bUnplaced)
	   {   SendMessage( hwndList, LB_ADDSTRING, 0,
			  (LONG)(LPSTR) planet->szPlanetName);
               bFound = TRUE;
           };
	   planet++;
        };
	GlobalUnlock( hPlanets );
	if (!bFound)
	{  MessageBeep(0);
	   MessageBox(hDlg, "There are no planets", szAppName, MB_OK | MB_ICONEXCLAMATION );
	   EndDialog( hDlg, FALSE );
           return FALSE;
        };

	/* Select a default planet */
        SetFocus( GetDlgItem( hDlg, ID_LISTBOX ) );
	SendMessage(hwndList, LB_SETCURSEL, 0, 0L);
	SendMessage(hDlg, WM_COMMAND, ID_LISTBOX, ((DWORD) LBN_SELCHANGE)<<16);

	return FALSE;

    case WM_COMMAND:
	if ( GetFocus() != GetDlgItem(hDlg, wParam) )
	   break;      // Prevent recursion via SetDlgItemText ect.

	switch(wParam)
	{
	  case ID_LISTBOX:
	      if ( HIWORD(lParam) == LBN_SELCHANGE )
              {
	         n = (WORD) SendMessage( hwndList, LB_GETCURSEL, 0, 0L);
		 if ( (WORD) SendMessage( hwndList, LB_GETTEXT, n, (LONG) (LPSTR) szSelName) )
                 {
                    /* Find planet coordinates */
		    planet = (PLANET FAR*) GlobalLock( hPlanets );
		    while (0!=_fstrcmp(planet->szPlanetName,szSelName)) planet++;
                    SelPlanet = *planet;
		    GlobalUnlock( hPlanets );

		    /* Update planet name & coordinates */
		    wsprintf( szBuffer, "(%d,%d)", SelPlanet.xCoord, SelPlanet.yCoord);
		    SetWindowText ( hwndCoords, szBuffer   );
		    SetDlgItemInt ( hDlg, IDC_PLANETNR, SelPlanet.ID, FALSE );
                    SetDlgItemText( hDlg, ID_INPUT, szSelName );
		    InvalidateRect( hwndCoords, NULL, TRUE );
		 };
	      };
	      if ( HIWORD(lParam) == LBN_DBLCLK )
                 SendMessage(hDlg, WM_COMMAND, IDOK, 0L );
	      return TRUE;

	  case IDC_PLANETNR:
	      if ( HIWORD(lParam)==EN_UPDATE )
	      {  n = GetDlgItemInt( hDlg, IDC_PLANETNR, &bTranslated, FALSE );
		 if ( bTranslated && (n>=1) && (n<=500))
		 {
		    /* Find planet */
		    planet = (PLANET FAR*) GlobalLock( hPlanets );
		    while ( --n ) planet++;
		    SelPlanet = *planet;

		    /* Update planet name & coordinates */
		    wsprintf( szBuffer, "(%d,%d)", planet->xCoord, planet->yCoord);
		    SetWindowText ( hwndCoords, szBuffer );
                    SetDlgItemText( hDlg, ID_INPUT, planet->szPlanetName );
		    InvalidateRect( hwndCoords, NULL, TRUE );
                    SendMessage( hwndList, LB_SELECTSTRING, -1, (LONG) planet->szPlanetName);
		 };
              };
	      break;

	  case ID_INPUT:
	      if ( HIWORD(lParam)==EN_UPDATE )
	      {  GetDlgItemText( hDlg, ID_INPUT, szBuffer, 80);                 
		 n = SendMessage( hwndList, LB_SELECTSTRING, -1, (LONG) (LPSTR) szBuffer);
                 if ( (n>=1) && (n<=500) )
                 {
                      /* Find planet coordinates */
		      planet = (PLANET FAR*) GlobalLock( hPlanets );
		      while ( planet->ID != n ) planet++;
                      SelPlanet = *planet;
		      GlobalUnlock( hPlanets );

		      /* Update planet name & coordinates */
		      wsprintf( szBuffer, "(%d,%d)", SelPlanet.xCoord, SelPlanet.yCoord);
		      SetWindowText ( hwndCoords, szBuffer   );
		      SetDlgItemInt ( hDlg, IDC_PLANETNR, n, FALSE );
		      InvalidateRect( hwndCoords, NULL, TRUE );
		 };
              };
              break;

	  case IDOK:      // Find button pressed
	      BarInfo.bOK    = TRUE;               // Update info bar
	      BarInfo.planet = SelPlanet;
              SendMessage( hDlgInfo, WM_COMMAND, IDM_UPDATE, 0L);
	      GotoCoord( GetParent(hDlg),  SelPlanet.xCoord,
	                 SelPlanet.yCoord, pDisOp );

	      // Fall through

	  case IDCANCEL:
	      EndDialog( hDlg, FALSE );
	      return TRUE;
        };
  };
  return FALSE;
};
 
/*------------------------------------------------------------------------*/
/*    Race dialog proc                                                    */
/*------------------------------------------------------------------------*/

BOOL FAR PASCAL RaceDlgProc( HWND hDlg, WORD message,WORD wParam, LONG lParam)
{ int i, j;
  switch(message)
  {
    case WM_INITDIALOG:
	for (j=0; j<3; j++)
	   for (i=0; i<NUM_RACES; i++)
	   {  SendDlgItemMessage( hDlg, IDM_RACEBASE+i+NUM_RACES*j,
				  EM_LIMITTEXT, RACE_LENGTH(j), 0L );
	      SetDlgItemText( hDlg, IDM_RACEBASE+i+NUM_RACES*j, szRaceNames[i][j]);
	   };
	return FALSE;

    case WM_COMMAND:
	switch(wParam)
	{
	  case IDOK:
	       for (j=0; j<3; j++)
	         for (i=0; i<NUM_RACES; i++)
		     GetDlgItemText( hDlg, IDM_RACEBASE+i+NUM_RACES*j,
				     szRaceNames[i][j],RACE_LENGTH(j));

	      // Fall through     

	  case IDCANCEL:
	      EndDialog( hDlg, FALSE );
	      return TRUE;
        };
  };
  return FALSE;
};


/*------------------------------------------------------------------------*/
/*    Spiral dialog proc                                                  */
/*------------------------------------------------------------------------*/

BOOL FAR PASCAL SpiralDlgProc( HWND hDlg, WORD message,WORD wParam, LONG lParam)
{ int  i,iAngle, iEnd, iStart, iRand, iRots,
       iXCent, iYCent, iAsY, iAsX, sgn, iRotAngle;
  float t0, t1, t, f, step, p, q, x, y, x1, y1;
  BOOL  bTranslated;
  PLANET FAR* planet;

  switch(message)
  {
    case WM_INITDIALOG:
        // Input box are already initialized
	return FALSE;

    case WM_COMMAND:
	switch(wParam)
	{
	  case IDOK:
	      iAngle = GetDlgItemInt( hDlg, IDC_ANGLE, &bTranslated, FALSE);
	      if ((iAngle>360) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid angle", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iRotAngle = GetDlgItemInt( hDlg, IDC_ROTANGLE, &bTranslated, FALSE);
	      if ((iRotAngle>360) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid rotation angle", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iRots  = GetDlgItemInt( hDlg, IDC_ROTATIONS,  &bTranslated, FALSE);
	      if ((iRots>100) || (iRots<1) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid number of rotations", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iXCent = GetDlgItemInt( hDlg, IDC_XCENTER, &bTranslated, FALSE);
	      if ((iXCent>wGalaxyXTo) || (iXCent<wGalaxyXFrom)|| (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid x-coordinate", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
	      };
	      iYCent = GetDlgItemInt( hDlg, IDC_YCENTER, &bTranslated, FALSE);
	      if ((iYCent>wGalaxyYTo) || (iYCent<wGalaxyYFrom)|| (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid y-coordinate", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
	      };
	      iStart = GetDlgItemInt( hDlg, IDC_START_RADIUS, &bTranslated, FALSE);
	      if ((iStart<10) || (iStart>3000) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid starting radius", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iEnd   = GetDlgItemInt( hDlg, IDC_END_RADIUS,   &bTranslated, FALSE);
	      if ((iEnd<10) || (iEnd>3000) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid ending radius", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iAsX   = GetDlgItemInt( hDlg, IDC_ASX,   &bTranslated, FALSE);
	      if ((iAsX<1) || (iAsX>100) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid dx", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iAsY   = GetDlgItemInt( hDlg, IDC_ASY,   &bTranslated, FALSE);
	      if ((iAsY<1) || (iAsY>100) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid dy", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iRand  = GetDlgItemInt( hDlg, IDC_RANDOMFACTOR, &bTranslated, FALSE);
	      if ((iRand>1000) || (iRand<1) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid random factor", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
	      };
	      if (iEnd>iStart)
	      {  i = iStart; iStart = iEnd; iEnd=i; sgn = -1; }
              else sgn = 1;

              t0   = ((float)iAngle/180)*M_PI;
	      t1   = t0 + iRots*2*M_PI;
	      f    = -log((float)iStart/iEnd)/(iRots*2*M_PI);
	      p    = (float)iAsY/iAsX;
              q    = ((float)iRotAngle/180)*M_PI;
	      step = (t1-t0)/NUM_PLANETS;

	      t    = t0;
	      randomize();
	      planet = (PLANET FAR*) GlobalLock(hPlanets);
	      for (i=0; i<NUM_PLANETS; i++, t+=step, planet++)
              {
		  x  = iStart*exp(f*(t-t0))*sgn*sin(t);  // The spiral
		  y  = p*iStart*exp(f*(t-t0))*cos(t);
		  x1 = x*cos(q)+y*sin(q);                // rotate
                  y1 = y*cos(q)-x*sin(q);
		  x  = x1 + (float)iXCent + (float)(rand()%(2*iRand))-(float)iRand;
		  y  = y1 + (float)iYCent + (float)(rand()%(2*iRand))-(float)iRand;
                  // Update delta checksum
                  BarInfo.dx += (LONG)((x>GALAXY_BORDER)?x:GALAXY_BORDER) - planet->xCoord;
		  planet->xCoord = (x>GALAXY_BORDER) ? x:GALAXY_BORDER;
		  planet->yCoord = (y>GALAXY_BORDER) ? y:GALAXY_BORDER;
	      };
              GlobalUnlock(hPlanets);

	      // Fall through     

	  case IDCANCEL:
	      EndDialog( hDlg, FALSE );
              InvalidateRect(hDlgInfo, NULL, TRUE);
	      InvalidateRect(GetParent(hDlg), NULL, TRUE);
	      return TRUE;
        };
  };
  return FALSE;
};


/*------------------------------------------------------------------------*/
/*    Ellipse dialog proc                                                 */
/*------------------------------------------------------------------------*/

BOOL FAR PASCAL EllipseDlgProc( HWND hDlg, WORD message,WORD wParam, LONG lParam)
{ int  i, iEnd, iStart, iRand, iEllipses,
       iXCent, iYCent, iAsY, iAsX, iRotAngle;
  float r,t,step, p, q, x, y, x1, y1;
  BOOL  bTranslated;
  PLANET FAR* planet;

  switch(message)
  {
    case WM_INITDIALOG:
        // Input box are already initialized
	return FALSE;

    case WM_COMMAND:
	switch(wParam)
	{
	  case IDOK:
	      iRotAngle = GetDlgItemInt( hDlg, IDC_ROTANGLE, &bTranslated, FALSE);
	      if ((iRotAngle>360) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid rotation angle", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iEllipses  = GetDlgItemInt( hDlg, IDC_ROTATIONS,  &bTranslated, FALSE);
	      if ((iEllipses>100) || (iEllipses<1) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid number of ellipses", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iXCent = GetDlgItemInt( hDlg, IDC_XCENTER, &bTranslated, FALSE);
	      if ((iXCent>wGalaxyXTo) || (iXCent<wGalaxyXFrom)|| (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid x-coordinate", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
	      };
	      iYCent = GetDlgItemInt( hDlg, IDC_YCENTER, &bTranslated, FALSE);
	      if ((iYCent>wGalaxyYTo) || (iYCent<wGalaxyYFrom)|| (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid y-coordinate", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
	      };
	      iStart = GetDlgItemInt( hDlg, IDC_START_RADIUS, &bTranslated, FALSE);
	      if ((iStart<10) || (iStart>3000) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid starting radius", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iEnd   = GetDlgItemInt( hDlg, IDC_END_RADIUS,   &bTranslated, FALSE);
	      if ((iEnd<10) || (iEnd>3000) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid ending radius", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iAsX   = GetDlgItemInt( hDlg, IDC_ASX,   &bTranslated, FALSE);
	      if ((iAsX<1) || (iAsX>100) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid dx", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iAsY   = GetDlgItemInt( hDlg, IDC_ASY,   &bTranslated, FALSE);
	      if ((iAsY<1) || (iAsY>100) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid dy", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iRand  = GetDlgItemInt( hDlg, IDC_RANDOMFACTOR, &bTranslated, FALSE);
	      if ((iRand>1000) || (iRand<1) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid random factor", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
	      };
	      if (iEnd>iStart)
	      {  i = iStart; iStart = iEnd; iEnd=i; }

	      p    = (float)iAsY/iAsX;
	      q    = ((float)iRotAngle/180)*M_PI;
	      step = (float)iEllipses*2*M_PI/NUM_PLANETS;
              r    = iStart;
	      t    = 2*M_PI;

	      randomize();
	      planet = (PLANET FAR*) GlobalLock(hPlanets);
	      for (i=0; i<NUM_PLANETS; i++, t-=step, planet++)
	      {   if (t<=0)
		  {  t  = 2*M_PI;    // Next (smaller) ellipse
                     r -= (float)(iStart-iEnd)/iEllipses;
                  };
		  x  =   r*sin(t);     // The spiral
		  y  = p*r*cos(t);
		  x1 = x*cos(q)+y*sin(q);   // rotate
                  y1 = y*cos(q)-x*sin(q);
		  x  = x1 + (float)iXCent + (float)(rand()%(2*iRand))-(float)iRand;
		  y  = y1 + (float)iYCent + (float)(rand()%(2*iRand))-(float)iRand;
                  // Update delta checksum
                  BarInfo.dx += (LONG)((x>GALAXY_BORDER)?x:GALAXY_BORDER) - planet->xCoord;
		  planet->xCoord = (x>GALAXY_BORDER) ? x:GALAXY_BORDER;
		  planet->yCoord = (y>GALAXY_BORDER) ? y:GALAXY_BORDER;
	      };
              GlobalUnlock(hPlanets);

	      // Fall through     

	  case IDCANCEL:
	      EndDialog( hDlg, FALSE );
              InvalidateRect(hDlgInfo, NULL, TRUE);
	      InvalidateRect(GetParent(hDlg), NULL, TRUE);
	      return TRUE;
        };
  };
  return FALSE;
};


/*------------------------------------------------------------------------*/
/*    Rectangular dialog proc                                             */
/*------------------------------------------------------------------------*/

BOOL FAR PASCAL RectDlgProc( HWND hDlg, WORD message,WORD wParam, LONG lParam)
{ int  i, iHeight, iWidth, iXCent, iYCent;
  long x, y;
  BOOL  bTranslated;
  PLANET FAR* planet;

  switch(message)
  {
    case WM_INITDIALOG:
        // Input box are already initialized
	return FALSE;

    case WM_COMMAND:
	switch(wParam)
	{
	  case IDOK:
	      iXCent = GetDlgItemInt( hDlg, IDC_XCENTER, &bTranslated, FALSE);
	      if ((iXCent>wGalaxyXTo) || (iXCent<wGalaxyXFrom)|| (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid x-coordinate", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
	      };
	      iYCent = GetDlgItemInt( hDlg, IDC_YCENTER, &bTranslated, FALSE);
	      if ((iYCent>wGalaxyYTo) || (iYCent<wGalaxyYFrom)|| (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid y-coordinate", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
	      };
	      iWidth = GetDlgItemInt( hDlg, IDC_WIDTH, &bTranslated, FALSE);
	      if ((iWidth<100) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid width\nThe width must be >=100", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iHeight   = GetDlgItemInt( hDlg, IDC_HEIGHT,   &bTranslated, FALSE);
	      if ((iHeight<100) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid height\nThe height must be >=100", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };

	      randomize();
	      planet = (PLANET FAR*) GlobalLock(hPlanets);
	      for (i=0; i<NUM_PLANETS; i++, planet++)
	      {   
		  x  = (long) iXCent + rand()%iWidth-iWidth/2;
		  y  = (long) iYCent + rand()%iHeight-iHeight/2;
                  // Update delta checksum
                  BarInfo.dx += (LONG)((x>GALAXY_BORDER)?x:GALAXY_BORDER) - planet->xCoord;
		  planet->xCoord = (x>GALAXY_BORDER) ? x:GALAXY_BORDER;
		  planet->yCoord = (y>GALAXY_BORDER) ? y:GALAXY_BORDER;
	      };
              GlobalUnlock(hPlanets);

	      // Fall through     

	  case IDCANCEL:
	      EndDialog( hDlg, FALSE );
              InvalidateRect(hDlgInfo, NULL, TRUE);
	      InvalidateRect(GetParent(hDlg), NULL, TRUE);
	      return TRUE;
        };
  };
  return FALSE;
};


/*------------------------------------------------------------------------*/
/*    Bar-sprial  dialog proc                                             */
/*------------------------------------------------------------------------*/

BOOL FAR PASCAL BarDlgProc( HWND hDlg, WORD message,WORD wParam, LONG lParam)
{ int  i, iHeight, iWidth, iXCent, iYCent;
  long x, y;
  BOOL  bTranslated;
  PLANET FAR* planet;

  switch(message)
  {
    case WM_INITDIALOG:
        // Input box are already initialized
	return FALSE;

    case WM_COMMAND:
	switch(wParam)
	{
	  case IDOK:
	      iXCent = GetDlgItemInt( hDlg, IDC_XCENTER, &bTranslated, FALSE);
	      if ((iXCent>wGalaxyXTo) || (iXCent<wGalaxyXFrom)|| (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid x-coordinate", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
	      };
	      iYCent = GetDlgItemInt( hDlg, IDC_YCENTER, &bTranslated, FALSE);
	      if ((iYCent>wGalaxyYTo) || (iYCent<wGalaxyYFrom)|| (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid y-coordinate", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
	      };
	      iWidth = GetDlgItemInt( hDlg, IDC_WIDTH, &bTranslated, FALSE);
	      if ((iWidth<100) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid width\nThe width must be >=100", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };
	      iHeight   = GetDlgItemInt( hDlg, IDC_HEIGHT,   &bTranslated, FALSE);
	      if ((iHeight<100) || (!bTranslated) )
	      {  MessageBox( hDlg, "Invalid height\nThe height must be >=100", szAppName, MB_OK | MB_ICONEXCLAMATION);
		 break;
              };

	      randomize();
	      planet = (PLANET FAR*) GlobalLock(hPlanets);
	      for (i=0; i<NUM_PLANETS; i++, planet++)
	      {   
		  x  = (long) iXCent + (((long)rand()<<16)+rand())%iWidth-iWidth/2;
		  y  = (long) iYCent + (((long)rand()<<16)+rand())%iHeight-iHeight/2;
                  // Update delta checksum
                  BarInfo.dx += (LONG)((x>GALAXY_BORDER)?x:GALAXY_BORDER) - planet->xCoord;
		  planet->xCoord = (x>GALAXY_BORDER) ? x:GALAXY_BORDER;
		  planet->yCoord = (y>GALAXY_BORDER) ? y:GALAXY_BORDER;
	      };
              GlobalUnlock(hPlanets);

	      // Fall through     

	  case IDCANCEL:
	      EndDialog( hDlg, FALSE );
              InvalidateRect(hDlgInfo, NULL, TRUE);
	      InvalidateRect(GetParent(hDlg), NULL, TRUE);
	      return TRUE;
        };
  };
  return FALSE;
};

/*------------------------------------------------------------------------*/
/*    Galaxy range dialog proc                                            */
/*------------------------------------------------------------------------*/

BOOL FAR PASCAL GalaxyRangeProc( HWND hDlg, WORD message,WORD wParam, LONG lParam)
{ BOOL bTrans1, bTrans2, bTrans3, bTrans4;
  WORD x0, y0, x1, y1;
  char szBuffer[80];

  switch(message)
  {
    case WM_INITDIALOG:
	SetDlgItemInt( hDlg, IDC_XFROM, wGalaxyXFrom, FALSE);
	SetDlgItemInt( hDlg, IDC_YFROM, wGalaxyYFrom, FALSE);
	SetDlgItemInt( hDlg, IDC_XTO,   wGalaxyXTo,   FALSE);
        SetDlgItemInt( hDlg, IDC_YTO,   wGalaxyYTo,   FALSE);
	return FALSE;

    case WM_COMMAND:
	switch(wParam)
	{
	  case IDOK:
	      x0 = GetDlgItemInt( hDlg, IDC_XFROM, &bTrans1, FALSE);
	      y0 = GetDlgItemInt( hDlg, IDC_YFROM, &bTrans2, FALSE);
	      x1 = GetDlgItemInt( hDlg, IDC_XTO,   &bTrans3, FALSE);
	      y1 = GetDlgItemInt( hDlg, IDC_YTO,   &bTrans4, FALSE);
	      if ( (x0+MIN_GALAXYSIZE < x1) && (y0+MIN_GALAXYSIZE < y1) &&
		   (x0 >= GALAXY_BORDER)      && (y0 >= GALAXY_BORDER) &&
		   (x1 <= MAX_GALAXY-GALAXY_BORDER) &&
		   (y1 <= MAX_GALAXY-GALAXY_BORDER) &&
		    bTrans1 && bTrans2 && bTrans3 && bTrans4 )
	      {  wGalaxyXFrom = x0;
		 wGalaxyYFrom = y0;
		 wGalaxyXTo   = x1;
		 wGalaxyYTo   = y1;
		 EndDialog( hDlg, FALSE );
                 return TRUE;
	      }
	      else
	      {  MessageBeep(0);
		 wsprintf( szBuffer, "Invalid range\nMax. X-range %d-%d\nMax. Y-range %d-%d",
			   GALAXY_BORDER, MAX_GALAXY-GALAXY_BORDER,
                           GALAXY_BORDER, MAX_GALAXY-GALAXY_BORDER);
		 MessageBox(hDlg, szBuffer, szAppName, MB_OK | MB_ICONEXCLAMATION);
                 return TRUE;
	      }; 

	  case IDCANCEL:
	      EndDialog( hDlg, FALSE );
	      return TRUE;
        };
  };
  return FALSE;
};


/*------------------------------------------------------------------------*/
/*    View ships dialog proc                                              */
/*------------------------------------------------------------------------*/

void DisplayShip( HWND hDlg, int iNr )
/* Fill dialog box with ship iNr data */
{  char szBuffer[21];
   SHIPDATA FAR* ship;
   int iIndex;

   ship = (SHIPDATA FAR*) GlobalLock(hShips);
   ship+=iNr-1;

   _fmemcpy( szBuffer, ship->sCode, 3);    // Friendly code
   szBuffer[3] = '\0';
   SetDlgItemText( hDlg, IDC_CODE,      szBuffer );

   _fmemcpy( szBuffer, ship->sName, 20 );
   szBuffer[20] = '\0';
   iIndex = SendDlgItemMessage( hDlg, IDC_NAME, CB_FINDSTRING, 0, (LONG) szBuffer);
   SendDlgItemMessage( hDlg, IDC_NAME,  CB_SETCURSEL, iIndex , 0L);

   SendDlgItemMessage( hDlg, IDC_OWNER,       CB_SETCURSEL, ship->wRace-1, 0L );
   SendDlgItemMessage( hDlg, IDC_ENGINELEVEL, CB_SETCURSEL, ship->wDriveLevel-1, 0L );
   SendDlgItemMessage( hDlg, IDC_BEAMLEVEL,   CB_SETCURSEL, ship->wBeamLevel, 0L );
   SendDlgItemMessage( hDlg, IDC_TORPLEVEL,   CB_SETCURSEL, ship->wTorpLevel, 0L );
   SetDlgItemInt ( hDlg, IDC_ID,        ship->wID,          FALSE);
   SetDlgItemInt ( hDlg, IDC_X,         ship->xCoord,       FALSE);
   SetDlgItemInt ( hDlg, IDC_Y,         ship->yCoord,       FALSE);
   SetDlgItemInt ( hDlg, IDC_FUEL,      ship->wFuel,        FALSE);
   SetDlgItemInt ( hDlg, IDC_CREW,      ship->wCrew,        FALSE);
   SetDlgItemInt ( hDlg, IDC_COLONISTS, ship->wColonists,   FALSE);
   SetDlgItemInt ( hDlg, IDC_SUPPLIES,  ship->wSupplies,    FALSE);
   SetDlgItemInt ( hDlg, IDC_TRI,       ship->wTri,         FALSE);
   SetDlgItemInt ( hDlg, IDC_DUR,       ship->wDur,         FALSE);
   SetDlgItemInt ( hDlg, IDC_MOL,       ship->wMol,         FALSE);
   SetDlgItemInt ( hDlg, IDC_MC,        ship->wMoney,       FALSE);
   SetDlgItemInt ( hDlg, IDC_HULL,      ship->wShipType,    FALSE);
   SetDlgItemInt ( hDlg, IDC_BEAMWEAPONS,   ship->wBeamWeapons, FALSE);
   SetDlgItemInt ( hDlg, IDC_FIGHTERBAYS,   ship->wFighterBays, FALSE);
   SetDlgItemInt ( hDlg, IDC_TORPLAUNCHERS, ship->wTorpsBays,   FALSE);
   SetDlgItemInt ( hDlg, IDC_TORPS,         ship->wTorps,       FALSE);

   GlobalUnlock(hShips);
};

/* Copy data from dialog box into high memory */
/* Return TRUE if succesfull                  */
int GetShip( HWND hDlg, int iNr )
{  BOOL bTrans;
   int  i;
   char szBuffer[25];
   DWORD dwRace;
   SHIPDATA FAR* ship;
   SHIPDATA LocalShip;

   /* Get standard ship data */
   ship = (SHIPDATA FAR*) GlobalLock(hShips);
   ship+=iNr-1;
   LocalShip = *ship;
   GlobalUnlock(hShips);

   GetDlgItemText( hDlg, IDC_NAME, szBuffer, 19 );
   for (i=_fstrlen(szBuffer); i<20; i++)
     szBuffer[i]=' ';
   szBuffer[20]=0;
   memcpy(LocalShip.sName, szBuffer, 20 );
   SendDlgItemMessage( hDlg, IDC_NAME, CB_DELETESTRING, iNr-1, 0L);
   SendDlgItemMessage( hDlg, IDC_NAME, CB_INSERTSTRING, iNr-1, (LONG) szBuffer );
   
   GetDlgItemText( hDlg, IDC_CODE, szBuffer, 4 );
   strcat(szBuffer, "   "); // Be sure there are enough characters
   memcpy( LocalShip.sCode, szBuffer, 3 );

   dwRace = SendDlgItemMessage( hDlg, IDC_OWNER, CB_GETCURSEL, 0, 0L);
   if ( dwRace == CB_ERR ) return FALSE;
   LocalShip.wRace = dwRace+1;

   LocalShip.xCoord = GetDlgItemInt( hDlg, IDC_X, &bTrans, FALSE );
   if (!bTrans) return FALSE;
   LocalShip.yCoord = GetDlgItemInt( hDlg, IDC_Y, &bTrans, FALSE );
   if (!bTrans) return FALSE;
   LocalShip.wTri = GetDlgItemInt( hDlg, IDC_TRI, &bTrans, FALSE );
   if (!bTrans) return FALSE;
   LocalShip.wDur = GetDlgItemInt( hDlg, IDC_DUR, &bTrans, FALSE );
   if (!bTrans) return FALSE;
   LocalShip.wMol = GetDlgItemInt( hDlg, IDC_MOL, &bTrans, FALSE );
   if (!bTrans) return FALSE;
   LocalShip.wSupplies = GetDlgItemInt( hDlg, IDC_SUPPLIES, &bTrans, FALSE );
   if (!bTrans) return FALSE;
   LocalShip.wMoney = GetDlgItemInt( hDlg, IDC_MC, &bTrans, FALSE );
   if (!bTrans) return FALSE;
   LocalShip.wColonists = GetDlgItemInt( hDlg, IDC_COLONISTS, &bTrans, FALSE );
   if (!bTrans) return FALSE;
   LocalShip.wFuel = GetDlgItemInt( hDlg, IDC_FUEL, &bTrans, FALSE );
   if (!bTrans) return FALSE;
   LocalShip.wDriveLevel= SendDlgItemMessage( hDlg, IDC_ENGINELEVEL, CB_GETCURSEL, 0, 0L)+1;
   if (!bTrans) return FALSE;
   LocalShip.wShipType = GetDlgItemInt( hDlg, IDC_HULL, &bTrans, FALSE );
   if (!bTrans) return FALSE;
   LocalShip.wBeamLevel = SendDlgItemMessage( hDlg, IDC_BEAMLEVEL, CB_GETCURSEL, 0, 0L);
   if (!bTrans) return FALSE;
   LocalShip.wBeamWeapons = GetDlgItemInt( hDlg, IDC_BEAMWEAPONS, &bTrans, FALSE );
   if (!bTrans) return FALSE;
   LocalShip.wFighterBays = GetDlgItemInt( hDlg, IDC_FIGHTERBAYS, &bTrans, FALSE );
   if (!bTrans) return FALSE;
   LocalShip.wTorpLevel = SendDlgItemMessage( hDlg, IDC_TORPLEVEL, CB_GETCURSEL, 0, 0L);
   if (!bTrans) return FALSE;
   LocalShip.wTorpsBays = GetDlgItemInt( hDlg, IDC_TORPLAUNCHERS, &bTrans, FALSE );
   if (!bTrans) return FALSE;
   LocalShip.wTorps = GetDlgItemInt( hDlg, IDC_TORPS, &bTrans, FALSE );
   if (!bTrans) return FALSE;


   /* Values are OK, copy ship data into high memory */
   ship = (SHIPDATA FAR*) GlobalLock(hShips);
   ship+=iNr-1;
   *ship = LocalShip;
   GlobalUnlock(hShips);
   return TRUE;
};

BOOL FAR PASCAL ViewShipDlgProc( HWND hDlg, WORD message,WORD wParam, LONG lParam)
{ static int iScrollPos = 1;
  static int iNewShipName = 1;
  BOOL bTrans;
  SHIPDATA FAR* ship;
  DWORD dwIndex;
  char szBuffer[25];
  int i;

  switch(message)
  {
    case WM_INITDIALOG:
	/* Initialize list boxes */
	for (i=0; i<NUM_RACES; i++)
	  SendDlgItemMessage( hDlg, IDC_OWNER, CB_ADDSTRING, 0,
			  (LONG)(LPSTR) szRaceNames[i][0]);

	ship = (SHIPDATA FAR*) GlobalLock( hShips );
	szBuffer[20]=0;
	for (i=0; i<iShipCount; i++, ship++)
	{  _fmemcpy( szBuffer, ship->sName, 20 );
	   SendDlgItemMessage(hDlg, IDC_NAME, CB_ADDSTRING, 0,
	                  (LONG) (LPSTR) szBuffer);
        };
	GlobalUnlock(hShips);

	for (i=0; i<9; i++)
	   SendDlgItemMessage(hDlg, IDC_ENGINELEVEL, CB_ADDSTRING, 0,
			      (LONG) (LPSTR) szDrive[i]);
	for (i=0; i<11; i++)
        {  SendDlgItemMessage(hDlg, IDC_BEAMLEVEL, CB_ADDSTRING, 0,
			      (LONG) (LPSTR) szBeam[i]);
	   SendDlgItemMessage(hDlg, IDC_TORPLEVEL, CB_ADDSTRING, 0,
			      (LONG) (LPSTR) szTorp[i]);
        };

	DisplayShip( hDlg, iScrollPos );

	/* Initialize the scroll bar */
	SetScrollRange( GetDlgItem(hDlg, IDC_SCROLL), SB_CTL, 1, iShipCount, FALSE);
	SetScrollPos  ( GetDlgItem(hDlg, IDC_SCROLL), SB_CTL, iScrollPos, TRUE);

	return FALSE;

    case WM_HSCROLL:
	 if ( (wParam==SB_LINEUP) || (wParam==SB_LINEDOWN) ||
	      (wParam==SB_PAGEUP) || (wParam==SB_PAGEDOWN) ||
	      (wParam==SB_THUMBPOSITION))
  	    if ( !GetShip(hDlg, iScrollPos) )
	    {   MessageBeep(0);
	        MessageBox( hDlg, "Invalid ship data!\nCannot scroll", szAppName, MB_OK | MB_ICONEXCLAMATION );
	        return TRUE;
	    };
	 switch(wParam)
	     {  case SB_LINEUP:        iScrollPos--;  break;
	        case SB_LINEDOWN:      iScrollPos++;  break;
	        case SB_PAGEUP:        iScrollPos--;  break;
	        case SB_PAGEDOWN:      iScrollPos++;  break;
		case SB_THUMBPOSITION: iScrollPos = LOWORD(lParam); break;
	     };
	     iScrollPos=max(1,min(iScrollPos, iShipCount));
	     SetScrollPos(GetDlgItem(hDlg,IDC_SCROLL), SB_CTL, iScrollPos, TRUE);
	     DisplayShip( hDlg, iScrollPos );
	     return TRUE;

    case WM_COMMAND:
	switch(wParam)
	{
	  case IDC_NAME:
	      if (HIWORD(lParam)==LBN_SELCHANGE)
	      {
		 dwIndex = SendDlgItemMessage( hDlg, IDC_NAME, CB_GETCURSEL, 0, 0L );
		 if ( dwIndex != CB_ERR )
		 {  iScrollPos = dwIndex+1;
		    SetScrollPos(GetDlgItem(hDlg,IDC_SCROLL), SB_CTL, iScrollPos, TRUE);
		    DisplayShip( hDlg, iScrollPos );
                 };
              };
	      return TRUE;

	  case IDC_ADDSHIP:
	       wsprintf( szBuffer, "No name %d", iNewShipName++ );
               while (strlen(szBuffer)!=20) strcat(szBuffer, " ");
	       ship = (SHIPDATA FAR*) GlobalLock(hShips);
	       for (i=iShipCount; i>iScrollPos-1; i--)
		  *(ship+i) = *(ship+i-1);
	       _fmemcpy((ship+iScrollPos-1)->sName, szBuffer, 20);
	       (ship+iScrollPos-1)->wID = iNewShipID++; 
	       GlobalUnlock(hShips);
	       SendDlgItemMessage( hDlg, IDC_NAME, CB_INSERTSTRING, iScrollPos-1,
                                   (LONG) (LPSTR) szBuffer );
	       iShipCount++;
	       SetScrollRange( GetDlgItem(hDlg, IDC_SCROLL), SB_CTL, 1, iShipCount, FALSE);
	       DisplayShip( hDlg, iScrollPos );
	       return TRUE;

	  case IDC_REMOVESHIP:
	       if ( iShipCount<=1 )  // At least one ship
	       {  MessageBeep(0); return TRUE; };
	       ship = (SHIPDATA FAR*) GlobalLock(hShips);
	       for (i=iScrollPos-1; i<iShipCount; i++)
		  *(ship+i) = *(ship+i+1);
	       GlobalUnlock(hShips);
	       SendDlgItemMessage( hDlg, IDC_NAME, CB_DELETESTRING, iScrollPos-1, 0L);
	       iShipCount--;
               if ( iScrollPos>iShipCount) iScrollPos--;
	       SetScrollPos(GetDlgItem(hDlg,IDC_SCROLL), SB_CTL, iScrollPos, TRUE);
	       SetScrollRange( GetDlgItem(hDlg, IDC_SCROLL), SB_CTL, 1, iShipCount, FALSE);
	       DisplayShip( hDlg, iScrollPos );
	       return TRUE;

	  case IDOK:
	      if ( !GetShip(hDlg, iScrollPos) )
	      {   MessageBeep(0);
	          MessageBox( hDlg, "Invalid ship data", szAppName, MB_OK | MB_ICONEXCLAMATION );
		  return FALSE;
	      };
	      // Fall through

	  case IDCANCEL:
	      EndDialog( hDlg, FALSE );
	      return TRUE;
        };
  };
  return FALSE;
};

/*------------------------------------------------------------------------*/
/*    Font selection                                                      */
/*------------------------------------------------------------------------*/

void CMUFont( HWND hWnd )
{
	static CHOOSEFONT cfTemp;
	static LOGFONT lfTemp = {-13,0,0,0,500,0,0,0,0,
				 OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS,
				 DEFAULT_QUALITY, DEFAULT_PITCH, "Courier"};
	HDC hdcPrn;

	if ( hfFont )
        {       hdcPrn = GetPrinterDC2();
		cfTemp.lStructSize = sizeof( CHOOSEFONT );
		cfTemp.hwndOwner   = hWnd;
		cfTemp.hDC         = hdcPrn;
		cfTemp.lpLogFont   = &lfTemp;	
		cfTemp.Flags       = CF_EFFECTS  | CF_INITTOLOGFONTSTRUCT |
		  		     CF_FORCEFONTEXIST | CF_BOTH;
		cfTemp.rgbColors   = crColor; // Global var.
		cfTemp.lCustData   = 0L;
		cfTemp.lpfnHook    = NULL;
		cfTemp.lpTemplateName = NULL;
		cfTemp.hInstance   = 0;
		cfTemp.lpszStyle   = NULL;
		cfTemp.nFontType   = PRINTER_FONTTYPE | SCREEN_FONTTYPE;
		cfTemp.nSizeMin    = 0;
		cfTemp.nSizeMax    = 0;
		
		if( ChooseFont( &cfTemp ) == TRUE )
		{
		   if( hfFont ) DeleteObject( hfFont );
                   lfTemp.lfHeight *= 20;            // convert points to twips
		   hfFont = CreateFontIndirect( &lfTemp );
		   lfTemp.lfHeight /= 20;            // to points again
		   crColor = cfTemp.rgbColors; 
		}
		DeleteDC( hdcPrn );

	} else
	{    // First time: default font and no dialog box
	     lfTemp.lfHeight *= 20;            // convert points to twips
	     hfFont = CreateFontIndirect( &lfTemp );
             lfTemp.lfHeight /= 20;            // to points again 
	};
	InvalidateRect( hWnd, NULL, TRUE );
}

/*------------------------------------------------------------------------*/
/*    About dialog proc                                                   */
/*------------------------------------------------------------------------*/

BOOL FAR PASCAL About(HWND hDlg, WORD message, WORD wParam, LONG lParam)
{   switch (message) 
    {   case WM_INITDIALOG: return (TRUE);
        case WM_COMMAND: EndDialog(hDlg, TRUE); return (TRUE);
    }
    return (FALSE);
}

/*------------------------------------------------------------------------*/
/*    Automatically correct the delta-checksum                            */
/*------------------------------------------------------------------------*/

void CorrectChecksum(void)
{  int i;
   int d;
   PLANET FAR* planet;
   PLANET FAR* planets_base;
   char szBuffer[40];

   /* Now move the planets */
   planets_base = (PLANET FAR*) GlobalLock(hPlanets);
   while ( BarInfo.dx )
   {  planet = planets_base;           // Start with planet ID 1
      d = (BarInfo.dx/NUM_PLANETS);    // Calculate planet move
      if (!d) d = (BarInfo.dx<0)?-1:+1;
      for (i=0; i<NUM_PLANETS; i++)
      {
	 if (((long)planet->xCoord-d > GALAXY_BORDER) &&
	      (planet->xCoord-d < MAX_GALAXY-GALAXY_BORDER))
	 {  if ( BarInfo.planet.ID == planet->ID )
		BarInfo.planet.xCoord -= d;
	    planet->xCoord -= d;
	    BarInfo.dx     -= d;
	    if ( !BarInfo.dx )
	       break;  // Exit for-loop at once
         }
         planet++;
      };
   };
   GlobalUnlock(hPlanets);

   /* Update info-bar */
   SendMessage(hDlgInfo, WM_COMMAND, IDM_UPDATE, 0L);
};

/*------------------------------------------------------------------------*/
/*    Move to closest planet                                              */
/*------------------------------------------------------------------------*/

void MoveToPlanet( HWND hwnd, HDC hdc, LONG lParam, DISPLAY_OPTIONS *pDisOp )
{  RECT rect = { 0, 0, 0, 0 };
   PLANET FAR* planet;
   PLANET FAR* planet1;
   LONG        dis, disMin;
   int  i;

   // Transform move postion into planet units
   rect.right  = LOWORD(lParam);  // Move mouse position into rect
   rect.bottom = HIWORD(lParam);
   DPtoLP( hdc, (LPPOINT) &rect, 2 );    // mouse pos in twips
   rect.right  = rect.right/pDisOp->iZoom  + pDisOp->xOffset;
   rect.bottom = rect.bottom/pDisOp->iZoom + pDisOp->yOffset;

   if ((rect.right>=MAX_GALAXY) || (rect.bottom>=MAX_GALAXY))
   	return;

   // Find the closest planet
   planet1 = planet = (PLANET FAR*) GlobalLock(hPlanets);
   disMin  = (LONG) 2*MAX_GALAXY*MAX_GALAXY;
   for (i=0; i<NUM_PLANETS; i++)
   {   if ( !planet->bUnplaced )
       {  dis = ((LONG) rect.right  - planet->xCoord) *
	        ((LONG) rect.right  - planet->xCoord) +
	        ((LONG) rect.bottom - planet->yCoord) *
	        ((LONG) rect.bottom - planet->yCoord);
          if ( dis < disMin )
          {   disMin = dis;
              planet1 = planet;
	  };
       };
       planet++;
   };

   // Jump to closest planet
   GotoCoord( hwnd, planet1->xCoord, planet1->yCoord, pDisOp );

   // Update info bar
   BarInfo.bOK    = TRUE;
   BarInfo.planet = *planet1;
   SendMessage( hDlgInfo, WM_COMMAND, IDM_UPDATE, 0L);

   GlobalUnlock(hPlanets);
};

/*------------------------------------------------------------------------*/
/*    Message processing procedure for the info bar (modeless dlg box)    */
/*------------------------------------------------------------------------*/

int UpdatePlanet( PLANET NewPlanet )
/* Replaces the planet in BarInfo by NewPlanet    */
/* returns false iff NewPlanet is somehow invalid */
{
   PLANET FAR* planet;

   // Test if the new planet is valid
   if (( '\0' == *NewPlanet.szPlanetName )          ||
       (NewPlanet.xCoord < wGalaxyXFrom)            ||
       (NewPlanet.xCoord > wGalaxyXTo)              ||
       (NewPlanet.yCoord < wGalaxyYFrom)            ||
       (NewPlanet.yCoord > wGalaxyYTo  ))
     return FALSE;

   // Search for original planet name in global memory
   planet = (PLANET FAR*) GlobalLock(hPlanets);
   while (BarInfo.planet.ID != planet->ID)
      planet++;

   // Update delta checksum
   BarInfo.dx += (LONG)NewPlanet.xCoord - planet->xCoord;

   // Remove unplaced flag
   NewPlanet.bUnplaced = 0;

   // Insert new planet
   BarInfo.planet = *planet = NewPlanet;

   GlobalUnlock(hPlanets);
   return TRUE;
};

LPSTR GetGalaxyName(LPSTR pBuf)
/* Get the galaxy name into pBuf, a pointer to a character buffer */
/* Return pBuf                                                    */
{  PLANET FAR* planet;
   LPSTR  pTemp = pBuf;
   int i;

   planet = (PLANET FAR*) GlobalLock(hPlanets);
   planet += 8; // Skip galaxy size
   for (i=8; i<MAX_GALAXYNAME; i++)
   {   *pTemp++ = planet->Char1;
       if ( !planet->Char1)
           break;
       planet++;
   };
   GlobalUnlock(hPlanets);
   if ( *pBuf == '\0' ) // Use default name if no galaxy name
      _fstrcpy( pBuf, DEFAULT_GALAXY_NAME );
   return pBuf;
};

void UpdateGalaxyName(void)
/* Copies the new galaxy name from the hDlgInfo window into global memory */
/* Copies the new galaxy size into global memory                          */
{  PLANET FAR* planet;
   char szBuffer[MAX_GALAXYNAME+2];
   int i;

   GetDlgItemText(hDlgInfo, IDC_GALAXYNAME, szBuffer, MAX_GALAXYNAME-1);
   planet = (PLANET FAR*) GlobalLock(hPlanets);
   for (i=0; i<=strlen(szBuffer)+8; i++ )
   {   switch (i)   // First four planets contain galaxy size
       {            // galaxy size = 0 -> Galaxy created by MASTER.EXE
	   case 0: planet->Char1 = wGalaxyXFrom & 0xFF; break;
	   case 1: planet->Char1 = wGalaxyXFrom >> 8;   break;
	   case 2: planet->Char1 = wGalaxyXTo & 0xFF;   break;
	   case 3: planet->Char1 = wGalaxyXTo >> 8;     break;
	   case 4: planet->Char1 = wGalaxyYFrom & 0xFF; break;
	   case 5: planet->Char1 = wGalaxyYFrom >> 8;   break;
	   case 6: planet->Char1 = wGalaxyYTo & 0xFF;   break;
	   case 7: planet->Char1 = wGalaxyYTo >> 8;     break;
	   default: planet->Char1 = szBuffer[i-8];   break;
       };
       planet++;
   };
   GlobalUnlock(hPlanets);
};

LPSTR GetNumPlan(LPSTR pBuf)  
{  PLANET FAR* planet;
   int i,Count = 0;

   planet = (PLANET FAR*) GlobalLock(hPlanets);
   for (i=0; i<NUM_PLANETS; i++)
   {   if (!planet->bUnplaced)
          Count++;
       planet++;
   };
   GlobalUnlock(hPlanets);
   wsprintf( pBuf, "Placed %d of 500", Count);
   return pBuf;
};


BOOL FAR PASCAL InfoDlgProc(HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
    static HBRUSH hGreyBrush;
    POINT  point;
    char   szBuffer[MAX_GALAXYNAME];
    BOOL   bFlag;
    static PLANET planet;
    RECT   rect;

    switch(message)
    {
	case WM_INITDIALOG:
	   SendDlgItemMessage( hwnd, IDM_PLANETNAME, EM_LIMITTEXT,
			       PLANETNAME_LENGTH, 0L );
	   SendDlgItemMessage( hwnd, IDC_GALAXYNAME, EM_LIMITTEXT,
                               MAX_GALAXYNAME-1, 0L);
           GetClientRect( hwnd, &rect );
	   MoveWindow( hwnd, -1, -1, 2048, rect.bottom+2, FALSE );
           hGreyBrush = CreateSolidBrush(GetSysColor(COLOR_BTNFACE));
	   SetDlgItemText( hwnd, IDC_GALAXYNAME, GetGalaxyName(szBuffer));
	   InvalidateRect( GetParent(hwnd), NULL, FALSE );
	   return TRUE;

	case WM_CTLCOLOR:
	   SetBkColor(wParam, GetSysColor(COLOR_BTNFACE));
	   SetTextColor(wParam, GetSysColor(COLOR_BTNTEXT));

           // Align the brush
	   UnrealizeObject(hGreyBrush);
	   point.x = point.y = 0;
	   ClientToScreen(hwnd, &point);
	   SetBrushOrg(wParam, point.x, point.y);

           return ((DWORD) hGreyBrush);

	case WM_COMMAND:
	   switch( wParam )
	   {
	       case IDM_REMOVE:   // Pass command to main window
                  SendMessage(GetParent(hwnd), WM_COMMAND, IDM_REMOVE, 0L);
		  break;

	       case IDM_UPDATEBUTTON:
		  if ( !BarInfo.bOK )
		  {  MessageBeep(0);
		     break;
		  };
		  UpdateGalaxyName();
                  planet = BarInfo.planet;
		  GetDlgItemText( hwnd, IDM_PLANETNAME,
		  		  planet.szPlanetName, PLANETNAME_LENGTH );
		  planet.xCoord = GetDlgItemInt( hwnd, IDM_XCOORD, &bFlag, FALSE);
                  if (bFlag) 
		     planet.yCoord = GetDlgItemInt( hwnd, IDM_YCOORD, &bFlag, FALSE);
		  if (bFlag)
		     if (UpdatePlanet( planet ))
		     {   InvalidateRect( GetParent(hwnd), NULL, TRUE );
			 SendMessage( GetParent(hwnd), WM_COMMAND, IDM_PLACENEXT, 0L);
                     }
		     else
		        bFlag = FALSE;	                       

                  if (!bFlag) 
		     MessageBox( hwnd, "Invalid Coordinates", szAppDescr, MB_OK | MB_ICONHAND );

                     // Fall through

               case IDM_UPDATE:
	   	if ( BarInfo.bOK )
		{  SetDlgItemText( hwnd, IDM_PLANETNAME, BarInfo.planet.szPlanetName);
		   SetDlgItemInt ( hwnd, IDM_XCOORD, BarInfo.planet.xCoord, FALSE);
		   SetDlgItemInt ( hwnd, IDM_YCOORD, BarInfo.planet.yCoord, FALSE);

		   SetWindowText( GetDlgItem(hwnd, IDM_PLAN_LEFT), GetNumPlan(szBuffer));
		   InvalidateRect( GetDlgItem(hwnd, IDM_PLAN_LEFT), NULL, TRUE);

		   if ((BarInfo.dx<(long)32000) && (BarInfo.dx>(long)-32000))
		      wsprintf( szBuffer, "dChecksum %d", LOWORD(BarInfo.dx));
		   else
                      wsprintf( szBuffer, "dChecksum %dE3", LOWORD(BarInfo.dx/1000));
		   SetWindowText( GetDlgItem(hwnd, IDM_ERROR), szBuffer);
		   InvalidateRect( GetDlgItem(hwnd, IDM_ERROR), NULL, TRUE);
		};
		break;
           }
	   return TRUE;

	case WM_DESTROY:
	   DeleteObject(GetClassWord(hwnd, GCW_HBRBACKGROUND));
           DeleteObject(hGreyBrush);
	   return TRUE;

    };
    return FALSE;
};


/*------------------------------------------------------------------------*/
/*    Main message-processing procedure                                   */
/*------------------------------------------------------------------------*/

long FAR PASCAL WndProc(HWND hwnd, WORD message, WORD wParam, LONG lParam)
{
   static short            cxClient, cyClient;
   static BYTE             bButtonFlag = TRUE;
   static short            InfoBarHeight;
   static HCURSOR          hCircleCursor;
   static DISPLAY_OPTIONS  DisplayOptions =
   { TRUE, TRUE, TRUE, FALSE, TRUE, TRUE,
     {1000,1000,500,500,250,250,250,200,200,150, // 36 Zoomfactors
      150,150,100,100,100,100,100,100,100,100,
      100,100,100,100,100, 50, 50, 50, 50, 50,
       50, 50, 50, 50, 50, 50 },
     25, 1000, 1000, "", NULL };
   DISPLAY_OPTIONS DisOp;
   PLANET FAR*     planet;
   FARPROC         lpfnDlgProc;
   HDC             hdc, hPrn;
   RECT            rect;
   PAINTSTRUCT     ps;
   HMENU           hMenu;
   long            i, cxSize, cySize;
   WORD            x,y;
   DWORD           dwFlagSave;
   char            szBuffer[40];

   /* Clipboard data */
   GLOBALHANDLE   hGMem = NULL;
   LPMETAFILEPICT lpMFP;
   HMETAFILE      hmf = NULL;

   switch (message)
   {
     case WM_CREATE:
	SetCursor(LoadCursor(NULL, IDC_WAIT));

	/* Allocate FAR global memory to store planet information        */
        /* FAR memory because future versions may store more planet info */
	hPlanets = GlobalAlloc( GMEM_MOVEABLE, NUM_PLANETS*sizeof(PLANET));
	hShips = GlobalAlloc( GMEM_MOVEABLE, MAXSHIP*sizeof(SHIPDATA));
	if ((NULL == hShips) || (NULL == hPlanets))
	{   // Error, cannot get memory !
	    MessageBox( hwnd, "Cannot allocate memory", szAppDescr,
			MB_OK | MB_ICONHAND | MB_SYSTEMMODAL );
	    SendMessage(hwnd, WM_CLOSE, 0, 0L);
	    return TRUE;
	};

	/* Now read the VGA PLANETS data files */
	if (ReadPlanets( hPlanets ))
	{   // Cannot read files !
	    MessageBox( hwnd, "Error reading PLANET.NM, XYPLAN.DAT, RACE.NM or SHIP.HST", szAppDescr,
			MB_OK | MB_ICONHAND | MB_SYSTEMMODAL );
	    SendMessage(hwnd, WM_CLOSE, 0, 0L);
	    return TRUE;
	};

	/* Initialize scroll bars */
	SetScrollRange(hwnd, SB_VERT, wGalaxyYFrom-GALAXY_BORDER, wGalaxyYTo, FALSE);
	SetScrollRange(hwnd, SB_HORZ, wGalaxyXFrom-GALAXY_BORDER, wGalaxyXTo, FALSE);
	SetScrollPos  (hwnd, SB_VERT, wGalaxyYTo-DisplayOptions.yOffset+wGalaxyYFrom, TRUE);
	SetScrollPos  (hwnd, SB_HORZ, DisplayOptions.xOffset, TRUE);

	/* Get handle to default font in hFont */
        CMUFont(hwnd);

	/* Place the planet-names, this takes some time! */
	OrderPlanetNames( hPlanets );

	/* fill in non-variant fields of PRINTDLG struct. */
        pd.lStructSize    = sizeof(PRINTDLG);
        pd.hwndOwner      = hwnd;
        pd.hDevMode       = NULL;
        pd.hDevNames      = NULL;
        pd.Flags          = PD_RETURNDC;
	pd.nCopies        = 1;

	/* Initiate info bar */
	BarInfo.bOK   = FALSE;   // No planet data in bar
	BarInfo.dx    = 0;       // Delta checksum OK at begin

	/* Initialize globals xSheet and ySheet using GetPrinterDC */
	DeleteDC(GetPrinterDC2());

	hCircleCursor = LoadCursor(hInst, "Circle");
	SetCursor(hCircleCursor);  // Remove hourglass cursor

	return 0;

     case WM_SIZE:
	cyClient = HIWORD(lParam);
	cxClient = LOWORD(lParam);
        return 0;

     case WM_VSCROLL:
	i = wGalaxyYTo-(DisplayOptions.yOffset-wGalaxyYFrom);
	switch(wParam)
	{  case SB_LINEUP:   i -= SCROLL_STEP; break;
	   case SB_LINEDOWN: i += SCROLL_STEP; break;
	   case SB_PAGEUP:   i -= DisplayOptions.iGridSize[DisplayOptions.iZoom-1];  break;
	   case SB_PAGEDOWN: i += DisplayOptions.iGridSize[DisplayOptions.iZoom-1];  break;
	   case SB_THUMBPOSITION: i = LOWORD(lParam); break;
	};
	DisplayOptions.yOffset=max(wGalaxyYFrom,
			       min((wGalaxyYTo-i)+wGalaxyYFrom,wGalaxyYTo));
	if ( i != GetScrollPos(hwnd, SB_VERT))
	{  SetScrollPos(hwnd, SB_VERT, wGalaxyYTo-(DisplayOptions.yOffset-wGalaxyYFrom), TRUE);
	   InvalidateRect(hwnd, NULL, TRUE);
	};
        return 0;


     case WM_HSCROLL:
	switch(wParam)
	{
	   case SB_LINEUP:   DisplayOptions.xOffset -= SCROLL_STEP; break;
	   case SB_LINEDOWN: DisplayOptions.xOffset += SCROLL_STEP; break;
	   case SB_PAGEUP:   DisplayOptions.xOffset -= DisplayOptions.iGridSize[DisplayOptions.iZoom-1];  break;
	   case SB_PAGEDOWN: DisplayOptions.xOffset += DisplayOptions.iGridSize[DisplayOptions.iZoom-1];  break;
	   case SB_THUMBPOSITION:
	   		     DisplayOptions.xOffset = LOWORD(lParam); break;
	};
	DisplayOptions.xOffset=max(wGalaxyXFrom,
			       min(DisplayOptions.xOffset, wGalaxyXTo));
	if ( DisplayOptions.xOffset != GetScrollPos(hwnd, SB_HORZ))
	{  SetScrollPos(hwnd, SB_HORZ, DisplayOptions.xOffset, TRUE);
	   InvalidateRect(hwnd, NULL, TRUE);
	};
	return 0;

     case WM_COMMAND:
        hMenu = GetMenu(hwnd);
	switch(wParam)
	{
	    case IDM_ABOUT:
               lpfnDlgProc = MakeProcInstance(About, hInst);
               DialogBox(hInst, "AboutBox", hwnd, lpfnDlgProc);
               FreeProcInstance(lpfnDlgProc);
	       break;

            case IDM_NAMES:
	       DisplayOptions.bNames = 1-DisplayOptions.bNames;
	       CheckMenuItem( hMenu, wParam,
			      DisplayOptions.bNames?MF_CHECKED:MF_UNCHECKED);
               InvalidateRect(hwnd, NULL, TRUE);
	       break;

	    case IDM_IDS:
	       DisplayOptions.bIDs = 1-DisplayOptions.bIDs;
	       CheckMenuItem( hMenu, wParam,
			      DisplayOptions.bIDs?MF_CHECKED:MF_UNCHECKED);
               InvalidateRect(hwnd, NULL, TRUE);
	       break;

	    case IDM_GRID:
	       DisplayOptions.bGrid = 1-DisplayOptions.bGrid;
	       CheckMenuItem( hMenu, wParam,
			      DisplayOptions.bGrid?MF_CHECKED:MF_UNCHECKED);
               InvalidateRect(hwnd, NULL, TRUE);
	       break;

	    case IDM_MARKSHEETS:
	       DisplayOptions.bMarkSheets = 1-DisplayOptions.bMarkSheets;
	       CheckMenuItem( hMenu, wParam,
			      DisplayOptions.bMarkSheets?MF_CHECKED:MF_UNCHECKED);
	       InvalidateRect(hwnd, NULL, TRUE);
	       break;

	    case IDM_ZOOM:
	       lpfnDlgProc = MakeProcInstance(ZoomDlgProc, hInst);
               DisplayOptions.pPd = &pd;
               DialogBoxParam(hInst, "ZoomDlgBox", hwnd, lpfnDlgProc, (DWORD) &DisplayOptions);
               FreeProcInstance(lpfnDlgProc);
               break;

	    case IDM_FIND:
	       lpfnDlgProc = MakeProcInstance(FindDlgProc, hInst);
               DialogBoxParam(hInst, "FindDlgBox", hwnd, lpfnDlgProc, (DWORD) &DisplayOptions);
	       FreeProcInstance(lpfnDlgProc);
	       BarInfo.bOK = TRUE;
	       SendMessage( GetParent(hwnd), WM_COMMAND, IDM_UPDATE, (LONG) (LPSTR) &BarInfo );
	       break;

	    case IDM_PRINT:
	       PrintPage(hwnd, hPlanets, &DisplayOptions );
	       break;

	    case IDM_PRINTERSETUP:
	       dwFlagSave = pd.Flags;
               pd.Flags |= PD_PRINTSETUP;      /* Set option */
	       if ( PrintDlg((LPPRINTDLG)&pd) !=0 )
	       {  xSheet = GetDeviceCaps( pd.hDC, HORZSIZE ); // Updat global
		  ySheet = GetDeviceCaps( pd.hDC, VERTSIZE ); // sheet size info.
		  DeleteDC( pd.hDC );
               };
               pd.Flags = dwFlagSave;          /* Remove option */
               break;

	    case IDM_SAVE:
	       if (BarInfo.dx!=0L)
	       {  if (IDYES == MessageBox( hwnd,
	       		      "dChecksum must be 0!\nCorrect checksum?",
	       		      szAppDescr, MB_YESNO | MB_ICONEXCLAMATION ))
		     CorrectChecksum();
                  else break;  // Never save; program works with relative checksum!
               };
	       if (IDOK == MessageBox( hwnd, "Save planet files\nPLANET.NM, XYPLAN.DAT, RACES.DAT and SHIP.HST",
				       szAppDescr, MB_OKCANCEL | MB_ICONEXCLAMATION ))
	       {   UpdateGalaxyName();
		   OrderPlanetNames(hPlanets);  // Eliminate duplicate names & coords
		   WritePlanets(hPlanets);
               };
	       break;

            case IDM_RACENAME:
	       lpfnDlgProc = MakeProcInstance(RaceDlgProc, hInst);
	       DialogBoxParam(hInst, "RaceDlgBox", hwnd, lpfnDlgProc, (DWORD) &DisplayOptions);
	       FreeProcInstance(lpfnDlgProc);
	       break;

            case IDM_SPIRAL:
	       lpfnDlgProc = MakeProcInstance(SpiralDlgProc, hInst);
	       DialogBoxParam(hInst, "SpiralDlgBox", hwnd, lpfnDlgProc, (DWORD) &DisplayOptions);
	       FreeProcInstance(lpfnDlgProc);
	       break;

	    case IDM_ELLIPSE:
	       lpfnDlgProc = MakeProcInstance(EllipseDlgProc, hInst);
	       DialogBoxParam(hInst, "EllipseDlgBox", hwnd, lpfnDlgProc, (DWORD) &DisplayOptions);
	       FreeProcInstance(lpfnDlgProc);
	       break;

	    case IDM_RECTANGLE:
	       lpfnDlgProc = MakeProcInstance(RectDlgProc, hInst);
	       DialogBoxParam(hInst, "RectDlgBox", hwnd, lpfnDlgProc, (DWORD) &DisplayOptions);
	       FreeProcInstance(lpfnDlgProc);
	       break;

	    case IDM_BAR:
	       MessageBox( hwnd, "Not implemented", szAppName, MB_OK);
               break;
	       lpfnDlgProc = MakeProcInstance(BarDlgProc, hInst);
	       DialogBoxParam(hInst, "BarDlgBox", hwnd, lpfnDlgProc, (DWORD) &DisplayOptions);
	       FreeProcInstance(lpfnDlgProc);
	       break;

	    case IDM_VIEWSHIPS:
	       lpfnDlgProc = MakeProcInstance(ViewShipDlgProc, hInst);
	       DialogBoxParam(hInst, "ViewShipDlgBox", hwnd, lpfnDlgProc, (DWORD) &DisplayOptions);
	       FreeProcInstance(lpfnDlgProc);
	       break;

	    case IDM_GALAXYSIZE:
	       lpfnDlgProc = MakeProcInstance(GalaxyRangeProc, hInst);
	       DialogBoxParam(hInst, "GalaxyRangeBox", hwnd, lpfnDlgProc, (DWORD) &DisplayOptions);
	       FreeProcInstance(lpfnDlgProc);
	       SetScrollRange(hwnd, SB_VERT, wGalaxyYFrom-GALAXY_BORDER, wGalaxyYTo, FALSE);
	       SetScrollRange(hwnd, SB_HORZ, wGalaxyXFrom-GALAXY_BORDER, wGalaxyXTo, FALSE);
	       SetScrollPos  (hwnd, SB_VERT, wGalaxyYTo-(DisplayOptions.yOffset-wGalaxyYFrom), TRUE);
	       SetScrollPos  (hwnd, SB_HORZ, DisplayOptions.xOffset, TRUE);
               break;

	    case IDM_CORRECTCHECKSUM:
	       if (IDOK == MessageBox( hwnd,
	       		   "The x-coordinates of the planets\nwill change a bit! Continue?",
			   "Correct checksum", MB_OKCANCEL | MB_ICONEXCLAMATION ))
	       {  CorrectChecksum();
		  InvalidateRect(hwnd, NULL, TRUE);  // Planets might have moved
               };
	       break;

	    case IDM_REMOVE:     // Remove a planet
	       if ( BarInfo.planet.bUnplaced )
	       {  MessageBeep(0);
		  break;
               };
	       planet = (PLANET FAR*) GlobalLock(hPlanets);
	       for (i=0; i<NUM_PLANETS; i++)
	       {   if ( planet->ID == BarInfo.planet.ID )
		   {  planet->bUnplaced = 1;
		      BarInfo.planet = *planet;
		      SendMessage( hDlgInfo, WM_COMMAND, IDM_UPDATE, 0L );
                      break; // leave for-loop
                   };
      		   planet++;
	       };
	       InvalidateRect(hwnd, NULL, TRUE);
	       break;

	    case IDM_PLACENEXT:  // Get next planet to place
	       planet = (PLANET FAR*) GlobalLock(hPlanets);
	       for (i=0; i<NUM_PLANETS; i++)
	       {   if ( planet->bUnplaced )
		   {  BarInfo.planet = *planet;
		      SendMessage( hDlgInfo, WM_COMMAND, IDM_UPDATE, 0L );
		      i=0;   // For if(i) below
                      break; // leave for-loop
                   };
      		   planet++;
	       };
	       GlobalUnlock(hPlanets);
	       break;

	    case IDM_REMOVEALL:  // Remove all planets
	       planet = (PLANET FAR*) GlobalLock(hPlanets);
	       for (i=0; i<NUM_PLANETS; i++)
	       {   planet->bUnplaced = 1;
                   if ( BarInfo.planet.ID == planet->ID)
		      BarInfo.planet = *planet;
		   planet++;
	       };
	       GlobalUnlock(hPlanets);
	       SendMessage( hDlgInfo, WM_COMMAND, IDM_UPDATE, 0L );
               SendMessage(hwnd, WM_COMMAND, IDM_PLACENEXT, 0L);
	       InvalidateRect(hwnd, NULL, TRUE);
	       break;

	    case IDM_RESTORE:    // Restore all planets
	       planet = (PLANET FAR*) GlobalLock(hPlanets);
	       for (i=0; i<NUM_PLANETS; i++)
	       {   planet->bUnplaced = 0;
		   if ( BarInfo.planet.ID == planet->ID )
		     BarInfo.planet = *planet;
      		   planet++;
               };
	       GlobalUnlock(hPlanets);
	       SendMessage( hDlgInfo, WM_COMMAND, IDM_UPDATE, 0L );
	       InvalidateRect(hwnd, NULL, TRUE);
	       break;

	    case IDM_ORDERPLANETNAMES:
	       OrderPlanetNames( hPlanets );
	       InvalidateRect( hwnd, NULL, TRUE);
	       break;

	    case IDM_PLACE:  // identical to place button in info-bar
	       SendMessage( hDlgInfo, WM_COMMAND, IDM_UPDATEBUTTON, 0L);
	       break;

	    case IDM_HELP:
	       WinHelp( hwnd, "PLAN_MAP.HLP", HELP_CONTENTS, 0L );
               break;

	    case IDM_FONT:
	       CMUFont(hwnd);
	       break;

	    case IDM_MMCOPY:
	       cxSize = (long)(wGalaxyXTo-wGalaxyXFrom)*8;
	       cySize = (long)(wGalaxyYTo-wGalaxyYFrom)*8;
	       if (NULL == (hdc = CreateMetaFile(NULL)))
	       {  MessageBox( hwnd, "Can't create meta file", szAppName, MB_OK | MB_ICONEXCLAMATION );
		  break; };

	       SetWindowExt(hdc, cxSize, cySize);
	       SetWindowOrg(hdc, 0, cySize);  // Bottom left
	       DisOp = DisplayOptions;
	       DisOp.xOffset = wGalaxyXFrom;
	       DisOp.yOffset = wGalaxyYFrom;
               DisOp.iZoom   = 8;
	       XYMap( hdc, hPlanets, &DisOp, cxSize, cySize);

	       if (NULL == (hmf = CloseMetaFile( hdc )))
	       {  MessageBox( hwnd, "Can't close meta file", szAppName, MB_OK | MB_ICONEXCLAMATION );
		  break; };

	       if (NULL == (hGMem = GlobalAlloc(GHND, (DWORD) sizeof(METAFILEPICT))))
	       {  MessageBox( hwnd, "Can't allocate memory", szAppName, MB_OK | MB_ICONEXCLAMATION );
		  break; };

	       lpMFP = (LPMETAFILEPICT) GlobalLock(hGMem);
	       lpMFP->mm   = MM_TWIPS;
	       lpMFP->xExt = cxSize;
	       lpMFP->yExt = cySize;
	       lpMFP->hMF  = hmf;
	       GlobalUnlock(hGMem);

	       OpenClipboard(hwnd);
	       EmptyClipboard();
	       SetClipboardData(CF_METAFILEPICT, hGMem);
	       CloseClipboard();
               break;

	    case IDM_EXIT:
	       SendMessage(hwnd, WM_CLOSE, 0, 0L);
	       break;
	};
	return 0;

     case WM_LBUTTONDOWN:  // Snap to closest planet
        if (!( wParam & MK_RBUTTON ))
	{  hdc = GetDC( hwnd );
	   SetMapMode( hdc, MM_TWIPS );
	   SetViewportOrg( hdc, 0, cyClient );   // Bottom left
           MoveToPlanet( hwnd, hdc, lParam, &DisplayOptions );
	   ReleaseDC( hwnd, hdc );
	   BarInfo.bOK = TRUE;
           SendMessage( hDlgInfo, WM_COMMAND, IDM_UPDATE, 0L );
	   return 0;
	}
	// Fall through; both buttons are pressed -> move planet

     case WM_RBUTTONDOWN:
     case WM_RBUTTONUP:
     case WM_MOUSEMOVE:
	if ((wParam & MK_RBUTTON) && (BarInfo.bOK))
	{  SetCursor(LoadCursor(NULL, IDC_CROSS));
	   hdc = GetDC( hwnd );
	   SetMapMode( hdc, MM_TWIPS );
	   SetViewportOrg( hdc, 0, cyClient );  // Bottom left
           rect.right  = LOWORD(lParam);        // Move mouse position into rect
           rect.bottom = HIWORD(lParam);
           DPtoLP( hdc, (LPPOINT) &rect, 2 );   // mouse pos in twips
	   ReleaseDC( hwnd, hdc );
	   x = rect.right/DisplayOptions.iZoom  + DisplayOptions.xOffset;
	   y = rect.bottom/DisplayOptions.iZoom + DisplayOptions.yOffset;
	   SetDlgItemInt ( hDlgInfo, IDM_XCOORD, x, FALSE);
	   SetDlgItemInt ( hDlgInfo, IDM_YCOORD, y, FALSE);
	   if ((BarInfo.dx-BarInfo.planet.xCoord+x<(long)32000) &&
	       (BarInfo.dx-BarInfo.planet.xCoord+x>(long)-32000))
	     wsprintf( szBuffer, "dChecksum %d", LOWORD(BarInfo.dx-BarInfo.planet.xCoord+x));
	   else
	     wsprintf( szBuffer, "dChecksum %dE3", LOWORD((BarInfo.dx-BarInfo.planet.xCoord+x)/1000));
	   SetWindowText( GetDlgItem(hDlgInfo, IDM_ERROR), szBuffer);
	   InvalidateRect( GetDlgItem(hDlgInfo, IDM_ERROR), NULL, TRUE);
	   if (wParam & MK_LBUTTON)   // Place planet
	   {   if ( bButtonFlag )
	       {   MessageBeep(0);
                   bButtonFlag = FALSE;
		   SendMessage( hDlgInfo, WM_COMMAND, IDM_UPDATEBUTTON, 0L);
	       } 
	   }
           else bButtonFlag = TRUE;
	}
	else
	   SetCursor( hCircleCursor );
        return 0;

     case WM_PAINT:
	if (!hDlgInfo) break;  // Wait for info bar to appear first

	/* Get height of InfoBar */
	GetWindowRect( hDlgInfo, &rect );
	InfoBarHeight = rect.bottom-rect.top; 

	hdc = BeginPaint(hwnd, &ps);
	SetMapMode(hdc, MM_TWIPS);
	SetViewportOrg(hdc, 0, cyClient);  // Bottom left
	rect.right = cxClient; rect.bottom = cyClient;
	rect.left  = 0;        rect.top    = InfoBarHeight;
	DPtoLP(hdc, (LPPOINT) &rect, 2);   
        XYMap( hdc, hPlanets, &DisplayOptions, rect.right, rect.top);
	EndPaint(hwnd, &ps);
	return 0;

     case WM_DESTROY:
        // Destroy handles in PRINTDLG sturcture
	if (pd.hDevNames) GlobalFree(pd.hDevNames);
        if (pd.hDevMode)  GlobalFree(pd.hDevMode);
	if (hPlanets) GlobalFree( hPlanets );
        if (hShips)   GlobalFree( hShips );
	DeleteObject( hfFont );
        DestroyCursor( hCircleCursor );
	PostQuitMessage(0);
	return 0;
   }
   return DefWindowProc(hwnd, message, wParam, lParam);
};
