/*************************************************************************
Legends Soldier Calculator
Copyright (c) 1993 by Mark R. Nuiver.  All rights reserved.

Module Name      => lscdata.c
Version          => 4.0.0
Change Date      => 94/03/09
Programmer       => M.R. Nuiver
Operating System => AmigaDOS 3.00                  MS-DOS 6.0
Compiler         => SAS/C C 5.10                   Power C 2.0.0
Modules Used     => strztok strblk
  (Amiga Only)   => c.o
Header Files     => lsc.h
Libraries Used   => lc.lib, lcm.lib amiga2.0.lib   PCLIB.MIX, PCAUTO.MIX
Compiler Options => None
Data Files       => ARMOR.LSC. MOUNTS.LSC, RACES.LSC, SHIELDS.LSC,
                    STATUS.LSC, TYPES.LSC, WEAPONS.LSC

Description

  This module provides the functions to read legion description files and
data files.

Revision Notes

Version Date      Programmer      Reason
------- --------  --------------  ---------------------------------------
4.0.0   94/03/09  Mark Nuiver     * Changed validate_terrain() to allow
                                    Defending Walls terrain even if there
                                    are no fortifications, since locations
                                    seem to always use that modifier whether
                                    they have any fortifications or not.
4.0.0   94/02/25  Mark Nuiver     * Fixed is_forts() - had Forts[OVER] which
                                    was invalid - detected it in DOS version.
4.0.0   94/02/22  Mark Nuiver     * Removed references to TWI (Winternight),
                                    added TAI (Air).
4.0.0   94/01/08  Mark Nuiver     * Added weapon class "Other".
                                  * Added expanded character data input.
4.0.0   94/01/02  Mark Nuiver     * Used enum reserve_options on
                                    Reservenames[].
                                  * Replaced hard-coded terrain and reserve
                                    codes with appropriate symbols.
                                  * Created is_reserve_slot().
4.0.0   93/12/27  Mark Nuiver     * Added Reserve Options and Acceptable
                                    Losses to load_legion(); adjusted
                                    characters, forts, and terrain for new
                                    lsc.h definitions.
                                  * Created validate_legion().
                                  * Added validate_terrain().
                                  * Modified print_terrain().
                                  * Added is_forts().
                                  * Added validate_reserve() and
                                    validate_reserve_forts().
                                  * Added Reservenames[].
                                  * Added print_reserve().
                                  * Added validate_soldiers().
                                  * Added validate_characters().
                                  * Added validate_forts().
                                  * Added validate_losses().
                                  * Added print_losses().
4.0.0   93/12/22  Mark Nuiver     * Added Terrain stuff to Fieldnames[],
                                    tables.  Made Fieldnames[] global.
                                  * Added Terrainfields[], modified
                                    load_legion() to load terrain data.
                                  * Moved print_terrain() from lsc.c to
                                    share with lbs.
4.0.0   93/12/21  Mark Nuiver     * Changed version.
                                  * Added GWC, GWB, SWID, SWB to RACES,
                                    TYPES, STATUS.  Removed SPC.
                                  * Added BWID to WEAPONS.
                                  * Added OVER, and used it in characters and
                                    fortifications instead of the misleading
                                    re-use of CAFB.
                                  * Removed claymore and dart/shortsword as
                                    unique weapon classes.
                                  * Removed TAC symbol.
------- --------  --------------  ---------------------------------------
3.0.1   93/12/19  Mark Nuiver     * Changed version.
                                  * Changed all int to short, *Tablefields[]
                                    to enum.
------- --------  --------------  ---------------------------------------
3.0.0   93/12/12  Mark Nuiver     * Added DAM and INV to tables.
3.0.0   93/12/11  Mark Nuiver     * Created this module from original lsc.c
                                    for re-use with LBS program.
*************************************************************************/

#include "lsc.h"

/* external functions */
extern char *strblk(char *);
extern char *strztok(char *, char *);
extern enum lsc_symbols decode_terrain(short);

/* global functions */
void cleanup(enum lsc_errors, char *, short);
void load_legion(char *, struct legion_struct *);
void load_tables(struct legion_struct *);
void init_tables(void);
short is_forts(struct legion_struct *);
short is_reserve_slot(short);
void print_terrain(short *);
void print_reserve(short *);
void print_losses(short *);
void validate_legion(struct legion_struct *);

/* global variables */
char *Sfieldnames[LASTSFIELD]; /* names of soldier fields */
char *Fieldnames[LASTFIELD];  /* names of the fields (strings) */
char *Reservenames[RO_END]; /* names of the Reserve Options */

/* local variables */
static char colons[] = ":";
static char line[LINESIZE + 1];
static FILE *filefp;

static char *Tablefiles[LASTTABLE];  /* the Table file names */
static enum lsc_symbols *Tablefields[LASTTABLE]; /* fields in the Tables */

/*
  the following arrays hold the fields in the tables in the order that they
  appear on each line, terminated by the meaningless end-symbol LASTFIELD.
*/
static enum lsc_symbols Racefields[LASTFIELD] =
  { ID, DESC, CF, DF, RAFB, AFB, CAFB, WAFB, MAFB, SAFB, MAD, SAD, SAV,
    DAM, INV, GWC, GWB, SWID, SWB, TGR, TFO, THF, TFH, TFM, TBH, TBM,
    TMO, TSW, TDE, TSE, TDW, TAW, TCI, TAI, TCO, LASTFIELD };
static enum lsc_symbols Typefields[LASTFIELD] =
  { ID, DESC, CF, DF, AFB, CAFB, RAFB, MORB, MAFB, SAFB, MAD, SAD, SAV,
    DAM, INV, GWC, GWB, SWID, SWB, TGR, TFO, THF, TFH, TFM, TBH, TBM, TMO,
    TSW, TDE, TSE, TDW, TAW, TCI, TAI, TCO, LASTFIELD };
static enum lsc_symbols Statusfields[LASTFIELD] =
  { ID, DESC, CF, DF, AFB, CAFB, RAFB, MORB, MAFB, SAFB, MAD, SAD, SAV,
    DAM, INV, GWC, GWB, SWID, SWB, TGR, TFO, THF, TFH, TFM, TBH, TBM, TMO,
    TSW, TDE, TSE, TDW, TAW, TCI, TAI, TCO, LASTFIELD };
static enum lsc_symbols Weaponfields[LASTFIELD] =
  { ID, DESC, AFB, CAFB, RAFB, MAFB, MAD, SAD, CLASS, DAM, BWID, TGR, TFO,
    THF, TFH, TFM, TBH, TBM, TMO, TSW, TDE, TSE, TDW, TAW, TCI, TAI, TCO,
    LASTFIELD };
static enum lsc_symbols Armorfields[LASTFIELD] =
  { ID, DESC, DF, RAFB, CAFB, MAD, SAD, INV, TGR, TFO, THF, TFH, TFM, TBH,
    TBM, TMO, TSW, TDE, TSE, TDW, TAW, TCI, TAI, TCO, LASTFIELD };
static enum lsc_symbols Shieldfields[LASTFIELD] =
  { ID, DESC, DF, RAFB, CAFB, MAD, SAD, INV, TGR, TFO, THF, TFH, TFM, TBH,
    TBM, TMO, TSW, TDE, TSE, TDW, TAW, TCI, TAI, TCO, LASTFIELD };
static enum lsc_symbols Mountfields[LASTFIELD] =
  { ID, DESC, CF, CAFB, RAFB, SAV, TGR, TFO, THF, TFH, TFM, TBH, TBM, TMO,
    TSW, TDE, TSE, TDW, TAW, TCI, TAI, TCO, LASTFIELD };

/*
  the following arrays hold the fields in the legion description file in the
  order that they appear on appropriate lines, terminated by the meaningless
  end-symbols.
*/
static enum terrain_symbols Terrainfields[LASTTERRAIN + 1] =
  { T_BASIC, T_WALLS, T_SPECIAL, LASTTERRAIN };
static enum forts_symbols Fortificationfields[LASTFORTS + 1] =
  { F_DF, F_MORB, F_AFB, F_OVER, LASTFORTS };
static enum lsc_symbols Characterfields[LASTCFIELD + 1] =
  { ID, DESC, CF, DF, AF, MAF, CAF, MAV, SAV, MAD, SAD, DAM, INV,
    AFB, OVER, LASTCFIELD };
static enum table_type Soldierfields[LASTSFIELD + 1] =
  { AMT, RACE, TYPE, STATUS, TRL, MOR, WEAPON, ARMOR, SHIELD, MOUNT,
    LASTSFIELD };
static enum reserve_symbols Reservefields[LASTRESERVE + 1] =
  { R_LEFT, R_CENTER, R_RIGHT, LASTRESERVE };
static enum losses_symbols Lossesfields[LASTLOSSES + 1] =
  { L_LOSSES, LASTLOSSES };

/* local function prototypes */
static void load_table(short, struct legion_struct *);
static void validate_soldiers(struct legion_struct *);
static void validate_characters(struct legion_struct *);
static void validate_reserve_forts(struct legion_struct *);
static void validate_forts(struct legion_struct *);
static void validate_terrain(struct legion_struct *);
static void validate_reserve(struct legion_struct *);
static void validate_losses(struct legion_struct *);

/*************************************************************************
cleanup:
*************************************************************************/
void cleanup(ferr, filename, line_count)
enum lsc_errors ferr;
char *filename;
short line_count;
  {

    if(ferr != ERROR_NO_ERROR)
      {

        printf("\n\nError in line %d of file %s", line_count, filename);
        if(*line != '\0') printf("\n%s", line);

      }

    switch(ferr)
      {

        case ERROR_NO_ERROR:
          {

            break;

          }

        case ERROR_NO_FILE:
          {

            printf("\nFile not found.\n");
            break;

          }

        case ERROR_NO_DATA:
          {

            printf("\nNo data in file.\n");
            break;

          }

        case ERROR_BAD_DATA:
          {

            printf("\nBad data in file.\n");
            break;

          }

        default:
          {

            printf("\nUndefined error in file.\n");
            break;

          }

      } /* end switch(ferr) */

    if(filefp != NULL) fclose(filefp);
    exit(ferr);

  } /* end of cleanup() */

/*************************************************************************
is_forts:
  returns TRUE if fortified, FALSE if not.
*************************************************************************/
short is_forts(Legion)
struct legion_struct *Legion;
  {

    if(Legion->Forts[F_DF] > 0 || Legion->Forts[F_MORB] > 0 ||
       Legion->Forts[F_AFB] > 0 || Legion->Forts[F_OVER] > 0)
      { /* fortified */

        return(TRUE);

      } /* fortified */

    return(FALSE);

  } /* end of is_forts() */

/*************************************************************************
is_reserve_slot:
  returns TRUE if slot is a reserve slot (6-8)
  returns FALSE otherwise
*************************************************************************/
short is_reserve_slot(slot)
short slot;
  {

    if(slot >= 6 && slot <= 8) return(TRUE);

    return(FALSE);

  } /* end of is_reserve_slot() */

/*************************************************************************
validate_soldiers:
  validates the soldier inputs.
*************************************************************************/
static void validate_soldiers(Legion)
struct legion_struct *Legion;
  {

    short slot;

    for(slot = 1; slot <= MAXSSLOTS; slot++)
      { /* all soldier slots */

        if(Legion->Slot[slot].SoldierItems[AMT] < 0)
          Legion->Slot[slot].SoldierItems[AMT] = 0;
        if(Legion->Slot[slot].SoldierItems[TRL] < 0)
          Legion->Slot[slot].SoldierItems[TRL] = 0;
        if(Legion->Slot[slot].SoldierItems[RACE] < 0)
          Legion->Slot[slot].SoldierItems[RACE] = 0;
        if(Legion->Slot[slot].SoldierItems[TYPE] < 0)
          Legion->Slot[slot].SoldierItems[TYPE] = 0;
        if(Legion->Slot[slot].SoldierItems[STATUS] < 0)
          Legion->Slot[slot].SoldierItems[STATUS] = 0;
        if(Legion->Slot[slot].SoldierItems[WEAPON] < 0)
          Legion->Slot[slot].SoldierItems[WEAPON] = 0;
        if(Legion->Slot[slot].SoldierItems[ARMOR] < 0)
          Legion->Slot[slot].SoldierItems[ARMOR] = 0;
        if(Legion->Slot[slot].SoldierItems[SHIELD] < 0)
          Legion->Slot[slot].SoldierItems[SHIELD] = 0;
        if(Legion->Slot[slot].SoldierItems[MOUNT] < 0)
          Legion->Slot[slot].SoldierItems[MOUNT] = 0;

      } /* all soldier slots */

  } /* end of validate_soldiers() */

/*************************************************************************
validate_characters:
  validates the character data inputs.
*************************************************************************/
static void validate_characters(Legion)
struct legion_struct *Legion;
  {

    short slot;

    for(slot = 1; slot <= MAXCSLOTS; slot++)
      { /* all character slots */

        if(Legion->Slot[slot].Character[CF] < 0)
          Legion->Slot[slot].Character[CF] = 0;
        if(Legion->Slot[slot].Character[DF] < 1)
          Legion->Slot[slot].Character[DF] = 1;
        if(Legion->Slot[slot].Character[AF] < 0)
          Legion->Slot[slot].Character[AF] = 0;
        if(Legion->Slot[slot].Character[MAF] < 0)
          Legion->Slot[slot].Character[MAF] = 0;
        if(Legion->Slot[slot].Character[CAF] < 0)
          Legion->Slot[slot].Character[CAF] = 0;
        if(Legion->Slot[slot].Character[MAV] < 0)
          Legion->Slot[slot].Character[MAV] = 0;
        if(Legion->Slot[slot].Character[SAV] < 0)
          Legion->Slot[slot].Character[SAV] = 0;
        if(Legion->Slot[slot].Character[MAD] < 1)
          Legion->Slot[slot].Character[MAD] = 1;
        if(Legion->Slot[slot].Character[SAD] < 1)
          Legion->Slot[slot].Character[SAD] = 1;
        if(Legion->Slot[slot].Character[DAM] < 0)
          Legion->Slot[slot].Character[DAM] = 0;
        if(Legion->Slot[slot].Character[INV] < 0)
          Legion->Slot[slot].Character[INV] = 0;

      } /* all character slots */

  } /* end of validate_characters() */

/*************************************************************************
validate_forts:
  validates the fortifications inputs.
*************************************************************************/
static void validate_forts(Legion)
struct legion_struct *Legion;
  {

    if(Legion->Forts[F_DF] < 0) Legion->Forts[F_DF] = 0;
    if(Legion->Forts[F_MORB] < 0) Legion->Forts[F_MORB] = 0;
    if(Legion->Forts[F_AFB] < 0) Legion->Forts[F_AFB] = 0;
    if(Legion->Forts[F_OVER] < 0) Legion->Forts[F_OVER] = 0;

  } /* end of validate_forts() */

/*************************************************************************
validate_reserve_forts:
  validates the reserve options with walls terrain.
*************************************************************************/
static void validate_reserve_forts(Legion)
struct legion_struct *Legion;
  {

    short *Reserve = &(Legion->Reserve[0]);

    /* some options aren't allowed when attacking walls */
    if(Legion->Terrain[T_WALLS] == TAW - TBEGIN)
      { /* attacking walls */

        if((1 << (Reserve[R_LEFT] - 1)) & M_R_NOATTACK)
           Reserve[R_LEFT] = RO_TACTICAL_RESERVE;
        if((1 << (Reserve[R_CENTER] - 1)) & M_R_NOATTACK)
           Reserve[R_CENTER] = RO_TACTICAL_RESERVE;
        if((1 << (Reserve[R_RIGHT] - 1)) & M_R_NOATTACK)
           Reserve[R_RIGHT] = RO_TACTICAL_RESERVE;

      } /* attacking walls */

    /* some options aren't allowed when defending walls */
    if(Legion->Terrain[T_WALLS] == TDW - TBEGIN)
      { /* defending walls */

        if((1 << (Reserve[R_LEFT] - 1)) & M_R_NODEFEND)
           Reserve[R_LEFT] = RO_TACTICAL_RESERVE;
        if((1 << (Reserve[R_CENTER] - 1)) & M_R_NODEFEND)
           Reserve[R_CENTER] = RO_TACTICAL_RESERVE;
        if((1 << (Reserve[R_RIGHT] - 1)) & M_R_NODEFEND)
           Reserve[R_RIGHT] = RO_TACTICAL_RESERVE;

      } /* defending walls */

  } /* end of validate_reserve_forts */

/*************************************************************************
validate_reserve:
  validates the reserve options inputs.
*************************************************************************/
static void validate_reserve(Legion)
struct legion_struct *Legion;
  {

    short *Reserve = &(Legion->Reserve[0]);

    if(Reserve[R_LEFT] <= RO_BEGIN || Reserve[R_LEFT] >= RO_END)
      Reserve[R_LEFT] = RO_TACTICAL_RESERVE;
    if(Reserve[R_CENTER] <= RO_BEGIN || Reserve[R_CENTER] >= RO_END)
      Reserve[R_CENTER] = RO_TACTICAL_RESERVE;
    if(Reserve[R_RIGHT] <= RO_BEGIN || Reserve[R_RIGHT] >= RO_END)
      Reserve[R_RIGHT] = RO_TACTICAL_RESERVE;

    /* strategic reserve becomes tactical reserve */
    if(Reserve[R_LEFT] == RO_STRATEGIC_RESERVE)
      Reserve[R_LEFT] = RO_TACTICAL_RESERVE;
    if(Reserve[R_CENTER] == RO_STRATEGIC_RESERVE) Reserve[R_CENTER] =
      RO_TACTICAL_RESERVE;
    if(Reserve[R_RIGHT] == RO_STRATEGIC_RESERVE) Reserve[R_RIGHT] =
      RO_TACTICAL_RESERVE;

    /* some options aren't allowed for all reserve slots */
    if((1 << (Reserve[R_LEFT] - 1)) & M_R_NOLEFT)
       Reserve[R_LEFT] = RO_TACTICAL_RESERVE;
    if((1 << (Reserve[R_CENTER] - 1)) & M_R_NOCENTER)
       Reserve[R_CENTER] = RO_TACTICAL_RESERVE;
    if((1 << (Reserve[R_RIGHT] - 1)) & M_R_NORIGHT)
       Reserve[R_RIGHT] = RO_TACTICAL_RESERVE;

    /* validate the reserve options with fortifications */
    validate_reserve_forts(Legion);

  } /* end of validate_reserve() */

/*************************************************************************
validate_terrain:
  validates the terrain inputs.
*************************************************************************/
static void validate_terrain(Legion)
struct legion_struct *Legion;
  {

    if(!decode_terrain(Legion->Terrain[T_BASIC]))
      { /* terrain out of range */

        Legion->Terrain[T_BASIC] = 0;

      } /* terrain out of range */
    else
      { /* terrain in range */

        if(!((1 << (Legion->Terrain[T_BASIC] - 1)) & M_T_BASIC))
          { /* invalid terrain */

            Legion->Terrain[T_BASIC] = 0;

          } /* invalid terrain */

      } /* terrain in range */

    if(!decode_terrain(Legion->Terrain[T_WALLS]))
      { /* terrain out of range */

        Legion->Terrain[T_WALLS] = 0;

      } /* terrain out of range */
    else
      { /* terrain in range */

        if(!((1 << (Legion->Terrain[T_WALLS] - 1)) & M_T_WALLS))
          { /* invalid terrain */

            Legion->Terrain[T_WALLS] = 0;

          } /* invalid terrain */

      } /* terrain in range */

    if(!decode_terrain(Legion->Terrain[T_SPECIAL]))
      { /* terrain out of range */

        Legion->Terrain[T_SPECIAL] = 0;

      } /* terrain out of range */
    else
      { /* terrain in range */

        if(!((1 << (Legion->Terrain[T_SPECIAL] - 1)) & M_T_SPECIAL))
          { /* invalid terrain */

            Legion->Terrain[T_SPECIAL] = 0;

          } /* invalid terrain */

      } /* terrain in range */

    /* if fortifications are being included in the calculations, */
    /* then wall terrain must be defending walls.  if there are no */
    /* fortifications, however, the walls terrain is still permitted */
    /* to be defending walls, as all locations appear use that modifier */
    /* whether there are any fortifications or not. */
    if(is_forts(Legion))
      { /* forts present */

        Legion->Terrain[T_WALLS] = TDW - TBEGIN;

      } /* forts present */

  } /* end of validate_terrain() */

/*************************************************************************
validate_losses:
  validates the Acceptable Losses input.
*************************************************************************/
static void validate_losses(Legion)
struct legion_struct *Legion;
  {

    if(Legion->Losses[L_LOSSES] < 1 || Legion->Losses[L_LOSSES] > 100)
      Legion->Losses[L_LOSSES] = 100;

  } /* end of validate_losses() */

/*************************************************************************
validate_legion:
  validates all the input data
*************************************************************************/
void validate_legion(Legion)
struct legion_struct *Legion;
  {

    /* validate the inputs */
    validate_soldiers(Legion);
    validate_characters(Legion);
    validate_forts(Legion);
    validate_terrain(Legion);
    validate_reserve(Legion);
    validate_losses(Legion);

  } /* end of validate_legion() */

/*************************************************************************
print_terrain:
  prints the terrain conditions for the analysis
  NOTE: assumes that terrain has already been validated, and is valid entry
or zero.
*************************************************************************/
void print_terrain(Terrain)
short Terrain[];
  {

    if(Terrain[T_BASIC] == 0)
      { /* no basic terrain */

        printf("\nFighting in unspecified terrain");

      } /* no basic terrain */
    else
      { /* specified basic terrain */

        printf("\nFighting in %s terrain",
               Fieldnames[decode_terrain(Terrain[T_BASIC])]);

      } /* specified basic terrain */

    if(Terrain[T_WALLS] != 0)
      { /* walls terrain specified */

        printf(", and %s",
               Fieldnames[decode_terrain(Terrain[T_WALLS])]);

      } /* walls terrain specified */

    if(Terrain[T_SPECIAL] != 0)
      { /* special terrain specified */

        printf(", and %s",
               Fieldnames[decode_terrain(Terrain[T_SPECIAL])]);

      } /* special terrain specified */

  } /* end of print_terrain() */

/*************************************************************************
print_reserve:
  prints the reserve options.
*************************************************************************/
void print_reserve(Reserve)
short Reserve[];
  {

    printf("\n");
    printf("\nLeft   Reserve Option: %s", Reservenames[Reserve[R_LEFT]]);
    printf("\nCenter Reserve Option: %s", Reservenames[Reserve[R_CENTER]]);
    printf("\nRight  Reserve Option: %s", Reservenames[Reserve[R_RIGHT]]);

  } /* end of print_reserve() */

/*************************************************************************
print_losses:
  prints the Acceptable Losses.
*************************************************************************/
void print_losses(Losses)
short Losses[];
  {

    printf("\n\nAcceptable Losses: %d%%", Losses[L_LOSSES]);

  } /* end of print_losses() */

/*************************************************************************
load_legion:
*************************************************************************/
void load_legion(Legionfile, Legion)
char *Legionfile;
struct legion_struct *Legion;
  {

    short soldier_count = 0; /* # of soldier slots loaded from file */
    short character_count = 0; /* # of character slots loaded from file */
    short all_count = 0; /* # of data lines of all kinds loaded */
    short line_count = 0; /* # of lines read from file */
    char *slot;
    char prefix;
    short slt;
    short i; /* for() counter */
    char *sfield[LASTSFIELD]; /* input soldier data */
    char *cfield[LASTCFIELD]; /* input character data */
    char *ffield[LASTFORTS]; /* input forts data */
    char *tfield[LASTTERRAIN]; /* input terrain data */
    char *rfield[LASTRESERVE]; /* input reserve options data */
    char *lfield[LASTLOSSES]; /* input acceptable losses data */
    char tline[LINESIZE + 1]; /* working copy of input line */

    line[0] = '\0';
    filefp = fopen(Legionfile, "r");
    if(filefp == NULL)
      {

        cleanup(ERROR_NO_FILE, Legionfile, line_count);

      }

    while(fgets(line, LINESIZE + 1, filefp) != NULL)
      { /* read legion data */

        line[strlen(line) - 1] = '\0'; /* remove LF */
        strcpy(tline, line);
        line_count = line_count + 1;
        if(line[0] != '#' && line[0] != '\0')
          { /* not a comment or blank line */

            /* get the first field - slot number & possible prefix */
            slot = strblk(strztok(tline, colons));
            if(isalpha(slot[0]))
              {

                prefix = toupper(slot[0]);

              }

            else
              {

                prefix = '\0';

              }

            switch(prefix)
              {

                case '\0':
                  { /* no prefix - soldier slot */

                    /* convert the slot number to integer */
                    slt = atoi(slot);
                    /* validate the slot number */
                    if(slt < 1 || slt > MAXSSLOTS)
                      {

                        cleanup(ERROR_BAD_DATA, Legionfile, line_count);

                      }

                    /* input the remaining fields */
                    i = 0;
                    while(Soldierfields[i] != LASTSFIELD)
                      { /* input the soldier slot fields */

                        sfield[Soldierfields[i]] = strztok(NULL, colons);
                        if(sfield[Soldierfields[i]] == NULL)
                          { /* ran out of tokens */

                            cleanup(ERROR_BAD_DATA, Legionfile, line_count);

                          } /* ran out of tokens */

                        else
                          { /* convert the strings to decimal */

                            /* v2.0.0 - atoi() replaces sscanf() */
                            Legion->Slot[slt].SoldierItems[Soldierfields[i]] =
                              atoi(sfield[Soldierfields[i]]);
                            #ifdef DEBUG
                            printf("\n%s:field[%s]='%s'",
                                   Legionfile, Sfieldnames[Soldierfields[i]],
                                   sfield[Soldierfields[i]]);
                            printf(",Slot[%d].SoldierItems[%s]=%d", slt,
                                  Sfieldnames[Soldierfields[i]],
                            Legion->Slot[slt].SoldierItems[Soldierfields[i]]);
                            #endif

                          } /* convert the strings to decimal */

                        i++; /* proceed to next field */

                      } /* input the soldier fields */

                    soldier_count = soldier_count + 1;
                    all_count = all_count + 1;
                    break;

                  } /* no prefix - soldier slot */

                case 'C':
                  { /* character slot */

                    /* convert the slot number to integer */
                    slt = atoi(&slot[1]);
                    /* validate the slot number */
                    if(slt < 1 || slt > MAXCSLOTS)
                      {

                        cleanup(ERROR_BAD_DATA, Legionfile, line_count);

                      }

                    /* input the remaining fields */
                    i = 0;
                    while(Characterfields[i] != LASTCFIELD)
                      { /* input the character slot fields */

                        cfield[Characterfields[i]] = strztok(NULL, colons);
                        if(cfield[Characterfields[i]] == NULL)
                          { /* ran out of tokens */

                            cleanup(ERROR_BAD_DATA, Legionfile, line_count);

                          } /* ran out of tokens */

                        else
                          { /* convert the strings to decimal */

                            /* this will result in a value of 0 if format */
                            /* is bad, as will definitely occur with */
                            /* the description */
                            /* v2.0.0 - atoi() replaces sscanf() */
                            Legion->Slot[slt].Character[Characterfields[i]] =
                              atoi(cfield[Characterfields[i]]);
                            #ifdef DEBUG
                            printf("\n%s:cfield[%s]='%s'",
                                   Legionfile, Fieldnames[Characterfields[i]],
                                   cfield[Characterfields[i]]);
                            printf(",Slot[%d].Character[%s]=%d", slt,
                                   Fieldnames[Characterfields[i]],
                            Legion->Slot[slt].Character[Characterfields[i]]);
                            #endif

                          } /* convert the strings to decimal */

                        i++; /* proceed to next field */

                      } /* input the character slot fields */

                    strncpy(&(Legion->Slot[slt].CharacterDesc[0]),
                            strblk(cfield[DESC]),
                            DESCSIZE);
                    Legion->Slot[slt].CharacterDesc[DESCSIZE] = '\0';
                    character_count = character_count + 1;
                    all_count = all_count + 1;
                    break;

                  } /* character slot */

                case 'F': /* v2.1.0 - added this case */
                  { /* fortifications */

                    /* input the remaining fields */
                    i = 0;
                    while(Fortificationfields[i] != LASTFORTS)
                      { /* input the fortification fields */

                        ffield[Fortificationfields[i]] =
                          strztok(NULL, colons);
                        if(ffield[Fortificationfields[i]] == NULL)
                          { /* ran out of tokens */

                            cleanup(ERROR_BAD_DATA, Legionfile, line_count);

                          } /* ran out of tokens */

                        else
                          { /* convert the strings to decimal */

                            /* this will result in a value of 0 if format */
                            /* is bad, as will definitely occur with */
                            /* the description */
                            /* v2.0.0 - atoi() replaces sscanf() */
                            Legion->Forts[Fortificationfields[i]] =
                              atoi(ffield[Fortificationfields[i]]);

                          } /* convert the strings to decimal */

                        i++; /* proceed to next field */

                      } /* input the fortifications fields */

                    all_count = all_count + 1;
                    break;

                  } /* fortifications */

                case 'T': /* v4.0.0 - added this case */
                  { /* Terrain */

                    /* input the remaining fields */
                    i = 0;
                    while(Terrainfields[i] != LASTTERRAIN)
                      { /* input the terrain fields */

                        tfield[Terrainfields[i]] =
                          strztok(NULL, colons);
                        if(tfield[Terrainfields[i]] == NULL)
                          { /* ran out of tokens */

                            cleanup(ERROR_BAD_DATA, Legionfile, line_count);

                          } /* ran out of tokens */

                        else
                          { /* convert the strings to decimal */

                            /* this will result in a value of 0 if format */
                            /* is bad, as will definitely occur with */
                            /* the description */
                            /* v2.0.0 - atoi() replaces sscanf() */
                            Legion->Terrain[Terrainfields[i]] =
                              atoi(tfield[Terrainfields[i]]);

                          } /* convert the strings to decimal */

                        i++; /* proceed to next field */

                      } /* input the terrain fields */

                    all_count = all_count + 1;
                    break;

                  } /* Terrain */

                case 'R': /* v4.0.0 - added this case */
                  { /* Reserve Options */

                    /* input the remaining fields */
                    i = 0;
                    while(Reservefields[i] != LASTRESERVE)
                      { /* input the Reserve Options fields */

                        rfield[Reservefields[i]] =
                          strztok(NULL, colons);
                        if(rfield[Reservefields[i]] == NULL)
                          { /* ran out of tokens */

                            cleanup(ERROR_BAD_DATA, Legionfile, line_count);

                          } /* ran out of tokens */

                        else
                          { /* convert the strings to decimal */

                            /* this will result in a value of 0 if format */
                            /* is bad, as will definitely occur with */
                            /* the description */
                            /* v2.0.0 - atoi() replaces sscanf() */
                            Legion->Reserve[Reservefields[i]] =
                              atoi(rfield[Reservefields[i]]);

                          } /* convert the strings to decimal */

                        i++; /* proceed to next field */

                      } /* input the Reserve Options fields */

                    all_count = all_count + 1;
                    break;

                  } /* Reserve Options */

                case 'L': /* v4.0.0 - added this case */
                  { /* Acceptable Losses */

                    /* input the remaining fields */
                    i = 0;
                    while(Lossesfields[i] != LASTLOSSES)
                      { /* input the Acceptable Losses fields */

                        lfield[Lossesfields[i]] =
                          strztok(NULL, colons);
                        if(lfield[Lossesfields[i]] == NULL)
                          { /* ran out of tokens */

                            cleanup(ERROR_BAD_DATA, Legionfile, line_count);

                          } /* ran out of tokens */

                        else
                          { /* convert the strings to decimal */

                            /* this will result in a value of 0 if format */
                            /* is bad, as will definitely occur with */
                            /* the description */
                            /* v2.0.0 - atoi() replaces sscanf() */
                            Legion->Losses[Lossesfields[i]] =
                              atoi(lfield[Lossesfields[i]]);

                          } /* convert the strings to decimal */

                        i++; /* proceed to next field */

                      } /* input the Acceptable Losses fields */

                    all_count = all_count + 1;
                    break;

                  } /* Acceptable Losses */

                default:
                  { /* unknown prefix */

                    cleanup(ERROR_BAD_DATA, Legionfile, line_count);
                    break;

                  } /* unknown prefix */

              } /* end switch(prefix) */

          } /* not a comment or blank line */

      } /* read legion data */

    if(all_count == 0) cleanup(ERROR_NO_DATA, Legionfile, line_count);
    fclose(filefp);

  } /* end of load_legion() */

/*************************************************************************
load_tables:
*************************************************************************/
void load_tables(Legion)
struct legion_struct *Legion;
  {

    short i;

    for(i = 0; i < LASTTABLE; i++)
      {

        load_table(i, Legion);

      }

  } /* end of load_tables() */

/*************************************************************************
load_table:
*************************************************************************/
static void load_table(id, Legion)
short id;
struct legion_struct *Legion;
  {

    short count = 0; /* lines of data loaded from file */
    short line_count = 0; /* total lines read from file */
    short slot = 1; /* current slot */
    char *field[LASTFIELD]; /* fields of the input data */
    short item[LASTFIELD]; /* converted input data */
    short i, j; /* while/for() counter */
    char tline[LINESIZE + 1]; /* working copy of input line */

    #ifdef DEBUG
    printf("\nload_table(%s)", Tablefiles[id]);
    #endif
    line[0] = '\0';
    filefp = fopen(Tablefiles[id], "r");
    if(filefp == NULL)
      {

        cleanup(ERROR_NO_FILE, Tablefiles[id], line_count);

      }

    while((slot <= MAXSSLOTS) &&
          (fgets(line, LINESIZE + 1, filefp) != NULL))
      { /* read line of data */

        while((slot <= MAXSSLOTS) &&
              Legion->Slot[slot].SoldierItems[id] == 0 ||
              Legion->Slot[slot].Item[id][ID] != 0)
          { /* skip slots with no item of this type or already found */

            #ifdef DEBUG
            printf("\nSlot %d complete:", slot);
            printf("Slot[%d].SoldierItems[%s]=%d",
                   slot, Sfieldnames[id],
                   Legion->Slot[slot].SoldierItems[id]);
            printf(",Slot[%d].Item[%s][%s]=%d", slot,
                  Sfieldnames[id],
                  Fieldnames[ID],
                  Legion->Slot[slot].Item[id][ID]);
            #endif
            slot++; /* proceed to next slot */

          } /* skip slots with no item of this type or already found */

        if(slot <= MAXSSLOTS)
          { /* slot needs an item */

            #ifdef DEBUG
            printf("\nSearching for slot %d item...", slot);
            #endif
            line[strlen(line) - 1] = '\0'; /* remove LF */
            strcpy(tline, line);
            line_count = line_count + 1;
            if(line[0] != '#' && line[0] != '\0')
              { /* not a comment or blank line */

                i = 0;
                while(Tablefields[id][i] != LASTFIELD)
                  { /* input the fields */

                    if(i == 0)
                      { /* first field use string pointer */

                        field[Tablefields[id][i]] = strztok(tline, colons);

                      } /* first field use string pointer */

                    else
                      { /* from then on use NULL pointer */

                        field[Tablefields[id][i]] = strztok(NULL, colons);

                      } /* from then on use NULL pointer */

                    if(field[Tablefields[id][i]] == NULL)
                      { /* ran out of tokens */

                        cleanup(ERROR_BAD_DATA, Tablefiles[id], line_count);

                      } /* ran out of tokens */

                    else
                      { /* convert the strings to decimal */

                        /* v2.0.0 - atoi() replaces sscanf() */
                        item[Tablefields[id][i]] =
                          atoi(field[Tablefields[id][i]]);
                        #ifdef DEBUG
                        printf("\n%s:field[%s]='%s'",
                               Tablefiles[id], Fieldnames[Tablefields[id][i]],
                               field[Tablefields[id][i]]);
                        printf(",item[%s][%s]=%d",
                              Sfieldnames[id],
                              Fieldnames[Tablefields[id][i]],
                              item[Tablefields[id][i]]);
                        #endif

                      } /* convert the strings to decimal */

                    i++; /* proceed to next field */

                  } /* input the fields */

                /* v2.0.0 - the ID field in the data table */
                /* is not permitted to be < 1 (blank/zero/bad format) */
                if(item[ID] < 1)
                  { /* Bad ID */

                    cleanup(ERROR_BAD_DATA, Tablefiles[id], line_count);

                  } /* Bad ID */

                for(j = slot; j <= MAXSSLOTS; j++)
                  { /* find applicable slots */

                    if(Legion->Slot[j].SoldierItems[id] == item[ID])
                      { /* item applies to slot soldiers */

                        #ifdef DEBUG
                        printf("\nUsing item#%d in slot %d", item[ID], j);
                        #endif
                        if(j == slot)
                          { /* found current slot item */

                            slot++; /* proceed to next slot */

                          } /* found current slot item */

                        i = 0;
                        while(Tablefields[id][i] != LASTFIELD)
                          { /* copy data into slot item */

                            switch(Tablefields[id][i])
                              {

                                case DESC:
                                  { /* copy description */

                                    strncpy(&(Legion->Slot[j].ItemDesc[id][0]),
                                           strblk(field[DESC]), DESCSIZE);
                                    Legion->Slot[j].ItemDesc[id][DESCSIZE] = '\0';
                                    #ifdef DEBUG
                                    printf("\nCopied description:");
                                    printf("Slot[%d].ItemDesc[%s]='%s'",
                                          j,
                                          Sfieldnames[id],
                                          Legion->Slot[j].ItemDesc[id]
                                          );
                                    #endif
                                    break;

                                  } /* copy description */

                                case CLASS:
                                  { /* convert weapon class */

                                    if(strchr(field[CLASS], W_SWORD) != NULL)
                                      { /* weapon is sword */

                                        Legion->Slot[j].Item[id][CLASS] |=
                                          M_SWORD;

                                      } /* weapon is sword */

                                    if(strchr(field[CLASS], W_AXE) != NULL)
                                      { /* weapon is AXE */

                                        Legion->Slot[j].Item[id][CLASS] |=
                                          M_AXE;

                                      } /* weapon is AXE */

                                    if(strchr(field[CLASS], W_BOW) != NULL)
                                      { /* weapon is BOW */

                                        Legion->Slot[j].Item[id][CLASS] |=
                                          M_BOW;

                                      } /* weapon is BOW */

                                    if(strchr(field[CLASS], W_OTHER) != NULL)
                                      { /* weapon is OTHER */

                                        Legion->Slot[j].Item[id][CLASS] |=
                                          M_OTHER;

                                      } /* weapon is OTHER */

                                    if(strchr(field[CLASS], W_2HANDED) != NULL)
                                      { /* weapon is 2HANDED */

                                        Legion->Slot[j].Item[id][CLASS] |=
                                          M_2HANDED;

                                      } /* weapon is 2HANDED */

                                    if(strchr(field[CLASS], W_MISSILE) != NULL)
                                      { /* weapon is MISSILE */

                                        Legion->Slot[j].Item[id][CLASS] |=
                                          M_MISSILE;

                                      } /* weapon is MISSILE */

                                    if(strchr(field[CLASS], W_MOUNT) != NULL)
                                      { /* weapon is MOUNT USABLE */

                                        Legion->Slot[j].Item[id][CLASS] |=
                                          M_MOUNT;

                                      } /* weapon is MOUNT USABLE */

                                    if(strchr(field[CLASS], W_MOUNT_ONLY) != NULL)
                                      { /* weapon is USABLE ONLY WITH MOUNT */

                                        Legion->Slot[j].Item[id][CLASS] |=
                                          M_MOUNT_ONLY;

                                      } /* weapon is USABLE ONLY WITH MOUNT */

                                    #ifdef DEBUG
                                    printf("\nConverted class:");
                                    printf("Slot[%d].Item[%s][%s]='%d'",
                                          j,
                                          Sfieldnames[id],
                                          Fieldnames[CLASS],
                                          Legion->Slot[j].Item[id][CLASS]
                                          );
                                    #endif
                                    break;

                                  } /* convert weapon class */

                                case GWC:
                                  { /* copy general weapon code */

                                    strblk(field[GWC]);
                                    if(field[GWC][0] != '\0')
                                      { /* general weapon code exists */

                                        Legion->Slot[j].Item[id][GWC] =
                                          toupper(field[GWC][0]);

                                      } /* general weapon code exists */

                                    else
                                      { /* no special code */

                                        Legion->Slot[j].Item[id][GWC] = 0;

                                      } /* no special code */

                                    #ifdef DEBUG
                                    printf("\nCopied GWC:");
                                    printf("Slot[%d].Item[%s][%s]='%c'",
                                          j,
                                          Sfieldnames[id],
                                          Fieldnames[Tablefields[id][i]],
                                          Legion->Slot[j].Item[id][GWC]
                                          );
                                    #endif
                                    break;

                                  } /* copy general weapon code */

                                default:
                                  {

                                    Legion->Slot[j].Item[id][Tablefields[id][i]] =
                                      item[Tablefields[id][i]];
                                    #ifdef DEBUG
                                    printf("\nCopied field:");
                                    printf("Slot[%d].Item[%s][%s]='%d'",
                                          j,
                                          Sfieldnames[id],
                                          Fieldnames[Tablefields[id][i]],
                                          Legion->Slot[j].Item[id][Tablefields[id][i]]
                                          );
                                    #endif
                                    break;

                                  } /* end default */

                              } /* end switch() */

                            i++; /* proceed to next field */

                          } /* copy data into slot item */

                      } /* item applies to slot soldiers */

                  } /* find applicable slots */

                count = count + 1;

              } /* not a comment or blank line */

          } /* slot needs an item */

      } /* read lines of data */

    fclose(filefp);
    #ifdef DEBUG
    for(slot = 1; slot <= MAXSSLOTS; slot++)
      {

        for(i = 0; i < LASTTABLE; i++)
          {

            for(j = 0; j < LASTFIELD; j++)
              {

                printf("\nSlot[%d].Item[%s][%s]=%d",
                      slot,
                      Sfieldnames[i],
                      Fieldnames[j],
                      Legion->Slot[slot].Item[i][j]);

              }

          }

      }

    #endif

  } /* end of load_table() */

/*************************************************************************
init_tables:
*************************************************************************/
void init_tables()
  {

    /* Define input fields for tables */
    Tablefields[RACE] = Racefields;
    Tablefields[TYPE] = Typefields;
    Tablefields[STATUS] = Statusfields;
    Tablefields[WEAPON] = Weaponfields;
    Tablefields[ARMOR] = Armorfields;
    Tablefields[SHIELD] = Shieldfields;
    Tablefields[MOUNT] = Mountfields;

    /* Table file names */
    Tablefiles[RACE] = RACETABLE;
    Tablefiles[TYPE] = TYPETABLE;
    Tablefiles[STATUS] = STATUSTABLE;
    Tablefiles[WEAPON] = WEAPONTABLE;
    Tablefiles[ARMOR] = ARMORTABLE;
    Tablefiles[SHIELD] = SHIELDTABLE;
    Tablefiles[MOUNT] = MOUNTTABLE;

    Fieldnames[ID] = "ID";
    Fieldnames[CF] = "CF";
    Fieldnames[DF] = "DF";
    Fieldnames[AF] = "AF";
    Fieldnames[AFB] = "AFB";
    Fieldnames[MAF] = "MAF";
    Fieldnames[MAFB] = "MAFB";
    Fieldnames[SAFB] = "SAFB";
    Fieldnames[CAF] = "CAF";
    Fieldnames[CAFB] = "CAFB";
    Fieldnames[RAFB] = "RAFB";
    Fieldnames[WAFB] = "WAFB";
    Fieldnames[MAV] = "MAV";
    Fieldnames[SAV] = "SAV";
    Fieldnames[MAD] = "MAD";
    Fieldnames[SAD] = "SAD";
    Fieldnames[DAM] = "DAM";
    Fieldnames[INV] = "INV";
    Fieldnames[MORB] = "MORB";
    Fieldnames[OVER] = "OVER";
    Fieldnames[GWC] = "GWC";
    Fieldnames[GWB] = "GWB";
    Fieldnames[SWID] = "SWID";
    Fieldnames[SWB] = "SWB";
    Fieldnames[DESC] = "DESC";
    Fieldnames[CLASS] = "CLASS";
    Fieldnames[LASTCFIELD] = "LASTCFIELD";
    Fieldnames[TBEGIN] = "TBEGIN";
    Fieldnames[TGR] = "Grassland";
    Fieldnames[TFO] = "Forest";
    Fieldnames[THF] = "Heavy Forest";
    Fieldnames[TFH] = "Forest Hills";
    Fieldnames[TFM] = "Forest Mountains";
    Fieldnames[TBH] = "Barren Hills";
    Fieldnames[TBM] = "Barren Mountains";
    Fieldnames[TMO] = "Moors";
    Fieldnames[TSW] = "Swamp";
    Fieldnames[TDE] = "Desert";
    Fieldnames[TSE] = "Sea";
    Fieldnames[TDW] = "Defending Walls";
    Fieldnames[TAW] = "Attacking Walls";
    Fieldnames[TCI] = "City";
    Fieldnames[TAI] = "Air";
    Fieldnames[TCO] = "Confined";
    Fieldnames[TEND] = "TEND";

    Sfieldnames[RACE] = "RACE";
    Sfieldnames[TYPE] = "TYPE";
    Sfieldnames[STATUS] = "STATUS";
    Sfieldnames[WEAPON] = "WEAPON";
    Sfieldnames[ARMOR] = "ARMOR";
    Sfieldnames[SHIELD] = "SHIELD";
    Sfieldnames[MOUNT] = "MOUNT";
    Sfieldnames[LASTTABLE] = "LASTTABLE";
    Sfieldnames[AMT] = "AMT";
    Sfieldnames[TRL] = "TRL";
    Sfieldnames[MOR] = "MOR";

    Reservenames[RO_FLANK_LEFT] = "Flank Left";
    Reservenames[RO_FLANK_RIGHT] = "Flank Right";
    Reservenames[RO_SUPPORT_LEFT_FLANK] = "Support Missile Fire Left Flank";
    Reservenames[RO_SUPPORT_LEFT_CENTER] = "Support Missile Fire Left Center";
    Reservenames[RO_SUPPORT_CENTER] = "Support Missile Fire Center";
    Reservenames[RO_SUPPORT_RIGHT_CENTER] = "Support Missile Fire Right Center";
    Reservenames[RO_SUPPORT_RIGHT_FLANK] = "Support Missile Fire Right Flank";
    Reservenames[RO_TACTICAL_RESERVE] = "Tactical Reserve";
    Reservenames[RO_CHARGE_LEFT] = "Charge Left";
    Reservenames[RO_CHARGE_CENTER] = "Charge Center";
    Reservenames[RO_CHARGE_RIGHT] = "Charge Right";
    Reservenames[RO_COUNTERCHARGE] = "Countercharge";
    Reservenames[RO_OPPORTUNITY_CHARGE] = "Opportunity Charge";
    Reservenames[RO_STRATEGIC_RESERVE] = "Strategic Reserve";

  } /* end init_tables() */
