/*************************************************************************
Legends Soldier Calculator
Copyright (c) 1994 by Mark R. Nuiver.  All rights reserved.

Module Name      => lsccalc.c
Version          => 4.0.0
Change Date      => 94/02/22
Programmer       => M.R. Nuiver
Operating System => AmigaDOS 3.00                  MS-DOS 6.0
Compiler         => SAS/C C 5.10                   Power C 2.0.0
Modules Used     => None
  (Amiga Only)   => c.o
Header Files     => lsc.h
Libraries Used   => lc.lib, lcm.lib amiga2.0.lib   PCLIB.MIX, PCAUTO.MIX
Compiler Options => None
Data Files       => None

Pre-processor Switches

ROT = Reserve Option Terrain: When undefined, soldiers in reserve slots that
      do not get the fortifications bonuses, e.g. Charge option, do not
      use the "Defending Walls" terrain modifier.  When defined, all
      soldiers in a fortified location use the "Defending Walls" terrain
      modifier, regardless of reserve slot options.

Description

  This module provides the functions to perform the calculations for a
legion.

Revision Notes

Version Date      Programmer      Reason
------- --------  --------------  ---------------------------------------
4.0.0   94/02/22  Mark Nuiver     * Reversed meaning of ROT; by default the
                                    Walls terrain modifier will not be used
                                    on reserve slots that are charging.  I
                                    am confused on this issue, and Edi's
                                    comments in SR issue 8 are the reverse
                                    of what he told me directly.
4.0.0   94/01/20  Mark Nuiver     * Added pre-processor symbol ROT (for
                                    "Reserve Option Terrain" for conditional
                                    compilation of code that keeps the
                                    Defending Walls terrain modifier from
                                    being used on reserve slots that do not
                                    get the fortifications bonuses.  Based
                                    on a conversation with Edi Birsan on
                                    this date, ALL soldiers in a fortified
                                    location use the Defending Walls terrain
                                    modifier, regardless of reserve options.
4.0.0   94/01/17  Mark Nuiver     * Created compute_minmax_afb() and
                                    compute_minmax_mafb().
                                  * Broke compute_slot_overall() into
                                    several separate functions:
                                      compute_forts_overall()
                                      compute_terrain_overall()
                                      compute_character_overall()
                                  * Created compute_raw_soldier_totals(),
                                    validate_soldier_totals(),
                                    compute_soldier_attack(), and broke
                                    compute_legion() into smaller pieces.
                                  * Broke compute_item() up and created
                                    sum_soldier_items().
                                  * Made most of compute functions global.
4.0.0   94/01/16  Mark Nuiver     * Changed maximum soldier AFB/MAFB to
                                    400%/500%.  Added minimum -80%.
4.0.0   94/01/08  Mark Nuiver     * Added GWC "Other".
4.0.0   93/12/27  Mark Nuiver     * Modified decode_terrain() to return 0
                                    on invalid.
                                  * Modified use of terrain symbols for
                                    revised lsc.h definitions.
                                  * Modified use of fort symbols for revised
                                    lsc.h definitions.
                                  * Removed compute_character_totals().
                                  * Added some more soldier totals min/max
                                    into compute_totals_per_soldier(), CF,
                                    AF, CAF, MAV, SAV cannot be < 0.
                                  * Changed compute_forts() to consider
                                    reserve slot option, whether slot
                                    receives fortification bonuses or not.
                                  * Added use_forts().
                                  * Changed compute_base_df() to consider
                                    reserve slot options, whether slot
                                    receives fort bonuses or not.
                                  * Modified compute_item() to consider
                                    reserve slot options with Defending Walls
                                    terrain.
                                  * Modified compute_terrain_overall() to
                                    consider reserve slot options with
                                    Defending Walls terrain.
4.0.0   93/12/22  Mark Nuiver     * Removed unnecessary code from
                                    compute_legion() (if logic).
                                  * Added compute_base_df().
                                  * Made most of functions local, due to
                                    changes in lsc.c.
                                  * Added decode_terrain().
                                  * Added terrain AF bonuses to
                                    compute_item().
                                  * Created compute_terrain_overall().
                                  * Changed compute_slot_overall() to include
                                    terrain overall calculations.
                                  * Racial mounted bonus was overwriting new
                                    terrain AF bonuses on mounts - fixed it.
                                  * Capped soldier INV at 3.
4.0.0   93/12/21  Mark Nuiver     * Changed version.
                                  * Changed Character[] and Forts[] to use
                                    the new OVER symbol instead of the
                                    misleading re-use of CAFB.
                                  * Removed SPC and changed code to use
                                    GWC, GWB, SWID, SWB for weapon bonuses.
                                  * Removed compute_special_type().
                                  * Removed compute_special_status().
                                  * Removed unused (commented out) code
                                    from compute_legion().
------- --------  --------------  ---------------------------------------
3.0.1   93/12/19  Mark Nuiver     * Changed version.
                                  * Changed all int to short, changed Forts
                                    to short.
------- --------  --------------  ---------------------------------------
3.0.0   93/12/12  Mark Nuiver     * Added DAM and INV calculations.
3.0.0   93/12/11  Mark Nuiver     * Created this module from original lsc.c
                                    for re-use with LBS program.
*************************************************************************/

#include "lsc.h"

/* global functions */
enum lsc_symbols decode_terrain(short);
short use_forts(struct legion_struct *, short);
long compute_minmax_afb(long);
long compute_minmax_mafb(long);
void compute_trl(struct slot_struct *, long *);
void compute_forts(struct legion_struct *, short, long *);
void compute_magic(struct slot_struct *, long *);
void compute_character_overall(struct legion_struct *, short, short, long *);
void compute_forts_overall(struct legion_struct *, short, long *);
void compute_terrain_overall(struct legion_struct *, short, long *);
void compute_totals_per_soldier(struct legion_struct *, short, long *);
void compute_legion(struct legion_struct *);

/* local functions */
static long compute_overall_df(long, long);
static long compute_base_df(struct legion_struct *, short);
static long compute_trl_cf(struct slot_struct *);
static long compute_trl_df(struct slot_struct *);
static void compute_items(struct legion_struct *, short);
static void sum_soldier_items(struct legion_struct *, short, long *);
static void compute_raw_totals_per_soldier(struct legion_struct *, short, long *);
static void validate_soldier_totals(long *);
static void compute_soldier_attack(struct slot_struct *, long *);
static void compute_soldier_totals(struct slot_struct *, long *);
static void compute_grand_totals(long *, struct slot_struct *);

/*************************************************************************
compute_minmax_afb:
  returns the AFB adjusted to minimum or maximum value
*************************************************************************/
long compute_minmax_afb(afb)
long afb;
  {

    if(afb > MAXAFB) afb = MAXAFB;
    if(afb < MINAFB) afb = MINAFB;

    return(afb);

  } /* end of compute_minmax_afb() */

/*************************************************************************
compute_minmax_mafb:
  returns the MAFB adjusted to minimum or maximum value
*************************************************************************/
long compute_minmax_mafb(mafb)
long mafb;
  {

    if(mafb > MAXMAFB) mafb = MAXMAFB;
    if(mafb < MINMAFB) mafb = MINMAFB;

    return(mafb);

  } /* end of compute_minmax_mafb() */

/*************************************************************************
decode_terrain:
  translates the terrain code in the LDF to lsc_symbol for same
  returns 0 if the terrain code is out of range.
*************************************************************************/
enum lsc_symbols decode_terrain(ter)
short ter;
  {

    if(ter > 0 && ter < TEND - TBEGIN)
      { /* in range */

        return((enum lsc_symbols)(ter + TBEGIN));

      } /* in range */

    return(0);

  } /* end of decode_terrain() */

/*************************************************************************
compute_trl_cf:
  returns training level CF bonus for given slot
*************************************************************************/
static long compute_trl_cf(Slot)
struct slot_struct *Slot;
  {
#ifdef DEBUG
    printf("\ncompute_trl_af():");
#endif

    return(Slot->SoldierItems[TRL] * 3);

  } /* end of compute_trl_cf() */

/*************************************************************************
compute_trl_df:
  returns training level DF bonus for given slot
*************************************************************************/
static long compute_trl_df(Slot)
struct slot_struct *Slot;
  {
#ifdef DEBUG
    printf("\ncompute_trl_df():");
#endif

    return((long)(Slot->SoldierItems[TRL] * .25));

  } /* end of compute_trl_df() */

/*************************************************************************
compute_trl:
  calculates the training level bonuses for given slot.
*************************************************************************/
void compute_trl(Slot, Total)
struct slot_struct *Slot;
long Total[];
  {
#ifdef DEBUG
    printf("\ncompute_trl():");
#endif

    /* add the training level bonuses into slot totals */
    Total[CF] += compute_trl_cf(Slot);
    Total[DF] += compute_trl_df(Slot);

  } /* end of compute_trl() */

/*************************************************************************
use_forts:
  returns TRUE if forts bonuses apply to given slot, else FALSE.
*************************************************************************/
short use_forts(Legion, slot)
struct legion_struct *Legion;
short slot;
  {

    short forts = TRUE;

    if(slot == 6 || slot == 7 || slot == 8)
      { /* reserve slots */

        if((1 << (Legion->Reserve[slot - 6] - 1)) & M_R_NOFORTS)
          forts = FALSE;

      } /* reserve slots */

    return(forts);

  } /* end of use_forts() */

/*************************************************************************
compute_forts:
  calculates the base fortifications bonuses for given slot.
*************************************************************************/
void compute_forts(Legion, slot, Total)
struct legion_struct *Legion;
short slot;
long Total[];
  {
#ifdef DEBUG
    printf("\ncompute_forts():");
#endif

    if(use_forts(Legion, slot))
      { /* slot gets base forts bonuses */

        /* add the fortifications bonuses into slot totals */
        Total[DF] += Legion->Forts[F_DF];
        Total[AFB] += Legion->Forts[F_AFB];
        Total[MAFB] += Legion->Forts[F_AFB];
        Total[MORB] += Legion->Forts[F_MORB];

      } /* slot gets base forts bonuses */

  } /* end of compute_forts() */

/*************************************************************************
compute_magic:
  adds character magic AFB into slot totals.
*************************************************************************/
void compute_magic(Slot, Total)
struct slot_struct *Slot;
long Total[];
  {
#ifdef DEBUG
    printf("\ncompute_magic():");
#endif

    /* add character magic effects to slot soldier totals */
    Total[AFB] += Slot->Character[AFB];
    Total[MAFB] += Slot->Character[AFB];

  } /* end of compute_magic() */

/*************************************************************************
compute_items:
    calculates all the numbers for the soldier items, e.g. race, training,
  weapon, armor, etc.
NOTE: this modifies the item values in the Legion structure, so should
never be called more than once for a given legion.
*************************************************************************/
static void compute_items(Legion, slot)
struct legion_struct *Legion;
short slot; /* current slot */
  {

    struct slot_struct *Slot = &(Legion->Slot[slot]);
    short *Terrain = &(Legion->Terrain[0]);
    short i; /* for() counter */
    short table_id; /* item type */
    int mask; /* weapon class mask */
    enum lsc_symbols ter;

#ifdef DEBUG
    printf("\ncompute_items():");
#endif

    for(table_id = 0; table_id < LASTTABLE; table_id++)
      { /* all tables */

        if(Slot->Item[table_id][ID] != Slot->SoldierItems[table_id])
          { /* unknown item */

            /* do nothing */
            return;

          } /* unknown item */

        else
          { /* item in table or no item */

            if(Slot->Item[table_id][ID] != 0)
              { /* slot has this item */

                /* 4.0.0 Terrain AF bonuses */
                for(i= 0; i < LASTTERRAIN; i++)
                  { /* all given terrains */

#ifndef ROT
                    /* when terrain is Defending Walls, only use the terrain */
                    /* modifier when appropriate on the reserve slots */
                    if(Terrain[i] != (TDW - TBEGIN) || use_forts(Legion, slot))
                      { /* not Defending Walls or slot uses DW modifier */
#endif

                        ter = decode_terrain(Terrain[i]);

                        if(ter != 0)
                          { /* legal terrain code */

                            Slot->Item[table_id][AFB] +=
                              Slot->Item[table_id][ter];

                            /* add terrain AF bonus to the MAFB also, unless */
                            /* this is weapon, and it is not a missile weapon */
                            if(table_id != WEAPON || (table_id == WEAPON &&
                               (Slot->Item[table_id][CLASS] & M_MISSILE)))
                              { /* item isn't weapon, or not a missile weapon */

                                Slot->Item[table_id][MAFB] +=
                                  Slot->Item[table_id][ter];

                              } /* item isn't weapon, or not a missile weapon */

                          } /* legal terrain code */

#ifndef ROT
                      } /* not Defending Walls or slot uses DW modifier */
#endif

                  } /* all given terrains */

                switch(table_id)
                  { /* special considerations */

                    case TYPE:
                    case STATUS:
                    case RACE:
                      { /* general and specific weapon bonuses */

                        if(Slot->Item[table_id][GWC] != '\0' &&
                           Slot->Item[table_id][GWB] != 0)
                          { /* has a general weapon class bonus */

                            switch(Slot->Item[table_id][GWC])
                              { /* switch() */

                                case W_SWORD:
                                  { /* general weapon class sword */

                                    mask = M_SWORD;
                                    break;

                                  } /* general weapon class sword */

                                case W_AXE:
                                  { /* general weapon class axe */

                                    mask = M_AXE;
                                    break;

                                  } /* general weapon class axe */

                                case W_BOW:
                                  { /* general weapon class bow */

                                    mask = M_BOW;
                                    break;

                                  } /* general weapon class bow */

                                case W_MISSILE:
                                  { /* general weapon class missile */

                                    mask = M_MISSILE;
                                    break;

                                  } /* general weapon class missile */

                                case W_OTHER:
                                  { /* general weapon class other */

                                    mask = M_OTHER;
                                    break;

                                  } /* general weapon class other */

                                default:
                                  { /* unknown general weapon class */

                                    mask = 0;
                                    break;

                                  } /* unknown general weapon class */

                              } /* switch() */

                            if(Slot->Item[WEAPON][CLASS] & mask)
                              { /* weapon in use is in general weapon class */

                                Slot->Item[table_id][AFB] +=
                                  Slot->Item[table_id][GWB];

                                /* receives MAFB too if missile weapon */
                                if(Slot->Item[WEAPON][CLASS] & M_MISSILE)
                                  { /* weapon in use is missile weapon */

                                    Slot->Item[table_id][MAFB] +=
                                      Slot->Item[table_id][GWB];

                                  } /* weapon in use is missile weapon */

                              } /* weapon in use is in general weapon class */

                          } /* has a general weapon class bonus */

                        if(Slot->Item[table_id][SWID] != 0 &&
                           Slot->Item[table_id][SWB] != 0)
                          { /* has specific weapon bonus */

                            /* the specific weapon ID can match either the */
                            /* weapon ID or the Base weapon ID */
                            if((Slot->Item[table_id][SWID] ==
                                Slot->Item[WEAPON][ID]) ||
                               (Slot->Item[table_id][SWID] ==
                                Slot->Item[WEAPON][BWID]))
                              { /* weapon in use gets specific weapon bonus */

                                Slot->Item[table_id][AFB] +=
                                  Slot->Item[table_id][SWB];

                                /* receives MAFB too if missile weapon */
                                if(Slot->Item[WEAPON][CLASS] & M_MISSILE)
                                  { /* weapon in use is missile weapon */

                                    Slot->Item[table_id][MAFB] +=
                                      Slot->Item[table_id][SWB];

                                  } /* weapon in use is missile weapon */

                              } /* weapon in use gets specific weapon bonus */

                          } /* has specific weapon bonus */

                        break;

                      } /* general and specific weapon bonuses */

                    case MOUNT:
                      { /* mount */

                        /* 4.0.0 FIX - was =, made += */
                        /* receives racial WAFB as AFB */
                        Slot->Item[MOUNT][AFB] += Slot->Item[RACE][WAFB];
                        Slot->Item[MOUNT][MAFB] += Slot->Item[RACE][WAFB];
                        break;

                      } /* mount */

                    default:
                      { /* default */

                        break;

                      } /* default */

                  } /* special considerations */

              } /* slot has this item */

          } /* item in table or no item */

      } /* all tables */

  } /* end of compute_items() */

/*************************************************************************
sum_soldier_items:
  adds up all the soldier items
*************************************************************************/
static void sum_soldier_items(Legion, slot, Total)
struct legion_struct *Legion;
short slot; /* current slot */
long Total[];
  {

    struct slot_struct *Slot = &(Legion->Slot[slot]);
    short table_id;
    short j; /* counter */

    for(table_id = 0; table_id < LASTTABLE; table_id++)
      { /* all tables */

        for(j = 0; j < LASTCFIELD; j++)
          { /* add this item to the slot soldier totals */

            /* 2.1.0 DAM and INV are not additive, everything else is */
            if(j == DAM || j == INV)
              { /* DAM or INV rating of item */

                if(Slot->Item[table_id][j] > Total[j])
                  { /* item DAM or INV best so far */

                    Total[j] = Slot->Item[table_id][j];

                  } /* item DAM or INV best so far */

              } /* DAM or INV rating of item */
            else
              { /* not DAM or INV */

                Total[j] += Slot->Item[table_id][j];

              } /* not DAM or INV */

          } /* add this item to the slot soldier totals */

      } /* all tables */

  } /* end of sum_soldier_items() */

/*************************************************************************
compute_overall_df:
  calculates the DF bonus to slot for given overall modifier using current
total DF.  Overall modifier must always be applied to the Base DF only.
*************************************************************************/
static long compute_overall_df(base_df, overall)
long base_df;
long overall;
  {
#ifdef DEBUG
    printf("\ncompute_overall_df():");
#endif

    return((long)(base_df * (overall / 100.0)));

  } /* end of compute_overall_df() */

/*************************************************************************
compute_base_df:
  calculates base df by adding
    training level DF
    items DF
    forts DF
*************************************************************************/
static long compute_base_df(Legion, slot)
struct legion_struct *Legion;
short slot;
  {

    short i;
    long base_df = 0;

    base_df += compute_trl_df(&(Legion->Slot[slot]));

    for(i = 0; i < LASTTABLE; i++)
      { /* each item */

        base_df += Legion->Slot[slot].Item[i][DF];

      } /* each item */

    if(use_forts(Legion, slot))
      { /* forts apply to slot */

        base_df += Legion->Forts[F_DF];

      } /* forts apply to slot */

    return(base_df);

  } /* end of compute_base_df() */

/*************************************************************************
compute_terrain_overall:
  computes the sum of the slot item Terrain Modifiers, across the specified
terrains, with the exception of the Weapon Terrain Modifier.  According to
the information that I have, the Weapon Terrain Modifier is AF bonuses only.
All others are Overall Modifiers.
*************************************************************************/
void compute_terrain_overall(Legion, slot, Total)
struct legion_struct *Legion;
short slot;
long Total[]; /* put results here */
  {

    struct slot_struct *Slot = &(Legion->Slot[slot]);
    short *Terrain = &(Legion->Terrain[0]);
    long base_df = compute_base_df(Legion, slot);
    enum lsc_symbols ter;  /* terrain symbol */
    long ter_over = 0;     /* sum of terrain overall bonuses */
    short table_id;        /* counter */
    short i;               /* counter */

    /* Terrain Overall Modifier */
    for(i = 0; i < LASTTERRAIN; i++)
      { /* all specified terrains */

#ifndef ROT
        if(Terrain[i] != (TDW - TBEGIN) || use_forts(Legion, slot))
          { /* not Defending Walls or reserve slot uses DW modifier */
#endif

            ter = decode_terrain(Terrain[i]);

            if(ter != 0)
              { /* valid terrain code */

                for(table_id = 0; table_id < LASTTABLE; table_id++)
                  { /* all slot items */

                    /* NOTE: According to the information I have, */
                    /* the Weapon Terrain Modifier is ONLY an AF bonus */
                    /* All of the other Item Terrain Modifiers are overall */
                    if(table_id != WEAPON)
                      { /* not weapon item */

                        ter_over += Slot->Item[table_id][ter];

                      } /* not weapon item */

                  } /* all slot items */

              } /* valid terrain code */

#ifndef ROT
          } /* not Defending Walls or reserve slot uses DW modifier */
#endif

      } /* all specified terrains */

    /* Terrain Overall Modifier */
    Total[DF] += compute_overall_df(base_df, ter_over);

  } /* end of compute_terrain_overall() */

/*************************************************************************
compute_character_overall:
  calculates overall bonus to slot from:
    character in slot casting a spell
        OR
    force leader tactics overall modifier
*************************************************************************/
void compute_character_overall(Legion, soldier_slot, character_slot, Total)
struct legion_struct *Legion;
short soldier_slot;
short character_slot;
long Total[];  /* add results here */
  {

    long base_df = compute_base_df(Legion, soldier_slot);

    /* add slot leader overall bonuses to slot soldiers total */
    Total[DF] += compute_overall_df(base_df,
                   Legion->Slot[character_slot].Character[OVER]);
    Total[AFB] += Legion->Slot[character_slot].Character[OVER];
    Total[MAFB] += Legion->Slot[character_slot].Character[OVER];

  } /* end of compute_character_overall() */

/*************************************************************************
compute_forts_overall:
  calculates Tactics overall modifiers to slot from:
    fortifications tactical bonus
*************************************************************************/
void compute_forts_overall(Legion, slot, Total)
struct legion_struct *Legion;
short slot;
long Total[];  /* add results here */
  {

    long base_df = compute_base_df(Legion, slot);

#ifdef DEBUG
    printf("\ncompute_slot_overall():");
#endif

    /* add fortifications overall bonuses to slot soldier total */
    Total[DF] += compute_overall_df(base_df, Legion->Forts[F_OVER]);
    Total[AFB] += Legion->Forts[F_OVER];
    Total[MAFB] += Legion->Forts[F_OVER];

  } /* end of compute_slot_overall() */

/*************************************************************************
validate_soldier_totals:
  validates the soldier totals
*************************************************************************/
static void validate_soldier_totals(Total)
long Total[];
  {

    /* test for minimum values */
    if(Total[CF] < 0) Total[CF] = 0;
    if(Total[DF] < 1) Total[DF] = 1;
    if(Total[MAV] < 0) Total[MAV] = 0;
    if(Total[SAV] < 0) Total[SAV] = 0;
    if(Total[MAD] < 1) Total[MAD] = 1;
    if(Total[SAD] < 1) Total[SAD] = 1;
    if(Total[DAM] < 0) Total[DAM] = 0;
    if(Total[INV] < 0) Total[INV] = 0;

    Total[AFB] = compute_minmax_afb(Total[AFB]);
    Total[MAFB] = compute_minmax_mafb(Total[MAFB]);

    /* 4.0.0 per an old "Notes from the Home Office", soldier INV can't */
    /* exceed 3 */
    if(Total[INV] > 3) Total[INV] = 3;

  } /* end of validate_soldier_totals() */

/*************************************************************************
compute_soldier_attack:
  calculates the total AF from bonuses, for soldiers only
*************************************************************************/
static void compute_soldier_attack(Slot, Total)
struct slot_struct *Slot;
long Total[];
  {

    /* compute the various AF from bonuses */
    Total[AF] = (Total[CF] * (1.0 + (Total[AFB] / 100.0)));
    Total[CAF] = (Total[CF] * (1.0 + ((Total[AFB] + Total[CAFB]) / 100.0)));
    if(Slot->Item[WEAPON][CLASS] & M_MISSILE)
      { /* soldiers have missile weapon */

        Total[MAF] = (Total[CF] * (1.0 + (Total[MAFB] / 100.0)));

      } /* soldiers have missile weapon */

    /* AF can't go negative */
    if(Total[AF] < 0) Total[AF] = 0;
    if(Total[MAF] < 0) Total[MAF] = 0;
    if(Total[CAF] < 0) Total[CAF] = 0;

    } /* end of compute_soldier_attack() */

/*************************************************************************
compute_raw_totals_per_soldier:
  performs all of the raw calculations for a slot of soldiers, on a per
soldier basis, but does not validate the results.  Also does not calculate
the attack factors, or add in the magic bonuses.
*************************************************************************/
static void compute_raw_totals_per_soldier(Legion, slot, Total)
struct legion_struct *Legion;
short slot;
long Total[]; /* put results here */
  {

    compute_trl(&(Legion->Slot[slot]), Total);
    sum_soldier_items(Legion, slot, Total);
    compute_forts(Legion, slot, Total);
    compute_forts_overall(Legion, slot, Total);
    compute_character_overall(Legion, slot, MAXCSLOTS, Total);
    compute_terrain_overall(Legion, slot, Total);

  } /* end of compute_raw_totals_per_soldier() */

/*************************************************************************
compute_totals_per_soldier:
  computes the raw numbers for a slot on a per soldier basis, then
validates the results and calculates the attack factors.
*************************************************************************/
void compute_totals_per_soldier(Legion, slot, Total)
struct legion_struct *Legion;
short slot;
long Total[]; /* put results here */
  {

    compute_raw_totals_per_soldier(Legion, slot, Total);
    validate_soldier_totals(Total);
    compute_soldier_attack(&(Legion->Slot[slot]), Total);

  } /* end of compute_totals_per_soldier() */

/*************************************************************************
compute_soldier_totals:
  calculates the total AF from bonuses, for soldiers only, all soldiers.
*************************************************************************/
static void compute_soldier_totals(Slot, Total)
struct slot_struct *Slot;
long Total[];
  {
#ifdef DEBUG
    printf("\ncompute_soldier_totals():");
#endif

    /* compute slot totals for all soldiers */
    Total[CF] *= Slot->SoldierItems[AMT];
    Total[AF] *= Slot->SoldierItems[AMT];
    Total[MAF] *= Slot->SoldierItems[AMT];
    Total[CAF] *= Slot->SoldierItems[AMT];
    Total[MAV] *= Slot->SoldierItems[AMT];
    Total[SAV] *= Slot->SoldierItems[AMT];

  } /* end of compute_soldier_totals() */

/*************************************************************************
compute_grand_totals:
    adds up all the slot totals.
*************************************************************************/
static void compute_grand_totals(Total, Slot)
long Total[];
struct slot_struct Slot[];
  {

    short i;

#ifdef DEBUG
    printf("\ncompute_grand_totals():");
#endif

    for(i = 1; i <= MAXSSLOTS; i++)
      { /* add up all of the slots */

        Total[CF] += Slot[i].Total[CF];
        Total[AF] += Slot[i].Total[AF];
        Total[MAF] += Slot[i].Total[MAF];
        Total[CAF] += Slot[i].Total[CAF];
        Total[MAV] += Slot[i].Total[MAV];
        Total[SAV] += Slot[i].Total[SAV];

      } /* add up all of the slots */

  } /* end of compute_grand_totals() */

/*************************************************************************
compute_legion:
  performs all the calculations on a Legion -- SOLDIER TOTALS ONLY!  The
grand total does not include the slot leader's CF, AF, MAF, CAF, MAV, or
SAV, although it does include the effects of magic on AFB and overall
modifier, and the overall modifier from the force leader.
  totals go into the Total[] array for each slot.
  grand totals go into the Total[] array for slot 0.
*************************************************************************/
void compute_legion(Legion)
struct legion_struct *Legion;
  {
    short slot;

#ifdef DEBUG
    printf("\ncompute_legion():");
#endif

    for(slot = 1; slot <= MAXSSLOTS; slot++)
      { /* do all soldier slots */

        /* adjust all the soldier items based on terrain etc. */
        compute_items(Legion, slot);

        /* calculate the numbers on a per soldier basis */
        compute_totals_per_soldier(Legion, slot,
                                   &(Legion->Slot[slot].Total[0]));
        /* NOTE: the order here assumes that magic bonuses can push */
        /* the soldier bonuses beyond the normally permissible limits. */
        /* add the magic bonuses */
        compute_magic(&(Legion->Slot[slot]), &(Legion->Slot[slot].Total[0]));
        compute_character_overall(Legion, slot, slot,
                             &(Legion->Slot[slot].Total[0]));
        /* recompute the total Attack Factors, with magic */
        compute_soldier_attack(&(Legion->Slot[slot]),
                               &(Legion->Slot[slot].Total[0]));
        /* compute the slot totals for all soldiers */
        compute_soldier_totals(&(Legion->Slot[slot]),
                               &(Legion->Slot[slot].Total[0]));

      } /* do all soldier slots */

    compute_grand_totals(&(Legion->Slot[0].Total[0]),
                         &(Legion->Slot[0]));

  } /* end of compute_legion() */
