/************************************************************************
Legends Battle Simulator
Copyright (C) 1994 by Mark R. Nuiver.  All rights reserved.

Module Name      => lbs.c
Version          => 2.0.0
Change Date      => 94/03/10
Programmer       => M.R. Nuiver
Operating System => AmigaDOS 3.00                  MS-DOS 6.0
Compiler         => SAS/C C 5.10                   Power C 2.0.0
Modules Used     => lscdata lsccalc
  (Amiga Only)   => c.o
Header Files     => lbs.h lsc.h
Libraries Used   => lcm.lib, lc.lib amiga2.0.lib   PCLIB.MIX, PCAUTO.MIX
Compiler Options => None
Data Files       => See lscdata.c
Documentation    => LBS.DOC LSC.DOC

Pre-processor Switches

NOCHARGE = No Charge Attack Factor: when defined, uses the Melee Attack
           Factor in the Charge Round, instead of the Charge Attack Factor.
           When undefined, uses the Charge Attack Factor in the Charge Round.

Description

  This purpose of this program is to simulate battles between legions in
the play-by-mail fantasy game Legends.  This is extremely tedious and prone
to error if done manually.  The program will permit the player to test a
battle before committing his legions and will make it easy to experiment
with different attack/defense formations in order to optimize the results.

  LBS is built on top of the Legends Soldier Calculator (LSC) program and
uses the same data files, legion description files, and the source code
that does all of the calculations.  As a result, LBS shares all of LSC's
limitations.  In addition, LBS has its own limitations.  Specifically,
there is no randomness in LBS's battles, thus no surprise missile rounds,
no routs, flanking maneuvers always fail, etc.  Thus, while LBS can be a
useful tool to get a feel for a battle, it can NEVER predict the exact
outcome of an actual battle.

Credits

  Legends is a play-by-mail fantasy wargame system designed by Jim Landes
and moderated commercially by Midnight Games.  Legends features aspects
of both power wargaming and role-playing, with cities, armies, civilians,
player characters, monsters, gods, and magic all interacting in a
complex, detailed, and often mysterious manner.

  Legends is a trademark of Midnight Games.

  If it hadn't been for Keith B. Verret a.k.a. Dra'Kon'Kyl the Holdbane
and a battle he was planning the first version of this program would never
have been developed.

  David Wrobel, Dragonlord and slayer of Mockers, provided many helpful
insights.

Revision Notes

Version Date      Programmer      Reason
------- --------  --------------  ---------------------------------------
2.0.0   94/03/10  Mark Nuiver     * Made the character rout threshold the
                                    same as Acceptable Losses rather than
                                    a fixed 50%.
2.0.0   94/02/25  Mark Nuiver     * Changed some %d to %ld for DOS compiler.
2.0.0   94/02/13  Mark Nuiver     * Neatened the legion summary.
2.0.0   94/02/02  Mark Nuiver     * More search pattern changes ...
2.0.0   94/01/25  Mark Nuiver     * Changed search pattern for reserve slots
                                    based on one of Wrobel's big battles.
2.0.0   94/01/20  Mark Nuiver     * Added NOCHARGE pre-processor switch,
                                    to conveniently switch between versions
                                    of the program "the way it should work"
                                    and "the way Legends battles are working"
                                    because the Legends battles do not seem
                                    to be using the Charge Attack Factor.
2.0.0   94/01/18  Mark Nuiver     * Changed search pattern based on some
                                    recent battle reports.
2.0.0   94/01/10  Mark Nuiver     * Created flank().
2.0.0   94/01/09  Mark Nuiver     * Created attack().
2.0.0   94/01/08  Mark Nuiver     * Created is_soldiers() and is_character().
                                  * Separated check_reserve() into separate
                                    functions: tactical_reserve(),
                                    opportunity_charge(), and
                                    countercharge().
                                  * Created support_fire().
2.0.0   94/01/03  Mark Nuiver     * Created compute_inv_factor().
2.0.0   94/01/02  Mark Nuiver     * Continued modifying fight().
                                  * Created copy_previous().
                                  * Created compute_legion_fraction().
                                  * Renamed destroyed() to defeated() and
                                    significant modifications.
                                  * Created compute_soldiers_remaining().
                                  * Created compute_damage_fraction().
                                  * Modified SearchBP[] for 14 position
                                    battlefield search.
                                  * Modified find_targets() for 14 position
                                    battlefield.
                                  * Modified print routines.
                                  * Created print_round_preliminary().
                                  * Created compute_soldier_fraction().
                                  * Created is_fighting_soldiers().
                                  * Created compute_soldiers_fighting().
                                  * Created compute_character_fraction().
                                  * Created compute_fraction_fighting().
                                  * Created reserve_charge().
                                  * Created check_reserve().
                                  * Created find_opposing_bp().
2.0.0   93/12/28  Mark Nuiver     * Added calls to print_reserve() and
                                    print_losses().
                                  * Created battle_terrain().
                                  * Massive changes associated with switching
                                    to 14 battleposition setup, using reserve
                                    options, etc.
2.0.0   93/12/23  Mark Nuiver     * Made help message appear when only one
                                    argument (unless it's "?").
2.0.0   93/12/22  Mark Nuiver     * Made print_summary() part of normal
                                    operation.  Added print_terrain() calls.
2.0.0   93/12/21  Mark Nuiver     * Changed version.
------- --------  --------------  ---------------------------------------
1.0.1   93/12/19  Mark Nuiver     * Changed version.
                                  * Changed int to short.
                                  * Changed float to double.
                                  * Made the print_round_finals() and
                                    print_remaining() "surviving soldiers"
                                    calculations identical.
                                  * Fixed battle routine logic, removed a
                                    divide-by-zero that seemd to create
                                    a massive "ghost" slot under the certain
                                    conditions.
************************************************************************/

#include "lbs.h"

/* external functions */
extern void cleanup(enum lsc_errors, char *, short);
extern void load_legion(char *, struct legion_struct *);
extern void load_tables(struct legion_struct *);
extern void load_table(short, struct legion_struct *);
extern void init_tables(void);
extern void compute_legion(struct legion_struct *);
extern void print_terrain(short *);
extern void print_reserve(short *);
extern void print_losses(short *);
extern void validate_legion(struct legion_struct *);
extern short is_forts(struct legion_struct *);
extern short is_reserve_slot(short);

/* local function prototypes */
static void help(void);
static short is_soldiers(short, short, short);
static short is_character(short, short, short);
static short is_fighting_soldiers(short, short, short);
static void find_targets(short, short, short, short *, short *);
static short find_opposing_bp(short, short, short);
static double compute_inv_factor(long inv, long dam);
static double compute_character_fraction(short, short, short);
static double compute_damage_fraction(short, short, short);
static double compute_soldier_fraction(short, short, short);
static long compute_soldiers_remaining(short, short, short);
static double compute_fraction_fighting(short, short, short);
static long compute_soldiers_fighting(short, short, short);
static double compute_legion_fraction(short, short);
static short defeated(short, short);
static void print_summary(short);
static void print_round_preliminary(short);
static void print_round_summary(short);
static void print_remaining(short);
static void print_round_finals(short);
static void print_round_action(short);
static void print_round_title(short);
static void print_battle_report(short);
static void magic_round(short);
static void special_round(short);
static void attack(short, short, short, short, enum lsc_symbols af, short);
static void battle_round(short, enum lsc_symbols, short, short);
static void flank(short);
static void reserve_charge(short);
static void tactical_reserve(short);
static void support_fire(short);
static void opportunity_charge(short);
static void countercharge(short);
static void copy_previous(short, short, short);
static short fight(void);
static void battle_terrain(struct legion_struct *, struct legion_struct *);
#ifdef DEBUG
static void print_legion(short);
#endif

/* local variables */
static struct legion_struct Legion[2]; /* attacking and defending legions */
static struct battle_struct Battle[2]; /* combat results */
static short SearchBP[MAXBP + 1][MAXBP] = /* battlefield search sequences */
  {
      {  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 }, /* BP 0 */
      { 11, 10,  9,  8, 14,  7,  6, 13,  5,  4,  3,  2, 12,  1 }, /* BP 1 */
      { 10,  9, 11,  8, 14,  7,  6, 13,  5,  4,  3,  2, 12,  1 }, /* BP 2 */
      {  9,  8, 10, 14,  7, 11,  6, 13,  5,  4,  3,  2, 12,  1 }, /* BP 3 */
      {  8,  7,  9,  6, 10, 14, 13,  5, 11,  4,  3,  2, 12,  1 }, /* BP 4 */
      {  7,  6,  8, 13,  5,  9,  4, 10, 14,  3, 11,  2, 12,  1 }, /* BP 5 */
      {  6,  7,  5,  8,  4, 13,  9,  3, 10,  2, 14, 12, 11,  1 }, /* BP 6 */
      {  5,  6,  4, 13,  7,  3,  8,  2, 12,  9,  1, 10, 14, 11 }, /* BP 7 */
      {  4,  5,  3,  6,  2, 12, 13,  7,  1,  8,  9, 10, 14, 11 }, /* BP 8 */
      {  3,  4,  2, 12,  5,  1,  6, 13,  7,  8,  9, 10, 14, 11 }, /* BP 9 */
      {  2,  3,  1,  4, 12,  5,  6, 13,  7,  8,  9, 10, 14, 11 }, /* BP10 */
      {  1,  2,  3,  4, 12,  5,  6, 13,  7,  8,  9, 10, 14, 11 }, /* BP11 */
      /* Reserve slots */
      { 10,  9, 11,  8, 14,  7,  6, 13,  5,  4,  3,  2, 12,  1 }, /* BP12 */
      {  6,  7,  5,  8,  4, 13,  9,  3, 10,  2, 14, 12, 11,  1 }, /* BP13 */
      {  2,  3,  1,  4, 12,  5,  6, 13,  7,  8,  9, 10, 14, 11 }, /* BP14 */
  };
    /*{  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0 },
      { 11, 10, 14,  9,  8,  7,  6, 13,  5,  4,  3,  2, 12,  1 },
      { 10,  9, 11,  8, 14,  7,  6, 13,  5,  4,  3,  2, 12,  1 },
      {  9,  8, 10, 14,  7, 11,  6, 13,  5,  4,  3,  2, 12,  1 },
      {  8,  7,  9,  6, 10, 13, 14,  5, 11,  4,  3,  2, 12,  1 },
      {  7,  6,  8, 13,  5,  9,  4, 10, 14, 11,  3,  2, 12,  1 },
      {  6,  7,  5,  8,  4, 13,  9,  3, 10,  2, 14, 12, 11,  1 },
      {  5,  6,  4, 13,  7,  3,  8,  2, 12,  9,  1, 10, 14, 11 },
      {  4,  5,  3,  6,  2, 13, 12,  7,  1,  8,  9, 10, 14, 11 },
      {  3,  4,  2, 12,  5,  1,  6, 13,  7,  8,  9, 10, 14, 11 },
      {  2,  3,  1,  4, 12,  5,  6, 13,  7,  8,  9, 10, 14, 11 },
      {  1,  2, 12,  3,  4,  5,  6, 13,  7,  8,  9, 10, 14, 11 },
      { 10,  9, 11,  8, 14,  7,  6, 13,  5,  4,  3,  2, 12,  1 },
      {  6,  7,  5,  8,  4, 13,  9,  3, 10,  2, 14, 12, 11,  1 },
      {  2,  3,  1,  4, 12,  5,  6, 13,  7,  8,  9, 10, 14, 11 },*/
static short StartBP[MAXSLOTS] = /* slot starting battlefield positions */
/*slot 1  2  3  4   5   6   7   8 */
  { 0, 2, 4, 6, 8, 10, 12, 13, 14 };

static char who[2][9] =
  {
      "Attacker" ,
      "Defender"
  };

/*************************************************************************
help:
*************************************************************************/
static void help()
  {

    printf("\n\
\n  Command syntax: 'lbs [<attacker> <defender> | ?]'\
\n\
\n                  '<attacker>' and '<defender>' are the names of\
\n                  legion description files for the attacking and\
\n                  defending legions, respectively.\
\n");
    printf("\
\n  No arguments or '?' displays this help.\
\n  Use I/O re-direction to write the output into a file, e.g.\
\n'lbs f1234 f4321 >lbs.out'.\
\n");
    cleanup(ERROR_NO_ERROR, NULL, NULL);

  } /* end of help() */

/*************************************************************************
is_soldiers:
  returns TRUE if there are soldiers alive in the slot.
  returns FALSE if there are no soldiers alive in the slot.
*************************************************************************/
static short is_soldiers(legn, rnd, slot)
short legn;
short rnd;
short slot;
  {

    if(Battle[legn].Round[rnd].Slot[slot].SoldierStatus == STATUS_OK)
      return(TRUE);

    return(FALSE);

  } /* end of is_soldiers() */

/*************************************************************************
is_character:
  returns TRUE if there is a character alive in the slot.
  returns FALSE if there is no character alive in the slot.
*************************************************************************/
static short is_character(legn, rnd, slot)
short legn;
short rnd;
short slot;
  {

    if(Battle[legn].Round[rnd].Slot[slot].CharacterStatus == STATUS_OK)
      return(TRUE);

    return(FALSE);

  } /* end of is_character() */

/*************************************************************************
is_fighting_soldiers:
  returns TRUE if there are soldiers STILL FIGHTING in the slot.
  returns FALSE otherwise.
*************************************************************************/
static short is_fighting_soldiers(legn, rnd, slot)
short legn;
short rnd;
short slot;
  {

    if(is_soldiers(legn, rnd, slot))
      { /* soldiers present/alive */

        if(!(Battle[legn].Round[rnd].Slot[slot].Event &
            (EVENT_ROUT | EVENT_WITHDRAW)))
          { /* soldiers didn't rout or slot withdraw this round */

            return(TRUE);

          } /* soldiers didn't rout or slot withdraw this round */

      } /* soldiers present/alive */

    return(FALSE);

  } /* end of is_fighting_soldiers() */

/*************************************************************************
find_targets:
  finds enemy slots to attack using battlefield search sequence.
*************************************************************************/
static void find_targets(attacker, bp, rnd, target1, target2)
short attacker;
short bp;
short rnd;
short *target1;
short *target2;
  {

    short defender;
    short search;
    short target_slot;
    short done = FALSE;

    defender = (attacker + 1) % 2;
    *target1 = 0;
    *target2 = 0;

    for(search = 0; search < MAXBP && !done; search++)
      { /* step through search sequence */

        target_slot =
          Battle[defender].Round[rnd].Battlefield[SearchBP[bp][search]];

        if(target_slot)
          { /* slot in battlefield position */

            if(is_soldiers(defender, rnd, target_slot) ||
               is_character(defender, rnd, target_slot))
              { /* something alive in target slot */

                if(*target1 == 0)
                  { /* don't have first target */

                    *target1 = target_slot;

                  } /* don't have first target */
                else
                  { /* already have first target */

                    *target2 = target_slot;
                    done = TRUE;

                  } /* already have first target */

              } /* something alive in target slot */

          } /* slot in battlefield position */

      } /* step through search sequence */

    /* if only one target (or none), both are on same slot */
    if(*target2 == 0) *target2 = *target1;

  } /* end of find_targets() */

/*************************************************************************
find_opposing_bp:
  finds an open battlefield position in the front line 2-10 that most nearly
directly opposes the specified opponent bp.
  returns 0 if it can't find a position.

  the logic used is essentially the same as find_targets(), except that
we are looking for the first EMPTY position in the attacker's bp 2-10.
the same search sequence works nicely.
*************************************************************************/
static short find_opposing_bp(defender, rnd, dbp)
short defender; /* the opponent legion */
short rnd; /* round of interest */
short dbp; /* opponent battlefield position */
  {

    short attacker = (defender + 1) % 2;
    short search;
    short target_bp;
    short target_slot;

    if(dbp < 1 || dbp > MAXBP) return(0);

    for(search = 0; search < MAXBP; search++)
      { /* entire search sequence */

        target_bp = SearchBP[dbp][search];

        if(target_bp > 1 && target_bp < 11)
          { /* target bp is normal front position */

            target_slot = Battle[attacker].Round[rnd].Battlefield[target_bp];

            if(!target_slot)
              { /* target bp is empty */

                return(target_bp);

              } /* target bp is empty */
            else
              { /* target bp has a slot */

                if(!is_soldiers(attacker, rnd, target_slot) &&
                   !is_character(attacker, rnd, target_slot))
                  { /* nothing left in slot */

                    return(target_bp);

                  } /* nothing left in slot */

              } /* target bp has a slot */

          } /* target bp is normal front position */

      } /* entire search sequence */

    return(0);

  } /* end of find_opposing_bp() */

/*************************************************************************
compute_inv_factor:
  calculates the DF multiplier for given defender INV and attacker DAM
*************************************************************************/
static double compute_inv_factor(inv, dam)
long inv;
long dam;
  {

    double inv_factor = (1.0 + (.5 * (inv - dam)));

    if(inv_factor < 1.0) inv_factor = 1.0;

    return(inv_factor);

  } /* end of compute_inv_factor() */

/*************************************************************************
compute_character_fraction:
  calculates the fraction of original total attack strength remaining to
character.
*************************************************************************/
static double compute_character_fraction(legn, rnd, slot)
short legn;
short rnd;
short slot;
  {

    double character_fraction = 0.0;

    if(is_character(legn, rnd, slot))
      { /* character alive/present */

        if(Legion[legn].Slot[slot].Character[CF] > 0)
          { /* slot had character to begin with */

            if(Battle[legn].Round[rnd].Slot[slot].CharacterDamage <
               Legion[legn].Slot[slot].Character[CF])
              { /* fraction remains */

                character_fraction = 1.0 -
                  (((double)Battle[legn].Round[rnd].Slot[slot].CharacterDamage) /
                    Legion[legn].Slot[slot].Character[CF]);
                if(character_fraction < 0.0) character_fraction = 0.0;

              } /* fraction remains */

          } /* slot had character to begin with */

      } /* character alive/present */

    return(character_fraction);

  } /* end of compute_character_fraction() */

/*************************************************************************
compute_damage_fraction:
  calculates the fraction of damage to allocate to soldiers.
  if there are 20 or more soldiers in a slot, the character in the slot
will not take any damage from enemy attacks.  if there are fewer than
20 soldiers, the character receives 5% of the damage per soldier under 20.
  NOTE: I have no idea how this is done in the Legends battle program,
except that it is generally known that 20 soldiers will "protect" a
character from taking damage.
*************************************************************************/
static double compute_damage_fraction(legn, rnd, slot)
short legn;
short rnd;
short slot;
  {

    double soldier_damage;

    if(!is_character(legn, rnd, slot))
      { /* no character to split damage with */

        soldier_damage = 1.0;

      } /* no character to split damage with */
    else
      { /* there's a character */

        soldier_damage = 0.05 * compute_soldiers_remaining(legn, rnd, slot);

        if(soldier_damage > 1.0) soldier_damage = 1.0;

      } /* there's a character */

    return(soldier_damage);

  } /* end of compute_damage_fraction() */

/*************************************************************************
compute_soldier_fraction:
  calculates the fraction of soldiers remaining in slot, which is also the
fraction of original total attack strength remaining.
*************************************************************************/
static double compute_soldier_fraction(legn, rnd, slot)
short legn;
short rnd;
short slot;
  {

    double soldier_fraction = 0.0;

    if(is_soldiers(legn, rnd, slot))
      { /* soldiers alive/present */

        if(Legion[legn].Slot[slot].Total[CF] > 0)
          { /* slot had soldiers to begin with */

            if(Battle[legn].Round[rnd].Slot[slot].SoldierDamage <
               Legion[legn].Slot[slot].Total[CF])
              { /* fraction remains */

                soldier_fraction = 1.0 -
                  (((double)Battle[legn].Round[rnd].Slot[slot].SoldierDamage) /
                    Legion[legn].Slot[slot].Total[CF]);
                if(soldier_fraction < 0.0) soldier_fraction = 0.0;

              } /* fraction remains */

          } /* slot had soldiers to begin with */

      } /* soldiers alive/present */

    return(soldier_fraction);

  } /* end of compute_soldier_fraction() */

/*************************************************************************
compute_soldiers_remaining:
  determines the number of surviving soldiers in a slot for a given round.
ignores possible rout conditions, etc, just returns the number based on
the original amount of soldiers and damage taken.
Note: this is soldiers only, excludes characters.
*************************************************************************/
static long compute_soldiers_remaining(legn, rnd, slot)
short legn;
short rnd;
short slot;
  {

    if(is_soldiers(legn, rnd, slot))
      { /* soldiers alive/present */

        return(Legion[legn].Slot[slot].SoldierItems[AMT] -
               (long)(Legion[legn].Slot[slot].SoldierItems[AMT] *
                (1.0 - compute_soldier_fraction(legn, rnd, slot))));

      } /* soldiers alive/present */

    return(0);

  } /* end of compute_soldiers_remaining() */

/*************************************************************************
compute_fraction_fighting:
  calculates the fraction of slot's soldiers that are STILL FIGHTING at
the end of the specified round, as follows:
  Soldiers that are STATUS_DEAD are not counted as present.
  Soldiers that are STATUS_GONE are not counted as present.
  Soldiers that are STATUS_OK and are in a slot that does not have event
flags EVENT_WITHDRAW or EVENT_ROUT are counted, and the number remaining
is determined from the SoldierDamage.
*************************************************************************/
static double compute_fraction_fighting(legn, rnd, slot)
short legn;
short rnd;
short slot;
  {

    if(is_fighting_soldiers(legn, rnd, slot))
      { /* soldiers still fighting */

        return(compute_soldier_fraction(legn, rnd, slot));

      } /* soldiers still fighting */

    return(0.0);

  } /* end of compute_fraction_fighting() */

/*************************************************************************
compute_soldiers_fighting:
  calculates the number of soldiers in a slot that are STILL FIGHTING at
the end of a given round, as follows:
  Soldiers that are STATUS_DEAD are not counted as present.
  Soldiers that are STATUS_GONE are not counted as present.
  Soldiers that are STATUS_OK and are in a slot that does not have event
flags EVENT_WITHDRAW or EVENT_ROUT are counted, and the number remaining
is determined from the SoldierDamage.
*************************************************************************/
static long compute_soldiers_fighting(legn, rnd, slot)
short legn;
short rnd;
short slot;
  {

    if(is_fighting_soldiers(legn, rnd, slot))
      { /* soldiers still fighting */

        return(compute_soldiers_remaining(legn, rnd, slot));

      } /* soldiers still fighting */

    return(0);

  } /* end of compute_soldiers_fighting() */

/*************************************************************************
compute_legion_fraction:
  determines fraction of legion STILL FIGHTING in legion at end of given
round, as follows:
  The starting number of soldiers is taken from Round 0, the before battle
condition.
  Soldiers/Characters that are STATUS_DEAD are not counted as present.
  Soldiers/Characters that are STATUS_GONE are not counted as present.
  Soldiers that are STATUS_OK and are in a slot that does not have event
flags EVENT_WITHDRAW or EVENT_ROUT are counted, and the number remaining
is determined from the SoldierDamage.
  Characters that are STATUS_OK and are in a slot that does not have event
flags EVENT_WITHDRAW or EVENT_CHARACTER_ROUT are counted.
  NOTE: characters count as 1 soldier each.
*************************************************************************/
static double compute_legion_fraction(legn, rnd)
short legn;
short rnd;
  {

    long total = 0; /* original number of soldiers */
    long remain = 0; /* remaining number of soldiers */
    short bp;
    short slot;

    /* determine the original total soldiers */
    for(bp = 1; bp <= MAXBP; bp++)
      { /* all battlefield positions */

        slot = Battle[legn].Round[0].Battlefield[bp];

        if(slot)
          { /* slot in battlefield position */

            if(is_soldiers(legn, 0, slot))
              { /* soldiers present */

                total += Legion[legn].Slot[slot].SoldierItems[AMT];

              } /* soldiers present */

            if(is_character(legn, 0, slot))
              { /* character present */

                total += 1;

              } /* character present */

          } /* slot in battlefield position */

      } /* all battlefield positions */

    /* determine the soldiers remaining in specified round */
    for(bp = 1; bp <= MAXBP; bp++)
      { /* all battlefield positions */

        slot = Battle[legn].Round[rnd].Battlefield[bp];

        if(slot)
          { /* slot in battlefield position */

            remain += compute_soldiers_fighting(legn, rnd, slot);

            if(is_character(legn, rnd, slot))
              { /* character is still fighting */

                remain += 1;

              } /* character is still fighting */

          } /* slot in battlefield position */

      } /* all battlefield positions */

    if(total == 0) return(0.0);
    else return(((double)remain) / total);

  } /* end of compute_legion_fraction() */

/*************************************************************************
defeated:
  determines whether a legion was defeated in a specified round, and how it
was defeated.  Legion is defeated when losses exceed acceptable losses (or
reach 100%).
  returns: 0 if legion is not defeated yet.
           1 if legion has been wiped out (100% losses).
           2 if legion withdraws due to excessive losses.
*************************************************************************/
static short defeated(legn, rnd)
short legn;
short rnd;
  {

    double remaining = compute_legion_fraction(legn, rnd);

    if(remaining == 0.0) return(1);

    if(((1.0 - remaining) * 100) > Legion[legn].Losses[L_LOSSES]) return(2);

    return(0);

  } /* end of defeated() */

/*************************************************************************
print_round_preliminary:
  prints things that happen to slots before round begins, i.e. reserve
slot movement.
*************************************************************************/
static void print_round_preliminary(rnd)
short rnd;
  {

    short legn;
    short bp;
    short slot;

    for(legn = 0; legn <= 1; legn++)
      { /* both legions */

        for(bp = 1; bp <= MAXBP; bp++)
          { /* all battlefield positions */

            slot = Battle[legn].Round[rnd].Battlefield[bp];

            if(slot)
              { /* slot in battlefield position */

                if(Battle[legn].Round[rnd].Slot[slot].Event &
                   EVENT_FLANK)
                  { /* slot flanks */

                    printf("\n%s slot %d flanks into battlefield position %d!",
                           who[legn],
                           slot,
                           bp);

                  } /* slot flanks */

                if(Battle[legn].Round[rnd].Slot[slot].Event &
                   EVENT_CHARGE)
                  { /* slot charged */

                    printf("\n%s slot %d charges into battlefield position %d!",
                           who[legn],
                           slot,
                           bp);

                  } /* slot charged */

                if(Battle[legn].Round[rnd].Slot[slot].Event &
                   EVENT_REPLACE)
                  { /* slot replaced a destroyed/routed slot */

                    printf("\n%s slot %d advances into battlefield position %d!",
                           who[legn],
                           slot,
                           bp);

                  } /* slot replaced a destroyed/routed slot */

                if(Battle[legn].Round[rnd].Slot[slot].Event &
                   EVENT_COUNTER)
                  { /* slot countercharged */

                    printf("\n%s slot %d countercharges into battlefield position %d!",
                           who[legn],
                           slot,
                           bp);

                  } /* slot countercharged */

                if(Battle[legn].Round[rnd].Slot[slot].Event &
                   EVENT_OPPORTUNITY)
                  { /* slot opportunity charged */

                    printf("\n%s slot %d opportunity charges into battlefield position %d!",
                           who[legn],
                           slot,
                           bp);

                  } /* slot oppportunity charged */

              } /* slot in battlefield position */

          } /* all battlefield positions */

      } /* both legions */

  } /* end of print_round_preliminary() */

/*************************************************************************
print_round_summary:
  prints when what happened to slot during round.
*************************************************************************/
static void print_round_summary(rnd)
short rnd;
  {

    short legn;
    short bp;
    short slot;

    for(legn = 0; legn <= 1; legn++)
      { /* both legions */

        for(bp = 1; bp <= MAXBP; bp++)
          { /* all battlefield positions */

            slot = Battle[legn].Round[rnd].Battlefield[bp];

            if(slot)
              { /* slot in battlefield position */

                if(Battle[legn].Round[rnd].Slot[slot].Event &
                   EVENT_CHARACTER_ROUT)
                  { /* character routed */

                    printf("\n%s slot %d character %s routs!",
                           who[legn],
                           slot,
                           Legion[legn].Slot[slot].CharacterDesc);

                  } /* character routed */

                if(Battle[legn].Round[rnd].Slot[slot].Event &
                   EVENT_CHARACTER_DIE)
                  { /* slot character died */

                    printf("\n%s slot %d character %s was killed!",
                           who[legn],
                           slot,
                           Legion[legn].Slot[slot].CharacterDesc);

                  } /* slot character died */

                if(Battle[legn].Round[rnd].Slot[slot].Event &
                   EVENT_DIE)
                  { /* slot soldiers died */

                    printf("\n%s slot %d soldiers were annihilated!",
                           who[legn],
                           slot);

                  } /* slot soldiers died */

                if(Battle[legn].Round[rnd].Slot[slot].Event &
                   EVENT_WITHDRAW)
                  { /* slot withdrew from battle */

                    printf("\n%s slot %d withdraws from the battle!",
                           who[legn],
                           slot);

                  } /* slot died */

              } /* slot in battlefield position */

          } /* all battlefield positions */

      } /* both legions */

  } /* end of print_round_summary() */

/*************************************************************************
print_remaining:
  prints % of soldiers remaining
*************************************************************************/
static void print_remaining(rnd)
short rnd;
  {

    double remain[2]; /* remaining number of soldiers */

    remain[ATTACKER] = 100.0 * compute_legion_fraction(ATTACKER, rnd);
    remain[DEFENDER] = 100.0 * compute_legion_fraction(DEFENDER, rnd);

    /* @@ Power C wants it broken up like this ... */
    printf("\n\nPercent Remaining:          %4.1f%%       ",
           remain[ATTACKER]);
    printf("%4.1f%%", remain[DEFENDER]);

  } /* end of print_remaining() */

/*************************************************************************
print_round_finals:
  prints end of round soldiers remaining per slot, both sides.
*************************************************************************/
static void print_round_finals(rnd)
short rnd;
  {

    long Soldiers[2][MAXSLOTS]; /* soldiers left */
    short slot;
    short legion;

    for(legion = 0; legion <= 1; legion++)
      { /* both legions */

        for(slot = 1; slot <= MAXSSLOTS; slot++)
          { /* all slots */

            Soldiers[legion][slot] =
              compute_soldiers_fighting(legion, rnd, slot);

          } /* all slots */

      } /* both legions */

    /* @@ have to break this up like this for Power C */
    printf("\n\
\nEnd of Round Results:\
\nAttacker                Battlefield Soldiers              Defender\
\nPosition                Attacker    Defender              Position");
    printf("\nLeft Flank---------(1)->  %6d      ", Soldiers[ATTACKER][1]);
    printf("%6d    <-(5)-----Right Flank", Soldiers[DEFENDER][5]);
    printf("\nLeft RESERVE-------(6)->  %6d      ", Soldiers[ATTACKER][6]);
    printf("%6d    <-(8)-----Right RESERVE", Soldiers[DEFENDER][8]);
    printf("\nLeft Center--------(2)->  %6d      ", Soldiers[ATTACKER][2]);
    printf("%6d    <-(4)-----Right Center", Soldiers[DEFENDER][4]);
    printf("\nCenter-------------(3)->  %6d      ", Soldiers[ATTACKER][3]);
    printf("%6d    <-(3)-----Center", Soldiers[DEFENDER][3]);
    printf("\nCenter RESERVE-----(7)->  %6d      ", Soldiers[ATTACKER][7]);
    printf("%6d    <-(7)-----Center RESERVE", Soldiers[DEFENDER][7]);
    printf("\nRight Center-------(4)->  %6d      ", Soldiers[ATTACKER][4]);
    printf("%6d    <-(2)-----Left Center", Soldiers[DEFENDER][2]);
    printf("\nRight RESERVE------(8)->  %6d      ", Soldiers[ATTACKER][8]);
    printf("%6d    <-(6)-----Left RESERVE", Soldiers[DEFENDER][6]);
    printf("\nRight Flank--------(5)->  %6d      ", Soldiers[ATTACKER][5]);
    printf("%6d    <-(1)-----Left Flank", Soldiers[DEFENDER][1]);

  } /* end of print_round_finals() */

/*************************************************************************
print_round_action:
  blow by blow report per round.
*************************************************************************/
static void print_round_action(rnd)
short rnd;
  {

    short bp;
    short slot;
    short attacker; /* current attacking legion */
    short defender; /* other legion */
    short t; /* targets counter */

    for(attacker = 0; attacker <= 1; attacker++)
      { /* both combatants */

        defender = (attacker + 1) % 2;

        for(bp = 1; bp <= MAXBP; bp++)
          { /* all battlefield positions */

            slot = Battle[attacker].Round[rnd].Battlefield[bp];

            if(slot)
              { /* slot in battlefield position */

                for(t = 0; t <= 1; t++)
                  { /* both targets */

                    if(Battle[attacker].Round[rnd].Slot[slot].CharacterAttack[t] > 0)
                      { /* character did some damage to target */

                        /* @@ broken up for Power C */
                        printf("\n%s %s attacks with",
                               who[attacker],
                               Legion[attacker].Slot[slot].CharacterDesc);
                        printf(" %ld",
                        Battle[attacker].Round[rnd].Slot[slot].CharacterAttack[t]);
                        printf(" points against %s slot %d.",
                               who[defender],
                               Battle[attacker].Round[rnd].Slot[slot].Target[t]);

                      } /* character did some damage to target */

                  } /* both targets */

                for(t = 0; t <= 1; t++)
                  { /* both targets */

                    if(Battle[attacker].Round[rnd].Slot[slot].SoldierAttack[t] > 0)
                      { /* soldiers did some damage to target */

                        /* @@ broken up for Power C */
                        printf("\n%s slot %d soldiers attack with",
                               who[attacker],
                               slot);
                        printf(" %ld",
                        Battle[attacker].Round[rnd].Slot[slot].SoldierAttack[t]);
                        printf(" points against %s slot %d.",
                               who[defender],
                               Battle[attacker].Round[rnd].Slot[slot].Target[t]);

                      } /* soldiers did some damage to target */

                  } /* both targets */

              } /* slot in battlefield position */

          } /* all battlefield positions */

      } /* both combatants */

  } /* end of print_round_action() */

/*************************************************************************
print_round_title:
  prints a heading to identify round
*************************************************************************/
static void print_round_title(rnd)
short rnd;
  {

    switch(rnd)
      {
        case ROUND_MAGIC:
          { /* magic round */

            printf("\n\nM A G I C   R O U N D\n");
            break;

          } /* magic round */

        case ROUND_SPECIAL:
          { /* special attack round */

            printf("\n\nS P E C I A L   A T T A C K   R O U N D\n");
            break;

          } /* special attack round */

        case ROUND_MISSILE:
          { /* missile round */

            printf("\n\nM I S S I L E   R O U N D\n");
            break;

          } /* missile round */

        case ROUND_CHARGE:
          { /* charge round */

            printf("\n\nC H A R G E   R O U N D\n");
            break;

          } /* charge round */

        default:
          { /* melee rounds */

            printf("\n\nM E L E E   R O U N D   %d\n", (rnd - 3));
            break;

          } /* special attack round */

      } /* end switch(rnd) */

  } /* end of print_round_title() */

/*************************************************************************
print_battle_report:
  prints all the battle results by round.
*************************************************************************/
static void print_battle_report(winner)
short winner;
  {

    short round;
    short done = FALSE;

    if(winner == ATTACKER)
      { /* ATTACKER wins battle */

        printf("\n\nThe defenders were defeated!");

      } /* ATTACKER wins battle */
    else
      { /* DEFENDER wins battle */

        printf("\n\nThe attackers were defeated!");

      } /* DEFENDER wins battle */

    for(round = 1; round <= MAXROUNDS && !done; round++)
      { /* print each round */

        print_round_title(round);
        print_round_preliminary(round);

        if(round == ROUND_MAGIC || round == ROUND_SPECIAL)
          { /* magic or special attack rounds */

            print_round_action(round);

          } /* magic or special attack rounds */
#ifdef DEBUG
        else
          { /* other rounds */

            print_round_action(round);

          } /* other rounds */
#endif
        print_round_summary(round);
        print_round_finals(round);
        print_remaining(round);
        printf("\n\
\n************************************************************************");

        if(defeated(ATTACKER, round) || defeated(DEFENDER, round))
          {
            done = TRUE;
          }

      } /* print each round */

    done = defeated((winner + 1) % 2, round - 1);

    switch(done)
      { /* switch() */

        case 1:
          { /* loser was annihilated */

            printf("\n%s has been annihilated.",
              who[(winner + 1) % 2]);
            break;

          } /* loser was annihilated */

        case 2:
          { /* loser exceeded acceptable losses */

            printf("\n%s has withdrawn due to excessive losses.",
              who[(winner + 1) % 2]);
            break;

          } /* loser exceeded acceptible losses */

        default:
          { /* ran out of rounds */

            printf("\n%s has run out of time and withdrawn.",
              who[(winner + 1) % 2]);
            break;

          } /* ran out of rounds */

      } /* switch() */

    printf("\n");

  } /* end of print_battle_report() */

/*************************************************************************
magic_round:
  conducts battle for magic round.
*************************************************************************/
static void magic_round(rnd)
short rnd;
  {

    short attacker; /* legion whose attacks are being computed */
    short defender; /* the other legion */
    short bp; /* battlefield position */
    short slot; /* slot number */
    short target[2]; /* target slots */
    double character_atk; /* character attack fraction remaining 0/1 */
    double soldier_damage; /* fraction of damage to allocate to soldiers */

    /* determine targets */
    for(attacker = 0; attacker <= 1; attacker++)
      { /* both legions */

        defender = (attacker + 1) % 2; /* defender is other legion */

        for(bp = 1; bp <= MAXBP; bp++)
          { /* all battlefield positions */

            slot = Battle[attacker].Round[rnd].Battlefield[bp];

            if(slot)
              { /* battlefield position contain a slot */

                character_atk =
                  compute_character_fraction(attacker, rnd - 1, slot);

                /* character is either fully present or dead for this */
                if(character_atk != 0.0) character_atk = 1.0;

                find_targets(attacker, bp, rnd, &target[0], &target[1]);

                if(target[0] != 0)
                  { /* first (only) target exists */

                    /* assign target slot and damage */
                    Battle[attacker].Round[rnd].Slot[slot].Target[0] =
                      target[0];
                    Battle[attacker].Round[rnd].Slot[slot].CharacterAttack[0] =
                      character_atk * Legion[attacker].Slot[slot].Character[MAV];

                    /* allocate damage to target slot */
                    soldier_damage =
                      compute_damage_fraction(defender, rnd - 1, target[0]);
                    Battle[defender].Round[rnd].Slot[target[0]].SoldierDamage +=
                      ((soldier_damage *
                       Legion[attacker].Slot[slot].Character[MAV]) /
                       Legion[defender].Slot[target[0]].Total[MAD]);
                    Battle[defender].Round[rnd].Slot[target[0]].CharacterDamage +=
                      (((1.0 - soldier_damage) *
                       Legion[attacker].Slot[slot].Character[MAV]) /
                       Legion[defender].Slot[target[0]].Character[MAD]);

                  } /* first (only) target exists */

              } /* battlefield position contains a slot */

          } /* all battlefield positions */

      } /* both legions */

  }/* end of magic_round() */

/*************************************************************************
special_round:
  conducts battle for special attack round
*************************************************************************/
static void special_round(rnd)
short rnd;
  {

    short attacker; /* legion whose attacks are being computed */
    short defender; /* the other legion */
    short bp; /* battlefield position */
    short slot; /* legion slot number */
    short target[2]; /* target slots */
    double soldier_atk; /* fraction of attacking soldiers remaining */
    double character_atk; /* fraction of attacking character, 0 or 1 */
    double soldier_damage; /* fraction of damage absorbed by soldiers */

    for(attacker = 0; attacker <= 1; attacker++)
      { /* attacker and defender attacks */

        defender = (attacker + 1) % 2; /* defender is other legion */

        for(bp = 1; bp <= MAXBP; bp++)
          { /* all battlefield positions */

            slot = Battle[attacker].Round[rnd].Battlefield[bp];

            if(slot)
              { /* slot in battlefield position */

                soldier_atk =
                  compute_fraction_fighting(attacker, rnd - 1, slot);

                character_atk =
                  compute_character_fraction(attacker, rnd - 1, slot);

                /* character is either fully present or dead for this */
                if(character_atk != 0.0) character_atk = 1.0;

                find_targets(attacker, bp, rnd, &target[0], &target[1]);

                if(target[0] != 0)
                  { /* first (only) target exists */

                    /* determine targets and attack values */
                    Battle[attacker].Round[rnd].Slot[slot].Target[0] =
                      target[0];
                    Battle[attacker].Round[rnd].Slot[slot].SoldierAttack[0] =
                      (soldier_atk * Legion[attacker].Slot[slot].Total[SAV]);
                    Battle[attacker].Round[rnd].Slot[slot].CharacterAttack[0] =
                      (character_atk * Legion[attacker].Slot[slot].Character[SAV]);

                    /* allocate damage to defender */
                    soldier_damage =
                      compute_damage_fraction(defender, rnd - 1, target[0]);
                    Battle[defender].Round[rnd].Slot[target[0]].SoldierDamage +=
                      (soldier_damage *
                      (Battle[attacker].Round[rnd].Slot[slot].SoldierAttack[0] +
                       Battle[attacker].Round[rnd].Slot[slot].CharacterAttack[0])) /
                      Legion[defender].Slot[target[0]].Total[SAD];
                    Battle[defender].Round[rnd].Slot[target[0]].CharacterDamage +=
                      ((1.0 - soldier_damage) *
                      (Battle[attacker].Round[rnd].Slot[slot].SoldierAttack[0] +
                       Battle[attacker].Round[rnd].Slot[slot].CharacterAttack[0])) /
                      Legion[defender].Slot[target[0]].Character[SAD];

                  } /* first (only) target exists */

              } /* slot in battlefield position */

          } /* all battlefield positions */

      } /* attacker and defender attacks */

  } /* end of special_round() */

/*************************************************************************
battle_round:
  conducts battle for missile, charge, and melee rounds
*************************************************************************/
static void battle_round(rnd, af, first_bp, last_bp)
short rnd;
enum lsc_symbols af;
short first_bp;
short last_bp;
  {

    short attacker; /* legion whose attacks are being computed */
    short bp; /* legion battlefield position */
    short slot; /* legion slot number */

    for(attacker = 0; attacker <= 1; attacker++)
      { /* attacker and defender attacks */

        for(bp = first_bp; bp <= last_bp; bp++)
          { /* all appropriate battlefield positions */

            slot = Battle[attacker].Round[rnd].Battlefield[bp];

            if(slot)
              { /* slot in battlefield position */

                attack(attacker, rnd, bp, slot, af, 0);

              } /* slot in battlefield position */

          } /* all appropriate battlefield positions */

      } /* attacker and defender attacks */

  } /* end of battle_round() */

/*************************************************************************
attack:
  selects targets for slot and allocates damage
  uses the provided attack type unless slot has charged this round.
*************************************************************************/
static void attack(attacker, rnd, bp, slot, rnd_af, afb)
short attacker; /* attacking legion */
short rnd; /* current round */
short bp; /* attacker battlefield position */
short slot; /* attacker slot */
enum lsc_symbols rnd_af; /* attack factor type to use */
short afb; /* extra attack factor bonus to apply to attack */
  {

      short defender = (attacker + 1) % 2; /* opposing legion */
      enum lsc_symbols af; /* attack factor type to use */
      double soldier_atk; /* fraction of slot's soldier attack left */
      double character_atk; /* fraction of slot's character attack left */
      double target_factor; /* attack fraction split between targets */
      double soldier_damage; /* fraction of damage for soldiers */
      long soldier_af; /* soldier's adjusted attack */
      long character_af; /* character's adjusted attack */
      short target[2]; /* target slots */
      short t; /* counter */

      if(Battle[attacker].Round[rnd].Slot[slot].Event &
         EVENT_CHARGE_EVENTS)
        { /* slot charged this round */

          /* use charge attack instead of given */
          af = CAF;

        } /* slot charged this round */
      else
        { /* slot didn't charge */

          /* use the given attack type */
          af = rnd_af;

        } /* slot didn't charge */

      soldier_atk = compute_fraction_fighting(attacker, rnd - 1, slot);

      character_atk = compute_character_fraction(attacker, rnd - 1, slot);

      find_targets(attacker, bp, rnd, &target[0], &target[1]);

      for(t = 0; t <= 1; t++)
        { /* both targets */

          if(target[t] != 0)
            { /* target exists */

              if(t == 0) target_factor = 0.8;
              else target_factor = 0.2;

              soldier_af =
                Legion[attacker].Slot[slot].Total[af] + (long)
                (Legion[attacker].Slot[slot].Total[CF] *
                 (((double)afb) / 100.0));

              if(soldier_af < 0) soldier_af = 0;

              character_af =
                Legion[attacker].Slot[slot].Character[af] + (long)
                (Legion[attacker].Slot[slot].Character[CF] *
                 (((double)afb) / 100.0));

              if(character_af < 0) character_af = 0;

              /* determine targets and attack values */
              Battle[attacker].Round[rnd].Slot[slot].Target[t] =
                target[t];
              Battle[attacker].Round[rnd].Slot[slot].SoldierAttack[t] =
                target_factor * soldier_atk * soldier_af;
              Battle[attacker].Round[rnd].Slot[slot].CharacterAttack[t] =
                target_factor * character_atk * character_af;

              /* allocate damage to opponent */
              soldier_damage =
                compute_damage_fraction(defender, rnd - 1, target[t]);
              Battle[defender].Round[rnd].Slot[target[t]].SoldierDamage +=
                (soldier_damage) *
                  (
                    (
                 Battle[attacker].Round[rnd].Slot[slot].SoldierAttack[t]
                    /
                      (
                 Legion[defender].Slot[target[t]].Total[DF] *
                  compute_inv_factor(Legion[defender].Slot[target[t]].Total[INV],
                                     Legion[attacker].Slot[slot].Total[DAM])
                      )
                    )
                  +
                    (
                 Battle[attacker].Round[rnd].Slot[slot].CharacterAttack[t]
                    /
                      (
                 Legion[defender].Slot[target[t]].Total[DF] *
                  compute_inv_factor(Legion[defender].Slot[target[t]].Total[INV],
                                     Legion[attacker].Slot[slot].Character[DAM])
                      )
                    )
                  );
              Battle[defender].Round[rnd].Slot[target[t]].CharacterDamage +=
                (1.0 - soldier_damage) *
                  (
                    (
                 Battle[attacker].Round[rnd].Slot[slot].SoldierAttack[t]
                    /
                      (
                 Legion[defender].Slot[target[t]].Character[DF] *
                  compute_inv_factor(Legion[defender].Slot[target[t]].Character[INV],
                                     Legion[attacker].Slot[slot].Total[DAM])
                      )
                    )
                  +
                    (
                 Battle[attacker].Round[rnd].Slot[slot].CharacterAttack[t]
                    /
                      (
                 Legion[defender].Slot[target[t]].Character[DF] *
                  compute_inv_factor(Legion[defender].Slot[target[t]].Character[INV],
                                     Legion[attacker].Slot[slot].Character[DAM])
                      )
                    )
                  );

            } /* target exists */

        } /* both targets */

    } /* end of attack() */

/*************************************************************************
flank:
  moves the flanking reserve slots into their new battlefield positions.
*************************************************************************/
static void flank(rnd)
short rnd;
  {

    short legn;
    short bp;
    short slot;

    for(legn = 0; legn <= 1; legn++)
      { /* both legions */

        for(bp = 12; bp <= MAXBP; bp++)
          { /* all reserve battlefield positions */

            slot = Battle[legn].Round[rnd].Battlefield[bp];

            switch(slot)
              { /* switch(slot) */

                case 6:
                  { /* slot 6 */

                    if(Legion[legn].Reserve[R_LEFT] == RO_FLANK_LEFT)
                      { /* "Flank Left" */

                        /* move the slot into new Battlefield position */
                        Battle[legn].Round[rnd].Battlefield[1] = slot;
                        Battle[legn].Round[rnd].Battlefield[bp] = 0;
                        /* add the event */
                        Battle[legn].Round[rnd].Slot[slot].Event |=
                          EVENT_FLANK;

                      } /* "Flank Left" */

                    break;

                  } /* slot 6 */

                case 8:
                  { /* slot 8 */

                    if(Legion[legn].Reserve[R_RIGHT] == RO_FLANK_RIGHT)
                      { /* "Flank Right" */

                        /* move the slot into new Battlefield position */
                        Battle[legn].Round[rnd].Battlefield[11] = slot;
                        Battle[legn].Round[rnd].Battlefield[bp] = 0;
                        /* add the event */
                        Battle[legn].Round[rnd].Slot[slot].Event |=
                          EVENT_FLANK;

                      } /* "Flank Right" */

                    break;

                  } /* slot 8 */

              } /* switch(slot) */

          } /* all reserve battlefield positions */

      } /* both legions */

  } /* end of flank() */

/*************************************************************************
reserve_charge:
  moves the charging reserve slots into their new battlefield positions.
*************************************************************************/
static void reserve_charge(rnd)
short rnd;
  {

    short legn;
    short bp;
    short slot;

    for(legn = 0; legn <= 1; legn++)
      { /* both legions */

        for(bp = 12; bp <= MAXBP; bp++)
          { /* all reserve battlefield positions */

            slot = Battle[legn].Round[rnd].Battlefield[bp];

            if(is_reserve_slot(slot))
              { /* reserve slot */

                switch(Legion[legn].Reserve[slot-6])
                  { /* switch() */

                    case RO_CHARGE_LEFT:
                      { /* "Charge Left" */

                        /* move the slot into new Battlefield position */
                        if(Battle[legn].Round[rnd].Battlefield[3] == 0)
                          { /* BP 3 free */

                            Battle[legn].Round[rnd].Battlefield[3] = slot;
                            Battle[legn].Round[rnd].Battlefield[bp] = 0;
                            /* add the event */
                            Battle[legn].Round[rnd].Slot[slot].Event |=
                              EVENT_CHARGE;

                          } /* BP 3 free */
                        else
                          { /* BP 3 not free */

                            if(Battle[legn].Round[rnd].Battlefield[5] == 0)
                              { /* BP 5 free */

                                Battle[legn].Round[rnd].Battlefield[5] = slot;
                                Battle[legn].Round[rnd].Battlefield[bp] = 0;
                                /* add the event */
                                Battle[legn].Round[rnd].Slot[slot].Event |=
                                  EVENT_CHARGE;

                              } /* BP 5 free */
                            else
                              { /* BP 5 not free */

                                /* no available slot! */
                                /* this should never happen, but if */
                                /* it does change the reserve option */
                                /* to TACTICAL RESERVE */
                                Legion[legn].Reserve[slot-6] =
                                  RO_TACTICAL_RESERVE;

                              } /* BP 5 not free */

                          } /* BP 3 not free */

                        break;

                      } /* "Charge Left" */

                    case RO_CHARGE_CENTER:
                      { /* "Charge Center" */

                        /* move the slot into new Battlefield position */
                        if(Battle[legn].Round[rnd].Battlefield[5] == 0)
                          { /* BP 5 free */

                            Battle[legn].Round[rnd].Battlefield[5] = slot;
                            Battle[legn].Round[rnd].Battlefield[bp] = 0;
                            /* add the event */
                            Battle[legn].Round[rnd].Slot[slot].Event |=
                              EVENT_CHARGE;

                          } /* BP 5 free */
                        else
                          { /* BP 5 not free */

                            if(Battle[legn].Round[rnd].Battlefield[7] == 0)
                              { /* BP 7 free */

                                Battle[legn].Round[rnd].Battlefield[7] = slot;
                                Battle[legn].Round[rnd].Battlefield[bp] = 0;
                                /* add the event */
                                Battle[legn].Round[rnd].Slot[slot].Event |=
                                  EVENT_CHARGE;

                              } /* BP 7 free */
                            else
                              { /* BP 7 not free */

                                /* no available slot! */
                                /* this could happen to the right reserve */
                                /* if all reserve slots charge center */
                                /* if it does, change the reserve option */
                                /* to TACTICAL RESERVE */
                                Legion[legn].Reserve[slot-6] =
                                  RO_TACTICAL_RESERVE;

                              } /* BP 7 not free */

                          } /* BP 5 not free */

                        break;

                      } /* "Charge Center" */

                    case RO_CHARGE_RIGHT:
                      { /* "Charge Right" */

                        /* move the slot into new Battlefield position */
                        if(Battle[legn].Round[rnd].Battlefield[7] == 0)
                          { /* BP 7 free */

                            Battle[legn].Round[rnd].Battlefield[7] = slot;
                            Battle[legn].Round[rnd].Battlefield[bp] = 0;
                            /* add the event */
                            Battle[legn].Round[rnd].Slot[slot].Event |=
                               EVENT_CHARGE;

                          } /* BP 7 free */
                        else
                          { /* BP 7 not free */

                            if(Battle[legn].Round[rnd].Battlefield[9] == 0)
                              { /* BP 9 free */

                                Battle[legn].Round[rnd].Battlefield[9] = slot;
                                Battle[legn].Round[rnd].Battlefield[bp] = 0;
                                /* add the event */
                                Battle[legn].Round[rnd].Slot[slot].Event |=
                                  EVENT_CHARGE;

                               } /* BP 9 free */
                             else
                               { /* BP 9 not free */

                                  /* no available slot! */
                                  /* this should never happen, but if */
                                  /* it does change the reserve option */
                                  /* to TACTICAL RESERVE */
                                  Legion[legn].Reserve[slot-6] =
                                    RO_TACTICAL_RESERVE;

                                } /* BP 9 not free */

                            } /* BP 7 not free */

                        break;

                      } /* "Charge Right" */

                  } /* switch(slot) */

              } /* reserve slot */

          } /* all reserve battlefield positions */

      } /* both legions */

  } /* end of reserve_charge() */

/*************************************************************************
tactical_reserve:
  checks for conditions which trigger the tactical reserve option during
normal melee rounds, and has any slots remaining in the reserve with
tactical reserve orders react accordingly.
  Tactical Reserve slots move into a front battlefield position vacated by
any slot that has been destroyed or routed.
NOTE: only the regular front battlefield positions (2-10) are considered.
*************************************************************************/
static void tactical_reserve(rnd)
short rnd;
  {

    short attacker;
    short defender;
    short bp;
    short slot;
    short done;
    short abp;
    short aslot;

    for(attacker = 0; attacker <= 1; attacker++)
      { /* both legions */

        defender = (attacker + 1) % 2;

        for(bp = 2; bp <= 10; bp++)
          { /* all normal front battlefield positions */

            slot = Battle[attacker].Round[rnd-1].Battlefield[bp];

            if(slot)
              { /* slot in battlefield position on previous round */

                if(
                   /* entire slot was dead/gone after previous round */
                   (!is_soldiers(attacker, rnd-1, slot) &&
                    !is_character(attacker, rnd-1, slot)) ||
                   /* soldiers routed previous round */
                   (Battle[attacker].Round[rnd-1].Slot[slot].Event &
                    EVENT_ROUT)
                  )
                  { /* slot is trigger for reserves */

                    /* look for a reserve slot on TacRes in same legion */
                    done = FALSE;

                    for(abp = 12; abp <= MAXBP && !done; abp++)
                      { /* all reserve battlefield positions */

                        aslot = Battle[attacker].Round[rnd].Battlefield[abp];

                        if(is_reserve_slot(aslot))
                          { /* slot in reserve battlefield position */

                            if(Legion[attacker].Reserve[aslot-6] ==
                               RO_TACTICAL_RESERVE)
                              { /* reserve slot is TR */

                                done = TRUE;
                                /* move reserve into position */
                                Battle[attacker].Round[rnd].Battlefield[bp] =
                                  aslot;
                                Battle[attacker].Round[rnd].Battlefield[abp] =
                                  0;
                                /* record event */
                                Battle[attacker].Round[rnd].Slot[aslot].Event |=
                                  EVENT_REPLACE;

                              } /* reserve slot is TR */

                          } /* slot in reserve battlefield position */

                      } /* all reserve battlefield positions */

                  } /* slot is trigger for reserves */

              } /* slot in battlefield position on previous round */

          } /* all normal front battlefield positions */

      } /* both legions */

  } /* end of tactical_reserve() */

/*************************************************************************
support_fire:
  selects targets and allocates damage for reserve slots with Support
Missile Fire options.
NOTE: Support Missile Fire is at -50% MAFB
*************************************************************************/
static void support_fire(rnd)
short rnd;
  {

    short attacker; /* legion whose attacks are being computed */
    short bp; /* legion battlefield position */
    short slot; /* legion slot number */
    short reserve_option; /* reserve slot option */
    short target_bp; /* firing target, depends on specific reserve option */

    for(attacker = 0; attacker <= 1; attacker++)
      { /* both legions */

        for(bp = 12; bp <= MAXBP; bp++)
          { /* all reserve battlefield positions */

            slot = Battle[attacker].Round[rnd].Battlefield[bp];

            if(is_reserve_slot(slot))
              { /* slot in reserve battlefield position */

                reserve_option = Legion[attacker].Reserve[slot-6];

                if((reserve_option == RO_SUPPORT_LEFT_FLANK) ||
                   (reserve_option == RO_SUPPORT_LEFT_CENTER) ||
                   (reserve_option == RO_SUPPORT_CENTER) ||
                   (reserve_option == RO_SUPPORT_RIGHT_CENTER) ||
                   (reserve_option == RO_SUPPORT_RIGHT_FLANK))
                  { /* slot has support missile fire */

                    switch(reserve_option)
                      { /* switch() */

                        case RO_SUPPORT_LEFT_FLANK:
                          { /* support left flank */

                            target_bp = 2;
                            break;

                          } /* support left flank */

                        case RO_SUPPORT_LEFT_CENTER:
                          { /* support left center */

                            target_bp = 4;
                            break;

                          } /* support left center */

                        case RO_SUPPORT_CENTER:
                          { /* support center */

                            target_bp = 6;
                            break;

                          } /* support center */

                        case RO_SUPPORT_RIGHT_CENTER:
                          { /* support right center */

                            target_bp = 8;
                            break;

                          } /* support right center */

                        case RO_SUPPORT_RIGHT_FLANK:
                          { /* support right flank */

                            target_bp = 10;
                            break;

                          } /* support right flank */

                      } /* switch() */

                    attack(attacker, rnd, target_bp, slot, MAF, -50);

                  } /* slot has support missile fire */

              } /* slot in reserve battlefield position */

          } /* all reserve battlefield positions */

      } /* both legions */

  } /* end of support_fire() */

/*************************************************************************
opportunity_charge:
  checks for conditions which trigger the opportunity charge reserve option
during normal melee rounds, and has all opponent reserve slots with
opportunity charge orders react.
  Opportunity Charge slots charge into a front battlefield position across
from an enemy slot that has been destroyed or routed.
NOTE: only the regular front battlefield positions (2-10) are considered.
*************************************************************************/
static void opportunity_charge(rnd)
short rnd;
  {

    short attacker;
    short defender;
    short bp;
    short slot;
    short dbp;
    short dslot;
    short newbp;

    for(attacker = 0; attacker <= 1; attacker++)
      { /* both legions */

        defender = (attacker + 1) % 2;

        for(bp = 2; bp <= 10; bp++)
          { /* all normal front battlefield positions */

            slot = Battle[attacker].Round[rnd-1].Battlefield[bp];

            if(slot)
              { /* slot in battlefield position on previous round */

                if(
                   /* entire slot was dead/gone after previous round */
                   (!is_soldiers(attacker, rnd-1, slot) &&
                    !is_character(attacker, rnd-1, slot)) ||
                   /* soldiers routed previous round */
                   (Battle[attacker].Round[rnd-1].Slot[slot].Event &
                    EVENT_ROUT)
                  )
                  { /* slot is trigger for reserves */

                    /* look for opponent reserve slots to opportunity chg */

                    for(dbp = 12; dbp <= MAXBP; dbp++)
                      { /* all opponent reserve battlefield positions */

                        dslot = Battle[defender].Round[rnd].Battlefield[dbp];

                        if(is_reserve_slot(dslot))
                          { /* slot in opponent reserve position */

                            if(Legion[defender].Reserve[dslot-6] ==
                               RO_OPPORTUNITY_CHARGE)
                              { /* reserve slot has oppty chg */

                                newbp = find_opposing_bp(attacker, rnd, bp);

                                if(newbp)
                                  { /* found open bp */

                                    /* move to new bp */
                                    Battle[defender].Round[rnd].Battlefield[newbp] =
                                      dslot;
                                    Battle[defender].Round[rnd].Battlefield[dbp] = 0;
                                    /* record the event */
                                    Battle[defender].Round[rnd].Slot[dslot].Event |=
                                      EVENT_OPPORTUNITY;

                                  } /* found open bp */

                              } /* reserve slot has oppty chg */

                          } /* slot in opponent reserve position */

                      } /* all opponent reserve battlefield positions */

                  } /* slot is trigger for reserves */

              } /* slot in battlefield position on previous round */

          } /* all normal front battlefield positions */

      } /* both legions */

  } /* end of opportunity_charge() */

/*************************************************************************
countercharge:
  checks for oppponent slots opportunity charging during normal melee
rounds and has any slots remaining in the reserve with countercharge orders
react.
  Countercharge slots charge into a front battlefield position across from
an enemy slot that Opportunity Charges.
NOTE: only the regular front battlefield positions (2-10) are considered.
*************************************************************************/
static void countercharge(rnd)
short rnd;
  {

    short attacker;
    short defender;
    short bp;
    short slot;
    short dbp;
    short dslot;
    short newbp;

    for(attacker = 0; attacker <= 1; attacker++)
      { /* both legions */

        defender = (attacker + 1) % 2;

        for(bp = 2; bp <= 10; bp++)
          { /* all normal front battlefield positions */

            slot = Battle[attacker].Round[rnd].Battlefield[bp];

            if(slot)
              { /* slot in battlefield position THIS round */

                if(Battle[attacker].Round[rnd].Slot[slot].Event &
                   EVENT_OPPORTUNITY)
                  { /* slot opportunity charged */

                    /* look for opponent reserve slots to countercharge */

                    for(dbp = 12; dbp <= MAXBP; dbp++)
                      { /* all reserve battlefield positions */

                        dslot = Battle[defender].Round[rnd].Battlefield[dbp];

                        if(is_reserve_slot(dslot))
                          { /* slot in battlefield reserve */

                            if(Legion[defender].Reserve[dslot-6] ==
                               RO_COUNTERCHARGE)
                              { /* reserve slot has countercharge */

                                newbp = find_opposing_bp(attacker, rnd, slot);

                                if(newbp)
                                  { /* found open bp */

                                    /* move to new bp */
                                    Battle[defender].Round[rnd].Battlefield[newbp] =
                                      dslot;
                                    Battle[defender].Round[rnd].Battlefield[dbp] = 0;
                                    /* record the event */
                                    Battle[defender].Round[rnd].Slot[dslot].Event |=
                                      EVENT_COUNTER;

                                  } /* found open bp */

                              } /* reserve slot has countercharge */

                          } /* slot in battlefield reserve */

                      } /* all reserve battlefield positions */

                  } /* slot opportunity charged */

              } /* slot in battlefield position THIS round */

          } /* all normal front battlefield positions */

      } /* both legions */

  } /* end of countercharge() */

/*************************************************************************
copy_previous:
  copies certain battle data from previous round into the new round.
*************************************************************************/
static void copy_previous(legn, rnd, slot)
short legn;
short rnd;
short slot;
  {

      Battle[legn].Round[rnd].Slot[slot].SoldierDamage =
        Battle[legn].Round[rnd - 1].Slot[slot].SoldierDamage;
      Battle[legn].Round[rnd].Slot[slot].SoldierStatus =
        Battle[legn].Round[rnd - 1].Slot[slot].SoldierStatus;
      Battle[legn].Round[rnd].Slot[slot].CharacterDamage =
        Battle[legn].Round[rnd - 1].Slot[slot].CharacterDamage;
      Battle[legn].Round[rnd].Slot[slot].CharacterStatus =
        Battle[legn].Round[rnd - 1].Slot[slot].CharacterStatus;

      /* add "sticky actions" from last round to any new events */
      /* for the coming round */
      Battle[legn].Round[rnd].Slot[slot].Event |=
        (Battle[legn].Round[rnd - 1].Slot[slot].Event & EVENT_STICKY);

  } /* end of copy_previous() */

/*************************************************************************
fight:
  conducts battle between attacking and defending legions.
  returns winning legion, ATTACKER or DEFENDER.
*************************************************************************/
static short fight()
  {

    short rnd;
    short legn;
    short slot;
    short bp;

    /* initialize the battle data */
    for(legn = 0; legn < 2; legn++)
      { /* both legions */

        for(slot = 1; slot <= MAXSSLOTS; slot++)
          { /* all slots */

            if(Legion[legn].Slot[slot].Total[CF] > 0 ||
               Legion[legn].Slot[slot].Character[CF] > 0)
              { /* something in slot */

                Battle[legn].Round[0].Battlefield[StartBP[slot]] = slot;

                if(Legion[legn].Slot[slot].Total[CF] <= 0)
                  Battle[legn].Round[0].Slot[slot].SoldierStatus =
                  STATUS_DEAD;

                if(Legion[legn].Slot[slot].Character[CF] <= 0)
                  Battle[legn].Round[0].Slot[slot].CharacterStatus =
                  STATUS_DEAD;

              } /* something in slot */

          } /* all slots */

      } /* both legions */

    for(rnd = 1; rnd <= MAXROUNDS; rnd++)
      { /* do each combat round */

        /* copy end state from previous round */
        for(legn = 0; legn <= 1; legn++)
          { /* both legions */

            for(slot = 1; slot <= MAXSSLOTS; slot++)
              { /* all slots */

                copy_previous(legn, rnd, slot);

              } /* all slots */

            for(bp = 1; bp <= MAXBP; bp++)
              { /* all battlefield positions */

                slot = Battle[legn].Round[rnd - 1].Battlefield[bp];

                if(slot)
                  { /* battlefield position occupied */

                    if(!(Battle[legn].Round[rnd - 1].Slot[slot].Event &
                       EVENT_WITHDRAW))
                      { /* slot didn't withdraw */

                        Battle[legn].Round[rnd].Battlefield[bp] = slot;

                      } /* slot didn't withdraw */

                  } /* battlefield position occupied */

              } /* all battlefield positions */

          } /* both legions */

        switch(rnd)
          { /* which round? */

            case ROUND_MAGIC:
              { /* magic round */

                magic_round(rnd);
                break;

              } /* magic round */

            case ROUND_SPECIAL:
              { /* special attack round */

                special_round(rnd);
                break;

              } /* special attack round */

            case ROUND_MISSILE:
              { /* missile round */

                battle_round(rnd, MAF, 1, MAXBP);
                break;

              } /* missile round */

            case ROUND_CHARGE:
              { /* charge round */

                flank(rnd);
#ifdef NOCHARGE
                battle_round(rnd, AF, 1, 11);
#else
                battle_round(rnd, CAF, 1, 11);
#endif
                break;

              } /* charge round */

            default:
              { /* melee round */

                tactical_reserve(rnd);
                support_fire(rnd);
                reserve_charge(rnd);
                opportunity_charge(rnd);
                countercharge(rnd);
                battle_round(rnd, AF, 1, 11);
                break;

              } /* melee round */

          } /* which round? */

        /* end of round slot updates */
        for(legn = 0; legn <= 1; legn++)
          { /* both legions */

            for(bp = 1; bp <= MAXBP; bp++)
              { /* all battlefield positions */

                slot = Battle[legn].Round[rnd].Battlefield[bp];

                if(slot)
                  { /* battlefield position occupied */

                    /* update the soldier status */
                    if(is_soldiers(legn, rnd, slot))
                      { /* soldiers were still fighting */

                        if(Battle[legn].Round[rnd].Slot[slot].SoldierDamage >=
                           Legion[legn].Slot[slot].Total[CF])
                          { /* soldiers are dead now */

                            Battle[legn].Round[rnd].Slot[slot].SoldierStatus =
                              STATUS_DEAD;
                            Battle[legn].Round[rnd].Slot[slot].Event |=
                              EVENT_DIE;

                          } /* soldiers are dead now */

                      } /* soldiers were still fighting */

                    /* update the character status */
                    if(is_character(legn, rnd, slot))
                      { /* character was still fighting */

                        if(Battle[legn].Round[rnd].Slot[slot].CharacterDamage >=
                           Legion[legn].Slot[slot].Character[CF])
                          { /* character is dead now */

                            Battle[legn].Round[rnd].Slot[slot].CharacterStatus =
                              STATUS_DEAD;
                            Battle[legn].Round[rnd].Slot[slot].Event |=
                              EVENT_CHARACTER_DIE;

                          } /* character is dead now */
                        else
                          { /* character still alive */

                            /* characters rout at Acceptable Losses wounds */
                            if(Battle[legn].Round[rnd].Slot[slot].CharacterDamage >=
                               (Legion[legn].Losses[L_LOSSES] / 100.0) *
                                Legion[legn].Slot[slot].Character[CF])
                              { /* character wounds exceed acceptable losses */

                                Battle[legn].Round[rnd].Slot[slot].CharacterStatus =
                                  STATUS_GONE;
                                Battle[legn].Round[rnd].Slot[slot].Event |=
                                  EVENT_CHARACTER_ROUT;

                              } /* character wounds exceed acceptable losses */

                          } /* character still alive */

                      } /* character was still fighting */

                    /* anybody left? */
                    if(!is_soldiers(legn, rnd, slot) &&
                       !is_character(legn, rnd, slot))
                      { /* everybody dead/gone */

                        /* slot withdraws from battle */
                        Battle[legn].Round[rnd].Slot[slot].Event |=
                          EVENT_WITHDRAW;

                      } /* everybody dead/gone */

                  } /* battlefield position occupied */

              } /* all battlefield positions */

          } /* both legions */

        if(defeated(ATTACKER, rnd))
          { /* attacking legion defeated */

            return(DEFENDER);

          } /* attacking legion defeated */
        else
          { /* attacking legion not defeated */

            if(defeated(DEFENDER, rnd))
              { /* defending legion defeated */

                return(ATTACKER);

              } /* defending legion defeated */

          } /* attacking legion not defeated */

      } /* do each combat round */

    /* defender wins by default */
    return(DEFENDER);

  } /* end of fight() */

/*************************************************************************
battle_terrain:
  makes the attacking and defending forces terrains make sense.
*************************************************************************/
void battle_terrain(Attacker, Defender)
struct legion_struct *Attacker;
struct legion_struct *Defender;
  {

    short i;

    /* basic and special terrain should be the same */
    Attacker->Terrain[T_BASIC] = Defender->Terrain[T_BASIC];
    Attacker->Terrain[T_SPECIAL] = Defender->Terrain[T_SPECIAL];

    /* the attacker cannot have any fortifications */
    if(is_forts(Attacker))
      { /* attacker has forts */

        for(i = 0; i < LASTFORTS; i++)
          { /* all forts data */

            Attacker->Forts[i] = 0;

          } /* all forts data */

      } /* attacker has forts */

    /* force the defender's walls terrain to Defending Walls if there */
    /* are fortifications present */
    if(is_forts(Defender)) Defender->Terrain[T_WALLS] = 12;

    /* if defender is defending walls, attacker should be attacking walls */
    /* if defender isn't defending walls, attacker shouldn't be attacking */
    /* walls, and in no case can defender be attacking walls. */
    if(Defender->Terrain[T_WALLS] == 12)
      { /* defending walls */

        Attacker->Terrain[T_WALLS] = 13;

      } /* defending walls */
    else
      { /* no walls */

        Defender->Terrain[T_WALLS] = 0;
        Attacker->Terrain[T_WALLS] = 0;

      } /* no walls */

  } /* end of battle_terrain() */

/*************************************************************************
print_summary:
  brief summary of legion
*************************************************************************/
static void print_summary(legion)
short legion;
  {

    short slot;

    for(slot = 1; slot <= MAXSSLOTS; slot++)
      { /* all slots */

        if(Legion[legion].Slot[slot].SoldierItems[AMT] > 0)
          { /* soldiers in slot */

            printf(
                    "\n%d: %5d TRL %2d %s",
                    slot,
                    Legion[legion].Slot[slot].SoldierItems[AMT],
                    Legion[legion].Slot[slot].SoldierItems[TRL],
                    Legion[legion].Slot[slot].ItemDesc[RACE]
                  );
            if(Legion[legion].Slot[slot].ItemDesc[STATUS][0] != '\0')
              printf(" %s",
                     Legion[legion].Slot[slot].ItemDesc[STATUS]
                    );
            if(Legion[legion].Slot[slot].ItemDesc[TYPE][0] != '\0')
              printf(" %s",
                     Legion[legion].Slot[slot].ItemDesc[TYPE]
                    );
            if(Legion[legion].Slot[slot].ItemDesc[WEAPON][0] != '\0')
              printf(", %s",
                     Legion[legion].Slot[slot].ItemDesc[WEAPON]
                    );
            if(Legion[legion].Slot[slot].ItemDesc[ARMOR][0] != '\0')
              printf(", %s",
                     Legion[legion].Slot[slot].ItemDesc[ARMOR]
                    );
            if(Legion[legion].Slot[slot].ItemDesc[SHIELD][0] != '\0')
              printf(", %s",
                     Legion[legion].Slot[slot].ItemDesc[SHIELD]
                    );
            if(Legion[legion].Slot[slot].ItemDesc[MOUNT][0] != '\0')
              printf(", %s",
                     Legion[legion].Slot[slot].ItemDesc[MOUNT]
                    );

          } /* soldiers in slot */

      } /* all slots */

  } /* end of print_summary() */

/*************************************************************************
print_legion:
  debugging check on legion calculations.
*************************************************************************/
#ifdef DEBUG
static void print_legion(legion)
short legion;
  {

    short slot;

    for(slot = 1; slot <= MAXSSLOTS; slot++)
      { /* all slots */

        /* @@ Power C has a problem with long printf() variables ... */
        printf("\nSlot %d:CF=%ld",
               slot,
               Legion[legion].Slot[slot].Total[CF]);
        printf(" DF=%ld",
               Legion[legion].Slot[slot].Total[DF]);
        printf(" AF=%ld",
               Legion[legion].Slot[slot].Total[AF]);
        printf(" MAF=%ld",
               Legion[legion].Slot[slot].Total[MAF]);
        printf(" CAF=%ld",
               Legion[legion].Slot[slot].Total[CAF]);
        printf(" MAV=%ld",
               Legion[legion].Slot[slot].Character[MAV]);
        printf(" SAV=%ld",
               Legion[legion].Slot[slot].Character[SAV]);
        printf(" MAD=%ld",
               Legion[legion].Slot[slot].Total[MAD]);
        printf(" SAD=%ld",
               Legion[legion].Slot[slot].Total[SAD]);

      } /* all slots */

  } /* end of print_legion() */
#endif

/*************************************************************************
main program
*************************************************************************/
void main(argc, argv)
int argc;
char *argv[];
  {

    char *Program = "Legends Battle Simulator";
#ifdef NOCHARGE
    char *Version = "2.0.0NC (no Charge Round CAFB)";
#else
    char *Version = "2.0.0";
#endif
    char *Date = "10-Mar-94";
    char *Author = "Mark R. Nuiver";
    long t;

    (void) time(&t);
    printf("\n%s version %s of %s.", Program, Version, Date);
    printf("\nCopyright (C) 1994 by %s.  All rights reserved.", Author);
    if(argc != 3) help();
    if(strcmp(argv[1], "?") == 0) help();
    /* initialize table data */
    init_tables();
    /* load the attacking legion data */
    load_legion(argv[1], &(Legion[ATTACKER]));
    /* load the defending legion data */
    load_legion(argv[2], &(Legion[DEFENDER]));
    /* load the predefined tables for attacker */
    load_tables(&(Legion[ATTACKER]));
    /* load the predefined tables for defender */
    load_tables(&(Legion[DEFENDER]));
    /* make the terrain compatible */
    battle_terrain(&(Legion[ATTACKER]), &(Legion[DEFENDER]));
    /* validate the legion data with the (revised) terrains */
    validate_legion(&(Legion[ATTACKER]));
    validate_legion(&(Legion[DEFENDER]));
    /* calculate the legion totals for attacker */
    compute_legion(&(Legion[ATTACKER]));
    /* calculate the legion totals for defender */
    compute_legion(&(Legion[DEFENDER]));
    printf("\n\n%s attacks %s on %s", argv[1], argv[2], ctime(&t));
    printf("\nSummary of Attacking Force:\n");
    print_summary(ATTACKER);
    printf("\n");
    print_terrain(&(Legion[ATTACKER].Terrain[0]));
    print_reserve(&(Legion[ATTACKER].Reserve[0]));
    print_losses(&(Legion[ATTACKER].Losses[0]));
    printf("\n\nSummary of Defending Force:\n");
    print_summary(DEFENDER);
    printf("\n");
    print_terrain(&(Legion[DEFENDER].Terrain[0]));
    print_reserve(&(Legion[DEFENDER].Reserve[0]));
    print_losses(&(Legion[DEFENDER].Losses[0]));
#ifdef DEBUG
    printf("\nAttacking legion:");
    print_legion(ATTACKER);
    printf("\nDefending legion:");
    print_legion(DEFENDER);
#endif
    print_battle_report(fight());
    cleanup(ERROR_NO_ERROR, NULL, NULL);
  } /* end of main() */
