/*
 * File: screen.c
 * Author: Douglas Selph
 * Date: Apr 1992
 * Purpose:
 *
 *   Provide a io interface using 'curses'.
 */
#include <stdio.h>
#include <curses.h>
#include <assert.h>
#include "common.h"
#include "screen.h"
/**
 **  Local Structures and Defines
 **/
static struct screen_data
{
    struct default_def
    {
      int (*func)();
      pointer arg;
    } def, redraw;

    struct flags {
      Boolean inited;
    } flag;
} gScr;

/**
 ** Functions:
 **/
static int read_next_line();
extern int g_cur_display;
extern char g_msg[MAX_MSG];
WINDOW *g_data_win[2];
int g_data_height;
int g_data_width[2];
int g_map_win;
int g_file_win;
int g_insert_mode;
int g_num_col;
/* 
 *  Intialize the screen
 */
ScreenInit()
{
    int row, col;

    initscr();
    cbreak();
    noecho();

    gScr.flag.inited = TRUE;
    row = 1;
    col = 0;
    g_data_height = LINES-2;
    g_data_width[0] = (g_data_height-4)*2+1;
    g_data_width[1] = COLS-g_data_width[0];

    /* If this is true we have 40X80 screen or something ridiculous like that */
    if (g_data_width[1] < 10)
    {
      g_data_width[0] = (COLS/4)*3;
      g_data_width[1] = COLS-g_data_width[0];
    }
    g_map_win = 0;
    g_file_win = 1;
    g_num_col = 2;

    g_data_win[0] = newwin(g_data_height, g_data_width[0], 
			   row, col);
    g_data_win[1] = newwin(g_data_height, g_data_width[1], 
			   row, col+g_data_width[0]);
    
    assert(g_data_win[0]);
    assert(g_data_win[1]);

    FileInit();

    return(DONE);
}

ScreenResize()
{
    int row, col;

    if (g_data_win[0])
      delwin(g_data_win[0]);
    if (g_data_win[1])
      delwin(g_data_win[1]);

    g_data_win[0] = 0;
    g_data_win[1] = 0;

    row = 1;
    col = 0;

    g_num_col = g_num_col % 3;	/* 0, 1 or 2 only */

    switch(g_num_col)
    {
      case 2 :
	g_data_width[0] = (g_data_height-4)*2+1;
	break;
      case 1 :
	g_data_width[0] = (g_data_height-4)+1;
	break;
      case 0 :
	g_data_width[0] = 0;
	break;
    }
    g_data_width[1] = COLS-g_data_width[0];

    /* If this is true we have 40X80 screen or something ridiculous like that */
    if (g_data_width[1] < 10)
    {
      g_data_width[0] = (COLS/4)*3;
      g_data_width[1] = COLS-g_data_width[0];
    }
    if (g_data_width[0] > 0)
    {
      g_data_win[0] = newwin(g_data_height, g_data_width[0], 
			     row, col);

      assert(g_data_win[0]);
    }
    g_data_win[1] = newwin(g_data_height, g_data_width[1], 
			   row, col+g_data_width[0]);

    assert(g_data_win[1]);
    
    PlanetMapResized();

    ScreenRedraw();

    return(DONE);
}

WINDOW *MapWin()
{
  assert(g_data_win[g_map_win]);
  return(g_data_win[g_map_win]);
}
WINDOW *FileWin()
{
  assert(g_data_win[g_file_win]);
  return(g_data_win[g_file_win]);
}
ScreenDataWinHeight()
{
  return g_data_height;
}
MapWinWidth()
{
  return g_data_width[g_map_win];
}
FileWinWidth()
{
  return g_data_width[g_file_win];
}

ScreenPlaceCursor()
{
  switch(g_cur_display)
  {
    case DISPLAY_MAP :
      PlanetPlaceCursor();
      break;
    case DISPLAY_FILE :
      FilePlaceCursor();
      break;
  }
  return(DONE);
}

ScreenTerm()
{
    if (!gScr.flag.inited)
      return;

    clear();
    refresh();
    endwin();    
    printf("\n");
}

ScreenRedraw()
{
    MsgRedisplay();
    FileRedisplay();
    PlanetRedisplay();
    ScreenPlaceCursor();
}

/*
 * Purpose:
 *
 *   Examine the passed inputted character against
 *   all registered callbacks.
 */
ScreenMainLoop()
{
    int chr;

    while(1)
    {
      chr = getch();

      if (ScreenProcessAccl(chr))
	continue;

      if (gScr.def.func)
	(*gScr.def.func)(0, chr, gScr.def.arg);
    }
}

ScreenProcessAccl(i_chr)
int i_chr;
{
    int num;

    num = ScreenAcclNewChr(i_chr);
    /*
     *  Did we get a pulldown menu?
     */
    if (ScreenChkPulldown(i_chr))
      return(TRUE);

    if (ScreenChkItem(i_chr))
      return(TRUE);
    
    if (!ScreenAcclPossible())
    {
      ScreenAcclClear();
      /* 
       * If there was something in the accleration
       * string before we started this function try
       * matching the above character again to see if
       * we can match another acceleration.
       */
      if (num > 1)
        return ScreenProcessAccl(i_chr);
    }
    return(FALSE);
}

ScreenSetDefaultCallback(I_func, arg)
int (*I_func)();
pointer arg;
{
    gScr.def.func = I_func;
    gScr.def.arg = arg;
}

ScreenSetRedrawCallback(I_func, arg)
int (*I_func)();
pointer arg;
{
    gScr.redraw.func = I_func;
    gScr.redraw.arg = arg;
}

ScreenCallRedraw()
{
    if (*gScr.redraw.func != 0)
      (*gScr.redraw.func)(gScr.redraw.arg);
}

ScreenJumpWorld(i_ymod, i_xmod)
int i_ymod;
int i_xmod;
{
  switch(g_cur_display)
  {
    case DISPLAY_MAP  : return PlanetJumpWorld(i_ymod, i_xmod);
    case DISPLAY_FILE : return FileJumpWorld(i_ymod, i_xmod);
  }
  return(NOT_OKAY);
}


/*  Move the cursor inside the grid */
ScreenCursorMove(i_ymod, i_xmod)
int i_xmod, i_ymod;
{
  switch(g_cur_display)
  {
    case DISPLAY_MAP  : return PlanetCursorMove(i_ymod, i_xmod);
    case DISPLAY_FILE : return FileCursorMove(i_ymod, i_xmod);
  }
  return(NOT_OKAY);
}
/*  Move the cursor inside the grid */
ScreenCursorJump(i_ymod, i_xmod)
int i_xmod, i_ymod;
{
  switch(g_cur_display)
  {
    case DISPLAY_MAP  : return PlanetCursorJump(i_ymod, i_xmod);
    case DISPLAY_FILE : return FileCursorJump(i_ymod, i_xmod);
  }
  return(NOT_OKAY);
}


