#include <stdio.h>
#include <ctype.h>
#include <assert.h>
#include "global.h"

#define MAX_TITLES 20
#define SHIP_FILE "save.ships"
#define MAX_SHIP_LEN 15

static struct ships
{
  char *name;
  float drive, weapon, shield, cargo;
  int attacks;
  float mass;
  float ly, ly_full;
  float attack_pow, defense, cost, eff_cargo;
  float eff_defense; 	/* after cargo effects */
  int race;		/* which race */
  int set;
} ship[NUM_SHIPS+1];

extern int line_num;
extern int do_more;
extern int galaxy_version;
extern int g_show_eff;
char *strchr();

test_for_ship_list()
{
    FILE *fp;

    if ((fp = fopen("shiplist", "r")) == 0)
      return 0;

    printf("Warning!  The shiplist is now obsolete!\n");
    printf("  Instead the ship types can be read in directly from the turn report.\n");
    printf("  To update do the following:\n");
    printf("    (1) Remove the files 'shiplist' and 'save.ships'.\n");
    printf("    (2) Restart this program.\n");
    printf("    (3) Read in your latest turn report using the 'rt' command.\n");
    printf("    (4) Quit using the 'q' command.\n");
    printf("The file 'save.ships' will now contain ship descriptions for every race!\n");
    exit(1);
}

init_ships()
{
    read_ship_data(SHIP_FILE);
}

save_ships()
{
   int i;
   do_more = 0;
   write_ship_data(SHIP_FILE);
   do_more = 1;
}

write_ship_data(cp_file)
char *cp_file;
{
  FILE *fp;
  int i;

  if ((fp = fopen(cp_file, "w")) == NULL)
  {
    perror(cp_file);
    return;
  }
  for(i = 0; i < get_num_races(); i++)
    write_all_ships(fp, i, TECH_ALL);

  printf("Wrote %s\n", cp_file);

  fclose(fp);
}

get_ship_race(start)
char *start;
{
  int i, id;
  char line[80];
  char *s;

  if (get_num_races() == 1)
    return(0);

  if (start != 0)
  {
    strcpy(line, start);
    if ((s = strchr(line, ',')) != 0)
      *s = '\0';

    if (strIsInt(line))
    {
      id = atoi(line);
      return(chk_race_id(id)); 
    }
  }
  while(1)
  {
    for (i = 0; i < get_num_races(); i++)
      printf(" %d - %s\n", i, get_race_name(i));

    printf("> ");
    gets(line);

    if ((chk_race_id(id = atoi(line))) >= 0)
      return(id);
  }
}

get_tech_level(race, start)
int race;
char *start;
{
  int i, id;
  char line[80];
  char *s;
  float drive, weapon, shield, cargo;

  if (get_num_techs(race) <= 1)
    return(0);

  if (start != 0)
  {
    if ((s = strchr(start, ',')) != 0)
    {
      if (strIsInt(s+1))
      {
	id = atoi(s+1);
	return(chk_tech_level(race,id)); 
      }
    }
  }

  printf("Select Tech Level\n");

  while(1)
  {
    for (i = 0; i < get_num_techs(race); i++)
    {
      get_race_techs(race, i, &drive, &weapon, &shield, &cargo);
      printf(" %d - %5.2f %5.2f %5.2f %5.2f\n", i,
          drive, weapon, shield, cargo);
    }
    printf("> ");
    gets(line);

    if ((chk_tech_level(race, id = atoi(line))) >= 0)
      return(id);
  }
}

print_ships(start)
char *start;
{
  int race;
  int techlevel;

  if ((race = get_ship_race(start)) < 0)
    return;
  if ((techlevel = get_tech_level(race, start)) < 0)
    return;

  write_all_ships(stdout, race, techlevel);
}

write_all_ships(fp, race, techlevel)
FILE *fp;
int race;
int techlevel;
{
  line_num = 0;
  write_ship_type(fp, race, techlevel);
}

write_ship_type(fp, race, techlevel)
FILE *fp;
int race;
int techlevel;
{
  int i, j;
  char line[256];
  char *title;
  float drive, weapon, shield, cargo;

  if (techlevel == TECH_ALL)
  {
    fprintf(fp, "\n\t\t%s %s\n", title = get_race_name(race),
	SHIP_TYPES);
    
    for (i = 0; i < get_num_techs(race); i++)
    {
      get_race_techs(race, i, &drive, &weapon, &shield, &cargo);

      fprintf(fp, "\t\tTech Level %5.2f %5.2f %5.2f %5.2f\n",
          drive, weapon, shield, cargo);
    }
    chk_more(i);
    fprintf(fp, "\n");
    chk_more(1);

    techlevel = 0;
  }
  else
  {
    assert(techlevel >= 0 && (techlevel < get_num_techs(race)));

    get_race_techs(race, techlevel, &drive, &weapon, &shield, &cargo);

    fprintf(fp, "\t\t%s %5.2f %5.2f %5.2f %5.2f\n",
	get_race_name(race),
	drive, weapon, shield, cargo);
    
    chk_more(1);
  }
  show_ship_header(fp);

  for (i = 0; i < NUM_SHIPS; i++)
  {
    if (ship[i].set && (ship[i].race == race))
    {
      calc_ship_data(i, techlevel);
      show_ship_line(i, line);
      fprintf(fp, "%s\n", line);
      if (fp == stdout) chk_more(1);
    }
  }
}

show_ship_header(fp)
FILE *fp;
{
  fprintf(fp, 
  "Name            Drive A Weap  Sh    Cargo Mass    LY/full   AT    DA    Cost\n"); 
  if (fp == stdout) chk_more(1);
}

show_ship_line(i, line)
int i;
char *line;
{
    char tmp[256];

    sprintf(line, "%-*s %-5.2f %-1d %-5.2f %-5.2f %-5.2f %-5.2f",
      MAX_SHIP_LEN,
      get_ship_name(i), 
      ship[i].drive, 
      ship[i].attacks,
      ship[i].weapon, 
      ship[i].shield,
      (g_show_eff ? ship[i].eff_cargo : ship[i].cargo),
      ship[i].mass);

    if (ship[i].ly == ship[i].ly_full)
    {
      sprintf(tmp, "   %-5.2f     ", ship[i].ly);
      strcat(line, tmp);
    }
    else
    {
      sprintf(tmp, " %-5.2f/%-5.2f ",
	ship[i].ly,
	ship[i].ly_full);
      strcat(line, tmp);
    }

    sprintf(tmp, "%-5.2f %-5.2f %-5.2f",
      ship[i].attack_pow,
      (g_show_eff ? ship[i].eff_defense : ship[i].defense),
      ship[i].cost);

    strcat(line, tmp);
}

get_ship_id(cp_name, race)
char *cp_name;
int race;
{
    int i;

    /* See if there is a ship of this name already */
    for (i = 0; i < NUM_SHIPS;  i++)
      if (ship[i].set && (ship[i].race == race) &&
	  !strcmp(ship[i].name, cp_name))
	return(i);

    return(IS_ERR);
}

get_new_ship_id()
{
    int i;

    /* Get first unset slot */
    for (i = 0; i < NUM_SHIPS;  i++)
      if (!ship[i].set)
	return(i);

    printf("Out of Room for new ships!\n");
    exit(1);
}

char *get_ship_name(id)
int id;
{
    static char text[20];

    if (id == TEST_SLOT)
      return "Test Ship";

    if ((id < 0) || (id >= NUM_SHIPS+1))
    {
      printf("get_ship_name(), line %d : Illegal ship number: %d\n",
	line_num, id);
      exit(1);
    }

    if (ship[id].name != NULL)
    {
      strncpy(text, ship[id].name, MAX_SHIP_LEN);
      text[MAX_SHIP_LEN] = '\0';
      return text;
    }
}

new_ship_data(cp_name, drive, attacks, weapon, shield, cargo, race)
char *cp_name;
float drive, weapon, shield, cargo;
int attacks, race;
{
    int id;

    if ((id = get_ship_id(cp_name, race)) == IS_ERR)
      id = get_new_ship_id();

    ship[id].set = 1;
    zrealloc_cpy(&ship[id].name, cp_name);

    set_ship_data(id, drive, attacks, weapon, shield, cargo, race, 0);
}

set_ship_data(id, drive, attacks, weapon, shield, cargo, race, techlev)
int id;
float drive, weapon, shield, cargo;
int attacks, race;
int techlev;
{
    ship[id].drive = drive;
    ship[id].attacks = attacks;
    ship[id].weapon = weapon;
    ship[id].shield = shield;
    ship[id].cargo = cargo;
    ship[id].race = race;

    calc_ship_data(id, techlev);
}

calc_ship_data(id, techlev)
int id;
int techlev;
{
    float drive, weapon, shield, cargo;
    float tdrive, tweapon, tshield, tcargo;
    float cargo_mass, mass, ly, ly_full, defense, cost, eff_cargo;
    float eff_defense;
    double cbrt();
    int attacks;
    int race;

    race = ship[id].race;
    drive = ship[id].drive;
    attacks = ship[id].attacks;
    weapon = ship[id].weapon;
    shield = ship[id].shield;
    cargo = ship[id].cargo;

    get_race_techs(race, techlev, &tdrive, &tweapon, &tshield, &tcargo);

    if (galaxy_version >= 3)
    {
      mass = drive + shield + cargo;
      if (attacks >= 1) 
      {
	mass += weapon;
	if (attacks > 1)
	  mass += ((attacks-1) * weapon) / 2.0;
      }
    }
    else
    {
      mass = drive + attacks * weapon + shield + cargo;
    }
    drive *= tdrive;
    weapon *= tweapon;
    shield *= tshield;
    cargo = COMPUTE_CARGO(cargo) * tcargo;
    cargo_mass = mass + cargo/tcargo;

    ly = (drive * 20.0) / mass;
    ly_full = (drive * 20.0) / cargo_mass;
    defense = shield / cbrt(mass) * 3.107;
    eff_defense = shield / cbrt(cargo_mass) * 3.107;
    cost = 10 * mass;

    ship[id].mass = mass;
    ship[id].ly = ly;
    ship[id].ly_full = ly_full;
    ship[id].defense = defense;
    ship[id].attack_pow = weapon;
    ship[id].cost = cost;
    ship[id].eff_cargo = cargo;
    ship[id].eff_defense = eff_defense;
}

get_ship_data(id, cp_basics)
int id;
char *cp_basics;
{
    if (ship[id].set)
    {
      sprintf(cp_basics, "%5.2f %d %5.2f %5.2f %5.2f",
	ship[id].drive, ship[id].attacks, ship[id].weapon,
	ship[id].shield, ship[id].cargo);
    }
    else
      cp_basics[0] = '\0';
}
get_ship_brief_data(id, cp_basics)
int id;
char *cp_basics;
{
    if (ship[id].set)
    {
      if (ship[id].ly == ship[id].ly_full)
      {
	if (ship[id].attacks > 1)
	{
	  sprintf(cp_basics, "(%5.2f,%5.2f*%d,%5.2f,%5.2f)",
	    ship[id].ly, ship[id].attack_pow, ship[id].attacks,
	      (g_show_eff ? ship[id].eff_defense : ship[id].defense), 
	      (g_show_eff ? ship[id].eff_cargo : ship[id].cargo));
	}
	else
	{
	  sprintf(cp_basics, "(%5.2f,%5.2f,%5.2f,%5.2f)",
	    ship[id].ly, ship[id].attack_pow, 
	      (g_show_eff ? ship[id].eff_defense : ship[id].defense), 
	      (g_show_eff ? ship[id].eff_cargo : ship[id].cargo));
	}
      }
      else
      {
	if (ship[id].attacks > 1)
	{
	  sprintf(cp_basics, "(%.2f/%.2f,%.2f*%d,%.2f,%.2f)",
	    ship[id].ly, ship[id].ly_full, ship[id].attack_pow,
	    ship[id].attacks, 
		(g_show_eff ? ship[id].eff_defense : ship[id].defense), 
		(g_show_eff ? ship[id].eff_cargo : ship[id].cargo));
	}
	else
	{
	  sprintf(cp_basics, "(%.2f/%.2f,%.2f,%.2f,%.2f)",
	    ship[id].ly, ship[id].ly_full, ship[id].attack_pow,
	        (g_show_eff ? ship[id].eff_defense : ship[id].defense), 
		(g_show_eff ? ship[id].eff_cargo : ship[id].cargo));
	}
      }
    }
    else
      cp_basics[0] = '\0';
}


get_ship_race_value(id)
int id;
{
  return ship[id].race;
}
float get_ship_mass(id)
int id;
{
  return ship[id].mass;
}

double get_ship_ly(id)
int id;
{
   return ship[id].ly;
}

#define LEVEL1 0
#define LEVEL2 2000
#define LEVEL3 4000
#define LEVEL4 6000
#define UNSET 10000

shipval(e1)
struct ships *e1;
{
  int val;

  if (!e1->set)
    return(UNSET);
  
  if (e1->cargo > 0)
  {
    return e1->mass + e1->cargo + LEVEL1;
  }
  if (e1->drive == 0)
  {
    return e1->mass + LEVEL4;
  }
  if (e1->attacks * e1->weapon == 0)
  {
    return e1->mass + LEVEL2;
  }
  return e1->mass + LEVEL3;
}

shippos(e1, e2)
struct ships *e1;
struct ships *e2;
{
  int val1, val2;

  val1 = shipval(e1);
  val2 = shipval(e2);

  return(val1 - val2);
}

sort_ships()
{
  qsort((char *) ship, NUM_SHIPS, sizeof(struct ships), shippos);
}

display_ship(id)
int id;
{
  char line[245];
  show_ship_header(stdout);
  show_ship_line(id, line);
  printf("%s\n", line);
}
