/**********************************************************************
 * SCRLOG.C                          Copyright (C) Damian Walker 1997 *
 *--------------------------------------------------------------------*
 * AstroWar 1.00 - Screen and Log Output Module.                      *
 *--------------------------------------------------------------------*
 * Author   Damian G Walker                                           *
 * Date     09-Apr-97                                                 *
 **********************************************************************/


/* included headers ***************************************************/


#include <stdio.h>
#include "astrowar.h"
#include "log.h"


/* global variables ***************************************************/


extern char pid[36];


/* main open/close routines *******************************************/


/* scrlog_open() - open screen and log */
result scrlog_open(void)
{
    setbuf(stdout, NULL);
    printf("\nCenturion %s\n", pid);
    puts(  "Copyright (C) Damian Walker 1997");
    puts(  "Fidonet 2:2502/666");
    puts(  "");

    return R_OK;
}

/* scrlog_openlog() - open the log file */
result scrlog_openlog(char *filename, short status)
{
    char fullpid[81];   /* "Centurion " + pid */

    sprintf(fullpid, "Centurion %s", pid);
    if( openlog(filename, fullpid) == LOGFILEERROR ) return R_FILE;
    setlogstatus(status);

    return R_OK;
}

/* scrlog_close() - close the library */
void scrlog_close(void)
{
    closelog();
}

/* void scrlog_cannotopenlog() - cannot open the log file */
void scrlog_cannotopenlog(char *filename)
{
    printf("! Cannot open log file %s\n", filename);
}


/* environment-related messages ***************************************/


/* scrlog_error() - command line error */
void scrlog_error(void)
{
    puts("! Command line error: type \'astrowar ?\' for help");
}

/* scrlog_running() - program already running */
void scrlog_running(void)
{
    puts("! AstroWar already running");
}

/* scrlog_help() - program help output */
void scrlog_help(void)
{
    puts("usage: astrowar <commands> [<switches>]");
    puts("");
    puts("commands =");
    puts("    create      process inbound order messages");
    puts("    turn        generate outbound report messages");
    puts("    ?           request this help");
    puts("");
    puts("switches =");
    puts("    -path <p>   path to configs, semaphores etc.");
    puts("    -game <n>   name of game to create or process");
}


/* general messages ***************************************************/


/* scrlog_create() - running turn input module */
void scrlog_create(void)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Running game creation module");
    printf("* %s\n", entry);
    writelog(LOGMAJOR, entry);
}

/* scrlog_doturn() - running turn output module */
void scrlog_doturn(void)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Running turn processing module");
    printf("* %s\n", entry);
    writelog(LOGMAJOR, entry);
}

/* scrlog_outofmemory() - general out of memory error */
void scrlog_outofmemory(char *act)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Out of memory %s", act);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_badreturncode() - unexpected result code from function */
void scrlog_badreturncode(result r, char *function)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Bad return code %d from %s", r, function);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_finished() - finished */
void scrlog_finished(void)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Finished!");
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_planetsdone() - done! */
void scrlog_done(void)
{
    puts("done!");
}


/* config-related messages ********************************************/


/* scrlog_badcommandinconfig() - bad command in config file */
void scrlog_badcommandinconfig(char *path, int line)
{
    printf("! Bad command in line %d of %sastrowar.data", line, path);
}

/* scrlog_cannotreadconfig() - file error opening config */
void scrlog_cannotreadconfig(char *path)
{
    printf("! Cannot read config file %sastrowar.data\n", path);
}

/* scrlog_cannotwriteconfig() - file error creating config */
void scrlog_cannotwriteconfig(char *path)
{
    printf("! Cannot write config file %sastrowar.data\n", path);
}

/* scrlog_configread() - config read successfully */
void scrlog_configread(char *path)
{
    char entry[128]; /* line for log entry */

    sprintf(entry, "Config file %sastrowar.data read successfully", path);
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}


/* detail-related errors **********************************************/


/* scrlog_cannotopendetail() - cannot read detail file */
void scrlog_cannotopendetail(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Cannot open detail file %sdetail.data", path);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_badcommandindetail() - bad command in detail file */
void scrlog_badcommandindetail(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Bad command in detail file %sdetail.data", path);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_detailread() - detail file read successfully */
void scrlog_detailread(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Detail file %sdetail.data read successfully", path);
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}


/* game list file related messages ************************************/


/* scrlog_cannotopengamelist() - error opening game list */
void scrlog_cannotopengamelist(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Cannot open game list %sgamelist.data", path);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_badfileingamelist() - bad command in game list */
void scrlog_badfileingamelist(char *path, int line)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Bad detail file (line %d in %sgamelist.data)", line,
        path);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_badlineingamelist() - bad command in game list */
void scrlog_badlineingamelist(char *path, int line)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Bad line %d in game list %sgamelist.data", line,
        path);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_gamelistread() - game list read successfully */
void scrlog_gamelistread(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Game list %sgamelist.data read successfully", path);
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_gamelistempty() - game list empty */
void scrlog_gamelistempty(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Game list %sgamelist.data is empty", path);
    printf("? %s\n", entry);
    writelog(LOGWARNING, entry);
}

/* scrlog_gamenotfound() - game not found */
void scrlog_gamenotfound(char *gamename)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Game \"%s\" not found", gamename);
    printf("! %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_cannotwritegamelist() - cannot write game list */
void scrlog_cannotwritegamelist(char *path)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Cannot write game list %sgamelist.data", path);
    printf("? %s\n", entry);
    writelog(LOGWARNING, entry);
}


/* general file-related messages **************************************/


/* scrlog_cannotopenfile() - cannot open file.* */
void scrlog_cannotopenfile(char *path, char *file)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Cannot open file %s%s", path, file);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_badheaderinfile() - bad header in file.* */
void scrlog_badheaderinfile(char *path, char *file)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Bad header in file %s%s", path, file);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_fileopened() - join request file opened */
void scrlog_fileopened(char *path, char *file)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "File %s%s opened successfully", path, file);
    printf("- %s\n", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_fileexists() - join request file exists */
void scrlog_fileexists(char *path, char *file)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "File %s%s already exists", path, file);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_cannotcreatefile() - cannot create join request file */
void scrlog_cannotcreatefile(char *path, char *file)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Cannot create file %s%s", path, file);
    printf("! %s\n", entry);
    writelog(LOGERROR, entry);
}

/* scrlog_filecreated() - join request file created */
void scrlog_filecreated(char *path, char *file)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "File %s%s created successfully", path, file);
    printf("+ %s\n", entry);
    writelog(LOGMINOR, entry);
}


/* game creation messages *********************************************/


/* scrlog_creatingplanet() - creating planet ... */
void scrlog_creatingplanet(long curr, long planets)
{
    printf(". Creating planet %ld of %ld ...\r", curr, planets);
}

/* scrlog_planetsdone() - planets created */
void scrlog_planetsdone(long planets)
{
    char entry[128]; /* line for screen and log */

    sprintf(entry, "%ld planets created", planets);
    printf("+ %-72s\n", entry);
    writelog(LOGMINOR, entry);
}


/* turn processing messages *******************************************/


/* scrlog_movingfleet() - moving fleet ... */
void scrlog_movingfleet(long fleetno)
{
    char entry[128]; /* line for log and screen */

    sprintf(entry, "Moving fleet %ld", fleetno);
    printf("- %s ...\r", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_fleetsdone() - fleets moved */
void scrlog_fleetsdone(void)
{
    char entry[128]; /* line for screen and log */

    sprintf(entry, "All fleets moved");
    printf("+ %-72s\n", entry);
    writelog(LOGMINOR, entry);
}

/* scrlog_processingorder() - processing ... order ... */
void scrlog_processingorder(char type)
{
    char entry[128]; /* line for log and screen */

    switch(type)
    {
        case 'J': sprintf(entry, "Processing join order");    break;
        case 'S': sprintf(entry, "Processing send order");    break;
        case 'P': sprintf(entry, "Processing planets order"); break;
        case 'F': sprintf(entry, "Processing fleets order");  break;
        case 'M': sprintf(entry, "Processing map order");     break;
        case 'W': sprintf(entry, "Processing write order");   break;
        case 'R': sprintf(entry, "Processing resign order");  break;
        default:  sprintf(entry, "Processing unknown order");
    }
    printf("- %s ...     \r", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_ordersdone() - processed all orders */
void scrlog_ordersdone(void)
{
    char entry[128]; /* line for screen and log */

    sprintf(entry, "All orders processed");
    printf("+ %-72s\n", entry);
    writelog(LOGMINOR, entry);
}

/* scrlog_processingplanet() - moving fleet ... */
void scrlog_processingplanet(char *name)
{
    printf(". Processing production for %s ...%15s\r", name, "");
}

/* scrlog_proddone() - prod moved */
void scrlog_productiondone(void)
{
    char entry[128]; /* line for screen and log */

    sprintf(entry, "All planets processed");
    printf("+ %-72s\n", entry);
    writelog(LOGMINOR, entry);
}

/* scrlog_checkingvictory() - checking victory */
void scrlog_checkingvictory(void)
{
    char entry[128]; /* line for screen and log */

    sprintf(entry, "Checking victory");
    printf("- %s\r", entry);
    writelog(LOGTRIVIAL, entry);
}

/* scrlog_victory() - victory for ... */
void scrlog_victory(char *vname)
{
    char entry[128]; /* line for screen and log */

    sprintf(entry, "Victory for %s", vname);
    printf("+ %-72s\n", entry);
    writelog(LOGMINOR, entry);
}

/* scrlog_novictory() - victory for ... */
void scrlog_novictory(void)
{
    char entry[128]; /* line for screen and log */

    sprintf(entry, "No victory this turn");
    printf("+ %-72s\n", entry);
    writelog(LOGMINOR, entry);
}
