/**************************************************************
 * EMPIRE.C                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroWar 1.00 - Empire database library.                   *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     28-Apr-97                                         *
 **************************************************************/


#define _EMPIRE_C_


/* included headers *******************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "astrowar.h"
#include "empire.h"


/* some macros ************************************************/


/* record sizes */
#define DATASIZE 33  /* size of packed data record */
#define IND1SIZE 20  /* size of packed fleet number index */

/* flags */
#define JF_DELETED 0x01 /* deleted flag */


/* structures *************************************************/


/* hidden empire structure */
typedef struct {

    /* record fields */
    char deleted,     /* 'deleted' flag */
         name[16],    /* empire name */
         homewld[16]; /* empire's homeworld */

    /* index maintenance fields */
    char  infile;    /* 0=new record, !0=somewhere in file */
    long  pos;       /* record position when last read */
    char  oname[16];  /* original empire name */

} empire;

/* indexes */
typedef struct {   /* empire number index */
    long pos;          /* position of record in data file */
    char name[16];     /* empire name */
} ind1;


/* global variables *******************************************/


FILE *empire_data, /* empire data file */
     *empire_ind1; /* empire number index file */


/* level 3 routines *******************************************/


/* empire_readind1() - read a number index record */
result empire_readind1(ind1 *i)
{
    long pos; /* first byte of position in file */

    if(( pos = fgetc(empire_ind1) ) == EOF)
        return R_EOF;
    i->pos = pos;
    FREADLONGR(i->pos, empire_ind1);
    fread(i->name, 16, 1, empire_ind1);
    return R_OK;
}


/* level 2 routines *******************************************/


/* empire_findind1() - find a trans index record */
long empire_findind1(ind1 *i, char *name)
{
    long pos,    /* original position in index file */
         start,  /* lower bound of binary search area */
         middle, /* midpoint of binary search area */
         end;    /* upper bound of binary search area */

    /* store current file pointer in case of no find */
    pos = ftell(empire_ind1); fseek(empire_ind1, 0, SEEK_END);
    if( ftell(empire_ind1) == 8 ) return 0;

    /* initialise start/end/middle of binary search */
    end = ( ftell(empire_ind1) - 8 ) / IND1SIZE - 1;
    start = 0; middle = start + (end - start) / 2;

    /* proceed with binary search */
    fseek(empire_ind1, 8 + IND1SIZE * middle, SEEK_SET);
    empire_readind1(i);
    while( stricmp(i->name, name) && start <= end )
    {
        if( stricmp(i->name, name) < 0 )
            start = middle + 1;
        else
            end = middle - 1;
        middle = start + (end - start) / 2;
        fseek(empire_ind1, 8 + IND1SIZE * middle, SEEK_SET);
        empire_readind1(i);
    }

    /* return value */
    if( !stricmp(i->name, name) )
        return 8 + middle * IND1SIZE;
    fseek(empire_ind1, pos, SEEK_SET);
    return 0;
}

/* empire_writeind1() - write a number index record */
result empire_writeind1(ind1 i)
{
    if( fputc(i.pos & 0xff, empire_ind1) == EOF ) return R_FILE;
    FWRITELONGR(i.pos,  empire_ind1);
    fwrite(i.name, 16, 1, empire_ind1);
    return R_OK;
}


/* level 1 routines *******************************************/


/* empire_readdata() - read a data record */
result empire_readdata(empire *e)
{
    int  deleted; /* deleted flag */
    long pos;     /* position in file */

    /* check for EOF */
    pos = ftell(empire_data);
    if(( deleted = fgetc(empire_data) ) == EOF)
        return R_EOF;

    /* read rest of record */
    e->deleted = deleted;
    fread(e->name,    16, 1, empire_data);
    fread(e->homewld, 16, 1, empire_data);

    /* set index maintenance fields and return */
    e->infile = 1;
    e->pos = pos;
    strcpy(e->oname, e->name);
    return R_OK;
}

/* empire_writedata() - write a data record */
result empire_writedata(empire *e)
{
    long pos; /* position in file */

    /* check that record can be written */
    pos = ftell(empire_data);
    if( fputc(e->deleted, empire_data) == EOF )
        return R_FILE;

    /* write rest of record */
    fwrite(e->name,    16, 1, empire_data);
    fwrite(e->homewld, 16, 1, empire_data);

    /* set index maintenance fields and return */
    e->infile = 1;
    e->pos = pos;
    strcpy(e->oname, e->name);
    return R_OK;
}

/* empire_sortind1() - single bi-directional pass of sort */
int empire_sortind1(void)
{
    long   pos;      /* stored position in index */
    char   name[16]; /* stored empire name */
    ind1   i1,       /* first index record to compare */
           i2;       /* second index record to compare */
    result r;        /* returned to calling process */

    /* obtain information about current position */
    pos = ftell(empire_ind1);
    if(pos > 8)
    {
        fseek(empire_ind1, -IND1SIZE, SEEK_CUR);
        empire_readind1(&i1);
        strcpy(name, i1.name);
    }
    else strcpy(name, "");

    /* ensure file is big enough to need sorting */
    fseek(empire_ind1, 0, SEEK_END);
    if( ftell(empire_ind1) <= 8 + IND1SIZE )
    {
        fseek(empire_ind1, pos, SEEK_SET);
        return R_OK;
    }
    r = R_OK;

    /* forward pass of sort */
    fseek(empire_ind1, 8, SEEK_SET); empire_readind1(&i1);
    while( empire_readind1(&i2) == R_OK )
    {
        if( stricmp(i1.name, i2.name) > 0 )
        {
            fseek(empire_ind1, -2 * IND1SIZE, SEEK_CUR);
            empire_writeind1(i2); empire_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(empire_ind1, -IND1SIZE, SEEK_CUR);
        empire_readind1(&i1);
    }

    /* backward pass of sort */
    fseek(empire_ind1, -IND1SIZE, SEEK_END);
    while( ftell(empire_ind1) > 8 )
    {
        fseek(empire_ind1, -IND1SIZE, SEEK_CUR);
        empire_readind1(&i1); empire_readind1(&i2);
        if( stricmp(i1.name, i2.name) > 0 )
        {
            fseek(empire_ind1, -2 * IND1SIZE, SEEK_CUR);
            empire_writeind1(i2); empire_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(empire_ind1, -2 * IND1SIZE, SEEK_CUR);
    }

    /* clean up */
    if(pos > 8)
        empire_findind1(&i1, name);
    else
        fseek(empire_ind1, pos, SEEK_SET);
    return r;
}


/* level 0 routines *******************************************/


/* empire_open() - open empire files */
result empire_open(char *path)
{
    char dataname[128], /* name of data file */
         ind1name[128], /* name of empire number index file */
         dataheader[8], /* data file header */
         ind1header[8]; /* index 1 header */

    /* initialise data names */
    sprintf(dataname, "%sempire.data", path);
    sprintf(ind1name, "%sempire.index", path);

    /* attempt to open files */
    if(( empire_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    if(( empire_ind1 = fopen(ind1name, "r+b") ) == NULL)
    {
        fclose(empire_data);
        return R_FILE;
    }

    /* verify files */
    fread(dataheader, 8, 1, empire_data);
    fread(ind1header, 8, 1, empire_ind1);
    if(( strncmp(dataheader, "AST100E", 8) ) ||
       ( strncmp(ind1header, "AST100e", 8) ))
    {
        fclose(empire_data);
        fclose(empire_ind1);
        return R_HEADER;
    }

    /* files checked out OK */
    return R_OK;
}

/* empire_create() - create empire files */
result empire_create(char *path)
{
    char  dataname[128], /* name of data file */
          ind1name[128]; /* name of empire number index file */
    FILE *test;          /* used to test existence of files */

    /* initialise data names */
    sprintf(dataname, "%sempire.data", path);
    sprintf(ind1name, "%sempire.index", path);

    /* check for files' existence */
    if(( test = fopen(dataname, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }
    if(( test = fopen(ind1name, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }

    /* attempt to create files */
    if(( empire_data = fopen(dataname, "wb") ) == NULL)
        return R_FILE;
    if(( empire_ind1 = fopen(ind1name, "wb") ) == NULL)
    {
        fclose(empire_data); remove(dataname);
        return R_FILE;
    }

    /* write headers */
    fwrite("AST100E", 8, 1, empire_data);
    fwrite("AST100e", 8, 1, empire_ind1);

    /* close files and return */
    fclose(empire_data);
    fclose(empire_ind1);
    return R_OK;
}

/* empire_close() - close empire files */
result empire_close(void)
{
    fclose(empire_data);
    fclose(empire_ind1);
    return R_OK;
}

/* empire_new() - allocate memory for new empire record */
empire *empire_new(void)
{
    return calloc( 1, sizeof(empire) );
}

/* empire_old() - free memory for new empire record */
void empire_old(empire *e)
{
    free(e);
}

/* empire_clear() - clear details from existing empire record */
void empire_clear(empire *e)
{
    memset( e, 0, sizeof(empire) );
}

/* empire_setname() - set the empire name */
char *empire_setname(empire *e, char *name)
{
    return strcpy(e->name, name);
}

/* empire_getname() - return the empire name */
char *empire_getname(char *name, empire *e)
{
    return strcpy(name, e->name);
}

/* empire_sethomewld() - set the homeworld name */
char *empire_sethomewld(empire *e, char *homewld)
{
    return strcpy(e->homewld, homewld);
}

/* empire_gethomewld() - return the homeworld */
char *empire_gethomewld(char *homewld, empire *e)
{
    return strcpy(homewld, e->homewld);
}

/* empire_write() - write a record */
result empire_write(empire *e)
{
    ind1   i;  /* trans index record */
    long   p1; /* position of index record 1 */

    /* write new record */
    if(!e->infile)
    {
        if( empire_findind1(&i, e->name) ) return R_DUPE;
        fseek(empire_data, 0, SEEK_END);
        fseek(empire_ind1, 0, SEEK_END);
        i.pos = ftell(empire_data);
        strcpy(i.name, e->name);
        if( empire_writedata(e) != R_OK )  return R_FILE;
        if( empire_writeind1(i) != R_OK )  return R_CORRUPT;
        empire_sortind1();
        return R_OK;
    }

    /* rewrite existing record */
    p1 = empire_findind1(&i, e->oname);
    if(!p1)                                return R_CORRUPT;
    fseek(empire_data, i.pos, SEEK_SET);
    if( empire_writedata(e) != R_OK )      return R_FILE;
    if( stricmp(i.name, e->name) )
    {
        strcpy(i.name, e->name);
        fseek(empire_ind1, p1, SEEK_SET);
        if( empire_writeind1(i) != R_OK )  return R_CORRUPT;
        empire_sortind1();
    }
    return R_OK;
}

/* empire_first() - read the first record */
result empire_first(empire *e, eindex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case E_NONE:
            fseek(empire_data, 8, SEEK_SET);
            return empire_readdata(e);
        case E_NAME:
            fseek(empire_ind1, 8, SEEK_SET);
            if(( r = empire_readind1(&i) ) != R_OK) return r;
            fseek(empire_data, i.pos, SEEK_SET);
            return empire_readdata(e);
        default:
            return R_INDEX;
    }
}

/* empire_next() - read the next record */
result empire_next(empire *e, eindex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case E_NONE:
            return empire_readdata(e);
        case E_NAME:
            if(( r = empire_readind1(&i) ) != R_OK) return r;
            fseek(empire_data, i.pos, SEEK_SET);
            return empire_readdata(e);
        default:
            return R_INDEX;
    }
}

/* empire_prev() - read the previous record */
result empire_prev(empire *e, eindex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case E_NONE:
            if(( ftell(empire_data) - 2 * DATASIZE ) < 8)
                return R_EOF;
            fseek(empire_data, -2 * DATASIZE, SEEK_CUR);
            return empire_readdata(e);
        case E_NAME:
            if(( ftell(empire_ind1) - 2 * IND1SIZE ) < 8)
                return R_EOF;
            fseek(empire_ind1, -2 * IND1SIZE, SEEK_CUR);
            if(( r = empire_readind1(&i) ) != R_OK) return r;
            fseek(empire_data, i.pos, SEEK_SET);
            return empire_readdata(e);
        default:
            return R_INDEX;
    }
}

/* empire_last() - read the last record */
result empire_last(empire *e, eindex inum)
{
    ind1   i;   /* number index record */
    result r;   /* result of index read */
    long   pos; /* current data/index file position */

    switch(inum)
    {
        case E_NONE:
            pos = ftell(empire_data);
            if( fseek(empire_data, -DATASIZE, SEEK_END) )
                return R_EOF;
            if( ftell(empire_data) < 8 )
            {
                fseek(empire_data, pos, SEEK_SET);
                return R_EOF;
            }
            return empire_readdata(e);
        case E_NAME:
            pos = ftell(empire_ind1);
            if( fseek(empire_ind1, -IND1SIZE, SEEK_END) )
                return R_EOF;
            if( ftell(empire_ind1) < 8 )
            {
                fseek(empire_ind1, pos, SEEK_SET);
                return R_EOF;
            }
            if(( r = empire_readind1(&i) ) != R_OK) return r;
            fseek(empire_data, i.pos, SEEK_SET);
            return empire_readdata(e);
        default:
            return R_INDEX;
    }
}

/* empire_find() - find a record by name or name */
result empire_find(empire *e, char *name)
{
    ind1   i;      /* empire name index record */

    if( empire_findind1(&i, name) == 0 ) return R_EOF;
    fseek(empire_data, i.pos, SEEK_SET);
    return empire_readdata(e);

    return R_OK;
}

/* empire_delete() - mark a record as deleted */
result empire_delete(empire *e)
{
    if(e->pos == 0) return R_EOF;
    e->deleted ^= JF_DELETED;
    fseek(empire_data, e->pos, SEEK_SET);
    return empire_writedata(e);
}

/* empire_deleted() - return the deleted status of a record */
int empire_deleted(empire *e)
{
    if(e->pos == 0) return R_EOF;
    return e->deleted & JF_DELETED;
}

/* empire_pack() - pack a close data file */
result empire_pack(char *path)
{
    empire *e;             /* empire record */
    char    dataname[128], /* name of data file */
            ind1name[128], /* name of empire number index file */
            tempname[128], /* name of temporary file */
            dataheader[8]; /* data file header */
    FILE   *empire_temp;   /* temporary file */
    ind1    i;             /* number index record */

    /* initialise data names */
    sprintf(dataname, "%sempire.data", path);
    sprintf(tempname, "%sempire.temp", path);

    /* attempt to open data & temp file */
    if(( empire_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    fread(dataheader, 8, 1, empire_data);
    if(( strncmp(dataheader, "AST100E", 8) ) ||
       ( empire_temp = fopen(tempname, "w+b") ) == NULL)
    {
        fclose(empire_data);
        return R_HEADER;
    }

    /* allocate memory */
    if( (e = malloc( sizeof(empire) )) == NULL )
    {
        fclose(empire_data);
        fclose(empire_temp);
        remove(tempname);
        return R_MEMORY;
    }

    /* copy non-deleted records to temporary file and back */
    while( empire_readdata(e) == R_OK )
        if( !(e->deleted & JF_DELETED) )
            fwrite(e, sizeof(empire), 1, empire_temp);
    fclose(empire_data);
    remove(dataname);
    if(( empire_data = fopen(dataname, "w+b") ) == NULL)
    {
        free(e);
        fclose(empire_temp);
        remove(tempname);
        return R_FILE;
    }
    fwrite("AST100E", 8, 1, empire_data);
    fseek(empire_temp, 0, SEEK_SET);
    while( fread(e, sizeof(empire), 1, empire_temp) == 1 )
        empire_writedata(e);
    fclose(empire_temp);
    remove(tempname);

    /* recreate fleet number index */
    sprintf(ind1name, "%sempire.index", path);
    remove(ind1name);
    if(( empire_ind1 = fopen(ind1name, "w+b") ) == NULL)
    {
        free(e);
        fclose(empire_data);
        return R_FILE;
    }
    fwrite("AST100e", 8, 1, empire_ind1);
    fseek(empire_data, 8, SEEK_SET); i.pos = ftell(empire_data);
    while( empire_readdata(e) == R_OK )
    {
        strcpy(i.name, e->name);
        empire_writeind1(i);
        i.pos = ftell(empire_data);
    }
    while( empire_sortind1() == R_SWAPS );
    fclose(empire_ind1);

    /* clean up */
    free(e);
    fclose(empire_data);
    return R_OK;
}
