/**************************************************************
 * DOTURN.C                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroWar 1.00 - Turn processing module.                    *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     01-Mar-97                                         *
 **************************************************************/


/* included headers *******************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "astrowar.h"
#include "detail.h"
#include "orders.h"
#include "empire.h"
#include "planet.h"
#include "fleets.h"
#include "report.h"
#include "scrlog.h"
#include "config.h"
#include "gamlst.h"


/* structures *************************************************/


typedef struct {
    char  name[16]; /* planet name */
    void *next;     /* next entry in list */
} plist;


/* level 5 routines *******************************************/


/* doturn_calcdist() - calculate distance between planets */
long doturn_calcdist(char *origin, char *dest)
{
    pos     opos,             /* position of origin */
            dpos;             /* position of destination */
    planet *p = planet_new(); /* planet record */
    long    dist;             /* distance between planets */

    planet_find(p, origin); opos = planet_getpos(p);
    planet_find(p, dest);   dpos = planet_getpos(p);
    planet_old(p);
    dist = abs(opos.x - dpos.x);
    if( abs(opos.y - dpos.y) > dist )
        dist = abs(opos.y - dpos.y);

    return dist;
}


/* level 4 routines *******************************************/


/* doturn_planets() - construct a planets report */
void doturn_planets(report *t, planet *p, char *ename)
{
    result  r;         /* result of read */
    char    pname[16], /* planet name */
            owner[16]; /* planet owner */

    /* initialise planet list */
    report_settype(t, 'P');
    report_setempire(t, ename);

    /* build planet list */
    r = planet_first(p, P_NAME);
    while(r == R_OK)
    {
        planet_getowner(owner, p);
        if( strcmp(owner, ename) == 0 )
        {
            report_addentry(t);
            planet_getname(pname, p);
            report_setname(  t, pname);
            report_setpos(   t, planet_getpos(p) );
            report_setprod(  t, planet_getprod(p) );
            report_setships( t, planet_getships(p) );
        }
        r = planet_next(p, P_NAME);
    }
}

/* doturn_map() - construct a map */
void doturn_map(report *t, planet *p, char *ename, char *centre)
{
    char   pname[16]; /* planet name */
    result r;         /* result of read */

    report_settype(  t, 'M');
    report_setempire(t, ename);
    report_setcentre(t, centre);
    r = planet_first(p, P_NAME);
    while(r == R_OK)
    {
        planet_getname(pname, p);
        if( doturn_calcdist(centre, pname) <= 7 )
        {
            report_addentry(t);
            report_setname(t, pname);
            report_setpos( t, planet_getpos(p) );
        }
        r = planet_next(p, P_NAME);
    }
}


/* level 3 routines *******************************************/


/* doturn_processjoin() - process a join order */
void doturn_processjoin(order *o)
{
    empire *e = empire_new(); /* empire details */
    planet *p = planet_new(); /* planet (possible homeworld) details */
    report *t = report_new(); /* report variable */
    char    ename[16],        /* empire name */
            pname[16];        /* planet name */
    result  r;                /* result of find or read */

    /* attempt to add empire to game if name not duplicated */
    orders_getempire(ename, o);
    if( empire_find(e, ename) != R_OK )
    {
        r = planet_first(p, P_NONE);
        while(!r)
        {
            if( !planet_getscouted(p) ) break;
            r = planet_next(p, P_NONE);
        }

        /* select a safe planet */
        if(!r)
        {
            planet_getname(pname, p);
            empire_sethomewld(e, pname); /* save empire */
            empire_setname(e, ename);
            empire_write(e);
            planet_setowner(p, ename);   /* save planet */
            planet_setscouted(p, 1);
            planet_setprod(p, 15);
            planet_setships(p, 30);
            planet_write(p);
            report_settype(t, 'J');      /* prepare report */
            report_settrans( t, orders_gettrans(o) );
            report_sethomewld(t, pname);
        }

        /* prepare 'game full' report */
        else
        {
            report_settype(t, 'j');
            report_settrans( t, orders_gettrans(o) );
            report_seterror(t, 'F');
        }

    }

    /* prepare 'duplicate name' report */
    else
    {
        report_settype(t, 'j');
        report_settrans( t, orders_gettrans(o) );
        report_seterror(t, 'D');
    }

    /* write join report */
    report_write(t);

    /* write planets report and map */
    if( report_gettype(t) == 'J' )
    {
        doturn_planets(t, p, ename);
        report_write(t);
        doturn_map(t, p, ename, pname);
        report_write(t);
    }

    /* clean up */
    report_old(t);
    empire_old(e);
    planet_old(p);
}

/* doturn_processsend() - process a send order */
void doturn_processsend(order *o, detail *d)
{
    planet *p = planet_new(); /* planet variable */
    fleet  *f = fleets_new(); /* variable for new fleet */
    report *t = report_new(); /* report variable */
    char    ecode,            /* for unsuccessful send */
            origin[16],       /* planet of origin name */
            dest[16],         /* destination planet name */
            owner[16],        /* planet owner */
            ename[16];        /* empire name */
    long    dist,             /* distance between planets */
            number;           /* number of last fleet in file */

    /* initialise some variables */
    orders_getempire(ename, o);
    orders_getorigin(origin, o);
    orders_getdest(dest, o);

    /* check details & send fleet if OK */
    if( planet_find(p, dest) != R_OK )
        ecode = 'D';
    else
    {
        /* check for various errors */
        orders_getorigin(origin, o);
        if( planet_find(p, origin) != R_OK )
            ecode = 'O';
        else if(strcmp( planet_getowner(owner, p), ename ))
            ecode = 'U';
        else if(( planet_getships(p) < orders_getships(o) ) ||
                ( orders_getships(o) < 1) )
            ecode = 'S';
        else if(( dist = doturn_calcdist(origin, dest) ) >
                  detail_getmaxdist(d))
            ecode = 'M';

        /* no errors-- send fleet */
        else
        {
            /* ensure correct capitalisation */
            planet_find(p, dest);
            planet_getname(dest, p);
            planet_find(p, origin);
            planet_getname(origin, p);

            /* send fleet */
            if( fleets_last(f, F_NUMBER) == R_OK )
            {
                number = fleets_getnumber(f);
                fleets_clear(f);
                fleets_setnumber(f, number + 1);
            }
            else fleets_setnumber(f, 1);
            fleets_setowner( f, ename);
            fleets_setorigin(f, origin);
            fleets_setdest(  f, dest);
            fleets_setships( f, orders_getships(o) );
            fleets_setdist(  f, (int)dist );
            fleets_write(f);

            /* update origin/dest planets */
            planet_setships(p, planet_getships(p) -
                orders_getships(o) );
            planet_write(p);
            planet_find(p, dest);
            planet_setscouted(p, 1);
            planet_write(p);

            /* prepare report */
            report_settype(    t, 'S');
            report_setempire(  t, ename);
            report_setfleetno( t, fleets_getnumber(f) );
            report_setorigin(  t, origin);
            report_setdest(    t, dest);
            report_setships(   t, orders_getships(o) );
            report_write(t);

            /* free up memory */
            fleets_old(f);
            report_old(t);
            planet_old(p);
            return;
        }
    }

    /* write failed send report */
    report_settype(  t, 's');
    report_setempire(t, ename);
    report_setorigin(t, origin);
    report_setdest(  t, dest);
    report_setships( t, orders_getships(o) );
    report_seterror( t, ecode);
    report_write(t);

    /* free up memory */
    fleets_old(f);
    report_old(t);
    planet_old(p);
}

/* doturn_processplanets() - process a planets order */
void doturn_processplanets(order *o)
{
    report *t = report_new(); /* planet list report entry */
    planet *p = planet_new(); /* planet variable */
    char    ename[16];        /* empire name */

    /* construct planets report */
    orders_getempire(ename, o);
    doturn_planets(t, p, ename);

    /* write planet list and clean up */
    report_write(t);
    report_old(t);
    planet_old(p);
}

/* doturn_processfleets() - process a fleets order */
void doturn_processfleets(order *o)
{
    report *t = report_new(); /* fleet list report entry */
    fleet  *f = fleets_new(); /* fleet variable */
    result  r;                /* result of read */
    char    ename[16],        /* empire name */
            owner[16],        /* fleet owner */
            origin[16],       /* planet of origin */
            dest[16];         /* destination planet */

    /* initialise fleet list */
    report_settype(t, 'F');
    orders_getempire(ename, o);
    report_setempire(t, ename);

    /* build fleet list */
    r = fleets_first(f, F_NUMBER);
    while(r == R_OK)
    {
        fleets_getowner(owner, f);
        if(( strcmp(owner, ename) == 0 ) &&
             !fleets_deleted(f) )
        {
            fleets_getorigin(origin, f);
            fleets_getdest(dest, f);
            report_addentry(t);
            report_setfleetno( t, fleets_getnumber(f) );
            report_setorigin(  t, origin);
            report_setdest(    t, dest);
            report_setships(   t, fleets_getships(f) );
            report_setdist(    t, fleets_getdist(f) );
        }
        r = fleets_next(f, F_NUMBER);
    }

    /* write fleet list and clean up */
    report_write(t);
    report_old(t);
    fleets_old(f);
}

/* doturn_processmap() - process a map order */
void doturn_processmap(order *o)
{
    report *t = report_new(); /* map image */
    planet *p = planet_new(); /* for scanning planets */
    char    ename[16],        /* empire name */
            centre[16],       /* name of centre planet */
            owner[16];        /* owner of planet */

    /* extract information from order record */
    orders_getempire(ename,  o);
    orders_getcentre(centre, o);

    if( planet_find(p, centre) != R_OK )
    {
        planet_getname(centre, p); /* correct capitalisation */
        report_settype(  t, 'm');
        report_setempire(t, ename);
        report_setcentre(t, centre);
        report_seterror( t, 'P');
    }
    else if(strcmp( planet_getowner(owner, p), ename ))
    {
        report_settype(  t, 'm');
        report_setempire(t, ename);
        report_setcentre(t, centre);
        report_seterror( t, 'U');
    }
    else doturn_map(t, p, ename, centre);

    /* write report and clean up */
    report_write(t);
    report_old(t);
    planet_old(p);
}

/* doturn_processwrite() - process a write order */
void doturn_processwrite(order *o)
{
    empire *e = empire_new(); /* empire for verification */
    report *t = report_new(); /* report record */
    result  r;                /* result of read */
    char    sender[16],       /* sender of message */
            recip[16],        /* recipient of message */
           *text;             /* message text */

    /* initialise */
    orders_getrecip(recip, o);
    orders_getempire(sender, o);

    /* message to all */
    if( strcmp(recip, "*") == 0 )
    {
        text = orders_gettext(o);
        report_settype(  t, 'W');
        report_setsender(t, sender);
        report_settext(  t, text);
        r = empire_first(e, E_NAME);
        while(r == R_OK)
        {
            empire_getname(recip, e);
            report_setempire(t, recip);
            report_write(t);
            r = empire_next(e, E_NAME);
        }
        free(text);
    }

    /* message to one empire (ignore messages to system [""]) */
    else if( strcmp(recip, "") )
    {
        if( empire_find(e, recip) == R_OK )
        {
            empire_getname(recip, e); /* correct capitalisation */
            text = orders_gettext(o);
            report_settype(  t, 'W');
            report_setempire(t, recip);
            report_setsender(t, sender);
            report_settext(  t, text);
            report_write(t);
            free(text);
        }
        else
        {
            report_settype(  t, 'w');
            report_setempire(t, sender);
            report_setrecip( t, recip);
            report_seterror( t, 'R');
            report_write(t);
        }
    }

    /* free memory */
    empire_old(e);
    report_old(t);
}

/* doturn_processresign() - process a resign order */
void doturn_processresign(order *o)
{
    empire *e = empire_new(); /* empire record */
    planet *p = planet_new(); /* to neutralise planets */
    fleet  *f = fleets_new(); /* to delete fleets */
    char    ename[16],        /* empire name */
            owner[16];        /* planet's owner */
    result  r;                /* result of planet/fleet read */

    /* delete empire record */
    orders_getempire(ename, o);
    if( !empire_find(e, ename) ) empire_delete(e);

    /* neutralise planets */
    r = planet_first(p, P_NAME);
    while(!r)
    {
        planet_getowner(owner, p);
        if( !stricmp(owner, ename) )
        {
            planet_setowner(p, "");
            planet_write(p);
        }
        r = planet_next(p, P_NAME);
    }

    /* remove fleets */
    r = fleets_first(f, F_NUMBER);
    while(!r)
    {
        fleets_getowner(owner, f);
        if( !stricmp(owner, ename) )
            fleets_delete(f);
        r = fleets_next(f, F_NUMBER);
    }

    /* return */
    empire_old(e);
    planet_old(p);
    fleets_old(f);
}

/* doturn_processround() - process a battle round */
void doturn_processround(long *attacker, long *defender)
{
    /* very simplistic */
    if(random() % 2) --*defender;
}


/* level 2 routines *******************************************/


/* doturn_processorder() - process a single order */
void doturn_processorder(order *o, detail *d)
{
    switch( orders_gettype(o) )
    {
        case 'J': doturn_processjoin(o);    break;
        case 'S': doturn_processsend(o, d); break;
        case 'P': doturn_processplanets(o); break;
        case 'F': doturn_processfleets(o);  break;
        case 'M': doturn_processmap(o);     break;
        case 'W': doturn_processwrite(o);   break;
        case 'R': doturn_processresign(o);  break;
    }
}

/* doturn_processarrival() - process an arrival */
void doturn_processarrival(fleet *f)
{
    planet *p = planet_new(); /* destination planet */
    report *t = report_new(); /* report record */
    char    ename[16],        /* fleet owner */
            dest[16];         /* destination planet name */

    /* add fleet's ships to those in orbit around planet */
    fleets_getdest(dest, f);
    if( planet_find(p, dest) == R_OK )
    {
        planet_setships( p, planet_getships(p) +
            fleets_getships(f) );
        planet_write(p);
    }

    /* write report */
    fleets_getowner(ename, f);
    report_settype(    t, 'A');
    report_setempire(  t, ename);
    report_setfleetno( t, fleets_getnumber(f) );
    report_write(t);

    /* delete fleet and clean up */
    fleets_delete(f);
    report_old(t);
    planet_old(p);
}

/* doturn_processscouting() - process a scouting */
void doturn_processscouting(fleet *f)
{
    planet *p = planet_new(); /* planet scouted */
    report *t = report_new(); /* intelligence report */
    char    dest[16],         /* destination planet name */
            ename[16],        /* empire name */
            owner[16],        /* planet owner */
            origin[16];       /* origin planet name */

    /* build intelligence report */
    fleets_getdest(dest, f);
    fleets_getowner(ename, f);
    if( planet_find(p, dest) == R_OK )
    {
        planet_getowner(owner, p);
        report_settype(    t, 'I');
        report_setempire(  t, ename);
        report_setfleetno( t, fleets_getnumber(f) );
        report_setowner(   t, owner);
        report_setprod(    t, planet_getprod(p) );
        report_setships(   t, planet_getships(p) );
        report_write(t);
    }

    /* return fleet home */
    fleets_getorigin(origin, f);
    fleets_setorigin(f, dest);
    fleets_setdest(  f, origin);
    fleets_setdist(  f, doturn_calcdist(dest, origin) );
    fleets_write(f);

    /* clean up */
    report_old(t);
    planet_old(p);
}

/* doturn_processbattle() - process a battle */
void doturn_processbattle(fleet *f)
{
    planet *p = planet_new(); /* planet record */
    report *t = report_new(); /* report record */
    char    dest[16],         /* destination planet name */
            ownerf[16],       /* fleet owner */
            ownerp[16];       /* planet owner */
    long    pships,           /* ships around planet */
            fships;           /* ships in fleet */

    /* check destination planet exists */
    fleets_getdest(dest, f);
    if( planet_find(p, dest) != R_OK )
    {
        planet_old(p);
        report_old(t);
        return;
    }

    /* blow-by-blow battle resolution */
    fships = fleets_getships(f);
    pships = planet_getships(p);
    while( (pships > 0) && (fships > 0) )
    {
        doturn_processround(&fships, &pships);
        if(pships > 0) doturn_processround(&pships, &fships);
    }
    if(fships < 0) fships = 0;
    if(pships < 0) pships = 0;

    /* write reports */
    fleets_getowner(ownerf, f);  /* fleet owner battle report */
    planet_getowner(ownerp, p);
    report_settype(    t, 'B');
    report_setempire(  t, ownerf);
    report_setfleetno( t, fleets_getnumber(f) );
    report_setowner(   t, ownerp);
    report_setships(   t, fships);
    report_setenemy(   t, pships);
    report_write(t);
    if( strcmp(ownerp, "") )
    {
        report_settype(  t, 'D');    /* planet defence report */
        report_setempire(t, ownerp);
        report_setplanet(t, dest);
        report_setowner( t, ownerf);
        report_setships( t, pships);
        report_setenemy( t, fships);
        report_write(t);
    }

    /* update planet record */
    if(pships == 0)
    {
        planet_setowner(p, ownerf);
        planet_setships(p, fships);
    }
    else planet_setships(p, pships);
    planet_write(p);

    /* delete fleet and clean up */
    fleets_delete(f);
    report_old(t);
    planet_old(p);
}


/* level 1 routines *******************************************/


/* doturn_interpret() - interpret doturn specific parameters */
result doturn_interpret(char *gamename, char **argp)
{
    int argc; /* argument counter */

    argc = 0; strcpy(gamename, "");
    while(argc < 10 && argp[argc] != NULL)
    {
        if(!strcmp( argp[argc], "-game" ) && argc < 9)
        {
            ++argc;
            if(strlen( argp[argc] ) > 15)
            {
                strncpy( gamename, argp[argc], 15 );
                gamename[15] = '\0';
            }
            else strcpy( gamename, argp[argc] );
        }
        else
        {
            scrlog_error();
            return R_BADCMD;
        }
        ++argc;
    }
    return R_OK;
}

/* doturn_open() - open the turn processing module */
result doturn_open(detail **d, char *gamepath, char *path, char
    *gamename)
{
    result r;            /* result of each open file attempt */
    int    line;         /* line number of error message */
    char   logfile[128]; /* name of log file */

    /* attempt to open config */
    config_open(path);
    switch( r = config_read(path, &line) )
    {
        case R_FILE:   scrlog_cannotreadconfig(path);
                       return R_FILE;
        case R_BADCMD: scrlog_badcommandinconfig(path, line);
                       return R_BADCMD;
        case R_OK:     config_getlogfile(logfile);
                       if( ( r = scrlog_openlog(logfile, 0xffff) )
                           != R_OK)
                       {
                           scrlog_cannotopenlog(logfile);
                           goto closeconfig;
                       }
                       scrlog_doturn();
                       scrlog_configread(path);
                       break;
        default:       scrlog_badreturncode(r, "config_read");
                       return r;
    }

    /* attempt to open game list */
    switch( r = gamlst_open(path, &line) )
    {
        case R_FILE:   scrlog_cannotopengamelist(path);
                       goto closeconfig;
        case R_EOF:    scrlog_badfileingamelist(path, line);
                       goto closeconfig;
        case R_BADCMD: scrlog_badlineingamelist(path, line);
                       goto closeconfig;
        case R_OK:     scrlog_gamelistread(path);
                       break;
        default:       scrlog_badreturncode(r, "gamlst_open");
                       goto closeconfig;
    }

    /* now open game detail file */
    if(( r = gamlst_find(gamename) ) != R_OK)
    {
        scrlog_gamenotfound(gamename);
        goto closegamelist;
    }
    gamlst_getpath(gamepath);
    switch( r = detail_open(d, gamepath) )
    {
        case R_FILE:  scrlog_cannotopendetail(gamepath);
                       goto closegamelist;
        case R_MEMORY: scrlog_outofmemory("loading detail file");
                       goto closegamelist;
        case R_BADCMD: scrlog_badcommandindetail(gamepath);
                       goto closegamelist;
        case R_OK:     scrlog_detailread(gamepath);
                       break;
        default:       scrlog_badreturncode(r, "detail_open");
                       goto closegamelist;
    }

    /* open empire file */
    switch( r = empire_open(gamepath) )
    {
        case R_FILE:   scrlog_cannotopenfile(gamepath, "empire.data");
                       goto closedetail;
        case R_HEADER: scrlog_badheaderinfile(gamepath, "empire.data");
                       goto closedetail;
        case R_OK:     scrlog_fileopened(gamepath, "empire.data");
                       break;
        default:       scrlog_badreturncode(r, "empire_open");
                       goto closedetail;
    }

    /* open planet file */
    switch( r = planet_open(gamepath) )
    {
        case R_FILE:   scrlog_cannotopenfile(gamepath, "planet.data");
                       goto closeempire;
        case R_HEADER: scrlog_badheaderinfile(gamepath, "planet.data");
                       goto closeempire;
        case R_OK:     scrlog_fileopened(gamepath, "planet.data");
                       break;
        default:       scrlog_badreturncode(r, "planet_open");
                       goto closeempire;
    }

    /* open fleets file */
    switch( r = fleets_open(gamepath) )
    {
        case R_FILE:   scrlog_cannotopenfile(gamepath, "fleets.data");
                       goto closeplanet;
        case R_HEADER: scrlog_badheaderinfile(gamepath, "fleets.data");
                       goto closeplanet;
        case R_OK:     scrlog_fileopened(gamepath, "fleets.data");
                       break;
        default:       scrlog_badreturncode(r, "fleets_open");
                       goto closeplanet;
    }

    /* open report file */
    switch( r = report_create(gamepath) )
    {
        case R_FILE:   scrlog_cannotcreatefile(gamepath, "report.data");
                       goto closefleets;
        case R_EXISTS: scrlog_fileexists(gamepath, "report.data");
                       goto closefleets;
        case R_OK:     if(( r = report_open(gamepath) ) != R_OK)
                       {
                           scrlog_cannotopenfile(gamepath,
                               "report.data");
                           goto closefleets;
                       }
                       break;
        default:       scrlog_badreturncode(r, "report_open");
                       goto closefleets;
    }

    /* randomise random() seed */
    srandom( time(0) );
    return R_OK;

    /* clean up after fatal error */
    closefleets:
        fleets_close();
    closeplanet:
        planet_close();
    closeempire:
        empire_close();
    closedetail:
        detail_close(*d);
    closegamelist:
        gamlst_close();
    closeconfig:
        config_close();
    return r;
}

/* doturn_processmoves() - process fleet movements */
void doturn_processmoves(detail *d)
{
    fleet  *f = fleets_new(); /* current fleet to process */
    planet *p = planet_new(); /* planet record */
    result  r;                /* result of read operation */
    char    dest[16],         /* destination planet */
            origin[16],       /* planet of origin */
            ownerd[16],       /* owner of destination */
            ownero[16],       /* owner of origin */
            ownerf[16];       /* owner of fleet */
    int     speed;            /* ship speed */

    /* initialise */
    speed = detail_getspeed(d);

    /* scan fleets */
    r = fleets_first(f, F_NUMBER);
    while(r == R_OK)
    {
        /* process fleet */
        if( !fleets_deleted(f) )
        {
            scrlog_movingfleet( fleets_getnumber(f) );
            fleets_setdist( f, fleets_getdist(f) - speed );
            if( fleets_getdist(f) <= 0 )
            {
                /* arrived... get planet details */
                fleets_getowner( ownerf, f);
                fleets_getorigin(origin, f);
                fleets_getdest(  dest,   f);
                if( planet_find(p, origin) == R_OK )
                    planet_getowner(ownero, p);
                else
                    strcpy(ownero, "");
                if( planet_find(p, dest) == R_OK )
                    planet_getowner(ownerd, p);
                else
                    strcpy(ownerd, "");

                /* process fleet arrival */
                if(( fleets_getships(f) == 1 )     &&
                   ( strcmp(ownerf, ownerd) != 0 ) &&
                   ( strcmp(ownerf, ownero) == 0 ))
                    doturn_processscouting(f);
                else if( strcmp(ownerf, ownerd) == 0 )
                    doturn_processarrival(f);
                else
                    doturn_processbattle(f);
            }
            else fleets_write(f);
        }

        r = fleets_next(f, F_NUMBER);
    }

    /* clean up */
    scrlog_fleetsdone();
    planet_old(p);
    fleets_old(f);
}

/* doturn_processorders() - process orders from orders file */
void doturn_processorders(detail *d, char *path)
{
    order *o = orders_new(); /* current order to process */
    result r;                /* result of read operation */

    if( orders_open(path) != R_OK )  return;
    r = orders_first(o);
    while(r == R_OK)
    {
        scrlog_processingorder( orders_gettype(o) );
        doturn_processorder(o, d);
        r = orders_next(o);
    }
    scrlog_ordersdone();
    orders_old(o);
    orders_close();
}

/* doturn_processprod() - process planetary production */
void doturn_processprod()
{
    planet *p = planet_new(); /* current planet to process */
    result  r;                /* result of read operation */
    char    name[16],         /* planet name */
            owner[16];        /* planet owner */
    long    ships;            /* ships in orbit */

    r = planet_first(p, P_NAME);
    while(r == R_OK)
    {
        planet_getname(name, p);
        scrlog_processingplanet(name);
        if(strcmp( planet_getowner(owner, p), "" ))
        {
            ships = planet_getships(p);
            ships += planet_getprod(p);
            planet_setships(p, ships);
            planet_write(p);
        }
        r = planet_next(p, P_NAME);
    }
    scrlog_productiondone();
    planet_old(p);
}

/* doturn_checkvictory() - process victory */
void doturn_checkvictory()
{
    int     homeworlds,       /* homeworlds owned by victor */
            players;          /* number of players */
    char    vname[16],        /* last potential victor name */
            owner[16],        /* planet owner name */
            ename[16],        /* empire name */
            homewld[16];      /* homeworld name */
    empire *e = empire_new(); /* empire record */
    report *t = report_new(); /* report record */
    planet *p = planet_new(); /* for checking planet name */
    result  r;                /* result of read */

    /* initialise */
    scrlog_checkingvictory();
    players = homeworlds = 0;
    strcpy(vname, "");

    /* read through empires */
    r = empire_first(e, E_NAME);
    while(r == R_OK)
    {
        empire_gethomewld(homewld, e);
        if( planet_find(p, homewld) == R_OK )
        {
            planet_getowner(owner, p);
            empire_getname(ename, e);

            /* owner of first homeworld = potential victor */
            if( !strcmp(vname, "") )
            {
                strcmp(vname, owner);
                homeworlds = 1;
            }

            /* potential victor owns another homeworld */
            else if( !strcmp(vname, owner) )
                homeworlds++;
        }
        players++;
        r = empire_next(e, E_NAME);
    }

    /* write victory reports */
    if( (homeworlds == players) && (players > 1) )
    {
        report_settype(t, 'V');
        report_setvictor(t, vname);
        r = empire_first(e, E_NAME);
        while(r == R_OK)
        {
            empire_getname(ename, e);
            report_setempire(t, ename);
            report_write(t);
            r = empire_next(e, E_NAME);
        }
        scrlog_victory(vname);
    }
    else scrlog_novictory();

    /* clean up */
    planet_old(p);
    empire_old(e);
    report_old(t);
}

/* doturn_close() - close down turn processing */
void doturn_close(detail *d, char *path)
{
    scrlog_finished();
    report_close();
    fleets_close();
    fleets_pack(path);
    gamlst_close();
    planet_close();
    empire_close();
    empire_pack(path);
    detail_close(d);
}


/* level 0 routines *******************************************/


/* doturn_main() - main turn processing function */
int doturn_main(char *path, char **argp)
{
    result  r;            /* returned to calling process */
    detail *d;            /* detail record */
    char    gamename[16], /* name of game */
            gamepath[72]; /* path to game files */

    if( doturn_interpret(gamename, argp) == R_OK )
    {
        if(( r = doturn_open(&d, gamepath, path, gamename) ) != R_OK)
            return r;
        if( detail_getmoves(d) )  doturn_processmoves(d);
        if( detail_getorders(d) ) doturn_processorders(d, gamepath);
        if( detail_getprod(d) )   doturn_processprod();
        if( detail_getvict(d) )   doturn_checkvictory();
        doturn_close(d, gamepath);
    }

    return R_OK;
}
