/**********************************************************************
 * TURNIN.C                          Copyright (C) Damian Walker 1997 *
 *--------------------------------------------------------------------*
 * AstroMail 1.01 - Inbound order processing.                         *
 *--------------------------------------------------------------------*
 * Author   Damian G Walker                                           *
 * Date     11-Apr-97                                                 *
 **********************************************************************/


/* included headers ***************************************************/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "astroml.h"  /* all sorts of defs for AstroMail */
#include "iomsg.h"    /* for input/output messages */
#include "detail.h"   /* detail file processing defs from AstroWar */
#include "user.h"     /* user file processing defs */
#include "config.h"   /* configuration file processing defs */
#include "gamlst.h"   /* game list processing defs */
#include "transn.h"   /* transaction number processing defs */
#include "plytxt.h"   /* temporary text file processing defs */
#include "pfleet.h"   /* player fleet processing defs */
#include "joinrq.h"   /* join request processing defs */
#include "player.h"   /* player processing defs */
#include "orders.h"   /* order file processing defs from AstroWar */
#include "scrlog.h"   /* for screen and log file output */


/* enums, structures &c. **********************************************/


#define P_PLAYER 0x01
#define P_PFLEET 0x04


/* global variables ***************************************************/


int turnin_pack; /* which database files need packing? */


/* level 4 routines ***************************************************/


/* turnin_sendreply() - send reply to user */
void turnin_sendreply(user *u)
{
    fido fromaddr,     /* address to post from */
         toaddr;       /* address to post to */
    char toname[36],   /* user to post to */
         subject[72],  /* the subject */
         gamename[16], /* name of the game */
        *text;         /* text message to send */
    int  attr;         /* attributes */

    /* return if no text to send */
    if( (text = plytxt_read( user_getnumber(u) )) == NULL ) return;

    /* set some message header fields */
    fromaddr = iomsg_gettoaddr();
    user_getname(toname, u);
    toaddr = user_getaddr(u);
    if( user_getdirect(u) )
        attr = config_getdattr();
    else
        attr = config_getrattr();
    gamlst_getname(gamename);
    sprintf(subject, "Reply to orders for game \"%s\"", gamename);
    scrlog_replyposted(toname, toaddr);

    /* post message */
    iomsg_post(fromaddr, toname, toaddr, subject, attr, text);
}

/* turnin_interpretorder() - interpret user order */
char turnin_interpretorder(char *orderline)
{
    if(*orderline == '\0' || *orderline == ';' || *orderline == '\1')
                                                   return ';';
    if( !strncmp(orderline, "--- ", 4) ||
        !stricmp(orderline,  "---") )              return '-';
    if( !strnicmp(orderline, "join", 4) )          return 'J';
    if( !strnicmp(orderline, "send", 4) )          return 'S';
    if( !strnicmp(orderline, "scout", 5) )         return 'C';
    if( !strnicmp(orderline, "list planets", 12) ||
        !strnicmp(orderline, "planets", 7) )       return 'P';
    if( !strnicmp(orderline, "list fleets", 11) ||
        !strnicmp(orderline, "fleets", 6) )        return 'F';
    if( !strnicmp(orderline, "map", 3) )           return 'M';
    if( !strnicmp(orderline, "write", 5) )         return 'W';
    if( !strnicmp(orderline, "info", 4) )          return 'I';
    if( !strnicmp(orderline, "routed", 6) )        return 'R';
    if( !strnicmp(orderline, "direct", 6) )        return 'D';
    if( !strnicmp(orderline, "resign", 6) )        return 'Q';
                                                   return '!';
}

/* turnin_rejectjoin() - reject join order from existing player */
void turnin_rejectjoin(user *u, char *orderline, int line)
{
    long number; /* user number */

    number = user_getnumber(u);
    scrlog_badorder(line);
    plytxt_write(number, " > ");
    plytxt_write(number, orderline);
    plytxt_write(number, "\r\rYou are already playing this game\r\r");
}

/* turnin_processsend() - process a send order */
void turnin_processsend(user *u, char *orderline)
{
    order  *o = orders_new();           /* 'send' order */
    player *p = player_new();           /* player issuing send order */
    static
    char    word[7][72];                /* words in list */
    char    txtnum[11],                 /* player number as text */
            ename[16];                  /* empire name */
    long    number = user_getnumber(u), /* user number */
            ships;                      /* ships to send */
    int     wordctr;                    /* word counter */

    /* log and initialise */
    scrlog_processingorder("send");
    orders_settype(o, 'S');
    if(sscanf( orderline, "%s%s%s%s%s%s%s", word[0], word[1], word[2],
        word[3], word[4], word[5], word[6] ) < 4)
    {
        plytxt_write(number, " > ");
        plytxt_write(number, orderline);
        plytxt_write(number, "\r\rError in send.\r\r");
        orders_old(o);
        return;
    }
    word[1][15] = word[2][15] = word[3][15] = word[4][15] = word[5][15] =
    word[6][15] = '\0';

    /* set empire */
    sprintf(txtnum, "%ld", number);
    player_find(p, txtnum);
    player_getempire(ename, p);
    orders_setempire(o, ename);

    /* check number of ships */
    if(sscanf( word[1], "%ld", &ships ) != 1)
    {
        plytxt_write(number, " > ");
        plytxt_write(number, orderline);
        plytxt_write(number, "\r\rError in send.\r\r");
        player_old(p);
        orders_old(o);
        return;
    }
    orders_setships(o, ships);
    if(!stricmp(word[2], "ships") || !stricmp(word[2], "ship"))
        wordctr = 3;
    else
        wordctr = 2;

    /* check planets of origin and destination */
    if(!stricmp( word[wordctr], "from" )) wordctr++;
    orders_setorigin(o, word[wordctr++]);
    if(!stricmp( word[wordctr], "to" )) wordctr++;
    orders_setdest(o, word[wordctr]);

    /* clean up and return */
    orders_write(o);
    orders_old(o);
    player_old(p);
}

/* turnin_processscout() - process a scout order */
void turnin_processscout(user *u, char *orderline)
{
    order  *o = orders_new();           /* 'send' order */
    player *p = player_new();           /* player issuing send order */
    static
    char    word[7][72];                /* words in list */
    char    txtnum[11],                 /* player number as text */
            ename[16];                  /* empire name */
    long    number = user_getnumber(u); /* user number */

    /* log and initialise */
    scrlog_processingorder("scout");
    orders_settype(o, 'S');
    if(sscanf( orderline, "%s%s%s%s", word[0], word[1], word[2],
        word[3] ) < 3)
    {
        plytxt_write(number, " > ");
        plytxt_write(number, orderline);
        plytxt_write(number, "\r\rError in scout order.\r\r");
        orders_old(o);
        player_old(p);
        return;
    }
    word[1][15] = word[2][15] = word[3][15] = '\0';

    /* set empire and ships */
    sprintf(txtnum, "%ld", number);
    player_find(p, txtnum);
    player_getempire(ename, p);
    orders_setempire(o, ename);
    orders_setships(o, 1);

    /* check planets of origin and destination */
    orders_setdest( o, word[1] );
    if(stricmp( word[2], "from" ))
        orders_setorigin( o, word[2] );
    else
        orders_setorigin( o, word[3] );

    /* clean up and return */
    orders_write(o);
    orders_old(o);
    player_old(p);
}

/* turnin_processplanets() - process a planets order */
void turnin_processplanets(user *u)
{
    order  *o = orders_new();           /* 'send' order */
    player *p = player_new();           /* player issuing send order */
    char    txtnum[11],                 /* player number as text */
            ename[16];                  /* empire name */
    long    number = user_getnumber(u); /* user number */

    /* log and initialise */
    scrlog_processingorder("planets");
    orders_settype(o, 'P');

    /* set empire */
    sprintf(txtnum, "%ld", number);
    player_find(p, txtnum);
    player_getempire(ename, p);
    orders_setempire(o, ename);

    /* clean up and return */
    orders_write(o);
    orders_old(o);
    player_old(p);
}

/* turnin_processfleets() - process a fleets order */
void turnin_processfleets(user *u)
{
    order  *o = orders_new();           /* 'send' order */
    player *p = player_new();           /* player issuing send order */
    char    txtnum[11],                 /* player number as text */
            ename[16];                  /* empire name */
    long    number = user_getnumber(u); /* user number */

    /* log and initialise */
    scrlog_processingorder("fleets");
    orders_settype(o, 'F');

    /* set empire */
    sprintf(txtnum, "%ld", number);
    player_find(p, txtnum);
    player_getempire(ename, p);
    orders_setempire(o, ename);

    /* clean up and return */
    orders_write(o);
    orders_old(o);
    player_old(p);
}

/* turnin_processmap() - process a map order */
void turnin_processmap(user *u, char *orderline)
{
    order  *o = orders_new();           /* 'send' order */
    player *p = player_new();           /* player issuing send order */
    static
    char    word[7][72];                /* words in list */
    char    txtnum[11],                 /* player number as text */
            ename[16];                  /* empire name */
    long    number = user_getnumber(u); /* user number */
    int     wordctr;                    /* word counter */

    /* log and initialise */
    scrlog_processingorder("map");
    orders_settype(o, 'M');
    if(sscanf( orderline, "%s%s%s%s", word[0], word[1], word[2],
        word[3] ) < 2)
    {
        plytxt_write(number, " > ");
        plytxt_write(number, orderline);
        plytxt_write(number, "\r\rError in map order.\r\r");
        orders_old(o);
        player_old(p);
        return;
    }
    word[1][15] = word[2][15] = word[3][15] = '\0';

    /* set empire */
    sprintf(txtnum, "%ld", number);
    player_find(p, txtnum);
    player_getempire(ename, p);
    orders_setempire(o, ename);

    /* set centre planet */
    wordctr = 1;
    if(!stricmp( word[wordctr], "planets" )) wordctr++;
    if(!stricmp( word[wordctr], "from" ))    wordctr++;
    orders_setcentre( o, word[wordctr] );

    /* clean up and return */
    orders_write(o);
    orders_old(o);
    player_old(p);
}

/* turnin_processwrite() - process a write order */
void turnin_processwrite(user *u, char *orderline)
{
    order  *o = orders_new();           /* 'write' order */
    player *p = player_new();           /* player writing message */
    char    ename[16],                  /* empire writing message */
            txtnum[11],                 /* user number as text */
            recip[72],                  /* recipient as requested */
           *text = malloc(1),           /* message text */
           *msgline;                    /* pointer to a message line */
    long    number = user_getnumber(u); /* user number */

    /* log and initialise */
    scrlog_processingorder("write");
    if(!stricmp(orderline, "write") || !stricmp(orderline, "write to"))
    {
        plytxt_write(number, " > ");
        plytxt_write(number, orderline);
        plytxt_write(number, "\r\rError in write order.\r\r");
        orders_old(o);
        player_old(p);
        return;
    }
    *text = '\0';
    orders_settype(o, 'W');

    /* set empire */
    sprintf(txtnum, "%ld", number);
    player_find(p, txtnum);
    player_getempire(ename, p);
    orders_setempire(o, ename);

    /* set recipient */
    if(!strnicmp( &orderline[6], "to ", 3 ))
        strcpy( recip, &orderline[9] );
    else
        strcpy( recip, &orderline[6] );
    if( !stricmp(recip, "all") )
        orders_setrecip(o, "*");
    else if( !stricmp(recip, "system") )
        orders_setrecip(o, "");
    else
        orders_setrecip(o, recip);

    /* set message text */
    while(( msgline = iomsg_nextline() ))
    {
        if(!*msgline) break;
        text = realloc(text, strlen(text) + strlen(msgline) + 2);
        strcat(text, msgline);
        strcat(text, "\r");
        free(msgline);
    }
    orders_settext(o, text);

    /* clean up */
    orders_write(o);
    if(msgline) free(msgline);
    free(text);
    player_old(p);
}

/* turnin_processinfo() - process an info order */
void turnin_processinfo(user *u)
{
    detail *d;        /* detail record to extract info from */
    char    path[72], /* path to game */
            name[36], /* name of game */
            buf[72];  /* buffer for line output */
    long    number;   /* user number */

    /* intialise */
    scrlog_processingorder("info");
    number = user_getnumber(u);

    /* attempt to read detail file */
    gamlst_getpath(path);
    if( detail_open(&d, path) ) return;

    /* write detail entries to user */
    detail_getname(name, d);
    sprintf(buf, "Settings for game \"%s\"\r", name);
    plytxt_write(number, buf);
    sprintf( buf, "Planets:             %ld\r", detail_getplanets(d) );
    plytxt_write(number, buf);
    sprintf( buf, "Galaxy size:         %d,%d..%d,%d\r",
        detail_getxmin(d), detail_getymin(d), detail_getxmax(d),
        detail_getymax(d));
    plytxt_write(number, buf);
    sprintf( buf, "Max travel distance: %d\r", detail_getmaxdist(d) );
    plytxt_write(number, buf);
    sprintf( buf, "Ship speed:          %d\r\r", detail_getspeed(d) );
    plytxt_write(number, buf);

    /* close detail file */
    detail_close(d);
}

/* turnin_processdirect() - process a 'direct' order */
void turnin_processdirect(user *u)
{
    scrlog_processingorder("direct");
    user_setdirect(u, 1);
    user_write(u);
    plytxt_write( user_getnumber(u),
        "All responses now sent directly.\r\r" );
}

/* turnin_processrouted() - process a 'routed' order */
void turnin_processrouted(user *u)
{
    scrlog_processingorder("routed");
    user_setdirect(u, 0);
    user_write(u);
    plytxt_write( user_getnumber(u),
        "All responses now sent routed.\r\r" );
}

/* turnin_processresign() - process a 'resign' order */
void turnin_processresign(user *u)
{
    order  *o = orders_new();           /* order record */
    player *p = player_new();           /* player issuing order */
    pfleet *f = pfleet_new();           /* player fleet record */
    result  r;                          /* result of fleet read */
    char    txtnum[11],                 /* user number as text */
            ename[16];                  /* name of empire */
    long    number = user_getnumber(u); /* user number */

    /* log and initialise */
    scrlog_processingorder("resign");
    orders_settype(o, 'R');

    /* set empire */
    sprintf(txtnum, "%ld", number);
    player_find(p, txtnum);
    player_getempire(ename, p);
    orders_setempire(o, ename);
    orders_write(o);

    /* delete player record */
    player_delete(p);
    turnin_pack |= P_PLAYER;

    /* delete player fleets */
    r = pfleet_first(f, F_FLEETNO);
    while(!r)
    {
        if( pfleet_getuser(f) == number ) pfleet_delete(f);
        r = pfleet_next(f, F_FLEETNO);
    }
    turnin_pack |= P_PFLEET;

    /* send notification */
    plytxt_write(number, " > resign\r\r"
                         "You have been removed from this game.\r\r");

    /* return */
    orders_old(o);
    pfleet_old(f);
    player_old(p);
}

/* turnin_processjoin() - process a join order */
void turnin_processjoin(user *u, char *orderline)
{
    char   *ptr,              /* current position in line */
            path[72];         /* path to game files */
    order  *o = orders_new(); /* order to write */
    long    trans,            /* transaction number */
            number;           /* user number */
    joinrq *j = joinrq_new(); /* join request record */
    result  r;                /* for scanning */

    /* check there are parameters */
    number = user_getnumber(u);
    if( !stricmp(orderline, "join") )
    {
        plytxt_write(number, " > join\r\r"
                             "You need to specify an empire name\r\r");
    }

    /* log & initialise */
    scrlog_processingorder("join");
    gamlst_getpath(path);

    /* check user is not already in join request file */
    r = joinrq_first(j, J_NONE);
    while(r == R_OK)
    {
        if( joinrq_getuser(j) == user_getnumber(u) )
        {
            plytxt_write(number, " > ");
            plytxt_write(number, orderline);
            plytxt_write(number, "\r\rYou have already reqested to "
                                 "join this game\r\r");
            orders_old(o);
            joinrq_old(j);
            return;
        }
        r = joinrq_next(j, J_NONE);
    }

    /* parse line */
    ptr = &orderline[5]; /* skip 'join' */
    if( !strnicmp(ptr, "as ", 3) ) ptr = &ptr[3];

    /* set order fields */
    if( strlen(ptr) > 15 ) ptr[15] = '\0';
    orders_settype(  o, 'J');
    orders_settrans( o, trans = transn_read(path) );
    orders_setempire(o, ptr);
    orders_write(    o);

    /* log the join request */
    joinrq_clear(    j);
    joinrq_settrans( j, trans);
    joinrq_setuser(  j, number);
    joinrq_setempire(j, ptr);
    joinrq_write(    j);

    /* clean up */
    joinrq_old(j);
    orders_old(o);
}

/* turnin_rejectorder() - reject an order */
void turnin_rejectorder(user *u, char *orderline, int line)
{
    long number; /* user number */

    number = user_getnumber(u);
    scrlog_badorder(line);
    plytxt_write(number, " > ");
    plytxt_write(number, orderline);
    plytxt_write(number, "\r\r");
    plytxt_write(number, "Order not valid until player accepted\r\r");
}

/* turnin_error() - command error in input message */
void turnin_error(user *u, char *orderline, int line)
{
    long number; /* user number */

    number = user_getnumber(u);
    scrlog_badorder(line);
    plytxt_write(number, " > ");
    plytxt_write(number, orderline);
    plytxt_write(number, "\r\rCannot understand this order\r\r");
}


/* level 3 routines ***************************************************/


/* turnin_opengame() - open game files */
result turnin_opengame(void)
{
    result r;        /* result of open attempt */
    char   path[72]; /* path to game files */

    /* establish path */
    gamlst_getpath(path);

    /* attempt to open player file */
    switch( r = player_open(path) )
    {
        case R_FILE:   if(( r = player_create(path) ) != R_OK)
                       {
                           scrlog_cannotopenfile(path, "player.data");
                           return R_FILE;
                       }
                       scrlog_filecreated(path, "player.data");
                       if(( r = player_open(path) ) != R_OK)
                       {
                           scrlog_cannotopenfile(path, "player.data");
                           return R_FILE;
                       }
                       scrlog_fileopened(path, "player.data");
                       break;
        case R_HEADER: scrlog_badheaderinfile(path, "player.data");
                       return R_HEADER;
        case R_OK:     scrlog_fileopened(path, "player.data");
                       break;
        default:       scrlog_badreturncode(r, "player_open");
                       return R_BADCMD;
    }

    /* attempt to open player fleet file */
    switch( r = pfleet_open(path) )
    {
        case R_FILE:   if(( r = pfleet_create(path) ) != R_OK)
                       {
                           scrlog_cannotopenfile(path, "pfleet.data");
                           goto closeplayer;
                       }
                       scrlog_filecreated(path, "pfleet.data");
                       if(( r = pfleet_open(path) ) != R_OK)
                       {
                           scrlog_cannotopenfile(path, "pfleet.data");
                           goto closeplayer;
                       }
                       scrlog_fileopened(path, "pfleet.data");
                       break;
        case R_HEADER: scrlog_badheaderinfile(path, "pfleet.data");
                       goto closeplayer;
        case R_OK:     scrlog_fileopened(path, "pfleet.data");
                       break;
        default:       scrlog_badreturncode(r, "pfleet_open");
                       goto closeplayer;
    }

    /* attempt to open join request file */
    switch( r = joinrq_open(path) )
    {
        case R_FILE:   if(( r = joinrq_create(path) ) != R_OK)
                       {
                           scrlog_cannotopenfile(path, "joinrq.data");
                           goto closepfleet;
                       }
                       scrlog_filecreated(path, "joinrq.data");
                       if(( r = joinrq_open(path) ) != R_OK)
                       {
                           scrlog_cannotopenfile(path, "joinrq.data");
                           goto closepfleet;
                       }
                       scrlog_fileopened(path, "joinrq.data");
                       break;
        case R_HEADER: scrlog_badheaderinfile(path, "joinrq.data");
                       goto closepfleet;
        case R_OK:     scrlog_fileopened(path, "joinrq.data");
                       break;
        default:       scrlog_badreturncode(r, "joinrq_open");
                       goto closepfleet;
    }

    /* attempt to open orders file */
    switch( r = orders_open(path) )
    {
        case R_FILE:  if(( r = orders_create(path) ) != R_OK)
                       {
                           scrlog_cannotopenfile(path, "orders.data");
                           r = R_FILE;
                           goto closejoinrq;
                       }
                       scrlog_filecreated(path, "orders.data");
                       if(( r = orders_open(path) ) != R_OK)
                       {
                           scrlog_cannotopenfile(path, "orders.data");
                           r = R_FILE;
                           goto closejoinrq;
                       }
                       scrlog_fileopened(path, "orders.data");
                       break;
        case R_HEADER: scrlog_badheaderinfile(path, "orders.data");
                       r = R_HEADER;
                       goto closejoinrq;
        case R_OK:     scrlog_fileopened(path, "orders.data");
                       break;
        default:       scrlog_badreturncode(r, "orders_open");
                       goto closejoinrq;
    }

    /* all OK so return */
    plytxt_open(path);
    turnin_pack = 0;
    return R_OK;

    /* clean up after fatal error */
    closejoinrq:
        joinrq_close();
    closepfleet:
        pfleet_close();
    closeplayer:
        player_close();
    return r;
}

/* turnin_existingplayer() - process existing player's record */
void turnin_existingplayer(user *u)
{
    char *orderline; /* line of order message */
    int   line = 1,  /* line number */
          eot = 0;   /* end of text flag */

    orderline = iomsg_firstline();
    while(orderline != NULL && !eot)
    {
        switch( turnin_interpretorder(orderline) )
        {
            case 'J': turnin_rejectjoin(u, orderline, line); break;
            case 'S': turnin_processsend(u, orderline);      break;
            case 'C': turnin_processscout(u, orderline);     break;
            case 'P': turnin_processplanets(u);              break;
            case 'F': turnin_processfleets(u);               break;
            case 'M': turnin_processmap(u, orderline);       break;
            case 'W': turnin_processwrite(u, orderline);     break;
            case 'I': turnin_processinfo(u);                 break;
            case 'D': turnin_processdirect(u);               break;
            case 'R': turnin_processrouted(u);               break;
            case 'Q': turnin_processresign(u); eot = 1;      break;
            case ';': /* ignore comments etc. */             break;
            case '-': eot = 1;                               break;
            default:  turnin_error(u, orderline, line);      break;
        }
        free(orderline);
        line++;
        orderline = iomsg_nextline();
    }
    scrlog_messagedone();
    turnin_sendreply(u);
}

/* turnin_newplayer() - process new player's record */
void turnin_newplayer(user *u)
{
    char *orderline; /* line of order message */
    int   line = 1,  /* line number */
          eot = 0;   /* end of message text marker */

    orderline = iomsg_firstline();
    while(orderline != NULL && !eot)
    {
        switch( turnin_interpretorder(orderline) )
        {
            case 'J': turnin_processjoin(u, orderline);       break;
            case 'S':
            case 'C':
            case 'P':
            case 'F':
            case 'M':
            case 'W': turnin_rejectorder(u, orderline, line); break;
            case 'I': turnin_processinfo(u);                  break;
            case 'D': turnin_processdirect(u);                break;
            case 'R': turnin_processrouted(u);                break;
            case ';': /* ignore comments etc. */              break;
            case '-': eot = 1;                                break;
            default:  turnin_error(u, orderline, line);       break;
        }
        line++;
        free(orderline);
        orderline = iomsg_nextline();
    }
    scrlog_messagedone();
    turnin_sendreply(u);
}

/* turnin_closegame() - close the game files */
void turnin_closegame(void)
{
    char path[72]; /* path to game files */

    /* close files */
    player_close();
    pfleet_close();
    joinrq_close();
    orders_close();
    plytxt_close();

    /* pack files if necessary */
    gamlst_getpath(path);
    if(turnin_pack & P_PLAYER) player_pack(path);
    if(turnin_pack & P_PFLEET) pfleet_pack(path);
}


/* level 2 routines ***************************************************/


/* turnin_checkuser() - ensure that user exists */
result turnin_checkuser(user *u, char *fromname)
{
    user *last = user_new(); /* last user in file */

    /* add user if not found */
    if( user_find(u, fromname) != R_OK )
    {
        if( user_last(last, U_NUMBER) == R_OK )
            user_setnumber( u, user_getnumber(last) + 1 );
        else
            user_setnumber(u, 1);
        user_setname(  u, fromname );
        user_setaddr(  u, iomsg_getfromaddr() );
        user_setdirect(u, config_getudirect() );
        user_write(u);
    }

    /* clean up and return */
    user_old(last);
    return R_OK;
}

/* turnin_duplicateuser() - duplicate user name/wrong address used */
void turnin_duplicateuser(user *u, fido toaddr)
{
    fido fromaddr;   /* address to post from */
    char toname[36], /* user to post to */
         text[256];  /* text message to send */
    int  attr;       /* attributes */

    /* set some message header fields */
    fromaddr = iomsg_gettoaddr();
    user_getname(toname, u);
    if( config_getudirect() )
        attr = config_getdattr();
    else
        attr = config_getrattr();
    sprintf(text,
        "Sorry, the user \"%s\" is already registered at this system "
        "from a different address.  If you suspect this user is "
        "someone else, please contact the sysop or write under a "
        "different user name.",
        toname);

    /* post message */
    iomsg_post(fromaddr, toname, toaddr, "Duplicate name", attr, text);
}

/* turnin_gamenotfound() - send user a 'game not found' message */
void turnin_gamenotfound(user *u, char *gamename)
{
    fido   fromaddr,     /* address to post from */
           toaddr;       /* address to post to */
    char   toname[36],   /* user to post to */
           text[1024],   /* text message to send */
           readname[16], /* name of game read from file */
           outname[19];  /* name of game to output */
    int    attr,         /* attributes */
           c;            /* line counter for output */
    result r;            /* result of game list read */

    /* set some message header fields */
    fromaddr = iomsg_gettoaddr();
    toaddr = iomsg_getfromaddr();
    user_getname(toname, u);
    if( user_getdirect(u) )
        attr = config_getdattr();
    else
        attr = config_getrattr();
    sprintf(text,
        "Sorry, the game \"%s\" is not one of the games on offer at "
        "this system.  A full list of available games is shown below:"
        "\r\r", gamename);

    /* build game list */
    c = 0;
    r = gamlst_first();
    while(r == R_OK && strlen(text) < 1000)
    {
        gamlst_getname(readname);
        sprintf(outname, "%18s", readname);
        strcat(text, outname);
        if(++c > 3) strcat(text, "\r");
        c %= 4;
        r = gamlst_next();
    }
    if(c) strcat(text, "\r");

    /* post message */
    iomsg_post(fromaddr, toname, toaddr, "Game not found", attr, text);
}

/* turnin_processorderlist() - process an order list */
result turnin_processorderlist(user *u)
{
    player *p = player_new(); /* player record */
    char    searchtext[12];   /* for searching player file */
    result  r;                /* returned to calling process */

    /* open game files */
    if(( r = turnin_opengame() ) != R_OK )
    {
        player_old(p);
        return r;
    }

    /* process new or existing player */
    sprintf( searchtext, "%ld", user_getnumber(u) );
    if(player_find(p, searchtext) == R_OK)
        turnin_existingplayer(u);
    else
        turnin_newplayer(u);

    /* clean up and return */
    turnin_closegame();
    player_old(p);
    return R_OK;
}


/* level 1 routines ***************************************************/


/* turnin_open() - open the turnin process */
result turnin_open(char *path)
{
    result r;            /* result returned to calling process */
    int    line;         /* file line number for log */
    char   netmail[128], /* netmail directory */
           logfile[128]; /* name of log file */

    /* attempt to open config */
    config_open(path);
    switch( r = config_read(path, &line) )
    {
        case R_FILE:   scrlog_cannotreadconfig(path);
                       return R_FILE;
        case R_BADCMD: scrlog_badcommandinconfig(path, line);
                       return R_BADCMD;
        case R_OK:     config_getlogfile(logfile);
                       if( ( r = scrlog_openlog(logfile, 0xffff) )
                           != R_OK)
                       {
                           scrlog_cannotopenlog(logfile);
                           goto closeconfig;
                       }
                       scrlog_turnin();
                       scrlog_configread(path);
                       break;
        default:       scrlog_badreturncode(r, "config_read");
                       return r;
    }

    /* attempt to open input message lib */
    switch( r = iomsg_open() )
    {
        case R_EOF:    scrlog_noprimaryaddress(path);
                       goto closeconfig;
        case R_FILE:   config_getnetmail(netmail);
                       scrlog_cannotopennetmailarea(netmail);
                       goto closeconfig;
        case R_MEMORY: scrlog_outofmemory("opening message area");
                       goto closeconfig;
        case R_OK:     config_getnetmail(netmail);
                       scrlog_netmailopened(netmail);
                       break;
        default:       scrlog_badreturncode(r, "iomsg_open");
                       goto closeconfig;
    }

    /* attempt to open game list */
    switch( r = gamlst_open(path, &line) )
    {
        case R_FILE:   scrlog_cannotopengamelist(path);
                       goto closeiomsg;
        case R_EOF:    scrlog_badfileingamelist(path, line);
                       goto closeiomsg;
        case R_BADCMD: scrlog_badlineingamelist(path, line);
                       goto closeiomsg;
        case R_OK:     scrlog_gamelistread(path);
                       break;
        default:       scrlog_badreturncode(r, "gamlst_open");
                       goto closeiomsg;
    }

    /* attempt to open user file */
    switch( r = user_open(path) )
    {
        case R_FILE:   if(( r = user_create(path) ) != R_OK)
                       {
                           scrlog_cannotopenfile(path, "user.data");
                           r = R_FILE;
                           goto closegamelist;
                       }
                       scrlog_filecreated(path, "user.data");
                       if(( r = user_open(path) ) != R_OK)
                       {
                           scrlog_cannotopenfile(path, "user.data");
                           r = R_FILE;
                           goto closegamelist;
                       }
                       scrlog_fileopened(path, "user.data");
                       break;
        case R_HEADER: scrlog_badheaderinfile(path, "user.data");
                       goto closegamelist;
        case R_OK:     scrlog_fileopened(path, "user.data");
                       break;
        default:       scrlog_badreturncode(r, "user_open");
                       goto closegamelist;
    }

    /* all OK so return */
    return R_OK;

    /* clean up after fatal error */
    closegamelist:
        gamlst_close();
    closeiomsg:
        iomsg_close();
    closeconfig:
        config_close();
    return r;
}

/* turnin_processmsg() - process a single message */
result turnin_processmsg(void)
{
    user *u = user_new(); /* user record */
    char  gamename[72],   /* game named on subject line */
          fromuser[36];   /* user name for log & screen */
    fido  fromaddr,       /* user address for log & screen */
          useraddr;       /* real address of user for security check */

    /* extract information from message */
    iomsg_getfromname(fromuser);
    fromaddr = iomsg_getfromaddr();
    iomsg_getgame(gamename);
    scrlog_msgread(fromuser, fromaddr);

    /* check user */
    if( turnin_checkuser(u, fromuser) == R_MEMORY )
    {
        scrlog_outofmemory("checking user");
        user_old(u);
        return R_MEMORY;
    }

    /* process message */
    useraddr = user_getaddr(u);
    if( useraddr.zone  != fromaddr.zone  ||
        useraddr.net   != fromaddr.net   ||
        useraddr.node  != fromaddr.node  ||
        useraddr.point != fromaddr.point )
        turnin_duplicateuser(u, fromaddr);
    else if( gamlst_find(gamename) != R_OK )
        turnin_gamenotfound(u, gamename);
    else
        turnin_processorderlist(u);

    /* clean up & return */
    iomsg_rcvd();
    user_old(u);
    return R_OK;
}

/* turnin_close() - close program */
void turnin_close(void)
{
    config_close();
    iomsg_close();
    gamlst_close();
    user_close();
    scrlog_finished();
}


/* level 0 routines ***************************************************/


/* turnin_main() - main turn processing */
result turnin_main(char *path, char **argp)
{
    result r; /* returned to calling process */

    if(( r = turnin_open(path) ) != R_OK) return r;
    scrlog_scanning(); r = iomsg_first();
    while(r == R_OK)
    {
        if(( r = turnin_processmsg() ) != R_OK)
        {
            turnin_close();
            return r;
        }
        scrlog_scanning(); r = iomsg_next();
    }
    turnin_close();
    return R_OK;
}
