/**********************************************************************
 * TRNOUT.C                          Copyright (C) Damian Walker 1997 *
 *--------------------------------------------------------------------*
 * AstroMail 1.00.  Sends reports to players via netmail.             *
 *--------------------------------------------------------------------*
 * Author   Damian G Walker                                           *
 * Date     17-Apr-97                                                 *
 **********************************************************************/


/* included headers ***************************************************/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "astroml.h"
#include "detail.h"
#include "report.h"
#include "config.h"
#include "user.h"
#include "gamlst.h"
#include "plytxt.h"
#include "pfleet.h"
#include "joinrq.h"
#include "player.h"
#include "scrlog.h"
#include "iomsg.h"


/* enums, structures &c. **********************************************/


#define P_PLAYER 0x01
#define P_JOINRQ 0x02
#define P_PFLEET 0x04
#define P_USER   0x08


/* global variables ***************************************************/


int trnout_pack; /* which database files need packing? */


/* level 2 routines ***************************************************/


/* trnout_processjoinreport() - process a 'join' report */
void trnout_processjoinreport(report *t, char *gamename)
{
    joinrq *j = joinrq_new(); /* join request record */
    player *p = player_new(); /* player record */
    char    empname[16],      /* empire name */
            joinrpt[256],     /* text of join report */
            homeworld[16];    /* player's homeworld */

    /* check player & log */
    if(joinrq_find( j, report_gettrans(t) ) != R_OK) return;
    scrlog_processingreport("join");

    /* add new player to player file & remove join request */
    joinrq_getempire(empname, j);
    player_setnumber( p, joinrq_getuser(j) );
    player_setempire(p, empname);
    player_write(p);
    joinrq_delete(j); trnout_pack |= P_JOINRQ;

    /* send join report, clean up and exit */
    report_gethomewld(homeworld, t);
    sprintf(joinrpt, "Your application to join game \'%s\' of "
        "AstroWar was successful.  Your empire name is \'%s\' and your "
        "homeworld is \'%s\'.  ", gamename, empname, homeworld);
    plytxt_write( joinrq_getuser(j), joinrpt );
    joinrq_old(j);
    player_old(p);
}

/* trnout_processjoinerror() - process a 'join' error */
void trnout_processjoinerror(report *t, char *gamename)
{
    joinrq *j = joinrq_new(); /* join request record */
    long    unum;             /* user number */

    /* check player & log */
    if(joinrq_find( j, report_gettrans(t) ) != R_OK) return;
    scrlog_processingreport("join error");

    /* add new player to player file & remove join request */
    unum = joinrq_getuser(j);
    joinrq_delete(j); trnout_pack |= P_JOINRQ;

    /* send join error, clean up and exit */
    plytxt_write(unum, "Your application to join game \"");
    plytxt_write(unum, gamename);
    plytxt_write(unum, "\" of AstroWar was unsuccessful (");
    switch( report_geterror(t) )
    {
        case 'F': plytxt_write(unum, "game full). ");          break;
        case 'D': plytxt_write(unum, "empire name in use). "); break;
        default:  plytxt_write(unum, "reason unknown). ");
    }

    /* clean up */
    joinrq_old(j);
}

/* trnout_processsendreport() - process a 'send' report */
void trnout_processsendreport(report *t)
{
    player *p = player_new(); /* player report is bound for */
    pfleet *f = pfleet_new(); /* player fleet */
    char    ename[16],        /* empire name */
            fleetno[11],      /* fleet number */
            ships[11],        /* ships in fleet */
            origin[16],       /* planet of origin */
            dest[16];         /* destination planet */
    long    unum;             /* user number */

    /* one of our players? */
    report_getempire(ename, t);
    if( player_find(p, ename) != R_OK )
    {
        player_old(p);
        pfleet_old(f);
        return;
    }
    unum = player_getnumber(p);
    scrlog_processingreport("send");

    /* save player fleet record */
    report_getorigin(origin, t);
    report_getdest(dest, t);
    pfleet_setfleetno( f, report_getfleetno(t) );
    pfleet_setuser(    f, unum);
    pfleet_setorigin(  f, origin);
    pfleet_setdest(    f, dest);
    pfleet_setships(   f, report_getships(t) );
    pfleet_write(f);

    /* output report */
    sprintf( fleetno, "%ld", report_getfleetno(t) );
    sprintf( ships, "%ld", report_getships(t) );
    plytxt_write(unum, "Fleet ");
    plytxt_write(unum, fleetno);
    plytxt_write(unum, " of ");
    plytxt_write(unum, ships);
    if(!stricmp(ships, "1"))
        plytxt_write(unum, " ship launched from ");
    else
        plytxt_write(unum, " ships launched from ");
    plytxt_write(unum, origin);
    plytxt_write(unum, " to ");
    plytxt_write(unum, dest);
    plytxt_write(unum, ". ");

    /* clean up and return */
    player_old(p);
    pfleet_old(f);
}

/* trnout_processsenderror() - process a 'send' error */
void trnout_processsenderror(report *t)
{
    player *p = player_new(); /* player report is bound for */
    char    ename[16],        /* empire name */
            ships[11],        /* ships in fleet */
            origin[16],       /* planet of origin */
            dest[16];         /* destination planet */
    long    unum;             /* user number */

    /* one of our players? */
    report_getempire(ename, t);
    if( player_find(p, ename) != R_OK )
    {
        player_old(p);
        return;
    }
    unum = player_getnumber(p);
    scrlog_processingreport("send error");

    /* get text info from report */
    report_getorigin(origin, t);
    report_getdest(dest, t);
    sprintf( ships, "%ld", report_getships(t) );

    /* output report */
    plytxt_write(unum, "Cannot launch fleet of ");
    plytxt_write(unum, ships);
    plytxt_write(unum, " ships from ");
    plytxt_write(unum, origin);
    plytxt_write(unum, " to ");
    plytxt_write(unum, dest);
    switch( report_geterror(t) )
    {
        case 'U': plytxt_write(unum, " (origin not your planet). ");
                  break;
        case 'O': plytxt_write(unum, " (origin unknown). ");
                  break;
        case 'D': plytxt_write(unum, " (destination unknown). ");
                  break;
        case 'S': plytxt_write(unum, " (you haven\'t enough ships). ");
                  break;
        case 'M': plytxt_write(unum, " (destination out of range). ");
                  break;
        default:  plytxt_write(unum, " (reason unknown). ");
    }

    /* clean up and return */
    player_old(p);
}

/* trnout_processplanetsreport() - process a 'planets' report */
void trnout_processplanetsreport(report *t)
{
    player *p = player_new(); /* player report is bound for */
    result  r;                /* result of list scan etc. */
    char    ename[16],        /* empire name */
            pname[16],        /* planet name */
            buf[72];          /* buffer for output line */
    pos     pp;               /* planet position */
    long    unum;             /* user number */

    /* one of our players? */
    report_getempire(ename, t);
    if( player_find(p, ename) != R_OK )
    {
        player_old(p);
        return;
    }
    unum = player_getnumber(p);
    scrlog_processingreport("planets");

    /* start outputting report */
    plytxt_write( unum,
        "Your planets are:\r\r"
        "Name               position prodn      ships\r"
        "--------------------------------------------\r" );
    r = report_firstentry(t);
    while(r == R_OK)
    {
        report_getname(pname, t);
        pp = report_getpos(t);
        sprintf( buf, "%-15s %5d,%5d %5d %10ld\r", pname, pp.x, pp.y,
            report_getprod(t), report_getships(t) );
        plytxt_write(unum, buf);
        r = report_nextentry(t);
    }
    plytxt_write(unum, "\r");

    /* clean up */
    player_old(p);
}

/* trnout_processfleetsreport() - process a 'fleets' report */
void trnout_processfleetsreport(report *t)
{
    player *p = player_new(); /* player report is bound for */
    result  r;                /* result of list scan etc. */
    char    ename[16],        /* empire name */
            origin[16],       /* origin planet */
            dest[16],         /* destination planet */
            buf[72];          /* buffer for output line */
    long    unum;             /* user number */

    /* one of our players? */
    report_getempire(ename, t);
    if( player_find(p, ename) != R_OK )
    {
        player_old(p);
        return;
    }
    unum = player_getnumber(p);
    scrlog_processingreport("fleets");

    /* start outputting report */
    plytxt_write( unum,
        "Your fleets are:\r\r"
        "     Fleet origin          dest                 ships  dist\r"
        "-----------------------------------------------------------\r" );
    r = report_firstentry(t);
    while(r == R_OK)
    {
        report_getorigin(origin, t);
        report_getdest(dest, t);
        sprintf( buf, "%10ld %-15s %-15s %10ld %5d\r",
            report_getfleetno(t), origin, dest, report_getships(t),
            report_getdist(t) );
        plytxt_write(unum, buf);
        r = report_nextentry(t);
    }
    plytxt_write(unum, "\r");

    /* clean up */
    player_old(p);
}

/* trnout_processmapreport() - process a 'map' report */
void trnout_processmapreport(report *t)
{
    player *p = player_new(); /* player record */
    char    map[15][15][3],   /* map image */
            centre[16],       /* planet at centre of map */
            pname[16],        /* temporary planet name */
            ename[16],        /* empire name */
            header[72],       /* line or column header */
            pdetail[72];      /* detail of planet for key */
    pos     cpos,             /* centre planet position */
            ppos;             /* temporary planet position */
    result  r;                /* result of report read */
    long    pnum;             /* player number */
    int     x, y;             /* used for map image calculation */

    /* attempt to find player */
    report_getempire(ename, t);
    if( player_find(p, ename) != R_OK )
    {
        player_old(p);
        return;
    }
    pnum = player_getnumber(p);
    scrlog_processingreport("map");

    /* get map centre details */
    report_getcentre(centre, t);
    report_firstentry(t);
    while(stricmp( report_getname(pname, t), centre ))
        report_nextentry(t);
    cpos = report_getpos(t);

    /* put planets on map image */
    for(x = 0; x <= 14; ++x)
        for(y = 0; y <= 14; ++y)
            strcpy( map[x][y], ".." );
    r = report_firstentry(t);
    while(r == R_OK)
    {
        x = 7 + report_getpos(t).x - cpos.x;
        y = 7 + report_getpos(t).y - cpos.y;
        report_getname(pname, t);
        strncpy( map[x][y], pname, 2 );
        r = report_nextentry(t);
    }

    /* write map image */
    plytxt_write(pnum, "Map of planets around ");
    plytxt_write(pnum, centre);
    plytxt_write(pnum, ":\r\r");
    sprintf(header, "%6s|%-6d%18s%6d\r", "", cpos.x - 7, "",
        cpos.x + 7);
    plytxt_write(pnum, header);
    plytxt_write(pnum, "------+------------------------------\r");
    for(y = 14; y >= 0; --y)
    {
        sprintf(header, "%6d|", cpos.y - 7 + y);
        plytxt_write(pnum, header);
        for(x = 0; x <= 14; ++x)
            plytxt_write(pnum, map[x][y]);
        plytxt_write(pnum, "\r");
    }
    plytxt_write(pnum, "\r");

    /* write planet key */
    r = report_firstentry(t);
    while(r == R_OK)
    {
        report_getname(pname, t);
        ppos = report_getpos(t);
        sprintf(pdetail, "%.2s=%s (%d,%d). ", pname, pname, ppos.x,
            ppos.y);
        plytxt_write(pnum, pdetail);
        r = report_nextentry(t);
    }

    /* clean up */
    player_old(p);
}

/* trnout_processmaperror() - process a 'map' error */
void trnout_processmaperror(report *t)
{
    player *p = player_new(); /* player record */
    char    ename[16],        /* empire name */
            centre[16];       /* planet at centre of map */
    long    unum;             /* user number */

    /* one of our players? */
    report_getempire(ename, t);
    if( player_find(p, ename) != R_OK )
    {
        player_old(p);
        return;
    }
    unum = player_getnumber(p);
    scrlog_processingreport("map error");

    /* extract information from report */
    report_getcentre(centre, t);

    /* output report */
    plytxt_write(unum, "Cannot map planets around ");
    plytxt_write(unum, centre);
    switch( report_geterror(t) )
    {
        case 'U': plytxt_write(unum, " (not your planet). "); break;
        case 'C': plytxt_write(unum, " (planet unknown). ");  break;
        default:  plytxt_write(unum, " (reason unknown). ");  break;
    }

    /* clean up */
    player_old(p);
}

/* trnout_processwritereport() - process a player message */
void trnout_processwritereport(report *t)
{
    player *p = player_new(); /* player record */
    char    ename[16],        /* empire name */
            sender[16],       /* sender's empire name */
           *text;             /* pointer to message text */
    long    unum;             /* user number */

    /* one of our players? */
    report_getempire(ename, t);
    if( player_find(p, ename) != R_OK )
    {
        player_old(p);
        return;
    }
    unum = player_getnumber(p);
    scrlog_processingreport("write");

    /* extract information from report */
    report_getsender(sender, t);
    text = report_gettext(t);

    /* output report */
    plytxt_write(unum, "Message from ");
    if(*sender)
        plytxt_write(unum, sender);
    else
        plytxt_write(unum, "the system");
    plytxt_write(unum, ":\r\r");
    plytxt_write(unum, text);
    if(text[ strlen(text) - 1 ] != '\r')
        plytxt_write(unum, "\r");
    plytxt_write(unum, "\r");

    /* clean up */
    free(text);
    player_old(p);
}

/* trnout_processwriteerror() - process a bounced message */
void trnout_processwriteerror(report *t)
{
    player *p = player_new(); /* player record */
    char    ename[16],        /* empire name */
            recip[16];        /* intended recipient */
    long    unum;             /* user number */

    /* one of our players? */
    report_getempire(ename, t);
    if( player_find(p, ename) != R_OK )
    {
        player_old(p);
        return;
    }
    unum = player_getnumber(p);
    scrlog_processingreport("write error");

    /* extract information from report */
    report_getrecip(recip, t);

    /* output report */
    plytxt_write(unum, "Message to ");
    plytxt_write(unum, recip);
    plytxt_write(unum, " bounced");
    switch( report_geterror(t) )
    {
        case 'R': plytxt_write(unum, " (recipient unknown). "); break;
        default:  plytxt_write(unum, " (reason unknown). ");    break;
    }

    /* clean up */
    player_old(p);
}

/* trnout_processvictoryreport() - process a victory report */
void trnout_processvictoryreport(report *t)
{
    player *p = player_new(); /* player report is bound for */
    char    ename[16],        /* empire name */
            victor[16];       /* victorious empire */
    long    unum;             /* user number */

    /* one of our players? */
    report_getempire(ename, t);
    if( player_find(p, ename) != R_OK )
    {
        player_old(p);
        return;
    }
    unum = player_getnumber(p);
    scrlog_processingreport("victory");
    report_getvictor(victor, t);

    /* output arrival report */
    plytxt_write(unum, "IMPORTANT NEWS: ");
    plytxt_write(unum, victor);
    plytxt_write(unum, " HAS WON THE GAME! ");

    /* clean up and return */
    player_old(p);
}

/* trnout_processarrivalreport() - process an 'arrival' report */
void trnout_processarrivalreport(report *t)
{
    player *p = player_new(); /* player report is bound for */
    pfleet *f = pfleet_new(); /* player fleet */
    char    ename[16],        /* empire name */
            fleetno[11],      /* fleet number */
            ships[11],        /* ships in fleet */
            dest[16];         /* destination planet */
    long    unum;             /* user number */

    /* one of our players? */
    report_getempire(ename, t);
    if( player_find(p, ename) != R_OK )
    {
        player_old(p);
        pfleet_old(f);
        return;
    }
    unum = player_getnumber(p);
    scrlog_processingreport("arrival");

    /* gather information from player fleet record and report */
    sprintf( fleetno, "%ld", report_getfleetno(t) );
    pfleet_find( f, report_getfleetno(t) );
    sprintf( ships, "%ld", pfleet_getships(f) );
    pfleet_getdest(dest, f);
    pfleet_delete(f);
    trnout_pack |= P_PFLEET;

    /* output arrival report */
    plytxt_write(unum, "Fleet ");
    plytxt_write(unum, fleetno);
    plytxt_write(unum, " of ");
    plytxt_write(unum, ships);
    if( !strcmp(ships, "1") )
        plytxt_write(unum, " ship has arrived at ");
    else
        plytxt_write(unum, " ships has arrived at ");
    plytxt_write(unum, dest);
    plytxt_write(unum, ". ");

    /* clean up and return */
    player_old(p);
    pfleet_old(f);
}

/* trnout_processintelligencereport() - process an intelligence rpt */
void trnout_processintelligencereport(report *t)
{
    player *p = player_new(); /* player report is bound for */
    pfleet *f = pfleet_new(); /* player fleet */
    char    ename[16],        /* empire name */
            fleetno[11],      /* fleet number */
            ships[11],        /* ships in fleet */
            origin[16],       /* planet of origin */
            dest[16],         /* destination planet */
            owner[16],        /* owner of the planet */
            prod[11];         /* production rating */
    long    unum;             /* user number */

    /* one of our players? */
    report_getempire(ename, t);
    if( player_find(p, ename) != R_OK )
    {
        player_old(p);
        pfleet_old(f);
        return;
    }
    unum = player_getnumber(p);
    scrlog_processingreport("intelligence");

    /* gather information from player fleet record and report */
    sprintf( fleetno, "%ld", report_getfleetno(t) );
    report_getowner(owner, t);
    sprintf( prod, "%d", report_getprod(t) );
    sprintf( ships, "%ld", report_getships(t) );
    pfleet_find( f, report_getfleetno(t) );
    pfleet_getorigin(origin, f);
    pfleet_getdest(dest, f);
    pfleet_setorigin(f, dest);
    pfleet_setdest(f, origin);
    pfleet_write(f);

    /* output intelligence report */
    plytxt_write(unum, "Intelligence report from scout fleet ");
    plytxt_write(unum, fleetno);
    plytxt_write(unum, " at ");
    plytxt_write(unum, dest);
    plytxt_write(unum, ":\r\rOwner:      ");
    if( !strcmp(owner, "") )
        plytxt_write(unum, "(neutral)");
    else
        plytxt_write(unum, owner);
    plytxt_write(unum, "\rProduction: ");
    plytxt_write(unum, prod);
    plytxt_write(unum, "\rShips:      ");
    plytxt_write(unum, ships);
    plytxt_write(unum, "\r\r");

    /* clean up and return */
    player_old(p);
    pfleet_old(f);
}

/* trnout_processbattlereport() - process a battle report */
void trnout_processbattlereport(report *t)
{
    player *p = player_new(); /* player report is bound for */
    pfleet *f = pfleet_new(); /* player fleet */
    char    ename[16],        /* empire name */
            fleetno[11],      /* fleet number */
            ships[11],        /* ships left */
            enemy[11],        /* enemy ships left */
            owner[16],        /* owner of destination planet */
            dest[16];         /* destination planet */
    long    unum;             /* user number */

    /* one of our players? */
    report_getempire(ename, t);
    if( player_find(p, ename) != R_OK )
    {
        player_old(p);
        pfleet_old(f);
        return;
    }
    unum = player_getnumber(p);
    scrlog_processingreport("battle");

    /* gather information from player fleet record and report */
    sprintf( fleetno, "%ld", report_getfleetno(t) );
    pfleet_find( f, report_getfleetno(t) );
    sprintf( ships, "%ld", report_getships(t) );
    sprintf( enemy, "%ld", report_getenemy(t) );
    pfleet_getdest(dest, f);
    pfleet_delete(f);
    report_getowner(owner, t);
    trnout_pack |= P_PFLEET;

    /* output arrival report */
    if(*ships != '0')
    {
        plytxt_write(unum, "Invasion of ");
        plytxt_write(unum, dest);
        plytxt_write(unum, " by fleet ");
        plytxt_write(unum, fleetno);
        plytxt_write(unum, " successful.  All ships of ");
        if(*owner)
            plytxt_write(unum, owner);
        else
            plytxt_write(unum, "neutral owner");
        plytxt_write(unum, " destroyed, with ");
        plytxt_write(unum, ships);
        plytxt_write(unum, " of your ships remaining. ");
    }
    else
    {
        plytxt_write(unum, "Invasion of ");
        plytxt_write(unum, dest);
        plytxt_write(unum, " by fleet ");
        plytxt_write(unum, fleetno);
        plytxt_write(unum, " failed. All of your ships have been "
                           "destroyed, with ");
        plytxt_write(unum, enemy);
        plytxt_write(unum, " ships of ");
        if(*owner)
            plytxt_write(unum, owner);
        else
            plytxt_write(unum, "neutral owner");
        plytxt_write(unum, " remaining. ");
    }

    /* clean up and return */
    player_old(p);
    pfleet_old(f);
}

/* trnout_processdefencereport() - process a defence report */
void trnout_processdefencereport(report *t)
{
    player *p = player_new(); /* player report is bound for */
    char    ename[16],        /* empire name */
            pname[16],        /* planet name */
            ships[11],        /* ships left */
            enemy[11],        /* enemy ships left */
            owner[16];        /* owner of attack fleet */
    long    unum;             /* user number */

    /* one of our players? */
    report_getempire(ename, t);
    if( player_find(p, ename) != R_OK )
    {
        player_old(p);
        return;
    }
    unum = player_getnumber(p);
    scrlog_processingreport("defence");

    /* gather information from player fleet record and report */
    sprintf( ships, "%ld", report_getships(t) );
    sprintf( enemy, "%ld", report_getenemy(t) );
    report_getplanet(pname, t);
    report_getowner(owner, t);

    /* output arrival report */
    if(*ships != '0')
    {
        plytxt_write(unum, "Invasion of ");
        plytxt_write(unum, pname);
        plytxt_write(unum, " by ");
        plytxt_write(unum, owner);
        plytxt_write(unum, " attempted. ");
        plytxt_write(unum, ships);
        plytxt_write(unum, " of your ships remain in orbit. ");
    }
    else
    {
        plytxt_write(unum, "Planet ");
        plytxt_write(unum, pname);
        plytxt_write(unum, " invaded by ");
        plytxt_write(unum, owner);
        plytxt_write(unum, ". All of your ships have been destroyed, "
                           "with ");
        plytxt_write(unum, enemy);
        plytxt_write(unum, " ships of the enemy remaining "
                           "after the battle. ");
    }

    /* clean up and return */
    player_old(p);
}

/* trnout_send() - send player report, if any */
void trnout_send(user *u, char *gamename, fido fromaddr)
{
    char *text,        /* text to send */
          usernum[11], /* user number to find */
          toname[36],  /* user name */
          subject[72]; /* subject line of outgoing message */
    fido  toaddr;      /* address to send to */
    int   attr;        /* attributes for message */

    /* attempt to read text of report */
    if( (text = plytxt_read( user_getnumber(u) )) == NULL ) return;

    /* prepare message and send */
    sprintf( usernum, "%ld", user_getnumber(u) );
    user_getname(toname, u);
    toaddr = user_getaddr(u);
    sprintf(subject, "Report for game \"%s\"", gamename);
    if( user_getdirect(u) )
        attr = config_getdattr();
    else
        attr = config_getrattr();
    iomsg_post(fromaddr, toname, toaddr, subject, attr, text);
    scrlog_reportposted(toname, toaddr);

    /* clean up */
    free(text);
}


/* level 1 routines ***************************************************/


/* trnout_interpret() - interpret trnout specific parameters */
result trnout_interpret(char *gamename, fido *fromaddr, char **argp)
{
    int argc; /* argument counter */

    argc = 0; strcpy(gamename, "");
    sscanf("0:0/0.0", "%d:%d/%d.%d", &fromaddr->zone, &fromaddr->net,
        &fromaddr->node, &fromaddr->point);
    while(argc < 10 && argp[argc] != NULL)
    {
        if(!stricmp( argp[argc], "-game" ) && argc < 9)
        {
            ++argc;
            if(strlen( argp[argc] ) > 15)
            {
                strncpy( gamename, argp[argc], 15 );
                gamename[15] = '\0';
            }
            else strcpy( gamename, argp[argc] );
        }
        else if(!stricmp( argp[argc], "-from" ) && argc < 9)
        {
            if(sscanf(argp[++argc], "%d:%d/%d.%d", &fromaddr->zone,
                &fromaddr->net, &fromaddr->node, &fromaddr->point) != 4)
                return R_BADCMD;
        }
        else
        {
            scrlog_error();
            return R_BADCMD;
        }
        ++argc;
    }
    return R_OK;
}

/* trnout_open() - open game files */
result trnout_open(char *gamepath, char *path, char *gamename, fido
    *fromaddr)
{
    result  r;             /* result returned to calling process */
    int     line;          /* file line number for log */
    char    netmail[128],  /* netmail directory */
            logfile[128];  /* name of log file */
    detail *d;             /* detail record for game */

    /* attempt to open config */
    config_open(path);
    switch( r = config_read(path, &line) )
    {
        case R_FILE:   scrlog_cannotreadconfig(path);
                       return R_FILE;
        case R_BADCMD: scrlog_badcommandinconfig(path, line);
                       return R_BADCMD;
        case R_OK:     config_getlogfile(logfile);
                       if( ( r = scrlog_openlog(logfile, 0xffff) )
                           != R_OK)
                       {
                           scrlog_cannotopenlog(logfile);
                           goto closeconfig;
                       }
                       scrlog_trnout();
                       scrlog_configread(path);
                       break;
        default:       scrlog_badreturncode(r, "config_read");
                       return r;
    }

    /* retrieve address to post from */
    if( fromaddr->zone == 0 && fromaddr->net == 0 &&
        fromaddr->node == 0 && fromaddr->point == 0)
    {
        if( config_firstaka(fromaddr) != R_OK )
        {
            scrlog_noprimaryaddress(path);
            goto closeconfig;
        }
    }

    /* attempt to open input message lib */
    switch( r = iomsg_open() )
    {
        case R_EOF:    scrlog_noprimaryaddress(path);
                       goto closeconfig;
        case R_FILE:   config_getnetmail(netmail);
                       scrlog_cannotopennetmailarea(netmail);
                       goto closeconfig;
        case R_MEMORY: scrlog_outofmemory("opening message area");
                       goto closeconfig;
        case R_OK:     config_getnetmail(netmail);
                       scrlog_netmailopened(netmail);
                       break;
        default:       scrlog_badreturncode(r, "iomsg_open");
                       goto closeconfig;
    }

    /* attempt to open game list */
    switch( r = gamlst_open(path, &line) )
    {
        case R_FILE:   scrlog_cannotopengamelist(path);
                       goto closeiomsg;
        case R_EOF:    scrlog_badfileingamelist(path, line);
                       goto closeiomsg;
        case R_BADCMD: scrlog_badlineingamelist(path, line);
                       goto closeiomsg;
        case R_OK:     scrlog_gamelistread(path);
                       break;
        default:       scrlog_badreturncode(r, "gamlst_open");
                       goto closeiomsg;
    }

    /* attempt to open user file */
    switch( r = user_open(path) )
    {
        case R_FILE:   if(( r = user_create(path) ) != R_OK)
                       {
                           scrlog_cannotopenfile(path, "user.data");
                           goto closegamelist;
                       }
                       scrlog_filecreated(path, "user.data");
                       if(( r = user_open(path) ) != R_OK)
                       {
                           scrlog_cannotopenfile(path, "user.data");
                           goto closegamelist;
                       }
                       scrlog_fileopened(path, "user.data");
                       break;
        case R_HEADER: scrlog_badheaderinfile(path, "user.data");
                       goto closegamelist;
        case R_OK:     scrlog_fileopened(path, "user.data");
                       break;
        default:       scrlog_badreturncode(r, "user_open");
                       goto closegamelist;
    }

    /* now open game detail file */
    if(( r = gamlst_find(gamename) ) != R_OK)
    {
        scrlog_gamenotfound(gamename);
        goto closeuserfile;
    }
    gamlst_getpath(gamepath);
    switch( r = detail_open(&d, gamepath) )
    {
        case R_FILE:  scrlog_cannotopendetail(gamepath);
                       goto closeuserfile;
        case R_MEMORY: scrlog_outofmemory("loading detail file");
                       goto closeuserfile;
        case R_BADCMD: scrlog_badcommandindetail(gamepath);
                       goto closeuserfile;
        case R_OK:     scrlog_detailread(gamepath);
                       break;
        default:       scrlog_badreturncode(r, "detail_open");
                       goto closeuserfile;
    }
    detail_close(d);

    /* open player file */
    switch( r = player_open(gamepath) )
    {
        case R_FILE:   if(( r = player_create(gamepath) ) != R_OK)
                       {
                           scrlog_cannotopenfile(gamepath, "player.data");
                           r = R_FILE;
                           goto closeuserfile;
                       }
                       scrlog_filecreated(gamepath, "player.data");
                       if(( r = player_open(gamepath) ) != R_OK)
                       {
                           scrlog_cannotopenfile(gamepath, "player.data");
                           r = R_FILE;
                           goto closeuserfile;
                       }
                       scrlog_fileopened(gamepath, "player.data");
                       break;
        case R_HEADER: scrlog_badheaderinfile(gamepath, "player.data");
                       goto closeuserfile;
        case R_OK:     scrlog_fileopened(gamepath, "player.data");
                       break;
        default:       scrlog_badreturncode(r, "player_open");
                       goto closeuserfile;
    }

    /* open pfleet file */
    switch( r = pfleet_open(gamepath) )
    {
        case R_FILE:   if(( r = pfleet_create(gamepath) ) != R_OK)
                       {
                           scrlog_cannotopenfile(gamepath, "pfleet.data");
                           r = R_FILE;
                           goto closeplayer;
                       }
                       scrlog_filecreated(gamepath, "pfleet.data");
                       if(( r = pfleet_open(gamepath) ) != R_OK)
                       {
                           scrlog_cannotopenfile(gamepath, "pfleet.data");
                           r = R_FILE;
                           goto closeplayer;
                       }
                       scrlog_fileopened(gamepath, "pfleet.data");
                       break;
        case R_HEADER: scrlog_badheaderinfile(gamepath, "pfleet.data");
                       goto closeplayer;
        case R_OK:     scrlog_fileopened(gamepath, "pfleet.data");
                       break;
        default:       scrlog_badreturncode(r, "pfleet_open");
                       goto closeplayer;
    }

    /* open joinrq file */
    switch( r = joinrq_open(gamepath) )
    {
        case R_FILE:   if(( r = joinrq_create(gamepath) ) != R_OK)
                       {
                           scrlog_cannotopenfile(gamepath, "joinrq.data");
                           r = R_FILE;
                           goto closepfleet;
                       }
                       scrlog_filecreated(gamepath, "joinrq.data");
                       if(( r = joinrq_open(gamepath) ) != R_OK)
                       {
                           scrlog_cannotopenfile(gamepath, "joinrq.data");
                           r = R_FILE;
                           goto closepfleet;
                       }
                       scrlog_fileopened(gamepath, "joinrq.data");
                       break;
        case R_HEADER: scrlog_badheaderinfile(gamepath, "joinrq.data");
                       goto closepfleet;
        case R_OK:     scrlog_fileopened(gamepath, "joinrq.data");
                       break;
        default:       scrlog_badreturncode(r, "joinrq_open");
                       goto closepfleet;
    }

    /* open report file */
    switch( r = report_open(gamepath) )
    {
        case R_FILE:   scrlog_cannotopenfile(gamepath, "report.data");
                       goto closejoinrq;
        case R_HEADER: scrlog_badheaderinfile(gamepath, "report.data");
                       goto closejoinrq;
        case R_OK:     scrlog_fileopened(gamepath, "report.data");
                       break;
        default:       scrlog_badreturncode(r, "report_open");
                       goto closejoinrq;
    }

    /* open player text file library */
    plytxt_open(gamepath); /* no need to check result since we know the
                              directory 'gamepath' exists by now */

    /* all OK so return */
    return R_OK;

    /* clean up after fatal error */
    closejoinrq:
        joinrq_close();
    closepfleet:
        pfleet_close();
    closeplayer:
        player_close();
    closeuserfile:
        user_close();
    closegamelist:
        gamlst_close();
    closeiomsg:
        iomsg_close();
    closeconfig:
        config_close();
    return r;
}

/* trnout_procesreport() - process a single report */
void trnout_processreport(report *t, char *gamename)
{
    switch( report_gettype(t) )
    {
        case 'J': trnout_processjoinreport(t, gamename); break;
        case 'j': trnout_processjoinerror(t, gamename);  break;
        case 'S': trnout_processsendreport(t);           break;
        case 's': trnout_processsenderror(t);            break;
        case 'P': trnout_processplanetsreport(t);        break;
        case 'F': trnout_processfleetsreport(t);         break;
        case 'M': trnout_processmapreport(t);            break;
        case 'm': trnout_processmaperror(t);             break;
        case 'W': trnout_processwritereport(t);          break;
        case 'w': trnout_processwriteerror(t);           break;
        case 'V': trnout_processvictoryreport(t);        break;
        case 'A': trnout_processarrivalreport(t);        break;
        case 'I': trnout_processintelligencereport(t);   break;
        case 'B': trnout_processbattlereport(t);         break;
        case 'D': trnout_processdefencereport(t);        break;
        default:
    }
}

/* trnout_close() - finish off and close the files */
void trnout_close(char *gamename, fido fromaddr, char *gamepath,
    char *path)
{
    user  *u = user_new(); /* user record */
    result r;              /* result of read */

    /* log message */
    scrlog_donereports();

    /* output the reports */
    r = user_first(u, P_NUMBER);
    while(r == R_OK)
    {
        trnout_send(u, gamename, fromaddr);
        r = user_next(u, P_NUMBER);
    }
    user_old(u);

    /* close the files */
    plytxt_close();
    report_close();
    joinrq_close();
    pfleet_close();
    player_close();
    user_close();
    gamlst_close();
    iomsg_close();
    config_close();

    /* purge the files */
    if(trnout_pack & P_JOINRQ)
    {
        joinrq_pack(gamepath);
        scrlog_purgedfile(gamepath, "joinrq.data");
    }
    if(trnout_pack & P_PFLEET)
    {
        pfleet_pack(gamepath);
        scrlog_purgedfile(gamepath, "pfleet.data");
    }
    if(trnout_pack & P_PLAYER)
    {
        player_pack(gamepath);
        scrlog_purgedfile(gamepath, "player.data");
    }
    if(trnout_pack & P_USER)
    {
        user_pack(path);
        scrlog_purgedfile(gamepath, "user.data");
    }

    /* screen/log message */
    scrlog_finished();
}


/* level 0 routines ***************************************************/


/* trnout_main() - main program */
result trnout_main(char *path, char **argp)
{
    report *t = report_new(); /* report to process */
    result  r;                /* result of read */
    char    gamename[16],     /* name of game */
            gamepath[72];     /* path of game files */
    fido    fromaddr;         /* address to post from */

    /* open, process and close */
    if( trnout_interpret(gamename, &fromaddr, argp) == R_OK )
        if( trnout_open(gamepath, path, gamename, &fromaddr) == R_OK )
        {
            r = report_first(t);
            while(r == R_OK)
            {
                trnout_processreport(t, gamename);
                r = report_next(t);
            }
            trnout_close(gamename, fromaddr, gamepath, path);
        }

    /* free report memory & return */
    report_old(t);
    return R_OK;
}
